/*******************************************************************************
    PROGRAM NAME : AMDHST.c

    DATE : April 23, 1982

    AUTHOR : Tsu-Phong Wu

    This software supports the file transfer between RTE16 and a host.

    This program constantly listens to RTE16 and tries to satisfy the request
    regardless of any invalid conditions.  Therefore RTE16 must monitor the
    communication and cancel the request under those circumstances.  A request
    to cancel is always accepted at any time.

    This program is written specifically to run under UNIX.  It needs
    modifications in order to run under any other operating system supporting
    "C" language.  It also serves as an example if one wants to write his own
    software to communicate with RTE16.

    The message format is described ad follows :

	header record
	-------------

	  start    filename				 end    carriage
	of header   length   type  filename  checksum  of text   return
	................................................................

	   SOH        xx      xx   xxxxxxxx     xx       ETX       CR

	where
	    type can be 1: download file,
			2: upsave file,
			3: upsave print file,
			4: close print file,
			5: reopen print file,
			6: delete file
	    filename is of length up to 8 characters,
	    checksum is the sum of all characters from length field to filename
		field,
	    CR is converted to LF by UNIX

	data record
	-----------

	 start    data    block   data              end    carriage
	of text  length  number  block  checksum  of text   return
	...........................................................

	  STX      xx      xx    xxxxx     xx       ETX       CR

	where
	    data block is of length up to 128 characters (corresponding to 64
		bytes after packed),
	    checksum is the sum of all characters from length field to data
		block field,
	    CR is converted to LF by UNIX

	end record
	----------

	 start		  block		     end    carriage
	of text  length  number  checksum  of text   return
	....................................................

	  STX      00      00       C0       ETX       CR

	where
	    CR is converted to LF by UNIX
*******************************************************************************/

#include	<stdio.h>

#define	TRUE		1
#define	FALSE		0

/* RTE16 request */
#define	DOWNLOAD	1	/* Host to RTE */
#define	UPSAVE		2	/* RTE to Host */
#define	PRINTSAVE	3	/* RTE to Host PRINT file */
#define	CLOSE		4	/* Close PRINT file */
#define	REOPEN		5	/* Reopen PRINT file */
#define	DELETE		6	/* Delete file */

/* sync chars */
#define	SOH		0x1	/* start of header */
#define	STX		0x2	/* start of text */
#define	ETX		'.'     /* end of text */
#define NEWDOT		0x1c    /* change period to 1c hex */
#define	ENQ		0x5	/* enquiry */
#define	ACK		0x6	/* acknowledge */
#define	LF		0x0a	/* line feed */
#define NEWLF           0x1e    /* the mapped line feed */
#define	NAK		0x16	/* negative acknowledge */
#define	CAN		0x18	/* cancel */
#define CR		0x0d    /* Carriage Return */
#define NEWCR           0x1D    /* New CR */
#define TAB  		0x09    /* tab */
#define NEWTAB		0x1f   /* new tab */
#define	ENDSIZE	8		/* end record size */
#define	FILENAMELEN	12      /* file name length */
#define	DATALEN	64		/* data block length */
#define	BLOCKSIZE	25	/* block buffer length */



/******************************************************************
    G L O B A L

    ENDRECORD: record to end transmission,
    BLOCK, BLOCKCOUNT & BLOCKINDEX: block buffer, count and index,
    BUFFER, LENGTH & INDEX: message buffer, length and index,
    CHECKSUM, CK12 & ETX: checksum and etx,
    FILENAME & PRINTNAME: file names,
    CANCEL: RTE16 request to cancel flag,
    SOH: SOH received, cancel flag also set,
    DEBUG: TRUE to enable debugging message,
    PBLOCKNO: print save block number
******************************************************************/

char	filename[FILENAMELEN+1],printname[FILENAME+1];
char	buffer[DATALEN+8],block[BLOCKSIZE],checksum,ck12,etx;
int	blockindex,blockcount,cancel,length,index,soh,debug,pblockno;
char	endrecord[ENDSIZE] = {STX,'0','0','0','0','C','0',ETX};
char    getrte();



/***************************************************************************
    m a i n

	LOCAL  - TRANSACTION: transaction type,
		 REOPEN: print save interrupted flag,
		 FD: file ptr,
		 I, ITEMP, CTEMP & C: miscellaneous;

    It waits for and processes the following request from RTE16 :

	download
	upsave
	print save
	close
	reopen
	delete

    The communication always starts at RTE16 sending a header record to the
    host.  Host will determine the request and talk back to RTE16.
***************************************************************************/

main ()
{
int	i,itemp,transaction,reopen;
FILE	*fd;
char	*ctemp,c;

    soh = reopen = debug = FALSE;

    /* sign on */
    printf("AMD	AMDHSTCX	V1.0	1/25/83\n");
    printf("---------------------------------------\n\n");

    /* enable debugging ? */
    printf("Want DEBUG option (Y/N) ? ");
    gets(buffer);
    for (ctemp=buffer; (c=(*ctemp++)) && ((c == ' ') || (c == '\t')););
    if ((c == 'Y') || (c == 'y')) debug = TRUE;
    /* debugging */


    while (1) {
	/* wait for SOH */
	if (!soh)
	    do {
		printsi("wait",SOH);
		cancel = FALSE;
	    } while (getrte() != SOH);
	printsi("got",SOH);

	/* clear soh and cancel again */
	cancel = soh = FALSE;

	/* get one message */
	transaction = receive();

	/* check for obvious error */
	if ((length <= FILENAMELEN) && (DOWNLOAD <= transaction) &&
	    (transaction <= DELETE) && (ck12 == checksum) && (etx == ETX))
	
	    /* processing */
	    switch (transaction) {
		case DOWNLOAD:
		    /* save file name */
		    for (i=0; i<length; i++) filename[i] = buffer[i];
		    filename[length] = '\0';
	
		    printss(filename," download\n");
		    if (fd = fopen(filename,"r")) {
			/* acknowledge */
			putrte(ACK);
	
			/* process DOWNLOAD */
			download(fd);
	
			/* close file */
			fclose(fd);
		    }
		    else putrte(NAK);	/* can't open file */

		    break;

		case UPSAVE:
		    /* save file name */
		    for (i=0; i<length; i++) filename[i] = buffer[i];
		    filename[length] = '\0';
	
		    printss(filename," upsave\n");
		    if (fd = fopen(filename,"w")) {
			/* acknowledge */
			putrte(ACK);
	
			/* process UPSAVE */
			itemp = upsave(fd);
	
			/* flush buffer and save file even if cancelled */
			putblock(fd);
			fclose(fd);
		    }
		    else putrte(NAK);	/* can't create file */
	
		    break;

		case PRINTSAVE:
		    /* save file name */
		    for (i=0; i<length; i++) printname[i] = buffer[i];
		    printname[length] = '\0';
	
		    printss(printname," printsave\n");
		    if (fd = fopen(printname,"w")) {
			/* acknowledge */
			putrte(ACK);
	
			/* process PRINTSAVE - initialize pblockno */
			pblockno = 0;
			itemp = printsave(fd);
	
			/* flush buffer and save file even if cancelled */
			putblock(fd);
			fclose(fd);

			/* SOH ? */
			if (itemp == SOH) reopen = TRUE;
		    }
		    else putrte(NAK);	/* can't create file */
	
		    break;

		case CLOSE:
		    printss("close","\n");

		    /* acknowledge */
		    putrte(ACK);
	
		    /* close PRINT file and ignore errors */
		    putblock(fd);
		    fclose(fd);
	
		    break;

		case REOPEN:
		    printss(printname," reopen\n");
		    if (reopen && (fd = fopen(printname,"a"))) {
			/* acknowledge */
			putrte(ACK);
	
			/* process PRINTSAVE - don't initialize pblockno */
			itemp = printsave(fd);
	
			/* flush buffer and save file even if cancelled */
			putblock(fd);
			fclose(fd);
	
			/* SOH ? */
			reopen = FALSE;
			if (itemp == SOH) reopen = TRUE;
		    }
		    else putrte(NAK);	/* can't reopen */
	
		    break;

		case DELETE:
		    printss("delete","\n");

		    /* acknowledge */
		    putrte(ACK);
	
		    /* delete file and ignore errors */
		    buffer[length] = '\0';
		    unlink(buffer);
	    }
	else putrte(NAK);	/* message error */
    }
}



/*************************************
    p r i n t s s

    ENTRY - str1,str2: ptrs to string

    It prints out debugging message.
*************************************/

printss(str1,str2)
char	*str1,*str2;
{
    if (debug) printf("%s%s",str1,str2);
}

/************************************
    p r i n t s i

    ENTRY - str1: ptr to string,
	    sync: sync char

    It prints out debugging message.
************************************/

printsi(str1,sync)
char	*str1,sync;
{
    if (debug) {
	printf("%s",str1);
	switch (sync) {
	    case ACK:	printf(" ACK");  break;
	    case CAN:	printf(" CAN");  break;
	    case ENQ:	printf(" ENQ");  break;
	    case NAK:	printf(" NAK");  break;
	    case SOH:	printf(" SOH");  break;
	    case STX:	printf(" STX");  break;
	}
    }
}

/*******************************************************************************
    g e t r t e

	GLOBAL - CANCEL: cancel flag,
		 CHECKSUM: checksum

	LOCAL  - CTEMP: char

    It reads one byte from RTE16 and accumulates CHECKSUM.

    Whenever CAN or SOH is received, it sets up "cancel" flag to force "getrte"
    to return CAN on future call.

    The purpose of "cancel" flag is to avoid hanging up when AMDHST has received
    unexpected cancel request or start of another header record from RTE16.

    It works by forcing control to flow to certain points where "cancel" is
    checked for.

    "cancel" flag will be cleared after flow of control returns to the top
    level where SOH is expected.

    RETN  - next byte from RTE16
*******************************************************************************/

char getrte ()
{
char	ctemp;

    /* check "cancel" flag */
    if (cancel == TRUE) return (CAN);

    /* compute "checksum" */
    ctemp = getchar();
    checksum = checksum + ctemp;

    /* set up "cancel" and "soh" flag*/
    if (ctemp == CAN) cancel = TRUE;
    if (ctemp == SOH) cancel = soh = TRUE;

    if (debug) printf(".%x",ctemp);

    return (ctemp);
}

/***********************************************************
    g e t r t e 2

	LOCAL  - CTEMP & CTEMP1: chars

    It packs two bytes from RTE16 and accumulates CHECKSUM.

    RETN  - two bytes from RTE16 packed into one
***********************************************************/

getrte2 ()
{
char	ctemp,ctemp1;

    ctemp = getrte();
    ctemp1 = getrte();

    /* convert to upper case if necessary */
    if (('a' <= ctemp) && (ctemp <= 'f')) ctemp = ctemp - 'a' + 'A';
    if (('a' <= ctemp1) && (ctemp1 <= 'f')) ctemp1 = ctemp1 - 'a' + 'A';

    /* convert hex to binary */
    if (ctemp > '9') ctemp = ctemp - '7';
    else ctemp = ctemp - '0';
    if (ctemp1 > '9') ctemp1 = ctemp1 - '7';
    else ctemp1 = ctemp1 - '0';

    return (((ctemp << 4) + ctemp1) & 0xff);
}

/*******************************
    p u t r t e

    ENTRY - BYTE: the byte

    It sends one byte to RTE16.
*******************************/

putrte (byte)
char	byte;
{
    putchar(byte);
}

/******************************************************************
    p u t b u f

    ENTRY - BYTE: the byte

	GLOBAL - BUFFER[] & INDEX: message buffer and index,
		 CHECKSUM: checksum

    It writes one byte into "buffer[index]" and computes CHECKSUM.

    "index" is incremented by 1.
******************************************************************/

putbuf (byte)
char	byte;
{
    /* computes "checksum" */
    buffer[index++] = byte;
    checksum = checksum + byte;
}

/********************************************************************
    p u t b u f 2

    ENTRY - BYTE: the byte

	LOCAL  - TEMP: char

    It unpacks the byte into two, stores them at "buffer[index]" and
    "buffer[index+1]", and computes "checksum".

    "index" is incremented by 2.
********************************************************************/

putbuf2 (byte)
char	byte;
{
char	temp;

    /* masking */
    temp = (byte >> 4) & 0xf;
    byte = byte & 0xf;

    /* convert binary to hex */
    if (temp > 9) putbuf(temp + '7');
    else putbuf(temp + '0');
    if (byte > 9) putbuf(byte + '7');
    else putbuf(byte + '0');
}

/*************************************************************************
    p a c k 2

	GLOBAL - BUFFER[] & INDEX: message buffer and index

	LOCAL  - TEMP & TEMP1: chars

    It packs two bytes at "buffer[index]" and "buffer[index+1]" into one.

    "index" is incremented by 2.

    RETN  - packed byte
*************************************************************************/

char pack2 ()
{
char	temp,temp1;

    temp = buffer[index++];
    temp1 = buffer[index++];

    /* convert hex to binary */
    if (temp > '9') temp = temp - '7';
    else temp = temp - '0';
    if (temp1 > '9') temp1 = temp1 - '7';
    else temp1 = temp1 - '0';

    return ((temp << 4) + temp1);
}

/*******************************************************************************
    g e t b l o c k

    ENTRY - FD: ptr to a file

	GLOBAL - BLOCK[], BLOCKCOUNT & BLOCKINDEX: block buffer, count and index

	LOCAL  - C: char

    It transfer one block from file.

    "blockindex" is set to 0 and "blockcount" is set to number of bytes
    transfered.
*******************************************************************************/

getblock (fd)
FILE	*fd;
{
char	c;
    for (blockcount=0; ((blockcount<BLOCKSIZE) && ((c=getc(fd))!=EOF)) ;) 
	switch (c) {
	    case CR   : block[blockcount++] = NEWCR;
			break;
	    case LF   : block[blockcount++] = NEWLF;
			break;
	    case TAB  : block[blockcount++] = NEWTAB;
			break;
	    case '.'  : block[blockcount++] = NEWDOT;
	     	        break;
	    default   : block[blockcount++] = c;
        }
    blockindex = 0;
}

/*******************************************************************************
    g e t c h

    ENTRY - FD: ptr to a file

	GLOBAL - BLOCK[], BLOCKCOUNT & BLOCKINDEX: block buffer, count and index

    It reads one byte from block buffer.

    Make sure both "blockcount" and "blockindex" are 0 before the first call
    to "getch".

    Or else do a "getblock" before the first call to "getch".

    RETN  - either next byte or EOF
*******************************************************************************/

char getch (fd)
FILE	*fd;
{
    /* check EOF */
    if ((blockcount == 0) && (blockindex == 0)) return (EOF);

    if (blockcount-- == 0) {
	/* get new block when block buffer empty */
	getblock(fd);
	return (getch(fd));
    }
    else return (block[blockindex++]);
}

/*******************************************************************************
    p u t b l o c k

    ENTRY - FD: ptr to a file

	GLOBAL - BLOCK[], BLOCKCOUNT & BLOCKINDEX: block buffer, count and index

    It writes one block to file.

    "blockcount" and "blockindex" are set to 0.
*******************************************************************************/

putblock (fd)
FILE	*fd;
{
    for (blockindex=0; blockindex<blockcount;) putc(block[blockindex++],fd);

    blockcount = blockindex = 0;
}

/*******************************************************************************
    p u t c h 

    ENTRY - FD: ptr to a file,
	    BYTE: the byte

	GLOBAL - BLOCK[], BLOCKCOUNT & BLOCKINDEX: block buffer, count and index
    It writes one byte to block buffer.

    Block buffer is flushed when already full.

    Make sure both "blockcount" and "blockindex" are 0 before the first call
    to "putch".

    Do a "putblock" after the last "putch" call.
*******************************************************************************/

putch (fd,byte)
FILE	*fd;
char	byte;
{
    /* check block buffer full */
    if (blockcount == BLOCKSIZE) putblock(fd);
    blockcount++;
    block[blockindex++] = byte;

    if (debug) printf(",%x",byte);
}


/*****************************************************************************
    r e c e i v e

	GLOBAL - BUFFER[] & LENGTH: message buffer and length,
		 CHECKSUM & CK12: checksum,
		 ETX: etx

	LOCAL  - BN: block number or transaction type,
		 CK: received checksum,
		 I: miscellaneous

    It accepts one message from RTE16 and computes checksum on those bytes
    before the checksum field.

    First two bytes are packed into "length".  Next two bytes are packed into
    "bn" for either transaction or block number.

    The message of length "length" is saved in the "buffer[]".

    Two bytes are then packed into "checksum" and the last byte is saved in
    "etx".

    For UNIX, one needs to ignore the next byte (LF).

    Checksum computation is included in "getrte" and the result is saved in
    "ck12".

    RETN  - block number or transaction type
*****************************************************************************/

receive ()
{
int	i,bn;
char	ck;

    checksum = 0;

    /* get length and block/transaction number */
    length = getrte2();
    bn = getrte2();

    /* get message */
    for (i=0; i<length; i++) buffer[i] = getrte();

    /* get checksum */
    ck12 = checksum & 0xff;	/* computed checksum */
    ck = getrte2();		/* received checksum */

    /* get ETX */
    etx = getrte();

    /* ignore LF - UNIX only */
    getrte();

    /* restore due to that "getrte" changes "checksum" */
    checksum = ck;

    return (bn);
}

/*****************************************************************************
    d o w n l o a d

    ENTRY - FD: ptr to an open file

	GLOBAL - BUFFER[] & INDEX: message buffer and index,
		 CHECKSUM: checksum

	LOCAL  - BLOCKNO: block number,
		 LEAVE: loop control flag,
		 ITEMP, I & CTEMP: miscellaneous

    It sends file (already open) including end record to RTE16.

    After the message is constructed, it waits for ENQ to start transmission.

    It retransmits upon receiving a NAK response.

    Transmission continues until either end of file is reached, CAN or SOH is
    received from RTE16.

    RETN - ACK: file transmission completed,
	   CAN: RTE16 request to cancel or start of new header received
*****************************************************************************/

download (fd)
FILE	*fd;
{
int	itemp,blockno,leave,i,ctemp;

    leave = FALSE;
    blockno = 0;
    buffer[0] = STX;
    getblock(fd);	/* initialize read buffer */

    do {
	/* construct message */
	checksum = 0;  index = 3;
	putbuf2(blockno++);	/* BN1 and BN2 */
	for (i=0; (i<DATALEN) && ((ctemp=getch(fd))!=EOF); i++) putbuf(ctemp);

	/* check end of file */
	if (i) {
	    itemp = index;
	    index = 1;
	    putbuf2(i);		/* L1 and L2 */
	    index = itemp;
	    putbuf2(checksum);	/* CK1 and CK2 */
	    buffer[index] = ETX;
	}
	else {
	    /* copy end record */
	    for (i=0; i<ENDSIZE; i++) buffer[i] = endrecord[i];
	    index = 8;

	    leave = TRUE;
	}

	/* wait for ENQ */
	printsi("wait",ENQ);
	while (((ctemp=getrte()) != ENQ) && (ctemp != CAN)&& (ctemp !=NAK));
	printsi("got",ctemp);
	if (ctemp == CAN) return (CAN);

	/* ignore LF - UNIX only */
	getrte();

	/* transmit */
	do {
	    for (i=0; i<=index; i++) putrte(buffer[i]);

	    /* wait for acknowledgement */
	    printsi("wait",ACK);
	    while (((ctemp=getrte())!=ACK) && (ctemp!=NAK) && (ctemp!=CAN)&& (ctemp !=ENQ));
	    printsi("got",ctemp);
	    if (ctemp == CAN) return (CAN);

	    /* ignore LF - UNIX only */
	    getrte();

	} while (ctemp == NAK);

    } while (leave == FALSE);

    return (ACK);
}

/************************************************************************
    u p s a v e

    ENTRY - FD: ptr to an open file

	GLOBAL - BUFFER[] & LENGTH: message buffer and length,
		 BLOCKCOUNT & BLOCKINDEX: block buffer count and index,
		 CHECKSUM & CK12: checksum,
		 ETX: etx

	LOCAL  - BLOCKNO & BN12: block number,
		 I & CTEMP: miscellaneous

    It receives a file from RTE16.

    It sends out ENQ to RTE16 and waits for the message.

    Reply with ACK to accept the message or NAK to reject it and request
    retransmission.

    Receiving process continues until end record, CAN or SOH is received.

    RETN  - ACK: file received successfully,
	    CAN: RTE16 request to cancel or start of new header received
************************************************************************/

upsave (fd)
FILE *fd;
{
int	bn12,i,blockno;
char	ctemp;

    blockno = blockcount = blockindex = 0;

    do {
	/* request message */
	putrte(ENQ);

	/* wait for STX or CAN */
	printsi("wait",STX);
	while (((ctemp=getrte()) != STX) && (ctemp != CAN));
	printsi("got",ctemp);
	if (ctemp == CAN) return (CAN);

	/* get one message */
	bn12 = receive();

	/* check for obvious errors */
	if (!(((length <= DATALEN)   && (ck12 == checksum) 
	    && (etx == ETX)) || ((length | bn12) == 0))) {
	    /* message error */
	    putrte(NAK);

	    length = 1;
	}
	else if (bn12 == blockno) {
	    /* acknowledge */
		blockno = (blockno + 1) & 0xff;

	    /* write buffer */
		for (i=0; i<length; i++) putch(fd,buffer[i]);
		putrte(ACK);
	    }
	/* check for retransmission of previous block 
	   if so ignore it */
	else if (bn12 == (blockno -1)) {
		putrte(ACK);
	} else {
	/* if it is not previous block then
	   message error */
	   putrte(NAK);
	   printss("in upsave retransmit ","\n");
	   length = 1;
	}
    } while (length || bn12);

    return (ACK);
}

/*************************************************************************
    p r i n t s a v e

    ENTRY - FD: ptr to an open file

	GLOBAL - INDEX & LENGTH: message buffer index and length,
		 BLOCKCOUNT & BLOCKINDEX: block buffer count and index,
		 CHECKSUM & CK12: checksum,
		 ETX: etx,
		 PBLOCKNO: block number

	LOCAL  - BN12: block number,
		 CTEMP: miscellaneous

    It receives a PRINT file from RTE16.

    It sends out ENQ to RTE16 and waits for the message.

    Reply with ACK to accept the message or NAK to reject it and request
    retransmission.

    Receiving process continues until end record, CAN or SOH is received.

    RETN  - ACK: file received successfully,
	    CAN: RTE16 request to cancel or start of new header received
		 unexpectedly,
	    SOH: start of new header received at proper time
*************************************************************************/

printsave (fd)
FILE	*fd;
{
int	bn12;
char	ctemp;

    blockcount = blockindex = 0;

    do {
	/* request message */
	putrte(ENQ);

	/* wait for SOH, STX or CAN */
	printsi("wait",STX);
	while (((ctemp=getrte()) != SOH) && (ctemp != STX) && (ctemp != CAN));
	printsi("got",ctemp);
	if ((ctemp == CAN) || (ctemp == SOH)) return (ctemp);

	/* get one message */
	bn12 = receive();

	/* check for obvious errors */
	if (!((length <= DATALEN)  && (ck12 == checksum) &&
	    (etx == ETX))) {
	    /* message error */
	    putrte(NAK);

	    length = 1;
	}
	else if (pblockno == bn12) {
	    /* acknowledge */
	    putrte(ACK);

	    pblockno = (pblockno + 1) & 0xff;

	    /* convert and write buffer */
	    for (index=0; index<length;) putch(fd,pack2());
	}
	/* check for retransmission of previous block */
	else if (bn12 == (pblockno -1)) {
	     putrte(ACK);
	} else {
	   /* incorrect block */
	   /* message error */
	   putrte(NAK);
	   printsi("in printsave retransmit ","\n");

	   length =1;
	}

    } while (length);

    return (ACK);
}
