/*************************************************************************
*
*
*	Name:  readpc.c
*
*	Description: Read p-code file into memory.
*
*
*	History:
*	Date		By	Comments
*
*	03/04/83	mas
*	04/20/83	mas	Added new function to check header
*				given the fd. This is used by reloadpc.
*				Also added new function to reload the
*				pcode. This is used by PEND, SWAP, and CHAIN.
*	04/21/83	mas	added check for revision number
*	05/09/83	mas	added support for not loading the line number
*				table
*	05/16/83	mas	removed unused automatic variables and made
*				some autos into registers
*	06/14/83	mas	fixed bug in calculation of end of memory
*	6/24/83 	mas	changed to use registers where possible
*	06/27/83	mas	changed pcmem to allocate 0x7fff at a time
*				did the same thing to readuns, and made
*				loadpc lseek over line table if noltab set
*	08/22/83	waf		Efficiency re-write (chksum computation).
*	11/16/83	waf		Implement program overlays.
*	12/15/83	waf		Re-organize for resetGFP() & updGFP().
*
*
*
*  This document contains confidential/proprietary information.
*
*  Copyright (c) 1983, 1984 by Digital Communications Assoc.
*
*************************************************************************
* BB/Xenix Runtime Module */




/*  Notes -

  If the 'no line# table' flag is set in the p-code header, then the line
# table is not loaded into memory.

  The runtime stk starts after the p-code and (if it is loaded) the line #
table.

Overlays -
  The pgm name given can be either a stand-alone (SA) pgm, or the name of an
overlay pgm segment (OV). If it is an OV, the name of the base pgm segment
(BASE) is in the header of the OV, and that code must be read in from the
.bp file.
  When swapping, only the OV code is written to the push file. If the pgm
swapped to uses the same base, the base code is left in memory.  When returning
from a swap, if the base code is not still in memory, it must be re-read in
(from the .bp file).
  In an overlay pgm header (in memory and on disk), the 'codsiz' value is the
size of the overlay code ONLY. The total code size of an overlay is
'codsiz + bcdsiz'.
  In a base pgm header (in memory and on disk), the 'codsiz' value is the 
# of bytes in the total pgm (including the dummy trailer). The 'bcdsiz' is the
# of bytes of code (excluding the dummy trailor). This value is used when
base code is read in, and the overlay code begins loading at bcdsiz+1.

  Note - base pgms can be run 'stand alone'. Any reference to 'external'
line#'s are trapped as errors. Running a base pgm alone is exactly the same 
as running a stand alone pgm (e.g. no base code swapping optimization).
  Note - if the pgm is stand-alone, bcdsiz is guaranteed to be '0'. Thus, the
total code size of NORMPGM & OVLYPGM pgm's is 'codsiz + bcdsiz'.

  The fn's resetGFP() & updGFP() are used to 'disable' the global frame
at certain times. In particular, when code is being loaded in the info
in the global frame may be incorrect. The fn resetGFP() sets the GFP to
point to the 'dummy global frame' DGFP. The info in this dummy frame will
insure that
	> ikey's and errors will not be 'trap'ed and redirected to user code.
	All ikey's and errors are reported.
	> the line# table ptr (LTP) is set to '0', which is recognized as a
	flag by code in lintable.c. When LTP = 0, the line# displayed
	when an error/ikey is reported will be '0000' (LTP is set to '0').

*/




#include "/bb/include/ptype.h"
#include "/bb/include/prevision.h"
#include "/bb/include/pextern.h"
#include "/bb/include/bberms.h"
#include "/bb/include/syerms.h"

#define HDWSTKSIZ 0x2000	/* leave 8k for hardware stack */

extern char *sbrk();


char  *readpc (file)

/* Read a p-code file into memory.

  The header, p-code, and line# table (if no -t) are read in.

  Return -
	Ret val	= addr of 1st p-code.
*/

char	*file;		/* p-code filename */
{
	register int	pfd;
	long	l;
	char	buf[PATHSIZE];

	strcpy(buf,file);

	if ((pfd = openpc(buf)) < 0)
		bbxerrno();		/* file not found */

	/* get header */
	l = lseek(pfd,0L,1);	/* get current position */
	loadhd(pfd, &pcheader) ;	/* read in header */
	updhd(pfd) ;		/* update vars */

	/* set mem ptrs */
	begmem = sbrk(0);	/* set start of p-code */
	pcmem();		/* allocate memory using header */

	/* get p-code */
	lseek(pfd,l,0);		/* point to beginning of header */
	resetGFP() ;		/* set GFP to dummy frame */
	loadpc(pfd,begmem);	/* load in the pcode */
	updGFP() ;			/* update GFP */

	/* don't close program library */
	if (pfd != ust.slibfd && pfd != ust.ulibfd)
		close(pfd);

	/* load name into ust */
	if (buf[0] == '#' || buf[0] == '%')
		strcpy(ust.pname,buf+1);
	else
		strcpy(ust.pname,buf);

	/* set begstk (start addr of runtime stk).
	   If 'noltab' is set, runtime stk starts at end of p-code,
	   else the runtime stk starts at end of line tbl */
	begstk = begmem + pcheader.codsiz + pcheader.bcdsiz ;
	if (pcheader.noltab == FALSE)
		begstk += pcheader.ltabsiz;
#ifdef	dbug
	printf("readpc: begmem,begstk = %u %u\n", begmem,begstk );
#endif

	return(begmem);
	}

static	pcmem ()
{
	register unsigned x;
	register int	j;

	/* do 0x7fff bytes at a time because sbrk wants integers */
	if (pcheader.maxsiz != 0) {
		x = (unsigned)begmem + pcheader.maxsiz;
		x = (x + 0x0fff) & ~0x0fff; /* round to next 4k */
		x -= (unsigned)begmem;
		}
	else
		/* get all memory */
		x = (0xffff - HDWSTKSIZ) - (unsigned)begmem;

	while (x != 0) {
		j = (x > 0x7fff) ? 0x7fff : (int)x;
		if (sbrk(j) == (char *)-1)
			bbxerrno();	/* no memory error */
		x -= (unsigned)j;
		}

	/* save end of memory */
	endmem = sbrk(0); 
	}

		/***  Read header  ***/


readhd (fd)

/* Read in header from 'fd', and update global & ust vars. */

int	fd;
{

	loadhd(fd,&pcheader);
	updhd(fd);
	}



static	checkhd (fd)
int	fd;
{
	struct phead pch;

	loadhd(fd,&pch);	/* do test load of header */
	pcheader = pch;		/* move it to real header */
	updhd(fd);			/* update ust & global vars out of header */
	}



static	loadhd (fd,buf)

/* Load p-code file header into 'buf'.
	Chks 'magic' number validity.
	Chks revision number compatability.
	Chks checksum.
*/

int	fd;
struct phead *buf;
{
	register int	i;
	

	i = read(fd, (char *)buf, sizeof(pcheader));
	if ( i != sizeof(pcheader) )
		bbxerrno();		/* pfile read error */

	/* chk magic number */
	i = buf->magic ;	/* 'magic' number */
	if ( i != NORMPGM && i != OVLYPGM && i != BASEPGM )
		bberr(BENSV);

	/* check revision number in header */
	if ((buf->revision & 0xff00) != (REVISION & 0xff00))
		bberr(BEISF);	/* major numbers not equal */
	if ((buf->revision & 0x00ff) > (REVISION & 0x00ff))
		bberr(BEISF);	/* pgm minor number > pmach minor number */

	/* check chksum */
	 i = hdchksum(buf) ;
	 if ( i != buf->chksum )
		bberr(BENSV) ;

	/* make sure there is something to load */
	if (buf->codsiz == 0)
		bberr(BENSV);
	}



static	updhd (fd)

/* From header, get pgm name, some ust vars, & some global vars.

  File ptr is assumed to be at pgm name entry in header block.
*/

int	fd;
{
	char	name[PATHSIZE];
	register int	i;
	register int	savikey;


	if(read(fd,name,PATHSIZE) < PATHSIZE)
		bbxerrno();
	if (name[0] != '\0')	/* only update name if it is there */
		strcpy(ust.pname,name);

	savikey = ust.stxfl.ikey;
	ust.stmtx.sxflag = pcheader.pcstmtx.sxflag;
	ust.stxfl.ikey = savikey;

	if (pcheader.runonly == TRUE)
		ust.noikey = TRUE;	/* run only - no ikeys */

	pcerr = pcheader.pcerr;
	pclast = pcheader.pclast;
	lastfileno = pcheader.lastfileno;
	lasterrno = pcheader.lasterrno;
	randseed = pcheader.randseed;
	timeleft = pcheader.timeleft;
	for (i = 0; i<4; ++i)
		cram[i] = pcheader.cram[i];

	/* set up local frame ptrs										??? */
	FP.B = pcheader.fp.B;
	BFP  = pcheader.bfp;

	updGFP();		/* set up global frame ptr */
	}



static	updGFP ()

/* Update Global Frame Pointer from pcheader. */

{

	GFP  = pcheader.gfp;
	}

loadpc (fd,pcaddr)

/* Load in p-code from (already opened) p-code file.
  The header has already been read into pcheader.
  The file ptr is at the start of the header.
  The logic is:

  If pgm is overlay pgm (OVLYPGM) {
	(* overlay *)
  	If the base pgm not already resident {
  		Load the base code (from .bp file).
  		Set up base info. (Flag 'base pgm resident').
  		}
  	If base pgm is not compatable with overlay pgm
  		Report error.
  	If base pgm is not 'current' 
  		Set dincbs flag for debugger (invalid line# table).
  	else
  		Clear dincbs flag.
	Increment code ptr to end of base segment.
	}
  else {
	(* stand-alone *)
	If pgm is base pgm
		convert to 'normal' pgm.
  	Clear base info. (Flag 'no base pgm resident').
	}
  Load overlay/stand-alone code.

  Notes -
  A base pgm (BASEPGM) being run 'stand-alone' is treated like a
	stand alone pgm, except that the base info is updated.
  Overlay code must start at begmem+(bcdsiz+1).
*/

int		fd;			/* fd of p-code file */
char	*pcaddr;	/* load addr */
{
	register unsigned codsize;


	/* skip p-code file header */
	lseek(fd, 512L, 1) ;

	/* process file type */
	if ( pcheader.magic == OVLYPGM ) {

		/**  Overlay program  **/

		/* chk for base resident */
		if ( basheader.bcodsiz == 0 ||
				strcmp(pcheader.basnam, basheader.bbasnam) != 0 ) {
			/* load base pgm & set up basheader */
			loadbase(pcaddr, pcheader.basnam);
			}

		/* modify pcaddr (ovly code loading addr) */
		pcaddr += pcheader.bcdsiz ;
		
		/* chk base/ovlay compatability */
		if ( pcheader.icksum != basheader.bicksum
				|| pcheader.bcdsiz != basheader.bbcdsiz
				/* || <data-size equal> */ )						/* ??? */
			bberr(BEISF);

		/* chk for 'current' base pgm */
		if ( pcheader.bcksum == basheader.bbcksum )					/* ??? */
			/* base pgm is current */
			pcheader.dincbs = 0 ;	/* reset Debugger flag */
		else {
			/* base pgm is not current.
			   Set Debugger flag and disable line# table operations */
			pcheader.dincbs = 1 ;	/* set Debugger flag */
			pcheader.noltab = TRUE ;	/* disable line# table ops */
			}
		}

	else {

		/** Stand-alone pgm **/

		/* convert base pgm to a 'normal' pgm */
		if ( pcheader.magic == BASEPGM ) {
			pcheader.bcdsiz = 0 ;
			pcheader.magic = NORMPGM ;
			}

		clearbase();	/* set 'no base resident' */
		}


	/** Load stand-alone/overlay pgm **/
	codsize = pcheader.codsiz ;			/* code size */
	if ( pcheader.noltab == FALSE )
		codsize += pcheader.ltabsiz ;	/* add line tbl size */
#ifdef	dbug
	printf("loadpc: reading main code\n");
#endif
	readuns(fd,pcaddr,codsize);
	if (pcheader.noltab == TRUE)
		lseek(fd,(long)pcheader.ltabsiz,1);		/* skip line# table data */


	/** Load stk's and tables **/
	/* read in eval stk */
	if (pcheader.stksiz != 0) {
#ifdef	dbug
		printf("loadpc: loading eval stack\n");
#endif
		readuns(fd,(char *)BFP,pcheader.stksiz);
		}

	/* read in ust */
	if (pcheader.ustsiz != 0) {
#ifdef	dbug
		printf("loadpc: loading ust\n");
#endif
		readuns(fd,(char *)&ust,pcheader.ustsiz);
		}

#ifdef	dump
	printf(">> Taking dump.\n");
	pmdump(0,0,1) ;
#endif
	}

static	loadbase ( loadaddr, basename )

/* Load base pgm code into memory & set up basheader info.
  
  The base pgm name can have a prefix of '#' or '%'.
  If '%', or no prefix, look for .bp file in current directory.
  If '#' prefix, look for .bp file in system library.
*/

unsigned loadaddr;		/* loading addr */
char	*basename;		/* name of base pgm */
{
	register int	i;
	register int	fd;
	char	c;
	char	*cp;
	struct phead	tmphead;
	long	lpos;
	char	tmpstr[PATHSIZE];


	/* find and open .bp file */
	strcpy(tmpstr, basename) ;	/* make copy of name */
	cp = tmpstr ;		/* -> 1st char of base name */
	if ( *cp == '%' )
		cp++ ;		/* ignore '%' prefix */
	fd = openpc(cp) ;		/* open bp file */
	if ( fd < 0 )
		bbxerrno();
	lpos = lseek(fd, 0L, 1) ;	/* save starting pos */

	/* read base header */
	loadhd(fd, &tmphead) ;
	if ( tmphead.magic != BASEPGM )
		panic();
	lseek(fd, (lpos + 512L), 0) ;		/* pos to code start */

	/* read base code.
	   Note - only bcdsiz bytes are needed. */
#ifdef	dbug
	printf("loadbase: loading base code\n");
#endif
	readuns(fd, loadaddr, tmphead.bcdsiz) ;

	/* close base code file */
	if ( fd != ust.slibfd )		/* don't close system lib */
		close(fd) ;

	/* set up base header info */
	setbashd(&tmphead, basename);
	}

static	setbashd ( hdptr, basename )

/* Update base header info from *hdptr & basename. */

struct phead	*hdptr;		/* ptr to pcode header */
char	*basename;			/* name of base pgm */
{


	strcpy(basheader.bbasnam, basename) ;	/* base pgm name */
	basheader.bcodsiz = hdptr->codsiz ;		/* p-code size */
	basheader.bbcdsiz = hdptr->bcdsiz ;		/* size of base code seg */
	basheader.bicksum = hdptr->icksum ;
	basheader.bbcksum = hdptr->bcksum ;
	}



clearbase ()

/* Clear base header (e.g. flag base as 'non-resident').
*/
{

	basheader.bbasnam[0] = '\0' ;	/* name of base pgm */
	basheader.bcodsiz = 0 ;			/* size of base code */
	}

static	readuns (fd,where,xamount)
int	fd;
char	*where;
unsigned xamount;
{
	register int	j;
	register unsigned amount;

#ifdef	dbug
	printf( "readuns: fd,where,xamount = %d %u %u.\n",fd,where,xamount );
#endif
	amount = xamount;
	
	/* do 0x7fff bytes at a time because read wants integers */
	while (amount != 0) {
		j = (amount > 0x7fff) ? 0x7fff : (int)amount; 
		if (read(fd,where,j) != j)
			bbxerrno();		/* *assume* error (should not find EOF) */
		amount -= (unsigned)j;
		where += j;
		}
	}

reloadpc (fd,lpos)
int	fd;
long	lpos;
{
register unsigned u;

	u = pcheader.maxsiz;	/* save old maxsiz */

	checkhd(fd);	/* verify header */

	/* reallocate memory if necessary */
	if (u != pcheader.maxsiz){
		brk(begmem);
		pcmem();
		}

	lseek(fd,lpos,0);	/* point to header */
	resetGFP() ;		/* set GFP to dummy frame */
	loadpc(fd,begmem);	/* load in the pcode */

	lseek(fd,lpos+(long)sizeof(pcheader),0); /* point to name */
	updhd(fd);		/* read in name */

	/* Pgm code is installed, update GFP */
	updGFP() ;

	/* don't close program library */
	if (fd != ust.slibfd && fd != ust.ulibfd && fd != pushfd)
		close(fd);

	/* set begstk */
	begstk = begmem + pcheader.codsiz ;
	if ( pcheader.magic == OVLYPGM )
		begstk += pcheader.bcdsiz ;		/* add base code size */
	if (pcheader.noltab == FALSE)
		begstk += pcheader.ltabsiz;
#ifdef	dbug
	printf("reloadpc: begmem,begstk = %u %u\n", begmem, begstk );
#endif
	}
