/*	sttyi.c		- set ikey characteristics.

This pgm initializes terminal chars for the following functions -
	intr	(interrupt)
	quit	(int with core dump)
These are characters which can not be set with 'stty'.

Values can be specified for either or both intr & quit, or the current
values can be displayed.
This utility is designed to emulate the operation of the 'stty' utility.

Note that the most common usage will be
	sttyi dg
which sets the intr & quit keys for the DG terminals.



Usage :
*/
#define	USAGE \
	"sttyi [dg[6053]] [vt100] [d0] [intr <ikey_val>] [quit <quit_val>]"
/*

	intr	- the following char is used for the interrupt char.
	quit	- the following char is used for the quit char.
			('char's are in the same format as for 'stty'. e.g. '^X'
			signifies 'control X').
	dg		- set intr & quit keys for the DG terminals (using values
			#define'ed below).
	dg6053	- same as 'dg'.
	vt,vt100 - (currently) same as 'dg'.

	If no args are given, the current values are displayed.

	Processing of args is left to right, e.g.
		sttyi - intr '^A' 
	will set intrc & quitc to the default values, then set intrc to /001, but
		sttyi intr '^A' -
	will set intrc & quitc to the default values.

*/

#include <sgtty.h>

/* set 'verbose' for diagnostics */
#define		verbose		0


/* define default values for DG terminals */
#define	DG_INTRC	'\033'		/* Int		= ESC */
#define	DG_QUITC	'\003'		/* quit		= ^C  */



int		intrc, quitc ;
struct	sgttyb argp;
struct	tchars targ;

main (argc, argv)

int		argc;
char	**argv;
{
	register char	*cptr;
	int		argptr, argcnt;


	/** get current value of intrc & quitc keys **/
	ioctl(0,TIOCGETC,&targ);
	intrc = targ.t_intrc ;
	quitc = targ.t_quitc ;


	/** chk args **/
	if ( argc == 1 ) {

		/* no args - show current chars */
		showtc() ;
		}

	else {

		/* process args */
		argptr = 1 ;		/* 1st arg */
		argcnt = --argc ;	/* # of args */
		while ( argcnt ) {

			/* process next arg */
			cptr = argv[argptr] ;
			if ( strcmp(cptr, "dg") == 0
			  || strcmp(cptr, "dg6053") == 0
			  || strcmp(cptr, "vt100") == 0
			  || strcmp(cptr, "d0") == 0 ) {

				/* set up for DG term */
				intrc = DG_INTRC ;
				quitc = DG_QUITC ;
				if ( verbose )
					puts(">>> Using DG chars") ;
				argptr++ ;
				argcnt-- ;
				}

			else if ( strcmp(cptr, "intr") == 0 ) {

				/* get intr char */
				intrc = getcval(argv[argptr+1]) ;
				argptr += 2 ;
				argcnt -= 2 ;
				}

			else if ( strcmp(cptr, "quit") == 0 ) {

				/* get quit char */
				quitc = getcval(argv[argptr+1]) ;
				argptr += 2 ;
				argcnt -= 2 ;
				}

			else

				/* bad argument */
				usage() ;

			if ( argcnt < 0 )
				usage() ;		/* bad args */
			}

		settc() ;		/* set chars */
		}

	}

settc ()

/* Set terminal characteristics, using values defined by the cmd args */
{


		if ( verbose )
			printf(">>> setting - i,q = %x %x\n", intrc,quitc);
		targ.t_intrc = intrc ;
		targ.t_quitc = quitc ;
		ioctl(0,TIOCSETC,&targ);

		}



showtc ()

/* Show current characteristics */
{


		printf("Intr = '") ;
		showchar(intrc) ;
		printf("' ; Quit = '") ;
		showchar(quitc) ;
		printf("'\n") ;

		}

showchar ( c )

/* Show the character, using control format (^X) if necessary. */

char	c;
{


	/* chk for hi bit set */
	if ( c & 0x80 ) {
		showchar(c & 0x7f) ;
		return ;
		}

	/* chk for control char */
	if ( c < 0x20 ) {
		putchar('^') ;
		putchar(c+0x40) ;
		return ;
		}

	/* show normal char */
	putchar(c) ;

	}

getcval ( ptr )

/* Given arg string ptr, return value of the char.
   Control chars are allowed in '^X' format.
*/

char	*ptr;
{
	int		cntrl;
	int		c;


	/* chk for '^' */
	cntrl = 0 ;		/* assume no '^' */
	if ( *ptr == '^' ) {
		cntrl = 1 ;
		ptr++ ;
		}

	if ( strlen(ptr) != 1 )
		usage() ;
	c = *ptr ;
	if ( cntrl )
		c -= 0x40 ;

	return(c) ;
	}

usage ()

/* Show usage & exit */
{


	printf("Usage: %s\n", USAGE) ;
	exit(-1) ;

	}
