/************************************************************************
 *  CtrlProp.cpp -- Utilities for getting information about Visual Basic
 *  controls' properties
 ************************************************************************/

#include <windows.h>
#include <vbapi.h>
#include "CtrlProp.h"

// StandardPropery -- holds information for a 'PPROPINFO_STD_' property
typedef struct
{
   PPROPINFO FakeAddress;
   PROPINFO  PropInfo;
} StandardProperty;

StandardProperty StandardProperties[] =
{
   PPROPINFO_STD_NAME,              {"Name",          0},
   PPROPINFO_STD_INDEX,             {"Index",         DT_SHORT},
   PPROPINFO_STD_HWND,              {"Hwnd",          DT_SHORT},
   PPROPINFO_STD_BACKCOLOR,         {"BackColor",     DT_COLOR},
   PPROPINFO_STD_FORECOLOR,         {"ForeColor",     DT_COLOR},
   PPROPINFO_STD_LEFT,              {"Left",          DT_XPOS},
   PPROPINFO_STD_TOP,               {"Top",           DT_YPOS},
   PPROPINFO_STD_WIDTH,             {"Width",         DT_XSIZE},
   PPROPINFO_STD_HEIGHT,            {"Height",        DT_YSIZE},
   PPROPINFO_STD_ENABLED,           {"Enabled",       DT_BOOL},
   PPROPINFO_STD_VISIBLE,           {"Visible",       DT_BOOL},
   PPROPINFO_STD_MOUSEPOINTER,      {"MousePointer",  DT_ENUM},
   PPROPINFO_STD_CAPTION,           {"Caption",       DT_HSZ},
   PPROPINFO_STD_FONTNAME,          {"FontName",      DT_HSZ},
   PPROPINFO_STD_FONTBOLD,          {"FontBold",      DT_BOOL},
   PPROPINFO_STD_FONTITALIC,        {"FontItalic",    DT_BOOL},
   PPROPINFO_STD_FONTSTRIKE,        {"FontStrikeThru",DT_BOOL},
   PPROPINFO_STD_FONTUNDER,         {"FontUnderline", DT_BOOL},
   PPROPINFO_STD_FONTSIZE,          {"FontSize",      DT_REAL},
   PPROPINFO_STD_TABINDEX,          {"TabIndex",      DT_SHORT},
   PPROPINFO_STD_PARENT,            {"Parent",        DT_LONG},
   PPROPINFO_STD_DRAGMODE,          {"DragMode",      DT_ENUM},
   PPROPINFO_STD_DRAGICON,          {"DragIcon",      DT_SHORT},
   PPROPINFO_STD_BORDERSTYLEOFF,    {"BorderStyleOff",DT_ENUM},
   PPROPINFO_STD_TABSTOP,           {"TabStop",       DT_BOOL},
   PPROPINFO_STD_TAG,               {"Tag",           DT_HSZ},
   PPROPINFO_STD_TEXT,              {"Text",          DT_HSZ},
   PPROPINFO_STD_BORDERSTYLEON,     {"BorderStyleOn", DT_ENUM},
   PPROPINFO_STD_CLIPCONTROLS,      {"ClipControls",  DT_BOOL},
   PPROPINFO_STD_NONE,              {"None",          0},
   PPROPINFO_STD_HELPCONTEXTID,     {"HelpContextID", DT_SHORT},
   PPROPINFO_STD_LINKMODE,          {"LinkMode",      DT_ENUM},
   PPROPINFO_STD_LINKITEM,          {"LinkItem",      DT_HSZ},
   PPROPINFO_STD_LINKTOPIC,         {"LinkTopic",     DT_HSZ},
   PPROPINFO_STD_LINKTIMEOUT,       {"LinkTimeout",   DT_SHORT},
   PPROPINFO_STD_LEFTNORUN,         {"LeftNoRun",     DT_XPOS},
   PPROPINFO_STD_TOPNORUN,          {"TopNoRun",      DT_YPOS},
   PPROPINFO_STD_ALIGN,             {"Align",         DT_ENUM},
   PPROPINFO_STD_IMEMODE,           {"ImeMode",       DT_BOOL},
   PPROPINFO_STD_DATASOURCE,        {"DataSource",    0},
   PPROPINFO_STD_DATAFIELD,         {"DataField",     DT_HSZ},
   PPROPINFO_STD_DATACHANGED,       {"DataChanged",   DT_BOOL},
   NULL,                            {"",              0},
};


PROPINFO UnknownStdProp = 
{"UNKNOWN_STD",       0};
  


/*
 *  Function:     GetPropertyArray()
 *
 *  Description:  Gets property array for a Visual Basic control
 *
 *  Parameters:   hctl -- handle to control
 *
 *  Returns:      far pointer to property array
 */
 
PPROPINFO FAR * GetPropertyArray (HCTL hctl)
{
   PPROPINFO FAR *lppPropInfo = NULL;
   MODEL FAR *lpModel = VBGetControlModel(hctl);
   if (lpModel)
      lppPropInfo = (PPROPINFO FAR*)
         MAKELONG(lpModel->npproplist, (_segment)lpModel);

   return lppPropInfo;
}

/*
 *  Function:     GetProperty()
 *
 *  Description:  gets specified property of a Visual Basic control
 *
 *  Parameters:   PropertyArray -- array returned by GetPropertyArray()
 *                index         -- index of control, from 0
 *
 *  Returns:      far pointer to a PROPINFO structure;  for a standard
 *                property, this will be a pointer into StandardProperties[]
 */

LPPROPINFO GetProperty (PPROPINFO FAR * PropertyArray, short index)
{
   PPROPINFO FAR *lppPropInfo = PropertyArray + index; // offset address
   
   if (*lppPropInfo == NULL)
      return NULL;
   if (IsStandardProperty(*lppPropInfo))
   {
      // 'standard' property pointers are not real addresses;
      // need to search StandardProperties[] list
      StandardProperty *pStdProp;
      for (pStdProp = StandardProperties; pStdProp->FakeAddress; pStdProp++)
      {
	      if (*lppPropInfo == pStdProp->FakeAddress)
	         return &(pStdProp->PropInfo);
      }
      return &UnknownStdProp; // standard property not found
   }
   else
   {
      return (LPPROPINFO)MAKELONG(*lppPropInfo, (_segment)PropertyArray);
   }
}

/*
 *  Function:     IsStandardProperty()
 *
 *  Description:  determines whether a property is a standard property,
 *                as defined in VBAPI.H
 *
 *  Parameters:   lpPropInfo -- pointer to PROPINFO structure
 *
 *  Returns:      TRUE if property is standard, FALSE otherwise
 */

BOOL IsStandardProperty (LPPROPINFO lpPropInfo)
{
   StandardProperty *pStdProp;
   NPPROPINFO pPropInfo = (NPPROPINFO)LOWORD(lpPropInfo); // near pointer
   
   if (pPropInfo >= PPROPINFO_STD_LAST)
      return TRUE;    // pPropInfo is a 'fake' address in VBAPI.H
      
   for (pStdProp = StandardProperties; pStdProp->FakeAddress; pStdProp++)
   {
	   if (pPropInfo == &(pStdProp->PropInfo))
	      return TRUE; // pPropInfo is contained in StandardProperties[]
   }   
      
   if (pPropInfo == &UnknownStdProp)
      return TRUE;    // pPropInfo is an 'unknown' standard property

   return FALSE;      // property is not standard
}   
      

