/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	debugger;
include	hardware;
include	file;
include	filesys;
include	disasm;
include	symtab;
include	alys;
include	error;
include	trace;

main:	entry	() =
	{
	core:	ref char;
	map:	[:] char;
	mbuf:	[MAXPATH] char;
	runf:	[:] char;
	rbuf:	[MAXPATH] char;
	buf:	[256] char;
	line:	[:] char;
	cp:	ref char;
	sym:	ref symbol;
	file:	[:] char;
	base:	[:] char;

//	catchSignal(SigBreak, &breakKey);
	if	(ArgumentCount < 1 ||
		 ArgumentCount > 2){
		printf("Use is: DEBUG run_file [ core_file ]\n");
		return;
		}
	file = getNextArgument();
	base = stripExtension(file);
	map = makePath(mbuf, "", base, ".map");
	runf = makePath(rbuf, "", base, ".run");
	loadMap(map);
	loadRun(runf);
	for	(;;){
		i:	int;
		x:	unsigned;

		printf("> ");
		i = stdin gets(buf);
		if	(i == 0)
			break;
		if	(i >= |buf)
			i = |buf - 1;
		buf[i] = 0;
		for	(cp = buf; isspace(*cp); cp++)
			;
		if	(*cp == 'q')
			break;
		switch	(*cp){
		case	0:	break;		// noop

		case	'?':
			helpScreen();
			break;

		case	's':
			if	(!Debugging)
				printf("Not debugging\n");
			else
				trace(&DB);
			break;

		case	'f':
			if	(!Debugging)
				printf("Not debugging\n");
			else	{
				ref process(DB.pid) flush();
				Debugging = FALSE;
				}
			break;

		case	'b':
			if	(!Debugging){
				printf("Not debugging\n");
				break;
				}
			cp++;
			while	(isspace(*cp))
				cp++;
			if	(*cp == 0){
				BreakPoints listAll();
				break;
				}
			sym = translateAddress(cp, CodeSymbols, -1);
			if	(LastAddress == INV_ADDRESS){
				printf("Invalid address\n");
				break;
				}
			setBreakPoint(LastAddress);
			break;

		case	'c':
			if	(!Debugging){
				printf("Not debugging\n");
				break;
				}
			cp++;
			while	(isspace(*cp))
				cp++;
			if	(*cp == 0)
				BreakPoints clearAll();
			else	{
				sym = translateAddress(cp, CodeSymbols, -1);
				if	(LastAddress == INV_ADDRESS){
					printf("Invalid address\n");
					break;
					}
				clearBreakPoint(LastAddress);
				}
			break;

		case	'd':
			if	(!Debugging){
				printf("Not debugging\n");
				break;
				}
			sym = translateAddress(cp + 1, DataSymbols, DataAddress);
			if	(LastAddress == INV_ADDRESS){
				printf("Invalid address\n");
				break;
				}
			DataAddress = LastAddress;
			if	(sym &&
				 LastAddress < RunHeader->dataTotalLen){
				printf("Displaying ");
				sym display(LastAddress);
				printf("\n");
				}
			dumpData(DataAddress, 320);
			DataAddress += 320;
			break;

		case	'g':
			if	(!Debugging){
				printf("Not debugging\n");
				break;
				}
			cp++;
			while	(isspace(*cp))
				cp++;
			if	(*cp == 0){
				x = DB.eip;
				BreakPoints disable(x);
				singleStep(DB.pid, &DB);
				BreakPoints enable(x);
				run(DB.pid, &DB);
				CodeAddress = DB.eip;
				displayState();
				break;
				}
			sym = translateAddress(cp, CodeSymbols, -1);
			if	(LastAddress == INV_ADDRESS){
				printf("Invalid address\n");
				break;
				}
			x = DB.eip;
			BreakPoints disable(x);
			singleStep(DB.pid, &DB);
			BreakPoints enable(x);
			runTo(LastAddress);
			CodeAddress = DB.eip;
			displayState();
			break;

		case	'r':
			displayState();
			break;

		case	't':
			if	(!Debugging)
				printf("Not debugging\n");
			else	{
				x = DB.eip;
				BreakPoints disable(x);
				singleStep(DB.pid, &DB);
				CodeAddress = DB.eip;
				BreakPoints enable(x);
				displayState();
				}
			break;

		case	'x':
			if	(Debugging){
				printf("Already running\n");
				break;
				}
			for	(cp++; isspace(*cp); cp++)
				;
			i = cp - buf;
			processCommand(runf, cp[:|buf - i]);
			if	(Debugging)
				displayState();
			break;

		case	'u':
			sym = translateAddress(cp + 1, CodeSymbols, CodeAddress);
			if	(LastAddress == INV_ADDRESS){
				printf("Invalid address\n");
				break;
				}
			CodeAddress = LastAddress;
			if	(sym){
				printf("Disassembling ");
				sym display(LastAddress);
				printf("\n");
				}
			for	(i = 0; i < 20; i++)
				CodeAddress = disassemble(CodeAddress, CodeAddress);
			break;

		default:
			printf("Bad command: %S", cp);
			helpScreen();
			}
		}
	}

translateAddress:	(cp: ref char, syms: ref symbol, last: unsigned) ref symbol =
	{
	sym:	ref symbol;

	LastAddress = INV_ADDRESS;
	while	(isspace(*cp))
		cp++;
	if	(isalpha(*cp)){
		nm:	[:] char;
		x:	unsigned;

		nm = cp[:1];
		while	(isalnum(*cp) ||
			 *cp == '_' ||
			 *cp == '.' ||
			 *cp == ':')
			cp++;
		if	(*cp == '+')
			x = xtou(cp + 1);
		else
			x = 0;
		|nm = cp - nm;
		sym = syms findName(nm);
		if	(sym){
			LastAddress = sym->address + x;
			return sym;
			}
		else
			return 0;
		}
	else if	(*cp == '$')
		return Start;
	else if	(isdigit(*cp))
		LastAddress = xtou(cp);
	else if	(*cp != 0)
		return 0;
	else if	(last == unsigned(-1))
		return 0;
	else
		LastAddress = last;
	return syms findSymbol(LastAddress);
	}

xtou:	(cp: ref char) unsigned =
	{
	x:	long;

	x = 0;
	while	(isxdigit(*cp)){
		x <<= 4;
		if	(isdigit(*cp))
			x += *cp - '0';
		else
			x += tolower(*cp) - 'a' + 10;
		cp++;
		}
	return x;
	}

LastAddress:	unsigned;
CodeAddress:	unsigned;
DataAddress:	unsigned;

INV_ADDRESS:	unsigned = -1;

helpScreen:	() =
	{
	printf(	"?		Display this screen\n"
		"b		List breakpoints\n"
		"b addr		Set breakpoint at addr\n"
		"c		CLear all breakpoints\n"
		"c addr		Clear breakpoint at addr\n"
		"d [ addr ]	Display data at addr\n"
		"f		Flush test\n"
		"g [ addr ]	Go to addr\n"
		"q		Quit\n"
		"r		Display registers\n"
		"s		Stack trace\n"
		"t		Single step\n"
		"u [ addr ]	Unassemble at addr\n"
		"x args		Execute program\n"
		"\n"
		"args are a sequence of command line arguments including\n"
		"redirection."
		"addr is a machine address, either in code or data space\n"
		"	if the address begin with a letter, it is a symbol\n"
		"	name.  A $ represents the program start address.\n"
		"	A string beginning with a digit is a hex value.\n"
		);
	}

displayState:	() =
	{
	printf("%s at ", StateMessages[DB.state]);
	sym:	ref symbol;

	sym = CodeSymbols findSymbol(DB.eip);
	if	(sym){
		sym display(DB.eip);
		printf("\n");
		}
	else
		printf("%x\n", DB.eip);
	if	(DB.state == DEB_DEAD)
		printf("exit code %d\n", DB.errorCode);
	else if	(DB.state == DEB_TRAP){
		if	(DB.signal == 6 &&
			 DB.eip < RunHeader->codeLen - 1){
			cp:	ref byte;

			cp = &Code[DB.eip];
			if	(*cp == 0x8d &&
				 cp[1] == 0xc3)
				DB.signal = 17;
			}
		printf("%s trap  error code = %x\n", 
			TrapNames[DB.signal],
			DB.errorCode);
		}
	else if	(DB.state == DEB_SIG)
		printf("signal %d\n", DB.signal);
	printf("  eax %08x ebx %08x ecx %08x edx %08x", DB.eax,
			DB.ebx,
			DB.ecx,
			DB.edx);
	printf("   ds %04x  es %04x\n",
			DB.ds & 0xFFFF,
			DB.es & 0xFFFF);
	printf("  ebp %08x esi %08x edi %08x esp %08x", 
			DB.ebp,
			DB.esi,
			DB.edi,
			DB.esp);
	printf("   fs %04x  gs %04x\n",
			DB.fs & 0xFFFF,
			DB.gs & 0xFFFF);
	printf("               ");
	if	(DB.eflags & 1)
		printf("CY ");
	else
		printf("NC ");
	if	(DB.eflags & 4)
		printf("PA ");
	else
		printf("NP ");
	if	(DB.eflags & 0x10)
		printf("AC ");
	else
		printf("NA ");
	if	(DB.eflags & 0x40)
		printf("ZF ");
	else
		printf("NZ ");
	if	(DB.eflags & 0x80)
		printf("SG ");
	else
		printf("NS ");
	if	(DB.eflags & 0x100)
		printf("TF ");
	else
		printf("NT ");
	if	(DB.eflags & 0x200)
		printf("IE ");
	else
		printf("NI ");
	if	(DB.eflags & 0x400)
		printf("DN ");
	else
		printf("UP ");
	if	(DB.eflags & 0x800)
		printf("OV ");
	else
		printf("NO ");
	if	(DB.eflags & 0x4000)
		printf("NT ");
	if	(DB.eflags & 0x10000)
		printf("RF ");
	if	(DB.eflags & 0x20000)
		printf("VM ");
	printf("iopl = %x\n", DB.eflags >> 12 & 3);
	disassemble(DB.eip, DB.eip);
	CodeAddress = DB.eip;
	}

StateMessages:	[] * char = [
	"- step -",				// DEB_STEP
	"New",					// DEB_NEW
	"Program finished",			// DEB_DEAD
	"- breakpoint -",			// DEB_BKPT
	"Signal raised",			// DEB_SIG
	"Trapped",				// DEB_TRAP
	];

TrapNames:	[] * char = [
	"Divide by zero",			// 00
	"Unexpected single step",		// 01
	"Trap 2",				// 02
	"Unexpected breakpoint",		// 03
	"Overflow",				// 04
	"Bounds exception",			// 05
	"Invalid opcode",			// 06
	"FP coprocessor not present",		// 07
	"Double fault",				// 08
	"Coprocessor segment overrun",		// 09
	"Invalid TSS",				// 0A
	"Segment not present",			// 0B
	"Stack segment",			// 0C
	"General protection",			// 0D
	"Page fault",				// 0E
	"Unknown",				// 0F
	"Math Overrun",				// 10
	"Compile time error",			// 11 - special
	];

Debugging:	boolean = FALSE;
DB:		task_t;
Target:		ref far debugger;

processCommand:	(runfile: [:] char, args: [:] char) =
	{
	spawnCommand(runfile, args);
	}

spawnCommand:	(cmd: [:] char, arg: [:] char) =
	{
	infile:		[:] char;
	outfile:	[:] char;
	appendout:	char;
	i:		int;
	cp:		ref char;
	c:		char;

	cmdClear();
	infile = "";
	outfile = "";
	appendout = 0;
	i = 0;
	c = cmd[i++];
	while	(isspace(c))
		c = cmd[i++];
	if	(c == 0)
		return;
	while	(!isspace(c) && c != 0){
		cdeposit(c);
		c = cmd[i++];
		}
	while	(isspace(c))
		c = cmd[i++];
	for	(;;){
		if	(c == 0)
			break;
		else if	(isspace(c)){
			c = cmd[i++];
			continue;
			}
		else if	(c == '"'){
			c = cmd[i++];
			while	(c != 0 && c != '"'){
				adeposit(c);
				c = cmd[i++];
				}
			adeposit(0);
			if	(c != '"')
				break;
			c = cmd[i++];
			}
		else if	(c == '<'){
			c = cmd[i++];
			while	(isspace(c))
				c = cmd[i++];
			cp = RedirIn;
			while	(c != 0 &&
				 !isspace(c)){
				*cp++ = c;
				c = cmd[i++];
				}
			infile = RedirIn[:cp - RedirIn];
			if	(c == 0)
				break;
			}
		else if	(c == '>'){
			c = cmd[i++];
			if	(c == '>'){
				appendout = 1;
				c = cmd[i++];		// >> is append
				}
			while	(isspace(c))
				c = cmd[i++];
			cp = RedirOut;
			while	(c != 0 &&
				 !isspace(c)){
				*cp++ = c;
				c = cmd[i++];
				}
			outfile = RedirOut[:cp - RedirOut];
			if	(c == 0)
				break;
			}
		else	{
			do	{
				adeposit(c);
				c = cmd[i++];
				}
				while	(c != 0 &&
					 !isspace(c));
			adeposit(0);
			if	(c == 0)
				break;
			c = cmd[i++];
			}
		}
	p:	ref far external;
	fcmd:		[:] char;
	lookup:	static	pathList;

	try
		lookup useEnvironment("PATH");
	fcmd = lookup search(cstring(), ".run", ".com", ".exe", 0);
	if	(|fcmd == 0){
		printf("Unknown command: %S\n", cstring());
		return;
		}
	if	(|infile ||
		 |outfile)
		forkArena();

	fd:	ref far channel;

	if	(|infile){
		fd = FileSystem open(infile, AR_READ);
		if	(fd == 0){
			printf("Couldn't open '%S'\n", infile);
			return;
			}
		makeLocal(fd, ref far channel(0));
		}
	if	(|outfile){
		if	(appendout){
			fd = FileSystem open(outfile, AR_WRITE);
			fd seek(0, SEEK_END);
			}
		else
			fd = FileSystem create(outfile, FA_WRITE|FA_READ);
		if	(fd == 0){
			printf("Couldn't create '%S'\n", outfile);
			return;
			}
		makeLocal(fd, ref far channel(1));
		}
	try	{
		args:	[:] char;

		args = astring();
		p = Host spawn(fcmd, args);
		if	(|infile ||
			 |outfile)
			discardArena();
		if	(p)
			Debugger wait(&external.childExit);
		}
	except	{
		printf("Spawn failed\n");
		if	(|infile ||
			 |outfile)
			discardArena();
		}
	}

cmdClear:	() =
	{
	ArgPtr = Arguments;
	CmdPtr = Command;
	}

cdeposit:	(c: char) =
	{
	if	(CmdPtr < Command + |Command)
		*CmdPtr++ = c;
	}

cstring:	() [:] char =
	{
	return Command[:CmdPtr - Command];
	}

adeposit:	(c: char) =
	{
	if	(ArgPtr < Arguments + |Arguments)
		*ArgPtr++ = c;
	}

astring:	() [:] char =
	{
	return Arguments[:ArgPtr - Arguments];
	}

Arguments:	[ARGS_LEN] char;
ArgPtr:		ref char;
Command:	[MAXPATH] char;
CmdPtr:		ref char;
RedirIn:	[MAXPATH] char;
RedirOut:	[MAXPATH] char;

dumpData: public	(addr: unsigned, len: int) =
	{
	data:	[16] byte;
	cp:	ref byte;
	i:	int;
	j:	int;
	line:	unsigned;

	line = addr & ~0xf;
	DB.dataFD seek(line, SEEK_ABS);
	len += addr - line;
	while	(len){
		cp = data;
		i = DB.dataFD read(data);
		if	(i < 16)
			len = i;
		if	(len == 0)
			break;
		printf("%08x: ", line);
		i = 8;
		j = 0;
		while	(i && len){
			if	(line + j >= addr)
				printf("%02x ", cp[8 - i]);
			else
				printf("   ");
			i--;
			j++;
			len--;
			}
		while	(i){
			printf("   ");
			i--;
			}
		i = 8;
		printf(" ");
		while	(i && len){
			if	(line + j >= addr)
				printf("%02x ", cp[16 - i]);
			else
				printf("   ");
			i--;
			j++;
			len--;
			}
		while	(i){
			printf("   ");
			i--;
			}
		printf("  ");
		for	(i = 0; i < j; i++){
			if	(line + i >= addr){
				if	(isprint(cp[i]))
					printf("%c", cp[i]);
				else
					stdout putc('.');
				}
			else
				stdout putc(' ');
			}
		line += 16;
		printf("\n");
		}
	}

breakPoint:	type	{
	public:

	next:		* breakPoint;
	location:	unsigned;

create:	factory	(loc: unsigned) ref breakPoint =
	{
	return new breakPoint[ 0, loc ];
	}

listAll:	() =
	{
	while	(self){
		sym:	ref symbol;

		sym = CodeSymbols findSymbol(location);
		printf("Break at ");
		if	(sym == 0)
			printf("%08x", location);
		else
			sym display(location);
		printf("\n");
		self = next;
		}
	}

defined:	(loc: unsigned) boolean =
	{
	while	(self){
		if	(loc == location)
			return TRUE;
		self = next;
		}
	return FALSE;
	}

disable:	(loc: unsigned) =
	{
	while	(self){
		if	(loc == location){
			DB.codeFD exchangeByte(loc, Code[loc]);
			return;
			}
		self = next;
		}
	}

enable:	(loc: unsigned) =
	{
	while	(self){
		if	(loc == location){
			DB.codeFD exchangeByte(loc, 0xcc);
			return;
			}
		self = next;
		}
	}

delete:	(loc: unsigned) =
	{
	if	(BreakPoints == 0)
		return;
	self = BreakPoints;
	if	(location == loc){
		BreakPoints = next;
		free(self);
		return;
		}
	while	(next){
		if	(loc == next->location){
			xb:	ref breakPoint;

			xb = next;
			DB.codeFD exchangeByte(xb->location, 
							Code[xb->location]);
			next = xb->next;
			free(xb);
			return;
			}
		self = next;
		}
	}

clearAll:	() =
	{
	b:	ref breakPoint;

	while	(self){
		b = next;
		DB.codeFD exchangeByte(location, Code[location]);
		free(self);
		self = b;
		}
	}

	};

BreakPoints:	ref breakPoint;

setBreakPoint:	(addr: unsigned) =
	{
	printf("Break point at %x\n", addr);
	if	(isBreakPoint(addr)){
		printf("Already set\n");
		return;
		}
	b:	ref breakPoint;

	b = breakPoint create(addr);
	b->next = BreakPoints;
	BreakPoints = b;
	DB.codeFD exchangeByte(addr, 0xcc);
	}

clearBreakPoint:	(addr: unsigned) =
	{
	if	(!isBreakPoint(addr)){
		printf("Not set\n");
		return;
		}
	BreakPoints delete(addr);
	DB.codeFD exchangeByte(addr, Code[addr]);
	}

runTo:	(addr: unsigned) =
	{
	if	(addr >= RunHeader->codeLen){
		printf("Cannot run to that address\n");
		return;
		}
	if	(isBreakPoint(addr)){
//		run(DB.pid, &DB);
		return;
		}
	DB.codeFD exchangeByte(addr, 0xcc);
//	run(DB.pid, &DB);
	DB.codeFD exchangeByte(addr, Code[addr]);
	}

isBreakPoint:	public	(addr: unsigned) boolean =
	{
	return BreakPoints defined(addr);
	}

header:	type	{
	public:

	size:	unsigned;
	next:	ref header;
	};

heapr:	type	{
	public:

	x:		pointer;
	base:		header;
	allocp:		* header;
	};

traceHeap:	() =
	{
	xp:	ref heapr;

	xp = pointer(&Heap);
	printf("base @%x = [ %x, %x ]\n", xp, xp->base.size, xp->base.next);
	printf("allocp = %x _brklvl = %x\n\n", xp->allocp, _brklvl);
	h:	ref header;

	for	(h = xp->allocp; ; h = h->next){
		printf("%8x: [ %x, %x ]\n", h, h->size, h->next);
		if	(h->next == xp->allocp)
			break;
		}
	}

dumpXData: public	(addr: unsigned, len: int) =
	{
	cp:	ref byte;
	i:	int;
	j:	int;
	line:	unsigned;

	line = addr & ~0xf;
	len += addr - line;
	cp = pointer(line);
	while	(len){
		printf("%08x: ", line);
		i = 8;
		j = 0;
		while	(i && len){
			if	(line + j >= addr)
				printf("%02x ", cp[8 - i]);
			else
				printf("   ");
			i--;
			j++;
			len--;
			}
		while	(i){
			printf("   ");
			i--;
			}
		i = 8;
		printf(" ");
		while	(i && len){
			if	(line + j >= addr)
				printf("%02x ", cp[16 - i]);
			else
				printf("   ");
			i--;
			j++;
			len--;
			}
		while	(i){
			printf("   ");
			i--;
			}
		printf("  ");
		for	(i = 0; i < j; i++){
			if	(line + i >= addr){
				if	(isprint(cp[i]))
					printf("%c", cp[i]);
				else
					stdout putc('.');
				}
			else
				stdout putc(' ');
			}
		line += 16;
		cp += 16;
		printf("\n");
		}
	}
/*
breakKey:	interrupt (ifr: interruptFrame) =
	{
	catchSignal(SigBreak, &breakKey);
	setSignalThreshold(ifr.threshold);
	}
 */

Debugger:	inherit external { public:

attention:	gate!	() = 
	{ 
	}
/*
rejectedMessage:	gate!	(code: int) =
	{
	e:	ref __exceptionFrame;

	printf("Rejected message: %d _Except = %x\n", code, _ExceptionFrame);
	for	(e = _ExceptionFrame; e; e = e->next){
		printf("frame = [ %x, %x, %x, %x ]\n",
			e->next, e->addr, e->bp, e->sp);
		}
	RejectTrap raise(code);
	}
 */
childExit:	gate	(exitCode: unsigned) =
	{
	if	(exitCode){
		if	(exitCode & EX_CAUSE){
			if	(exitCode & EX_CAUSE == EX_ABORT)
				printf("Aborted: ");
			else if	(exitCode & EX_CAUSE == EX_RAISE)
				printf("Uncaught exception: ");
			}
		if	(exitCode == EX_FAIL)
			printf("Failed");
		else
			printf("Exit: 0x%x", exitCode & EX_LEVEL);
		if	(exitCode & EX_CORE)
			printf(" - Core dumped");
		printf("\n");
		}
	}

	};
