/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	types;
include	file, string;
include	errmsg;
include	target;
include	symtab;
include	backend;
include	xtree, xcall;
include	ptree;
include	image;
include	link;
include	hash;
include	real;

FIXUP_SIZE:	const	int = 1024;

ValueIndex:	public	int;
ValueVector:	public	ref * value;

FixupBuf:	* long;
EndFixups:	* long;
EndBuf:		* long;

startLoadFixups:	public	() =
	{
	FixupBuf = alloc(FIXUP_SIZE * sizeof long);
	EndBuf = FixupBuf + FIXUP_SIZE;
	EndFixups = FixupBuf;
	}

finishLoadFixups:	public	() =
	{
	free(FixupBuf);
	}

generateLoadFixup:	public	(v: long, exeFile: stream) =
	{
	*EndFixups = v;
	EndFixups++;
	if	(EndFixups >= EndBuf)
		flushLoadFixups(exeFile);
	}

flushLoadFixups:	public	(exeFile: stream) =
	{
	if	(EndFixups != FixupBuf){
		bp:	ref byte;

		bp = pointer(FixupBuf);
		exeFile write(bp[:int(EndFixups) - int(FixupBuf)]);
		EndFixups = FixupBuf;
		}
	}

buildValueVector:	public	() =
	{
	u:	ref unit_s;
	i:	int;

	i = 0;
	for	(u = Project.units; u; u = u->next)
		i += u countValues();
	if	(ValueVector)
		free(ValueVector);
	ValueVector = 0;			// in case the next line traps
	ValueIndex = 0;
	if	(i == 0)
		return;
	ValueVector = alloc(i * sizeof ref value);
	i = i * sizeof ref value;
	memSet(ValueVector, 0, i);
	for	(u = Project.units; u; u = u->next){
		u->index = ValueIndex;
		u enumerateValues();
		}
	}

start:	entry	() =
	{
	defineLoadFuncs(U_VALUE, &value.load, sizeof value);
	defineLoadFuncs(U_XVALUE, loadFunction(&externValue.load),
					sizeof externValue);
	defineLoadFuncs(U_LITERAL, loadFunction(&literalValue.load),
					sizeof literalValue);
	defineLoadFuncs(U_LFIXUP, loadFunction(&literalFixup_.load),
					sizeof literalFixup_);
	defineLoadFuncs(U_DFIXUP, loadFunction(&dynamicFixup_.load),
					sizeof dynamicFixup_);
	defineLoadFuncs(U_VFIXUP, loadFunction(&valueFixup_.load),
					sizeof valueFixup_);
	defineLoadFuncs(U_RFIXUP, loadFunction(&selfRelativeFixup_.load),
					sizeof selfRelativeFixup_);
/*
	defineLoadFuncs(U_LINE, loadFunction(&ln.load));
 */
	}

anyValue:	public	type	{
	public:

	owner:		ref symbol_s;
	isCode:		boolean;

markUsed:	dynamic	() =
	{
	}

getAddress:	dynamic	() addr_t =
	{
	return 0;
	}

	};

value:	public	type	inherit	anyValue {
	public:

	next:		* value;
	related:	* value;
	dtype:		ref type_s;
	data:		* char;
	dataSize:	addr_t;
	vSize:		addr_t;
	fixups:		* fixup;
	align:		unsigned[16];		// alignment in bytes
	linked:		boolean;
	index:		unsigned[16];
	used:		boolean;
	address:	addr_t;
/*
	lines:		* lnum;
 */
create:	factory	(sym: * symbol_s, t: ref type_s) ref value =
	{
	self = Heap alloc(sizeof value);
	memSet(self, 0, sizeof value);
	self = [ sym ];
	dtype = t;
	return self;
	}

createVector:	factory	() ref value =
	{
	self = Heap alloc(sizeof value);
	memSet(self, 0, sizeof value);
	self = [ 0 ];
	align = PTRALIGN / BYTEBITS;
	return self;
	}
/*
	This is called after the symbol's type has been registered, but
	before the value is actually generated.  The data contents for the
	value are allocated.
 */
registerType:	() =
	{
	x:	addr_t;
	t:	ref type_s;

	t = dtype getType();
	vSize = t sizeOf();
	align = t alignmentOf() / BYTEBITS;
	if	(dataSize != vSize){
		if	(data)
			free(data);
		if	(vSize)
			data = Heap alloc(vSize);
		dataSize = vSize;
		}
	memSet(data, 0, dataSize);		// zero out the object
	}

recordValueSize:	() =
	{
	t:	ref type_s;

	if	(data)
		free(data);
	t = dtype getType();
	vSize = t sizeOf();
	align = t alignmentOf() / BYTEBITS;
	data = 0;
	dataSize = 0;
	}

sizeValue:	(size: addr_t, isCodeValue: boolean) =
	{
	dataSize = size;
	vSize = size;
	isCode = isCodeValue;
	align = FUNCALIGN / BYTEBITS;
	data = Heap alloc(size);
	memSet(data, 0, size);
//	lines = 0;
	}

markUsed:	dynamic	() =
	{
	used = TRUE;
	}

getAddress:	dynamic	() addr_t =
	{
	return address;
	}

depositInteger:	(v: int, offs: addr_t, len: addr_t) =
	{
	memCopy(&data[offs], &v, len);
	}

constructVector:	(t: ref struct_z) =
	{
	offset:	addr_t;
	m:	ref symbol_s;
	v:	ref value;

	align = DYNFUNC_SIZE;
	if	(data)
		free(data);
	vSize = dataSize = t->vectorSize;
	data = Heap alloc(dataSize);
	memSet(data, 0, dataSize);
	for	(offset = 0; offset < t->vectorSize; offset += DYNFUNC_SIZE){
		v = t findVectorEntry(offset);
		if	(v)
			valueFixup(offset, v, 0);
		}
	dtype = t;
	}
/*
	The value, assuming it has been correctly compiled, has been
	determined to have integer type.  Note that the algorithm for
	copying the data value out of a constant is designed for little-
	endian machines.  Big-endian machines will need a different approach.
 */
constantInteger:	() ref tree_p =
	{
	x:	long;
	t:	ref tree_p;

	x = 0;
	if	(dataSize <= sizeof long)
		memCopy(&x, data, dataSize);
	t = Func icon(x, INTBITS);
	t convert(dtype getType());
	return t;
	}
/*
	The value, assuming it has been correctly compiled, has been
	determined to have float type.  
 */
constantFloat:	() ref tree_p =
	{
	t:	ref tree_p;
	r:	real;
	d:	ref type_s;

	if	(dataSize == 4)
		r fromFloat(data);
	else if	(dataSize == 8)
		r fromDouble(data);
	else
		memCopy(&r, data, dataSize);
	d = dtype getType();
	if	(memCompare(&One, &r, sizeof r) != 0 &&
		 memCompare(&Zero, &r, sizeof r) != 0)
		return 0;
	t = Func fcon(&r, ref number_z(d)->width);
	t->dtype = d;
	return t;
	}
/*
	This function does static initialization of a value given the
	expression tree.
 */
initialize:	(x: ref tree_p) =
	{
	itemInitialize(0, dtype getType(), x);
	}

itemInitialize:	(offset: addr_t, t: ref type_s, x: ref tree_p) =
	{
	d:		ref type_s;
	u:		ref tree_p;
	index:		addr_t;
	i:		int;
	hasamemberflag:	int;
	mask:		int;
	ad:		ref array_z;

	switch	(t->topType){
	case	T_ARRAY:
		ad = ref array_z(t);
		t = t elementOf();

			// Character array initialized by a string literal

		if	(x && x->operator == O_LITERAL){
			if	(t->topType != T_SIGNED &&
				 t->topType != T_UNSIGNED){
				error(ErrBadInit);
				return;
				}

			nt:	ref number_z;

			nt = ref number_z(t);
			if	(nt->width != BYTEBITS){
				error(ErrBadInit);
				return;
				}
			x copyString(data + offset, ad->dimension);
			return;
			}
		else if	(x && x->operator != O_SCONST){
			error(ErrBadInit);
			return;
			}
		esize:	addr_t;

		esize = t sizeOf();
		if	(x)
			x = ref sconst_x(x)->components;
		for	(index = 0; x && index < ad->dimension; index++){
			if	(x->operator == O_ARG){
				u = ref binary_x(x)->left;
				x = ref binary_x(x)->right;
				}
			else	{
				u = x;
				x = 0;
				}
			itemInitialize(offset, t, u);
			offset += esize;
			}
		if	(x){
			printf("dimension = %d\n", ad->dimension);
			error(ErrTooManyInit);
			}
		while	(index < ad->dimension){
			itemInitialize(offset, t, 0);
			offset += esize;
			index++;
			}
		break;

	case	T_STRUCT:
		if	(x && x->operator != O_SCONST){
			error(ErrBadInit);
			return;
			}
		st:	ref struct_z;

		st = ref struct_z(t);
		if	(x)
			x = ref sconst_x(x)->components;
		x = structInit(offset, x, st);
		if	(st->gateCount){
			m:	ref symbol_s;
			id:	ref identifier;

			id = hash("__td__");
			m = st lookupMember(id, st);
			if	(m)
				valueFixup(offset + m->bitOffset / BYTEBITS,
							st->gateVector, 0);
			}
		if	(st hasDynamicVector())
			valueFixup(offset + st->vectorOffset / BYTEBITS, 
							st->dynamicVector, 0);
		if	(x)
			error(ErrTooManyInit);
		break;

	case	T_FUNC:
		error(ErrBadInit);
		break;

	case	T_DESCRIPTOR:
		if	(x){
			if	(x->operator == O_LITERAL){
				i:	addr_t;

				i = x->dtype sizeOf();
				x copyFixup(self, offset, t sizeOf());
				memCopy(&data[offset + DescrMaxBoundOffset], 
						&i, sizeof i);
				memCopy(&data[offset + DescrBoundOffset], 
						&i, sizeof i);
				}
			else if	(x->operator == O_ID){
				dim:	addr_t = 0;

				if	(x->dtype->topType != T_ARRAY)
					error(ErrBadInit);
				else
					dim = ref array_z(x->dtype)->dimension;
				x copyFixup(self, offset, t sizeOf());
				memCopy(&data[offset + DescrMaxBoundOffset], 
						&dim, sizeof dim);
				memCopy(&data[offset + DescrBoundOffset], 
						&dim, sizeof dim);
				}
			else	{
				x display(0);
				error(ErrUnfinished);
				}
			}
		break;

	default:
		if	(x){
			x = x promoteArrays();
			x = x convert(t);
			x = x fold();
			x copyFixup(self, offset, t sizeOf());
			}
		}
	}

structInit:	(offset: addr_t, x: ref tree_p, 
				d: ref struct_z) ref tree_p =
	{
	t:		ref tree_p;
	u:		ref tree_p;
	sz:		addr_t;
	bitsz:		addr_t;
	mt:		ref type_s;
	sym:		ref symbol_s;
	i:		int;
	j:		int;
	mask:		int;

		/* Look for a member function 'constructor' */

	if	(x){
		for	(sym = d->symbols; sym; sym = sym->next){
			if	(sym->name &&
				 sym->name isSpelled("constructor")){
				if	(sym->dtype->topType == T_FUNC){
					error(ErrBadInit);
					return 0;
					}
				break;
				}
			}
		}
	if	(d->base){
		sup:	ref type_s;

		sup = d->base getType();

				// structure base classes use special init code

		if	(sup->topType == T_STRUCT)
			x = structInit(offset, x, ref struct_z(sup));
		else		// must be an error
			return 0;
		}
	for	(sym = d->symbols; sym; sym = sym->next){

			/* Skip member functions and static members */

		if	(sym->storageClass != SC_MEMBER)
			continue;

			// If the member is private, then allow a static
			// initializer of a private member only for objects
			// defined inside the type itself.  This should be
			// a rare case.

		if	(sym->visibility == V_PRIVATE){
			if	(owner &&
				 !d encloses(owner->enclosing))
				continue;
			}
		if	(sym->name == 0)
			continue;
		if	(x && x->operator == O_ARG){
			u = ref argument_x(x)->left;
			x = ref argument_x(x)->right;
			}
		else	{
			u = x;
			x = 0;
			}
		mt = sym->dtype;
		if	(mt == 0 ||
			 mt->topType == T_ERROR)
			continue;
		mt = mt getType();
		if	(u == 0 ||
			 u->operator != O_ICON)		// not a bit field
			itemInitialize(offset + sym->bitOffset / BYTEBITS,
						mt, u);
		else	{				// maybe a bit field
			off:	addr_t;
			bitoff:	int;
			mask:	long;
			ip:	ref long;

			bitsz = mt bitSizeOf();
			off = offset + sym->bitOffset / BYTEBITS;
			bitoff = sym->bitOffset % BYTEBITS;
			mask = (1UL << bitsz - 1) << bitoff;
			ip = ref long(data + off);
			*ip &= ~mask;
			*ip |= ref icon_x(u)->integer << bitoff;
			}
		}
	return x;
	}

load:	factory	(v: pointer, image: ref loader) =
	{
	self = v;
	self = [ 0 ];
	next = image address();
	related = image address();
	owner = image address();
	dtype = image address();
	fixups = image address();
	address = image uinteger();
	isCode = image integer();
	align = image uinteger();
	vSize = image uinteger();
	dataSize = image integer();
	if	(dataSize){
		data = Heap alloc(dataSize);
		image literal(data, dataSize);
		}
	else
		data = 0;
/*
	lines = image list(offsetof lnum.next);
 */
	}

save:	(ifile: ref imageFile) =
	{
	if	(!ifile beginRecord(U_VALUE, self, sizeof *self))
		return;			// already saved
	ifile address(next);
	ifile address(related);
	ifile address(owner);
	ifile address(dtype);
	ifile address(fixups);
	ifile uinteger(address);
	ifile integer(isCode);
	ifile uinteger(align);
	ifile uinteger(vSize);
	ifile literal(data, dataSize);
	if	(next)
		next save(ifile);
	if	(related)
		related save(ifile);
	if	(fixups)
		fixups save(ifile);
/*
	l:	* lnum;

	for	(l = lines; l; l = l->next)
		l save(ifile);
	ifile nullTag();
 */
	}

addRelatedValue:	(v: ref value) =
	{
	if	(v == 0)
		return;
	v->next = related;
	v->owner = 0;
	v->dtype = 0;
	related = v;
	}

countRelated:	() int =
	{
	v:	ref value;
	i:	int;

	index = ValueIndex++;
	for	(i = 1, v = related; v; v = v->next, i++)
		v->index = ValueIndex++;
	return i;
	}

enumerateValues:	() =
	{
	v:	ref value;

	index = ValueIndex;
	ValueVector[ValueIndex++] = self;
	for	(v = related; v; v = v->next){
		v->index = ValueIndex;
		ValueVector[ValueIndex++] = v;
		}
	}

link:	() =
	{
	f:		* fixup;

	for	(f = fixups; f; f = f->next)
		f applyFixup(self);
	}

writeLoadFixups:	(exeFile: stream) =
	{
	f:		* fixup;

	for	(f = fixups; f; f = f->next)
		f applyLoadFixup(self, exeFile);
	}
/*
newLine:	(lastLine: * lnum, lineno: int, off: addr_t, 
						bkFlag: int) * lnum =
	{
	l:	* lnum;

	if	(lastLine){
		if	(lastLine->line == lineno)
			return lastLine;
		if	(lastLine->line > lineno){
			for	(l = lines; l; l = l->next)
				if	(l->line == lineno)
					return lastLine;
			}
		if	(lastLine->offset == off){
			lastLine->line = lineno;
			return lastLine;
			}
		}
	l = Heap alloc(sizeof lnum);
	if	(lastLine)
		lastLine->next = l;
	else
		lines = l;
	l->next = 0;
	l->line = lineno;
	l->offset = off;
	l->breakFlag = bkFlag;
	return l;
	}

dispose:	() =
	{
	uninitialize();
	if	(next)
		next dispose();
	free(self);
	}

 */
markReferencedValues:	() =
	{
	f:	ref fixup;

	for	(f = fixups; f; f = f->next)
		f markReferencedValue();
	}

valueFixup:	(location: addr_t, v: ref value, adjust: addr_t) =
	{
	if	(unsigned(v) < _EBP){
		printf("v = %x loc = %x adj = %x\n", v, location, adjust);
		return;
		}
	fixups = valueFixup_ create(fixups, location, v, adjust);
	}

selfRelativeFixup:	(location: addr_t, v: ref value, adjust: addr_t) =
	{
	fixups = selfRelativeFixup_ create(fixups, location, v, adjust);
	}

dynamicFixup:	(location: addr_t, s: ref struct_z, index: addr_t) =
	{
	fixups = dynamicFixup_ create(fixups, location, s, index);
	}

literalFixup:	(location: addr_t, lit: ref literalItem_p, adjust: addr_t,
						nullTerminate: boolean) =
	{
	l:	ref literalValue;

	l = literalValue create(lit, nullTerminate);
	fixups = literalFixup_ create(fixups, location, l, adjust);
	}

	};

externValue:	public	type	inherit	anyValue {
	public:

	target:		ref value;
	name:		[] char;

load:	factory	(xv: pointer, image: ref loader) =
	{
	cp:	[:] char;

	self = xv;
	self = [ 0 ];
	cp = image string();
	memCopy(name, cp, |cp + 1);
	target = image address();
	}

save:	(ifile: ref imageFile) =
	{
	if	(ifile beginRecord(U_XVALUE, self, sizeof externValue + 
					stringLength(name) + 1)){
		ifile putstring(name);
		ifile address(target);
		}
	}

markUsed:	dynamic	() =
	{
		// Look up name.

	cp:	ref char;

	if	(target == 0){

			// isolate the unit name first

		for	(cp = name; *cp != ':'; cp++)
			if	(*cp == 0){
//				error(ErrBadFixup, name);
				return;
				}
		sc:	ref scope_s;
		sc = Project locateUnit(hash(name[:cp - name]));
		if	(sc == 0){
//			error(ErrBadFixup, name);
			return;
			}
		while	(*cp == ':')
			cp++;

		nm:	ref char;
		sym:	ref symbol_s;
		n:	ref identifier;
		typeSelect:	boolean;

		typeSelect = FALSE;
		nm = cp;
		while	(*cp){
			if	(*cp == '.'){
				n = hash(nm[:cp - nm]);
				sym = sc lookupLocal(n, sc);
				if	(sym == 0){
//					error(ErrBadFixup, &name);
					return;
					}
				if	(cp[1] == 0){
					typeSelect = TRUE;
					break;
					}
				sc = sym->dtype;
				nm = cp + 1;
				}
			cp++;
			}
		if	(typeSelect){
			t:	ref type_s;

			t = sym->dtype getType();
			if	(t->topType != T_STRUCT){
//				error(ErrBadFixup, &name);
				return;
				}
			target = ref struct_z(t)->dynamicVector;
			isCode = FALSE;
			}
		else if	(stringCompare(nm[:cp - nm], "gate") == 0){
			t:	ref type_s;

			t = sym->dtype getType();
			if	(t->topType != T_STRUCT){
//				error(ErrBadFixup, &name);
				return;
				}
			target = ref struct_z(t)->gateVector;
			isCode = FALSE;
			}
		else	{
			n = hash(nm[:cp - nm]);
			sym = sc lookupLocal(n, sc);
			if	(sym == 0){
//				error(ErrBadFixup, &name);
				return;
				}
			target = sym->currentValue;
			isCode = target->isCode;
			}
		if	(target == 0){
//			error(ErrBadFixup, &name);
			return;
			}
		}
	target markUsed();
	}

getAddress:	dynamic	() addr_t =
	{
	if	(target)
		return target getAddress();
	else
		return 0;
	}

	};

LiteralValue:	public	value;

fakeLiteralValue:	public	(lit: ref literalItem_p) ref value =
	{
	LiteralValue.data = pointer(lit);
	return &LiteralValue;
	}

literalValue:	public	type	{
	public:

	length:		int;
	address:	addr_t;
	value:		[] char;

create:	factory	(lit: ref literalItem_p, 
				nullTerminate: boolean) ref literalValue =
	{
	l:	ref literalItem_p;
	i:	int;
	cp:	ref char;

	if	(nullTerminate)
		i = 1;
	else
		i = 0;
	for	(l = lit; l; l = l->next)
		i += l->length;
	self = Heap alloc(sizeof literalValue + i);
	self = [ i ];
	for	(l = lit, cp = value; l; l = l->next){
		memCopy(cp, l->value, l->length);
		cp += l->length;
		}
	if	(nullTerminate)
		*cp = 0;		// null terminate
	return self;
	}

load:	factory	(lv: pointer, image: ref loader) =
	{
	self = lv;
	length = image integer();
	image literal(&value, length);
	}

save:	(ifile: ref imageFile) =
	{
	if	(ifile beginRecord(U_LITERAL, self, sizeof *self + length))
		ifile literal(&value, length);
	}

	};

/*
lnum:	public	type	{
	next:		* lnum;
	offset:		unsigned;
	line:		unsigned[16];
	byteValue:	char;		// Byte at the location
	breakFlag:	char;		// non-zero if the line has a breakpoint

load:	factory	(image: * loadFile, byte) * lnum =
	{
	self = alloc(sizeof lnum);
	line = image integer();
	offset = image integer();
	byteValue = 0;
	breakFlag = 0;
	return self;
	}

save:	(ifile: ref imageFile) =
	{
	ifile beginRecord(U_LINE);
	ifile integer(line);
	ifile integer(offset);
	}

dispose:	() =
	{
	n:	* lnum;

	while	(self){
		n = next;
		free(self);
		self = n;
		}
	}

	};
 */
fixup:	public	type	{
	public:

	next:		* fixup;
	location:	addr_t;
	adjust:		addr_t;

save:	(ifile: ref imageFile) =
	{
	while	(self){
		savePart(ifile);
		self = next;
		}
	}

savePart:	dynamic	(ref imageFile) =
	{
	}

targetAddress:	dynamic	(ref value) addr_t =
	{
	return 0;
	}

applyFixup:	(val: ref value) =
	{
	*ref addr_t(&val->data[location]) = targetAddress(val) + adjust;
	}

applyLoadFixup:	(val: * value, exeFile: stream) =
	{
	seg:	long;

	seg = loadFixupSegment();
	if	(seg == ~0)
		return;				// default to no fixup
	seg += location + val->address;
	if	(!val->isCode)
		seg += 0x40000000;		// ref is in data
	generateLoadFixup(seg, exeFile);
	}

loadFixupSegment:	dynamic	() long =
	{
	return ~0;
	}

assignLiteralLocation:	dynamic	() =
	{
	}

linkLiteral:	dynamic	() addr_t =
	{
	return 0;
	}

markReferencedValue:	dynamic	() =
	{
	}

	};

valueFixup_:	public	type	inherit	fixup {
	public:

	target:		ref value;

create:	factory	(n: ref fixup, loc: addr_t, t: ref value, 
					adj: addr_t) ref valueFixup_ =
	{
	self = Heap alloc(sizeof valueFixup_);
	self = [ n, loc, adj, t ];
	return self;
	}

load:	factory	(vf: pointer, image: ref loader) =
	{
	self = vf;
	self = [ 0 ];
	next = image address();
	location = image uinteger();
	adjust = image uinteger();
	target = image address();
	}

savePart:	dynamic	(ifile: ref imageFile) =
	{
	if	(ifile beginRecord(U_VFIXUP, self, sizeof *self)){
		ifile address(next);
		ifile uinteger(location);
		ifile uinteger(adjust);
		spitValue(ifile, target);
		}
	}

targetAddress:	dynamic	(ref value) addr_t =
	{
	return target getAddress();
	}

loadFixupSegment:	dynamic	() long =
	{
	if	(target->isCode)
		return 0;			// ref to code
	else
		return 0x80000000;		// ref to data
	}

markReferencedValue:	dynamic	() =
	{
//	printf("target = %x", target);
	target markUsed();
	}

	};

selfRelativeFixup_:	public	type	inherit	fixup {
	public:

	target:		ref value;

create:	factory	(n: ref fixup, loc: addr_t, t: ref value, 
					adj: addr_t) ref selfRelativeFixup_ =
	{
	self = Heap alloc(sizeof selfRelativeFixup_);
	self = [ n, loc, adj, t ];
	return self;
	}

load:	factory	(vf: pointer, image: ref loader) =
	{
	self = vf;
	self = [ 0 ];
	next = image address();
	location = image uinteger();
	adjust = image uinteger();
	target = image address();
	}

savePart:	dynamic	(ifile: ref imageFile) =
	{
	if	(ifile beginRecord(U_RFIXUP, self, sizeof *self)){
		ifile address(next);
		ifile uinteger(location);
		ifile uinteger(adjust);
		spitValue(ifile, target);
		}
	}

targetAddress:	dynamic	(v: ref value) addr_t =
	{
	return target getAddress() - (v->address + location + 4);
	}

markReferencedValue:	dynamic	() =
	{
	target markUsed();
	}

	};

literalFixup_:	public	type	inherit	fixup {
	public:

	target:		ref literalValue;
	address:	addr_t;

create:	factory	(n: ref fixup, loc: addr_t, t: ref literalValue,
					adj: addr_t) ref literalFixup_ =
	{
	self = Heap alloc(sizeof literalFixup_);
	self = [ n, loc, adj, t, 0 ];
	return self;
	}

load:	factory	(lf: pointer, image: ref loader) =
	{
	self = lf;
	self = [ 0 ];
	next = image address();
	location = image uinteger();
	adjust = image uinteger();
	target = image address();
	}

savePart:	dynamic	(ifile: ref imageFile) =
	{
	if	(ifile beginRecord(U_LFIXUP, self, sizeof *self)){
		ifile address(next);
		ifile uinteger(location);
		ifile uinteger(adjust);
		ifile address(target);
		if	(target)
			target save(ifile);
		}
	}

targetAddress:	dynamic	(ref value) addr_t =
	{
	return target->address;
	}

loadFixupSegment:	dynamic	() long =
	{
	return 0x80000000;		// ref to data
	}

assignLiteralLocation:	dynamic	() =
	{
	RunFile assignLiteralLocation(target);
	}

linkLiteral:	dynamic	() addr_t =
	{
	RunFile linkLiteral(target);
	return target->length;
	}

	};

spitValue:	(ifile: ref imageFile, target: ref value) =
	{
		// Saving to a library unit needs special effort

	if	(Project.projectSave){
		ifile address(target);
		return;
		}

	u:	ref unit_s;

	if	(target->owner)
		u = target->owner->enclosing getUnit();
	else if	(target->dtype)
		u = target->dtype getUnit();
	else
		u = Project.saveUnit;
	if	(Project.saveUnit == u){
		ifile address(target);
		return;
		}
	x:	pointer;
	cp:	[:] char;

	if	(target->owner)
		cp = target->owner objectName();
	else
		cp = target->dtype dynamicVectorName(target);
	x = ifile->saveHeap alloc(1);
	ifile address(x);
	if	(ifile beginRecord(U_XVALUE, x, 
				sizeof externValue + 
					|cp + 1)){
		ifile putstring(cp);
		ifile address(0);
		}
	}


dynamicFixup_:	public	type	inherit	fixup {
	public:

	target:		ref struct_z;
	index:		int;

create:	factory	(n: ref fixup, loc: addr_t, s: ref struct_z, 
					idx: int) ref dynamicFixup_ =
	{
	self = Heap alloc(sizeof dynamicFixup_);
	self = [ n, loc, 0, s, idx ];
	return self;
	}

load:	factory	(lf: pointer, image: ref loader) =
	{
	self = lf;
	self = [ 0 ];
	next = image address();
	location = image uinteger();
	target = image address();
	index = image integer();
	}

savePart:	dynamic	(ifile: ref imageFile) =
	{
	if	(ifile beginRecord(U_DFIXUP, self, sizeof *self)){
		ifile address(next);
		ifile uinteger(location);
		ifile address(target);
		ifile integer(index);
		}
	}

targetAddress:	dynamic	(val: ref value) addr_t =
	{
	return * ref long(&val->data[location]) - adjust;
	}

	};

