/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	process;
include	pc_hdw;
include	disk;
include	node;
include	kprintf;
include	hardware;
include	vector;
include	sound;

pcHardDiskInitialization:	public	() =
	{

		// Initialize the controller

	DiskController = [ FD_CONTROL, HD_PORT, HDISK_INT ];

		// Define the disk drives

	driveType:	byte;

	driveType = getCMOSbyte(0x12);

	DiskController.drive[0] = [ 0, driveType >> 4,	0x104 ];
	DiskController.drive[1] = [ 1, driveType & 0xf,	0x118 ];
	}
/*
	The DiskController object is the actual device driver for the
	hard disk controller of a PC.  The controller needs three hardware
	parameters: the control I/O port, the address of the block of eight
	hard disk I/O ports, and the interrupt number used by the controller.

	A controller object supports two operations: read and write.  The
	controller can only issue one operation at a time, so only one drive
	at a time can be used.

	The acquire event is used to enforce serial access to the controller.

	Note that the hardware does not use DMA to transfer data, but instead
	requires the CPU to copy the data two bytes at a time.

	Disk accesses are currently being performed on a first-come, first-
	served basis.  In fact, disk accesses that change the cylinder
	number are significantly slower than accesses that stay within a
	cylinder, because the servo motor that moves the read/write heads is
	much slower than the disk rotation speed.  As a result, in a disk
	server on a LAN, considerable utilization improvements can be made
	by minimizing the cylinder-to-cylinder movement of the heads.  For a
	personal machine, such improvements may not be worth the trouble.  Of
	course most elevator algorithms will work fine when access requests
	are sequential.  Those algorithms involve minimal CPU overhead over a
	simple first-come first-served strategy.

	One can also separate the seek and transfer commands so that additional
	reads or writes that arrive after a seek has been issued may be bundled
	into a single transfer command.  Since adjacent sectors are frequently
	accessed together, allowing additional transfers to accumulate while
	a seek is underway may further help cut down on rotational latency.

	This last improvement will require some experimentation since it does
	involve more communications between the CPU and the disk controller,
	and that extra work may be greater than the savings achieved through 
	the added parallelism.
 */
DiskController:	public	{
	public:

acquire:		semaphore;
wait:			ref semaphore;
controlPort:		unsigned[16];
diskPort:		unsigned[16];
drive:			[2] hardDiskDrive;

constructor:	(cp: unsigned[16], dp: unsigned[16], ivect: int) =
	{
	controlPort = cp;
	diskPort = dp;
	wait = vector semaphoreInterrupt(ivect);
	acquire = [ 1 ];
	}

	};

DISK_READ:	const	int = 0x20;
DISK_WRITE:	const	int = 0x30;

/*
	The diskDrive object holds the drive characteristics for each drive
	attached to the hard disk controller.
 */
hardDiskDrive:	type	inherit	diskDrive_t {
	public:

constructor:	(ds: byte, dType: byte, BIOSvect: unsigned) =
	{
	r:	realModePtr;
	bd:	BIOSdisk;

	driveSelect = ds;
	driveType = dType;
	if	(dType == 0)
		return;
	copyIn(&r, BIOSvect, sizeof r);
	copyIn(&bd, realToPhysical(r), sizeof bd);
	heads = bd.heads;
	cylinders = bd.cylinders;
	sectors = bd.sectors;
	writePrecomp = bd.writePrecomp;
	control = bd.control;
	landingZone = bd.landingZone;
	loadPartition(0);
	}

/*
	The read operation is carried out one sector at a time.  The
	controller is started for each sector, then the reader waits for
	an disk interrupt event to occur.  The hardware interrupt controllers
	are reset, the sector-full of data is copied into the caller's buffer
	and status is checked.  If there was some sort of failure, a retry
	is started by looping back to the beginning of the function.  
	Otherwise, the transfers increments to the next sector and operation
	continues.
 */
read:	dynamic	(sector: long, buf: pointer, count: int) =
	{
	DiskController.acquire down(FALSE);
	for	(;;){
		executeIO(DISK_READ, sector, count);

		for	(;;){
			DiskController.wait down(FALSE);
			Secondary8259 clearInterrupt();
			Primary8259 clearInterrupt();

			_ECX = SECTOR_SIZE / 2;
			_EDI = int(buf);
			_DX = DiskController.diskPort;
			_emit(0xfc);			// CLD
			_emit(0xf3, 0x66, 0x6D);	// REP INSW

				// Check status

			i:	char;

			i = _inportByte(DiskController.diskPort + 7);

				// Controller still busy?

			if	(i & 0x80){

					// must be a multi-sector operation

				count--;
				buf = ref char(buf) + SECTOR_SIZE;
				sector++;
				}
			else if	(i & 0x70 != 0x50){
						// Drive ready, no write
						// fault, seek complete
						// indicates a good operation
						// otherwise, retry the 
						// operation
				AlysNode.diskRetries++;
				break;
				}
			else	{
				DiskController.acquire up();
				return;		// Operation is complete
				}
			}
		}
	}
/*
	The write operation is carried out one sector at a time.  The
	controller is started, then a sector-full of data is copied to
	the controller data buffer.  The writer waits for a disk interrupt
	event to happen, resets the interrupt controllers, and then status
	is checked to determine whether the operation succeeded.  If not,
	the sector is retried.  If so, the transfer is stepped to the next
	sector until the transfer is complete.
 */
write:	dynamic	(sector: long, buf: pointer, count: int) =
	{
	DiskController.acquire down(FALSE);
	for	(;;){
		executeIO(DISK_WRITE, sector, count);

		for	(;;){
			_ECX = SECTOR_SIZE / 2;
			_ESI = int(buf);
			_DX = DiskController.diskPort;
			_emit(0xfc);			// CLD
			_emit(0xf3, 0x66, 0x6F);		// REP OUTSW

			DiskController.wait down(FALSE);
			Secondary8259 clearInterrupt();
			Primary8259 clearInterrupt();

				// Check status

			i:	char;

			i = _inportByte(DiskController.diskPort + 7);
			if	(i & 0x70 != 0x50){
						// Drive ready, no write
						// fault, seek complete
						// indicates a good operation
						// otherwise, retry the 
						// operation
				AlysNode.diskRetries++;
				break;
				}

				// Controller waiting for data?

			else if	(i & 0x8){

					// must be a multi-sector operation

				count--;
				buf = ref char(buf) + SECTOR_SIZE;
				sector++;
				}
			else	{
				DiskController.acquire up();
				return;		// Operation is complete
				}
			}
		}
	}


private:
/*
	This function maps the logical sector number into a sector,
	head, cylinder combination for the target drive.  The needed
	values are written to the controller I/O port registers.  Note
	that the command register is written last.  The controller will
	not act on any of the other registers until a command is written.
 */
executeIO:	(op: int, sector: long, count: byte) =
	{
	head:		int;

	_outportByte(DiskController.controlPort, control);
	_outportByte(DiskController.diskPort + 1, writePrecomp >> 2);
	_outportByte(DiskController.diskPort + 2, count);
	_outportByte(DiskController.diskPort + 3, sector % sectors + 1);
	sector /= sectors;
	head = sector % heads;
	sector /= heads;
	_outportByte(DiskController.diskPort + 4, sector);
	_outportByte(DiskController.diskPort + 5, sector >> 8);
	_outportByte(DiskController.diskPort + 6, 0xA0 + (head & 0xf) + 
							(driveSelect << 4));
	_outportByte(DiskController.diskPort + 7, op);
	}
/*
	This function loads a partition table and defines teh resulting 
	partitions.
 */
loadPartition:	(sector: unsigned) =
	{
	b:	bootBlock;
	p:	ref partitionDef;
	plow:	ref partitionDef;
	i:	int;
	j:	int;
	px:	ref partition_t;

	read(sector, &b, 1);			// Read the partition table
	if	(b.signature != 0xAA55){
		kprintf("Boot Signature not found\n");
		return;
		}
	for	(j = 0; j < 4; j++){
		p = b.partitions;
		plow = 0;
		for	(i = 0; i < 4; i++, p++){
			if	(p->sectCount == 0 ||
				 p->systInd == 0)
				continue;
			if	(plow == 0 ||
				 plow->relSect > p->relSect)
				plow = p;
			}
		if	(plow == 0)
			break;

		u, v:	unsigned;
		cyl, sect:	unsigned;
		cyl = plow->bcyl + (plow->bsect & 0xc0) << 2;
		sect = plow->bsect & 0x3f;
		u = (cyl * heads + plow->bhead) * sectors + sect - 1;
		cyl = plow->ecyl + (plow->esect & 0xc0) << 2;
		sect = plow->esect & 0x3f;
		v = (cyl * heads + plow->ehead) * sectors + sect - 1;
		if	(plow->systInd == SI_EXTENDED){	// Extended partition

				// recursively load it's partition table

			loadPartition(u);
			}
		else	{
			px = partition_t create(plow->systInd, FALSE);
			if	(px == 0)
				return;
			px->sectorCount = 1 + v - u;
			px->sectorOffset = u;
			px->drive = self;
			px display();
			}
		plow->systInd = 0;		// drop the partition
						// for the next round of
						// searching
		}
	}

driveSelect:	byte;
driveType:	byte;
heads:		byte;
sectors:	byte;
cylinders:	unsigned[16];
writePrecomp:	unsigned[16];
control:	byte;
landingZone:	unsigned[16];

	};

BIOSdisk:	type	packed	{
	public:

	cylinders:	unsigned[16];
	heads:		byte;
	fill:		unsigned[16];
	writePrecomp:	unsigned[16];
	fill2:		byte;
	control:	byte;
	fill3:		[3] byte;
	landingZone:	unsigned[16];
	sectors:	byte;
	};

/*
driveDef:	type	{
	public:

	heads:		byte;
	sectors:	byte;
	cylinders:	unsigned[16];
	writePrecomp:	unsigned[16];
	control:	byte;
	landingZone:	unsigned[16];
	};

DriveTable:	[] driveDef = [
	[  0,   0,    0,    0,    0,    0 ],		// Drive type 0
	[  4,  17,  306,  128,    0,  305 ],		// Drive type 1
	[  4,  17,  615,  300,    0,  615 ],		// Drive type 2
	[  6,  17,  615,  300,    0,  615 ],		// Drive type 3
	[  8,  17,  940,  512,    0,  940 ],		// Drive type 4
	[  6,  17,  940,  512,    0,  940 ],		// Drive type 5
	[  4,  17,  615,   -1,    0,  615 ],		// Drive type 6
	[  8,  17,  462,  256,    0,  511 ],		// Drive type 7
	[  5,  17,  733,   -1,    0,  733 ],		// Drive type 8
	[ 15,  17,  900,   -1,    8,  901 ],		// Drive type 9
	[  3,  17,  820,   -1,    0,  820 ],		// Drive type 10
	[  5,  17,  855,   -1,    0,  855 ],		// Drive type 11
	[  7,  17,  855,   -1,    0,  855 ],		// Drive type 12
	[  8,  17,  306,  128,    0,  319 ],		// Drive type 13
	[  7,  17,  733,   -1,    0,  733 ],		// Drive type 14
	[  0,   0,    0,    0,    0,    0 ]		// Drive type 15
	];
 */

bootBlock:	type	packed	{
			[0x1be] char;
	public:
	partitions:	[4] partitionDef;
	signature:	unsigned[16];
	};

partitionDef:	type	packed	{
	public:

	bootInd:	byte;
	bhead:		byte;
	bsect:		byte;
	bcyl:		byte;
	systInd:	byte;
	ehead:		byte;
	esect:		byte;
	ecyl:		byte;
	relSect:	unsignedLong;
	sectCount:	unsignedLong;
	};
