/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	alys;
include	kprintf;

BIG_ENDIAN:	public	const	int = 0;		// little endian = 0
							// big endian = 1
KERNEL_STACK:	public	const int = 3500;
NODE_BITS:	public	const int = 10;		// Number of node bits in id
OBJ_BITS:	public	const int = 16;		// Number of obj bits in id
NET_SHIFT:	public	const int = NODE_BITS;
NET_MASK:	public	const int = ~(1 << NODE_BITS - 1);
NODE_SHIFT:	public	const int = OBJ_BITS;
NODE_MASK:	public	const int = ~(1 << OBJ_BITS - 1);
KERNEL_VIRTUAL_ADDRESS:	
		public	const	paddr_t = 0x80000000;

vaddr_t:	public	type	unsigned[32];	// virtual address
paddr_t:	public	type	unsigned[32];	// physical address

taskState_t:	public	type	byte = {
	TS_ERROR,				// Not used - error condition
	TS_RUN,					// Normal running
	TS_TRAP,				// Hardware trap encountered
	TS_REJECT,				// Message rejected
	TS_INTERRUPT,				// Message interrupted
	TS_ABORT,				// Abort called
	TS_EXCEPTION,				// Uncaught exception
	TS_EXIT,				// Exit called
	TS_BREAK,				// Breakpoint
	TS_STEP,				// Single step
	};

task_t:	public	type	packed	{
	public:

	backlink:	unsigned[32];
	esp0:		unsigned[32];
	ss0:		unsigned[32];
	esp1:		unsigned[32];
	ss1:		unsigned[32];
	esp2:		unsigned[32];
	ss2:		unsigned[32];
	cr3:		unsigned[32];
	eip:		unsigned[32];
	eflags:		unsigned[32];
	eax:		unsigned[32];
	ecx:		unsigned[32];
	edx:		unsigned[32];
	ebx:		unsigned[32];
	esp:		unsigned[32];
	ebp:		unsigned[32];
	esi:		unsigned[32];
	edi:		unsigned[32];
	es:		unsigned[32];
	cs:		unsigned[32];
	ss:		unsigned[32];
	ds:		unsigned[32];
	fs:		unsigned[32];
	gs:		unsigned[32];
	ldt:		unsigned[32];
	ioperm:		unsigned[32];
	fpu:		fpu_t;
	pageTable0:	paddr_t;
	idt:		paddr_t;
	state:		taskState_t;
			[3] byte;
	errorCode:	signed[32];

constructor:	() =
	{
	memSet(self, 0, sizeof task_t);
	}

trigger:	(base: ref task_t, ifr: ref interruptFrame_t, intr: int, 
						errword: int) =
	{
	tf:	ref i386trapFrame;

	*self = *base;
	idt = intr;
	pageTable0 = 0;

	if	(errword){
		tf = ref i386trapFrame(&ifr->cs);
		pageTable0 = ifr->eip;
		}
	else
		tf = ref i386trapFrame(&ifr->eip);

	eflags = tf->eflags;
	eip = tf->eip;

	eax = ifr->eax;
	ebx = ifr->ebx;
	ecx = ifr->ecx;
	edx = ifr->edx;
	ebp = ifr->ebp;
	esi = ifr->esi;
	edi = ifr->edi;

	cs = tf->cs;
	ds = ifr->ds;
	es = ifr->es;
	fs = ifr->fs;
	gs = ifr->gs;
	if	(calledFromUserMode()){
		esp = tf->esp;
		ss = tf->ss;
		}
	else	{
		esp = unsigned(&tf->esp);
		ss = _SS;
		}
	}

calledFromUserMode:	() boolean =
	{
	if	(eflags & VM ||				// v8086 mode
		 cs & RING == USER_RING)		// ring == 3
		return TRUE;
	else
		return FALSE;
	}

display:	() =
	{
	lab:	static	[] ref char = [
		"Divide error",
		"Debug Exception",
		"- 2 -",
		"Unexpected breakpoint",
		"Overflow instruction",
		"Bounds check",
		"Invalid opcode",
		"Coprocessor trap",
		"Double fault",
		"Coprocessor segment overrun",
		"Invalid TSS",
		"Segment not present",
		"Stack segment fault",
		"General protection fault",
		"Page fault",
		"- 15 -",
		"Coprocessor error",
		];

	kprintf("%s at %08x\n", lab[idt], eip);
	switch	(idt){
	case	0x0e:
		if	(pageTable0 & 4)
			kprintf("user");
		else
			kprintf("kernel");
		kprintf(" mode ");
		if	(pageTable0 & 2)
			kprintf("write");
		else
			kprintf("read");
		if	(pageTable0 & 1)
			kprintf(" protection error");
		else
			kprintf(" not present error");
		_emit(0x0f, 0x20, 0xd0);		// mov eax,cr2
		kprintf(" at %x\n", _EAX);
		break;

	case	0x08:
	case	0x0a:
	case	0x0b:
	case	0x0c:
		if	(pageTable0 & 2)
			kprintf("Error in IDT vector 0x%x", pageTable0 >> 3);
		else if	(pageTable0 & 4)
			kprintf("Error in LDT selector 0x%04x", pageTable0 & ~3);
		else if	(pageTable0 & ~7)
			kprintf("Error in GDT selector 0x%04x", pageTable0 & ~3);
		else
			kprintf("Error in NULL selector");
		if	(pageTable0 & 1)
			kprintf(" while servicing a nested interrupt");
		kprintf("\n");
/*
		if	(pageTable0 & 2)
			dumpVector(pageTable0 >> 3);
		else if	(pageTable0 & 4)
			dumpMapping(pageTable0 & ~3);
		else if	(pageTable0 & ~7)
			dumpMapping(pageTable0 & ~3);
 */
		}
	if	(ss != 0x000f){
		kprintf(" link %08x ss0:esp0 %04x:%08x\n", backlink, ss0, esp0);
		kprintf("  cr3 %08x  ss:esp  %04x:%08x\n", cr3, ss, esp);
		}
	else
		kprintf("                   esp       %08x\n", esp);
	if	(cs != 0x7)
		kprintf("flags %08x  cs:eip  %04x:%08x\n", eflags, cs, eip);
	kprintf("  eax %08x ebx %08x ecx %08x edx %08x\n", eax, ebx, ecx, edx);
	kprintf("  ebp %08x esi %08x edi %08x\n", ebp, esi, edi);
	if	(ds != 0x000f ||
		 es != 0x000f ||
		 fs != 0x000f ||
		 gs != 0x000f){
		kprintf("   ds     %04x  es     %04x  fs     %04x  gs     %04x\n",
						ds, es, fs, gs);
		}
	if	(eflags & 1)
		kprintf("CY ");
	else
		kprintf("NC ");
	if	(eflags & 4)
		kprintf("PA ");
	else
		kprintf("NP ");
	if	(eflags & 0x10)
		kprintf("AC ");
	else
		kprintf("NA ");
	if	(eflags & 0x40)
		kprintf("ZF ");
	else
		kprintf("NZ ");
	if	(eflags & 0x80)
		kprintf("SG ");
	else
		kprintf("NS ");
	if	(eflags & 0x100)
		kprintf("TF ");
	else
		kprintf("NT ");
	if	(eflags & 0x200)
		kprintf("IE ");
	else
		kprintf("NI ");
	if	(eflags & 0x400)
		kprintf("DN ");
	else
		kprintf("UP ");
	if	(eflags & 0x800)
		kprintf("OV ");
	else
		kprintf("NO ");
	if	(eflags & 0x4000)
		kprintf("NT ");
	if	(eflags & 0x10000)
		kprintf("RF ");
	if	(eflags & VM)
		kprintf("VM ");
	kprintf("iopl = %x\n", eflags >> 12 & 3);
/*
	if	(cs & RING ||			// ring 3 trap
		 eflags & VM)			// v8086 trap
		abort(EX_LEVEL);
	enableAll();
	for	(;;)
		;
 */
	}

	};

RING:		const	int = 3;
USER_RING:	const	int = 3;

VM:	const	unsigned = 0x20000;		// v8086 mode

i386trapFrame:	public	type	packed	{
	public:

	eip:			unsigned[32];
	cs:			unsigned[16];
	extra:			unsigned[16];
	eflags:			unsigned[32];
	esp:			unsigned[32];
	ss:			unsigned[16];
				unsigned[16];
	};

i386gateFrame:	public	type	packed	{
	public:

	eip:			unsigned[32];
	cs:			unsigned[16];
	extra:			unsigned[16];
	esp:			unsigned[32];
	ss:			unsigned[16];
				unsigned[16];
	};

fpu_t:	public	type	packed	{
	public:
	environ:	fenv_t;
	st:		[8] extended;
	};

interruptVector_t:	public	type	packed	{
	public:
	itype:		ivType_t;
	attribute:	byte;
	selector:	unsigned[16];
	address:	unsigned[32];
	pointer0:	pointer;
	pointer1:	pointer;
	};

ivType_t:	public	type	char = {
	IVT_UNUSED,
	IVT_CALLOUT,
	IVT_EVENT,
	IVT_SPECIAL
	};
/*
	The following functions and primitives provide controls to enable
	and disable hardware interrupts.  Note that on the 80386 these can
	only be used in kernel mode.
 */
enableI:		const	byte = 0xFB;
disableI:		const	byte = 0xFA;
_idle_:			const	signed[16] 		= 0x00EB;

allowInterrupts:	public	() =
	{
	_emit(enableI);
	_emit(_idle_);
	_emit(disableI);
	}

enableAll:	public	() =
	{
	_emit(enableI);
	}

disableAll:	public	() =
	{
	_emit(disableI);
	}

threadLock:	public	type	{
	intrMask:	unsigned[16];

	public:

lock:	() =
	{
	intrMask = _FLAGS;
	_emit(disableI);
	}

unlock:	() =
	{
	_FLAGS = intrMask;
	}

	};

DSbase:		public	paddr_t;
CSbase:		public	paddr_t;
MapBase:	public	paddr_t;
PhysicalBase:	public	paddr_t;

physicalToAddressable:	public	(src: paddr_t) pointer =
	{
	return pointer(src + PhysicalBase);
	}

addressableToPhysical:	public	(src: pointer) paddr_t =
	{
	dest:	paddr_t;

	dest = paddr_t(src) - PhysicalBase;
	return(dest);
	}

addressableToMapped:	public	(src: pointer) paddr_t =
	{
	dest:	paddr_t;

	dest = paddr_t(src) - MapBase;
	return(dest);
	}

mappedToAddressable:	public	(src: paddr_t) pointer =
	{
	dest:	pointer;

	dest = ref char(src + MapBase);
	return dest;
	}

mappedToCodeAddress:	public	(src: paddr_t) paddr_t =
	{
	return paddr_t(src) - KERNEL_VIRTUAL_ADDRESS;
	}

codeAddressToMapped:	public	(src: paddr_t) paddr_t =
	{
	return src + KERNEL_VIRTUAL_ADDRESS;
	}

TIM_INCR:	public	const unsigned = 16384L;
TIM_ROLL:	public	const unsigned = 298295L;
CLOCK_HZ:	public	const int = TIM_ROLL / TIM_INCR;

secondsToTicks:	public	(sec: time_t) tick_t =
	{
	secHi:		unsignedLong;
	secLo:		unsignedLong;
	t:		tick_t;

	secLo = sec & 0xFFF;
	secHi = unsignedLong(sec) >> 12;
	t = (secLo * TIM_ROLL) / TIM_INCR;
	t += (secHi * TIM_ROLL) / (TIM_INCR / 0x1000);
	return t;
	}
