/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	file;

Infinity:	public	real = [ [ 0, 0x80000000 ], INFINITE_EXP ];
NaN:		public	real = [ [ 0, 0xC0000000 ], 0xFFFF ];
Zero:		public	real = [ [ 0, 0 ], 0 ];
One:		public	real = [ [ 0, 0x80000000 ], EXTEND_EXP_BIAS ];

EXTEND_EXP_BIAS:	public	const	int = 0x3fff;
DOUBLE_EXP_BIAS:	public	const	int = 0x3ff;
FLOAT_EXP_BIAS:		public	const	int = 0x3f;
INFINITE_EXP:		public	const	int = 0x7fff;

real:	public	type	packed	{
	public:

	fraction:	[2] unsigned[32];
	exp:		unsigned[16];

fromSigned32:	(i: signed[32]) =
	{
	u:	[2] unsigned;

	u[0] = i;
	u[1] = 0;
	if	(i < 0)
		u[1] = ~0;
	fromSigned64(u);
	}

fromUnsigned32:	(i: unsigned[32]) =
	{
	u:	[2] unsigned;

	u[0] = i;
	u[1] = 0;
	fromUnsigned64(u);
	}
/*
	This function converts a 64-bit signed integer, represented as
	an array of two 32-bit integers, to an extendedReal value.
*/
fromSigned64:	(src: [2] unsigned[32]) =
	{
	x, a, b:	int;

	a = src[0];
	b = src[1];
	if	(b < 0){		/* negative ? */
		a = -a;
		b = ~b;
		if	(a == 0)
			b++;
		x = 0x8000;		// set the sign bit
		}
	else if	(a == 0 &&
		 b == 0){
		*self = Zero;
		return;
		}
	else
		x = 0;
	x += 64;
	if	(b == 0){
		b = a;
		a = 0;
		x -= 32;
		}
	while	(b >= 0){		// normalize b, 
		b <<= 1;
		if	(a < 0)
			b++;
		a <<= 1;
		x--;
		}
	exp = x + EXTEND_EXP_BIAS - 1;
	fraction[0] = a;
	fraction[1] = b;
	}
/*
	This function converts a 64-bit unsigned integer, represented as
	an array of two 32-bit integers, to an extendedReal value.
*/
fromUnsigned64:	(src: [2] unsigned[32]) =
	{
	x, a, b:	int;

	a = src[0];
	b = src[1];
	x = 64;
	if	(b == 0){
		if	(a == 0){
			*self = Zero;
			return;
			}
		b = a;
		a = 0;
		x = 32;
		}
	while	(b >= 0){		// normalize b, 
		b <<= 1;
		if	(a < 0)
			b++;
		a <<= 1;
		x--;
		}
	exp = x + EXTEND_EXP_BIAS - 1;
	fraction[0] = a;
	fraction[1] = b;
	}
/*
	This function converts an extended to a 64 bit signed integer,
	truncating any result.
 */
toSigned64:	(dest: [2] unsigned) =
	{
	x:	int;
	a:	unsigned;
	b:	unsigned;
	c:	unsigned;
	d:	unsigned;
	carry:	int;

	x = (exp & 0x7fff) - (EXTEND_EXP_BIAS - 1);
	if	(x > 63){
		dest[1] = 0x7fffffff;
		dest[0] = ~0;
		if	(exp & 0x8000){
			dest[1]++;
			dest[0] = 0;
			}
		return;
		}

		/* Negative exponent means value is less than 1 */

	if	(x < 0){
		dest[0] = dest[1] = 0;
		return;
		}

	a = fraction[0];
	b = fraction[1];

	while	(x > 32){
		x -= 32;
		a = b;
		b = 0;
		}

	while	(x > 0){
		a >>= 1;
		if	(b & 1)
			a |= 0x80000000;
		b >>= 1;
		x--;
		}

	if	(exp & 0x8000){		/* negative? */
		a = -a;
		b = ~b;
		if	(a == 0)
			b++;
		}
	dest[0] = a;
	dest[1] = b;
	}

toSigned32:	() signed[32] =
	{
	u:	[2] unsigned;

	toSigned64(u);
	return u[0];
	}

toDouble:	(dp: ref byte) =
	{
	x:	int;

	fraction[0] &= ~0x7FF;
	x = exp & 0x7FFF;
	if	(x == 0)
		*self = Zero;
	x -= EXTEND_EXP_BIAS;
	if	(x > 0x3FF)
		*self = Infinity;
	if	(x < -0x3FF)
		*self = Zero;
	xp:	ref unsigned;
	e:	int;

	xp = pointer(dp);
	e = exp;
	xp[0] = 0;
	xp[1] = (e & 0x8000) << 16;		// put in the sign
	e &= 0x7FFF;				// strip the exponent
	if	(e == 0x7FFF){				// infinity or NaN
		xp[1] |= 0x7ff000000;
		xp[1] |= (fraction[1] >> 7) & 0xffffff;	// get the fraction
		xp[0] = fraction[0] >> 11 + fraction[1] << 21;
		return;
		}
	e -= EXTEND_EXP_BIAS;
//	printf("e = %d\n", e);
	if	(e > 0x3ff)
		xp[1] |= 0x7ff00000;		// infinity
	else if	(e >= -0x3ff){
		xp[1] |= (e + DOUBLE_EXP_BIAS) << 20;
		xp[1] |= (fraction[1] >> 11) & 0xfffff;	// get the fraction
		xp[0] = fraction[0] >> 11 + fraction[1] << 21;
		}
//	else
//		underflow yields signed zero
	}

toFloat:	(fp: ref byte) =
	{
	xp:	ref unsigned[32];
	e:	int;

	xp = pointer(fp);
	e = exp;
	*xp = (e & 0x8000) << 16;			// put in the sign
	e &= 0x7FFF;					// strip the exponent
	if	(e == 0x7FFF){				// infinity or NaN
		*xp |= 0x7f000000;
		*xp |= (fraction[1] >> 7) & 0xffffff;	// get the fraction
		return;
		}
	e -= EXTEND_EXP_BIAS;
	if	(e > 0x3f)
		*xp |= 0x7f000000;			// infinity
	else if	(e >= -0x3f){
		*xp |= (e + FLOAT_EXP_BIAS) << 24;
		*xp |= (fraction[1] >> 7) & 0xffffff;	// get the fraction
		}
//	else
//						underflow yields signed zero
	}

fromFloat:	(fp: ref byte) =
	{
	xp:	ref unsigned[32];
	e:	int;

	xp = pointer(fp);
	exp = (*xp >> 16) & 0x8000;			// extract the sign
	e = (*xp >> 24) & 0x7f;				// extract the exponent
	if	(e == 0x7F)				// infinity or NaN
		exp |= 0x7FFF;
	else
		exp |= e + EXTEND_EXP_BIAS - FLOAT_EXP_BIAS;
	fraction[0] = 0;
	fraction[1] = 0x80000000 | ((*xp & 0xffffff) << 7);
	}

fromDouble:	(dp: ref byte) =
	{
	xp:	ref unsigned[32];
	e:	int;

	xp = pointer(dp);
	exp = (xp[1] >> 16) & 0x8000;			// extract the sign
	e = (xp[1] >> 20) & 0x7ff;			// extract the exponent
	if	(e == 0x7FF)				// infinity or NaN
		exp |= 0x7FFF;
	else
		exp |= e + EXTEND_EXP_BIAS - DOUBLE_EXP_BIAS;
	fraction[0] = xp[0] << 11;
	fraction[1] = 0x80000000 | ((xp[1] & 0xfffff) << 11) |
				(xp[0] >> 21);
	}

roundDouble:	() =
	{
	x:	int;

	fraction[0] &= ~0x7FF;
	x = exp & 0x7FFF;
	if	(x == 0)
		return;			// it's a zero, or close
	else if	(x == 0x7FFF)
		return;			// infinity or NaN
	x -= EXTEND_EXP_BIAS;
	if	(x > 0x3FF)
		*self = Infinity;
	if	(x < -0x3FF)
		*self = Zero;
	}

roundFloat:	() =
	{
	x:	int;

	fraction[0] = 0;
	fraction[1] &= ~0x7F;
	x = exp & 0x7FFF;
	if	(x == 0)
		return;			// it's a zero, or close
	else if	(x == 0x7FFF)
		return;			// infinity or NaN
	x -= EXTEND_EXP_BIAS;
	if	(x > 0x3F)
		*self = Infinity;
	if	(x < -0x3F)
		*self = Zero;
	}

negate:	() =
	{
	exp ^= 0x8000;
	}

add:	(b: ref real) =
	{
	addsub(FALSE, b);			// *xp += *yp;
	}
 
subtract:	(b: ref real) =
	{
	addsub(TRUE, b);			// *xp -= *yp;
	}

/*
  algorithm notes
 
  subtraction is done as (a - b) = (a + (-b))
 
  align the lesser operand rightwards to match the greater.
  add them together, realign fraction rounded if overflow.
*/
addsub:	private	(isSub: boolean, b: ref real) =
	{
	align:	unsigned;		/* used for shift counts	*/
	excess:	unsigned;		/* bits 64..95 of calculation	*/
	x:	tempReal;
	y:	tempReal;
	xp:	ref tempReal;
	yp:	ref tempReal;
 
	loadExtended(&x, self);
 	loadExtended(&y, b);

	y.sign ^= isSub;		// the transformation of the sign of
					// y means that from here on we are
					// doing an addition
	isSub = y.sign ^ x.sign;	// isSub now is TRUE if subtraction is required

	if	(x.exp == INFINITE_EXP)
		return;			// keep x
	if	(y.exp == INFINITE_EXP){
		storeExtended(self, &y);
		return;
		}

	xp = &x;
	yp = &y;
	if	(x.exp < y.exp){
		yp = &x;
		xp = &y;
		}
	else if	(x.exp == y.exp){
		if	(x.frac[1] < y.frac[1] ||
			 (x.frac[1] == y.frac[1] &&
			  x.frac[0] < y.frac[0])){
			yp = &x;
			xp = &y;
			}
		}

		// xp is now the larger, yp the smaller number (or else the
		// exponents are equal).

	align = xp->exp - yp->exp;
	if	(yp->exp == 0 ||
		 align >= 64){

			// y is too small to matter

		storeExtended(self, xp);
		return;
		}

	excess = 0;
	if	(align >= 32){
		excess = yp->frac[0];
		yp->frac[0] = yp->frac[1];
		yp->frac[1] = 0;
		align -= 32;
		}

		// align is now < 32

	if	(align){
		excess += yp->frac[0] << (32 - align);
		yp->frac[0] >>= align;
		yp->frac[0] += yp->frac[1] << (32 - align);
		yp->frac[1] >>= align;
		}

	if	(isSub){
		x:	int;

		x = 0;
		xp->frac[1] -= yp->frac[1];		// can't underflow

			// if the low order word underflows, then that means
			// the high word must be definitely bigger in xp,
			// otherwise yp would be bigger.

		xp->frac[0] -= yp->frac[0];
		if	(_FLAGS & 1)
			xp->frac[1]--;			// still can't underflow
		if	(signed(excess) < 0){		// need to propagate a 
							// carry.
			if	(xp->frac[0] == 0)
				xp->frac[1]--;
			xp->frac[0]--;
			}

			// Check for zero result

		if	(xp->frac[1] == 0 &&
			 xp->frac[0] == 0 &&
			 signed(excess) >= 0){
			*self = Zero;
			return;
			}

			// now normalize

		while	(signed(xp->frac[1]) >= 0){
			xp->exp--;
			if	(xp->exp == 0){
				*self = Zero;
				return;
				}
			xp->frac[1] <<= 1;
			if	(signed(xp->frac[0]) < 0)
				xp->frac[1]++;
			xp->frac[0] <<= 1;
			if	(signed(excess) < 0){
				xp->frac[0]++;
				excess = 0;
				}
			}
		}
	else	{		// true addition
		x:	int;
		y:	int;

		x = 0;
		y = 0;
		xp->frac[0] += yp->frac[0];
		if	(_FLAGS & 1)
			x = 1;
		xp->frac[1] += yp->frac[1];
		if	(_FLAGS & 1)
			y = 1;
		xp->frac[1] += x;
		if	(y || (x && xp->frac[1] == 0)){
			if	(xp->frac[0] & 1)
				excess = ~0;	// force round
			xp->frac[0] >>= 1;
			if	(xp->frac[1] & 1)
				xp->frac[0] |= 0x80000000;
			xp->frac[1] >>= 1;
			xp->frac[1] |= 0x80000000;
			xp->exp++;
			}
		if	(signed(excess) < 0){		// round needed
			xp->frac[0]++;
			if	(xp->frac[0] == 0){
				xp->frac[1]++;
				if	(xp->frac[1] == 0){
					xp->exp++;
					xp->frac[1] = 0x80000000;
					}
				}
			}
		if	(xp->exp >= INFINITE_EXP){
			*self = Infinity;
			exp |= xp->exp << 15;
			return;
			}
		}
	storeExtended(self, xp);
	}

/*
  Multiplication first involves checks for the special cases of zero
  and infinity.  If these are encountered, an exception may be raised
  and a special value is returned.  Otherwise, a subroutine is called
  to perform a fixed point 64-bit multiplication.  This is then
  normalised and rounded, and the result and flags set.
 
 
  Algorithm:
 
	Let N = 2^32

	The problem is to multiply ab * uv where each letter represents 
	32 bits.
 
	  result = a*u*N*N + a*v*N + u*b*N + b*v
 
	and so, fortunately, the calculation can be done in terms of the
	unsigned 32-bit integers a, b, u and v.
 
	A 96 bit intermediate result is used, from which the low order
	bits will be dropped after normalization and rounding.  This
	provides the necessary 2 guard bits, guaranteeing an accurate
	multiply.

	The accumulation of products begins with the least significant.
	In this way, the lowest order 32 bits can be dropped immediately.
	The specialized multiply and add routines are used to accumulate the
	necessary result.
 
	The multiplication terms are aligned as follows:
 
	       |b v
	  +   a|v
	  +   b|u
	  + a u|
	    - - - -
	   (64 bits)
 */
multiply:	(b: ref real) =
	{
	x:	tempReal;
	y:	tempReal;
	e:	[2] unsigned;
	d:	[3] unsigned;

	loadExtended(&x, self);
	loadExtended(&y, b);

	x.sign ^= y.sign;			// calculate the result sign
	if	(x.exp == INFINITE_EXP)
		return;				// leave x as infinity
	if	(y.exp == INFINITE_EXP){
		y.sign = x.sign;
		storeExtended(self, &y);
		return;
		}

	if	(x.exp == 0 ||
		 y.exp == 0){
		*self = Zero;
		return;
		}

	result_exp:	long;

	result_exp = x.exp + (y.exp - (EXTEND_EXP_BIAS - 1));
	if	(result_exp < 0){
		*self = Zero;
		return;
		}
	if	(result_exp >= INFINITE_EXP){
		*self = Infinity;
		return;
		}

	x.exp = result_exp;

	mul64(e, x.frac[0], y.frac[0]);		// b*v
	d[2] = 0;
	d[1] = 0;
	d[0] = e[1];				// ignore the low order bits
	mul64(e, x.frac[1], y.frac[0]);		// a*v
	add96(d, e);
	mul64(e, x.frac[0], y.frac[1]);		// b*u
	add96(d, e);
	mul64(e, x.frac[1], y.frac[1]);		// a*u
	add64(d + 1, e);

		// Normalize the result (no more than one shift needed)

	if	(signed(d[2]) >= 0){	// no high bit, normalize
		d[2] <<= 1;
		if	(signed(d[1]) < 0)
			d[2]++;
		d[1] <<= 1;
		if	(signed(d[0]) < 0)
			d[1]++;
		d[0] <<= 1;
		x.exp--;
		}

		// Round the low order bit

	if	(signed(d[0]) < 0){
		d[1]++;
		if	(d[1] == 0){
			d[2]++;
			if	(d[2] == 0){
				d[2] = 0x80000000;
				x.exp++;
				}
			}
		}
	x.frac[1] = d[2];
	x.frac[0] = d[1];
	storeExtended(self, &x);
	}
/*
	*numP /= *dvrP;
 
  algorithm notes for divide.
 
  We are looking for the solution to the ratio:
 
	aN + b
       -------- = (vN + w) / K
	pN + q
 
  Where  a, p, v are in the range N/2 .. N-1 and the other lower-case
  symbols range from  0..N-1 , and K is either NN or NN/2.
 
  The machine provides a primitive for NN / N division.
 
  Knuth proposed a method for double length divide which uses an
  approximation:
 
      AN + B   AN + B     PN      AN + B
      ------ = ------ x ------ =  ------ x ( 1 - Q/PN + (Q/PN)^2 ...)
      PN + Q     PN     PN + Q      PN
 
  We will use this method, with some modifications for the quad length.
  Its theme is that using PN as the divisor gives an order(N) estimate.
 
  An important first step is to remove the possibility that ab > pq,
  since that makes the process liable to overflow when dividing.  The
  method for this is:
 
  transform PQ = pq, where P is rounded up and Q = N-q, unless
			Q == 0, treated as a special case.
 
  if ab >= pq, then AB = ab - pq, and excess = 1,
	       else AB = ab, and excess = 0;
 
  So now we have guaranteed P0 > AB.
 
  Let us make a first estimate, V:
 
     AB = PQ * V + error,	with	V = AB / PN
 
	  = (PN + Q) * (AB / PN) + error
	  = AB  + Q*V + error
 
     error = - Q * V
 
  How accurate is this ?  Well,  Q <= N-1, and V < 1, so error > 1-NNN.
  This looks promising: however, it is not likely to be fast enough if we
  work V through to maximum precision and then multiply by QRS, needing an
  extra 2 divisions and around 10 multiplies.  So we will need to work with
  a value of V rounded to 16 bits.  Further, we will always round upwards
  because it is best to plan for a fixed sign of error: if error could be
  either positive or negative it would be a bit faster, but many sections
  of code will be double sized.
 
  Now, rounding V upwards gives us 1/N maximum error in V, for a new error
  bound of
 
     0 > error > 1 - NNN - PNN
 
  and we can calculate error by using
 
     error = (pqrs * V) - abcd.
 
  V is then added into the quotient.
 
  We can now see how to make the algorithm recursive.  We substitute
  ABCD = error, and repeat.  However, note this time that 0 <= A <= 1,
  so that the next step is to reduce:
 
    while (-error >= P00)
	error += pqrs/N,  excess:V -= 1
 
  so that excess:V accumulate the top 18 bits of the eventual quotient,
  while error is reduced until it requires merely triple-word value.
  The next 16 bits of quotient, W, are then estimated as  error/P, and
  W is rounded up and added to the quotient  excess:V:W.  The process
  continues until excess:V:W:X:Y are accumulated, which are then aligned
  and delivered.
 
  What is the accumulated accuracy ?  Well, each stage has an accurate
  feedback where error is calculated using the original pqrs, and that
  prevents errors building up from stage to stage.  However, the final
  stage will have an error of up to (P+N)/N, or two least bits, biased
  larger.  Furthermore, if abcd was originally near NNNN/2, while
  pqrs was near maximal NNNN, then the result will need normalisation
  shifts so the normalised error can be up to 4 least bits, biased
  away from zero.
*/
divide:	(b: ref real) =
	{
	*ref extended(self) /= *ref extended(b);
/*
	num:		tempReal;
	dvr:		tempReal;
	result:		tempReal;
	guardBits:	unsigned;
	excess:		byte;
	x:		int;

	loadExtended(&num, self);
	loadExtended(&dvr, b);
	result.sign = num.sign ^ dvr.sign;

	if	(dvr.exp >= INFINITE_EXP){
		*self = Zero;
		exp = result.sign << 15;
		return;
		}

	if	(num.exp >= INFINITE_EXP){
		num.sign = result.sign;
		storeExtended(self, &num);
		}

	if	(num.exp == 0){
		if	(dvr.exp == 0)
			*self = NaN;		// 0.0 / 0.0 -> NaN
		return;
		}

	if	(dvr.exp == 0){
		loadExtended(&num, &Infinity);
		num.sign = result.sign;
		storeExtended(self, &num);
		return;
		}

	x = num.exp + (EXTEND_EXP_BIAS - 1);
	if	(x < dvr.exp){

			// underflow

		*self = Zero;
		exp = result.sign << 15;
		return;
		}
	x -= dvr.exp;
	if	(x >= INFINITE_EXP){
		loadExtended(&num, &Infinity);
		num.sign = result.sign;
		storeExtended(self, &num);
		return;
		}

	result.exp = x;

		// Avoid the case where the numerator fraction is greater
		// than the denominator fraction.

	excess = 0;
	if	(num.frac[1] > dvr.frac[1] ||
		 (num.frac[1] == dvr.frac[1] &&
		  num.frac[0] >= dvr.frac[0])){
		excess = 1;

			// record any borrow needed
		if	(num.frac[0] < dvr.frac[0])
			num.frac[1]--;
		num.frac[0] -= dvr.frac[0];
		num.frac[1] -= dvr.frac[1];
		}

	r, q, wm, wl:	unsigned;

//	printf("dvr = [ [ %x, %x ], %x ]\n", dvr.frac[0], dvr.frac[1], dvr.exp);
//	printf("num = [ [ %x, %x ], %x ]\n", num.frac[0], num.frac[1], num.exp);
	div64(&wm, &r, num.frac, dvr.frac[1]);

	w:	[2] unsigned;
	w[0] = 0;
	w[1] = r;
	div64(&wl, &r, w, dvr.frac[1]);

	w[1] = r;
	div64(&guardBits, &r, w, dvr.frac[1]);

	w[1] = dvr.frac[0] >> 1;
	w[0] = dvr.frac[0] << 31;

	div64(&q, &r, w, dvr.frac[1]);

	mul64(result.frac, q, wm);

	result.frac[0] = result.frac[1] << 1 + result.frac[0] >> 31;
	result.frac[1] >>= 31;

	result.frac[0] += wl;
	if	(_FLAGS & 1)
		result.frac[1]++;

	result.frac[1] += wm;
	if	(excess){
		r = result.frac[0];
		result.frac[0] >>= 1;
		if	(result.frac[1] & 1)
			result.frac[0] |= 0x80000000;
		result.frac[1] >>= 1;
		if	(r & 1){
			result.frac[0]++;
			if	(result.frac[0] == 0)
				result.frac[1]++;
			}
		result.exp++;
		result.frac[1] |= 0x80000000;
		}
	else if	(signed(guardBits) < 0){
		result.frac[0]++;
		if	(result.frac[0] == 0)
			result.frac[1]++;
		}

	if	(result.frac[1] == 0){
		result.frac[1] = result.frac[0];
		result.frac[0] = 0;
		result.exp -= 32;
		}

		// This normalizes the result.

	while	(signed(result.frac[1]) >= 0){
		result.frac[1] <<= 1;
		if	(signed(result.frac[0]) < 0)
			result.frac[1]++;
		result.frac[0] <<= 1;
		result.exp--;
		}
	storeExtended(self, &result);
 */
	return;
	}
/*
  return  10^p;

Calculate 10 raised to power.  A lookup table is used for values from
10 thru 10^7, then this is augmented by multiplying with table entries
for 10^8/16/32/64/128/256, which allows any power up to 511.  The usual
range of double precision is 10e308 but pow10 has a wider range so that
overflow does not occur when used to convert strings near the limits,
where powers beyond the limit may be wanted.

Only powers in the range 0..511 are computed.
*/
pow10:	public	(p: unsigned) =
	{
	if	(p == 0){
		*self = One;
		return;
		}

	if	(p > 511){
		*self = Infinity;
		return;
		}

	fromSigned32(smallPowers[p & 0x7]);

	if	(p < 8)
		return;

	if	(p & 0x008)
		multiply(&e8);

	if	(p & 0x010)
		multiply(&e16);

	if	(p & 0x020)
		multiply(&e32);

	if	(p & 0x040)
		multiply(&e64);

	if	(p & 0x080)
		multiply(&e128);

	if	(p & 0x100)
		multiply(&e256);

	}

	};

smallPowers:	const   [8] long =
		[ 1, 10, 100, 1000, 10000, 100000, 1000000, 10000000 ];

e8:	public	const	real = [ [ 0,          0xBEBC2000 ], 0x4019 ];
e16:	public	const   real = [ [ 0x04000000, 0x8E1BC9BF ], 0x4034 ];
e32:		const   real = [ [ 0x2B70B59E, 0x9DC5ADA8 ], 0x4069 ];
e64:		const   real = [ [ 0xFFCFA6D5, 0xC2781F49 ], 0x40D3 ];
e128:		const   real = [ [ 0x80E98CDF, 0x93BA47C9 ], 0x41A8 ];
e256:		const   real = [ [ 0x9DF9DE8C, 0xAA7EEBFB ], 0x4351 ];

tempReal:	public	type	{
	public:

	frac:		[2] unsigned[32];
	exp:		unsigned[16];
	sign:		unsigned[16];
	};

loadExtended:	(dest: ref tempReal, src: ref real) =
	{
	memCopy(dest->frac, src, 8);	
						// copy the sign/exponent
	dest->exp = src->exp;
	dest->sign = dest->exp >> 15;		// extract the sign bit
	dest->exp &= 0x7FFF;			// and strip it
	}

storeExtended:	(dest: ref real, src: ref tempReal) =
	{
	memCopy(dest, src->frac, 8);
	dest->exp = src->exp | (src->sign << 15);
	}

mul64:	(r: ref unsigned, a: unsigned, b: unsigned) =
	{
	_EAX = a;
	_EDX = b;
	_emit(0xf7, 0xe2);	// mul eax,edx
	_ECX = _EAX;
	r[1] = _EDX;
	r[0] = _ECX;
	}

div64:	(q: ref unsigned, r: ref unsigned, n: ref unsigned, d: unsigned) =
	{
	_EDX = n[1];
	_EAX = n[0];
	_ECX = d;
	_emit(0xf7, 0xf1);	// div eax:edx,ecx
	_ECX = _EAX;
	*q = _ECX;
	*r = _EDX;
	}

add96:	(d: ref unsigned, s: ref unsigned) =
	{
	d[1] += s[1];
	if	(_FLAGS & 1)
		d[2]++;
	d[0] += s[0];
	if	(_FLAGS & 1){
		d[1]++;

			// Can't use the carry flag, must check for overflow
			// explicitly (INC instruction doesn't set carry)

		if	(d[1] == 0)
			d[2]++;
		}
	}

add64:	(d: ref unsigned, s: ref unsigned) =
	{
	d[1] += s[1];
	d[0] += s[0];
	if	(_FLAGS & 1)
		d[1]++;
	}

/*
Convert a string to a double precision real.  The syntax of the
string must be:

        float    ::= [realnum] [exponent] [suffix]

        realnum  ::= {digit [digit]* ['.' [digit]* ]} | {'.' digit [digit]*}

        exponent ::= 'e'|'E' [sign] digit [digit]*
 
        suffix   ::= 'F' | 'L'
 
"src"   is a pointer to the string from which characters are scanned.
        Leading spaces are NOT accepted.  The string will not be changed.
	The string ends with a \0.
 
The digits must be decimal.

The format of the string has already been verified by the scanner.  There is
no need to do format checking.

If the number is too large or too tiny then the
result is infinity or zero.
 
METHOD
------
 
The conversion proceeds in two stages.  Firstly, the decimal strings
for fraction and exponent must be captured.
 
The fraction is held as a 64-bit unsigned integer (18 decimals of
precision), with separate sign.  Digits beyond the 18th are truncated.
 
The exponent is held as a short integer in binary format, and is adjusted
to note the position of the decimal point in the fraction so that the
"fraction" is normalised as an integer with decimal point to the right.
 
When both fraction and exponent have been captured, the second stage is
to combine them.  This is done with the formula:

        result = 10^(exponent) * fraction * sign.
 
If the result overflows + or - infinity will be returned.  If the result is
an underflow, zero is returned.
 
The following diagram may be helpful with undestanding the relations
between the variables:

   000012345789012345.098765432109876E+99
      |---decimals-->|
      |--------------.----digits---->|   not counting the '.'
 
Decimals are counted negative if the '.' is left of the first digit.
Digits are positive unless no non-zero digit is ever seen.
*/
stringToFloat:	public	(r: ref real, src: ref char) =
	{
	exponent, decimals, digits:		int;
	precision:				char;
	expNegative, pointSeen:			boolean;
	actual:					int;
	c:					char;
	pow, result:				real;
	fraction:				[2] unsigned;

	pointSeen = FALSE;
	decimals = 0;
	digits = 0;

	fraction[0] = 0;
	fraction[1] = 0;
	for	(;;){
		c = *src++;
		if	(c == '.'){
			if	(pointSeen)
				break;
			pointSeen = TRUE;
			decimals = digits;
			continue;
			}
		if	(!isdigit(c))
			break;
		c -= '0';
		if	(digits == 0){
			if	(c == 0)
				decimals--;
			else	{
				fraction[0] = c;
				actual = digits = 1;
				}
			}
		else	{
			digits++;
			if	(digits > 9){
				if	(fraction[1] >= 0xCCC0000)
					continue;
				fraction[1] *= 10;
				_EAX = fraction[0];
				_EDX = 10;
				_emit(0xf7, 0xe2);	// mul eax,edx
				fraction[0] = _EAX;
				fraction[1] += _EDX;
				fraction[0] += c;
				if	(_FLAGS & 1)
					fraction[1]++;
				}
			else
				fraction[0] = fraction[0] * 10 + c;
			actual = digits;
			}
		}

		// If there are no non-zero digits, the answer must be zero
		// and we don't care what the exponent is.

	if	(digits == 0){
		*r = Zero;
		return;
		}
	if	(!pointSeen)
		decimals = digits;

	exponent = 0;
	expNegative = FALSE;
	if	(c == 'E' || c == 'e'){
		c = *src++;
		if	(c == '-'){
			expNegative = TRUE;
			c = *src++;
			}
		else if	(c == '+')
			c = *src++;
		for	(;;){
			if	(!isdigit(c))
				break;
			c -= '0';
			exponent = exponent * 10 + c;

				// large exponents become infinity

			if	(exponent > 9999)
				exponent = 9999;
			c = *src++;
			}
		if	(expNegative)
			exponent = -exponent;
		}
	precision = tolower(c);		// in case it's 'f' or 'l'

	x:	int;

	x = decimals + exponent;

		// Check for overflow and underflow

	if	(x >= 308){
		*r = Infinity;
		return;
		}
	if	(x <= -306){
		*r = Zero;
		return;
		}

	exponent += decimals - actual;

	result fromUnsigned64(fraction);
	if	(exponent){
		pow pow10(abs(exponent));
		if	(exponent < 0)
			result divide(&pow);
		else
			result multiply(&pow);
 		}

	if	(precision == 'f')
		result roundFloat();
	else if	(precision != 'l')
		result roundDouble();
	*r = result;
	return;
}

hexToFloat:	public	(r: ref real, src: ref char) =
	{
	exponent, decimals, digits:		int;
	precision:				char;
	expNegative, pointSeen:			boolean;
	actual:					int;
	c:					char;
	pow, result:				real;
	fraction:				[2] unsigned;

	pointSeen = FALSE;
	decimals = 0;
	digits = 0;

	fraction[0] = 0;
	fraction[1] = 0;
	while	(*src == '0')
		src++;
	for	(;;){
		c = *src++;
		if	(c == '.'){
			if	(pointSeen)
				break;
			pointSeen = TRUE;
			decimals = digits;
			continue;
			}
		if	(!isxdigit(c))
			break;
		if	(isdigit(c))
			c -= '0';
		else
			c = tolower(c) - 'a' + 10;
		if	(fraction[1] & 0xF0000000)
			continue;
		if	(digits == 0 && c == 0){
			decimals -= 4;
			continue;
			}
		digits += 4;
		fraction[1] <<= 4;
		fraction[1] += fraction[0] >> 28;
		fraction[0] <<= 4;
		fraction[0] += c;
		actual = digits;
		}

		// If there are no non-zero digits, the answer must be zero
		// and we don't care what the exponent is.

	if	(digits == 0){
		*r = Zero;
		return;
		}
	if	(!pointSeen)
		decimals = digits;

	exponent = 0;
	expNegative = FALSE;
	if	(c == 'P' || c == 'p'){
		c = *src++;
		if	(c == '-'){
			expNegative = TRUE;
			c = *src++;
			}
		else if	(c == '+')
			c = *src++;
		for	(;;){
			if	(!isdigit(c))
				break;
			c -= '0';
			exponent = exponent * 10 + c;

				// large exponents become infinity

			if	(exponent > 99999)
				exponent = 99999;
			c = *src++;
			}
		if	(exponent == 97081){	// magic value!
			if	(expNegative)
				*r = NaN;
			else
				*r = Infinity;
			return;
			}
		if	(expNegative)
			exponent = -exponent;
		}
	precision = tolower(c);		// in case it's 'f' or 'l'

	x:	int;

	exponent += decimals;

		// Check for overflow and underflow

	if	(exponent > 0x3FFF){
		*r = Infinity;
		return;
		}
	if	(exponent <= -0x3FFF){
		*r = Zero;
		return;
		}

	exponent -= actual;
	result fromUnsigned64(fraction);
	result.exp += exponent;
	if	(precision == 'f')
		result roundFloat();
	else if	(precision != 'l')
		result roundDouble();
	*r = result;
	return;
}
