/*
	Copyright (c) 1993 by Robert Jervis
	All rights reserved.

	Permission to use, copy, modify and distribute this software is
	subject to the license described in the READ.ME file.
 */
include	alys, error, file, filesys, format, environ, heap, string;

include	errmsg;
include	hash;
include	parser;
include	ptree, backend, xtree, xstmt;
include	sbuffer;
include	scanner;
include	target;
include	value;
include	image;
include	gencode;
include	asmcode;
include	link;
include	mapout;
include	types;

uRecType:	public	type	char = {
/*
	U_HEADER = 0xa3,
 */
	U_PROJECT,
	U_LIBUNIT,
	U_SRCUNIT,
	U_INCLUDE,
	U_SYMBOL,
/*
	U_MESSAGE,
 */
	U_PARAMETER,
	U_VALUE,
	U_XVALUE,
	U_VFIXUP,
	U_RFIXUP,
	U_DFIXUP,
	U_LFIXUP,
	U_LITERAL,
/*
	U_LINE,
	U_MSGLIST,
 */
	U_ERROR_TYPE,
	U_VOID_TYPE,
	U_TYPE_TYPE,
	U_NUMBER_TYPE,
	U_NAMED_TYPE,
	U_STRUCT_TYPE,
	U_PTR_TYPE,
	U_ARRAY_TYPE,
	U_FUNCTION_TYPE,
	U_DESCRIPTOR_TYPE
	};

DebugList:	public	boolean;

projectTypes:	public	type	byte = {
	PT_RUN,
	PT_DRIVER,
	PT_LIBRARY
	};

outputModes:	public	type	char = {
	OM_ASM =	0x04,
	OM_H = 		0x08,
	OM_C =		0x10,
	OM_MAKE =	0x20,
	};

start:	entry	() =
	{
	defineLoadFuncs(U_LIBUNIT, loadFunction(&libraryUnit_s.load),
					sizeof libraryUnit_s);
	defineLoadFuncs(U_INCLUDE, loadFunction(&include_s.load),
					sizeof include_s);
	defineLoadFuncs(U_SYMBOL, loadFunction(&symbol_s.load),
					sizeof symbol_s);
	defineLoadFuncs(U_VOID_TYPE, loadFunction(&void_z.load),
					sizeof void_z);
	defineLoadFuncs(U_NUMBER_TYPE, loadFunction(&number_z.load),
					sizeof number_z);
	defineLoadFuncs(U_NAMED_TYPE, loadFunction(&named_z.load),
					sizeof named_z);
	defineLoadFuncs(U_STRUCT_TYPE, loadFunction(&struct_z.load),
					sizeof struct_z);
	defineLoadFuncs(U_PTR_TYPE, loadFunction(&ptr_z.load),
					sizeof ptr_z);
	defineLoadFuncs(U_ARRAY_TYPE, loadFunction(&array_z.load),
					sizeof array_z);
	defineLoadFuncs(U_FUNCTION_TYPE, loadFunction(&function_z.load),
					sizeof function_z);
	defineLoadFuncs(U_PARAMETER, loadFunction(&parameter_s.load),
					sizeof parameter_s);
	defineLoadFuncs(U_DESCRIPTOR_TYPE, loadFunction(&descriptor_z.load),
					sizeof descriptor_z);
	}

Project:	public	project_s;

project_s:	public	type	{
	public:

	name:		[:] char;
	projectType:	projectTypes;
	prjFile:	[:] char;
	runFile:	[:] char;
	mapFile:	[:] char;
	lodFile:	[:] char;
	stackSize:	addr_t;
	globalChange:	boolean;
	smartLink:	boolean;
	projectUnit:	ref unit_s;
	machineUnit:	ref unit_s;
	units:		* unit_s;
	outputAs:	outputModes;
	outputFd:	ref stream;
	asmFd:		ref stream;
	varArgsType:	ref type_s;
	trapPtrType:	ref type_s;
	allocSymbol:	ref symbol_s;
	exceptContextType:ref type_s;
	exceptFrameType:ref type_s;
	lockType:	ref type_s;
	threadValue:	ref value;
	exceptionFrame:	ref symbol_s;
	selfSymbol:	ref symbol_s;
	startupSymbol:	ref symbol_s;
	libraryPath:	pathList;
	optimize:	boolean;
	projectSave:	boolean;
	saveUnit:	ref unit_s;

constructor:	(nm: [:] char, pt: projectTypes, inc: [:] char, smart: boolean) =
	{
	libBuf:		[MAXPATH] char;
	libPath:	[:] char;
	i:		int;

	name = stripExtension(nm);
//	printf("name = %S\n", name);
	projectType = pt;
//	prjFile = heapFile(name, ".prp");
	libPath = libBuf;
	switch	(pt){
	case	PT_RUN:
		mapFile = heapFile(name, ".map");
		runFile = heapFile(name, ".run");
		stackSize = 4096;
		if	(|inc == 0)
			libraryPath useEnvironment("PARASOL_LIB");
		else
			libraryPath useString(inc);
		break;

	case	PT_DRIVER:
		mapFile = heapFile(name, ".map");
		lodFile = heapFile(name, ".lod");
		stackSize = 0;
		if	(|inc == 0)
			libraryPath useEnvironment("DRIVER_LIB");
		else
			libraryPath useString(inc);
		break;

	case	PT_LIBRARY:
		if	(|inc == 0)
			libraryPath useEnvironment("PARASOL_LIB");
		else
			libraryPath useString(inc);
		break;
		}
//	clock = 0;
	optimize = TRUE;
	smartLink = smart;
	projectUnit = sourceUnit_s create(hash(name), 
						heapFile(name, ".p"));
//	messages = messageList create();
	globalChange = TRUE;
	}
/*
setLibrary:	(l: [:] char) =
	{
	libraryPath = new [|l] char;
	libraryPath [:]= l;
	}
 */
addUnit:	(n: ref unit_s) =
	{
	u:	ref unit_s;
	last:	ref unit_s;

	last = 0;
	for	(u = units; u; last = u, u = u->next){
		if	(n->name compare(u->name) < 0)
			break;
		}
	n->next = u;
	if	(last)
		last->next = n;
	else
		units = n;
	}

locateUnit:	(n: ref identifier) ref unit_s =
	{
	u:	ref unit_s;

	for	(u = units; u; u = u->next)
		if	(n == u->name)
			return u;
	return 0;
	}

save:	() =
	{
	ifile:		imageFile;
	i:		int;

	i = ifile create(prjFile, FA_READ|FA_WRITE);
	if	(i){
		printf("Could not create project file: '%S'\n", prjFile);
		return;
		}
	if	(ifile beginRecord(U_PROJECT, self, sizeof *self)){
		u:	ref unit_s;

		ifile putstring(name);
		ifile integer(globalChange);
		ifile address(projectUnit);
		ifile address(units);
//		ifile putstring(libraryPath);
		ifile putstring(runFile);
		ifile putstring(lodFile);
		ifile putstring(mapFile);
		ifile integer(outputAs);
		ifile integer(optimize);
		ifile uinteger(stackSize);
		for	(u = units; u; u = u->next){
			printf("%-32S sp\r", u->name spelling());
			u saveProject(&ifile);
			}
		printf("\n");
		}
	ifile close();
	}

build:	(om: outputModes) =
	{
	changed:	boolean;
	parsedAny:	boolean;
	u:		ref unit_s;
	s:		ref symbol_s;

	outputAs = om;
	for	(u = units; u; u = u->next)
		for	(s = u->symbols; s; s = s->next)
			s clearChanges();

	for	(changed = FALSE, u = units; u; u = u->next)
		if	(u hasChanged())
			changed = TRUE;
	if	(changed == FALSE)
		return;
	parsedAny = FALSE;
	setDateTime();
	do	{
		changed = FALSE;
		for	(u = units; u; u = u->next){
			CurrentContext.obj = u;
			if	(machineUnit == 0 &&
				 u->name isSpelled("machine"))
				machineUnit = u;
			if	(u hasChanged()){
				parsedAny = TRUE;
				changed = TRUE;
//				messages purge(u, PH_CARVE);
				printf("%-32S pa\r", u->name spelling());
				u parse();
				}
			}
//		printf("changed = %d\n", changed);
		}
		while	(changed);
	printf("\n");
	if	(!(parsedAny || globalChange))
		return;
/*
	if	(om & OM_MAKE){
		outputMakefile();
		om &= ~OM_MAKE;
		}
 */
	varArgsType = 0;
	machineUnit locateMagicSymbols();
	CurrentContext.phase = PH_INTERFACE;
	for	(u = units; u; u = u->next){
//		printf("%-32S ri\r", u->name spelling());
		u reconnectInterfaces();
		}
//	printf("\n");
	for	(u = units; u; u = u->next){
//		printf("%-32S ci\r", u->name spelling());
		u constructInterfaces();
		}
//	printf("\n");
/*
	if	(om & OM_H){
		for	(u = units; u; u = u->next){
			printf("%-32S hd\r", u->name spelling());
			u outputHeader();
			}
		printf("\n");
		return;
		}
 */
	if	(om & OM_ASM){
		printf("%-32S as\n", projectUnit->name spelling());
		projectUnit constructValues();
		return;
		}
	else if	(om & OM_C){
		printf("%-32S cf\n", projectUnit->name spelling());
		projectUnit constructValues();
		return;
		}
	for	(u = units; u; u = u->next){
		printf("%-32S cv\r", u->name spelling());
		u constructValues();
		}
	printf("\n");
	if	(projectType == PT_LIBRARY){
		for	(u = units; u; u = u->next){
			if	(u == projectUnit)
				continue;
			printf("%-32S us\r", u->name spelling());
			u saveLibrary();
			}
		printf("\n");
		}
	else	{
		RunFile link();
		if	(projectType == PT_RUN){
			printf("%-32S rn\n", runFile);
			RunFile write();
			}
		else	{
			printf("%-32S ld\n", lodFile);
			RunFile writeDriver();
			}
		printf("%-32S mp\n", mapFile);
		writeMap(mapFile);
		}
	}
/*
outputMakefile:	() =
	{
	fd:	stream;
	i:	int;
	u:	ref unit_s;

	if	(fd create([ "makefile.mak" ], AR_READ|AR_WRITE)){
		printf("Couldn't create 'makefile.mak'\n");
		return;
		}
	fd printf("%S.exe: %S.prp ", name, name);
	for	(u = units; u; u = u->next)
		fd printf("%S.obj ", u->name spelling());
	fd printf("\n\ttlink @pmake.lnk\n\n%S.prp: ", name);
	for	(u = units; u; u = u->next)
		if	(u isSourceUnit())
			fd printf("%S.p ", u->name spelling());
	fd printf("\n\tparasol -c -h %S\n", name);
	fd close();
	if	(fd create([ "pmake.lnk" ], AR_READ|AR_WRITE)){
		printf("Couldn't create 'makefile.mak'\n");
		return;
		}
	fd printf("p0");
	for	(u = units; u; u = u->next)
		if	(u isSourceUnit())
			fd printf("+%S", u->name spelling());
	fd printf(",%S,%S/m,plib\n", name, name);
	fd close();
	}
 */
	};


unit_s:	public	type	inherit	scope_s {
	public:
	next:		* unit_s;
	name:		ref identifier;
	includes:	* include_s;
	changed:	boolean;
	codeSize:	addr_t;
	dataSize:	addr_t;
	bssSize:	addr_t;
	index:		unsigned;
	built:		boolean;

isSourceUnit:	dynamic	() boolean =
	{
	return FALSE;
	}

create:	factory	(nm: ref identifier) ref unit_s =
	{
	fpath:	[:] char;

	fpath = heapFile(nm spelling(), ".p");
//	printf("fpath = '%S'\n", fpath);
	if	(FileSystem access(fpath, AR_READ) == SUCCESS)
		return sourceUnit_s create(nm, fpath);

	s:	[:] char;

	s = Project.libraryPath search(nm spelling(), ".u", 0);
	if	(|s)
		return libraryUnit_s create(nm, s);
	else
		return sourceUnit_s create(nm, fpath);	// bad unit
	}

saveLibrary:	dynamic	() =
	{
	}

saveProject:	dynamic	(ref imageFile) =
	{
	}

locateMagicSymbols:	() =
	{

		// Only the machine unit does this code

	s:	ref symbol_s;
	vat:	ref identifier;
	efi:	ref identifier;
	eci:	ref identifier;
	efoi:	ref identifier;
	ti:	ref identifier;
	si:	ref identifier;
	ai:	ref identifier;
	tci:	ref identifier;
	tpi:	ref identifier;
	efmi:	ref identifier;
	lti:	ref identifier;

	vat = 	hash("varArgs");
	efi = 	hash("__exceptionFrame");
	eci = 	hash("exceptionContext");
	efmi =	hash("exceptionFrame");
	tpi =   hash("_Thread");
	tci =   hash("threadContext");
	ti = 	hash("trap");
	si = 	hash("__startup__");
	ai = 	hash("alloc");
	lti =   hash("lock");
	for	(s = symbols; s; s = s->next)
		if	(s->name == vat)
			Project.varArgsType = s->dtype;
		else if	(s->name == efi)
			Project.exceptFrameType = s->dtype;
		else if	(s->name == lti)
			Project.lockType = s->dtype;
		else if	(s->name == eci)
			Project.exceptContextType = s->dtype;
		else if	(s->name == tpi)
			Project.threadValue = s->currentValue;
		else if	(s->name == tci)
			Project.exceptionFrame = 
					s->dtype lookupLocal(efmi, s->dtype);
		else if	(s->name == ti)
			Project.trapPtrType = ptr_z 
						create(FALSE, s->dtype, 0);
		else if	(s->name == si)
			Project.startupSymbol = s;
		else if	(s->name == ai)
			Project.allocSymbol = s;
	}

lookup:	dynamic	(n: ref identifier, context: ref scope_s) ref symbol_s =
	{
	s:	ref symbol_s;
	i:	ref include_s;

	s = super lookup(n, context);
	if	(s)
		return s;
	for	(i = includes; i; i = i->next){
		if	(i->unit == 0)
			continue;
		s = i->unit lookupExported(n, context);
		if	(s)
			return s;
		}
	return 0;
	}

lookupExported:	dynamic	(n: ref identifier, context: ref scope_s) ref symbol_s =
	{
	return super lookup(n, context);
	}

clearChanges:	() =
	{
	changed = FALSE;
	}

objectName:	dynamic	() [:] char =
	{
	return name spelling();
	}

fileName:	dynamic	() [:] char =
	{
	return "";
	}

scopeName:	dynamic	() =
	{
	NameEmitter printf("%S::", name spelling());
	}

csourceName:	dynamic	() =
	{
	Project.outputFd printf("%S_", name spelling());
	}

hasChanged:	dynamic	() boolean =
	{
	return FALSE;
	}

parse:	dynamic	() =
	{
	}

sourceEverExisted:	dynamic	() boolean =
	{
	return FALSE;
	}

getUnit:	dynamic	() ref unit_s =
	{
	return self;
	}

outputHeader:	dynamic	() =
	{
	}

	};

/*
	A source unit object is the main symbol table entry for a unit.
	Amongst other things, this object contains a description of the
	current state of the source file for the unit.  At the minimum,
	the sourceFile member contains the name of the source file for the
	unit.  By default, it is the unit name with a .p extension.  The
	source file name is stored explicitly to allow future expansion
	where the source file name can be any arbitrary path.

	If the source file has ever been parsed, then the lastBuiltAge 
	member contains the file time stamp for the source file used in
	the build.  If the source member points at a source buffer, then
	that buffer is the source actually used.

	When the project is written, baseline source file backups need to
	be written for each source file parsed since the last save.  The
	project need not be written first.  If the backup source file does
	not exist, or else doesn't match the project's version of the time
	stamp, then the entire unit must be built from scratch.

	A separate everBuilt flag must be used to insure that the initial
	creation of the unit doesn't accidentally pick up some old backup
	file.  This might arise if you want to force a rebuild from
	scratch for the whole project.  It might also happen if the project
	directory hasn't been scrubbed clean since a previous project was
	built there.
 */
sourceUnit_s:	public	type	inherit	unit_s	{
	public:

	sourceFile:		[:] char;
	source:			* sourceParser;
	lastBuiltAge:		long;
	everBuilt:		boolean;

isSourceUnit:	dynamic	() boolean =
	{
	return TRUE;
	}

create:	factory	(nm: ref identifier, srcFile: [:] char) ref sourceUnit_s =
	{
	self = alloc(sizeof sourceUnit_s);
	self = [ 0, 0, 0, nm, 0 ];
	sourceFile = srcFile;
	lastBuiltAge = 0;
	everBuilt = FALSE;
	source = 0;
	includes = 0;
	changed = TRUE;
	Project addUnit(self);
	return self;
	}

fileName:	dynamic	() [:] char =
	{
	return sourceFile;
	}

saveLibrary:	dynamic	() =
	{
	ifile:		imageFile;
	fpath:		[MAXPATH] char;
	f:		[:] char;
	i:		int;

	f = makePath(fpath, "", name spelling(), ".u");	
						// Make up a unit file name
	i = ifile create(f, FA_READ|FA_WRITE);
	if	(i){
		printf("Could not create unit file");
		return;
		}
	Project.projectSave = FALSE;
	Project.saveUnit = self;
	save(&ifile, U_LIBUNIT, sizeof libraryUnit_s);
	ifile close();
	}

saveProject:	dynamic	(ifile: ref imageFile) =
	{
	save(ifile, U_SRCUNIT, sizeof *self);
	}

save:	(ifile: ref imageFile, x: uRecType, sz: size_t) =
	{
	if	(!ifile beginRecord(x, self, sz))
		return;
	ifile address(next);
	ifile address(symbols);
	ifile putstring(name spelling());
	ifile address(includes);
	ifile integer(changed);
	ifile uinteger(codeSize);
	ifile uinteger(dataSize);
	ifile uinteger(bssSize);
	ifile putstring(sourceFile);
	ifile integer(lastBuiltAge);
	ifile integer(everBuilt);
	if	(symbols)
		symbols save(ifile);
	if	(includes)
		includes save(ifile);
	}

lineno:		dynamic	(f: fileOffset) int =
	{
	if	(f == NO_LINE)
		return 0;
	else if	(source)
		return source tellLine(f);
	else
		return 0;
	}
/*
	A source unit that has no source file for it will produce error 
	messages at the places where it is included.  Of course, after it
	has been built once, if it disappears the include statements won't
	necessarily be revisited, so a message should be generated here.
 */
hasChanged:	dynamic	() boolean =
	{
	fs:	file_t;

	fs = FileSystem stat(sourceFile);
	if	(!fs isValid()){
		if	(everBuilt)
			error(ErrUnitNotFound, name spelling());
		return FALSE;		// couldn't find the file, don't
					// read it.
		}
	if	(everBuilt)
		return lastBuiltAge != fs.ftime;
	else
		return TRUE;
	}

sourceEverExisted:	dynamic	() boolean =
	{
	fs:	file_t;

	if	(everBuilt)
		return TRUE;
	fs = FileSystem stat(sourceFile);
	if	(fs isValid())
		return TRUE;
	else
		return FALSE;
	}
/*
	This function parses the source for a unit, and if there is a
	valid baseline file for reference, only differences are propagated
	into the symbol table.  This elaborate process assures us of
	enough information that a loosely coupled editor will not wreak
	havoc with the build process.

	Note that memory management is accomplished by created a temporary
	heap for the blocks allocated in the parsing process (along with 
	the source buffer for the baseline file).  When the differences
	have been generated, this memory is discarded as a whole.  This
	ends up saving a lot fo time and trouble.
 */
parse:	dynamic	() =
	{
	fs:		file_t;
	fpath:		[MAXPATH] char;
	f:		[:] char;
	u:		ref unit_p;
	u2:		ref unit_p;
	in:		ref sourceParser;
	tempHeap:	incrementalHeap;
	mainHeap:	ref heap;

	tempHeap = [ &Heap ];
	mainHeap = tempHeap activate();
	f = makePath(fpath, "", name spelling(), ".psb");
	fs = FileSystem stat(f);
	if	(fs isValid() &&
		 fs.ftime == lastBuiltAge){
		in = sourceParser create(f);
		u = in parseUnit(PH_NO_MESSAGES, name);
		}
	else
		u = unit_p create(name, 0);
	mainHeap activate();
	source = sourceParser create(sourceFile);
	tempHeap activate();
	fs = FileSystem stat(sourceFile);
	u2 = source parseUnit(PH_CARVE, name);
	if	(source){
		lastBuiltAge = fs.ftime;
		everBuilt = TRUE;
//		source save([ fpath ], lastBuiltAge);
		}
	mainHeap activate();
	u generateDifferences(u2, self);
	tempHeap freeAll();
	includes checkForSources();
	}

addInclude:	(n: ref identifier, off: fileOffset) =
	{
	if	(includes findInclude(n))
		return;
	i:	ref include_s;

	i = include_s create(includes, n, off);
	if	(i)
		includes = i;
	}

addDeclaration:	(n: ref identifier, off: fileOffset,
				hasError: boolean, d: ref declaration_p) =
	{
	sym:	ref symbol_s;
	nsym:	ref symbol_s;

	sym = symbol_s create(self, n, off, hasError, d);
	if	(symbols == 0)
		symbols = sym;
	else	{
		for	(nsym = symbols; nsym->next; nsym = nsym->next)
			;
		nsym->next = sym;
		}
	}

constructValues:	dynamic	() =
	{
	s:	ref symbol_s;

	Func startValue(self);
	for	(s = symbols; s; s = s->next)
		s constructValue(TRUE);
	Func endValue();
	}

getSourceBuffer:	dynamic	() ref sourceParser =
 	{
	return source;
	}

outputHeader:	dynamic	() =
	{
	fd:	stream;
	fpath:	[MAXPATH] char;
	f:	[:] char;

	f = makePath(fpath, "", name spelling(), ".h");
	if	(fd create(f, AR_WRITE|AR_READ)){
		printf("Couldn't create '%S'\n", f);
		return;
		}
	s:	ref symbol_s;

	for	(s = symbols; s; s = s->next)
		s outputHeader(TRUE, &fd);
	fd close();
	}

	};

libraryUnit_s:	public	type	inherit	unit_s	{
	public:

	sourceFile:		[:] char;
	unitFile:		[:] char;
	lastBuiltAge:		long;
	everBuilt:		boolean;
	connected:		boolean;

create:	factory	(nm: ref identifier, filename: [:] char) ref libraryUnit_s =
	{
	lx:	loader;

	if	(lx open(filename))
		return 0;
	lx decode();
	self = lx getObject(1);
	lx close();
	if	(self){
		unitFile = heapFile(filename, "");
		name = nm;
		Project addUnit(self);

		i:	ref include_s;

		for	(i = includes; i; i = i->next){
			i->unit = Project locateUnit(i->name);
			if	(i->unit == 0)
				i->unit = unit_s create(i->name);
			}
		connected = FALSE;
		}
	return self;
	}

load:	factory	(lu: pointer, image: ref loader) =
	{
	self = lu;
	self = [ 0, 0 ];
	next = image address();
	symbols = image address();
	name = hash(image string());
	includes = image address();
	changed = image integer();
	codeSize = image uinteger();
	dataSize = image uinteger();
	bssSize = image uinteger();
	sourceFile = heapFile(image string(), "");
	lastBuiltAge = image integer();
	everBuilt = image integer();
	}

fileName:	dynamic	() [:] char =
	{
	return unitFile;
	}

saveProject:	dynamic	(ifile: ref imageFile) =
	{
	if	(!ifile beginRecord(U_LIBUNIT, self, sizeof *self))
		return;
	ifile address(next);
	ifile address(symbols);
	ifile putstring(name spelling());
	ifile address(includes);
	ifile integer(changed);
	ifile uinteger(codeSize);
	ifile uinteger(dataSize);
	ifile uinteger(bssSize);
	ifile putstring(sourceFile);
	ifile integer(lastBuiltAge);
	ifile integer(everBuilt);
	if	(symbols)
		symbols save(ifile);
	if	(includes)
		includes save(ifile);
	}

sourceEverExisted:	dynamic	() boolean =
	{
	return TRUE;
	}

reconnectInterfaces:	dynamic	() =
	{
	s:	ref symbol_s;

	if	(!connected){
		for	(s = symbols; s; s = s->next)
			s reconnectInterface();
		connected = TRUE;
		}
	}

	};

Deferred:	ref deferredInterfaces;
DeferredHeap:	incrementalHeap;
DeferFlag:	boolean;

deferredInterfaces:	type	{
	dtype:		ref type_s;
	enclosing:	ref scope_s;
	msg:		ref messageGenerator;
	varAllowed:	boolean;

	public:

	next:		ref deferredInterfaces;

create:	factory	(d: ref type_s, b: boolean, e: ref scope_s) =
	{
	self = DeferredHeap alloc(sizeof deferredInterfaces);
	next = Deferred;
	Deferred = self;
	dtype = d;
	enclosing = e;
	msg = CurrentContext.obj;
	varAllowed = b;
	}

constructInterface:	() =
	{
	objsave:	ref messageGenerator;

	objsave = CurrentContext.obj;
	CurrentContext.obj = msg;
	dtype constructInterface(enclosing, varAllowed);
	CurrentContext.obj = objsave;
	}

	};

scope_s:	public	type	inherit	messageGenerator {
	public:

	enclosing:		ref scope_s;
	symbols:		* symbol_s;
	thisHeap:		ref heap;
	reservedInScope:	regMask;

constructor:	(e: ref scope_s, s: ref symbol_s) =
	{
	enclosing = e;
	symbols = s;
	thisHeap = currentHeap();
	reservedInScope = 0;
	}

lookup:	dynamic	(n: ref identifier, context: ref scope_s) ref symbol_s =
	{
	s:	ref symbol_s;

	for	(s = symbols; s; s = s->next)
		if	(s->name == n){
			if	(s->visibility != V_PRIVATE)
				return s;
			if	(encloses(context))
				return s;
			else
				return 0;
			}
	if	(enclosing)
		return enclosing lookup(n, context);
	else
		return 0;
	}

encloses:	(sc: ref scope_s) boolean =
	{
	while	(sc){
		if	(self == sc)
			return TRUE;
		sc = sc->enclosing;
		}
	return FALSE;
	}

lookupLocal:	(n: ref identifier, context: ref scope_s) ref symbol_s =
	{
	s:	ref symbol_s;

	for	(s = symbols; s; s = s->next)
		if	(s->name == n){
			if	(s->visibility != V_PRIVATE)
				return s;
			if	(encloses(context))
				return s;
			else
				return 0;
			return s;
			}
	return 0;
	}
/*
	This function appends its name to the Name buffer for display
	purposes.
 */
scopeName:	dynamic	() =
	{
	}

csourceName:	dynamic	() =
	{
	}

returnTypeOf:	dynamic	() ref type_s =
	{
	return 0;
	}

getSourceBuffer:	dynamic	() ref sourceParser =
 	{
	if	(enclosing)
		return enclosing getSourceBuffer();
	else
		return 0;
	}

getUnit:	dynamic	() ref unit_s =
	{
	if	(enclosing)
		return enclosing getUnit();
	else
		return 0;
	}

getEnclosingType:	dynamic	() ref type_s =
	{
	if	(enclosing)
		return enclosing getEnclosingType();
	else
		return 0;
	}

reconnectInterfaces:	dynamic	() =
	{
	}

constructInterfaces:	dynamic	() =
	{
	s:	ref symbol_s;

	Deferred = 0;
	DeferredHeap = [ &Heap ];
	DeferFlag = TRUE;

	for	(s = symbols; s; s = s->next)
		s constructInterface();

	d:	ref deferredInterfaces;

	while	(Deferred){
		d = Deferred;
		Deferred = d->next;
		d constructInterface();
		}

	DeferFlag = FALSE;
	DeferredHeap freeAll();
	}

constructValues:	dynamic	() =
	{
	}

deferInterface:	(d: ref type_s, b:boolean) =
	{
	if	(DeferFlag)
		deferredInterfaces create(d, b, self);
	else
		d constructInterface(self, b);
	}

addErrorDeclaration:	(n: ref identifier) =
	{
	sym:	ref symbol_s;

	sym = symbol_s createError(self, n, thisHeap);
	sym->next = symbols;
	symbols = sym;
	}

unnamedLocal:	(t: ref type_s) ref symbol_s =
	{
	sym:	ref symbol_s;

	sym = symbol_s createUnnamed(self, t, thisHeap);
	sym->next = symbols;
	symbols = sym;
	return sym;
	}

unnamedStatic:	(t: ref type_s) ref symbol_s =
	{
	sym:	ref symbol_s;

	sym = symbol_s createUnnamedStatic(self, t, thisHeap);
	sym->next = symbols;
	symbols = sym;
	return sym;
	}

foldStaticValues:	dynamic	(v: ref value) =
	{
	sym:	* symbol_s;

	for	(sym = symbols; sym; sym = sym->next){
//		if	(sym->dtype)
//			sym->dtype foldStaticValues(v);
		v addRelatedValue(sym->currentValue);
		}
	}

threadChild:	dynamic	(ref blockScope) =
	{
	}

findLabel:	dynamic	(ref identifier) ref label_x =
	{
	return 0;
	}

countValues:	dynamic	() int =
	{
	sym:		ref symbol_s;
	vCount:		int;

	vCount = 0;
	for	(sym = symbols; sym; sym = sym->next){
		if	(sym->currentValue)
			vCount += sym->currentValue countRelated();
		if	(sym->dtype)
			vCount += sym->dtype countValues();
		}
	return vCount;
	}

enumerateValues:	dynamic	() =
	{
	sym:		ref symbol_s;

	for	(sym = symbols; sym; sym = sym->next){
		if	(sym->currentValue)
			sym->currentValue enumerateValues();
		if	(sym->dtype)
			sym->dtype enumerateValues();
		}
	}

	};

blockScope:	public	type	inherit	scope_s {
	public:

	child:			* blockScope;
	sibling:		* blockScope;
	labels:			* label_s;
	function:		ref symbol_s;
	usedInScope:		regMask;

constructor:	(e: ref scope_s, f: ref symbol_s) =
	{
	super constructor(e, 0);
	child = 0;
	sibling = 0;
	labels = 0;
	if	(e)
		e threadChild(self);
	function = f;
	usedInScope = 0;
	}

create:	factory	() ref blockScope =
	{
	s:	ref blockScope;

	self = alloc(sizeof blockScope);
	self = [ 0, 0 ];
	return self;
	}

constructValues:	dynamic	() =
	{
	s:		ref symbol_s;

	for	(s = symbols; s; s = s->next)
		s constructValue(TRUE);
	}

defineParameters:	public	(parm: ref parameter_s, fQual: dQualifiers) =
	{
	s:		ref symbol_s;
	d:		ref type_s;

	if	(usesIndirectReturn()){
		d = refTo(TargetData.funcType->returnType);
		TargetData.indirectReturn = symbol_s createParameter(self,
				hash("return"), 0, d);
		}
	fQual &= DQ_MEMBERFUNC|DQ_FACTORY;
	if	(fQual == DQ_MEMBERFUNC){
		d = refTo(enclosing getEnclosingType());
		Project.selfSymbol = symbol_s createParameter(self, 
							hash("self"), 0, d);
		if	(TargetData.funcType->callingConvention == FC_GATE){
			TargetData.gateArgsSymbol = 
				symbol_s createParameter(self, 
							hash("gate"), 0, d);
			for	(; parm; parm = parm->next){
				if	(parm->pType->topType == T_ARRAY)
					d = refTo(parm->pType elementOf());
				else
					d = parm->pType getType();
				s = symbol_s createBased(self, parm->name, 
							parm->offset, d);
				s->next = symbols;
				symbols = s;
				}
			locateGateParameters(symbols);
			return;
			}
		}
	for	(; parm; parm = parm->next){
		if	(parm->pType->topType == T_ARRAY)
			d = refTo(parm->pType elementOf());
		else
			d = parm->pType getType();
		s = symbol_s createParameter(self, parm->name, 
							parm->offset, d);
		s->next = symbols;
		symbols = s;
		}
	if	(fQual == DQ_MEMBERFUNC|DQ_FACTORY){
		d = refTo(enclosing getEnclosingType());
		Project.selfSymbol = unnamedLocal(d);
		Project.selfSymbol->name = hash("self");
		Project.selfSymbol->var->dtype = d;
		}
	}

locateAutos:	(baseAmount: int, allocateBig: boolean) int =
	{
	sym:		ref symbol_s;
	m:		addr_t;
	align:		addr_t;
	d:		ref type_s;
	maxbase:	int;
	v:		ref variable;

	for	(sym = symbols; sym; sym = sym->next){
		if	(sym->storageClass != SC_AUTO)
			continue;
		if	(sym->dtype == 0)
			continue;
		v = sym->var;
		if	(v->totalCount == 0 ||
			 (v->flags & VF_REG))
			continue;		/* skip register vars */
		d = sym->dtype getType();
		m = d sizeOf();
		if	(m > BigObjectThreshold){
			if	(!allocateBig)
				continue;
			}
		else if	(allocateBig)
			continue;
		v->offset = baseAmount - m;
		align = d alignmentOf() / BYTEBITS;
		m = -v->offset;
		m = m % align;
		if	(m)
			v->offset -= align - m;
		baseAmount = v->offset;
		}
	maxbase = -baseAmount;

	s:	ref blockScope;

	for	(s = child; s; s = s->sibling){
		m = s locateAutos(baseAmount, allocateBig);
		if	(m > maxbase)
			maxbase = m;
		}
	TargetData.autoCount = maxbase;
	return maxbase;
	}

foldStaticValues:	dynamic	(v: ref value) =
	{
	while	(self){
		super foldStaticValues(v);
		if	(child)
			child foldStaticValues(v);
		self = sibling;
		}
	}

returnTypeOf:	dynamic	() ref type_s =
	{
	if	(function){
		if	(function->dtype)
			return function->dtype returnTypeOf();
		else
			return 0;
		}
	else if	(enclosing)
		return enclosing returnTypeOf();
	else
		return 0;
	}

threadChild:	dynamic	(s: ref blockScope) =
	{
	s->sibling = child;
	child = s;
	}

addDeclaration:	(d: ref declaration_p) =
	{
	n:	ref name_p;

	for	(n = d->idList; n; n = n->next){
		sym:	ref symbol_s;

		for	(sym = symbols; sym; sym = sym->next)
			if	(n->name == sym->name){
				CurrentContext.offset = n->offset;
				error(ErrRedecl, n->name spelling());
				sym->dtype = ErrorType;
				break;
				}
		if	(sym)
			continue;
		sym = symbol_s create(self, n->name, n->offset, 
							n->hasError, d);
		sym->next = symbols;
		symbols = sym;
		}
	}

addLabel:	(i: ref identifier, t: ref label_x) =
	{
	lab:	ref label_s;

	for	(lab = labels; lab; lab = lab->next)
		if	(lab->name == i){
			lab->loc = 0;		// discard old label
			error(ErrRedecl, i spelling());
			return;
			}
	lab = label_s create(i, t);
	lab->next = labels;
	labels = lab;
	}

findLabel:	dynamic	(n: ref identifier) ref label_x =
	{
	lab:	ref label_s;

	for	(lab = labels; lab; lab = lab->next)
		if	(lab->name == n)
			return lab->loc;
	if	(enclosing)
		return enclosing findLabel(n);
	else
		return 0;
	}

scopeName:	dynamic	() =
	{
	if	(function){
		s:	[:] char;

		s = function objectName();
		NameEmitter putc('.');
		}
	else if	(enclosing)
		enclosing scopeName();
	}

	};

label_s:	public	type	{
	public:

	name:		ref identifier;
	loc:		ref label_x;
	next:		* label_s;

create:	factory	(i: ref identifier, t: ref label_x) ref label_s =
	{
	self = alloc(sizeof label_s);
	self = [ i, t ];
	return self;
	}

	};

symbol_s:	public	type	inherit	messageGenerator {
	public:

	next:		* symbol_s;
	name:		ref identifier;
	offset:		fileOffset;
	visibility:	visibilities;
	storageClass:	storageClasses;
	qualifier:	dQualifiers;
	location:	ref identifier;
	dtype:		* type_s;
	initializer:	textRange;
	initBase:	int;			// A base value for
	ifaceChanged:	boolean;
	valueChanged:	boolean;
	working:	boolean;
	declEmitted:	boolean;
	enclosing:	ref scope_s;
	currentValue:	* value;
	var:		ref variable;
	bitOffset:	addr_t;

create:	factory	(e: ref scope_s, n: ref identifier, 
				off: fileOffset,
				hasError: boolean, 
				d: ref declaration_p) ref symbol_s =
	{
	t:	ref type_s;

	self = alloc(sizeof symbol_s);
	if	(hasError)
		t = ErrorType;
	else if	(d->dtype)
		t = d->dtype symbolTableCopy(e);
	else
		t = ErrorType;
	self = [ 0, n, off, d->visibility, 
				d->storageClass, d->qualifier,
				d->location, t ];
	initializer = d->initializer;
	initBase = d->initBase;
	ifaceChanged = TRUE;
	valueChanged = TRUE;
	enclosing = e;
	working = FALSE;
	declEmitted = FALSE;
	if	(storageClass == SC_STATIC ||
		 (storageClass == SC_INTRINSIC &&
		  dtype &&
		  dtype->topType != T_FUNC))		// not a function
		currentValue = value create(self, dtype);
	else
		currentValue = 0;
	bitOffset = 0;
	if	(storageClass == SC_AUTO ||
		 storageClass == SC_PARM)
		var = variable create(self);
	else
		var = 0;
	return self;
	}

createError:	factory	(e: ref scope_s, n: ref identifier, 
						h: ref heap) ref symbol_s =
	{
	self = h alloc(sizeof symbol_s);
	self = [ 0, n, 0, V_PRIVATE, SC_STATIC, 0, 0, ErrorType ];
	initializer = [ 0, 0 ];
	initBase = 0;
	ifaceChanged = FALSE;
	valueChanged = FALSE;
	declEmitted = FALSE;
	enclosing = e;
	working = FALSE;
	currentValue = 0;
	var = 0;
	return self;
	}

createUnnamed:	factory	(e: ref scope_s, t: ref type_s, 
						h: ref heap) ref symbol_s =
	{
	self = h alloc(sizeof symbol_s);
	self = [ 0, 0, 0, V_PRIVATE, SC_AUTO, 0, 0, t ];
	initializer = [ 0, 0 ];
	initBase = 0;
	ifaceChanged = FALSE;
	valueChanged = FALSE;
	declEmitted = FALSE;
	enclosing = e;
	working = FALSE;
	currentValue = 0;
	var = variable create(self);
	return self;
	}

createUnnamedStatic:	factory	(e: ref scope_s, t: ref type_s, 
						h: ref heap) ref symbol_s =
	{
	self = h alloc(sizeof symbol_s);
	self = [ 0, 0, 0, V_PRIVATE, SC_STATIC, 0, 0, t ];
	initializer = [ 0, 0 ];
	initBase = 0;
	ifaceChanged = FALSE;
	valueChanged = FALSE;
	enclosing = e;
	working = FALSE;
	declEmitted = FALSE;
	currentValue = value create(self, dtype);
	bitOffset = 0;
	var = 0;
	return self;
	}

createParameter:	factory	(e: ref scope_s, n: ref identifier, 
				off: fileOffset, t: ref type_s) ref symbol_s =
	{
	self = alloc(sizeof symbol_s);
	self = [ 0, n, off, V_PRIVATE, SC_PARM, 0, 0, t ];
	initializer = [ 0, 0 ];
	initBase = 0;
	ifaceChanged = FALSE;
	valueChanged = FALSE;
	declEmitted = FALSE;
	enclosing = e;
	working = FALSE;
	currentValue = 0;
	var = variable create(self);
	var->dtype = t;
	return self;
	}

createBased:	factory	(e: ref scope_s, n: ref identifier, 
				off: fileOffset, t: ref type_s) ref symbol_s =
	{
	self = alloc(sizeof symbol_s);
	self = [ 0, n, off, V_PRIVATE, SC_BASED, 0, 0, t ];
	initializer = [ 0, 0 ];
	initBase = 0;
	ifaceChanged = FALSE;
	valueChanged = FALSE;
	declEmitted = FALSE;
	enclosing = e;
	working = FALSE;
	currentValue = 0;
	var = 0;
	return self;
	}

load:	factory	(s: pointer, image: ref loader) =
	{
	cp:	[:] char;

	self = s;
	self = [ 0 ];
	next = image address();
	cp = image string();
	if	(|cp)
		name = hash(cp);
	else
		name = 0;
	offset = image uinteger();
	visibility = image integer();
	storageClass = image integer();
	qualifier = image integer();
	cp = image string();
	if	(|cp)
		location = hash(cp);
	else
		location = 0;
	dtype = image address();
	initializer.start = image uinteger();
	initializer.end = image uinteger();
	initBase = image integer();
	enclosing = image address();
	currentValue = image address();
	bitOffset = image uinteger();
	}

save:	(ifile: ref imageFile) =
	{
	for	(; self; self = next){
		if	(!ifile beginRecord(U_SYMBOL, self, sizeof *self))
			continue;
		ifile address(next);
		if	(name)
			ifile putstring(name spelling());
		else
			ifile putc(0);
		ifile uinteger(offset);
		ifile integer(visibility);
		ifile integer(storageClass);
		ifile integer(qualifier);
		if	(location)
			ifile putstring(location spelling());
		else
			ifile putc(0);
		ifile address(dtype);
		ifile uinteger(initializer.start);
		ifile uinteger(initializer.end);
		ifile integer(initBase);
		ifile address(enclosing);
		ifile address(currentValue);
		ifile uinteger(bitOffset);
		if	(dtype)
			dtype save(ifile);
		if	(currentValue)
			currentValue save(ifile);
		}
	}

outputHeader:	(fileScope: boolean, fd: ref stream) =
	{
	t:	ref type_s;

	if	(dtype == 0)
		return;
	if	(dtype->topType == T_ERROR)
		return;
	if	(fileScope){
		if	(visibility == V_PRIVATE)
			return;
		}
	switch	(storageClass){
	case	SC_TYPE:
		t = dtype getType();
		if	(t == 0)
			return;
		if	(t->topType == T_SIGNED ||
			 t->topType == T_UNSIGNED ||
			 t->topType == T_FLOAT)
			return;
		fd printf("typedef ");
		break;

	case	SC_INTRINSIC:
	case	SC_REGISTER:
		return;

	case	SC_EXTERN:
	case	SC_STATIC:
		if	(dtype->topType == T_FUNC)
			break;
		fd printf("extern ");
		}
	dtype outputHeaderPrefix(fd);
	outputHeaderName(fd);
	dtype outputHeaderSuffix(fd);
	fd printf(";\n");
	}

outputHeaderName:	(fd: ref stream) =
	{
	if	(enclosing)
		enclosing csourceName();
	fd printf("%S_", name spelling());
	}

emitNeededDeclaration:	(fd: ref stream) =
	{
	if	(!declEmitted){
		declEmitted = TRUE;
		dtype emitNeededTypedefs(fd);
		outputHeader(0, fd);
		}
	}

clearChanges:	() =
	{
	ifaceChanged = FALSE;
	valueChanged = FALSE;
	}

objectName:	dynamic	() [:] char =
	{
	NameEmitter = [ Name ];
	if	(enclosing)
		enclosing scopeName();

	if	(name)
		NameEmitter printf(name spelling());
	else
		NameEmitter printf("<no name>");
	return NameEmitter result();
	}

lineno:	dynamic	(o: fileOffset) int =
	{
	if	(enclosing)
		return enclosing getUnit() lineno(o);
	else
		return 0;
	}

display:	() =
	{
	if	(name)
		printf("%S", name spelling());
	if	(offset != NO_LINE)
		printf(" %d", lineno(offset));
	printf(":");
	if	(ifaceChanged)
		printf(" *iface");
	if	(valueChanged)
		printf(" *value");
	switch	(visibility){
	case	V_PUBLIC:	printf(" public");	break;
	case	V_VISIBLE: 	printf(" visible");	break;
		}
	switch	(storageClass){
	case	SC_STATIC:	printf(" static");	break;
	case	SC_TYPE:	printf(" type");	break;
	case	SC_MEMBER:	printf(" member");	break;
	case	SC_INTRINSIC:	printf(" intrinsic");	break;
	case	SC_REGISTER:	printf(" __reg__");	break;
	case	SC_REMOTE:	printf(" remote");	break;
		}
	if	(qualifier & DQ_CONST)		printf(" const");
	if	(qualifier & DQ_VOLATILE)	printf(" volatile");
	if	(qualifier & DQ_ENTRY)		printf(" entry");
	if	(qualifier & DQ_CLEANUP)	printf(" cleanup");
	if	(qualifier & DQ_DYNAMIC)	printf(" dynamic");
	if	(qualifier & DQ_FAR)		printf(" far");
	if	(qualifier & DQ_FACTORY)	printf(" factory");
	if	(qualifier & DQ_MEMBERFUNC)	printf(" memberfunc");
	if	(location)
		printf(" loc %S", location spelling());
	if	(storageClass == SC_MEMBER)
		printf(" bit %d", bitOffset);
	if	(dtype)
		dtype display(TRUE);
	if	(initializer.start){
		printf(" = init [ %d, %d ]", initializer.start, initializer.end);
		if	(initBase)
			printf(" + %d;", initBase);
		}
	else if	(initBase)
		printf(" = %d;", initBase);
	}

reconnectInterface:	() =
	{
	objsave:	ref messageGenerator;

	working = TRUE;
	objsave = CurrentContext.obj;
	CurrentContext.obj = self;
	CurrentContext.offset = offset;
	if	(dtype){
		dtype setTypeName(name);
		dtype reconnectInterface(enclosing);
		}
	CurrentContext.obj = objsave;
	working = FALSE;
	}

constructInterface:	() =
	{
	if	(ifaceChanged){
		objsave:	ref messageGenerator;

		working = TRUE;
		objsave = CurrentContext.obj;
		CurrentContext.obj = self;
		CurrentContext.offset = offset;
		if	(dtype){
			dtype setTypeName(name);
			dtype constructInterface(enclosing, FALSE);
			}
		CurrentContext.obj = objsave;
		working = FALSE;
		ifaceChanged = FALSE;
		if	(storageClass == SC_REMOTE){
			if	(initializer.start)
				bitOffset = constantExpression(initializer, 
								enclosing);
			else
				bitOffset = initBase;
			}
		}
	}

constructValue:	(forceOutput: boolean) boolean =
	{
	if	(dtype == 0 ||
		 dtype == ErrorType){
		valueChanged = FALSE;
		return FALSE;
		}
	dtype constructValue(enclosing);
	if	(valueChanged || forceOutput){
		if	(currentValue){
			tempHeap:	incrementalHeap;
			mainHeap:	ref heap;
			objsave:	ref messageGenerator;
			symsave:	ref symbol_s;

			tempHeap = [ &Heap ];
			mainHeap = tempHeap activate();
			objsave = CurrentContext.obj;
			symsave = Project.selfSymbol;
			Project.selfSymbol = 0;
			CurrentContext.obj = self;
			if	(DebugList){
				display();
				printf("\n");
				}
			if	(dtype->topType == T_FUNC)		// a function
				functionDefinition();
			else					// data
				Func generateData(self);
			CurrentContext.obj = objsave;
			Project.selfSymbol = symsave;
			mainHeap activate();
			tempHeap freeAll();
			}
		valueChanged = FALSE;
		}
	return TRUE;
	}
/*
	This function is called after it is determined that the symbol
	is a constant integer, signed or unsigned, and the value is
	valid.
 */
constantInteger:	() ref tree_p =
	{
	if	(constructValue(FALSE) &&
		 currentValue)
		return currentValue constantInteger();
	else
		return ErrorTree;
	}

constantFloat:	() ref tree_p =
	{
	if	(constructValue(FALSE) &&
		 currentValue)
		return currentValue constantFloat();
	else
		return ErrorTree;
	}

functionDefinition:	() =
	{
	sc:		blockScope;
	src:		ref sourceParser;
	x:		ref stmt_x;
	ft:		ref function_z;

	src = enclosing getSourceBuffer();
	if	(src == 0)
		return;
	sc = [ enclosing, self ];
	ft = ref function_z(dtype);
	memSet(&TargetData, 0, sizeof TargetData);
	TargetData.rootScope = &sc;
	TargetData.funcType = ft;
	TargetData.funcSymbol = self;
	sc defineParameters(ft->parameters, qualifier);
	x = src parseFunctionBody(initializer);
	if	(x == 0)
		return;
	x assignTypes(&sc, FALSE);
	x fold();
	Func generateCode(self, x, &sc);
	}

	};

include_s:	public	type	{
	public:

	next:		* include_s;
	offset:		fileOffset;		// offset of include statement
	name:		ref identifier;
	unit:		ref unit_s;

public:

create:	factory	(nx: ref include_s, n: ref identifier, 
						i: fileOffset) ref include_s =
	{
	u:	ref unit_s;

	u = Project locateUnit(n);
	if	(u == 0)
		u = unit_s create(n);
	self = alloc(sizeof include_s);
	self = [ nx, i, n, u ];
	return self;
	}

load:	factory	(i: pointer, image: ref loader) =
	{
	self = i;
	next = image address();
	offset = image uinteger();
	name = hash(image string());
	unit = 0;
	}

save:	(ifile: ref imageFile) =
	{
	if	(!ifile beginRecord(U_INCLUDE, self, sizeof *self))
		return;
	ifile address(next);
	ifile uinteger(offset);
	ifile putstring(name spelling());
	if	(next)
		next save(ifile);
	}

checkForSources:	() =
	{
	while	(self){
		if	(unit == 0 ||
			 !unit sourceEverExisted()){
			CurrentContext.offset = offset;
			error(ErrUnitNotFound, unit->name spelling());
			}
		self = next;
		}
	}

findInclude:	(n: ref identifier) ref include_s =
	{
	while	(self &&
		 name != n)
		self = next;
	return self;
	}

display:	(u: ref unit_s) =
	{
	i:	int;

	i = u lineno(offset);
	if	(i)
		printf("%4d: ", i);
	else
		printf("      ");
	printf("include %S;", name spelling());
	}

	};

heapFile:	public	(cp: [:] char, extension: [:] char) [:] char =
	{
	fpath:		[MAXPATH] char;
	fext:		[:] char;
	n:		[:] char;

	if	(|extension){
		fext = getExtension(cp);
		if	(|fext == 0)
			cp = makePath(fpath, "", cp, extension);
		}
	n = new [|cp] char;
	n [:]= cp;
	return n;
	}

/*
 *	FUNCTION:	constantExpression
 *
 *	DESCRIPTION:
 *		This function parses a constant expression and if
 *		successful sets Token.i.icon (global integer constant) to the
 *		expression's value.  A context parameter determines the
 *		terminator required for the expression.  Exact error handling
 *		and scan position depends on the context.
 *
 *		In any of the contexts, if the expression parse
 *		failed, or the terminator was not acceptable, a resync
 *		occurs.  If the expression
 *		failed in type checking, or a case statement was missing the
 *		colon, no resync occurs, Token.i.icon is set to zero.
 *
 *		In general, this returns zero if and only if a resync has
 *		taken place.
 */

constantExpression:	public	(source: textRange, ctx: ref scope_s) long =
	{
	t:		ref tree_p;
	tempHeap:	incrementalHeap;
	mainHeap:	ref heap;
	src:		ref sourceParser;

	src = ctx getSourceBuffer();
	if	(src == 0)
		return 0;
	tempHeap = [ &Heap ];
	mainHeap = tempHeap activate();
	t = src parseWholeExpression(source);
	i:	long;

	i = 0;
	if	(t){
		t = t assignTypes(ctx, FALSE);
		t = t fold();
		if	(t->operator == O_ICON)
			i = t integerValue();
		else if	(t->operator != O_ERROR){
			t display(0);
			error(ErrConstantExpr);
			}
		}
	tempHeap freeAll();
	mainHeap activate();
	return i;
	}

extractExpression:	public	(source: textRange, 
					ctx: ref scope_s) ref tree_p =
	{
	src:		ref sourceParser;

	src = ctx getSourceBuffer();
	if	(src == 0)
		return ErrorTree;
	return src parseWholeExpression(source);
	}

NameEmitter:	public	stringEmitter;
Name:		public	[256] char;
