/* DBREAD, written 1992 by Peter D. Hipson */
/* This program reads dBASE III files. /*

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>

/* Some defines useful for dBASE files: */

/* Actual record status defines (first byte of each record) */
#define DELETED_RECORD '*'
#define USABLE_RECORD ' '

/* Field (column) definitions (capital letters, please) */

#define NUMERIC_FIELD     'N'
#define CHARACTER_FIELD   'C'
#define LOGICAL_FIELD     'L'
#define MEMO_FIELD        'M'
#define DATE_FIELD        'D'
#define FLOAT_FIELD       'F'
#define PICTURE_FIELD     'P'

/* End of dBASE defines */

#pragma pack(1) /* Pack to byte boundaries */

typedef struct   {
/*  Bitfields are arranged from least significant to most significant */
    unsigned int   bfVersion:7;
    unsigned int   bfHasMemo:1;
    unsigned int   bYear:8;
    unsigned char  bMonth;
    unsigned char  bDay;
    long int       lNumberRecords;
    short int      nFirstRecordOffset;
    short int      nRecordLength;
    unsigned char  szReserved[20];
    } DB3HEADER;


typedef struct   {
    char           szColumnName[11];
    char           chType;
    long           lFieldPointer;
    unsigned char  byLength;
    unsigned char  byDecimalPlace;
    char           szReserved[14];
    } COLUMNDEF;


int    main()

{

FILE    *DBFile;

DB3HEADER   db3Header;

COLUMNDEF   *ColumnDef;

unsigned char * pBuffer;

char    szFileName[25];

int     i;
int     nColumnCount = 0;
int     nResult;

long    lCurrentRecord = 0;

double  dSales = 0.0; /* Forces loading of floating point support.*/



    printf("sizeof(DB3HEADER) = %d\n", sizeof(DB3HEADER));
    printf("sizeof(COLUMNDEF) = %d\n", sizeof(COLUMNDEF));

    printf("Please enter customer database name: ");

    gets(szFileName);

    DBFile = fopen(szFileName, "rb");

    if (DBFile == NULL)
    {
        fprintf(stderr,
            "ERROR: File '%s' couldn't be opened.\n", szFileName);

        exit(4);
    }

    nResult = fread((char *)&db3Header,
        sizeof(DB3HEADER),
        1,
        DBFile);

    if (nResult != 1)
    {
        if (!feof(DBFile))
        {
            fprintf(stderr, "ERROR: File '%s', read error (Database header).\n",
                szFileName);

            fclose(DBFile);

            exit(4);
        }
        else
        {
            fprintf(stderr, "Unexpected end of database file '%s'.\n",
                szFileName);

            fclose(DBFile);

            exit(4);
        }
    }

    if (db3Header.bfHasMemo)
    {
        printf("There is a .DBT memo\n");
    }
    else
    {
        printf("There is no a .DBT memo\n");
    }

    printf("Created with version %d of dBASE. \n",
        db3Header.bfVersion);

    if (db3Header.bfVersion != 3 &&
        db3Header.bfVersion != 4)
    {
        printf("The version of dBASE that created this file "
            "may not be compatable.\n");
    }

    printf("Updated last on: %d/", db3Header.bMonth);
    printf("%d", db3Header.bDay);
    printf(" 19%d\n", db3Header.bYear);

    printf("There are %ld records in the database. \n",
        db3Header.lNumberRecords);

    printf("The first record starts at byte %d. \n",
        db3Header.nFirstRecordOffset);

    printf("Each record is %d bytes long. \n",
        db3Header.nRecordLength);

    printf("The reserved field contains '%s' \n",
        db3Header.szReserved);

    nColumnCount =
        (db3Header.nFirstRecordOffset - sizeof(DB3HEADER)) /
        sizeof(COLUMNDEF);

    printf("There are %d columns in each record. \n", nColumnCount);

/*  Now allocate memory for each of the column definitions: */

    ColumnDef = (COLUMNDEF *)calloc(sizeof(COLUMNDEF), nColumnCount);

    if (ColumnDef == (COLUMNDEF *)NULL)
    {
        fprintf(stderr,
            "Couldn't allocate memory for the column definations \n");

        fclose(DBFile);

        exit(4);
    }

    nResult = fread((char *)ColumnDef,
        sizeof(COLUMNDEF),
        nColumnCount,
        DBFile);

    if (nResult != nColumnCount)
    {
        if (!feof(DBFile))
        {
            fprintf(stderr, "ERROR: File '%s', read error (Column definations).\n",
                szFileName);

            fclose(DBFile);

            exit(4);
        }
        else
        {
            fprintf(stderr, "Unexpected end of database file '%s'"
                " while reading column definations.\n",
                szFileName);

            fclose(DBFile);

            exit(4);
        }
    }

    printf("Column definations: \n");

    for (i = 0; i < nColumnCount; i++)
    {
        printf("Name: '%10.10s' ", ColumnDef[i].szColumnName);

        switch(ColumnDef[i].chType)
        {
            case NUMERIC_FIELD: /* Number field */
                printf(" (Numeric)\n");
                break;

            case CHARACTER_FIELD: /* Character field */
                printf(" (Character)\n");
                break;

            case LOGICAL_FIELD: /* Logical (using 'Y', 'N', etc */
                printf(" (Logical)\n");
                break;

            case MEMO_FIELD: /* Memo Index field */
                printf(" (Memo file .DBT Index)\n");
                break;

            case DATE_FIELD: /* Date in YYYYMMDD format */
                printf(" (Date in YYYYMMDD)\n");
                break;

            case FLOAT_FIELD: /* Floating point field */
                printf(" (Floating point)\n");
                break;

            case PICTURE_FIELD: /* Date in YYYYMMDD format */
                printf(" (Picture format)\n");
                break;

            default: /* Unknown type of field */
                printf(" (Field type unknown)\n");
                break;
        }

        printf("Length: %d\n", ColumnDef[i].byLength);
        printf("DecimalPoint: %d\n", ColumnDef[i].byDecimalPlace);
        printf("Reserved '%s'\n", ColumnDef[i].szReserved);
    }

/*  Next allocate the buffer to hold a database record
 *  We add a byte for the terminating \0, which is not supplied by
 *  dBASE as part of the record.
 */

    pBuffer = (unsigned char *)calloc(sizeof(char),
        db3Header.nRecordLength + 1);

    if (pBuffer == (unsigned char *)NULL)
    {
        fprintf(stderr,
            "Couldn't allocate memory for the column buffer\n");

        fclose(DBFile);

        exit(4);
    }

    nResult = fseek(DBFile,
        (long)db3Header.nFirstRecordOffset, SEEK_SET);

    if (nResult != 0)
    {
        if (!feof(DBFile))
        {
            fprintf(stderr, "ERROR: File '%s', seek error.\n",
                szFileName);

            fclose(DBFile);

            exit(4);
        }
        else
        {
            fprintf(stderr, "Unexpected end of database file '%s'"
                " while reading record.\n",
                szFileName);

            fclose(DBFile);

            exit(4);
        }
    }

    for (lCurrentRecord = 0;
        lCurrentRecord < db3Header.lNumberRecords;
        ++lCurrentRecord)
    {
        nResult = fread((char *)pBuffer,
            db3Header.nRecordLength, 1, DBFile);

        if (nResult != 1)
        {
            if (!feof(DBFile))
            {
                fprintf(stderr, "ERROR: File '%s', read error (records).\n",
                    szFileName);

                fclose(DBFile);

                exit(4);
            }
            else
            {
                fprintf(stderr, "Unexpected end of database file '%s'"
                    " while reading records.\n",
                    szFileName);

                fclose(DBFile);

                exit(4);
            }
        }

        pBuffer[db3Header.nRecordLength] = '\0';

/*      Where we print inside the switch, a program that would use the
 *          database records would parse, and read each field, probably
 *          using a built format string and a call to sscanf().
 */

        switch(pBuffer[0])
        {
            case USABLE_RECORD: /* Valid, undeleted record */
                printf("Record  '%s'\n", &pBuffer[1]);
                break;

            case DELETED_RECORD: /* Record has been deleted. Usually, you'd ignore it. */
                printf("Deleted '%s'\n", &pBuffer[1]);
                break;

            default: /* The record's status is unknown */
                printf("Unknown '%s'\n", &pBuffer[1]);
                break;
        }
    }

    return(0);

}
