/* DBWRITE, written 1992 by Peter D. Hipson  */
/* This program creates a dBASE compatible file. */
/* Derived from DBREAD.C */

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/* Some defines that are useful for dBASE files: */

/* Record status defines (first byte of each record) */
#define DELETED_RECORD '*'
#define USABLE_RECORD ' '

/* Field (column) definitions (capital letters, please) */
#define NUMERIC_FIELD     'N'
#define CHARACTER_FIELD   'C'
#define LOGICAL_FIELD     'L'
#define MEMO_FIELD        'M'
#define DATE_FIELD        'D'
#define FLOAT_FIELD       'F'
#define PICTURE_FIELD     'P'

/* End of dBASE defines */

#pragma pack(1) /* Pack to byte boundaries */

typedef struct   {
/*  Bit fields are arranged from least significant to most significant */
    unsigned int   bfVersion:7;
    unsigned int   bfHasMemo:1;
    unsigned int   bYear:8;
    unsigned char  bMonth;
    unsigned char  bDay;
    long int       lNumberRecords;
    short int      nFirstRecordOffset;
    short int      nRecordLength;
    unsigned char  szReserved[20];
    } DB3HEADER;


typedef struct   {
    char           szColumnName[11];
    char           chType;
    long           lFieldPointer;
    unsigned char  byLength;
    unsigned char  byDecimalPlace;
    char           szReserved[14];
    } COLUMNDEF;



int    main()

{

FILE    *DBFile;

DB3HEADER   db3Header;

struct _DBRECORD {
    char    sStatus[1]; /* Status does not count as a member */
    char    sName[40];
    char    sAddr1[40];
    char    sAddr2[40];
    char    sCity[20];
    char    sState[2];
    char    sZip[5];
    } OurRecord;

char        *pOurRecord;

COLUMNDEF   ColumnDef[6]; /* Six members in OurRecord */

char        szBuffer[200];  /* Work buffer */
char        szTime[26];

char        szFileName[25];

int         i;
int         nColumnCount = 0;
int         nResult;

long        lCurrentRecord = 0;

double      dSales = 0.0; /* Forces loading of floating-point support */

struct tm   *NewTime;
time_t      aClock;

/* Step 1. Determine the layout of the columns (fields). In this 
 *         example, they are predefined. In other programs, you 
 *         might determine the column layout by prompting the user
 *         or by examining the user's data.
 */

    printf("Please enter new database name: ");

    gets(szFileName);

    DBFile = fopen(szFileName, "wb");

    if (DBFile == NULL)
    {
        fprintf(stderr,
            "ERROR: File '%s' couldn't be opened.\n", szFileName);

        exit(4);
    }

/* Step 2. Initialize and write the header record.
 */

    time( &aClock );

    NewTime = localtime(&aClock);

    memset(&db3Header, 0, sizeof(db3Header));

    /* Make it dBASE III compatable */
    db3Header.bfVersion = 3;

    /* Make it a database with no memo fields */
    db3Header.bfHasMemo = 0;

    /* Set the date to now, but UPDATE when closing */
    /* may because date have changed */
    db3Header.bYear = NewTime->tm_year;
    db3Header.bMonth = (unsigned char)(NewTime->tm_mon + 1);
    db3Header.bDay   = (unsigned char)NewTime->tm_mday;

    /* No records in the database yet */
    db3Header.lNumberRecords = 0;

    /* File header, plus column headers, plus a byte for the carriage return  */
    db3Header.nFirstRecordOffset = sizeof(DB3HEADER) + sizeof(ColumnDef) + 2;

    /* Make it the size of a record in the database */
    db3Header.nRecordLength = sizeof(OurRecord);

    nResult = fwrite((char *)&db3Header,
        sizeof(DB3HEADER),
        1,
        DBFile);

    if (nResult != 1)
    {
        fprintf(stderr, "ERROR: File '%s', write error (Database header).\n",
            szFileName);

        fclose(DBFile);

        exit(4);
    }

/* Step 3. Initialize the column headers using the information from step 1.
 */

    memset(ColumnDef, 0, sizeof(ColumnDef));

/*  Do the following for each column, either inline or using a loop */

    strcpy(ColumnDef[0].szColumnName, "Name");
    ColumnDef[0].chType = CHARACTER_FIELD;
    ColumnDef[0].byLength = sizeof(OurRecord.sName);
    ColumnDef[0].byDecimalPlace = 0;

    strcpy(ColumnDef[1].szColumnName, "Address1");
    ColumnDef[1].chType = CHARACTER_FIELD;
    ColumnDef[1].byLength = sizeof(OurRecord.sAddr1);
    ColumnDef[1].byDecimalPlace = 0;

    strcpy(ColumnDef[2].szColumnName, "Address2");
    ColumnDef[2].chType = CHARACTER_FIELD;
    ColumnDef[2].byLength = sizeof(OurRecord.sAddr2);
    ColumnDef[2].byDecimalPlace = 0;

    strcpy(ColumnDef[3].szColumnName, "City");
    ColumnDef[3].chType = CHARACTER_FIELD;
    ColumnDef[3].byLength = sizeof(OurRecord.sCity);
    ColumnDef[3].byDecimalPlace = 0;

    strcpy(ColumnDef[4].szColumnName, "State");
    ColumnDef[4].chType = CHARACTER_FIELD;
    ColumnDef[4].byLength = sizeof(OurRecord.sState);
    ColumnDef[4].byDecimalPlace = 0;

    strcpy(ColumnDef[5].szColumnName, "Zipcode");
    ColumnDef[5].chType = CHARACTER_FIELD;
    ColumnDef[5].byLength = sizeof(OurRecord.sZip);
    ColumnDef[5].byDecimalPlace = 0;

    nResult = fwrite((char *)ColumnDef,
        sizeof(ColumnDef),
        1,
        DBFile);

    if (nResult != 1)
    {
        fprintf(stderr, "ERROR: File '%s', write error (Column headers).\n",
            szFileName);

        fclose(DBFile);

        exit(4);
    }

/* Step 4. Write a carriage return (and a NULL) to meet dBASE standards
 */

    nResult = fwrite((char *)"\x0D\0",
        sizeof(char) * 2,
        1,
        DBFile);

    if (nResult != 1)
    {
        fprintf(stderr, "ERROR: File '%s', write error (Column headers).\n",
            szFileName);

        fclose(DBFile);

        exit(4);
    }

    db3Header.nFirstRecordOffset = (int)ftell(DBFile);

/* Step 5. Get some records for the database.
 */

    memset(&OurRecord, 0, sizeof(OurRecord));

    printf("Enter the name: (or no name to end)");
    gets(szBuffer);
    strncpy(OurRecord.sName, szBuffer, sizeof(OurRecord.sName));

    while (strlen(szBuffer) > 0)
    {
        OurRecord.sStatus[0] = USABLE_RECORD;

        printf("Enter address line 1: ");
        gets(szBuffer);
        strncpy(OurRecord.sAddr1, szBuffer, sizeof(OurRecord.sAddr1));

        printf("Enter address line 2:");
        gets(szBuffer);
        strncpy(OurRecord.sAddr2, szBuffer, sizeof(OurRecord.sAddr2));

        printf("Enter city:");
        gets(szBuffer);
        strncpy(OurRecord.sCity, szBuffer, sizeof(OurRecord.sCity));

        printf("Enter state (2 characters only):");
        gets(szBuffer);
        strncpy(OurRecord.sState, szBuffer, sizeof(OurRecord.sState));

        printf("Enter Zipcode:");
        gets(szBuffer);
        strncpy(OurRecord.sZip, szBuffer, sizeof(OurRecord.sZip));

/*      dBASE records do not contain NULLs, but are padded with
 *      blanks instead, so we convert the NULLs to spaces
 */
        pOurRecord = (char *)&OurRecord;

        for (i = 0; i < sizeof(OurRecord); i++)
        {
            if (pOurRecord[i] == '\0')
            {
                pOurRecord[i] = ' ';
            }
        }

        nResult = fwrite((char *)&OurRecord,
            sizeof(OurRecord),
            1,
            DBFile);

        if (nResult != 1)
        {
            fprintf(stderr, "ERROR: File '%s', write error (Column headers).\n",
                szFileName);

            fclose(DBFile);

            exit(4);
        }
        else
        {
            ++db3Header.lNumberRecords;
        }

        memset(&OurRecord, 0, sizeof(OurRecord));

        printf("Enter the name: (or no name to end)");
        gets(szBuffer);
        strncpy(OurRecord.sName, szBuffer, sizeof(OurRecord.sName));
    }

/* Step 6. Update the file header with the current time and 
 *         the number of records.
 */

    time( &aClock );

    NewTime = localtime(&aClock);

    /* Set the date to now */
    db3Header.bYear = NewTime->tm_year;
    db3Header.bMonth = (unsigned char)(NewTime->tm_mon + 1);
    db3Header.bDay   = (unsigned char)NewTime->tm_mday;

    /* The number of records is already set */

    nResult = fseek(DBFile, (long)0, SEEK_SET);

    if (nResult != 0)
    {
        fprintf(stderr, "ERROR: File '%s', seek error (rewrite of header).\n",
            szFileName);

        fclose(DBFile);

        exit(4);
    }

    nResult = fwrite((char *)&db3Header,
        sizeof(DB3HEADER),
        1,
        DBFile);

    if (nResult != 1)
    {
        fprintf(stderr, "ERROR: File '%s', write error (Database header rewrite).\n",
            szFileName);

        fclose(DBFile);

        exit(4);
    }

/* Finished. Close the file and end the program.
 */

    fclose(DBFile);

    return(0);
}
