/* 
 * Mach Operating System
 * Copyright (c) 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	afs_daemons.c,v $
 * Revision 2.8  90/02/09  12:33:18  berman
 * 	Added jsb's fixes for rpause mechanism.
 * 	Set BUQUIT flag in afs_BQueue() when user has u_rpause cleared.
 * 	Also clear shouldpause in afs_fspause() if u_rpause is clear.
 * 	This prevents resource pausing in afs if the user requested not 
 * 	to, or the kernel is core dumping a user process.
 * 	[90/02/09            berman]
 * 
 * Revision 2.7  89/08/28  22:30:37  af
 * 	Added logic to termination code so that all servers are
 * 	terminated.
 * 	[89/08/05            af]
 * 
 * Revision 2.6  89/08/09  09:13:08  jsb
 * 	Added support for afs resource pausing. Unfortunately, the use of
 * 	daemons complicates issues such as printing the right message on the
 * 	right tty, and catching user interrupts.
 * 	[89/08/08  10:01:17  jsb]
 * 
 * Revision 2.5  89/06/03  15:27:06  jsb
 * 	Merged with newer ITC sources.
 * 	[89/06/02  22:14:55  jsb]
 * 
 * Revision 2.4  89/04/22  15:13:28  gm0w
 * 	Updated to RX version.
 * 	[89/04/14            gm0w]
 * 
 */
/*
 * P_R_P_Q_# (C) COPYRIGHT IBM CORPORATION 1987, 1988
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */

#include <afs/param.h>
#include <sys/types.h>
#include <sys/param.h>
#ifdef	AFS_AUX_ENV
#include <sys/mmu.h>
#include <sys/seg.h>
#include <sys/sysmacros.h>
#include <sys/signal.h>
#include <sys/errno.h>
#endif
#if	!defined(AFS_IBM_ENV) || !defined(sys_rt_r3)
#include <sys/time.h>
#endif	AFS_IBM_ENV
#ifdef	AFS_AIX_ENV
#include <sys/errno.h>
#include <afs/aix_vfs.h>
#else
#include <sys/kernel.h>
#endif
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/protosw.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/file.h>
#include <sys/uio.h>
#include <sys/buf.h>
#ifdef	AFS_GFS_ENV
#include <afs/gfs_vfs.h>
#include <afs/gfs_vnode.h>
#else
#ifdef	AFS_MACH_ENV
#include <vfs/vfs.h>
#include <vfs/vnode.h>
#include <sys/inode.h>
#include <sys/proc.h>
#else	AFS_MACH_ENV
#include <sys/vfs.h>
#include <sys/vnode.h>
#include <ufs/inode.h>
#endif	AFS_MACH_ENV
#endif	AFS_GFS_ENV
#include <netinet/in.h>
#include <sys/mbuf.h>
#include <rpc/types.h>
#include <rpc/xdr.h>

#include <afs/osi.h>
#include <rx/rx.h>

#include <afs/lock.h>
#include <afs/volerrors.h>
#include <afsint/afsint.h>
#include <afs/afs.h>
#include <afs/prs_fs.h>
#include <afs/dir.h>
#ifdef	AFS_MACH_ENV
#include <sys/fs.h>
#endif

/* background request queue size */
struct afs_lock afs_xbrs;		/* lock for brs */
static brsInit = 0;
short afs_brsDaemons = 0;	/* number of daemons waiting for brs requests */
short afs_brsWaiters = 0;	/* number of users waiting for brs buffers */
struct brequest	afs_brs[NBRS];	/* request structures */
struct osi_WaitHandle AFSWaitHandler;

extern char afs_rootVolumeName[];
extern struct VenusFid afs_rootFid;
extern struct osi_dev cacheDev;
extern char *afs_indexFlags;
extern struct afs_lock afs_xvcache;
extern struct vcache *afs_NewVCache();

afs_Daemon() {
    register long code;
    long now;
    long last3MinCheck, last10MinCheck, last60MinCheck;

    last3MinCheck = last10MinCheck = last60MinCheck = 0;
    afs_rootFid.Fid.Volume = 0;
    while (afs_initState < 101) osi_Sleep(&afs_initState);

    /* start off with afs_initState >= 101 (basic init done) */
    while(1) {
	now = osi_Time();

	/* things to do every minute */
	afs_CheckSize(0);		/* make sure we're under disk quota */
	DFlush();			/* write out dir buffers */
	afs_WriteThroughDSlots();	/* write through cacheinfo entries */
	afs_KeepFlocksAlive();		/* keep flocks held */

	if (last3MinCheck + 180 < now) {
	    if (afs_debug & AFSDEB_GENERAL) afs_dp("checking down servers periodically\n");
	    afs_CheckServers(1);	/* only check down servers */
	    last3MinCheck = now;
	}
	if (last10MinCheck + 600 < now) {
	    if (afs_debug & AFSDEB_GENERAL) afs_dp("checking working servers periodically\n");
	    afs_CheckServers(0);
	    afs_GCUserData();	    /* gc old conns */
	    last10MinCheck = now;
	}
	if (last60MinCheck + 3600 < now) {
	    if (afs_debug & AFSDEB_GENERAL) afs_dp("checking volume names periodically\n");
	    afs_CheckRootVolume();
	    afs_CheckVolumeNames();
	    last60MinCheck = now;
	}
	if (afs_initState < 300) {	/* while things ain't rosy */
	    code = afs_CheckRootVolume();
	    if (code ==	0) afs_initState = 300;		    /* succeeded */
	    if (afs_initState <	200) afs_initState = 200;   /* tried once */
	    osi_Wakeup(&afs_initState);
	}

	/* finally sleep for a minute and see what's up */
#ifdef	AFS_GATEWAY
	/* call afs_gateway_sync every 30 seconds to flush lingering files */
	afs_gateway_sync();
	code = osi_Wait(30*1000, &AFSWaitHandler);
	afs_gateway_sync();
	code = osi_Wait(30*1000, &AFSWaitHandler);
#else	AFS_GATEWAY
	code = osi_Wait(60*1000, &AFSWaitHandler);
#endif	AFS_GATEWAY

 	if (afs_termState == AFSOP_STOP_AFS) {
 	    afs_termState = AFSOP_STOP_BKG;
 	    osi_Wakeup(&afs_termState);
 	    return;
 	}
	if (code) { 
	    if (afs_debug & AFSDEB_GENERAL) afs_dp("afs_daemon: sleep code %d\n", code); 
	}
    };
}

afs_CheckRootVolume () {
    char rootVolName[32];
    register struct volume *tvp;

    if (afs_debug & AFSDEB_GENERAL) afs_dp("searching for root volume\n");
    if (*afs_rootVolumeName == 0) {
	strcpy(rootVolName, "root.afs");
    }
    else {
	strcpy(rootVolName, afs_rootVolumeName);
    }
    tvp = afs_GetVolumeByName(rootVolName, LOCALCELL, 1, (struct vrequest *) 0);
    if (tvp) {
	afs_rootFid.Cell = LOCALCELL;
	afs_rootFid.Fid.Volume = (tvp->roVol? tvp->roVol : tvp->volume);
	afs_rootFid.Fid.Vnode = 1;
	afs_rootFid.Fid.Unique = 1;
	afs_initState = 300;    /* won */
	osi_Wakeup(&afs_initState);
	afs_PutVolume(tvp);
    }
#ifdef AFS_GFS_ENV
/* This is to make sure that we update the root gnode */
/* every time root volume gets released */
    {
	extern struct vfs *afs_globalVFS;
	extern int afs_root();
	struct gnode *rootgp;
	struct mount *mp;
	int code;

	if (code = afs_root(afs_globalVFS, &rootgp))
	    return code;
	mp = (struct mount *) afs_globalVFS->vfs_data ;
	mp->m_rootgp = rootgp;
    }
#endif
    if (afs_rootFid.Fid.Volume) return 0;
    else return ENOENT;
}

/* parm 0 is the pathname, parm 1 to the fetch is the chunk number */
BPath(ab)
    register struct brequest *ab; {
    register struct dcache *tdc;
    struct vcache *tvc;
    long offset, len;
    struct vrequest treq;
    long code;

    afs_InitReq(&treq, ab->cred);   /* probably should keep credentials */
    
    code = gop_lookupname(ab->parm[0], AFS_UIOSYS, 1,  (struct vnode **) 0, &tvc);
    osi_FreeSendSpace(ab->parm[0]); /* free path name buffer here */
    if (code) return;
    /* now path may not have been in afs, so check that before calling our cache manager */
#ifdef	AFS_MACH_ENV
    if (tvc->v.v_type != ITYPE_AFS) {
#else	AFS_MACH_ENV
    if (tvc->v.v_op != afs_ops) {
#endif	AFS_MACH_ENV
	/* release it and give up */
	VN_RELE((struct vnode *) tvc);
	return;
    }
    /* here we know its an afs vnode, so we can get the data for the chunk */
    tdc = afs_GetDCache(tvc, ab->parm[1], &treq, &offset, &len, 1);
    if (tdc) {
	afs_PutDCache(tdc);
    }
    afs_PutVCache(tvc);
}

/* parm 0 to the fetch is the chunk number; parm 1 is null or a dcache entry to wakeup */
BPrefetch(ab)
    register struct brequest *ab; {
    register struct dcache *tdc;
    register struct vcache *tvc;
    long offset, len;
    struct vrequest treq;

    afs_InitReq(&treq, ab->cred);   /* probably should keep credentials */
    tvc = ab->vnode;
    tdc = afs_GetDCache(tvc, ab->parm[0], &treq, &offset, &len, 1);
    if (tdc) {
	afs_PutDCache(tdc);
    }
    /* now, dude may be waiting for us to clear DFFetchReq bit; do so */
    tdc = (struct dcache *) (ab->parm[1]);
    if (tdc) {
	/* can't use tdc from GetDCache since afs_GetDCache may fail, but someone
	    may be waiting for our wakeup anyway */
	tdc->flags &= ~DFFetchReq;
	osi_Wakeup(&tdc->validPos);
    }
}

/* already write-locked vcache */
BStore(ab)
    register struct brequest *ab; {
    register struct vcache *tvc;
    register long code;
    struct vrequest treq;

    afs_InitReq(&treq, ab->cred);
    code = 0;
    tvc = ab->vnode;
    UpgradeSToWLock(&tvc->lock);
    tvc->execsOrWriters--;
    if (tvc->execsOrWriters == 0) {
	/* put the file back */
	code = afs_StoreAllSegments(tvc, &treq);
    }
    ReleaseWriteLock(&tvc->lock);
    /* now set final return code, and wakeup anyone waiting */
    if ((ab->flags & BUVALID) == 0) {
	ab->code = afs_CheckCode(code, &treq);    /* set final code, since treq doesn't go across processes */
	ab->flags |= BUVALID;
	if (ab->flags & BUWAIT) {
	    ab->flags &= ~BUWAIT;
	    osi_Wakeup(ab);
	}
    }
}

/* release a held request buffer */
afs_BRelease(ab)
    register struct brequest *ab; {

    ObtainWriteLock(&afs_xbrs);
    if (--ab->refCount <= 0) {
	ab->flags = 0;
    }
    if (afs_brsWaiters) osi_Wakeup(&afs_brsWaiters);
    ReleaseWriteLock(&afs_xbrs);
}

/* return true if bkg fetch daemons are all busy */
int afs_BBusy() {
    if (afs_brsDaemons > 0) return 0;
    return 1;
}

struct brequest *afs_BQueue(aopcode, avc, ause, aparm0, aparm1, aparm2, aparm3, aparm4)
    register short aopcode;
    long ause;
    register struct vcache *avc;
    long aparm1, aparm2, aparm3, aparm0; {
    register int i;
    register struct brequest *tb;

    ObtainWriteLock(&afs_xbrs);
    while (1) {
	tb = afs_brs;
	for(i=0;i<NBRS;i++,tb++) {
	    if (tb->refCount == 0) break;
	}
	if (i < NBRS) {
	    /* found a buffer */
	    tb->opcode = aopcode;
	    tb->vnode = avc;
	    tb->cred = u.u_cred;    /* really should get this as a parm */
	    crhold(tb->cred);
	    if (avc) avc->vrefCount++;
	    tb->refCount = ause+1;
	    tb->parm[0] = aparm0;
	    tb->parm[1] = aparm1;
	    tb->parm[2] = aparm2;
	    tb->parm[3] = aparm3;
	    tb->flags = 0;
	    tb->code = 0;
#ifdef	AFS_MACH_ENV
	    tb->tty = u.u_ttyp;
	    tb->comm = u.u_comm;
	    if (u.u_rpause == 0)	/* Look at user modes for rpause */
		tb->flags |= BUQUIT;
	    tb->sleep = (char *) &tb->sleep;
#endif
	    /* if daemons are waiting for work, wake them up */
	    if (afs_brsDaemons > 0) {
		osi_Wakeup(&afs_brsDaemons);
	    }
	    ReleaseWriteLock(&afs_xbrs);
	    return tb;
	}
	/* no free buffers, sleep a while */
	afs_brsWaiters++;
	ReleaseWriteLock(&afs_xbrs);
	osi_Sleep(&afs_brsWaiters);
	ObtainWriteLock(&afs_xbrs);
	afs_brsWaiters--;
    }
}

afs_BackgroundDaemon() {
    register struct brequest *tb;
    register int i;
    int foundAny;
    static int num_servers= 0;
#ifdef	AFS_MACH_ENV
    struct tty *saved_tty;
#endif

    /* initialize subsystem */
    if (brsInit == 0) {
	Lock_Init(&afs_xbrs);
	bzero(afs_brs, sizeof(afs_brs));
	brsInit = 1;
    }
    num_servers++;
    ObtainWriteLock(&afs_xbrs);
    while (1) {
 	if (afs_termState == AFSOP_STOP_BKG) {
	    ReleaseWriteLock(&afs_xbrs);
	    if (--num_servers > 0)
	    	osi_Wakeup(&afs_brsDaemons);
	    else
		afs_termState = 0;
 	    osi_Wakeup(&afs_termState);
 	    return;
 	}
	/* find a request */
	tb = afs_brs;
	foundAny = 0;
	for(i=0;i<NBRS;i++,tb++) {
	    /* look for request */
	    if ((tb->refCount > 0) && !(tb->flags & BSTARTED)) {
		/* new request, not yet picked up */
		tb->flags |= BSTARTED;
		ReleaseWriteLock(&afs_xbrs);
		foundAny = 1;
		if (afs_debug & AFSDEB_GENERAL) afs_dp("bkg found request %d\n", tb->opcode);
#ifdef	AFS_MACH_ENV
		saved_tty = u.u_ttyp;
		u.u_ttyp = tb->tty;
		tb->self = (char *)u.u_procp;
#endif
		if (tb->opcode == BOP_FETCH)
		    BPrefetch(tb);
		else if (tb->opcode == BOP_STORE)
		    BStore(tb);
		else if (tb->opcode == BOP_PATH)
		    BPath(tb);
		else panic("background bop");
#ifdef	AFS_MACH_ENV
		u.u_ttyp = saved_tty;
		tb->self = (char *)0;
#endif
		if (tb->vnode) {
		    tb->vnode->vrefCount--;	    /* fix up reference count */
		    tb->vnode = (struct vcache *) 0;
		}
		if (tb->cred) {
		    tb->cred->cr_ref--;
		    tb->cred = (struct ucred *) 0;
		}
		afs_BRelease(tb);   /* this grabs and releases afs_xbrs lock */
		ObtainWriteLock(&afs_xbrs);
	    }
	}
	if (!foundAny) {
	    /* wait for new request */
	    afs_brsDaemons++;
	    if (afs_debug & AFSDEB_GENERAL) afs_dp("bkg about to wait\n");
	    ReleaseWriteLock(&afs_xbrs);
	    osi_Sleep(&afs_brsDaemons);
	    ObtainWriteLock(&afs_xbrs);
	    if (afs_debug & AFSDEB_GENERAL) afs_dp("bkg woke up\n");
	    afs_brsDaemons--;
	}
    }
}

#ifdef	AFS_MACH_ENV
struct brequest *afs_BSelf()
{
    register struct brequest *tb;

    ObtainReadLock(&afs_xbrs);
    for (tb = &afs_brs[0]; tb < &afs_brs[NBRS]; tb++) {
	if (tb->self == (char *)u.u_procp) {
	    ReleaseReadLock(&afs_xbrs);
	    return tb;
	}
    }
    ReleaseReadLock(&afs_xbrs);
    return 0;
}

int afs_BSleep(ab)
    struct brequest *ab; {

    if (sleep(ab, PWAIT | PCATCH) == 1) {
	/* we were interrupted */
	ab->flags |= BUQUIT;
	wakeup(ab->sleep);
    }
}

int afs_BWaitProc(rock)
    char *rock; {
    wakeup(rock);
}

/* return 1 if should retry operation, else 0 */
int afs_fspause(avc, acode)
    struct vcache *avc;
    int acode; {
    char *fspath, volName[32];
    char *nospcmsg, *comm;
    struct brequest *tb, *afs_BSelf();
    int pri, ret, shouldpause, didpause;
    char *rock;
    struct fs *fs;
    extern int afs_removeChan;

    shouldpause = (acode == ENOSPC || acode == EDQUOT);
    didpause = (u.u_rpswhich & URPW_NOTIFY);
    if (! shouldpause && ! didpause) {
	return 0;
    }
    rock = (char *) &afs_removeChan;
    nospcmsg = "is full";
    fs = u.u_rpsfs;
    if (fs) {
	if (u.u_rpswhich & URPW_FNOSPC) {
	    rock = (char *) &fs->fs_cstotal.cs_nffree;
	} else if (u.u_rpswhich & URPW_INOSPC) {
	    rock = (char *) &fs->fs_cstotal.cs_nifree;
	    nospcmsg = "out of inodes";
	}
	fspath = fs->fs_fsmnt;
    } else {
	if (!avc || afs_GetVolumeName(&avc->fid, volName)) {
	    fspath = "/afs";
	} else {
	    fspath = volName;
	}
    }
    tb = afs_BSelf();
    if (tb) {
	if (tb->flags & BUQUIT) shouldpause = 0;
	comm = tb->comm;
	pri = PZERO - 3;
	tb->sleep = rock;
    } else {
	if (u.u_rpause == 0) {
		shouldpause = 0; /* user does not want to pause */
	}
	comm = u.u_comm;
	pri = PWAIT | PCATCH;
    }
    if (! shouldpause) {
	if (didpause) {
	    u.u_rpswhich = 0;
	    uprintf("[%s: ... continuing]\r\n", comm);
	}
	return 0;
    }
    if (! didpause) {
	u.u_rpswhich |= URPW_NOTIFY;
	uprintf("[afs_fspause: uid %d]\r\n",u.u_uid);
	uprintf("[%s: %s %s %s, pausing ...]\r\n",
		comm, fspath,
		(fspath == volName ? "volume" : "filesystem"),
		(acode == EDQUOT ? "quota exhausted" : nospcmsg));
    }
    if (rock == (char *) &afs_removeChan) afs_removeChan++;
    osi_CallProc(afs_BWaitProc, rock, 20000);
    ret = sleep(rock, pri);
    osi_CancelProc(afs_BWaitProc, rock);
    if (rock == (char *) &afs_removeChan) afs_removeChan--;
    if (tb) {
	if (!(tb->flags & BUQUIT)) return 1;
    } else {
	if (ret != 1) return 1;
    }
    u.u_rpswhich = 0;
    uprintf("[%s: ... continuing]\r\n", comm);
    return 0;
}
#endif

shutdown_daemons()
{
    bzero(&afs_xbrs, sizeof(struct afs_lock));
    afs_brsDaemons = afs_brsWaiters = brsInit = 0;
    bzero(afs_brs, sizeof(afs_brs));
}
