/* 
 * Mach Operating System
 * Copyright (c) 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	afs_resource.c,v $
 * Revision 2.9  89/08/09  09:48:03  jsb
 * 	Added afs_GetVolumeName for more informative resource pausing messages.
 * 	[89/08/09  09:37:05  jsb]
 * 
 * Revision 2.8  89/08/02  08:00:20  jsb
 * 	Use osi_Zalloc instead of osi_Alloc whereever possible.
 * 	[89/07/31  18:43:54  jsb]
 * 
 * Revision 2.7  89/06/24  23:58:43  jsb
 * 	Newer ITC sources.
 * 	[89/06/24  23:41:29  jsb]
 * 
 * Revision 2.6  89/06/03  15:29:13  jsb
 * 	Merged with newer ITC sources. Undid ntp changes
 * 	(use afsd -nosettime instead).
 * 	[89/06/02  01:21:38  jsb]
 * 
 * Revision 2.5  89/04/22  15:15:11  gm0w
 * 	Updated to RX version.  Added code to print internet addresses.
 * 	At CMUCS we run ntp so do not use fileserver as time server.
 * 	[89/04/14            gm0w]
 * 
 */
/*
 * P_R_P_Q_# (C) COPYRIGHT IBM CORPORATION 1987, 1988
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */

#include <afs/param.h>
#include <sys/types.h>
#include <sys/param.h>
#ifdef	AFS_AUX_ENV
#include <sys/mmu.h>
#include <sys/seg.h>
#include <sys/sysmacros.h>
#include <sys/signal.h>
#include <sys/errno.h>
#endif
#if	!defined(AFS_IBM_ENV) || !defined(sys_rt_r3)
#include <sys/time.h>
#endif	AFS_IBM_ENV
#ifdef	AFS_AIX_ENV
#include <sys/errno.h>
#include <afs/aix_vfs.h>
#else
#include <sys/kernel.h>
#endif
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/protosw.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/file.h>
#include <sys/uio.h>
#ifdef	AFS_GFS_ENV
#include <afs/gfs_vfs.h>
#include <afs/gfs_vnode.h>
#else
#ifdef	AFS_MACH_ENV
#include <vfs/vfs.h>
#include <vfs/vnode.h>
#include <sys/inode.h>
#else	AFS_MACH_ENV
#include <sys/vfs.h>
#include <sys/vnode.h>
#include <ufs/inode.h>
#endif	AFS_MACH_ENV
#endif	AFS_GFS_ENV
#include <netinet/in.h>
#include <sys/mbuf.h>
#include <rpc/types.h>
#include <rpc/xdr.h>

#include <afs/osi.h>
#include <rx/rx.h>

#include <afs/lock.h>
#include <afs/volerrors.h>
#include <afs/voldefs.h>
#include <afsint/afsint.h>
#include <afs/afs.h>
#include <afs/rxkad.h>
#include <afsint/afsvlint.h>

/* This file contains routines used for dealing with users, volumes, cells
 * and connections.  They are related, unfortunately, as follows:
 *
 * 1.  A unix uid and a cell map
 * to a viceid and a set of tokens for authenticating as that vice user.
 *
 * 2.  A vice fid (including cell) maps to the set of servers providing service
 * for the cell and volume of the file named by the fid.
 *
 * 3.  A viceid and server maps to a particular connection that can be used
 * for authenticating as the particular user to that file server.
 *
 * The most-often performed operation starts with a Unix uid and a vice fid,
 * and iterates through all the connections to the servers that both provide
 * service for the file named by the fid and are authenticated as the uid
 * pair in the specified cell (the authentication is expressed as a cell, viceid
 * pair).
 *
 * Note that afs_rlock is the parent lock for cells, users, servers and volumes.
 * See large comment in lock.h about how parent locking really works.
 */

struct server *afs_FindServer ();
extern char *afs_cv2string();
extern struct rx_securityClass *rxvab_NewClientSecurityObject();
extern struct rx_securityClass *rxnull_NewClientSecurityObject();
extern struct rx_securityClass *rxnull_NewServerSecurityObject();
extern char *afs_GetMariner();

extern long afs_waitForever;
extern short afs_waitForeverCount;
static long fvTable[NFENTRIES];
long afs_debug = 0;	    /* general console debug flag */
extern long volumeInode;
extern long afs_setTime;
long afs_setTimeHost=0;	    /* last host we used for time */
extern struct osi_dev cacheDev;		    /* cache device */ 
long afs_volCounter = 1;    /* for allocating volume indices */
struct cell *afs_cells = 0;
struct unixuser *afs_users[NUSERS];
struct server *afs_servers[NSERVERS];
struct volume *afs_volumes[NVOLS];
extern struct afs_lock afs_xvcache;	/* lock for cache */
struct afs_lock afs_xconn;	    /* allocation lock for new things */
struct afs_lock afs_xvolume;    /* allocation lock for volumes */
struct afs_lock afs_xuser;	    /* allocation lock for unixusers */
struct afs_lock afs_xcell;	    /* allocation lock for cells */
struct afs_lock afs_xserver;    /* allocation lock for servers */
struct rx_service *afs_server;	/* server for andrew file system */
struct volume *afs_freeVolList, *Initialafs_freeVolList;
extern struct vcache *afs_vhashTable[];
extern struct dcache **afs_indexTable;
extern char *afs_indexFlags;
extern int RXAFSCB_ExecuteRequest();
long afs_marinerHost;
unsigned short afs_marinerPort;
long afs_logFileInode = 0;
struct afs_lock afs_puttofileLock;

/* convert a volume name to a #; return 0 if can't parse as a number */
afs_vtoi(aname)
register char *aname; {
    register long temp;
    register int tc;
    temp = 0;
    while(tc = *aname++) {
	if (tc > '9' ||	tc < '0') return 0; /* invalid name */
	temp *= 10;
	temp += tc - '0';
    }
    return temp;
}

static char mbuffer[100];
afs_MarinerLog(astring, avc)
    register struct vcache *avc;
    register char *astring; {
    struct sockaddr_in taddr;
    register char *tp, *up;

    taddr.sin_family = AF_INET;
    taddr.sin_addr.s_addr = afs_marinerHost;
    taddr.sin_port = htons(2106);
    tp = mbuffer;
    strcpy(tp, astring);
    tp += strlen(astring);
    *tp++ = ' ';
    up = afs_GetMariner(avc);
    strcpy(tp, up);
    tp += strlen(up);
    *tp++ = '\n';
    /* note, console doesn't want a terminating null */
    osi_NetSend(afs_server->socket, &taddr, mbuffer, tp-mbuffer);
}

/* called with afs_xuser, afs_xserver and afs_xconn locks held, to delete appropriate
    conn structures for au */
static RemoveUserConns(au)
    register struct unixuser *au; {
    register int i;
    register struct server *ts;
    register struct conn *tc, **lc;
    for(i=0;i<NSERVERS;i++) {
	for(ts = afs_servers[i]; ts; ts=ts->next) {
	    lc = &ts->conns;
	    for(tc = *lc; tc; lc = &tc->next, tc = *lc) {
		if (tc->user == au && tc->refCount == 0) {
		    *lc = tc->next;
		    rx_DestroyConnection(tc->id);
		    osi_Zfree(afs_conn_zone, tc);
		    break;  /* at most one instance per server */
		}
	    }
	}
    }
}

/* called from afs_Daemon to garbage collect unixusers no longer using system, and their R conns */
afs_GCUserData() {
    register struct unixuser *tu, **lu, *nu;
    register int i;
    long now, delFlag;

    /* obtain locks in valid order */
    ObtainWriteLock(&afs_xuser);
    ObtainReadLock(&afs_xserver);
    ObtainWriteLock(&afs_xconn);
    now = osi_Time();
    for(i=0;i<NUSERS;i++) {
	for(lu = &afs_users[i], tu = *lu; tu; tu = nu) {
	    delFlag = 0;	/* should we delete this dude? */
	    /* don't garbage collect users in use now (refCount) */
	    if (tu->refCount == 0) {
		if (tu->states & UHasTokens) {
		    if (tu->tokenTime < now - TOKENTIMEOUT) delFlag = 1;
		}
		else {
		    if (tu->tokenTime < now - NOTOKTIMEOUT) delFlag = 1;
		}
	    }
	    nu = tu->next;
	    if (delFlag) {
		*lu = tu->next;
		RemoveUserConns(tu);
		if (tu->stp) osi_Free(tu->stp, tu->stLen);
		osi_Zfree(afs_unixuser_zone, tu);
	    }
	    else {
		lu = &tu->next;
	    }
	}
    }
    ReleaseWriteLock(&afs_xuser);
    ReleaseReadLock(&afs_xserver);
    ReleaseWriteLock(&afs_xconn);
}

afs_CopyError(afrom, ato)
register struct vrequest *afrom;
register struct vrequest *ato; {
    if (!afrom->initd) return;
    afs_FinalizeReq(ato);
    if (afrom->accessError) ato->accessError = 1;
    if (afrom->volumeError) ato->volumeError = 1;
    if (afrom->networkError) ato->networkError = 1;
    return;
}

afs_FinalizeReq(areq)
    register struct vrequest *areq; {
    if (areq->initd) return;
    areq->busyCount = 0;
    areq->accessError = 0;
    areq->volumeError = 0;
    areq->networkError = 0;
    areq->initd = 1;
}

afs_CheckCode(acode, areq)
    register long acode;
    register struct vrequest *areq; {
    if (acode) { 
	if (afs_debug & AFSDEB_GENERAL) afs_dp("returning code %d\n", acode); 
    }
    if (!areq || !areq->initd) return acode;
    if (areq->networkError) return ETIMEDOUT;
    if (acode == 0) return 0;
    if (areq->accessError) return EACCES;
    if (areq->volumeError) return ENODEV;
    return acode;
}

#define	VSleep(at)	osi_Wait((at)*1000, 0)

#define	CVBS	20
char *afs_cv2string(aval)
    register unsigned long aval; {
    static char tbuffer[CVBS];
    register char *tp;
    register int  i;
    int any;
    
    any = 0;
    tp = &tbuffer[CVBS];
    *(--tp) = 0;
    while (aval != 0) {
	i = aval % 10;
	*(--tp) = '0' + i;
	aval /= 10;
	any = 1;
    }
    if (!any) *(--tp) = '0';
    return tp;
}

/*
 * All of the vol cache routines must be called with the afs_xvolume
 * lock held in exclusive mode, since they use static variables.
 * In addition, we don't want two people adding the same volume
 * at the same time.
 */

static struct fvolume staticFVolume;
static long FVIndex = -1;
struct fvolume *afs_GetVolCache(aslot)
    register long aslot; {
    register struct osi_file *tfile;
    register long code;
    if (FVIndex != aslot) {
	tfile = osi_UFSOpen(&cacheDev, volumeInode);
	if (!tfile) panic("open volumeinode");
	osi_Seek(tfile, sizeof(struct fvolume) * aslot);
	code = osi_Read(tfile, &staticFVolume, sizeof(struct fvolume));
	if (code != sizeof(struct fvolume)) panic("read volumeinfo");
	osi_Close(tfile);
	FVIndex = aslot;
    }
    return &staticFVolume;
}

struct fvolume *afs_FindVolCache(acell, avol)
    register long avol;
    register long acell; {
    register long i;
    register struct fvolume *tf;
    i = FVHash(acell,avol);
    for(i=fvTable[i]; i!=0; i=tf->next) {
	tf = afs_GetVolCache(i);
	if (tf->cell == acell && tf->volume == avol) return tf;
    }
    return (struct fvolume *) 0;
}

afs_WriteVolCache() {
    register struct osi_file *tfile;
    register long code;

    if (FVIndex < 0 || FVIndex >= afs_volCounter) panic("volumeinfo alloc");
    tfile = osi_UFSOpen(&cacheDev, volumeInode);
    if (!tfile) panic("open volumeinode");
    osi_Seek(tfile, sizeof(struct fvolume) * FVIndex);
    code = osi_Write(tfile, &staticFVolume, sizeof(struct fvolume));
    if (code != sizeof(struct fvolume)) panic("write volumeinfo");
    osi_Close(tfile);    
    return 0;
}

/* routine to get a slot from the volume list--must be called under afs_xvolume
    lock (write-locked). */
struct volume *afs_GetVolSlot() {
    register struct volume *tv, **lv;
    register long i;
    long bestTime;
    struct volume *bestVp, **bestLp;

    if (!afs_freeVolList) {
	/* get free slot */
	bestTime = 0x7fffffff;
	bestVp = 0;
	bestLp = 0;
	for(i=0;i<NVOLS;i++) {
	    lv = &afs_volumes[i];
	    for(tv = *lv; tv; lv = &tv->next, tv = *lv) {
		if (tv->refCount == 0) {    /* is this one available? */
		    if (tv->accessTime < bestTime) {	/* best one available? */
			bestTime = tv->accessTime;
			bestLp = lv;
			bestVp = tv;
		    }
		}
	    }
	}
	if (!bestVp) panic("getvolslot none");
	tv = bestVp;
	*bestLp = tv->next;
	if (tv->name) osi_Free(tv->name, strlen(tv->name)+1);
	tv->name = (char *) 0;
	/* now write out volume structure to file */
	if (tv->vtix < 0) {
	    tv->vtix = afs_volCounter++;
	    /* now put on hash chain */
	    i = FVHash(tv->cell, tv->volume);
	    staticFVolume.next = fvTable[i];
	    fvTable[i]=tv->vtix;
	}
	else {
	    /* haul the guy in from disk so we don't overwrite hash table next chain*/
	    afs_GetVolCache(tv->vtix);
	}
	FVIndex = tv->vtix;
	staticFVolume.volume = tv->volume;
	staticFVolume.cell = tv->cell;
	staticFVolume.mtpoint = tv->mtpoint;
	staticFVolume.dotdot = tv->dotdot;
	afs_WriteVolCache();
    }
    else {
	tv = afs_freeVolList;
	afs_freeVolList = tv->next;
    }
    return tv;
}

/* reset volume name to volume id mapping cache */
afs_CheckVolumeNames() {
    register long i;
    register struct volume *tv;
    register struct vcache *tvc;

    ObtainReadLock(&afs_xvolume);
    for(i=0;i<NVOLS;i++) {
	for(tv = afs_volumes[i]; tv; tv=tv->next) {
	    ObtainWriteLock(&tv->lock);
	    tv->states |= VRecheck;
	    if (tv->name) {
		osi_Free(tv->name, strlen(tv->name)+1);
		tv->name = (char *) 0;
	    }
	    ReleaseWriteLock(&tv->lock);
	}
    }
    ReleaseReadLock(&afs_xvolume);

    /* next ensure all mt points are re-evaluated */
    ObtainReadLock(&afs_xvcache);
    for(i=0;i<VCSIZE;i++) {
	for(tvc = afs_vhashTable[i]; tvc; tvc=tvc->hnext) {
	    tvc->states &= ~CMValid;
	    /* support for new volume invalidation scheme, whereby volume
		numbers may not change when making a backup or doing a volume
		release.  Requires checking data version in stat block again. */
	    if (tvc->states & CRO) tvc->states &= ~CStatd;
	}
    }
    ReleaseReadLock(&afs_xvcache);
}

static print_internet_address(preamble, address, postamble)
char *preamble, *postamble;
unsigned long address;
{
    address = ntohl(address);
    printf("%s %d.%d.%d.%d %s", preamble, (address >> 24), (address >> 16) & 0xff, (address >> 8) & 0xff,(address) & 0xff, postamble);
}

static ServerDown(aserver)
    register struct server *aserver; {
    if (aserver->isDown) return;
    print_internet_address("afs: Lost contact with server", aserver->host,"\n");
    aserver->isDown = 1;
}

/* return true if we have any callback promises from this server */
static HaveCallBacksFrom(aserver)
struct server *aserver; {
    register long thost;
    register long now;
    register int i;
    register struct vcache *tvc;

    now	= osi_Time();	    /* for checking for expired callbacks */
    thost = aserver->host;  /* host we're looking for */
    for(i=0;i<VCSIZE;i++) { /* for all guys in the hash table */
	for(tvc = afs_vhashTable[i]; tvc; tvc=tvc->hnext) {
	    /* check to see if this entry has an unexpired callback promise
		from the required host */
	    if (thost == tvc->callback && tvc->cbExpires >= now)
		return 1;
	}
    }
    return 0;
}

#ifndef	AFS_MINCHANGE	/* So that some can increase it in param.h */
#define AFS_MINCHANGE 2
#endif
/* check down servers (if adown), or running servers (if !adown) */
afs_CheckServers(adown)
    int	adown; {
    struct vrequest treq;
    register struct server *ts;
    register struct conn *tc;
    register long i;
    register long code;
    long start, end, delta;
    struct timeval tv;
    int setTimer;

    afs_InitReq(&treq, &osi_cred);
    ObtainReadLock(&afs_xserver);
    for(i=0;i<NSERVERS;i++) {
	for(ts = afs_servers[i]; ts; ts=ts->next) {

	    /* see if this server is the type we're checking this time */
	    if (adown != ts->isDown) continue;
	    /* get a connection, even if host is down; bumps conn ref count */
	    if (ts->cell) tc = afs_ConnByHost(ts, AFS_FSPORT, ts->cell->cell, &treq, 1);
	    else continue;
	    /* can't do anything without a connection */
	    if (!tc) continue;

	    if (ts->isDown || HaveCallBacksFrom(ts)) {
		if (ts->isDown) {
		    rx_SetConnDeadTime(tc->id, 3);
		    setTimer = 1;
		}
		else setTimer = 0;
		start = osi_Time();	/* time the gettimeofday call */
		code = RXAFS_GetTime(tc->id, &tv.tv_sec, &tv.tv_usec);
		end = osi_Time();
		/* if we're supposed to set the time, and the call worked quickly
		    (same second response) and this is the host we use for the time
		    and the time is really different, then really set the time */
		if (code == 0 && start == end && afs_setTime != 0 &&
		    (afs_setTimeHost == 0 || tc->server->host == afs_setTimeHost)) {
		    /* set the time */
		    delta = end - tv.tv_sec;   /* how many secs fast we are */
		    /* see if clock has changed enough to make it worthwhile */
		    if (delta >= AFS_MINCHANGE || delta <= -AFS_MINCHANGE) {
			osi_SetTime(&tv);
			if (delta > 0)
			    printf("afs: setting clock back %d seconds.\n", delta);
			else {
#if defined(IBMRTPC) && defined(ATR)
			    /* don't print this message on ATRs, they're always losing time */
#else
			    printf("afs: setting clock ahead %d seconds.\n", -delta);
#endif
			}
		    }
		    afs_setTimeHost = tc->server->host;
		}
		if (setTimer) rx_SetConnDeadTime(tc->id, 50);
		if (code >= 0 && ts->isDown) {
		    /* server back up */
		    print_internet_address("afs: Server", ts->host, "back up\n");
		    ts->isDown = 0;
		    if (afs_waitForeverCount) {
			osi_Wakeup(&afs_waitForever);
		    }
		}
		else if (code < 0) {
		    /* server crashed */
		    ServerDown(ts);
		}
	    }

	    afs_PutConn(tc);	/* done with it now */
	}   /* for each server loop */
    }	    /* for each server hash bucket loop */
    ReleaseReadLock(&afs_xserver);
}

int afs_ResetAccessCache(auser)
    register struct unixuser *auser; {
    register int i, j;
    register struct vcache *tvc;
    ObtainReadLock(&afs_xvcache);
    for(i=0;i<VCSIZE;i++) {
	for(tvc=afs_vhashTable[i]; tvc; tvc=tvc->hnext) {
	    for(j=0;j<CPSIZE;j++) {
		/* really should do this under cache write lock, but that.
		    is hard to under locking hierarchy */
#ifdef	AFS_GATEWAY
		if (tvc->randomUid[j] == auser->uid && tvc->randomHostaddr[j] == auser->hostaddr) {
		    tvc->randomUid[j] = -1;
		    tvc->randomHostaddr[j] = -1;
		}
#else	AFS_GATEWAY
		if (tvc->randomUid[j] == auser->uid)
		    tvc->randomUid[j] = -1;
#endif	AFS_GATEWAY
	    }
	}
    }
    ReleaseReadLock(&afs_xvcache);
}

/* ensure all connections make use of new tokens.  Discard incorrectly-cached access info. */
int afs_ResetUserConns (auser)
    register struct unixuser *auser; {
    register int i;
    register struct server *ts;
    register struct conn *tc;

    ObtainReadLock(&afs_xserver);
    ObtainWriteLock(&afs_xconn);
    for(i=0;i<NSERVERS;i++) {
	for(ts = afs_servers[i]; ts; ts=ts->next) {
	    for (tc = ts->conns; tc; tc=tc->next) {
		if (tc->user == auser)
		    tc->forceConnectFS = 1;
	    }
	}
    }
    ReleaseWriteLock(&afs_xconn);
    ReleaseReadLock(&afs_xserver);
    afs_ResetAccessCache(auser);
}

/* warning: probably shouldn't bump refCount without afs_xuser lock */
#ifdef	AFS_GATEWAY
struct unixuser *afs_FindUser(auid, acell, ahostaddr)
    long ahostaddr;
#else	AFS_GATEWAY
struct unixuser *afs_FindUser(auid, acell)
#endif	AFS_GATEWAY
    long acell;
    register long auid; {
    register struct unixuser *tu;
    register long i;
    i = UHash(auid);
    for(tu = afs_users[i]; tu; tu = tu->next) {
#ifdef	AFS_GATEWAY
	if (tu->uid == auid && tu->cell == acell && tu->hostaddr == ahostaddr) {
#else	AFS_GATEWAY
	if (tu->uid == auid && tu->cell == acell) {
#endif	AFS_GATEWAY
	    tu->refCount++;
	    return tu;
	}
    }
    return (struct unixuser *) 0;
}

afs_ResourceInit() {
    register long i;
    register struct volume *tv;
    static struct rx_securityClass *secobj;

    Lock_Init(&afs_xconn);
    Lock_Init(&afs_xuser);
    Lock_Init(&afs_xvolume);
    Lock_Init(&afs_xcell);
    Lock_Init(&afs_xserver);
    Lock_Init(&afs_puttofileLock);
    secobj = rxnull_NewServerSecurityObject();
    afs_server = rx_NewService(0, 1, "afs", &secobj, 1, RXAFSCB_ExecuteRequest);
    rx_StartServer(0);	/* barf */

    /* create volume list structure */
    tv = (struct volume *) osi_Alloc(MAXVOLS * sizeof(struct volume));
    for(i=0;i<MAXVOLS-1;i++) tv[i].next = &tv[i+1];
    tv[MAXVOLS-1].next = (struct volume *) 0;
    afs_freeVolList = Initialafs_freeVolList = tv;
    for(i=0;i<NFENTRIES;i++)
	fvTable[i] = 0;

    osi_Wakeup(&afs_server);	/* wakeup anyone waiting for it */
    return 0;
}

struct cell *afs_GetCellByName(acellName)
    register char *acellName; {
    register struct cell *tc;
    ObtainWriteLock(&afs_xcell);
    for(tc = afs_cells; tc; tc=tc->next) {
	if (!strcmp(tc->cellName, acellName)) {
	    ReleaseWriteLock(&afs_xcell);
	    return tc;
	}
    }
    ReleaseWriteLock(&afs_xcell);
    return (struct cell *) 0;
}

struct cell *afs_GetCell(acell)
    register long acell; {
    register struct cell *tc;
    ObtainWriteLock(&afs_xcell);
    for(tc = afs_cells; tc; tc=tc->next) {
	if (tc->cell == acell) {
	    ReleaseWriteLock(&afs_xcell);
	    return tc;
	}
    }
    ReleaseWriteLock(&afs_xcell);
    return (struct cell *) 0;
}

struct cell *afs_NewCell(acellName, acellHosts, aflags)
    int aflags;
    char *acellName;
    register long *acellHosts; {
    register struct cell *tc;
    register long i, temp;
    struct server *ts;

    if (*acellHosts == 0) return (struct cell *) 0; /* need >= one host to gen cell # */
    ObtainWriteLock(&afs_xcell);
    for(tc = afs_cells; tc; tc = tc->next) {
	if (!strcmp(tc->cellName, acellName)) {
	    break;
	}
    }
    if (!tc) {
	tc = (struct cell *) osi_Zalloc(afs_cell_zone);
	tc->next = afs_cells;
	afs_cells = tc;
	tc->cellName = (char *) osi_Alloc(strlen(acellName)+1);
	strcpy(tc->cellName, acellName);
	if (aflags & CPrimary)
	    tc->cell = 1;	/* primary cell is always 1 */
	else
	    tc->cell = *acellHosts; /* won't be reused by another cell */
	tc->states = 0;
    }
    tc->states |= (aflags & ~CPrimary);
    if (aflags & CPrimary) tc->states |= CPrimary;  /* only one way */
    bzero(tc->cellHosts, sizeof(tc->cellHosts));
    for(i=0; i<MAXHOSTS; i++) {
	temp = acellHosts[i];
	if (temp == 0) break;
	ts = tc->cellHosts[i] = afs_GetServer(temp, 0);
	ts->cell = tc;
    }
    afs_SortServers(tc->cellHosts);	/* randomize servers */
    ReleaseWriteLock(&afs_xcell);
    return tc;
}

#ifdef	AFS_GATEWAY
struct unixuser *afs_GetUser(auid, acell, ahostaddr)
#else	AFS_GATEWAY
struct unixuser *afs_GetUser(auid, acell)
#endif	AFS_GATEWAY
    long acell;
    register long auid; {
    register struct unixuser *tu;
    register long i;
    i = UHash(auid);
    ObtainWriteLock(&afs_xuser);
    for(tu = afs_users[i]; tu; tu = tu->next) {
#ifdef	AFS_GATEWAY
	if (tu->uid == auid && tu->cell == acell && tu->hostaddr == ahostaddr) {
#else	AFS_GATEWAY
	if (tu->uid == auid && tu->cell == acell) {
#endif	AFS_GATEWAY
	    tu->refCount++;
	    ReleaseWriteLock(&afs_xuser);
	    return tu;
	}
    }
    tu = (struct unixuser *) osi_Zalloc(afs_unixuser_zone);
    bzero(tu, sizeof(struct unixuser));
    tu->next = afs_users[i];
    afs_users[i] = tu;
    tu->uid = auid;
#ifdef	AFS_GATEWAY
    tu->hostaddr = ahostaddr;
#endif	AFS_GATEWAY
    tu->cell = acell;
    tu->vid = UNDEFVID;
    tu->refCount = 1;
    tu->tokenTime = osi_Time();
    ReleaseWriteLock(&afs_xuser);
    return tu;
}

afs_PutUser(au)
    register struct unixuser *au; {
    --au->refCount;
}

/* set the primary flag on a unixuser structure, ensuring that exactly one dude has
    the flag set at any time for a particular unix uid. */
afs_SetPrimary(au, aflag)
    register struct unixuser *au;
    register int aflag; {
    register struct unixuser *tu;
    register int i;
    struct unixuser *pu;

    i = UHash(au->uid);
    pu = (struct unixuser *) 0;
    ObtainWriteLock(&afs_xuser);
    /* see if anyone is this uid's primary cell yet; recording in pu the corresponding user */
    for(tu=afs_users[i]; tu; tu=tu->next) {
#ifdef	AFS_GATEWAY
	if (tu->uid == au->uid && tu->primary && tu->hostaddr == au->hostaddr) pu = tu;
#else	AFS_GATEWAY
	if (tu->uid == au->uid && tu->primary) pu = tu;
#endif	AFS_GATEWAY
    }
    if (pu && !(pu->states & UHasTokens)) {
	/* primary user has unlogged, don't treat him as primary any longer;
	    note that we want to treat him as primary until now, so that
	    people see a primary identity until now. */
	pu->primary = 0;
	pu = (struct unixuser *) 0;
    }
    if (aflag == 1) {
	/* setting au to be primary */
	if (pu) pu->primary = 0;
	au->primary = 1;
    }
    else if (aflag == 0) {
	/* we don't know if we're supposed to be primary or not */
	if (!pu || au == pu) {
	    au->primary = 1;
	}
	else au->primary = 0;
    }
    ReleaseWriteLock(&afs_xuser);
}

/* note that areq may be null, in which case we don't bother to set any
    request status information.
*/
struct volume *afs_GetVolume(afid, areq)
    struct vrequest *areq;
    register struct VenusFid *afid; {
    register struct volume *tv;
    register long i;
    char tbuffer[50];
    i = VHash(afid->Fid.Volume);
    ObtainWriteLock(&afs_xvolume);
    for(tv = afs_volumes[i]; tv; tv=tv->next) {
	if(tv->volume == afid->Fid.Volume && tv->cell == afid->Cell
	   && (tv->states & VRecheck) == 0) {
	    tv->refCount++;
	    ReleaseWriteLock(&afs_xvolume);
	    return tv;
	}
    }
    strcpy(tbuffer, afs_cv2string(afid->Fid.Volume));
    ReleaseWriteLock(&afs_xvolume);
    return afs_GetVolumeByName(tbuffer, afid->Cell, 0, areq);
}

#ifdef	AFS_MACH_ENV
/* aname must point to storage; I think 32 bytes is enough */
afs_GetVolumeName(afid, aname)
    register struct VenusFid *afid;
    char *aname; {
    struct vrequest treq;
    char offLineMsg[256], *OfflineMsg = offLineMsg;
    char motd[256], *MOTD = motd;
    register struct conn *tc;
    register long code;
    struct VolumeStatus volstat;

    afs_InitReq(&treq, &osi_cred);
    do {
	tc = afs_Conn(afid, &treq);
	if (tc)
	    code = RXAFS_GetVolumeStatus(tc->id, afid->Fid.Volume,
					&volstat, &aname, &OfflineMsg, &MOTD);
	else code = -1;
    } while (afs_Analyze(tc, code, afid, &treq));
    return code;
}
#endif

static long CheckVLDB (afid, areq)
struct VenusFid *afid;
register struct vrequest *areq; {
    struct vrequest treq;
    struct conn *tconn;
    register long i;
    struct vldbentry tve;
    register struct volume *tvp;
    struct cell *tcell;
    char tbuffer[50];

    afs_FinalizeReq(areq);
    if (areq->busyCount++ > 20)	{
	areq->volumeError = 1;
	return ENODEV;   /* failed */
    }
    afs_InitReq(&treq, &osi_cred);	/* *must* be unauth for vldb */
    tcell = afs_GetCell(afid->Cell);
    strcpy(tbuffer, afs_cv2string(afid->Fid.Volume));
    do {
	tconn = afs_ConnByMHosts(tcell->cellHosts, AFS_VLPORT, tcell->cell, &treq);
	if (tconn) i = VL_GetEntryByName(tconn->id, tbuffer, &tve);
	else i = -1;
    } while (afs_Analyze(tconn, i, (struct ViceFid *) 0, &treq));
    if (i) {
	areq->volumeError = 1;
	return i;
    }
    /* have info, copy into serverHost array */
    tvp = afs_GetVolume(afid, (struct vrequest *) 0);
    if (tvp) {
	InstallVolumeEntry(tvp, &tve, afid->Cell);
	afs_PutVolume(tvp);
	VSleep(1);	/* Better safe than sorry. */
    }
    else {	/* can't get volume */
	areq->volumeError = 1;
	return ENODEV;
    }
    return 0;
}

struct volume *afs_GetVolumeByName(aname, acell, agood, areq)
    struct vrequest *areq;
    long acell;
    int agood;
    register char *aname; {
    register long code, i;
    register struct volume *tv;
    register struct VolumeInfo *volInfo;
    register struct vldbentry *tve;
    struct cell *tcell;
    int	whichType;  /* which type of volume to look for */
    struct fvolume *tf;
    char *tbuffer;
    struct conn *tconn;
    struct vrequest treq;

    /* allow null request if we don't care about ENODEV/ETIMEDOUT distinction */
    if (!areq) {
	areq = &treq;
	afs_InitReq(&treq, &osi_cred);
    }

    ObtainWriteLock(&afs_xvolume);
    for(code=0;code<NVOLS;code++) {
	for(tv = afs_volumes[code]; tv; tv=tv->next) {
	    if (tv->name && !strcmp(aname,tv->name) && tv->cell == acell && (tv->states&VRecheck) == 0) {
		tv->refCount++;
		ReleaseWriteLock(&afs_xvolume);
		return tv;
	    }
	}
    }
    /* here we don't think we'll find one, so we make the call
	and then see if we have the appropriate volume. */
    ReleaseWriteLock(&afs_xvolume);
    tcell = afs_GetCell(acell);
    if (!tcell) {
	return (struct volume *) 0;
    }
    if (afs_debug & AFSDEB_GENERAL) afs_dp("getvolumebyname about to look for volume %s\n", aname);
    tbuffer = osi_AllocSendSpace();
    volInfo = (struct VolumeInfo *)tbuffer;	/* too big to allocate locally */
    tve = (struct vldbentry *) (tbuffer+1024);
    if (!(tcell->states & CNewVLDB)) {
	/* not known to be using new vldb, try old */
	do {
	    tconn = afs_ConnByMHosts(tcell->cellHosts, AFS_FSPORT, tcell->cell, areq);
	    if (tconn) code = RXAFS_GetVolumeInfo(tconn->id, aname, volInfo);
	    else code = -1;
	} while (afs_Analyze(tconn, code, (struct ViceFid *) 0, areq));
    }
    else code =	1;  /* fake error forces us into new-style code */
    if (code && (tcell->states & COldVLDB)) {
	/* failed, and using old, give up */
	osi_FreeSendSpace(tbuffer);
	return (struct volume *) 0;
    }
    if (code != 0 || volInfo->Type4 == 0xabcd9999) {
	struct vrequest treq;

	afs_InitReq(&treq, &osi_cred);	/* *must* be unauth for vldb */
	/* special hint from file server to use vlserver */
	code = strlen(aname);
	if (code >= 8 && strcmp(aname+code-7, ".backup") == 0)
	    whichType = BACKVOL;
	else if (code >= 10 && strcmp(aname+code-9, ".readonly")==0)
	    whichType = ROVOL;
	else whichType = RWVOL;
	do {
	    tconn = afs_ConnByMHosts(tcell->cellHosts, AFS_VLPORT, tcell->cell, &treq);
	    if (tconn) code = VL_GetEntryByName(tconn->id, aname, tve);
	    else code = -1;
	} while (afs_Analyze(tconn, code, (struct ViceFid *) 0, &treq));
	if (code) {
	    afs_CopyError(&treq, areq);
	    osi_FreeSendSpace(tbuffer);
	    return (struct volume *) 0;
	}
	tcell->states |= CNewVLDB;  /* set flag only if call works */
	tcell->states &= ~COldVLDB; /* clear other flag */
    }
    else {
	tcell->states |= COldVLDB;
	tcell->states &= ~CNewVLDB; /* clear other flag */
    }

    /* otherwise we create the entry and insert this info */
    if (tcell->states & COldVLDB)
	code = volInfo->Vid;	    /* volume number */
    else {
	/* figure out which one we're really interested in (a set is returned) */
	code = afs_vtoi(aname);
	if (code == 0) code = tve->volumeId[whichType];
    }
    i = VHash(code);
    ObtainWriteLock(&afs_xvolume);
    for(tv = afs_volumes[i]; tv; tv=tv->next) {
	if (tv->volume == code && tv->cell == tcell->cell) {
	    break;
	}
    }
    if (!tv) {
	tv = afs_GetVolSlot();
	bzero(tv, sizeof(struct volume));
	tv->cell = tcell->cell;
	Lock_Init(&tv->lock);
	tv->volume = code;
	tv->next = afs_volumes[i];	/* thread into list */
	afs_volumes[i] = tv;
	tf = afs_FindVolCache(tv->cell, code);
	if (tf) {
	    tv->vtix = FVIndex;
	    tv->mtpoint = tf->mtpoint;
	    tv->dotdot = tf->dotdot;
	}
	else tv->vtix = -1;
    }
    tv->refCount++;
    tv->states &= ~VRecheck;	    /* just checked it */
    tv->accessTime = osi_Time();
    ReleaseWriteLock(&afs_xvolume);
    ObtainWriteLock(&tv->lock);
    if (tcell->states & COldVLDB)
	InstallVolumeInfo(tv, volInfo, acell);
    else
	InstallVolumeEntry(tv, tve, acell);
    if (agood) {
	if (!tv->name) {
	    tv->name = osi_Alloc(strlen(aname) + 1);
	    strcpy(tv->name, aname);
	}
    }
    ReleaseWriteLock(&tv->lock);
    osi_FreeSendSpace(tbuffer);
    return tv;
}

/* call this with the volume structure locked */
InstallVolumeInfo(av, avi, acell)
    register struct volume *av;
    register struct VolumeInfo *avi; {
    register int i;
    register struct server *ts;
    register unsigned long *servers;
    long *p;

    p = (long *) &avi->Type0;
    av->rwVol = p[readwriteVolume];
    av->roVol = p[readonlyVolume];
    av->backVol = p[backupVolume];

    servers = &avi->Server0;
    if (avi->ServerCount > MAXHOSTS) panic("InstallVolumeInfo");
    for(i=0;i<avi->ServerCount; i++) {
	ts = afs_GetServer(htonl(*servers), acell);
	av->serverHost[i] = ts;
	servers++;
    }
    if (avi->ServerCount < MAXHOSTS) av->serverHost[avi->ServerCount] = 0;
    if (avi->Type != RWVOL) av->states |= VRO;
    if (avi->Type == BACKVOL) av->states|= VBackup;
    afs_SortServers(av->serverHost);
}

/* call this with the volume structure locked; used for new-style vldb requests */
InstallVolumeEntry(av, ave, acell)
    register struct volume *av;
    register struct vldbentry *ave; {
    register int i, j;
    long mask;

    /* figure out the type of teh volume by looking finding it in the vldb entry */
    if ((ave->flags&VLF_RWEXISTS) && av->volume == ave->volumeId[RWVOL]) {
	mask = VLSF_RWVOL;
    }
    else if ((ave->flags&VLF_ROEXISTS) && av->volume == ave->volumeId[ROVOL]) {
	mask = VLSF_ROVOL;
	av->states |= VRO;
    }
    else if ((ave->flags&VLF_BACKEXISTS) && av->volume == ave->volumeId[BACKVOL]) {
	mask = VLSF_RWVOL;  /* backup always is on the same volume as parent */
	av->states |= (VRO|VBackup);
    }
    else return 1;	/* error: can't find volume in vldb entry */

    /* fill in volume types */
    av->rwVol =	0;	/* first clear these guys out */
    av->roVol = 0;
    av->backVol = 0;
    if (ave->flags & VLF_RWEXISTS) av->rwVol = ave->volumeId[RWVOL];
    if (ave->flags & VLF_ROEXISTS) av->roVol = ave->volumeId[ROVOL];
    if (ave->flags & VLF_BACKEXISTS) av->backVol = ave->volumeId[BACKVOL];

    for(i=0,j=0; i<ave->nServers; i++) {
	if ((ave->serverFlags[i] & mask) == 0) continue;    /* wrong volume */
	av->serverHost[j] = afs_GetServer(htonl(ave->serverNumber[i]), acell);
	j++;
    }
    if (j < MAXHOSTS) av->serverHost[j++] = 0;
    afs_SortServers(av->serverHost);
}

afs_PutVolume(av)
    register struct volume *av; {
    av->refCount--;
}

struct server *afs_FindServer (aserver)
    register long aserver; {
    register struct server *ts;
    register int i;
    i = SHash(aserver);
    for(ts = afs_servers[i]; ts; ts=ts->next) {
	if (ts->host == aserver) {
	    break;
	}
    }
    return ts;
}

/* 16777213 is prime; Knuth volume 2 p 15 random generator */
#define	ranstage(x)	(x)=((0xa0a0a0*(x)+111) % 16777213)
long afs_random() {
    static state = 0;
    register unsigned long temp, i;
    if (!state) {
	temp = osi_Time();
	for(i=0;i<20;i++) {
	    ranstage(temp);
	}
    }
    else
	temp = state;
    ranstage(temp);
    state = temp;
    return temp;
}
    
/* resort up to MAXHOSTS servers by random field */
afs_SortServers(aservers)
struct server **aservers; {
    struct server *tservers[MAXHOSTS];
    unsigned short tvals[MAXHOSTS];
    register struct server *ts;
    register int i,j;
    int count;
    register long temp;
    
    /* initialize for sort */
    count = 0;
    for(i=0;i<MAXHOSTS;i++) {
	ts = tservers[i] = aservers[i];
	if (!ts) break;
	count++;	    /* another */
	tvals[i] = ts->random;
    }
    
    /* sort */
    for(i=0;i<count;i++) {
	for(j=i;j<count;j++) {
	    if (tvals[i] > tvals[j]) {
		/* out of order */
		temp = tvals[i];
		tvals[i] = tvals[j];
		tvals[j] = temp;
		ts = tservers[i];
		tservers[i] = tservers[j];
		tservers[j] = ts;
	    }
	}
    }
    
    /* copy back */
    for(i=0;i<count;i++) {
	aservers[i] = tservers[i];
    }
}

struct server *afs_GetServer (aserver, acell)
    long acell;
    register long aserver; {
    register struct server *ts;
    register int i;
    i = SHash(aserver);
    ObtainSharedLock(&afs_xserver);
    for(ts = afs_servers[i]; ts; ts=ts->next) {
	if (ts->host == aserver) {
	    ReleaseSharedLock(&afs_xserver);
	    return ts;
	}
    }
    UpgradeSToWLock(&afs_xserver);
    ts = (struct server *) osi_Zalloc(afs_server_zone);
    bzero(ts, sizeof(struct server));
    if (acell) ts->cell = afs_GetCell(acell);
    ts->host = aserver;
    ts->next = afs_servers[i];
    /* compute random value for funny sorting */
    afs_servers[i] = ts;
    ts->random = afs_random();
    ReleaseWriteLock(&afs_xserver);
    return ts;
}

struct conn *afs_Conn (afid, areq)
    register struct VenusFid *afid;
    register struct vrequest *areq; {
    register long cell;
    register struct volume *tv;
    register struct conn *tconn;

    cell = afid->Cell;
    tv = afs_GetVolume(afid, areq);
    if (!tv) {
	if (areq) {
	    afs_FinalizeReq(areq);
	    areq->volumeError = 1;
	}
	return (struct conn *) 0;
    }
    tconn = afs_ConnByMHosts(tv->serverHost, AFS_FSPORT, cell, areq);
    afs_PutVolume(tv);
    return tconn;
}

/* forceConnectFS is set whenever we must recompute the connection. UTokensBad
    is true only if we know that the tokens are bad.  We thus clear this flag
    when we get a new set of tokens.
    Having force... true and UTokensBad true simultaneously means that the tokens
    went bad and we're supposed to create a new, unauthenticated, connection.
*/
struct conn *afs_ConnByHost(aserver, aport, acell, areq, aforce)
    register struct server *aserver;
    long acell;
    unsigned short aport;
    register struct vrequest *areq;
    int aforce; {
    register struct unixuser *tu;
    register struct conn *tc;
    struct rx_securityClass *csec;
    int maxTag;
    int service;
    int forceConnect;

    forceConnect = 0;
    if (aserver->isDown	&& !aforce) return (struct conn	*) 0;	/* known down */
#ifdef	AFS_GATEWAY
    tu = afs_GetUser(areq->uid, acell, areq->hostaddr);
#else	AFS_GATEWAY
    tu = afs_GetUser(areq->uid, acell);
#endif	AFS_GATEWAY
    if (tu->states & UTokensBad) {
	/* tokens are bad, use user 0 instead */
	if (tu->uid != 0) {
	    afs_PutUser(tu);
#ifdef	AFS_GATEWAY
	    tu = afs_GetUser(0, acell, 0);
#else	AFS_GATEWAY
	    tu = afs_GetUser(0, acell);
#endif	AFS_GATEWAY
	}
	else {
	    /* user 0's tokens went bad! */
	    tu->vid = UNDEFVID;	/* forcibly disconnect the authentication info */
	    forceConnect = 1;	/* force recreation of connection */
	}
    }
    ObtainSharedLock(&afs_xconn);
    maxTag = -1;    /* biggest tag seen so far */
    for(tc = aserver->conns; tc; tc=tc->next) {
	if (tc->user == tu && tc->port == aport) {
	    break;
	}
    }
    if (!tc) {
	UpgradeSToWLock(&afs_xconn);
	tc = (struct conn *) osi_Zalloc(afs_conn_zone);
	tc->user = tu;
	tc->tag = maxTag+1;
	tc->port = aport;
	tc->server = aserver;
	tc->refCount = 0;   /* bumped below */
	tc->forceConnectFS = 1;
	tc->id = (struct rx_connection *) 0;
	tc->next = aserver->conns;
	aserver->conns = tc;
	ConvertWToSLock(&afs_xconn);
    }
    tc->refCount++;

    if (tc->forceConnectFS || forceConnect) {
	UpgradeSToWLock(&afs_xconn);
	if (tc->id) rx_DestroyConnection(tc->id);
	/* stupid hack to determine if using vldb service or file system service */
	if (aport == htons(7003)) service = 52;
	else service = 1;
	if (tu->vid != UNDEFVID) {
	    /* if AuthHandle is -1 (old) we use vab tickets.  If it is 0 we use kab tickets */
	    if (tu->ct.AuthHandle >= 0 && tu->ct.AuthHandle <= 255) {
		/* kerberos tickets on channel 2 */
		csec = rxkad_NewClientSecurityObject(rxkad_clear, tu->ct.HandShakeKey, 0, tu->stLen, tu->stp);
		tc->id = rx_NewConnection(aserver->host, aport, service, csec, 2);
	    }
	    else {
		csec = rxvab_NewClientSecurityObject(tu->ct.HandShakeKey, tu->stp, 0);
		tc->id = rx_NewConnection(aserver->host, aport, service, csec, 1);
	    }
	}
	else {
	    csec = rxnull_NewClientSecurityObject();
	    tc->id = rx_NewConnection(aserver->host, aport, service, csec, 0);
	}
	tc->forceConnectFS = 0;	/* apparently we're appropriately connected now */
	ConvertWToSLock(&afs_xconn);
    }
    ReleaseSharedLock(&afs_xconn);

    afs_PutUser(tu);
    return tc;
}

struct conn *afs_ConnByMHosts(ahosts, aport, acell, areq)
register struct server *ahosts[];
long acell;
unsigned short aport;
register struct vrequest *areq; {
    register long i;
    register struct conn *tconn;
    register struct server *ts;

    /* try to find any connection from the set */
    for(i=0;i<MAXHOSTS;i++) {
	if ((ts = ahosts[i]) == (struct server *) 0) break;
	tconn = afs_ConnByHost(ts, aport, acell, areq, 0);
	if (tconn) return tconn;
    }
    return (struct conn *) 0;
}

/* you better have a write lock on the volume before calling this one */
afs_ResetVolumeInfo(tv)
    register struct volume *tv; {
    tv->states |= VRecheck;
    if (tv->name) osi_Free(tv->name, strlen(tv->name)+1);
    tv->name = (char *) 0;
}

afs_PutConn(ac)
    register struct conn *ac; {
    ac->refCount--;
}

/* the connection should be held (by refCount) (afs_Conn locks it).  It will
    be released when this routine is done. */
int afs_Analyze(aconn, acode, afid, areq)
    register struct conn *aconn;
    long acode;
    register struct vrequest *areq;
    struct VenusFid *afid; {
    long *pp;
    register long i, code;
    struct server *tsp;
    struct VolumeInfo volInfo;
    register struct volume *tvp;
    long returnCode;
    char tname[20];

    if (afs_debug & AFSDEB_GENERAL) afs_dp("analyzing conn %x, code %d, for user %d\n", aconn, acode, areq->uid);

    if (!aconn) {
	afs_FinalizeReq(areq);
	if (!areq->volumeError)
	    areq->networkError = 1;
	return 0;				/* Fail if no connection. */
    }

    /* Find server associated with this connection. */
    tsp = aconn->server;

    /* If network troubles, mark server as having bogued out again. */
    if (acode < 0)
	ServerDown(tsp);

    if (acode == 0) {
	afs_PutConn(aconn);
	return 0;
    }
    afs_FinalizeReq(areq);
    if (acode == VBUSY) {
        uprintf("afs: Waiting for busy volume %u\n", (afid? afid->Fid.Volume : 0));
	if (areq->busyCount++ > 20) returnCode = 0;
	else {
	    VSleep(20);	    /* poll periodically */
	    returnCode = 1;
	}
    }
    else if (acode == VICETOKENDEAD || (acode & ~0xff) == ERROR_TABLE_BASE_rxk) {
	/* any rxkad error is treated as token expiration */
	struct unixuser *tu;
#ifdef	AFS_GATEWAY
	tu = afs_FindUser(areq->uid, tsp->cell->cell, areq->hostaddr);
#else	AFS_GATEWAY
	tu = afs_FindUser(areq->uid, tsp->cell->cell);
#endif	AFS_GATEWAY
	if (tu)
	    printf("afs: Tokens for user of AFS id %d have expired\n", tu->vid);
	else /* The else case shouldn't be possible and should probably be replaced by a panic? */
	    printf("afs: Tokens for user %d have expired\n", areq->uid);
	aconn->forceConnectFS = 0;		/* don't check until new tokens set */
	aconn->user->states |= UTokensBad;
	returnCode = 1;				/* Try again (as root). */
    }
    /* Check for access violation. */
    else if (acode == EACCES) {
	/* should mark access error in non-existent per-user global structure */
	areq->accessError = 1;
	returnCode = 0;
    }
    /* Check for bad volume data base / missing volume. */
    else if (acode == VNOVOL) {
	struct cell *tcell;
	if (afid)
	    tcell = afs_GetCell(afid->Cell);
	else tcell = (struct cell *) 0;
	if (tcell && (tcell->states & CNewVLDB)) {
	    returnCode = (CheckVLDB(afid, areq) == 0);
	    if (returnCode == 0) areq->volumeError = 1;
	}
	else {
	    /* should set something so ENODEV can be returned by caller */
	    areq->volumeError = 1;
	    returnCode = 0;
	}
    }
    /* Check for moving volume. */
    else if (acode == VMOVED) {
	struct cell *tcell;
	returnCode = 0;	/* default */
        if (afid) {
	    tcell = afs_GetCell(afid->Cell);
	    if (tcell && (tcell->states & CNewVLDB)) {
		if (CheckVLDB(afid, areq)==0) {
		    /* found a new home */
		    returnCode = 1; /* retry failing operation */
		}
		else
		    returnCode = 0;
	    }
	    else {
		strcpy(tname, afs_cv2string(afid->Fid.Volume));
		code = RXAFS_GetVolumeInfo(aconn->id, tname, &volInfo);
		uprintf("afs: Volume %u moved\n", afid->Fid.Volume);
		if (code) {
		    if (afs_debug & AFSDEB_GENERAL) afs_dp("afs: can't find my way home (code %d)\n", code);
		}
		else {
		    tvp = afs_GetVolume(afid, (struct vrequest *) 0);
		    if (tvp) {
			bzero(tvp->serverHost, sizeof(tvp->serverHost));
			pp = (long *) (&volInfo.Server0);
			for(i=0;i<volInfo.ServerCount;i++) {
			    tvp->serverHost[i] = afs_GetServer(htonl(*pp), afid->Cell);
			    pp++;
			}
			afs_PutVolume(tvp);
			VSleep(1);	/* Better safe than sorry. */
			returnCode = 1;	/* try again */
		    }
		}
	    }
	}
    }
    else if (acode >= 0) returnCode = 0;		/* Other random Vice error. */

    if (acode < 0) {
	/* If we get here, code < 0 and we have network/Server troubles.
	 * areq->networkError is not set here, since we always
	 * retry in case there is another server.  However, if we find
	 * no connection (aconn == 0) we set the networkError flag.
	 */
	tsp->isDown = 1;
	VSleep(1);		/* Just a hack for desperate times. */
	returnCode = 1;
    }
    
    /* now unlock the connection and return */
    afs_PutConn(aconn);
    return returnCode;
}

afs_SetLogFile(afile)
    register char *afile; {
    register long code;
    register struct osi_file *tfile;
    struct vnode *filevp;

    code = gop_lookupname(afile, AFS_UIOSYS, 0, (struct vnode *) 0, &filevp);
    if (code) {
	return code;
    }
    /* otherwise we have a VN_HOLD on filevp.  Get the useful info out and return.
      we make use here of the fact that the cache is in the UFS file system,
      and just record the inode number. */
    afs_logFileInode = VTOI(filevp)->i_number;
    VN_RELE(filevp);
    tfile = osi_UFSOpen(&cacheDev, afs_logFileInode);
    osi_Truncate(tfile, 0);
    osi_Close(tfile);
    return 0;
}

int LogFileInUse = 0;
struct osi_file *Logtfile;

StartLogFile() {
    if (LogFileInUse)
	osi_Close(Logtfile);
    Logtfile = osi_UFSOpen(&cacheDev, afs_logFileInode);
    if (!Logtfile) panic("open LogFile");
    osi_Truncate(Logtfile, 0);	    /* to start afresh */
    LogFileInUse = 1;
}

EndLogFile() {
    if (!LogFileInUse)
	return;	    /* Should we complain here? */
    LogFileInUse = 0;
    osi_Close(Logtfile);
}

afs_dp(a,b,c,d,e,f,g,h,i,j)
long a,b,c,d,e,f,g,h,i,j; {
    if (afs_debug)
	if (LogFileInUse)
	    fprint(a,b,c,d,e,f,g,h,i,j);
	else
	    printf(a,b,c,d,e,f,g,h,i,j);
}

fprint(fmt, x1)
char *fmt;
unsigned x1;
{
    fprf(fmt, &x1);
}

fprf(fmt, adx)
	register char *fmt;
	register u_int *adx;
{
	register int b, c, i;
	char *s;
	int any;

loop:
	while ((c = *fmt++) != '%') {
		if(c == '\0')
			return;
		puttofile(c);
	}
again:
	c = *fmt++;
	/* THIS CODE IS VAX DEPENDENT IN HANDLING %l? AND %c */
	switch (c) {

	case 'l':
		goto again;
	case 'x': case 'X':
		b = 16;
		goto number;
	case 'd': case 'D':
	case 'u':		/* what a joke */
		b = 10;
		goto number;
	case 'o': case 'O':
		b = 8;
number:
		fprintn((u_long)*adx, b);
		break;
	case 'c':
		b = *adx;
		for (i = 24; i >= 0; i -= 8)
			if (c = (b >> i) & 0x7f)
				puttofile(c);
		break;
	case 'b':
		b = *adx++;
		s = (char *)*adx;
		fprintn((u_long)b, *s++);
		any = 0;
		if (b) {
			puttofile('<');
			while (i = *s++) {
				if (b & (1 << (i-1))) {
					if (any)
						puttofile(',');
					any = 1;
					for (; (c = *s) > 32; s++)
						puttofile(c);
				} else
					for (; *s > 32; s++)
						;
			}
			if (any)
				puttofile('>');
		}
		break;

	case 's':
		s = (char *)*adx;
		while (c = *s++)
			puttofile(c);
		break;

	case '%':
		puttofile('%');
		break;
	}
	adx++;
	goto loop;
}

/*
 * Printn prints a number n in base b.
 * We don't use recursion to avoid deep kernel stacks.
 */
fprintn(n, b)
	u_long n;
{
	char prbuf[11];
	register char *cp;

	if (b == 10 && (int)n < 0) {
		puttofile('-');
		n = (unsigned)(-(int)n);
	}
	cp = prbuf;
	do {
		*cp++ = "0123456789abcdef"[n%b];
		n /= b;
	} while (n);
	do
		puttofile(*--cp);
	while (cp > prbuf);
}

puttofile(c)
register int c;
{
    static int bufindex = 0;
    static char filebuffer[4096];
    int code;

    /* These locks are only experimental for now! */
    ObtainWriteLock(&afs_puttofileLock);
    if (c == '\n') {
	filebuffer[bufindex++] = '\r';
	filebuffer[bufindex++] = c;
	code = osi_Write(Logtfile, filebuffer, bufindex);
	if (code != bufindex) printf("puttofile: osi_write(c=%d, i=%d)\n", code, bufindex); /* the printf should be replaced by panic! */
	bufindex = 0;
	ReleaseWriteLock(&afs_puttofileLock);
	return;
    }
    filebuffer[bufindex++] = c;
    ReleaseWriteLock(&afs_puttofileLock);
}

/* run everywhere, checking locks */
afs_CheckLocks() {
    register int i;
    printf("Looking for locked data structures.\n");
    printf("conn %x, volume %x, user %x, cell %x, server %x\n",
	    afs_xconn, afs_xvolume, afs_xuser, afs_xcell, afs_xserver);
    {
	register struct vcache *tvc;
	for(i=0;i<VCSIZE;i++) {
	    for(tvc = afs_vhashTable[i]; tvc; tvc=tvc->hnext) {
		if (tvc->vrefCount)
		    printf("Stat cache entry at %x is held\n", tvc);
		if (CheckLock(&tvc->lock))
		    printf("Stat entry at %x is locked\n", tvc);
	    }
	}
    }
    {
	register struct dcache *tdc;
	for(i=0;i<afs_cacheFiles;i++) {
	    tdc = afs_indexTable[i];
	    if (tdc) {
		if (tdc->refCount)
		    printf("Disk entry %d at %x is held\n", i, tdc);
	    }
	    if (afs_indexFlags[i] & IFDataMod)
		printf("Disk entry %d at %x has IFDataMod flag set.\n", i, tdc);
	}
    }
    {
	register struct server *ts;
	register struct conn *tc;
	for(i=0;i<NSERVERS;i++) {
	    for(ts = afs_servers[i]; ts; ts=ts->next) {
		if (ts->isDown)
		    printf("Server entry for host %x (at %x) is marked down\n", ts->host, ts);
		for(tc = ts->conns; tc; tc=tc->next) {
		    if (tc->refCount)
			printf("conn at %x (server %x) is held\n", tc, ts->host);
		}
	    }
	}
    }
    {
	register struct volume *tv;
	for(i=0;i<NVOLS;i++) {
	    for(tv = afs_volumes[i]; tv; tv=tv->next) {
		if (CheckLock(&tv->lock))
		    printf("volume at %x is locked\n", tv);
		if (tv->refCount)
		    printf("volume at %x is held\n", tv);
	    }
	}
    }
    {
	register struct unixuser *tu;
	for(i=0;i<NUSERS;i++) {
	    for(tu = afs_users[i]; tu; tu=tu->next) {
		if (tu->refCount) printf("user at %x is held\n", tu);
	    }
	}
    }
    printf("Done.\n");
}


shutdown_AFS() {
    int i;

    /* Free Cells table allocations */
    { 
	struct cell *tc, *ntc;
	for (tc = afs_cells; tc;) {
	    ntc = tc->next;
	    if (tc->cellName) osi_Free(tc->cellName, strlen(tc->cellName)+1);
	    osi_Zfree(afs_cell_zone, tc);
	    tc = ntc;
	}
    }

    /* Free Volumes table allocations */
    { 
	struct volume *tv;
	for (i = 0; i < NVOLS; i++) {
	    for (tv = afs_volumes[i]; tv; tv = tv->next) {
		if (tv->name) {
		    osi_Free(tv->name, strlen(tv->name)+1);
		    tv->name = 0;
		}
	    }
	    afs_volumes[i] = 0;
	}
    }

    /* Free FreeVolList allocations */
    osi_Free(Initialafs_freeVolList, MAXVOLS * sizeof(struct volume));
    afs_freeVolList = 0;

    /* Free Users table allocation */
    { 
	struct unixuser *tu, *ntu;
	for (i=0; i < NUSERS; i++) {
	    for (tu=afs_users[i]; tu;) {
		ntu = tu->next;
		osi_Zfree(afs_unixuser_zone, tu);
		tu = ntu;
	    }
	    afs_users[i] = 0;
	}
    }

    /* Free Servers table allocation */
    { 
	struct server *ts, *nts;
	struct conn *tc, *ntc;
	for (i=0; i < NSERVERS; i++) {
	    for (ts = afs_servers[i]; ts;) {
		nts = ts->next;
		if (ts->conns) {
		    tc = ts->conns;
		    while (tc) {
			ntc = tc->next;
			osi_Zfree(afs_conn_zone, tc);
			tc = ntc;
		    }
		}
		osi_Zfree(afs_server_zone, ts);
		ts = nts;
	    }
	    afs_servers[i] = 0;
	}
    }
    for (i=0; i<NFENTRIES; i++)
	fvTable[i] = 0;
    /* Reinitialize local globals to defaults */
    LogFileInUse = 0;
    afs_logFileInode = 0;
    afs_marinerHost = afs_marinerPort = 0;
    afs_cells = 0;
    afs_setTimeHost = 0;
    afs_volCounter = 1;
    afs_waitForever = afs_waitForeverCount = 0;
    afs_debug = 0;
#ifdef notdef
    /* do something for rx, too */
    osi_FreeSocket(afs_server->socket);
    afs_server = 0;
    osi_FreeSocket(afs_rftpServer->socket);
    afs_rftpServer = 0;
#endif notdef
    FVIndex = -1;
    bzero(&afs_rootFid, sizeof(struct VenusFid));
}
