/* 
 * Mach Operating System
 * Copyright (c) 1988, 1989, 1990 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */

/*
 *	Apple Macintosh II Mach (macmach)
 *
 *	File: mac2emul/macmemory.c
 *	Author: David E. Bohman II (CMU macmach)
 */

/*
 * RCS documentation
 *
 * $Header$
 *
 * $Log$
 */

/*
 * Macintosh memory
 * management
 */

#include <sys/param.h>
#include <sys/user.h>

#include <kern/task.h>
#include <vm/vm_map.h>

#include <mach/vm_param.h>

#include <mac2slotmgr/Types.h>
#include <mac2slotmgr/Slots.h>
#include <mac2slotmgr/slotmgr.h>

#include <mac2emul/macdefs.h>
#include <mac2emul/macmemory.h>

/*
 * map in physical page frames
 */
#define PHYS_DEV makedev(3, 3)
int macmmap();
#define PHYS_MAPFUN macmmap

/*
 * allocate zero-fill memory
 */
#define ZERO_DEV NODEV

struct mac_object_mapping mac_24bit_device_mappings[] = {
    {
	ROM_24B_ADDR, ROM_24B_SIZE, VM_PROT_READ,
	PHYS_DEV, PHYS_MAPFUN, ROM_32B_ADDR,
    },
    {
	SND_24B_ADDR, SND_24B_SIZE, VM_PROT_READ|VM_PROT_WRITE,
	PHYS_DEV, PHYS_MAPFUN, SND_32B_ADDR,
    },
    {
	IWM_24B_ADDR, IWM_24B_SIZE, VM_PROT_READ|VM_PROT_WRITE,
	PHYS_DEV, PHYS_MAPFUN, IWM_32B_ADDR,
    },
    {
	SCC_24B_ADDR, SCC_24B_SIZE, VM_PROT_READ|VM_PROT_WRITE,
	PHYS_DEV, PHYS_MAPFUN, SCC_32B_ADDR,
    },
    { 0 }
};

struct mac_object_mapping mac_32bit_device_mappings[] = {
    {
	ROM_32B_ADDR, ROM_32B_SIZE, VM_PROT_READ,
	PHYS_DEV, PHYS_MAPFUN, ROM_32B_ADDR,
    },
    {
	ROM_HIGH_32B_ADDR, ROM_32B_SIZE, VM_PROT_READ,
	PHYS_DEV, PHYS_MAPFUN, ROM_32B_ADDR,
    },
    {
	SND_32B_ADDR, SND_32B_SIZE, VM_PROT_READ|VM_PROT_WRITE,
	PHYS_DEV, PHYS_MAPFUN, SND_32B_ADDR,
    },
    {
	IWM_32B_ADDR, IWM_32B_SIZE, VM_PROT_READ|VM_PROT_WRITE,
	PHYS_DEV, PHYS_MAPFUN, IWM_32B_ADDR,
    },
    {
	SCC_32B_ADDR, SCC_32B_SIZE, VM_PROT_READ|VM_PROT_WRITE,
	PHYS_DEV, PHYS_MAPFUN, SCC_32B_ADDR,
    },
    { 0 }
};

mac_set_addr_mode(pcb, to32bit)
register struct pcb *pcb;
register boolean_t to32bit;
{
    register struct mac_internal *mac = pcb->pcb_emul;

    if (mac->flags&MAC_F_24BIT) {
	if (to32bit == TRUE) {
	    pmap_set_32bit_mode();
	    mac->flags &= ~MAC_F_24BIT;
	}
    }
    else {
	if (to32bit == FALSE) {
	    pmap_set_24bit_mode();
	    mac->flags |= MAC_F_24BIT;
	}
    }
}

mac_setup_memory(pcb)
{
    mac_enable_device_mappings(mac_24bit_device_mappings);
    mac_enable_device_mappings(mac_32bit_device_mappings);
    mac_enable_slot_mappings(pcb);
}

mac_enable_device_mappings(m)
register struct mac_object_mapping *m;
{
    register vm_map_t map = current_task()->map;
    register memory_object_t pager;
    memory_object_t device_pager_create();
    
    for (; m->m_task_address; m++) {
	if (m->m_object_dev == ZERO_DEV) {
	    vm_allocate(map, &m->m_task_address, m->m_task_size, FALSE);
	    continue;
	}
	pager = device_pager_create(m->m_object_dev, m->m_object_mapfun,
				    m->m_task_prot, m->m_object_offset,
				    m->m_task_size);
	if (pager == PORT_NULL)
	    return;
	
	vm_map(map, &m->m_task_address, m->m_task_size, 0, FALSE, pager,
	       m->m_object_offset, FALSE,
	       m->m_task_prot, m->m_task_prot,
	       VM_INHERIT_SHARE);
	
	port_release(pager);
    }
}

mac_disable_device_mappings(m)
register struct mac_object_mapping *m;
{
    register vm_map_t map  = current_task()->map;
    
    for (; m->m_task_address; m++)
	vm_deallocate(map, m->m_task_address, m->m_task_size);
}

mac_enable_slot_mappings(pcb)
{
    register vm_map_t map = current_task()->map;
    register memory_object_t pager;
    memory_object_t device_pager_create();
    register slot;
    vm_offset_t task_addr;

    for (slot = 0x9; slot <= 0xe; slot++)
	if (slot_to_slotdata_ptr(slot)->SFlags&SLOT_MAPPABLE) {
	    pager = device_pager_create(PHYS_DEV, PHYS_MAPFUN,
					VM_PROT_READ|VM_PROT_WRITE,
					SLOT_32B_ADDR(slot), SLOT_32B_SIZE);
	    if (pager == PORT_NULL)
		return;

	    task_addr = SLOT_32B_ADDR(slot);
	    vm_map(map, &task_addr, SLOT_32B_SIZE, 0, FALSE, pager,
		   SLOT_32B_ADDR(slot), FALSE,
		   VM_PROT_READ|VM_PROT_WRITE, VM_PROT_READ|VM_PROT_WRITE,
		   VM_INHERIT_SHARE);

	    task_addr = SLOT_24B_ADDR(slot);
	    vm_map(map, &task_addr, SLOT_24B_SIZE, 0, FALSE, pager,
		   SLOT_32B_ADDR(slot), FALSE,
		   VM_PROT_READ|VM_PROT_WRITE, VM_PROT_READ|VM_PROT_WRITE,
		   VM_INHERIT_SHARE);

	    port_release(pager);
	    mac_slot_enb(pcb, slot);
	}
}

mac_disable_slot_mappings()
{
    register vm_map_t map = current_task()->map;
    register slot;

    for (slot = 0x9; slot <= 0xe; slot++) {
	vm_deallocate(map, SLOT_32B_ADDR(slot), SLOT_32B_SIZE);
	vm_deallocate(map, SLOT_24B_ADDR(slot), SLOT_24B_SIZE);
    }
}

macmmap(dev, offset, prot)
register vm_offset_t offset;
{
    return (mac2_btop(offset));
}
