/* 
 * Mach Operating System
 * Copyright (c) 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	rx.c,v $
 * Revision 2.5  89/08/28  22:40:24  af
 * 	Added termination logic and code.  No guarantees.
 * 	[89/08/05            af]
 * 
 * Revision 2.4  89/08/02  08:07:29  jsb
 * 	Use {osi,rxi}_Zalloc instead of {osi,rxi}_Alloc. Allocate rx_packets
 * 	on demand instead of preallocating a huge free list.
 * 	[89/07/31  17:47:06  jsb]
 * 
 * Revision 2.3  89/06/03  15:37:53  jsb
 * 	Merged with newer ITC sources.
 * 	[89/06/02  00:54:38  jsb]
 * 
 * Revision 2.2  89/04/22  15:28:02  gm0w
 * 	Updated to RX version.
 * 	[89/04/14            gm0w]
 * 
 */

/*
****************************************************************************
*        Copyright IBM Corporation 1988, 1989 - All Rights Reserved        *
*                                                                          *
* Permission to use, copy, modify, and distribute this software and its    *
* documentation for any purpose and without fee is hereby granted,         *
* provided that the above copyright notice appear in all copies and        *
* that both that copyright notice and this permission notice appear in     *
* supporting documentation, and that the name of IBM not be used in        *
* advertising or publicity pertaining to distribution of the software      *
* without specific, written prior permission.                              *
*                                                                          *
* IBM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL *
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL IBM *
* BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY      *
* DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER  *
* IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING   *
* OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.    *
****************************************************************************
*/

/* RX:  Extended Remote Procedure Call */

#ifdef	KERNEL
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <afs/param.h>
#include <afs/osi.h>
#if	(defined(AFS_AUX_ENV) || defined(AFS_AIX_ENV))
#include <sys/systm.h>
#endif
#ifdef RXDEBUG
#undef RXDEBUG	    /* turn off debugging */
#endif RXDEBUG
#include <rx/rx_kernel.h>
#include <rx/rx_clock.h>
#include <rx/rx_queue.h>
#include <rx/rx.h>
#include <rx/rx_globals.h>
#else	KERNEL
# include <sys/types.h>
# include <sys/errno.h>
# include <sys/socket.h>
# include <sys/file.h>
# include <netdb.h>
# include <sys/stat.h>
# include <netinet/in.h>
# include <sys/time.h>
# include <afs/param.h>
# include "rx_user.h"
# include "rx_clock.h"
# include "rx_queue.h"
# include "rx.h"
# include "rx_globals.h"
extern int errno;
#endif	KERNEL

#define rxi_OverQuota(packetclass) (rx_nFreePackets - 1 < rx_packetQuota[packetclass])

long rxi_packetReclaims=0;
long rxi_minReclaim = RX_MIN_RECLAIM;
long rxi_nCalls = 0;

/* ---------------------------------------Exported Interfaces--------------------------------------------- */

/* Initialize rx.  A port number may be mentioned, in which case this becomes the default port number for any service installed later.  If 0 is provided for the port number, a random port will be chosen by the kernel.  Whether this will ever overlap anything in /etc/services is anybody's guess...  Returns 0 on success, -1 on error. */
int rx_Init(port)
{
    static zones_initialized = 0;
    static int status = 1;
    struct timeval tv;
    register long i;
    register struct rx_packet *p, *e;
    SPLVAR;

    if (status != 1) return status; /* Already initialized; return previous error code. */
    NETPRI;

    if (! zones_initialized) {
	rx_zone_init();
	rxvab_zone_init();
	rxkad_zone_init();
	zones_initialized = 1;
    }
    osi_GetTime(&tv);
    rx_epoch = tv.tv_sec;   /* Start time of this package */
    /* *Slightly* random start time for the cid.  This is just to help out with the hashing function at the peer */
    rx_nextCid = ((tv.tv_sec ^ tv.tv_usec) << RX_CIDSHIFT);
    rx_stats.minRtt.sec = 9999999;
    rx_port = port;
    rx_socket =	rxi_GetUDPSocket(port);  /* Allocate and initialize a socket for client and perhaps server connections */
    if (rx_socket == OSI_NULLSOCKET) {
	status = -1;
	USERPRI;
	return status;
    }
    rx_connHashTable = (struct rx_connection **) rxi_Alloc(rx_hashTableSize*sizeof(struct rx_connection *));
    rx_peerHashTable = (struct rx_peer **) rxi_Alloc(rx_hashTableSize*sizeof(struct rx_peer *));    

    rx_lastAckDelay.sec = 0;
    rx_lastAckDelay.usec = 500000;

    clock_Init();
    rxevent_Init(20, rxi_ReScheduleEvents);

    /* Malloc up a bunch of packet buffers */
    if (rx_nPackets < RX_MIN_PACKETS) rx_nPackets = RX_MIN_PACKETS;
    USERPRI;
    i = rx_nPackets/3;
    if (i > rxi_minReclaim) rxi_minReclaim = i;
    NETPRI;
    rx_nFreePackets = rx_nPackets;

    /* Initialize various global queues */
    queue_Init(&rx_idleServerQueue);
    queue_Init(&rx_incomingCallQueue);
    queue_Init(&rx_freeCallQueue);

    /* Start listener process (exact function is dependent on the implementation environment--kernel or user space) */
    rxi_StartListener();

    USERPRI;
    status = 0;
    return status;
}

int rx_Shutdown()
{
    register struct rx_serverQueueEntry *sq;

    if (queue_IsEmpty(&rx_idleServerQueue))
         return 0;

    sq = queue_First(&rx_idleServerQueue, rx_serverQueueEntry);
    queue_Remove(sq);

    osi_Wakeup(sq);
    return 1;
}


/* This routine must be called if any services are exported.  If the donateMe flag is set, the calling process is donated to the server process pool */
void rx_StartServer(donateMe)
{
    SPLVAR;
    clock_NewTime();
    NETPRI;
    /* Start server processes, if necessary (exact function is dependent on the implementation environment--kernel or user space).  DonateMe will be 1 if there is 1 pre-existing proc, i.e. this one.  In this case, one less new proc will be created rx_StartServerProcs. */
    rxi_StartServerProcs(donateMe);

    /* Turn on reaping of idle server connections */
    rxi_ReapConnections();

    if (donateMe) rx_ServerProc(); /* Never returns */
    USERPRI;
    return;
}

/* Create a new client connection to the specified service, using the specified security object to implement the security model for this connection. */
struct rx_connection *rx_NewConnection(shost, sport, sservice, securityObject, serviceSecurityIndex)
    register u_long shost;	    /* Server host */
    u_short sport;		    /* Server port */
    u_short sservice;		    /* Server service id */
    register struct rx_securityClass *securityObject;
    int serviceSecurityIndex;
{
    int hashindex;
    long cid;
    register struct rx_connection *conn;
    SPLVAR;
    clock_NewTime();
#ifdef RXDEBUG
    if (Log) fprintf(Log, "rx_NewConnection(host %x, port %u, service %u, securityObject %x, serviceSecurityIndex %d)\n", shost, sport, sservice, securityObject, serviceSecurityIndex);
#endif
    cid = (rx_nextCid += RX_MAXCALLS);
    hashindex = CONN_HASH(shost, sport, cid, rx_epoch, RX_CLIENT_CONNECTION);
    NETPRI;
    conn = rxi_AllocConnection();
    conn->type = RX_CLIENT_CONNECTION;
    conn->cid = cid;
    conn->epoch = rx_epoch;
    conn->peer = rxi_FindPeer(shost, sport);
    conn->serviceId = sservice;
    conn->securityObject = securityObject;
    /* This doesn't work in all compilers with void (they're buggy), so fake it with VOID */
    conn->securityData = (VOID *) 0;
    conn->securityIndex = serviceSecurityIndex;
    rx_SetConnDeadTime(conn, rx_connDeadTime);

    conn->next = rx_connHashTable[hashindex];
    rx_connHashTable[hashindex] = conn;
    RXS_NewConnection(securityObject, conn);
    USERPRI;
    rx_stats.nClientConns++;
    return conn;
}

void rxi_SetConnDeadTime(conn, seconds)
    register struct rx_connection *conn;
    register int seconds;
{
    /* This is completely silly; perhaps exponential back off would be more reasonable? XXXX */
    /* But, then, the retransmit stuff is all bogus, too XXXX */
    conn->secondsUntilDead = seconds;
    conn->secondsUntilPing = seconds/6;
    if (conn->secondsUntilPing == 0) conn->secondsUntilPing = 1; /* XXXX */
}

/* Destroy the specified connection */
void rx_DestroyConnection(conn) 
    register struct rx_connection *conn;
{
    register struct rx_connection **conn_ptr;
    register int i;
    register int havecalls = 0;
    SPLVAR;

    clock_NewTime();

    NETPRI;
    /* If the client previously called rx_NewCall, but it is still waiting, treat this as a running call, and wait to destroy the connection later when the call completes. */
    if (conn->type == RX_CLIENT_CONNECTION && (conn->flags & RX_CONN_MAKECALL_WAITING)) {
	conn->flags |= RX_CONN_DESTROY_ME;
	USERPRI;
	return;
    }
    /* Check for extant references to this connection */
    for (i = 0; i<RX_MAXCALLS; i++) {
	register struct rx_call *call = conn->call[i];
	if (call) {
	    havecalls = 1;
	    if (conn->type == RX_CLIENT_CONNECTION) {
		havecalls = 1;
		if (call->delayedAckEvent) {
		    /* Push the final acknowledgment out now--there won't be a subsequent call to acknowledge the last reply packets */
		    rxevent_Cancel(call->delayedAckEvent);
		    rxi_AckAll((struct rxevent *)0, call, 0);
		}
	    }
	}
    }

    if (havecalls) {
	/* Don't destroy the connection if there are any call structures still in use */
	conn->flags |= RX_CONN_DESTROY_ME;
	USERPRI;
	return;
    }

    /* Remove from connection hash table before proceeding */
    for (conn_ptr = &rx_connHashTable[CONN_HASH(peer->host, peer->port, conn->cid, conn->epoch, conn->type)]; *conn_ptr; conn_ptr = &(*conn_ptr)->next) {
	if (*conn_ptr == conn) {
	    *conn_ptr = conn->next;
	    break;
	}
    }

    /* Notify the service exporter, if requested, that this connection is being destroyed */
    if (conn->type == RX_SERVER_CONNECTION && conn->service->destroyConnProc) (*conn->service->destroyConnProc)(conn);

    /* Notify the security module that this connection is being destroyed */
    RXS_DestroyConnection(conn->securityObject, conn);
    
    /* Make sure that the connection is completely reset, before deleting it. */
    rxi_ResetConnection(conn);

    if (--conn->peer->refCount == 0) conn->peer->idleWhen = clock_Sec();

    if (conn->type == RX_SERVER_CONNECTION) rx_stats.nServerConns--;
    else rx_stats.nClientConns--;

    rxi_FreeConnection(conn);
    USERPRI;
}

/* Start a new rx remote procedure call, on the	specified connection.  If wait is set to 1, wait for a free call channel; otherwise return 0.  Maxtime gives the maximum number of seconds this call may take, after rx_MakeCall returns.  After this time interval, a call to any of rx_SendData, rx_ReadData, etc. will fail with RX_CALL_TIMEOUT. */
struct rx_call *rx_NewCall(conn)
    register struct rx_connection *conn;
{
    register int i;
    register struct rx_call *call;
    SPLVAR;

    clock_NewTime();

#ifdef RXDEBUG
    if (Log) fprintf(Log, "rx_MakeCall(conn %x)\n", conn);
#endif
    NETPRI;
    for (;;) {
	for (i=0; i<RX_MAXCALLS; i++) {
	    call = conn->call[i];
	    if (call) {
		if (call->state == RX_STATE_DALLY) {
		    rxi_ResetCall(call);
		    (*call->callNumber)++;
		    break;
		}
	    }
	    else {
		call = rxi_NewCall(conn, i);
		break;
	    }
	}
	if (i < RX_MAXCALLS) {
	    break;
	}
	conn->flags |= RX_CONN_MAKECALL_WAITING;
	osi_Sleep(conn);
    }

    /* Client is initially in send mode */
    call->state = RX_STATE_ACTIVE;
    call->mode = RX_MODE_SENDING;

    /* remember start time for call in case we have hard dead time limit */
    call->startTime = clock_Sec();

    /* Turn on busy protocol. */
    rxi_KeepAliveOn(call);
    USERPRI;
    return call;
}

/* Advertise a new service.  A service is named locally by a UDP port number plus a 16-bit service id.  Returns (struct rx_service *) 0 on a failure. */
struct rx_service *rx_NewService(port, serviceId, serviceName, securityObjects, nSecurityObjects, serviceProc)
    u_short port;
    u_short serviceId;
    char *serviceName;	/* Name for identification purposes (e.g. the service name might be used for probing for statistics) */
    struct rx_securityClass **securityObjects;
    int nSecurityObjects;
    long (*serviceProc)();
{    
    osi_socket socket = OSI_NULLSOCKET;
    register int i;
    SPLVAR;

    clock_NewTime();

    if (serviceId == 0) {
	(osi_Msg "rx_NewService:  service id for service %s is not non-zero.\n", serviceName);
	return 0;
    }

    if (port == 0) {
	if (rx_port == 0) {
	    (osi_Msg "rx_NewService: A non-zero port must be specified on this call if a non-zero port was not provided at Rx initialization (service %s).\n", serviceName);
	    return 0;
	}
	port = rx_port;
	socket = rx_socket;
    }

    NETPRI;
    for (i = 0; i<RX_MAX_SERVICES; i++) {
	register struct rx_service *service = rx_services[i];
	if (service) {
	    if (port == service->servicePort) {
		if (service->serviceId == serviceId) {
		    /* The identical service has already been installed; if the caller was intending to change the security classes used by this service, he/she loses. */
		    (osi_Msg "rx_NewService: tried to install service %s with service id %d, which is already in use for service %s\n", serviceName, service->serviceName, serviceId);
		    USERPRI;
		    return service;
		}
		/* Different service, same port:  re-use the socket which is bound to the same port */
	    }
	} else {
	    if (socket == OSI_NULLSOCKET) {
		/* If we don't already have a socket (from another service on same port) get a new one */
		socket = rxi_GetUDPSocket(port);
		if (socket == OSI_NULLSOCKET) {
		    USERPRI;
		    return 0;
		}
	    }
	    service = rxi_AllocService();
	    service->socket = socket;
	    service->servicePort = port;
	    service->serviceId = serviceId;
	    service->serviceName = serviceName;
	    service->nSecurityObjects = nSecurityObjects;
	    service->securityObjects = securityObjects;
	    service->minProcs = 0;
	    service->maxProcs = 1;
	    service->idleDeadTime = 60;
	    service->connDeadTime = rx_connDeadTime;
	    service->executeRequestProc = serviceProc;
	    rx_services[i] = service;	/* not visible until now */
	    USERPRI;
	    return service;
	}
    }
    USERPRI;
    (osi_Msg "rx_NewService: cannot support > %d services\n", RX_MAX_SERVICES);
    return 0;
}

/* This is the server process's request loop.  This routine should either be each server proc's entry point, or it should be called by the server process (depending upon the rx implementation environment). */
void rx_ServerProc()
{
    register struct rx_call *call;
    register long code;
    register struct rx_service *tservice;

    for (;;) {
	call = rx_GetCall();
	if (call == 0)
		return;
	tservice = call->conn->service;
	if (tservice->beforeProc) (*tservice->beforeProc)(call);
	code = call->conn->service->executeRequestProc(call);
	if (tservice->afterProc) (*tservice->afterProc)(call, code);
	rx_EndCall(call, code);
	rxi_nCalls++;
    }
}

/* allocate and free these entries from static space rather than on the stack
    because in the kernel, one can not address data on other processes' stacks.
    This function allocates an rx_serverQueueEntry from a free list, or from
    the memory allocator if the free list is depleted */
struct rx_serverQueueEntry *rxi_NewSQE() {
    register struct rx_serverQueueEntry *np;
    if (np = rx_FreeSQEList) {
	rx_FreeSQEList = *(struct rx_serverQueueEntry **)np;
	return np;
    }
    /* otherwise allocate a new one and return that */
    np = (struct rx_serverQueueEntry *) osi_Zalloc(rx_serverQueueEntry_zone);
    return np;
}

/* this function returns a server queue entry to the free list */
rxi_FreeSQE(ae)
register struct	rx_serverQueueEntry *ae; {
    *(struct rx_serverQueueEntry **)ae = rx_FreeSQEList;
    rx_FreeSQEList = ae;
}

/* Sleep until a call arrives.  Returns a pointer to the call, ready for an rx_Read. */
struct rx_call *rx_GetCall()
{
    struct rx_serverQueueEntry *sq;
    register struct rx_call *call = (struct rx_call *) 0;
    struct rx_service *service;
    extern long afs_termState;
    SPLVAR;

    NETPRI;
    if (queue_IsNotEmpty(&rx_incomingCallQueue)) {
	register struct rx_call *tcall, *ncall;
	/* Scan for eligible incoming calls.  A call is not eligible if the maximum number of calls for its service type are already executing */
	for (queue_Scan(&rx_incomingCallQueue, tcall, ncall, rx_call)) {
	    service = tcall->conn->service;
	    if (service->nRequestsRunning < service->maxProcs) {
		call = tcall;
		break;
	    }
	}
    }
    if (call) {
	queue_Remove(call);
	call->flags &= (~RX_CALL_WAIT_PROC);
	service->nRequestsRunning++;
    }
    else {
	/* If there are no eligible incoming calls, add this process to the idle server queue, to wait for one */
	sq = rxi_NewSQE();
	queue_Append(&rx_idleServerQueue, sq);
	osi_Sleep(sq);
	/* Check for termination */
	if (afs_termState == 210) {
		osi_Wakeup(&afs_termState);
		return 0;
	}
	/* When we're woken up, the call structure pointer is left in the queue element, and the queue element is removed from the idle queue */
	call = sq->newcall;
	rxi_FreeSQE(sq);
    }
    call->state = RX_STATE_ACTIVE;
    call->mode = RX_MODE_RECEIVING;
#ifdef RXDEBUG
    if (Log) {
	service = call->conn->service;
	fprintf(Log, "rx_GetCall(port=%d, service=%d) ==> call %x\n", 
service->servicePort, service->serviceId, call);
    }
#endif
    USERPRI;

    return call;
}

/* Establish a procedure to be called when a packet arrives for a call.  This routine will be called at most once after each call, and will also be called if there is an error condition on the or the call is complete.  Used by multi rx to build a selection function which determines which of several calls is likely to be a good one to read from.  NOTE:  the way this is currently implemented it is probably only a good idea to (1) use it immediately after a newcall (clients only) and (2) only use it once.  Other uses currently void your warranty */

void rx_SetArrivalProc(call, proc, handle, arg)
    register struct rx_call *call;
    register VOID (*proc)();
    register VOID *handle;
    register VOID *arg;
{
    call->arrivalProc = proc;
    call->arrivalProcHandle = handle;
    call->arrivalProcArg = arg;
}

/* Call is finished (possibly prematurely).  Return rc to the peer, if appropriate, and return the final error code from the conversation to the caller */
long rx_EndCall(call, rc)
    register struct rx_call *call;
    long rc;
{
    register struct rx_connection *conn = call->conn;
    long error;
    SPLVAR;

#ifdef RXDEBUG
    if (Log) fprintf(Log, "rx_EndCall(call %x)\n", call);
#endif
    NETPRI;
    call->arrivalProc = (VOID (*)()) 0;
    if (rc && call->error == 0) {
	rxi_CallError(call, rc);
	/* Send an abort message to the peer if this error code has only just been set.  If it was set previously, assume the peer has already been sent the error code or will request it */
	rxi_SendCallAbort(call, (struct rx_packet *) 0);
    }
    if (conn->type == RX_SERVER_CONNECTION) {
	/* Make sure reply or at least dummy reply is sent */
	if (call->mode == RX_MODE_RECEIVING) rx_WriteProc(call, 0, 0);
	if (call->mode == RX_MODE_SENDING) rx_FlushWrite(call);
	conn->service->nRequestsRunning--;
    }
    else { /* Client connection */
	char dummy;
	/* Make sure server receives input packets, in the case where no reply arguments are expected */
	if (call->mode == RX_MODE_SENDING) (void) rx_ReadProc(call, &dummy, 1);
	if (conn->flags & RX_CONN_MAKECALL_WAITING) {
	    conn->flags &= (~RX_CONN_MAKECALL_WAITING);
	    osi_Wakeup(conn);/* Wakeup anyone waiting to establish a new connection */
	}
    }
    call->state = RX_STATE_DALLY;
    error = call->error;
    /* currentPacket, nLeft, and NFree must be zeroed here, because ResetCall cannot:  ResetCall may be called at splnet(), in the kernel version, and may interrupt the macros rx_Read or rx_Write, which run at normal priority for efficiency. */
    if (call->currentPacket) {
	rxi_FreePacket(call->currentPacket);
	call->currentPacket = (struct rx_packet *) 0;
	call->nLeft = call->nFree = 0;
    }
    USERPRI;
    return error;
}

/* Call this routine when shutting down a server or client (especially clients).  This will allow Rx to gracefully garbage collect server connections, and reduce the number of retries that a server might make to a dead client */
void rx_Finalize() {
    register struct rx_connection **conn_ptr, **conn_end;
    SPLVAR;
    NETPRI;
    if (rx_connHashTable)
	for (conn_ptr = &rx_connHashTable[0], conn_end = &rx_connHashTable[rx_hashTableSize]; conn_ptr < conn_end; conn_ptr++) {
	    struct rx_connection *conn, *next;
	    for (conn = *conn_ptr; conn; conn = next) {
		next = conn->next;
		if (conn->type == RX_CLIENT_CONNECTION) rx_DestroyConnection(conn);
	    }
	}
    USERPRI;
}

/* In the two packet freeing routines, below, the assumption is that we want all of the packets to be used equally frequently, so that we don't get packet buffers paging out.  It would be just as valid to assume that we DO want them to page out if not many are being used.  In any event, we assume the former, and append the packets to the end of the free list */

/* if we wakeup packet waiter too often, can get in loop with two AllocSendPackets
    each waking each other up (from ReclaimPacket calls) */
static CheckSendPackets() {
    if (!rx_waitingForPackets) return;
    if (rxi_OverQuota(RX_PACKET_CLASS_SEND)) return;	/* still over quota */
    rx_waitingForPackets = 0;
    osi_Wakeup(&rx_waitingForPackets);
    return;
}

/* Free the packet p.  P is assumed not to be on any queue, i.e. remove it yourself first if you call this routine. */
rxi_FreePacket(p)
    register struct rx_packet *p;
{
    SPLVAR;
#ifdef RXDEBUG
      if (Log)fprintf(Log, "Free %x\n", p);
#endif
    rx_nFreePackets++;
    NETPRI;
    osi_Zfree(rx_packet_zone, p);
    /* Wakeup anyone waiting for packets */
    CheckSendPackets();
    USERPRI;
}

int rx_WriteProc(call, buf, nbytes)
    register struct rx_call *call;
    register char *buf;
    register int nbytes;
{
    int requestCount = nbytes;
    SPLVAR;
    NETPRI;
    if (call->mode != RX_MODE_SENDING) {
	if (call->conn->type == RX_SERVER_CONNECTION && call->mode == RX_MODE_RECEIVING) {
	    call->mode = RX_MODE_SENDING;
	    if (call->currentPacket) {
		rxi_FreePacket(call->currentPacket);
		call->currentPacket = (struct rx_packet *) 0;
		call->nLeft = 0;
	    }
	}
	else {
	    USERPRI; 
	    return 0;
	}
    }

    /* Loop condition is checked at end, so that a write of 0 bytes will force a packet to be created--specially for the case where there are 0 bytes on the stream, but we must send a packet anyway. */
    do {
	if (call->nFree == 0) {
	    struct rx_connection *conn = call->conn;
	    if (!call->error && call->currentPacket) {
		struct rx_packet *cp = call->currentPacket;
		clock_NewTime(); /* Bogus:  need new time package */
		cp->length = rx_MaxUserDataSize(conn);
		/* The 0, below, specifies that it is not the last packet:  there will be others */
		/* The following routine may alter the packet length by up to conn->securityMaxTrailerSize */
		rxi_PrepareSendPacket(call, cp, 0);
		queue_Append(&call->tq, cp);
		rxi_Start(0, call);
	    }
	    /* Wait for transmit window to open up */
	    while (!call->error && call->tnext + 1 > call->tfirst + call->twind) {
		clock_NewTime();
		call->startWait = clock_Sec();
		call->flags |= RX_CALL_WAIT_WINDOW_ALLOC;
		osi_Sleep(&call->twind);
		call->startWait = 0;
	    }
	    if (call->currentPacket = rxi_AllocSendPacket(call)) {
		call->nFree = rx_MaxUserDataSize(conn);
		call->bufPtr = rx_UserDataOf(conn, call->currentPacket);
	    }
	    if (call->error) {
		USERPRI;
		return 0;
	    }
	}
	/* If the remaining bytes fit in the buffer, then store them and return.  Don't ship a buffer that's full immediately to the peer--we don't know if it's the last buffer yet */
	if (nbytes <= call->nFree) {
	    bcopy(buf, call->bufPtr, nbytes);
	    call->nFree -= nbytes;
	    call->bufPtr += nbytes;
	    USERPRI;
	    return requestCount;
	}
	bcopy(buf, call->bufPtr, call->nFree);
	buf += call->nFree;
	nbytes -= call->nFree;
	call->nFree = 0;
    } while(nbytes);
    USERPRI;
    return requestCount - nbytes;
}

/* Flush any buffered data to the stream, switch to read mode (clients) or to EOF mode (servers) */
void rx_FlushWrite(call)
    register struct rx_call *call;
{
    SPLVAR;
    NETPRI;
    if (call->mode == RX_MODE_SENDING) {
	register struct rx_packet *cp;
	register struct rx_connection *conn = call->conn;
	call->mode = (conn->type == RX_CLIENT_CONNECTION? RX_MODE_RECEIVING: RX_MODE_EOF);
	if (call->currentPacket) {
	    cp = call->currentPacket;
	    call->currentPacket = (struct rx_packet *) 0;
	    cp->length = rx_MaxUserDataSize(conn) - call->nFree;
	}
	else {
	    cp = rxi_AllocSendPacket(call);
	    if (!cp) {
		/* Mode can no longer be MODE_SENDING */
		USERPRI;
		return;
	    }
	    cp->length = 0;
	}
	call->nFree = 0;
	/* The 1 specifies that this is the last packet */
	rxi_PrepareSendPacket(call, cp, 1);
	queue_Append(&call->tq, cp);
	rxi_Start(0, call);
    }
    USERPRI;
}

int rx_ReadProc(call, buf, nbytes)
    register struct rx_call *call;
    register char *buf;
    register int nbytes;
{
    register struct rx_packet *rp; 
    register int requestCount;
    SPLVAR;
/* XXXX took out clock_NewTime from here.  Was it needed? */
    requestCount = nbytes;
    NETPRI;

    while (nbytes) {
	if (call->nLeft == 0) {
	    /* Get next packet */
	    for (;;) {
		if (call->error || call->mode != RX_MODE_RECEIVING) {
		    if (call->error) {
			USERPRI;
			return 0;
		    }
		    if (call->mode == RX_MODE_SENDING) {
			rx_FlushWrite(call);
			continue;
		    }
		}
		if (queue_IsNotEmpty(&call->rq)) {
		    /* Check that next packet available is next in sequence */
		    rp = queue_First(&call->rq, rx_packet);
		    if (rp->header.seq == call->rnext) {
			long error;
			register struct rx_connection *conn = call->conn;
			queue_Remove(rp);
			/* RXS_CheckPacket called to undo RXS_PreparePacket's work.  It may reduce the length of the packet by up to conn->maxTrailerSize, to reflect the length of the data + the header. */
			if (error = RXS_CheckPacket(conn->securityObject, call, rp)) {
			    rxi_CallError(call, error);
			    rp = rxi_SendCallAbort(call, rp);
			    rxi_FreePacket(rp);
			    USERPRI;
			    return 0;
			}
			call->rnext++;
			call->currentPacket = rp;
			/* Notice that this code works correctly if the data size is 0 (which it may be--no reply arguments from server, for example).  This relies heavily on the fact that the code below immediately frees the packet (no yields, etc.).  If it didn't, this would be a problem because a value of zero for call->nLeft normally means that there is no read packet */
			call->nLeft = rp->length;
			call->bufPtr = rx_UserDataOf(conn, rp);
			if (rp->header.flags&RX_LAST_PACKET)
			    call->flags |= RX_CALL_RECEIVE_DONE;
			
			/* now, if we haven't send a hard ack for window/2 packets
			    we spontaneously generate one, to take care of the case
			    where (most likely in the kernel) we receive a window-full
			    of packets, and ack all of them before any are read
			    by the user, thus not hard-acking any of them.  The sender
			    retransmits in this case only under a timer, which is a
			    real loser */
			if (call->rnext > call->lastAcked + (rx_Window>>1)) {
			    /* too many packets have been read sans ack, send one now */
			    rxi_SendAck(call, 0, 0, 0, 0, RX_ACK_DELAY);
			}
			break;
		    }
		}
		/* Are there ever going to be any more packets? */
		if (call->flags & RX_CALL_RECEIVE_DONE) {
		    USERPRI;
		    return requestCount - nbytes;
		}
		/* Wait for in-sequence packet */
		call->flags |= RX_CALL_READER_WAIT;
		clock_NewTime();
		call->startWait = clock_Sec();
		osi_Sleep(&call->rq);
		call->startWait = 0;
	    }
	}
	if (nbytes < call->nLeft) {
	    call->nLeft -= nbytes;
	    bcopy(call->bufPtr, buf, nbytes);
	    call->bufPtr += nbytes;
	    USERPRI;
	    return requestCount;
	}
	bcopy(call->bufPtr, buf, call->nLeft);
	buf += call->nLeft;
	nbytes -= call->nLeft;
	rxi_FreePacket(call->currentPacket);
	call->currentPacket = (struct rx_packet *)0;
	call->nLeft = 0;
    }
    USERPRI;
    return requestCount;
}



/* -------------------------------------Internal interfaces-------------------------------------------- */

/* Return this process's service structure for the specified socket and service */
struct rx_service *rxi_FindService(socket, serviceId)
    register osi_socket socket;
    register u_short serviceId;
{
    register struct rx_service **sp;	
    for (sp = &rx_services[0]; *sp; sp++) {
	if ((*sp)->serviceId == serviceId && (*sp)->socket == socket) return *sp;
    }
    return 0;
}

struct rx_packet *rxi_AllocPacket(class)
    int class;
{
    register struct rx_packet *p;

    if (rxi_OverQuota(class)) {
	/* if we won't get any packets, we should try to see if all the good
	    packets are in conns waiting for lwps; if so we reclaim them now */
	if (class != RX_PACKET_CLASS_SPECIAL) {
	    /* don't reclaim for special packets, since they're always enough */
	    rxi_ReclaimPackets();
	}
	if (rxi_OverQuota(class)) { /* still over quota, we fail */
	    rx_stats.noPackets[class]++;
	    return (struct rx_packet *) 0;
	}
    }
    rx_stats.packetRequests++;
    rx_nFreePackets--;
    return (struct rx_packet *)osi_Zalloc(rx_packet_zone);
}

struct rx_packet *rxi_AllocSendPacket(call)
register struct rx_call *call;
{
    register struct rx_packet *p = (struct rx_packet *) 0;
    SPLVAR;

    for(;;) {
	/* if an error occurred, or we get the packet we want, we're done */
	if (call->error || (p = rxi_AllocPacket(RX_PACKET_CLASS_SEND))) break;
	/* no error occurred, and we didn't get a packet, so we sleep.  At this point,
	    we assume that packets will be returned sooner or later, as packets are
	    acknowledged, and so we just wait.  There's a little race here if a burst
	    of packets arrives for a connection after doing the ReclaimPackets */
	NETPRI;
    	rx_waitingForPackets = 1;
	call->flags |= RX_CALL_WAIT_PACKETS;
	osi_Sleep(&rx_waitingForPackets);
	call->flags &= ~RX_CALL_WAIT_PACKETS;
	USERPRI;
    }
    return p;
}

/* Allocate a call structure, for the indicated channel of the supplied connection.  The mode and state of the call must be set by the caller. */
struct rx_call *rxi_NewCall(conn, channel)
    register struct rx_connection *conn;
    register int channel;
{
    register struct rx_call *call;
    /* Grab an existing call structure, or allocate a new one.  Existing call structures are assumed to have been left reset by rxi_FreeCall */
    if (queue_IsNotEmpty(&rx_freeCallQueue)) {
	call = queue_First(&rx_freeCallQueue, rx_call);
	queue_Remove(call);
	rx_stats.nFreeCallStructs--;
    }
    else {
	call = (struct rx_call *) rxi_Zalloc(rx_call_zone);
	rx_stats.nCallStructs++;
	/* Initialize once-only items */
	queue_Init(&call->tq);
	queue_Init(&call->rq);
	rxi_ResetCall(call);
    }
    /* Bind the call to it's connection structure */
    call->conn = conn;
    call->channel = channel;
    call->callNumber = &conn->callNumber[channel];
    /* Note that the next expected call number is retained (in conn->callNumber[i]), even if we reallocate the call structure */
    conn->call[channel] = call;
    /* if the channel's never been used (== 0), we should start at 1, otherwise
	the call number is valid from the last time this channel was used */
    if (*call->callNumber == 0) *call->callNumber = 1;
    return call;
}

/* A call has been inactive long enough that so we can throw away state, including the call structure, which is placed on the call free list. */
void rxi_FreeCall(call)
    register struct rx_call *call;
{
    register channel = call->channel;
    register struct rx_connection *conn = call->conn;
    if (call->state == RX_STATE_DALLY) (*call->callNumber)++;
    rxi_ResetCall(call);
    call->conn->call[channel] = (struct rx_call *) 0;
    queue_Append(&rx_freeCallQueue, call);
    rx_stats.nFreeCallStructs++;
    /* Destroy the connection if it was previously slated for destruction, i.e. the Rx client code previously called rx_DestroyConnection (client connections), or rxi_ReapConnections called the same routine (server connections).  Only do this, however, if there are no outstanding calls */
    if (conn->flags & RX_CONN_DESTROY_ME) rx_DestroyConnection(conn);
}

char *rxi_Alloc(size)
register int size;
{
    register char *p;
    p = (char *) osi_Alloc(size);
    if (!p) osi_Panic("rxi_Alloc error");
    bzero(p, size);
    return p;
}

char *rxi_Zalloc(zone)
osi_zone_t zone;
{
    register char *p;
    p = (char *) osi_Zalloc(zone);
    if (!p) osi_Panic("rxi_Zalloc error");
    bzero(p, osi_Zsize(zone));
    return p;
}

/* Find the peer process represented by the supplied (host,port) combination.  If there is no appropriate active peer structure, a new one will be allocated and initialized */
struct rx_peer *rxi_FindPeer(host, port)
    register u_long host;
    register u_short port;
{
    register struct rx_peer *pp;
    int hashIndex;
    hashIndex = PEER_HASH(host, port);
    for (pp = rx_peerHashTable[hashIndex]; pp; pp = pp->next) {
	if (pp->host == host && pp->port == port) break;
    }
    if (!pp) {
	pp = rxi_AllocPeer(); /* This bzero's *pp */
	pp->host = host;
	pp->port = port;
	/*pp->refCount = 0;*/
	/* XXXX Initialize various congestion control parameters */
	/* pp->burstSize = 0; */
	/* pp->burst = 0; */
	/* pp->burstWait.sec = 0; */
	/* pp->burstWait.usec = 0; */
	pp->timeout.sec = 2;
	/* pp->timeout.usec = 0; */
	queue_Init(&pp->congestionQueue);
	pp->next = rx_peerHashTable[hashIndex];
	rx_peerHashTable[hashIndex] = pp;
	pp->packetSize = RX_LOCAL_PACKET_SIZE;	/* XXXXX */
	/* pp->nSent = 0; */
	/* pp->reSends = 0; */
	/* pp->rtt = 0; */
	rx_stats.nPeerStructs++;
    }
    pp->refCount++;
    return pp;
}

/* Destroy the specified peer structure, removing it from the peer hash table */
void rxi_DestroyPeer(peer)
    register struct rx_peer *peer;
{
    register struct rx_peer **peer_ptr;
    for (peer_ptr = &rx_peerHashTable[PEER_HASH(peer->host, peer->port)]; *peer_ptr; peer_ptr = &(*peer_ptr)->next) {
	if (*peer_ptr == peer) {
	    *peer_ptr = peer->next;
	    rxi_FreePeer(peer);
	    rx_stats.nPeerStructs--;
	    return;
	}
    }
}

/* Find the connection at (host, port) started at epoch, and with the given connection id.  Creates the server connection if necessary.  The type specifies whether a client connection or a server connection is desired.   In both cases, (host, port) specify the peer's (host, pair) pair.  Client connections are not made automatically by this routine.  The parameter socket gives the socket descriptor on which the packet was received.  This is used, in the case of server connections, to check that *new* connections come via a valid (port, serviceId).  Finally, the securityIndex parameter must match the existing index for the connection.  If a server connection is created, it will be created using the supplied index, if the index is valid for this service */
struct rx_connection *rxi_FindConnection(socket, host, port, serviceId, cid, epoch, type, securityIndex)
    osi_socket socket;
    register long host;
    register u_short port;
    u_short serviceId;
    u_long cid;
    u_long epoch;
    int type;
    u_int securityIndex;
{
    int hashindex;
    register struct rx_connection *conn;
    hashindex = CONN_HASH(host, port, cid, epoch, type);
    for (conn = rx_connHashTable[hashindex]; conn; conn = conn->next) {
	if (conn->type == type && (cid&RX_CIDMASK) == conn->cid && epoch == conn->epoch && securityIndex == conn->securityIndex) {
	    register struct rx_peer *pp = conn->peer;
	    if (pp->host == host && pp->port == port) break;
	}
    }
    if (!conn) {
	struct rx_service *service;
	if (type == RX_CLIENT_CONNECTION) return (struct rx_connection *) 0; 
	service = rxi_FindService(socket, serviceId);
	if (!service || securityIndex >= service->nSecurityObjects || service->securityObjects[securityIndex] == 0)
	    return (struct rx_connection *) 0;
	conn = rxi_AllocConnection(); /* This bzero's the connection */
	conn->next = rx_connHashTable[hashindex];
	rx_connHashTable[hashindex] = conn;
	conn->peer = rxi_FindPeer(host, port);
	conn->type = RX_SERVER_CONNECTION;
	conn->lastSendTime = clock_Sec();   /* don't GC immediately */
	conn->epoch = epoch;
	conn->cid = cid & RX_CIDMASK;
	/* conn->serial = conn->lastSerial = 0; */
	/* conn->rock = 0; */
	/* conn->timeout = 0; */
	/* conn->refCount = 0; */
	conn->service = service;
	conn->serviceId = serviceId;
	conn->securityIndex = securityIndex;
	conn->securityObject = service->securityObjects[securityIndex];
	rx_SetConnDeadTime(conn, service->connDeadTime);
	/* Notify security object of the new connection */
	RXS_NewConnection(conn->securityObject, conn);
	/* XXXX Connection timeout? */
	if (service->newConnProc) (*service->newConnProc)(conn);
	rx_stats.nServerConns++;
    }
    return conn;
}

/* send a response to a debug packet */
struct rx_packet *rxi_ReceiveDebugPacket(ap, asocket, ahost, aport)
osi_socket asocket;
long ahost;
short aport;
register struct rx_packet *ap; {
    register char *tp = (char *) ap->wire.data;
    struct rx_debugIn tin;
    struct rx_debugStats tstat;

    bcopy(tp, &tin, sizeof(struct rx_debugIn));
    tin.type = ntohl(tin.type);
    tin.index = ntohl(tin.index);
    switch(tin.type) {
	case RX_DEBUGI_GETSTATS:
	    /* get basic stats */
	    tstat.waitingForPackets = rx_waitingForPackets;
	    tstat.nFreePackets = htonl(rx_nFreePackets);
	    tstat.callsExecuted = htonl(rxi_nCalls);
	    tstat.packetReclaims = htonl(rxi_packetReclaims);
	    bcopy(&tstat, tp, sizeof(tstat));
	    ap->length = sizeof(tstat);
	    rxi_SendDebugPacket(ap, asocket, ahost, aport);
	    break;

	case RX_DEBUGI_GETCONN: {
	    int i, j;
	    register struct rx_connection *tc;
	    struct rx_call *tcall;
	    struct rx_debugConn tconn;
	    /* get N'th "interesting" connection info */
	    for(i=0;i<rx_hashTableSize;i++) {
		for(tc=rx_connHashTable[i]; tc; tc=tc->next) {
		    if (interesting(tc) && tin.index-- <= 0) {
			tconn.host = tc->peer->host;
			tconn.port = tc->peer->port;
			tconn.cid = htonl(tc->cid);
			tconn.serial = htonl(tc->serial);
			for(j=0;j<RX_MAXCALLS;j++) {
			    tconn.callNumber[j] = tc->callNumber[j];
			    if (tcall=tc->call[j]) {
				tconn.callState[j] = tcall->state;
				tconn.callMode[j] = tcall->mode;
				tconn.callFlags[j] = tcall->flags;
				if (queue_IsNotEmpty(&tcall->rq))
				    tconn.callOther[j] |= RX_OTHER_IN;
				if (queue_IsNotEmpty(&tcall->tq))
				    tconn.callOther[j] |= RX_OTHER_OUT;
			    }
			    else tconn.callState[j] = RX_STATE_NOTINIT;
			}
			tconn.error = htonl(tc->error);
			tconn.flags = tc->flags;
			tconn.type = tc->type;
			tconn.securityIndex = tc->securityIndex;
			bcopy(&tconn, tp, sizeof(struct rx_debugConn));
			ap->length = sizeof(struct rx_debugConn);
			rxi_SendDebugPacket(ap, asocket, ahost, aport);
			return ap;
		    }
		}
	    }
	    /* if we make it here, there are no interesting packets */
	    tconn.cid = htonl(0xffffffff); /* means end */
	    bcopy(&tconn, tp, sizeof(struct rx_debugConn));
	    ap->length = sizeof(struct rx_debugConn);
	    rxi_SendDebugPacket(ap, asocket, ahost, aport);
	    break;
	    }

	default:
	    break;
    }
    return ap;
}

/* A packet has been received off the interface.  Np is the packet, socket is the socket number it was received from (useful in determining which service this packet corresponds to), and (host, port) reflect the host,port of the sender.  This call returns the packet to the caller if it is finished with it, rather than de-allocating it, just as a small performance hack */
struct rx_packet *rxi_ReceivePacket(np, socket, host, port)
    register struct rx_packet *np;
    osi_socket socket;
    u_long host;
    u_short port;
{
    register struct rx_call *call;
    register struct rx_connection *conn;
    register struct rx_peer *peer;
    int channel;
    unsigned long currentCallNumber;
    int type;
    int skew;
#ifdef RXDEBUG
    char *packetType;
#endif

#ifdef RXDEBUG
/* We don't print out the packet until now because (1) the time may not be accurate enough until now in the lwp implementation (rx_Listener only gets the time after the packet is read) and (2)  from a protocol point of view, this is the first time the packet has been seen */
    packetType = (np->header.type > 0 && np->header.type < RX_N_PACKET_TYPES)? rx_packetTypes[np->header.type-1]: "*UNKNOWN*";
    dpf(("R %d %s: %x.%d.%d.%d.%d.%d.%d flags %d, packet %x", np->header.serial, packetType, host, port, np->header.serviceId, np->header.epoch, np->header.cid, np->header.callNumber, np->header.seq, np->header.flags, np));
#endif

    /* If the packet was not sent by the client, then *we* must be the client */
    type = ((np->header.flags&RX_CLIENT_INITIATED) != RX_CLIENT_INITIATED)?RX_CLIENT_CONNECTION:RX_SERVER_CONNECTION;

    if (np->header.type == RX_PACKET_TYPE_DEBUG)
	return rxi_ReceiveDebugPacket(np, socket, host, port);

    /* Find the connection (or fabricate one, if we're the server & if necessary) associated with this packet */
    conn = rxi_FindConnection(socket, host, port, np->header.serviceId, np->header.cid, np->header.epoch, type, np->header.securityIndex);
    if (!conn) return np;   /* If no connection found or fabricated, just ignore the packet.  (An argument could be made for sending an abort packet for the connection) */

    /* compute the max serial number seen on this connection */
    if (conn->maxSerial < np->header.serial) conn->maxSerial = np->header.serial;

    /* If the connection is in an error state, send an abort packet and ignore the incoming packet */
    if (conn->error) {
	/* Don't respond to an abort packet--we don't want loops! */
	if (np->header.type != RX_PACKET_TYPE_ABORT)
	    np = rxi_SendConnectionAbort(conn, np);
	return np;
    }

    peer = conn->peer;

    /* Check for connection-only requests (i.e. not call specific). */
    if (np->header.callNumber == 0) {
	switch (np->header.type) {
	    case RX_PACKET_TYPE_ABORT:
		/* What if the supplied error is zero? */
		rxi_ConnectionError(conn, ntohl(*(long *)rx_DataOf(np)));
		return np;
	    case RX_PACKET_TYPE_CHALLENGE:
		return rxi_ReceiveChallengePacket(conn, np);
	    case RX_PACKET_TYPE_RESPONSE:
		return rxi_ReceiveResponsePacket(conn, np);
	    default:
		/* Should not reach here, unless the peer is broken:  send an abort packet */
		rxi_ConnectionError(conn, RX_PROTOCOL_ERROR);
		return rxi_SendConnectionAbort(conn, np);
	}
    }

    channel = np->header.cid & RX_CHANNELMASK;
    call = conn->call[channel];
    currentCallNumber = conn->callNumber[channel];
    if (type == RX_SERVER_CONNECTION) { /* We're the server */
	if (np->header.callNumber < currentCallNumber) {
	    rx_stats.spuriousPacketsRead++;
	    return np;
	}
	if (!call) {
	    call = rxi_NewCall(conn, channel);
	    *call->callNumber = np->header.callNumber;
	    call->state = RX_STATE_PRECALL;
	    rxi_KeepAliveOn(call);
	}
	else if (np->header.callNumber != currentCallNumber) {
	    /* If the new call cannot be taken right now send a busy and set the error condition in this call, so that it terminates as quickly as possible */
	    if (call->state == RX_STATE_ACTIVE) {
		rxi_CallError(call, RX_CALL_DEAD);
		return rxi_SendSpecial(call, conn, np, RX_PACKET_TYPE_BUSY, (char *) 0, 0);
	    }
	    /* If the new call can be taken right now (it's not busy) then accept it. */
	    rxi_ResetCall(call);
	    *call->callNumber = np->header.callNumber;
	    call->state = RX_STATE_PRECALL;
	    rxi_KeepAliveOn(call);
	}
	else {
	    /* Continuing call; do nothing here. */
	}
    } else { /* we're the client */
	/* Ignore anything that's not relevant to the current call.   If there isn't a current call, then no packet is relevant. */
	if (!call || np->header.callNumber != currentCallNumber) {
	    rx_stats.spuriousPacketsRead++;
	    return np;	
	}
	/* If the service security object index stamped in the packet does not match the connection's security index, ignore the packet */
	if (np->header.securityIndex != conn->securityIndex) return np;

	/* now check to see if this is an ack packet acknowledging that the server actually *lost* some hard-acked data.  If this happens we ignore this packet, as it may indicate that the server restarted in the middle of a call.  It is also possible that this is an old ack packet.  We don't abort the connection in this case, because this *might* just be an old ack packet.   The right way to detect a server restart in the midst of a call is to notice that the server epoch changed, btw.  */
	if (np->header.type == RX_PACKET_TYPE_ACK) {
	    struct rx_ackPacket *ap = (struct rx_ackPacket *) np->wire.data;
	    if (ntohl(ap->firstPacket) < call->tfirst) {
		rx_stats.spuriousPacketsRead++;
		return np;
	    }
	}

	/* If we're receiving the response, then all transmit packets are implicitly acknowledged.  Get rid of them. */
	if (np->header.type == RX_PACKET_TYPE_DATA) rxi_ClearTransmitQueue(call);
    }

    /* Set remote user defined status from packet */
    call->remoteStatus = np->header.userStatus;

    /* Note the maximum gap between the expected next packet and the actual packet that arrived, when the new packet has a smaller serial number than expected. */
    skew = conn->lastSerial - np->header.serial;
    conn->lastSerial = np->header.serial;
    /* Skew is checked against 0 here to avoid any dependence on the type of inPacketSkew (which may be unsigned).  In C, -1 > (unsigned) 0 is always true! */
    if (skew > 0 && skew > peer->inPacketSkew) {
#ifdef RXDEBUG	
	if (Log) fprintf(Log, "*** In skew changed from %d to %d\n", peer->inPacketSkew, skew);
#endif
	peer->inPacketSkew = skew;
    }

    /* Now do packet type-specific processing */
    switch (np->header.type) {
	case RX_PACKET_TYPE_DATA:
	    np = rxi_ReceiveDataPacket(call, np);
	    break;
	case RX_PACKET_TYPE_ACK:
	    /* Respond immediately to ack packets requesting acknowledgement (ping packets) */
	    if (np->header.flags & RX_REQUEST_ACK) {
		if (call->error) (void) rxi_SendCallAbort(call, 0);
		else (void) rxi_SendAck(call, 0, 0, 0, 0, RX_ACK_PING_RESPONSE);
	    }
	    np = rxi_ReceiveAckPacket(call, np);
	    break;
	case RX_PACKET_TYPE_ABORT:
	    /* An abort packet:  reset the connection, passing the error up to the user */
	    /* What if error is zero? */
	    rxi_CallError(call, ntohl(*(long *)rx_DataOf(np)));
	    break;
	case RX_PACKET_TYPE_BUSY:
	    /* XXXX */
	    break;
	case RX_PACKET_TYPE_ACKALL:
	    /* All packets acknowledged, so we can drop all packets previously readied for sending */
	    rxi_ClearTransmitQueue(call);
	    break;
	default:
	    /* Should not reach here, unless the peer is broken:  send an abort packet */
	    rxi_CallError(call, RX_PROTOCOL_ERROR);
	    np = rxi_SendCallAbort(call, np);
	    break;
    };
    /* Note when this last legitimate packet was received, for keep-alive processing.  Note, we delay getting the time until now in the hope that the packet will be delivered to the user before any get time is required (if not, then the time won't actually be re-evaluated here). */
    call->lastReceiveTime = clock_Sec();
    return np;
}

/* reclaim held incoming packets from calls that aren't running yet.
 * Note that this doesn't get all packets, just all that aren't allocated to an lwp yet.
 * But, this means you can allocate WINDOWSIZE packets * LWPS + <YOUR GUESS HERE>
 * and probably avoid losing.
 */
rxi_ReclaimPackets() {
    struct rx_call *victim, *nextvictim;
    SPLVAR;

    NETPRI;
    rxi_packetReclaims++;
    for (queue_ScanBackwards(&rx_incomingCallQueue, victim, nextvictim, rx_call)) {
	if (queue_IsNotEmpty(&victim->rq)) {
	    rxi_ClearReceiveQueue(victim);
	    if (rx_nFreePackets >= rxi_minReclaim) break;
	}
    }
    /* give output side a chance to get its share */
    CheckSendPackets();
    USERPRI;
}

/* send a debug packet back to the sender */
rxi_SendDebugPacket(apacket, asocket, ahost, aport)
struct rx_packet *apacket;
osi_socket asocket;
long ahost;
short aport; {
    struct sockaddr_in taddr;
    
    taddr.sin_family = AF_INET;
    taddr.sin_port = aport;
    taddr.sin_addr.s_addr = ahost;

    osi_NetSend(asocket, &taddr, &apacket->wire, apacket->length+RX_HEADER_SIZE);
}

/* return true if this is an "interesting" connection from the point of view
    of someone trying to debug the system */
static interesting(aconn)
register struct rx_connection *aconn; {
    register int i;
    register struct rx_call *tcall;

    if (aconn->flags) return 1;	/* make call waiting or destroy me is set */
    for(i=0;i<RX_MAXCALLS;i++) {
	tcall = aconn->call[i];
	if (tcall) {
	    if (tcall->state == RX_STATE_PRECALL || tcall->state == RX_STATE_ACTIVE)
		return 1;
	    if (tcall->mode == RX_MODE_SENDING || tcall->mode == RX_MODE_RECEIVING)
		return 1;
	}
    }
    return 0;
}

/* A data packet has been received off the interface.  This packet is appropriate to the call (the call is in the right state, etc.).  This routine can return a packet to the caller, for re-use */
struct rx_packet *rxi_ReceiveDataPacket(call, np)
    register struct rx_call *call;
    register struct rx_packet *np;
{
    register struct rx_connection *conn = call->conn;
    u_long seq, serial, flags;
    seq = np->header.seq;
    rx_stats.dataPacketsRead++;

    /* If the call is in an error state, send an abort message */
    /* XXXX this will send too many aborts for multi-packet calls */
    if (call->error) return rxi_SendCallAbort(call, np);

    /* If there are no packet buffers, drop this new packet, unless we can find packet buffers from inactive calls */
    if (rxi_OverQuota(RX_PACKET_CLASS_RECEIVE)) {
	/* HACK for this version of rx.  If the call is poised to read the next packet, and this is it, then reclaim packets from some call which is waiting for a process. */
	if (np->header.seq == call->rnext && (call->flags & RX_CALL_READER_WAIT)) {
	    rxi_ReclaimPackets();
	    if (rxi_OverQuota(RX_PACKET_CLASS_RECEIVE)) {
		/* Still didn't make it?  Then clear our own queue.  We should actually go through all receive queues, but that's too hard, for now */
		rxi_ClearReceiveQueue(call);
	    }
	}
	/* If we're still over quota, drop the packet */
	if (rxi_OverQuota(RX_PACKET_CLASS_RECEIVE)) {
	    rx_stats.noPacketBuffersOnRead++;
	    /* np = rxi_SendAck(call, np, np->header.seq, np->header.serial, np->header.flags, RX_ACK_NOSPACE);*/
	    call->rprev = seq;
	    return np;
	}
    }

    /* The usual case is that this is the expected next packet */
    if (np->header.seq == call->rnext) {

	/* Check to make sure it is not a duplicate of one already queued */
	if (queue_IsNotEmpty(&call->rq) && queue_First(&call->rq, rx_packet)->header.seq == seq) {
	    rx_stats.dupPacketsRead++;
	    np = rxi_SendAck(call, np, seq, np->header.serial, np->header.flags, RX_ACK_DUPLICATE);
	    call->rprev = seq;
	    return np;
	}

	/* It's the next packet.  Stick it on the receive queue for this call */
	queue_Prepend(&call->rq, np);

	/* Save info for possible acknowledge, later, when we don't actually have the packet */
	serial = np->header.serial;
	flags = np->header.flags;
	np = 0;	/* We can't use this any more */

	/* Provide asynchronous notification for those who want it (e.g. multi rx) */
	if (call->arrivalProc) {
	    (*call->arrivalProc)(call, call->arrivalProcHandle, call->arrivalProcArg);
	    call->arrivalProc = (VOID (*)()) 0;
	}

	/* Wakeup the reader, if any */
	if (call->flags & RX_CALL_READER_WAIT) {
	    call->flags &= ~RX_CALL_READER_WAIT;
	    osi_Wakeup(&call->rq);
	}

	if (flags&RX_REQUEST_ACK) {
	    /* Acknowledge, if requested */
	    np = rxi_SendAck(call, 0, seq, serial, flags, RX_ACK_REQUESTED);
	}
	else {
	    struct clock when;
	    /* Or schedule an acknowledgement later on. */
	    rxevent_Cancel(call->delayedAckEvent);
	    clock_GetTime(&when);
	    clock_Add(&when, &rx_lastAckDelay);
	    call->delayedAckEvent = rxevent_Post(&when, rxi_SendDelayedAck, call, 0);
	}

	/* Update last packet received */
	call->rprev = seq;

	/* If there is no server process serving this call, grab one, if available */
	if (conn->type == RX_SERVER_CONNECTION && call->state == RX_STATE_PRECALL) {
	    /* Don't attach until we have authentication information, if required. */
	    if (RXS_CheckAuthentication(call->conn->securityObject, conn) == 0) {
		rxi_AttachServerProc(call);
		/* Note:  this does not necessarily succeed; there may not any proc available */
	    }
	    else {
		rxi_ChallengeOn(call->conn);
	    }
	}
    }	
    /* This is not the expected next packet */
    else {
	/* Determine whether this is a new or old packet, and if it's a new one whether it fits into the current receive window.  It's also useful to know if the packet arrived out of sequence, so that we can force an acknowledgement in that case.   We have a slightly complex definition of out-of-sequence:  the previous packet number received off the wire is remembered.  If the new arrival's sequence number is less than previous, then previous is reset (to 0).  The new packet is then declared out-of-sequence if there are any packets missing between the "previous" packet and the one which just arrived (because the missing packets should have been filled in between the previous packet and the new arrival).  This works regardless of whether the peer's retransmission algorithm has been invoked, or not (i.e. whether this is the first or subsequent pass over the sequence of packets).  All this assumes that "most" of the time, packets are delivered in the same *order* as they are transmitted, with, possibly, some packets lost due to transmission errors along the way. */

	u_long prev;			/* "Previous packet" sequence number */
	register struct rx_packet *tp;	/* Temporary packet pointer */
	register struct rx_packet *nxp;	/* Next packet pointer, for queue_Scan */
	int nTwixt;			/* Number of packets between previous and new one */

	/* If the new packet's sequence number has been sent to the application already, then this is a duplicate */
	if (np->header.seq < call->rnext) {
	    rx_stats.dupPacketsRead++;
	    np = rxi_SendAck(call, np, seq, np->header.serial, np->header.flags, RX_ACK_DUPLICATE);
	    call->rprev = seq;
	    return np;
	}

	/* If the sequence number is greater than what can be accomodated by the current window, then send a negative acknowledge and drop the packet */
	if (call->rnext + call->rwind <= np->header.seq) {
	    np = rxi_SendAck(call, np, seq, np->header.serial, np->header.flags, RX_ACK_EXCEEDS_WINDOW);
	    call->rprev = seq;
	    return np;
	}

	/* Look for the packet in the queue of old received packets */
	prev = call->rprev;
	if (prev > np->header.seq) prev = 0;
	for (nTwixt = 0, queue_Scan(&call->rq, tp, nxp, rx_packet)) {
	    /*Check for duplicate packet */
	    if (np->header.seq == tp->header.seq) {
		rx_stats.dupPacketsRead++;
		np = rxi_SendAck(call, np, np->header.seq, np->header.serial, np->header.flags, RX_ACK_DUPLICATE);
		call->rprev = seq;
		return np;
	    }
	    /* Count the number of packets received 'twixt the previous packet and the new packet */
	    if (tp->header.seq > prev && tp->header.seq <np->header.seq) nTwixt++;
	    /* If we find a higher sequence packet, break out and insert the new packet here. */
	    if (np->header.seq < tp->header.seq) break;
	}

	/* It's within the window:  add it to the the receive queue.  tp is left by the previous loop either pointing at the packet before which to insert the new packet, or at the queue head if the queue is empty or the packet should be appended. */
	queue_InsertBefore(tp, np);

	/* Acknowledge the packet if it's out of sequence or if requested by peer */
	if (np->header.flags&RX_REQUEST_ACK) {
	    np = rxi_SendAck(call, 0, np->header.seq, np->header.serial, np->header.flags, RX_ACK_REQUESTED);
	    call->rprev = seq;
	    return np;
	}
	call->rprev = seq;
	np = 0;
    }
    return np;
}

/* The real smarts of the whole thing.  Right now somewhat short-changed. */
struct rx_packet *rxi_ReceiveAckPacket(call, np)
    register struct rx_call *call;
    struct rx_packet *np;
{
    struct rx_ackPacket *ap = (struct rx_ackPacket *) np->wire.data;
    int nAcks = ap->nAcks;
    register struct rx_packet *tp;
    register struct rx_packet *nxp;	/* Next packet pointer for queue_Scan */
    register struct rx_connection *conn = call->conn;
    struct rx_peer *peer = conn->peer;
    u_long first = ntohl(ap->firstPacket);
    u_long serial = ntohl(ap->serial);
#ifdef notdef
    u_long skew = ntohs(ap->maxSkew);
#else
    /* because there are CM's that are bogus, sending weird values for this. */
    u_long skew = 0;
#endif
    int needRxStart = 0;

    rx_stats.ackPacketsRead++;

#ifdef RXDEBUG
    if (Log) {
	fprintf(Log, "RACK: reason %x previous %u seq %u serial %u skew %d first %u", ap->reason, ntohl(ap->previousPacket), np->header.seq, serial, skew, ntohl(ap->firstPacket));
	if (nAcks) {
	    int offset;
	    for (offset = 0; offset < nAcks; offset++) 
		putc(ap->acks[offset] == RX_ACK_TYPE_NACK? '-' : '*', Log);
	}
	putc('\n', Log);
    }
#endif

    /* if a server connection has been re-created, it doesn't remember what
	serial # it was up to.  An ack will tell us, since the serial field
	contains the largest serial received by the other side */
    if (conn->type == RX_SERVER_CONNECTION && conn->serial < serial) {
	conn->serial = serial+1;
    }

    /* Update the outgoing packet skew value to the latest value of the peer's incoming packet skew value.  The ack packet, of course, could arrive out of order, but that won't affect things much */
    peer->outPacketSkew = skew;

    /* Check for packets that no longer need to be transmitted, and discard them.  This only applies to packets positively acknowledged as having been sent to the peer's upper level.  All other packets must be retained.  So only packets with sequence numbers <  ap->firstPacket are candidates. */
    while (queue_IsNotEmpty(&call->tq)) {
	struct clock thisRtt;
	tp = queue_First(&call->tq, rx_packet);
	if (tp->header.seq >= first) break;
	call->tfirst = tp->header.seq + 1;
	if (tp->header.serial == serial) rxi_ComputeRoundTripTime(tp, &thisRtt);
	queue_Remove(tp);
	rxi_FreePacket(tp); /* rxi_FreePacket mustn't wake up anyone, preemptively. */
    }

    /* N.B. we don't turn off any timers here.  They'll go away by themselves, anyway */

    /* Now go through explicit acks/nacks and record the results in the waiting packets.  These are packets that can't be released yet, even with a positive acknowledge.  This positive acknowledge only means the packet has been received by the peer, not that it will be retained long enough to be sent to the peer's upper level.  In addition, reset the transmit timers of any missing packets (those packets that must be missing because this packet was out of sequence) */

    for (queue_Scan(&call->tq, tp, nxp, rx_packet)) {
	struct clock thisRtt;
	/* Update round trip time if the ack was stimulated on receipt of this packet */
	if (tp->header.serial == serial) rxi_ComputeRoundTripTime(tp, &thisRtt);

	/* Set the acknowledge flag per packet based on the information in the ack packet.  It's possible for an acknowledged packet to be downgraded */
	if (tp->header.seq < first + nAcks) {
	    /* Explicit ack information:  set it in the packet appropriately */
	    tp->acked = (ap->acks[tp->header.seq - first] == RX_ACK_TYPE_ACK);
	}
	else {
	    /* No ack information:  the packet may have been acknowledged previously, but that is now rescinded (the peer may have reduced the window size) */
	    tp->acked = 0;
	}
	
	/* If the packet isn't yet acked, and its serial number indicates that it was transmitted before the packet which prompted the acknowledge (that packet's serial number is supplied in the ack packet), then schedule the packet to be transmitted *soon*.  This is done by resetting the retransmit time in the packet to the current time.  Actually this is slightly more intelligent:  to guard against packets that have been transmitted out-of-order by the network (this even happens on the local token ring with our IBM RT's!), the degree of out-of-orderness (skew) of the packet is compared against the maximum skew for this peer.  If it is less, we don't retransmit yet.  Note that we don't do this for packets with zero serial numbers:  they never have been transmitted. */
	if (!tp->acked && tp->header.serial && tp->header.serial + skew <= serial) {
	    rx_stats.dataPacketsPushed++;
	    clock_GetTime(&tp->retryTime);
	    needRxStart = 1;
#ifdef RXDEBUG
	    if (Log) fprintf(Log, "Pushed packet seq %d serial %d, new time %d.%d\n", tp->header.seq, tp->header.serial, tp->retryTime.sec, tp->retryTime.usec/1000);
#endif			     
	}
    }

    /* If the window has been extended by this acknowledge packet, then wakeup a sender waiting in alloc for window space, or try sending packets now, if he's been sitting on packets due to lack of window space */
    if (call->tnext < call->tfirst + call->twind)  {
	if (call->flags & RX_CALL_WAIT_WINDOW_ALLOC) {
	    call->flags &= ~RX_CALL_WAIT_WINDOW_ALLOC;
	    osi_Wakeup(&call->twind);
	}
	if (call->flags & RX_CALL_WAIT_WINDOW_SEND) {
	    call->flags &= ~RX_CALL_WAIT_WINDOW_SEND;
	    needRxStart = 1;
	}
    }

    /*if (needRxStart) rxi_Start(0, call);*/
    rxi_Start(0, call); /* Force rxi_Restart for now:  skew problems!!! */
    return np;
}

/* Received a response to a challenge packet */
struct rx_packet *rxi_ReceiveResponsePacket(conn, np)
    register struct rx_connection *conn;
    register struct rx_packet *np;
{
    int error;

    /* Ignore the packet if we're the client */
    if (conn->type == RX_CLIENT_CONNECTION) return np;

    /* If already authenticated, ignore the packet (it's probably a retry) */
    if (RXS_CheckAuthentication(conn->securityObject, conn) == 0)
	return np;

    /* Otherwise, have the security object evaluate the response packet */
    error = RXS_CheckResponse(conn->securityObject, conn, np);
    if (error) {
	/* If the response is invalid, reset the connection, sending an abort to the peer */
	rxi_ConnectionError(conn, error);
	return rxi_SendConnectionAbort(conn, np);
    }
    else {
	/* If the response is valid, any calls waiting to attach servers can now do so */
	int i;
	for (i=0; i<RX_MAXCALLS; i++) {
	    struct rx_call *call = conn->call[i];
	    if (call && call->state == RX_STATE_PRECALL) rxi_AttachServerProc(call);
	}
    }
    return np;
}

/* A client has received an authentication challenge:  the security object is asked to cough up a respectable response packet to send back to the server.  The server is responsible for retrying the challenge if it fails to get a response. */
struct rx_packet *rxi_ReceiveChallengePacket(conn, np)
    register struct rx_connection *conn;
    register struct rx_packet *np;
{
    int error;

    /* Ignore the challenge if we're the server */
    if (conn->type == RX_SERVER_CONNECTION) return np;

    /* Send the security object the challenge packet.  It is expected to fill in the response. */
    error = RXS_GetResponse(conn->securityObject, conn, np);

    /* If the security object is unable to return a valid response, reset the connection and send an abort to the peer.  Otherwise send the response packet to the peer connection. */
    if (error) {
	rxi_ConnectionError(conn, error);
	np = rxi_SendConnectionAbort(conn, np);
    }
    else {
	np = rxi_SendSpecial((struct rx_call *)0, conn, np, RX_PACKET_TYPE_RESPONSE, (char *) 0, -1);
    }
    return np;
}

/* Find an available server process to service the current request in the given call structure.  If one isn't available, queue up this call so it eventually gets one */
void rxi_AttachServerProc(call)
register struct rx_call *call;
{
    register struct rx_serverQueueEntry *sq;
    register struct rx_service *service = call->conn->service;
    /* May already be attached */
    if (call->state == RX_STATE_ACTIVE) return;

    if (service->nRequestsRunning >= service->maxProcs || queue_IsEmpty(&rx_idleServerQueue)) {
	/* If there are no processes available to service this call, put the call on the incoming call queue (unless it's already on the queue) */
	if (!(call->flags & RX_CALL_WAIT_PROC)) {
	    call->flags |= RX_CALL_WAIT_PROC;
	    queue_Append(&rx_incomingCallQueue, call);
	}
    }
    else {
	sq = queue_First(&rx_idleServerQueue, rx_serverQueueEntry);
	queue_Remove(sq);
	sq->newcall = call;
	if (call->flags & RX_CALL_WAIT_PROC) {
	    /* Conservative:  I don't think this should happen */
	    call->flags &= ~RX_CALL_WAIT_PROC;
	    queue_Remove(call);
	}
	call->state = RX_STATE_ACTIVE;
	call->mode = RX_MODE_RECEIVING;
	if (call->flags & RX_CALL_CLEARED) {
	    /* send an ack now to start the packet flow up again */
	    call->flags &= ~RX_CALL_CLEARED;
	    rxi_SendAck(call, 0, 0, 0, 0, RX_ACK_DELAY);
	}
	service->nRequestsRunning++;
	osi_Wakeup(sq);
    }
}

/* Delay the sending of an acknowledge event for a short while, while a new call is being prepared (in the case of a client) or a reply is being prepared (in the case of a server).  Rather than sending an ack packet, an ACKALL packet is sent. */
void rxi_AckAll(event, call, dummy)
struct rxevent *event;
register struct rx_call *call;
char *dummy;
{
    if (event) call->delayedAckEvent = (struct rxevent *) 0;
    rxi_SendSpecial(call, call->conn, (struct rx_packet *) 0, RX_PACKET_TYPE_ACKALL, (char *) 0, 0);
}

void rxi_SendDelayedAck(event, call, dummy)	
struct rxevent *event;
register struct rx_call *call;
char *dummy;
{
    if (event) call->delayedAckEvent = (struct rxevent *) 0;
    (void) rxi_SendAck(call, 0, 0, 0, 0, RX_ACK_DELAY);
}

/* Clear out the transmit queue for the current call (all packets have been received by peer) */
void rxi_ClearTransmitQueue(call)
    register struct rx_call *call;
{
    register struct rx_packet *p, *tp;
    for (queue_Scan(&call->tq, p, tp, rx_packet)) queue_Remove(p), rxi_FreePacket(p);
    rxevent_Cancel(call->resendEvent);
}

void rxi_ClearReceiveQueue(call)
    register struct rx_call *call;
{
    register struct rx_packet *p, *tp;
    for (queue_Scan(&call->rq, p, tp, rx_packet)) queue_Remove(p), rxi_FreePacket(p);
    if (call->state == RX_STATE_PRECALL) call->flags |= RX_CALL_CLEARED;
}

/* Send an abort packet for the specified call */
struct rx_packet *rxi_SendCallAbort(call, packet)
    register struct rx_call *call;
    struct rx_packet *packet;
{
    if (call->error) {
	long error;
	error = htonl(call->error);
	packet = rxi_SendSpecial(call, call->conn, packet, RX_PACKET_TYPE_ABORT, (char *)&error, sizeof(error));
    }
    return packet;
}

/* Send an abort packet for the specified connection.  Np is an optional packet that can be used to send the abort. */
struct rx_packet *rxi_SendConnectionAbort(conn, packet)
    register struct rx_connection *conn;
    struct rx_packet *packet;
{
    if (conn->error) {
	long error;
	error = htonl(conn->error);
	packet = rxi_SendSpecial((struct rx_call *)0, conn, packet, RX_PACKET_TYPE_ABORT, (char *)&error, sizeof(error));
    }
    return packet;
    }

/* Associate an error all of the calls owned by a connection.  Called with error non-zero.  This is only for really fatal things, like bad authentication responses.   The connection itself is set in error at this point, so that future packets received will be rejected. */
void rxi_ConnectionError(conn, error)
    register struct rx_connection *conn;
    register long error;
{
    if (error) {
	register int i;
	if (conn->challengeEvent)
	    rxevent_Cancel(conn->challengeEvent);
	for (i=0; i<RX_MAXCALLS; i++) {
	    struct rx_call *call = conn->call[i];
	    if (call) rxi_CallError(call, error);
	}
	conn->error = error;
    }
}

/* Reset all of the calls associated with a connection. */
void rxi_ResetConnection(conn)
    register struct rx_connection *conn;
{
    register int i;
    for (i=0; i<RX_MAXCALLS; i++) {
	struct rx_call *call = conn->call[i];
	if (call) rxi_ResetCall(call);
    }

    /* get rid of pending events that could zap us later */
    if (conn->challengeEvent)
	rxevent_Cancel(conn->challengeEvent);
}

void rxi_CallError(call, error)
    register struct rx_call *call;
    long error;
{
    if (call->error) error = call->error;
    rxi_ResetCall(call);
    call->error = error;
    call->mode = RX_MODE_ERROR;
}

/* Reset various fields in a call structure, and wakeup waiting processes.  Some fields aren't changed:	 state & mode are not touched (these must be set by the caller), and bufptr, nLeft, and nFree are not reset, since these fields are manipulated by unprotected macros, and may only be reset by non-interrupting code. */
void rxi_ResetCall(call)
    register struct rx_call *call;
{
    register int flags;

    /* Notify anyone who is waiting for asynchronous packet arrival */
    if (call->arrivalProc) {
	(*call->arrivalProc)(call, call->arrivalProcHandle, call->arrivalProcArg);
	call->arrivalProc = (VOID (*)()) 0;
    }

    flags = call->flags;
    rxi_ClearReceiveQueue(call);
    rxi_ClearTransmitQueue(call);
    call->error = 0;
    call->flags = 0;
    call->twind = call->rwind = rx_Window; /* XXXX */
    call->tfirst = call->rnext = call->tnext = 1;
    call->rprev = 0;
    call->lastAcked = 0;
    call->localStatus = call->remoteStatus = 0;
    if (flags&RX_CALL_READER_WAIT) osi_Wakeup(&call->rq);
    if (flags&RX_CALL_WAIT_PACKETS) {
	CheckSendPackets();
    }
    if (flags&RX_CALL_WAIT_WINDOW_ALLOC) osi_Wakeup(&call->twind);
    if (queue_IsOnQueue(call)) queue_Remove(call);
    rxi_KeepAliveOff(call);
    rxevent_Cancel(call->delayedAckEvent);
}

/* Send an acknowledge for the indicated packet (seq,serial)  of the indicated call, for the indicated reason (reason).  This acknowledge will specifically acknowledge receiving the packet, and will also specify which other packets for this call have been received.  This routine returns the packet that was used to the caller.  The caller is responsible for freeing it or re-using it.  This acknowledgement also returns the highest sequence number actually read out by the higher level to the sender; the sender promises to keep around packets that have not been read by the higher level yet (unless, of course, the sender decides to abort the call altogether).   Any of p, seq, serial, pflags, or reason may be set to zero without ill effect.   That is, if they are zero, they will not convey any information.  */
struct rx_packet *rxi_SendAck(call, optionalPacket, seq, serial, pflags, reason)
    register struct rx_call *call;
    register struct rx_packet *optionalPacket; /* Optional packet to use for the send ack (or null) */
    int	seq;			/* Sequence number of the packet we are acking */
    int	serial;			/* Serial number of the packet */
    int	pflags;			/* Flags field from packet header */
    int	reason;			/* Reason an acknowledge was prompted */
{
    struct rx_peer *peer = call->conn->peer;
    struct rx_ackPacket *ap;
    register struct rx_packet *rqp;
    register struct rx_packet *nxp;  /* For queue_Scan */
    register struct rx_packet *p;
    u_char offset;

    if (call->rnext > call->lastAcked) call->lastAcked = call->rnext;
    p = optionalPacket;
    if (!p) p = rxi_AllocPacket(RX_PACKET_CLASS_SPECIAL);
    if (!p) osi_Panic("rxi_SendAck");
    ap = (struct rx_ackPacket *) p->wire.data;
    ap->bufferSpace = htonl(0);	/* Something should go here, sometime */
    ap->reason = reason;

    /* The skew computation is bullshit anyway, since we forgot to check for 0
	in the np->header.serial *and* there are some dudes out there that use
	ntohl instead of ntohs.  */
    ap->serial = htonl(call->conn->maxSerial);
    ap->maxSkew	= 0;	/* used to be peer->inPacketSkew */

    ap->firstPacket = htonl(call->rnext); /* First packet not yet forwarded to reader */
    ap->previousPacket = htonl(call->rprev); /* Previous packet received */
    for (offset = 0, queue_Scan(&call->rq, rqp, nxp, rx_packet)) {
	while (rqp->header.seq > call->rnext + offset) ap->acks[offset++] = RX_ACK_TYPE_NACK;
	ap->acks[offset++] = RX_ACK_TYPE_ACK;
    }
    ap->nAcks = offset;
    p->length = rx_AckDataSize(offset);
    p->header.serviceId = call->conn->serviceId;
    p->header.cid = (call->conn->cid | call->channel);
    p->header.callNumber = *call->callNumber;
    p->header.seq = seq;
    p->header.securityIndex = call->conn->securityIndex;
    p->header.epoch = call->conn->epoch;
    p->header.type = RX_PACKET_TYPE_ACK;
    p->header.flags = 0;
    if (reason == RX_ACK_PING)
	p->header.flags |= RX_REQUEST_ACK;
    if (call->conn->type == RX_CLIENT_CONNECTION) p->header.flags |= RX_CLIENT_INITIATED;
#ifdef RXDEBUG
    if (Log) {
	fprintf(Log, "SACK: reason %x previous %u seq %u first %u", ap->reason, ntohl(ap->previousPacket), p->header.seq, ntohl(ap->firstPacket));
	if (ap->nAcks) {
	    for (offset = 0; offset < ap->nAcks; offset++) 
		putc(ap->acks[offset] == RX_ACK_TYPE_NACK? '-' : '*', Log);
	}
	putc('\n', Log);
    }
#endif
    rxi_Send(call, p);
    rx_stats.ackPacketsSent++;
    if (!optionalPacket) rxi_FreePacket(p);
    return optionalPacket;   /* Return packet for re-use by caller */
}

void rxi_PrepareSendPacket(call, p, last)
    register struct rx_call *call;
    register struct rx_packet *p;
    register int last;
{
    register struct rx_connection *conn = call->conn;
    p->acked = 0;
    p->header.cid = (conn->cid | call->channel);
    p->header.serviceId = conn->serviceId;
    p->header.securityIndex = conn->securityIndex;
    p->header.callNumber = *call->callNumber;
    p->header.seq = call->tnext++;
    p->header.epoch = conn->epoch;
    p->header.type = RX_PACKET_TYPE_DATA;
    p->header.flags = 0;
    if (conn->type == RX_CLIENT_CONNECTION) p->header.flags |= RX_CLIENT_INITIATED;
    if (last) p->header.flags |= RX_LAST_PACKET;
    clock_Zero(&p->retryTime);   /* Never yet transmitted */
    p->header.serial = 0;	     /* Another way of saying it hasn't been transmitted... */
    RXS_PreparePacket(conn->securityObject, call, p);
}

/* This routine is called when new packets are readied for transmission and when retransmission may be necessary, or when the transmission window or burst count are favourable.  This should be better optimized for new packets, the usual case, now that we've got rid of queues of send packets. XXXXXXXXXXX */
void rxi_Start(event, call)
    struct rxevent *event;
    register struct rx_call *call;
{
    int nSent = 0;
    struct rx_packet *p;
    register struct rx_packet *nxp;  /* Next pointer for queue_Scan */
    register struct rx_packet *lastPacket;
    struct rx_peer *peer = call->conn->peer;
    struct clock now, retryTime;
    int haveEvent;
    
    /* If rxi_Start is being called as a result of a resend event, then make sure that the event pointer is removed from the call structure, since there is no longer a per-call retransmission event pending. */
    if (event && event == call->resendEvent) call->resendEvent = 0;

    if (queue_IsNotEmpty(&call->tq)) { /* If we have anything to send */
	/* Get clock to compute the re-transmit time for any packets in this burst.  Note, if we back off, it's reasonable to back off all of the packets in the same manner, even if some of them have been retransmitted more times than more recent additions */
	clock_GetTime(&now);
	retryTime = now;    /* initialize before use */
	clock_Add(&retryTime, &peer->timeout);

	/* Send (or resend) any packets that need it, subject to window restrictions and congestion burst control restrictions.  Ask for an ack on the last packet sent in this burst.  For now, we're relying upon the window being considerably bigger than the largest number of packets that are typically sent at once by one initial call to rxi_Start.  This is probably bogus (perhaps we should ask for an ack when we're half way through the current window?).  Also, for non file transfer applications, this may end up asking for an ack for every packet.  Bogus. XXXX */
	for (lastPacket = (struct rx_packet *) 0, queue_Scan(&call->tq, p, nxp, rx_packet)) {
	    if (p->acked) {
		rx_stats.ignoreAckedPacket++;
		continue;	  /* Ignore this packet if it has been acknowledged */
	    }

	    /* Turn off all flags except these ones, which are the same on each transmission */
	    p->header.flags &= RX_PRESET_FLAGS;


	    if (p->header.seq >= call->tfirst + call->twind) {
		call->flags |= RX_CALL_WAIT_WINDOW_SEND; /* Wait for transmit window */
		/* Note:  if we're waiting for more window space, we can still send retransmits; hence we don't return here, but break out to schedule a retransmit event */
		break;
	    }

	    /* If we're not allowed to send any more in the current burst, make sure we get scheduled later.  Also schedule an event to "give back" the packets we've used, when the burst time has elapsed (if we used any packets at all). */
	    if (peer->burstSize && !peer->burst) {
		if (nSent) {
		    /* Send off the last packet */
		    if ((lastPacket->header.flags & RX_LAST_PACKET) == 0) ACKHACK(lastPacket);
		    rxi_Send(call, lastPacket);
		    rxi_ScheduleDecongestionEvent(call, nSent);
		}
		rxi_CongestionWait(call);
		/* Note: if we're waiting for congestion to ease, we can't send any packets, including retransmits.  Hence we do not schedule a new retransmit event right now */
		return;
	    }

	    /* Transmit the packet if it has never been sent, or retransmit it if the current timeout for this host for this packet has elapsed */
	    if (!clock_IsZero(&p->retryTime)) {
		if (clock_Lt(&now, &p->retryTime)) continue;
		/* Always request an ack on a retransmitted packet; this will help to get the data moving again, especially if the packet is near the beginning of the window.  Actually, XXXX, we may want to do just that:  only request the acks if the packet is in, say, the first half of the window */
		p->header.flags |= RX_REQUEST_ACK;
		peer->reSends++, rx_stats.dataPacketsReSent++;
	    }
	    else {
		peer->nSent++, rx_stats.dataPacketsSent++;
	    }

	    /* Install the new retransmit time for the packet, and record the time sent */
	    p->retryTime = retryTime;
	    p->timeSent = now;

	    /* Send the previous packet, and remember this one--we don't send it immediately, so we can tag it as requiring an ack later, if necessary */
	    if (peer->burstSize) peer->burst--;
	    nSent++;
	    if (lastPacket) {
		/* Tag this packet as not being the last in this group, for the receiver's benefit */
		lastPacket->header.flags |= RX_MORE_PACKETS;
		rxi_Send(call, lastPacket);
	    }
	    lastPacket = p;
	}

	/* If any packets are to be sent, send them and post a decongestion event to bump the burst count that we used up sending the packets */
	if (nSent) {
	    /* lastPacket should be non-null, here.  Since it's the last packet for the forseeable future, post an ack request for it XXXX  */
	    if ((lastPacket->header.flags & RX_LAST_PACKET) == 0) ACKHACK(lastPacket);
	    rxi_Send(call, lastPacket);
	    if (peer->burstSize) rxi_ScheduleDecongestionEvent(call, nSent);
	}

	/* Always post a resend event, if there is anything in the queue, and resend is possible.  There should be at least one unacknowledged packet in the queue ... otherwise none of these packets should be on the queue in the first place. */
	if (call->resendEvent) {
	    /* If there's an existing resend event, then if its expiry time is sooner than the new one, then it must be less than any possible expiry time (because it represents all previous packets sent that may still need retransmitting).  In this case, just leave that event as scheduled */
	    if (clock_Le(&call->resendEvent->eventTime, &retryTime)) return;
	    /* Otherwise, cancel the existing event and post a new one */
	    rxevent_Cancel(call->resendEvent);
	}
	/* Loop to find the earliest event.  I *know* XXXX that this can be coded more elegantly (perhaps rolled into the above code) */
	for (haveEvent = 0, queue_Scan(&call->tq, p, nxp, rx_packet)) {
	    if (!p->acked && !clock_IsZero(&p->retryTime)) {
		haveEvent = 1;
		if (clock_Lt(&p->retryTime, &retryTime)) retryTime = p->retryTime;
	    }
	}
	/* Post a new event to re-run rxi_Start when retries may be needed */
	if (haveEvent) {
	    call->resendEvent = rxevent_Post(&retryTime, rxi_Start, (char *)call, 0);
	}
    }
}

/* Also adjusts the keep alive parameters for the call, to reflect that we have just sent a packet (so keep alives aren't sent immediately) */
rxi_Send(call, p)
    register struct rx_call *call;		
    register struct rx_packet *p;
{
    register struct rx_connection *conn = call->conn;

    /* Stamp each packet with the user supplied status */
    p->header.userStatus = call->localStatus;

    /* Allow the security object controlling this call's security to make any last-minute changes to the packet */
    RXS_SendPacket(conn->securityObject, call, p);

    /* Actually send the packet, filling in more connection-specific fields */
    rxi_SendPacket(conn, p);

    /* Update last send time for this call (for keep-alive processing), and for the connection (so that we can discover idle connections) */
    conn->lastSendTime = call->lastSendTime = clock_Sec();

    /* Since we've just sent SOME sort of packet to the peer, it's safe to nuke any scheduled end-of-packets ack */
    rxevent_Cancel(call->delayedAckEvent);
}

/* Send the packet to appropriate destination for the specified connection.  The header is first encoded and placed in the packet. */
rxi_SendPacket(conn, p)
    register struct rx_connection *conn;		
    register struct rx_packet *p;
{
    struct sockaddr_in addr;
    register struct rx_peer *peer = conn->peer;
    osi_socket socket;
#ifdef RXDEBUG
    char deliveryType;
#endif

    /* The address we're sending the packet to */
    addr.sin_family = AF_INET;
    addr.sin_port = peer->port;
    addr.sin_addr.s_addr = peer->host;

    /* This stuff should be revamped, I think, so that most, if not all, of the header stuff is always added here.  We could probably do away with the encode/decode routines. XXXXX */

    /* Stamp each packet with a unique serial number.  The serial number is maintained on a connection basis because some types of security may be based on the serial number of the packet, and security is handled on a per authenticated-connection basis. */
    p->header.serial = ++conn->serial; /* Pre-increment, to guarantee no zero serial number; a zero serial number means the packet was never sent. */

    /* Get network byte order header */
    rxi_EncodePacketHeader(p);	

    /* Send the packet out on the same socket that related packets are being received on */
    socket = (conn->type == RX_CLIENT_CONNECTION? rx_socket: conn->service->socket);

#ifdef RXDEBUG
    /* Possibly drop this packet,  for testing purposes */
    if (rx_intentionallyDroppedPacketsPer100 > 0 && random() % 100 < rx_intentionallyDroppedPacketsPer100) {
	deliveryType = 'D';  /* Drop the packet */
    }
    else {
	deliveryType = 'S';  /* Send the packet */
#endif RXDEBUG
	/* Loop until the packet is sent.  We'd prefer just to use a blocking socket, but unfortunately the interface doesn't allow us to have the socket block in send mode, and not block in receive mode */
	osi_NetSend(socket, &addr, &p->wire, p->length+RX_HEADER_SIZE);
#ifdef RXDEBUG    
    }
    dpf(("%c %d %s: %x.%d.%d.%d.%d.%d.%d flags %d, packet %x resend %d.%d", deliveryType, p->header.serial, rx_packetTypes[p->header.type-1], peer->host, peer->port, p->header.serial, p->header.epoch, p->header.cid, p->header.callNumber, p->header.seq, p->header.flags, p, p->retryTime.sec, p->retryTime.usec/1000));
#endif
    rx_stats.packetsSent[p->header.type-1]++;
}

/* Send a "special" packet to the peer connection.  If call is specified, then the packet is directed to a specific call channel associated with the connection, otherwise it is directed to the connection only. Uses optionalPacket if it is supplied, rather than allocating a new packet buffer.  Nbytes is the length of the data portion of the packet.  If data is non-null, nbytes of data are copied into the packet.  Type is the type of the packet, as defined in rx.h.   Bug:  there's a lot of duplication between this and other routines.  This needs to be cleaned up. */
struct rx_packet *rxi_SendSpecial(call, conn, optionalPacket, type, data, nbytes)
    register struct rx_call *call;
    register struct rx_connection *conn;
    struct rx_packet *optionalPacket;
    int type;
    char *data;
    int nbytes;
{
    /* Some of the following stuff should be common code for all packet sends (it's repeated elsewhere) */
    register struct rx_packet *p;
    int channel, callNumber;
    if (call) {
	channel = call->channel;
	callNumber = *call->callNumber;
    } else {
	channel = 0;
	callNumber = 0;
    }
    p = optionalPacket;
    if (!p) {
	p = rxi_AllocPacket(RX_PACKET_CLASS_SPECIAL);
	if (!p) osi_Panic("rxi_SendSpecial failure");
    }
    if (nbytes != -1) p->length = nbytes;
    p->header.serviceId = conn->serviceId;
    p->header.securityIndex = conn->securityIndex;
    p->header.cid = (conn->cid | channel);
    p->header.callNumber = callNumber;
    p->header.seq = 0;
    p->header.epoch = conn->epoch;
    p->header.type = type;
    p->header.flags = 0;
    if (conn->type == RX_CLIENT_CONNECTION) p->header.flags |= RX_CLIENT_INITIATED;
    if (data) bcopy(data, rx_DataOf(p), nbytes);
    if (call) rxi_Send(call, p);
    else rxi_SendPacket(conn, p);
    if (!optionalPacket) rxi_FreePacket(p);
    return optionalPacket;
}

/* Check if a call needs to be destroyed.  Called by keep-alive code to ensure
    that things are fine.  Also called periodically to guarantee that nothing
    falls through the cracks (e.g. (error + dally) connections have keepalive
    turned off.  Returns 0 if conn is well, -1 otherwise.  If otherwise, call may be freed!  */
int rxi_CheckCall(call)
    register struct rx_call *call;
{
    register struct rx_connection *conn = call->conn;
    u_long now;

    now = clock_Sec();
    /* These are computed to the second (+- 1 second).  But that's good enough for these values, which should be a significant number of seconds. */
    if (now > call->lastReceiveTime + conn->secondsUntilDead) {
	if (call->state == RX_STATE_ACTIVE) rxi_CallError(call, RX_CALL_DEAD);
	else rxi_FreeCall(call);
	/* Non-active calls are destroyed if they are not responding to pings;  active calls are simply flagged in error, so the attached process can die reasonably gracefully. */
	return -1;
    }
    /* see if we have a non-activity timeout */
    if (conn->type == RX_SERVER_CONNECTION && call->startWait && call->startWait + conn->service->idleDeadTime < now) {
	if (call->state == RX_STATE_ACTIVE) {
	    rxi_CallError(call, RX_CALL_TIMEOUT);
	    return -1;
	}
    }
    /* see if we have a hard timeout */
    if (conn->hardDeadTime && now > conn->hardDeadTime + call->startTime) {
	if (call->state == RX_STATE_ACTIVE) rxi_CallError(call, RX_CALL_TIMEOUT);
	return -1;
    }
    return 0;
}


/* When a call is in progress, this routine is called occasionally to make sure that some traffic has arrived (or been sent to) the peer. If nothing has arrived in a reasonable amount of time, the call is declared dead; if nothing has been sent for a while, we send a keep-alive packet (if we're actually trying to keep the call alive) */
void rxi_KeepAliveEvent(event, call, dummy)
    struct rxevent *event;
    register struct rx_call *call;
{
    register struct rx_connection *conn = call->conn;
    u_long now;
    call->keepAliveEvent = (struct rxevent *) 0;
    now = clock_Sec();

    if (rxi_CheckCall(call)) return;

    /* Don't try to keep alive dallying calls */
    if (call->state != RX_STATE_DALLY && now - call->lastSendTime > conn->secondsUntilPing) {
	(void) rxi_SendAck(call, 0, 0, 0, 0, RX_ACK_PING);
    }
    rxi_ScheduleKeepAliveEvent(call);
}

void rxi_ScheduleKeepAliveEvent(call)
    register struct rx_call *call;
{
    if (!call->keepAliveEvent) {
	struct clock when;
	clock_GetTime(&when);
	when.sec += call->conn->secondsUntilPing;
	call->keepAliveEvent = rxevent_Post(&when, rxi_KeepAliveEvent, call, 0);
    }
}

/* N.B. rxi_KeepAliveOff:  is defined earlier as a macro */
void rxi_KeepAliveOn(call)
    register struct rx_call *call;
{
    /* Pretend last packet received was received now--i.e. if another packet isn't received within the keep alive time, then the call will die; Initialize last send time to the current time--even if a packet hasn't been sent yet.  This will guarantee that a keep-alive is sent within the ping time */
    call->lastReceiveTime = call->lastSendTime = clock_Sec();
    rxi_ScheduleKeepAliveEvent(call);
}

/* This routine is called periodically (every RX_AUTH_REQUEST_TIMEOUT seconds) to ask the client to authenticate itself.  The routine issues a challenge to the client, which is obtained from the security object associated with the connection */
void rxi_ChallengeEvent(event, conn, dummy)
    struct rxevent *event;
    register struct rx_connection *conn;
    char *dummy;
{
    conn->challengeEvent = (struct rxevent *) 0;
    if (RXS_CheckAuthentication(conn->securityObject, conn) != 0) {
	register struct rx_packet *packet;
	struct clock when;
	packet = rxi_AllocPacket(RX_PACKET_CLASS_SPECIAL);
	if (!packet) osi_Panic("rxi_ChallengeEvent");
	RXS_GetChallenge(conn->securityObject, conn, packet);
	rxi_SendSpecial((struct rx_call *) 0, conn, packet, RX_PACKET_TYPE_CHALLENGE, (char *) 0, -1);
	rxi_FreePacket(packet);
	clock_GetTime(&when);
	when.sec += RX_CHALLENGE_TIMEOUT;
	conn->challengeEvent = rxevent_Post(&when, rxi_ChallengeEvent, conn, 0);
    }
}

/* Call this routine to start requesting the client to authenticate itself.  This will continue until authentication is established, the call times out, or an invalid response is returned.  The security object associated with the connection is asked to create the challenge at this time.  N.B.  rxi_ChallengeOff is a macro, defined earlier. */
void rxi_ChallengeOn(conn)
    register struct rx_connection *conn;
{
    if (!conn->challengeEvent) {
	RXS_CreateChallenge(conn->securityObject, conn);
	rxi_ChallengeEvent((struct rxevent *)0, conn, 0);
    };
}

/* Called by event.c when a decongestion event (setup by rxi_CongestionWait) occurs.  This adds back in to the burst count for the specified host the number of packets that were sent at the time the event was scheduled.  It also calls rxi_Start on as many waiting calls as possible before the burst count goes down to zero, again. */
void rxi_DecongestionEvent(event, peer, nPackets)
    struct rxevent *event;
    register struct rx_peer *peer;
    int nPackets;
{
    register struct rx_call *call;
    register struct rx_call *nxcall; /* Next pointer for queue_Scan */
    peer->burst	+= nPackets;
    if (peer->burst > peer->burstSize) peer->burst = peer->burstSize;
    for (queue_Scan(&peer->congestionQueue, call, nxcall, rx_call)) {
	queue_Remove(call);
	/* The rxi_Start may put the call back on the congestion queue.  In that case, peer->burst should be 0 (otherwise no congestion was encountered).  It should go on the end of the queue, to allow other calls to proceed when the next burst is allowed */
	rxi_Start((struct rxevent *)0, call);
	if (!peer->burst) return;
    }
}

/* Schedule an event at a host-dependent time in the future which will add back nPackets to the current allowed burst window.   Any number of these events may be scheduled. */
void rxi_ScheduleDecongestionEvent(call, nPackets)
    register struct rx_call *call;
    int nPackets;
{
    register struct rx_peer *peer = call->conn->peer;
    struct clock tmp;
    clock_GetTime(&tmp);
    clock_Add(&tmp, &peer->burstWait);
    rxevent_Post(&tmp, rxi_DecongestionEvent, (char *)peer, nPackets);
}

/* The caller wishes to have rxi_Start called when the burst count has gone up, and more packets can therefore be sent.  Add the caller to the end of the list of calls waiting for decongestion events to happen.  It's important that it's added to the end so that the rxi_DecongestionEvent procedure always terminates (aside from matters of scheduling fairness). */
void rxi_CongestionWait(call)
    register struct rx_call *call;
{
    if (queue_IsOnQueue(call)) return;
    queue_Append(&call->conn->peer->congestionQueue, call);
}

/* Encode the packet's header (from the struct header in the packet to the net byte order representation in the wire representation of the packet, which is what is actually sent out on the wire) */
void rxi_EncodePacketHeader(p)
register struct rx_packet *p;
{
    register u_long *buf = p->wire.head;

    *buf++ = htonl(p->header.epoch);
    *buf++ = htonl(p->header.cid);
    *buf++ = htonl(p->header.callNumber);
    *buf++ = htonl(p->header.seq);
    *buf++ = htonl(p->header.serial);
    *buf++ = htonl((p->header.type<<24) | (p->header.flags<<16) | (p->header.userStatus<<8) | p->header.securityIndex);
    *buf++ = htonl(p->header.serviceId&0xffff); /* Note: top 16 bits of this word are reserved */
}

/* Decode the packet's header (from net byte order to a struct header) */
void rxi_DecodePacketHeader(p)
register struct rx_packet *p;
{
    register u_long *buf = p->wire.head;
    u_long temp;

    p->header.epoch = ntohl(*buf++);
    p->header.cid = ntohl(*buf++);
    p->header.callNumber = ntohl(*buf++);
    p->header.seq = ntohl(*buf++);
    p->header.serial = ntohl(*buf++);
    temp = ntohl(*buf++);
    /* C will truncate byte fields to bytes for me */
    p->header.type = temp>>24;
    p->header.flags = temp>>16;
    p->header.userStatus = temp>>8;
    p->header.securityIndex = temp>>0;
    temp = ntohl(*buf++);
    p->header.serviceId = (temp&0xffff);
    /* Note: top 16 bits of this last word are reserved */
}

/* Compute round trip time of the packet provided, in *rttp.  Currently, adjust global rtt parameters.  Eventually do this on a per connection basis. */
void rxi_ComputeRoundTripTime(p, rttp)
    register struct rx_packet *p;
    register struct clock *rttp;
{
    clock_GetTime(rttp);
    clock_Sub(rttp, &p->timeSent);
    if (clock_Lt(rttp, &rx_stats.minRtt)) rx_stats.minRtt = *rttp;
    if (clock_Gt(rttp, &rx_stats.maxRtt)) rx_stats.maxRtt = *rttp;
    clock_Add(&rx_stats.totalRtt, rttp);
    rx_stats.nRttSamples++;
}

/* Find all server connections that have not been active for a long time, and toss them */
void rxi_ReapConnections()
{
    struct clock now;
    clock_GetTime(&now);

    /* Find server connection structures that haven't been used for greater than rx_idleConnectionTime */
    {	register struct rx_connection **conn_ptr, **conn_end;
	register int i;
	for (conn_ptr = &rx_connHashTable[0], conn_end = &rx_connHashTable[rx_hashTableSize]; conn_ptr < conn_end; conn_ptr++) {
	    struct rx_connection *conn, *next;
	    for (conn = *conn_ptr; conn; conn = next) {
		next = conn->next;
		/* once a minute look at everything to see what's up */
		for(i=0;i<RX_MAXCALLS;i++) {
		    if (conn->call[i])
			rxi_CheckCall(conn->call[i]);
		}
		if (conn->type == RX_SERVER_CONNECTION) {
		    /* This only actually destroys the connection if there are no outstanding calls */
		    if (conn->lastSendTime + rx_idleConnectionTime < now.sec)
			rx_DestroyConnection(conn);
		}
	    }
	}
    }

    /* Find any peer structures that haven't been used (haven't had an associated connection) for greater than rx_idlePeerTime */
    {	struct rx_peer **peer_ptr, **peer_end;
	for (peer_ptr = &rx_peerHashTable[0], peer_end = &rx_peerHashTable[rx_hashTableSize]; peer_ptr < peer_end; peer_ptr++) {
	    struct rx_peer *peer, *next;
	    for (peer = *peer_ptr; peer; peer = next) {    
		next = peer->next;
		if (peer->refCount == 0 && peer->idleWhen + rx_idlePeerTime < now.sec)
		    rxi_DestroyPeer(peer);
	    }
	}
    }

    /* THIS HACK IS A TEMPORARY HACK.  The idea is that the race condition in
	rxi_AllocSendPacket, if it hits, will be handled at the next conn
	GC, just below.  Really, we shouldn't have to keep moving packets from
	one place to another, but instead ought to always know if we can afford
	to hold onto a packet in its particular use.  */
    if (rx_waitingForPackets) {
	rx_waitingForPackets = 0;
	osi_Wakeup(&rx_waitingForPackets);
    }

    now.sec += RX_REAP_TIME; /* Check every RX_REAP_TIME seconds */
    rxevent_Post(&now, rxi_ReapConnections, 0, 0);
}

#ifdef RXDEBUG
/* Don't call this debugging routine directly; use dpf */
void rxi_DebugPrint(format, a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, a11, a12, a13, a14, a15)
    char *format;
{
    struct clock now;
    clock_GetTime(&now);
    fprintf(Log, " %u.%u:", now.sec, now.usec/1000);
    fprintf(Log, format, a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, a11, a12, a13, a14, a15);
    putc('\n', Log);
}
#endif

#ifdef RXDEBUG
void rx_PrintStats(file)
FILE *file;
{
#define X rx_stats
    int i;

    fprintf(file, "rx stats: free packets %d, allocs %d, alloc-failures(rcv %d,send %d,ack %d)\n", rx_nFreePackets, X.packetRequests, X.noPackets[0], X.noPackets[1], X.noPackets[2]);
    fprintf(file, "   greedy %d, bogusReads %d (last from host %x), noPackets %d, noBuffers %d, selects %d, sendSelects %d\n", X.socketGreedy, X.bogusPacketOnRead, X.bogusHost, X.noPacketOnRead, X.noPacketBuffersOnRead, X.selects, X.sendSelects);
    fprintf(file, "   packets read: ");
    for (i = 0; i<RX_N_PACKET_TYPES; i++) fprintf(file, "%s %d ", rx_packetTypes[i], rx_stats.packetsRead[i]);
    fprintf(file, "\n");
    fprintf(file, "   other read counters: data %d, ack %d, dup %d spurious %d\n", X.dataPacketsRead, X.ackPacketsRead, X.dupPacketsRead, X.spuriousPacketsRead);
    fprintf(file, "   packets sent: ");
    for (i = 0; i<RX_N_PACKET_TYPES; i++) fprintf(file, "%s %d ", rx_packetTypes[i], rx_stats.packetsSent[i]);
    fprintf(file, "\n");
    fprintf(file, "   other send counters: ack %d, data %d (not resends), resends %d, pushed %d, acked&ignored %d\n", X.ackPacketsSent, X.dataPacketsSent, X.dataPacketsReSent, X.dataPacketsPushed, X.ignoreAckedPacket);
    fprintf(file, "   Average rtt is %0.3f, with %d samples\n", clock_Float(&X.totalRtt)/X.nRttSamples, X.nRttSamples);
    fprintf(file, "   Minimum rtt is %0.3f, maximum is %0.3f\n", clock_Float(&X.minRtt), clock_Float(&X.maxRtt));
    fprintf(file, "   %d server connections, %d client connections, %d peer structs, %d call structs, %d free call structs\n", X.nServerConns, X.nClientConns, X.nPeerStructs, X.nCallStructs, X.nFreeCallStructs);
    fprintf(file, "   %d clock updates\n", clock_nUpdates);
#undef X
}

void rx_PrintPeerStats(file, peer)
FILE *file;
struct rx_peer *peer;
{
    fprintf(file, "Peer %x.%d.  Burst size %d, burst wait %u.%d.\n", ntohl(peer->host), peer->port, peer->burstSize, peer->burstWait.sec, peer->burstWait.usec);
    fprintf(file, "   Rtt %d, retry time %u.%d, total sent %d, resent %d\n", peer->rtt, peer->timeout.sec, peer->timeout.usec, peer->nSent, peer->reSends);
    fprintf(file, "   Packet size %d, max in packet skew %d, max out packet skew %d\n", peer->packetSize, peer->inPacketSkew, peer->outPacketSkew);
}
#endif	RXDEBUG

osi_zone_t rx_serverQueueEntry_zone;
osi_zone_t rx_call_zone;
osi_zone_t rx_securityClass_zone;
osi_zone_t rx_service_zone;
osi_zone_t rx_peer_zone;
osi_zone_t rx_connection_zone;
osi_zone_t rx_event_zone;
osi_zone_t rx_packet_zone;

rx_zone_init()
{
    rx_serverQueueEntry_zone	= osi_Zinit(sizeof(struct rx_serverQueueEntry),
					    "rx serverQueueEntry");

    rx_call_zone		= osi_Zinit(sizeof(struct rx_call),
					    "rx call");

    rx_securityClass_zone	= osi_Zinit(sizeof(struct rx_securityClass),
					    "rx securityClass");

    rx_service_zone		= osi_Zinit(sizeof(struct rx_service),
					    "rx service");

    rx_peer_zone		= osi_Zinit(sizeof(struct rx_peer),
					    "rx peer");

    rx_connection_zone		= osi_Zinit(sizeof(struct rx_connection),
					    "rx connection");

    rx_event_zone		= osi_Zinit(sizeof(struct rxevent),
					    "rx event");

    rx_packet_zone		= osi_Zinit(sizeof(struct rx_packet),
					    "rx packet");
}
