/* 
 * Mach Operating System
 * Copyright (c) 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	rx_vab.c,v $
 * Revision 2.4  89/08/02  08:08:48  jsb
 * 	Replace osi_Allocs with osi_Zallocs.
 * 	[89/07/31  17:29:51  jsb]
 * 
 * Revision 2.3  89/06/03  15:40:50  jsb
 * 	Merged with newer ITC sources. Changes include an rxvab_DestroyConn
 * 	routine, which might fix some leakiness we think we have seen.
 * 	[89/05/26  22:17:42  jsb]
 * 
 * Revision 2.2  89/04/22  15:29:41  gm0w
 * 	Updated to RX version.
 * 	[89/04/14            gm0w]
 * 
 */

/*
****************************************************************************
*        Copyright IBM Corporation 1988, 1989 - All Rights Reserved        *
*                                                                          *
* Permission to use, copy, modify, and distribute this software and its    *
* documentation for any purpose and without fee is hereby granted,         *
* provided that the above copyright notice appear in all copies and        *
* that both that copyright notice and this permission notice appear in     *
* supporting documentation, and that the name of IBM not be used in        *
* advertising or publicity pertaining to distribution of the software      *
* without specific, written prior permission.                              *
*                                                                          *
* IBM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL *
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL IBM *
* BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY      *
* DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER  *
* IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING   *
* OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.    *
****************************************************************************
*/

#ifdef	KERNEL
#include <sys/types.h>
#include <netinet/in.h>
#include <rx/rx.h>
#include <rx/rx_kernel.h>
#include <rx/rx_vab.h>
#else	KERNEL
#include <sys/types.h>
#include <netinet/in.h>
#include "rx.h"
#include "rx_vab.h"
#endif	KERNEL

/* The rxvab security object.  Authentication using a bcrypt-encrypted VICE-style ticket. */

static int
    rxvab_Close(),
    rxvab_CheckAuthentication(),
    rxvab_CreateChallenge(),
    rxvab_GetChallenge(),
    rxvab_GetResponse(),
    rxvab_CheckResponse(),
    rxvab_DestroyConn();

extern int
    rxvab_CheckPacket(),
    rxvab_PreparePacket();


static struct rx_securityOps rxvab_ops = {
    rxvab_Close,
    0,			/* new connection */
    rxvab_PreparePacket,    /* once per packet creation */
    0,			/* send packet (once per packet retransmission) */
    rxvab_CheckAuthentication,
    rxvab_CreateChallenge,
    rxvab_GetChallenge,
    rxvab_GetResponse,
    rxvab_CheckResponse,
    rxvab_CheckPacket,	/* check data packet after duplicate elimination */
    rxvab_DestroyConn,	/* destroy conn */
    0,			/* spare 1 */
    0,			/* spare 2 */
    0,			/* spare 3 */
    0,			/* spare 4 */
};

static struct rx_securityOps rxvab_simpleOps = {
    rxvab_Close,
    0,			/* new connection */
    0,			/* prepare packet */
    0,			/* send packet (once per packet retransmission) */
    rxvab_CheckAuthentication,
    rxvab_CreateChallenge,
    rxvab_GetChallenge,
    rxvab_GetResponse,
    rxvab_CheckResponse,
    0,			/* CheckPacket */
    rxvab_DestroyConn,	/* destroy conn */
    0,			/* spare 1 */
    0,			/* spare 2 */
    0,			/* spare 3 */
    0,			/* spare 4 */
};

static rxvab_DestroyConn(aobj, aconn)
register struct rx_securityClass *aobj;
register struct rx_connection *aconn; {
    if (aconn->securityData) {
	osi_Zfree(rxvab_conn_zone, aconn->securityData);
	aconn->securityData = 0;
    }
    return 0;
}

/* called with one parameter, the server's encryption key */
struct rx_securityClass *rxvab_NewServerSecurityObject(akey, aencrypt)
int aencrypt;
register struct rxvab_EncryptionKey *akey; {
    register struct rx_securityClass *tsc;
    register struct rxvab_sprivate *tsp;
    tsc = (struct rx_securityClass *) osi_Zalloc(rx_securityClass_zone);
    tsc->refCount = 1;
    tsc->ops = (aencrypt? &rxvab_ops : &rxvab_simpleOps);
    tsp = (struct rxvab_sprivate *) osi_Zalloc(rxvab_sprivate_zone);
    tsc->privateData = (char *) tsp;
    /* compute key schedule, good for duration of key */
    bcopy(akey, &tsp->key, sizeof(struct rxvab_EncryptionKey));
    tsp->type =	2;	/* so can identify later */
    return tsc;
}

/* allocate a new client security object.  called with two parms, the session
    key and the ticket for the other side containing the encrypted
    session key from the auth server */
struct rx_securityClass *rxvab_NewClientSecurityObject(asession, aticket, aencrypt)
int aencrypt;
struct rxvab_Ticket *aticket;
struct rxvab_EncryptionKey *asession; {
    register struct rx_securityClass *tsc;
    register struct rxvab_cprivate *tcp;
    tsc = (struct rx_securityClass *) osi_Zalloc(rx_securityClass_zone);
    tcp = (struct rxvab_cprivate *) osi_Zalloc(rxvab_cprivate_zone);
    tsc->privateData = (char *) tcp;
    tcp->type = 1;
    bcopy(asession, &tcp->key, sizeof(struct rxvab_EncryptionKey));
    bcopy(aticket, &tcp->ticket, sizeof(struct rxvab_Ticket));
    tsc->refCount = 1;
    tsc->ops = (aencrypt? &rxvab_ops : &rxvab_simpleOps);
    return tsc;
}

/* called by rx with the security class object as a parameter when a security
    object is to be discarded */
static rxvab_Close(aobj)
register struct rx_securityClass *aobj; {
    register struct rxvab_cprivate *tcp;
    tcp = (struct rxvab_cprivate *)aobj->privateData;
    osi_Zfree(rx_securityClass_zone, aobj);
    if (tcp->type == 1) {
	/* client */
	osi_Zfree(rxvab_cprivate_zone, tcp);
    }
    else {
	/*server */
	osi_Zfree(rxvab_sprivate_zone, tcp);
    }
    return 0;
}

/* server: called to tell if a connection authenticated properly */
static rxvab_CheckAuthentication(aobj, aconn)
register struct rx_securityClass *aobj;
register struct rx_connection *aconn; {
    register struct rxvab_conn *tc;
    /* first make sure the object exists */
    if (!aconn->securityData) {
	/* <<<<when does this get freed?>>>> */
	aconn->securityData = (char *) osi_Zalloc(rxvab_conn_zone);
	bzero(aconn->securityData, sizeof(struct rxvab_conn));
    }
    tc = (struct rxvab_conn *) aconn->securityData;
    return !tc->worked;
}

static long challengeCounter = 2000;
/* server: put the current challenge in the connection structure for later use by packet sender */
static rxvab_CreateChallenge(aobj, aconn)
register struct rx_securityClass *aobj;
register struct rx_connection *aconn; {
    register struct rxvab_conn *tc;
    tc = (struct rxvab_conn *) aconn->securityData;
    tc->challengeID = challengeCounter++;
    return 0;
}

/* server: fill in a challenge in the packet */
static rxvab_GetChallenge(aobj, aconn, apacket)
register struct rx_securityClass *aobj;
register struct rx_packet *apacket;
register struct rx_connection *aconn; {
    register struct rxvab_conn *tc;
    long temp;

    tc = (struct rxvab_conn *) aconn->securityData;
    temp = htonl(tc->challengeID);
    bcopy(&temp, apacket->wire.data, sizeof(long));
    temp = htonl(0);
    bcopy(&temp, &apacket->wire.data[1], sizeof(long));
    rx_SetDataSize(apacket, 8);
    return 0;
}

/* client: respond to a challenge packet */
static rxvab_GetResponse(aobj, aconn, apacket)
struct rx_securityClass *aobj;
register struct rx_packet *apacket;
register struct rx_connection *aconn; {
    register struct rxvab_cprivate *tc;
    register char *tp;
    long a[2];

    tc = (struct rxvab_cprivate *) aobj->privateData;
    bcopy(apacket->wire.data, a, sizeof(a));	/* copy data out */
    if (ntohl(a[1]) != 0) return -1;	/* can't even figure out the question */
    a[0] = htonl(ntohl(a[0])+1);		/* perform challenge computation */
    tp = (char *) apacket->wire.data;
    bcopy(&tc->ticket, tp, sizeof(struct rxvab_Ticket));
    tp += sizeof(struct rxvab_Ticket);
    bcrypt_encrypt(a, a, sizeof(a), &tc->key);    /* encrypt it with session key */
    bcopy(a, tp, sizeof(a));
    apacket->length = sizeof(struct rxvab_Ticket) + 8;
    return 0;
}

/* server: process a response to a challenge packet */
static rxvab_CheckResponse(aobj, aconn, apacket)
struct rx_securityClass *aobj;
register struct rx_packet *apacket;
register struct rx_connection *aconn; {
    register struct rxvab_conn *tc;
    register struct rxvab_sprivate *ts;
    register char *tp;
    long a[2];	/* decryption buffer */
    struct rxvab_Ticket tix;

    tc = (struct rxvab_conn *) aconn->securityData;
    ts = (struct rxvab_sprivate *) aobj->privateData;
    tc->tried = 1;
    if (rx_GetDataSize(apacket) != 8+sizeof(struct rxvab_Ticket)) return -1;
    /* get the ticket from the packet */
    tp = (char *) apacket->wire.data;
    bcopy(tp, &tix, sizeof(tix));
    tp += sizeof(struct rxvab_Ticket);
    bcrypt_decrypt(&tix, &tix, sizeof(tix), &ts->key);	/* decrypt ticket with server key */
    bcopy(&tix.HandShakeKey, &tc->key, sizeof(struct rxvab_EncryptionKey));
    bcopy(tp, a, sizeof(a));
    bcrypt_decrypt(a, a, sizeof(a), &tc->key);    /* decrypt challenge with session key */
    if (ntohl(a[1]) !=	0) return -1;	/* failed to decrypt ticket */
    if (ntohl(a[0]) != tc->challengeID+1) return -1;    /* replay attempt */
    /* otherwise things are ok */
    tc->expirationTime = ntohl(tix.EndTimestamp);
    tc->viceID = ntohl(tix.ViceId);
    tc->worked = 1;
    return 0;
}

/* return useful authentication info about a server-side connection */
rxvab_GetServerInfo(aconn, awho, atime)
long *atime, *awho;
register struct rx_connection *aconn; {
    register struct rxvab_conn *tc;
    tc = (struct rxvab_conn *) aconn->securityData;
    if (!tc) return -1;
    if (!tc->tried || !tc->worked) return -1;
    if (atime) *atime = tc->expirationTime;
    if (awho) *awho = tc->viceID;
    return 0;
}

/* either: decode packet */
rxvab_CheckPacket(aobj, acall, apacket)
struct rx_securityClass *aobj;
struct rx_call *acall;
register struct rx_packet *apacket; {
    struct rx_connection *tconn;
    struct rxvab_EncryptionKey *tkey;

    tconn = acall->conn;
    if (rx_ServerConn(tconn)) {
	tkey = &((struct rxvab_conn *) (tconn->securityData))->key;
    }
    else {
	/* client connection */
	tkey = &((struct rxvab_cprivate *)(aobj->privateData))->key;
    }

    bcrypt_decrypt(apacket->wire.data, apacket->wire.data, apacket->length, tkey);
    return 0;
}

/* either: encode packet */
rxvab_PreparePacket(aobj, acall, apacket)
struct rx_securityClass *aobj;
struct rx_call *acall;
register struct rx_packet *apacket; {
    struct rx_connection *tconn;
    struct rxvab_EncryptionKey *tkey;

    tconn = acall->conn;
    if (rx_ServerConn(tconn)) {
	tkey = &((struct rxvab_conn *) (tconn->securityData))->key;
    }
    else {
	/* client connection */
	tkey = &((struct rxvab_cprivate *)(aobj->privateData))->key;
    }

    /* round packet size up to 8 byte boundary */
    apacket->length = ((apacket->length-1)|7)+1;
    
    bcrypt_encrypt(apacket->wire.data, apacket->wire.data, apacket->length, tkey);
    return 0;
}

/* server only: change server key */
rxvab_SetKey(aobj, akey)
struct rx_securityClass *aobj;
struct rxvab_EncryptionKey *akey; {
    register struct rxvab_sprivate *tp;
    tp = (struct rxvab_sprivate *) aobj->privateData;
    bcopy(akey, &tp->key, sizeof(struct rxvab_EncryptionKey));
    return 0;
}

#ifdef	MACH
zone_t rxvab_cprivate_zone;
zone_t rxvab_sprivate_zone;
zone_t rxvab_conn_zone;

rxvab_zone_init()
{
    rxvab_cprivate_zone		= zinit(sizeof(struct rxvab_cprivate),
					1024*1024, 0, FALSE,
					"rxvab cprivate");

    rxvab_sprivate_zone		= zinit(sizeof(struct rxvab_sprivate),
					1024*1024, 0, FALSE,
					"rxvab sprivate");

    rxvab_conn_zone		= zinit(sizeof(struct rxvab_conn),
					1024*1024, 0, FALSE,
					"rxvab conn");
}
#endif
