/* fixit -- a directory utility */

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <time.h>

#define BUFFSIZ 0x800 /* buffer size -- one page */
#define PATHSIZ 1000  /* maximum file name */

#define NLINE 2       /* lines output when files differ */
#define NBACK 80      /* how far to look back for beginning of line */

/* fixit codes for fix_files() */
#define CHECK_ONLY  0
#define FIX_FILES    1
#define MODE_ONLY    2
#define UPDATE_ONLY  3

extern char *alloca();

/* this and that for link_this(), diff_this() and diff_that() */
char this_path[PATHSIZ], that_path[PATHSIZ];
char *this_file, *that_file;

int uid, gid, gid_access; /* for secure_file() */

int extra, changed, missing; /* diff() results */

int error; /* global error flag, non-zero if errors */
int cksum; /* if non-zero, list_file() will do checksum's */

/* apply function to all files in directory */
/* this version does not gobble up file descriptors */
apply(path, function)
char *path;
int (*function)();
{
  DIR *dirp;
  struct direct *dp;
  struct stat s;
  int end,fix;
  struct list { char *d_name; struct list *next; } *list, *l;

  /* give up if file does not exist */
  if (lstat(path, &s)) return;
  /* give up if function returns non-zero */
  if ((*function)(path, &s)) return;
  /* give up if file is not a directory */
  if ((s.st_mode & S_IFMT) != S_IFDIR) return;
  /* save position in file path */
  fix = end = strlen(path);
  /* read directory contents into alloca() structure */
  /* this way, the directory is not open while we operate on its contents */
  if (!(dirp = opendir(path))) {
    printf("can't opendir(%s)\n", path);
    exit(1);
  }
  if ((fix > 0) && (path[fix - 1] != '/')) path[fix++] = '/';
  list = 0;
  for (dp = readdir(dirp); dp; dp = readdir(dirp)) {
    if (!strcmp(dp->d_name, ".") || !strcmp(dp->d_name, "..")) continue;
    if (!list) list = l = (struct list *)alloca(sizeof(struct list));
    else {
      l->next = (struct list *)alloca(sizeof(struct list));
      l = l->next;
    }
    l->next = 0;
    l->d_name = alloca(strlen(dp->d_name) + 1);
    strcpy(l->d_name, dp->d_name);
  }
  closedir(dirp);
  /* apply function to files in directory */
  for (l = list; l; l = l->next) {
    path[fix] = 0;
    apply(strcat(path, l->d_name), function);
  }
  path[end] = 0;
}

/* fast comparison of two files, return index of diff, -1 if none */
long diff_file(fileA, fileB)
char *fileA, *fileB;
{
  register long *A, *B, *C;
  int a, b;
  int n, nA, nB, nC;
  long buffA[BUFFSIZ], buffB[BUFFSIZ];
  long position;
  char *x, *y;
  int i;

  if ((a = open(fileA, O_RDONLY)) == -1) return 0;
  if ((b = open(fileB, O_RDONLY)) == -1) { close(a); return 0; }
  position = 0;
  while (1) {
    if ((nA = read(a, (char *)buffA, sizeof(buffA))) < 0) nA = 0;
    if ((nB = read(b, (char *)buffB, sizeof(buffB))) < 0) nB = 0;
    if (nA != nB) {
      if (nA < nB) ((char *)buffB)[nA] = ((char *)buffA)[nA] ^ 1;
      else ((char *)buffA)[nB] = ((char *)buffB)[nB] ^ 1;
    }
    else if (nA == 0) {
      close(a);
      close(b);
      return -1;
    }
    nC = (nA + nA % sizeof(*buffA)) / sizeof(*buffA);
    for (n = nA; n < nC * sizeof(*buffA); n++)
      ((char *)buffA)[n] = ((char *)buffB)[n] = 0;
    for (A = buffA, B = buffB, C = &buffA[nC]; A != C; )
      if (*(A++) != *(B++)) {
        close(a);
        close(b);
        x = (char *)(--A);
        y = (char *)(--B);
        position += x - (char *)buffA;
        for (i = 0; i < sizeof(*buffA); i++)
          if (*(x++) == *(y++)) position++;
          else break;
        return position;
      }
    position += nA;
  }
}

/* create "that" directory by symbolic links to "this" directory */
int link_this(path, s)
char *path;
struct stat *s;
{
  struct stat sbuf;
  char link[PATHSIZ];
  int i;

  *that_file = 0;
  strcpy(that_file, this_file);
  if (!lstat(that_path, &sbuf)) printf("EXISTS: %s\n", that_path);
  else switch (s->st_mode & S_IFMT) {
    case S_IFDIR:
      printf("directory: %s\n", that_path);
      if (mkdir(that_path, 0700)) {
        printf("can't mkdir(%s, 0700)\n", that_path);
        exit(1);
      }
      break;
    case S_IFREG:
      printf("file: %s\n", that_path);
      if (symlink(this_path, that_path)) {
        printf("can't symlink(%s, %s)\n", this_path, that_path);
        exit(1);
      }
      break;
    case S_IFLNK:
      printf("link: %s\n", that_path);
      if ((i = readlink(this_path, link, sizeof(link))) == -1) {
        printf("can't readlink(%s, *, *)\n", this_path);
        exit(1);
      }
      link[i] = 0;
      if (symlink(link, that_path)) {
        printf("can't symlink(%s, %s)\n", link, that_path);
        exit(1);
      }
      break;
    default:
      printf("IGNORE: %s\n", that_path);
      break;
  }
  fflush(stdout);
  return 0;
}

/* check for changed or extra files in "that" directory */
int diff_this(path, s)
char *path;
struct stat *s;
{
  struct stat sbuf;
  char link1[PATHSIZ], link2[PATHSIZ];
  int i;
  long position, p;
  FILE *f;
  int c;

  position = -1;
  *that_file = 0;
  strcpy(that_file, this_file);
  if (lstat(that_path, &sbuf)) {
    printf("extra: %s\n", this_file);
    fflush(stdout);
    extra++;
    return 1;
  }
  if ((sbuf.st_mode & S_IFMT) == (s->st_mode & S_IFMT))
      switch(sbuf.st_mode & S_IFMT) {
    case S_IFDIR:
    case S_IFSOCK:
      return 0;
    case S_IFCHR:
    case S_IFBLK:
      if (sbuf.st_dev == s->st_dev) return 0;
      break;
    case S_IFREG:
      if ((position = diff_file(this_path, that_path)) == -1) return 0;
      break;
    case S_IFLNK:
      if ((i = readlink(this_path, link1, sizeof(link1))) == -1) {
        printf("can't readlink(%s, *, *)\n", this_path);
        exit(1);
      }
      link1[i] = 0;
      if ((i = readlink(that_path, link2, sizeof(link2))) == -1) {
        printf("can't readlink(%s, *, *)\n", that_path);
        exit(1);
      }
      link2[i] = 0;
      if (!strcmp(link1, link2)) return 0;
      break;
  }
  printf("changed: %s\n", this_file);
  changed++;
  if (position >= 0) {
    if (!(f = fopen(this_path, "r"))) {
      printf("can't fopen(%s, *)\n", this_path);
      exit(1);
    }
    for (i = 0, p = position; (i < NBACK) && (p >= 0); i++, p--) {
      if (fseek(f, p, 0)) {
        printf("can't fseek(%s, %d, 0)\n", this_path, position);
      	exit(1);
      }
      if ((c = getc(f)) == '\n') break;
    }
    if ((c == '\n') || (p < 1)) for (i = 0; i < NLINE; i++) {
      char s[PATHSIZ];
      if (fgets(s, sizeof(s), f)) {
        fputs("  ", stdout);
        fputs(s, stdout);
      }
    }
    else printf("  Difference at position %d\n", position);
    fclose(f);
  }
  fflush(stdout);
  return 0;
}

/* check for missing files in "this" directory */
int diff_that(path, s)
char *path;
struct stat *s;
{
  struct stat sbuf;

  *this_file = 0;
  strcpy(this_file, that_file);
  if (lstat(this_path, &sbuf)) {
    printf("missing: %s\n", this_file);
    fflush(stdout);
    missing++;
    return 1;
  }
  return 0;
}

/* make file secure for global uid and gid (gid_access flag) */
int secure_file(path, s)
char *path;
struct stat *s;
{
  register int i;
  int mode = 0;
  int printed = 0;

  if ((s->st_mode & S_IFMT) == S_IFLNK) {
    for (i = strlen(path); path[i] != '/'; i--) path[i + 2] = path[i];
    path[i + 1] = '/';
    path[i + 2] = '/';
  }
  if ((s->st_uid != uid) || (s->st_gid != gid)) {
    printed++;
    printf("%s: set uid %d, gid %d", path, uid, gid);
    if (chown(path, uid, gid)) {
      perror(path);
      error++;
    }
  }
  if (gid_access) {
    /* remove group & others access, insure owner read/write */
    if ((s->st_mode & 07007) || ((s->st_mode & 00660) != 00660))
      mode = (s->st_mode & 07777 & ~07007) | 00660;
  }
  else {
    /* remove others access, insure owner and group read/write */
    if ((s->st_mode & 07077) || ((s->st_mode & 00600) != 00600))
      mode = (s->st_mode & 07777 & ~07077) | 00600;
  }
  if (mode) {
    if (!printed) printf("%s: ", path);
    else printf(", ");
    printf("set mode 0%o\n", mode);
    if (chmod(path, mode)) {
      perror(path);
      error++;
    }
  }
  else if (printed) printf("\n");
  return 0;
}

/* gather information about a file, do checksum if global cksum!=0 */
char *list_file(path, s)
char *path;
struct stat *s;
{
  int i, c;
  char link[PATHSIZ];
  unsigned sum;
  long nbytes;
  FILE *f;
  char dir[80];
  static char buffer[BUFFSIZ];

  switch(s->st_mode & S_IFMT) {
    case S_IFDIR:
      sprintf(dir, "d");
      break;
    case S_IFSOCK:
      sprintf(dir, "s");
      break;
    case S_IFCHR:
      sprintf(dir, "c %d %d", major(s->st_rdev), minor(s->st_rdev));
      break;
    case S_IFBLK:
      sprintf(dir, "b %d %d", major(s->st_rdev), minor(s->st_rdev));
      break;
    case S_IFREG:
      if (!cksum) sprintf(dir, "r %d", s->st_size);
      else {
        if (!(f = fopen(path, "r"))) {
          printf("can't fopen %s", path);
          exit(1);
        }
        sum = 0;
        nbytes = 0;
        while ((c = getc(f)) != EOF) {
          nbytes++;
          if (sum & 01) sum = (sum >> 1) + 0x8000;
          else sum >>= 1;
          sum += c;
          sum &= 0xFFFF;
        }
        if (ferror(f)) {
          printf("can't read %s\n", path);
          exit(1);
        }
        sprintf(dir, "r %05u %ld", sum, (nbytes+BUFSIZ-1)/BUFSIZ);
        fclose(f);
      }
      break;
    case S_IFLNK:
      if ((i = readlink(path, link, sizeof(link))) == -1) {
        printf("can't readlink(%s, *, *)\n", path);
        exit(1);
      }
      link[i] = 0;
      sprintf(dir, "l %s", link);
      break;
    default:
      printf("unexpected mode 0x%X for %s\n", s->st_mode & S_IFMT, path);
      exit(1);
  }
  sprintf(buffer, "%s 0%o %d %d %d %s",
    path, s->st_mode & 07777, s->st_uid, s->st_gid, s->st_mtime, dir);
  return buffer;
}

/* write list_file() to stdout to make file list */
int dir_file(path,s)
char *path;
struct stat *s;
{
  puts(list_file(path,s));
  return 0;
}

/* fix files specified by file list on stdin */
int fix_files(fixit)
int fixit;
{
  struct stat s;
  char *buffer1, buffer2[BUFFSIZ];
  char lpath[PATHSIZ], path[PATHSIZ];
  int mode1, mode2;
  int uid1, uid2;
  int gid1, gid2;
  time_t mtime1, mtime2;
  char type1, type2;
  char arg1a[BUFFSIZ], arg1b[BUFFSIZ], arg2a[BUFFSIZ], arg2b[BUFFSIZ];
  int nargs;
  time_t timep[2];
  char *slash;
  int dev_major, dev_minor;
  int line;

  /* loop through all lines of input */
  line = 0;
  while (gets(buffer2)) {
    line++;
    /* skip blank lines and comment lines */
    if (!*buffer2 || *buffer2 == '#') {
      /* copy blank lines and comments if -S */
      if (fixit == UPDATE_ONLY) puts(buffer2);
      continue;
    }
    /* parse the previous output of list_file() */
    nargs = sscanf(buffer2, "%s %o %d %d %d %c %s %s",
      path, &mode2, &uid2, &gid2, &mtime2, &type2, arg2a, arg2b);
    /* should be 6, 7 or 8 things parsed by sscanf() */
    if ((nargs < 6) || (nargs > 8)) {
      printf("line %d bad format: %s\n", line, buffer2);
      error++;
      continue;
    }
    /* path must be absolute */
    if (*path != '/') {
      printf("line %d relative path: %s\n", line, buffer2);
      error++;
      continue;
    }
    /* check type-specific format */
    switch (type2) {
      /* directory has no arguments */
      case 'd':
        if (nargs == 6) break;
        printf("line %d bad directory: %s\n", line, buffer2);
        error++;
        continue;
      /* socket has no arguments */
      case 's':
        if (nargs == 6) break;
        printf("line %d bad socket: %s\n", line, buffer2);
        error++;
        continue;
      /* character and block files have major/minor numbers */
      case 'c':
      case 'b':
        if (nargs == 8 &&
            sscanf(arg2a, "%d", &dev_major) == 1 &&
            sscanf(arg2b, "%d", &dev_minor) == 1)
          break;
        printf("line %d bad device: %s\n", line, buffer2);
        error++;
        continue;
      /* regular file has size, checksum or nothing */
      case 'r':
        if (nargs <= 8) break;
        printf("line %d bad file: %s\n", line, buffer2);
        error++;
        continue;
      /* link has path string */
      case 'l':
        if (nargs == 7) break;
        printf("line %d bad link: %s\n", line, buffer2);
        error++;
        continue;
      /* bad type code */
      default:
        printf("line %d bad type: %s\n", line, buffer2);
        error++;
        continue;
    }
    /* this is an undocumented way to chown/chmod a symbolic link */
    /* it seems to fail on AFS */
    strcpy(lpath, path);
    if (type2 == 'l')
      strcat(strcpy(rindex(lpath, '/'), "//"), rindex(path, '/'));
    /* if actual file does not exist, try to create it */
    if (lstat(path, &s)) {
      if (fixit == UPDATE_ONLY) {
      	printf("%s: missing\n", path);
        /* file is missing, go to next input line */
        error++;
        continue;
      }
      switch (type2) {
        case 'd':
          if (!fixit) printf("%s: missing directory\n", path);
          else if (mkdir(path, mode2)) perror(path);
          else if (chown(path, uid2, gid2)) perror(path);
          else printf("%s: fixed, directory created\n", path);
          break;
        case 's':
          continue; /* don't count missing sockets */
        case 'c':
        case 'b':
          if (!fixit) printf("%s: missing device\n", path);
          else if (mknod(path,
              mode2 | (type2 == 'c' ? S_IFCHR : S_IFBLK),
              makedev(dev_major, dev_minor)))
            perror(path);
          else if (chown(path, uid2, gid2)) perror(path);
          else printf("%s: fixed, %c device created\n", path, type2);
          break;
        case 'r':
          printf("%s: missing file\n", path);
          break;
        case 'l':
          if (!fixit) printf("%s: missing link\n", path);
          else if (symlink(arg2a, path)) perror(path);
          else if (chmod(lpath, mode2)) perror(lpath);
          else if (chown(lpath, uid2, gid2)) perror(lpath);
          else printf("%s: fixed, link created\n", path);
          break;
      }
      /* file is missing, go to next input line */
      error++;
      continue;
    }
    /* calculate checksum for regular file if specified by input line */
    /* note that MODE_ONLY runs much faster because it never checksum's */
    cksum = ((type2 == 'r') && (nargs > 7) && (fixit != MODE_ONLY));
    /* list actual information for existing file */
    buffer1 = list_file(path, &s);
    /* parse list_file() output, assume that the format is correct */
    sscanf(buffer1, "%*s %o %d %d %d %c %s %s",
       &mode1, &uid1, &gid1, &mtime1, &type1, arg1a, arg1b);
    /* check file type */
    if (type1 != type2) {
      error++; /* count bad file */
      printf("%s: type %c should be %c\n", path, type1, type2);
      /* bad file type, go to next input line */
      continue;
    }
    /* don't check or fix sockets, go to next input line */
    if (type2 == 's') {
      if (fixit == UPDATE_ONLY) puts(buffer2);
      continue;
    }
    /* check protection mode */
    if (mode1 != mode2) {
      error++; /* count bad file */
      if (!fixit || (fixit == UPDATE_ONLY))
        printf("%s: mode 0%o should be 0%o\n", path, mode1, mode2);
      else if (chmod(lpath, mode2)) perror(lpath);
      else printf("%s: fixed, set mode 0%o\n", path, mode2);
    }
    /* check uid and gid */
    if (uid1 != uid2 || gid1 != gid2) {
      error++; /* count bad file */
      if (!fixit || (fixit == UPDATE_ONLY)) {
        if (uid1 != uid2)
          printf("%s: uid %d should be %d\n", path, uid1, uid2);
        if (gid1 != gid2)
          printf("%s: gid %d should be %d\n", path, gid1, gid2);
      }
      else if (chown(lpath, uid2, gid2)) perror(lpath);
      else {
      	if (uid1 != uid2)
      	  printf("%s: fixed, set uid %d\n", path, uid2);
      	if (gid1 != gid2)
      	  printf("%s: fixed, set gid %d\n", path, gid2);
      }
    }
    /* don't do type-specific checks if mode-only */
    if (fixit == MODE_ONLY) continue;
    /* if update-cksum output file specs now, don't do type-specific checks */
    if (fixit == UPDATE_ONLY) {
      printf("%s 0%o %d %d %d %c",
        path, mode2, uid2, gid2, mtime2 ? mtime1 : 0, type2);
      if (nargs > 7) printf(" %s %s \n", arg1a, arg1b);
      else if (nargs > 6) printf(" %s\n", arg1a);
      else printf("\n");
      /* go to next input line -- don't do type-specific checks */
      continue;
    }
    /* type-specific checks */
    switch (type2) {
      case 'd':
        break;
      case 'c':
      case 'b':
        if (strcmp(arg1a, arg2a) || strcmp(arg1b, arg2b)) {
          error++; /* count bad file */
          if (!fixit) printf("%s: device %c %s %s should be %c %s %s\n",
            path, type1, arg1a, arg1b, type2, arg2a, arg2b);
          else {
            if (unlink(path)) perror(path);
            else {
              if (mknod(path,
                  mode2 | (type2 == 'c' ? S_IFCHR : S_IFBLK),
                  makedev(dev_major, dev_minor)))
                perror(path);
              if (chown(path, uid2, gid2)) perror(path);
              else printf("%s: fixed, set device %c %s %s\n",
                path, type2, arg2a, arg2b);
            }
          }
        }
        break;
      case 'r':
        /* don't check mtime below if no size/sum */
        if (nargs == 6) continue;
        if (cksum) {
          if (strcmp(arg1a, arg2a) || strcmp(arg1b, arg2b)) {
            error++; /* count bad file */
            printf("%s: sum %s %s should be %s %s\n",
              path, arg1a, arg1b, arg2a, arg2b);
            continue; /* don't check mtime below */
          }
        }
        else {
          if (strcmp(arg1a, arg2a)) {
            error++; /* count bad file */
            printf("%s: size %s should be %s\n", path, arg1a, arg2a);
            continue; /* don't check mtime below */
          }
        }
        break;
      case 'l':
        if (strcmp(arg1a, arg2a)) {
          if (fixit == 3) break; /* allow changed link if -S */
          error++; /* count bad file */
          if (!fixit) printf("%s: link %s should be %s\n",
            path, arg1a, arg2a);
          else if (unlink(path)) perror(path);
          else if (symlink(arg2a, path)) perror(path);
          else if (chmod(lpath, mode2)) perror(lpath);
          else if (chown(lpath, uid2, gid2)) perror(lpath);
          else printf("%s: fixed, set link to %s\n", path, arg2a);
        }
        break;
    }
    /* check modification times */
    if (mtime2 && (mtime1 != mtime2)) {
      error++; /* count bad file */
      if (!fixit) printf("%s: mtime %d should be %d\n",
        path, mtime1, mtime2);
      else if (lstat(path, &s)) perror(path);
      else {
        timep[0] = s.st_atime;
        timep[1] = mtime2;
        if (utime(lpath, timep)) perror(lpath);
        else printf("%s: fixed, set mtime %d\n", path, mtime2);
      }
    }
  }
}

/* extract names from file list */
extract_names(root)
char *root;
{
  char buffer[BUFFSIZ], path[PATHSIZ];

  while (gets(buffer)) {
    if (!*buffer || *buffer == '#') continue;
    sscanf(buffer, "%s %*o %*d %*d %*d %*c %*s %*s", path);
    printf("%s%s\n", root, path);
  }
}

/* re-root file list -- prepend string to paths */
reroot_files(root)
char *root;
{
  char buffer[BUFFSIZ], path[PATHSIZ];

  while (gets(buffer)) {
    if (!*buffer || *buffer == '#') printf("%s\n", buffer);
    else printf("%s%s\n", root, buffer);
  }
}

/* compare directories, count extra, changed and missing files */
diff(this, that)
char *this, *that;
{
  printf("comparing: %s\n", this);
  printf("against: %s\n", that);
  fflush(stdout);
  strcpy(this_path, this);
  strcpy(that_path, that);
  this_file = &this_path[strlen(this_path)];
  that_file = &that_path[strlen(that_path)];
  extra = 0;
  changed = 0;
  *this_file = 0;
  apply(this_path, diff_this);
  missing = 0;
  *that_file = 0;
  apply(that_path, diff_that);
  printf("summary: %d changed, %d extra, %d missing\n",
    changed, extra, missing);
}

/* secure directory for user */
secure(usr, grp, dir)
char *usr, *grp, *dir;
{
  struct passwd *p;
  struct group *g;

  if (!(p = getpwnam(usr))) {
    printf("%s: unknown user\n", usr);
    exit(1);
  }
  uid = p->pw_uid;
  gid = p->pw_gid;
  if (*grp) {
    if (!(g = getgrnam(grp))) {
      printf("%s: unknown group\n", grp);
      exit(1);
    }
    gid = g->gr_gid;
    gid_access = 1;
  }
  else gid_access = 0;
  printf("making %s secure for user %s, uid %d, gid %d\n",
    dir, usr, uid, gid);
  error = 0;
  apply(strcpy(this_path, dir), secure_file);
}

/* duplicate directory using symbolic links for files */
ln(this, that)
char *this, *that;
{
  printf("linking: %s\n", this);
  printf("to create: %s\n", that);
  fflush(stdout);
  strcpy(this_path, this);
  strcpy(that_path, that);
  this_file= &this_path[strlen(this_path)];
  that_file= &that_path[strlen(that_path)];
  *this_file = 0;
  apply(this_path, link_this);
}

main(argc, argv)
int argc;
char **argv;
{

  /* send all output to stdout */
  dup2(1, 2);

  /* check files -- the default operation when no arguments are given */
  if (argc <= 1) {
    if (isatty(fileno(stdin))) goto usage;
    error = 0;
    fix_files(CHECK_ONLY);
    exit(error ? 1 : 0);
  }

  /* fix files */
  if (!strcmp(argv[1], "-fix")) {
    if (argc > 2 || isatty(fileno(stdin))) goto usage;
    error = 0;
    fix_files(FIX_FILES);
    exit(error ? 1 : 0);
  }

  /* fix files -- mode, uid, gid only */
  if (!strcmp(argv[1], "-mode")) {
    if (argc > 2 || isatty(fileno(stdin))) goto usage;
    error = 0;
    fix_files(MODE_ONLY);
    exit(error ? 1 : 0);
  }

  /* update mtime and type-specific arguments */
  if (!strcmp(argv[1], "-update")) {
    if (argc > 2 || isatty(fileno(stdin))) goto usage;
    error = 0;
    fix_files(UPDATE_ONLY);
    exit(error ? 1 : 0);
  }

  /* extract file names */
  if (!strcmp(argv[1], "-name")) {
    if (argc > 3 || isatty(fileno(stdin))) goto usage;
    if (argc == 3) extract_names(argv[2]);
    else extract_names("");
    exit(0);
  }

  /* re-root file list -- prepend string to paths */
  if (!strcmp(argv[1], "-root")) {
    if (argc != 3 || isatty(fileno(stdin))) goto usage;
    if (*argv[2] != '/') goto usage;
    reroot_files(argv[2]);
    exit(0);
  }

  /* secure directory for user */
  if (!strcmp(argv[1], "-user")) {
    if (argc == 6) {
      if (strcmp(argv[3], "-group")) goto usage;
      if (argv[2][0] == '/') goto usage;
      if (argv[4][0] == '/') goto usage;
      if (argv[5][0] != '/') goto usage;
    }
    else if (argc == 4) {
      if (argv[2][0] == '/') goto usage;
      if (argv[3][0] != '/') goto usage;
    }
    else goto usage;
    secure(argv[2], argc == 6 ? argv[4] : "", argv[argc == 6 ? 5 : 3]);
    exit(error ? 1 : 0);
  }

  /* duplicate directory with symbolic links */
  if (!strcmp(argv[1], "-ln")) {
    if (argc != 4) goto usage;
    if (argv[2][0] != '/' || argv[3][0] != '/') goto usage;
    ln(argv[2], argv[3]);
    exit(0);
  }

  /* compare directories */
  if (!strcmp(argv[1], "-diff")) {
    if (argc != 4) goto usage;
    if (argv[2][0] != '/' || argv[3][0] != '/') goto usage;
    diff(argv[2], argv[3]);
    exit((extra || changed || missing) ? 1 : 0);
  }

  /* list all files -- the default operation, directory is argument */
  if (argc == 2) cksum = 0;
  else if (argc == 3 && ! strcmp(argv[1], "-sum")) cksum = 1;
  else goto usage;
  if (argv[cksum + 1][0] != '/') goto usage;
  error = 0;
  apply(strcpy(this_path, argv[cksum + 1]), dir_file);
  exit(error ? 1 : 0);

  /* display usage instructions */
usage:
  printf("usage:\n");
  printf("  fixit [-sum] /dir >list                  -- list files\n");
  printf("  fixit <list >errs                        -- check files\n");
  printf("  fixit -fix <list >errs                   -- fix files\n");
  printf("  fixit -mode <list >errs                  -- fix modes only\n");
  printf("  fixit -update <list >list                -- update checksums\n");
  printf("  fixit -name [root] <list >names          -- extract names\n");
  printf("  fixit -root /root <list >list            -- re-root file list\n");
  printf("  fixit -user usr [ -group grp ] /dir >log -- secure directory\n");
  printf("  fixit -ln /from /to >log                 -- symbolic link files\n");
  printf("  fixit -diff /this /that >diffs           -- compare directories\n");
  printf("file listing format:\n");
  printf("  <path> <octal mode> <uid> <gid> <mtime> <type specific>\n");
  printf("type specific:\n");
  printf("  d                 -- directory\n");
  printf("  s                 -- socket\n");
  printf("  c <major> <minor> -- char device\n");
  printf("  b <major> <minor> -- block device\n");
  printf("  r <size>          -- regular file without -s option\n");
  printf("  r <sum>           -- regular file with -s (see sum(1))\n");
  printf("  l <link>          -- symbolic link\n");
  printf("note: if mtime==0, mtime is not checked\n");
  printf("note: if r <null>, size/sum is not checked\n");
  exit(1);
}
