/******************************************************************************
* CLOCK.C - OEM Timer Driver					Version 1.3.00
*******************************************************************************
* Version   Date   Who  Description
* ======= ======== ===  =======================================================
* v1.3.00 02/17/87 AM	Schedule only one clock asr to satisfy one event.
*  1.2.03 08/19/86 ldt	tikcnt now static.
*  1.2.02 06/09/86 mei  High C port.
* v1.2.01 05/19/86 FRH	Close window in a_delay allowing negative ticks to be
*			scheduled
* v1.2.00 04/21/86 FRH	Creation
*******************************************************************************
*
*	This file contains the routines required to manipulate the hardware
*	clock for the IBM PC/AT, the Motorola MC146818 RT/CMOS RAM Chip.
*
*	Allow RT/CMOS to maintain MONTH, DAY, YEAR.
*	Time of Day maintained by Adding Ticks
*	Normalized every 24 hours ( Set alarm for midnight on 24 hour clock)
*
******************************************************************************/

#include "portab.h"
#include "sysbuild.h"
#include "struct.h"
#include "system.h"
#include "baspag.h"

#if METAWARE 
#define CLOCK
#include "protos.h"
#endif


#define	ASMTIKISR	1 	/* 1 = TIK_ISR in ASM, 0 = local	*/
#define	ASMTIKASR	1 	/* 1 = TIK_ASR in ASM, 0 = local	*/
#define	ASMRTC		1 	/* 1 = RTC Tools in ASM, 0 = local	*/

/******************************************************************************
*	TIMEDATE Table Structure and other Definitions
******************************************************************************/

#define TIMEDATE	struct _timedate

TIMEDATE
{
	UWORD	td_year;
	BYTE	td_month;
	BYTE	td_day;
	LONG	td_time;
	UWORD	td_timezone;
	BYTE	td_weekday;
	BYTE	td_reserved;
};

#define	ONE_DAY		86400000L	/* # milliseconds in 1 day */
#define	TICKPRIOR	150		/* tick asr priority */

		/* RT/CMOS RAM Locations and Bits */

#define	RT_SECOND	0x0
#define RT_SECALARM	0x1
#define RT_MINUTE	0x2
#define RT_MINALARM	0x3
#define RT_HOUR		0x4
#define RT_HRALARM	0x5
#define RT_WEEKDAY	0x6
#define RT_DAY		0x7
#define RT_MONTH	0x8
#define RT_YEAR		0x9
#define RT_AREG		0xa

#define RTA_UPDATE	0x80

#define RTA_16HZ	0x2c	/* 62.5 ms */
#define RTA_32HZ	0x2b	/* 31.25 ms */
#define RTA_64HZ	0x2a	/* 15.625 ms */
#define RTA_128HZ	0x29	/* 7.8125 ms */

#define RT_BREG		0xb

#define RTB_DSE		0x01	/* Daylight Savings Enabled */
#define RTB_24HR	0x02	/* 24 Hour Mode */
#define RTB_DM		0x04	/* Date Mode, 1=binary, 0=BCD */
#define RTB_SQWE	0x08	/* Square Wave Enabled */
#define RTB_UIE		0x10	/* Update-Ended Interrupt Enabled */
#define RTB_AIE		0x20	/* Alarm Interrupt Enabled */
#define RTB_PIE		0x40	/* Periodic Interrupt Enabled */
#define RTB_SET		0x80	/* Set Clock, Disable Update */

#define RT_CREG		0xc

#define RTC_UF		0x10	/* Update Interrupt Occured */
#define RTC_AF		0x20	/* Alarm Interrupt Occured */
#define RTC_PF		0x40	/* Periodic Interrupt Occured */
#define RTC_IRQF	0x80	/* Interrupt Request Flag */

#define RT_DREG		0xd

#define	RTD_VRB		0x80	/* Valid Ram Bit */


	/* Tick Granularity
	 *
	 *		if ((tikcnt & CORR_MASK) < CORR_VAL)
	 *		    tick = ELAPSED + 1;
	 *		else
	 *		    tick = ELAPSED;
	 */

#define TICK16		0
#define TICK32		1

#if TICK16	/* 64 HZ - 15.625 msec each */
#define ELAPSED		15L		/* Normally 15 milliseconds*/
#define CORR_MASK	7		/* 	every 8 ticks	*/
#define CORR_VAL	5		/*	increment 5	*/
#endif

#if TICK32	/* 32 HZ - 31.25 msec each */
#define ELAPSED		31L		/* Every 31 milliseconds*/
#define CORR_MASK	3		/* 	every 4 ticks	*/
#define CORR_VAL	1		/*	increment 1	*/
#endif

	/* Programmable Interrupt Controller */
	/*  Two INTEL 8259A Chips */

#define	PIC_M0		0x20		/* MASTER PIC */
#define	PIC_M1		0x21
#define	ICW1_M		0x11		/* ICW1 for master */
#define	ICW2_M		0x68		/* start of master interrupt numbers */
#define	ICW3_M		0x04
#define	ICW4_M		0x01

#define	PIC_S0		0xa0		/* SLAVE PIC */
#define	PIC_S1		0xa1
#define	ICW1_S		0x11
#define	ICW2_S		0x70		/* slave int. numbers start here */
#define	ICW3_S		0x02
#define	ICW4_S		0x01

#define	NSEOI		0x20		/* Non-specific End-of-Interrupt */

/******************************************************************************
*	External Functions
******************************************************************************/

EXTERN	EVB	*asetup();
EXTERN	VOID	_evdone();
EXTERN	LONG	_adummy();
EXTERN 	VOID	setvec(), okdisp(), int_enable();
EXTERN	WORD	no_isr(), ok_isr();

#if ASMTIKASR
EXTERN	VOID	tik_asr();
#endif

#if ASMTIKISR
EXTERN	UWORD	tik_isr();
#else
EXTERN	VOID	outp();
EXTERN 	VOID	doasr();
#endif

#if ASMRTC
EXTERN	VOID	rtcwrt();
EXTERN	VOID	rtcwrt_bcd();
EXTERN	UWORD	rtcrd();
EXTERN	UWORD	rtcrd_bcd();
#else
EXTERN	BYTE	inp();
#endif

/******************************************************************************
*	External Data
******************************************************************************/

EXTERN	WORD	dlatyp;
EXTERN	EVB	*dlr;

/******************************************************************************
*	Local Data
******************************************************************************/

static	LONG	tikcnt;

LONG	msec;			/* number of milliseconds since midnight */
UWORD	timezone = 960;		/* local timezone PST = 16 hours from GMT*/
UWORD	didtick;

/******************************************************************************
*	Forward References
******************************************************************************/

#if ASMTIKASR == 0
VOID	tik_asr();
#endif

#if ASMTIKISR == 0
UWORD	tik_isr();
#endif

#if ASMRTC == 0
VOID	rtcwrt();
VOID	rtcwrt_bcd();
UWORD	rtcrd();
UWORD	rtcrd_bcd();
#endif

VOID	tget(), tset();

/******************************************************************************
*	TIK_OPEN
*
*	Initialize the timer and time of day when system is booted.
******************************************************************************/

VOID tik_open()
{
	tikcnt = 0;

	setvec(tik_isr,0x70);		/* set up tick vector (on timer 1) */

		/* setup tick on MC146818 RTC cmos chip */

	rtcwrt(RT_BREG,RTB_SET);		/* Disable Update */

	rtcwrt(RT_SECALARM,0);			/* Set Alarm at */
	rtcwrt(RT_MINALARM,0);			/*  Midnight */
	rtcwrt(RT_HRALARM,0);
#if TICK32
	rtcwrt(RT_AREG, RTA_32HZ); /* 32 Hz */
#endif
#if TICK16
	rtcwrt(RT_AREG, RTA_64HZ); /* 64 Hz */
#endif

		/* initialize msec since midnight */

	msec = ((rtcrd_bcd(RT_HOUR) * 3600L) +
		(rtcrd_bcd(RT_MINUTE) * 60) +
		 rtcrd_bcd(RT_SECOND)) * 1000L;

		/* Enable Update */

	rtcwrt(RT_BREG, RTB_PIE|RTB_AIE|RTB_24HR|RTB_DSE);

		/* enable tick on pic */

	int_enable(8);
}

#if ASMTIKISR == 0

MLOCAL	BOOLEAN	didtikasr;		/* AM - Scheduled asr flag	   */
MLOCAL	LONG	msecstodo;		/* AM - # of msecs since last tick */

/******************************************************************************
*	TIK_ISR
*
*	Clock Interrupt Handler.
******************************************************************************/

UWORD tik_isr()
{
pragma On(Asm);

    UWORD n;		/* number of milliseconds since last tick 	*/
    UWORD creg;

	creg = rtcrd(RT_CREG);		/* clear int - read reg C */

	outp(PIC_M0, NSEOI);	/* non-specific EOI */
	outp(PIC_S0, NSEOI);

	if (creg & RTC_PF)	/* Is it a tick? */
	{
	    if (((UWORD)(++tikcnt) & CORR_MASK) < CORR_VAL)
		n = ELAPSED + 1;
	    else
		n = ELAPSED;
	    msec += n;		/* Increment # milliseconds since midnight */
	}

	if (creg & RTC_AF)	/* Is it the midnight alarm? */
	{
	    msec = 0;
	    if (!(creg & RTC_PF))	/* If no tick, we're all done */
		return(FALSE);
	}

	didtick++;
	if (dlr)				/* anyone delaying? */
	{
	  if (didtikasr)
	    msecstodo += n;
	  else
	  {
	    if (dlr->e_parm <= n)		/* time to wake up? */
	    {
		didtikasr = TRUE;
		msecstodo = n;
		doasr(tik_asr,0L,0L,TICKPRIOR);	/* yes - let tik_asr do it */
		return(TRUE);
	    }
	    else
		dlr->e_parm -= n;		/* no - decrement its delay */
	  }
	}

	/* Returning TRUE forces a dispatch and allows ASR's to
	 * run if scheduled.  A TRUE return also causes processes to be
	 * recheduled, breaking up CPU bound processes.  If "time slice"
	 * isn't wanted, return FALSE unless doasr() has been called above.
	 */

	return(TRUE);	/* return(FALSE); */

pragma Off(Asm);
}

#endif		/* End of ASMTIKISR == 0 */

#if ASMTIKASR == 0

/*****************************************************************************
*	TIK_ASR(n)
*
*	Check all events waiting for a specific time and wake up
*	those that have completed.
******************************************************************************/

VOID tik_asr()
{
pragma On(Asm);
	UWORD	mask;
	LONG	n;

	mask = no_isr();
	didtikasr = FALSE;
	n = msecstodo;

	while (dlr && (dlr->e_parm <= n))
	{
	    n -= dlr->e_parm;
	    ok_isr(mask);
	    _evdone(dlr);
	    mask = no_isr();
	}

	if (dlr)
	    dlr->e_parm -= n;

	ok_isr(mask);

pragma Off(Asm);
}

#endif		/* End of ASMTIKASR == 0 */

/*****************************************************************************
*	DLACAN(p)
*
*	Routine called when cancelling a timer event.
******************************************************************************/

LONG dlacan(p)
REG EVB *p;
{
	if (p->e_link)
	    p->e_link->e_parm += (LONG)p->e_parm;
	return(0);
}

/*****************************************************************************
*	A_DELAY(c,aswi)
*
*	Wait C milliseconds, and run ASWI on completion.
******************************************************************************/

LONG a_delay(c,aswi)
REG LONG c;
    BYTE *aswi;
{
    REG EVB *p,*q,*e;
    LONG temp;
    UWORD oldint;

	if (c > 0)
	    c += ELAPSED;		/* 1 Tick Minimum */
	else
	    return(_adummy(aswi,0L));	/* No Delay Wanted */

	if (( e = asetup(aswi)) != NULLPTR)
	{
	    e->e_type = dlatyp;
	    didtick = 0;		
oncemore:
	    NODISP
	    temp = c;
	    q = (EVB *)((BYTE *)&dlr - elinkoff);
	    for (p = dlr; p; p = (q = p)->e_link)
	    {
		if (c <= p->e_parm)
		    break;
		c -= p->e_parm;
	    }
	    oldint = no_isr();
	    if (didtick)
	    {
		c = temp - (didtick * ELAPSED);
		didtick = 0;
		ok_isr(oldint);
		DISPON
		if (c > 0)
		    goto oncemore;	/* A tick occured and decremented */
					/* a cnt on the DLR, try again */

			/* OOPS, we spun the time away ... */
		e->e_return = 0;
		_evdone(e);
		goto a_dexit;
	    }
	    e->e_pred = q;
	    q->e_link = e;
	    e->e_parm = c;
	    e->e_link = p;
	    if (p)
	    {
		p->e_pred = e;
		p->e_parm -= c;
	    }
	    ok_isr(oldint);
a_dexit:
	    DISPON
	    return(e->e_mask);
	}
	else
	    return(EM_KERN | E_EMASK);
}

/***************************************************************************
*	TIMER
*
*	Wait 'c' milliseconds or until absolute time 'c' and run swi
*	on completion.  The 'abs' flag tells whether this is a relative
*	or absolute delay.
****************************************************************************/

LONG timer(c,swi,abs)
    LONG c;
    LONG swi;
    UWORD abs;
{
	if (abs)
	    if ((c -= msec) < 0)
		c += ONE_DAY;

	return( a_delay(c,(BYTE *)swi) );
}

/***************************************************************************
*	TGET
*
*	
****************************************************************************/

VOID tget(buf,size)
TIMEDATE *buf;
UWORD size;
{

		/* read current time and date from RTC chip */
	while (rtcrd(RT_AREG) & RTA_UPDATE);   /* wait if update in progress */

	switch(size)
	{
	default:
	case 11:buf->td_weekday = rtcrd(RT_WEEKDAY);
	case 10:buf->td_timezone = timezone;
	case 9:
	case 8:	buf->td_time = msec;
	case 7:
	case 6:
	case 5:
	case 4:	buf->td_day = rtcrd_bcd(RT_DAY);
	case 3:	buf->td_month = rtcrd_bcd(RT_MONTH);
	case 2:	buf->td_year = rtcrd_bcd(RT_YEAR) + 1900;
	case 1:
	case 0:	break;
	}
}

/***************************************************************************
*	TSET
*
*	Set the current Time and Date	
****************************************************************************/

VOID tset(buf,size)
    TIMEDATE *buf;
    UWORD size;
{
    LONG time;
    UWORD hour,minute,second;

		/* disable time/date update on MC146818 RTC cmos chip
		 * periodic int enable,24-hour clock,dst features enabled
		 */

	rtcwrt(RT_BREG, RTB_SET | RTB_AIE | RTB_PIE | RTB_24HR | RTB_DSE);

		/* write info into RTC chip */

	switch(size)
	{
	default:
	case 11:rtcwrt(RT_WEEKDAY,buf->td_weekday);
	case 10:timezone = buf->td_timezone;
	case 9:
	case 8: msec = buf->td_time;

		time = buf->td_time / 1000L;	/* msecs to seconds */
		hour = time / 3600L;		/* seconds to hours */
		time -= hour * 3600L;		/* remove hours */
		minute = time / 60L;		/* seconds to minutes */
		second = time % 60L;		/* remove minutes */
		rtcwrt_bcd(RT_HOUR,hour);
		rtcwrt_bcd(RT_MINUTE,minute);
		rtcwrt_bcd(RT_SECOND,second);
	case 7:
	case 6:
	case 5:
	case 4:	rtcwrt_bcd(RT_DAY,buf->td_day);
	case 3:	rtcwrt_bcd(RT_MONTH,buf->td_month);
	case 2:	rtcwrt_bcd(RT_YEAR,buf->td_year - 1900);
	case 1:
	case 0:	break;
	}



		/* re-enable time/date update on MC146818 RTC cmos chip */
		/* periodic int enable,24-hour clock,dst feature */

	rtcwrt(RT_BREG, RTB_PIE | RTB_AIE | RTB_24HR | RTB_DSE);
}


#if ASMRTC == FALSE

/******************************************************************************
*
*	CMOS / Real Time Clock I/O routines
*
******************************************************************************/

VOID rtcwrt(add,dat)
    WORD add,dat;
{
    UWORD mask;

	mask = no_isr();
	outp(0x70,add);
	outp(0x71,dat);
	ok_isr(mask);
}

VOID rtcwrt_bcd(add,dat)	/* as above, with binary to bcd conversion */
    UWORD add,dat;
{
    UWORD item;
    UWORD mask;

	item = (dat % 10) + ((dat / 10 ) << 4);
	mask = no_isr();
	outp(0x70,add);
	outp(0x71,item);
	ok_isr(mask);
}

UWORD rtcrd(add)
    UWORD add;
{
    UWORD item;	
    UWORD mask;

	mask = no_isr();
	outp(0x70,add);
	item = inp(0x71);
	ok_isr(mask);
	return(item);
}

UWORD rtcrd_bcd(add)		/* as above, with bcd to binary conversion */
    UWORD add;
{
    UWORD item;
    UWORD mask;	

	mask = no_isr();
	outp(0x70, add);
	item = inp(0x71);
	ok_isr(mask);
	return ((item&0xf)+(item>>4)*10);
}

#endif		/* End of ASMRTC == FALSE	*/
