/*======================================================================*
 *   Version 2.4        Serial PORT Driver				*
 *			Main Code Section				*
 *----------------------------------------------------------------------*
 * VERSION   DATE    TIME  BY   CHANGE/COMMENTS				*
 *----------------------------------------------------------------------*
 *    .00   6/08/84       DR-K  started					*
 *   1.00   3/19/85  8:12 DR-K  Added this intro to files		*
 *   1.01   3/26/85 11:20 DR-K	shortened port_tbl to spec, added	*
 *   1.02   3/27/85 14:46 DR-K	Add p_hinit,p_set,p_special(x13,x93)	*
 *   1.03   4/11/85 11:11 DR-K	p_special move dp_delim to dp_bufsiz	*
 *					  and dp_offset to dp_buffer	*
 *   1.04   5/15/85 15:14 DR-K	change BASE to use second serial port	*
 *   1.05   6/05/85 18:35 DR-K	start using IO.H, two units, ports[]	*
 *				 untangle dual interrupts		*
 *   1.06   6/12/85 14:01 DR-K	use DPBLK rather then DPB		*
 *   1.07   6/25/85 17:12 DR-K	fix p1_intrp for pin[1] typo		*
 *   1.08   7/30/85  9:29 DR-K	p_pin[u] passes LONGs not BYTEs		*
 *   1.09   8/05/85 16:44 DR-K	move sfree(pdev[u]) from flush to uninit*
 *   1.10   12/03/85      gat	Ports interrupt at 0x6b and 0x6c	*
 *   2.0    12/18/85	  DR-K	fix byte overwrite init of mode field	*
 *   2.1    03/14/86      RFW   added mouse support to select code  	*
 *   2.2     4/29/86	  DR-K	fix bug in mouse support pdev allocation*
 *   2.3    06/11/86      mei	High C port.				*
 *   2.4    09/18/86	  RFW	Changed PORT_TBL portab[MAXpUNITS] to 	*
 * 				set the default setup for each port     *
 *				using defines in IPDRV.H these are 	*
 *				PTx_TYPE,PTx_BAUD,PTx_MODE,and PTx_CTRL.*
 *======================================================================*/
/*	INCLUDES							*/

#include "portab.h"
#include "system.h"
#include "io.h"
#include "ipdrv.h"

#if METAWARE 
#define IPDRV
#include "protos.h"
#endif

/************************************************************************/
/*	PROGRAM CONTROLS						*/
/*									*/
#define POLLING  FALSE	/* set TRUE for nonworking kb_interrupts 	*/
			/* when interrupts work turn it to FALSE 	*/
#define	TXINTS	 FALSE	/* set FALSE for nonworking tx_intrps	 	*/
			/* when interrupts work turn this to TRUE	*/
/************************************************************************/

#define	 BASE0		0x2F8	/* Base Port address of Serial Board 2	*/
				/* which we use as unit 0		*/
#define	 BASE1		0x3F8	/* Base Port address of Serial Board 1	*/
				/* which we use as unit 1		*/
#define  DATAPORT	BASE+0	/* Serial data port 			*/
#define	 TIntrp_REG	BASE+4	/* Serial Transmitter interupt port	*/


#define  INP	inp
#define  OUTP   outp
EXTERN	OUTP();		/* for asm language port routines */
EXTERN	INP();
EXTERN	LONG	_bmove();

/************************************************************************/
/* forward referenced functions						*/
/************************************************************************/
GLOBAL	LONG
	p_init(),
	p_subdrvr(),
	p_uninit(),
	p_select(),
	p_flush(),
	p_read(),
	p_write(),
	p_get(),
	p_set(),
	p_special(),
	p_deQ();

GLOBAL	VOID
	p_enQ(),
	p_out();

VOID	IsendQc(), p_hinit(); 

#if POLLING
VOID 	p0_getchar();
VOID	p1_getchar();
WORD	p_stat0();
WORD	p_stat1();
EXTERN	LONG	pollevent();
#else
EXTERN  LONG	setvec();
#endif

/************************************************************************/
/* local initializations      						*/
/************************************************************************/
	 	
#define MAXpUNITS 2
#define FLAGVAL  8	/* sequential access only */

/************************************************************************/
/*      Driver header     						*/
/*  This header must be the first structure in the data segment		*/
/*   so, don't put any data generating code before this section		*/
/************************************************************************/
	 	
DH  p_dh =
{	DVR_PORT,MAXpUNITS,FLAGVAL,
	p_init,
	p_subdrvr,
	p_uninit,
	p_select,
	p_flush,
	p_read,
	p_write,
	p_get,
	p_set,
	p_special,
	0L,0L,0L,0L,0L
};

#define	XON	0x11
#define XOFF	0x13

GLOBAL	BYTE	xonflag	= 0x0FF	;/* bitwise 1 = output, 0=not to output */
GLOBAL	BYTE	QEmpty	= 0x0FF	;/* a 0 bit means there are queued chars*/
GLOBAL	BYTE	QFull	= 0	;/* a 1 bit means queue is full		*/
GLOBAL	BYTE	intrpREG = 0	;/* the latest value in Interrupt REG */
GLOBAL	LONG	RdpFlag[MAXpUNITS] ;
GLOBAL	LONG	WrpFlag[MAXpUNITS] ;
WORD	p0_intrp()		;/* forward ref the interupt routine */
WORD	p1_intrp()		;/* forward ref the interupt routine */

#define MAXOUTB 128		/* must always be a power of 2 */

#define	PHYSBLK	struct	PhysDevQBlock
PHYSBLK
{
	int	Qrear;
	int	Qfront;
	char	Q[MAXOUTB];
};

GLOBAL	PHYSBLK	*pdev[MAXpUNITS];

typedef VOID	(*PINPTR)();

PINPTR	p_pin[MAXpUNITS];

#define	PT_MODE		PTM_NOPAR + PTM_1STOP + PTM_LENGTH
#define	PT_CNTRL	PTC_TXENAB + PTC_RXENAB + PTC_DTR + PTC_RTS

/* an array of base port addresses for each unit. */
WORD	ports[MAXpUNITS] = { BASE0 , BASE1 };

#if  POLLING
/* list of status routines, one for each unit.  Neccessary because POLLEVENT */
/* doesn't let the routine have any arguments, so one routine per unit */
WORD	(*p_stat[MAXpUNITS])() = { p_stat0, p_stat1 };
#endif	/* if polling */

PORT_TBL  portab[MAXpUNITS] = {
		/*** Initialize all our default parameters here ***/
/* NOTE BAUD0 BAUD1 are defined in IPDRV.H */
/*  type ,state,baud    , mode    , cntrl   ,p1 */ 
{PT0_TYPE,   0 ,PT0_BAUD, PT0_MODE, PT0_CTRL, 0 },
{PT1_TYPE,   0 ,PT1_BAUD, PT1_MODE, PT1_CTRL, 0 }
} ;

WORD	  baudtab[] =
    {	BD50,	BD75,	BD110,	BD134,	BD150,	BD300,	BD600,	BD1200,
	BD1800,	BD2000,	BD2400,	BD3600,	BD4800,	BD7200,	BD9600,	BD19200
     };

/************************************************************************/
/*      p_init                						*/
/************************************************************************/
	 	
LONG	p_init(unitno)

	BYTE	unitno;
{
	p_hinit(unitno);	/* initialize the USART */
	OUTP(ports[unitno]+1,0);		/* no INTERRUPTS NOW**/

	RdpFlag[unitno] = FLAGGET();
	WrpFlag[unitno] = FLAGGET();
#if (POLLING == FALSE)
	if (unitno ==0) setvec(p0_intrp,0x6BL);
	if (unitno ==1) setvec(p1_intrp,0x6CL);
#endif
	return(  (LONG) DVR_PORT );
}

/************************************************************************/
/*      p_subdrvr                					*/
/*		   *** nothing to do here ***				*/
/************************************************************************/

LONG	p_subdrvr(pb)
	DPBLK	*pb;
{
	return(E_SUCCESS);
}

/************************************************************************/
/*      p_uninit                						*/
/************************************************************************/
	 	
LONG	p_uninit(unitno)

	BYTE	unitno;

{
	LONG	r;

	if ((r = FLAGREL( RdpFlag[unitno] ) ) != NULLPTR ) return(r);
	if ((r = FLAGREL( WrpFlag[unitno] ) ) != NULLPTR ) return(r);
	if ((r = sfree ( pdev[unitno] ) ) != NULLPTR ) return(r);

#if  (POLLING==FALSE)	
	if (unitno ==0) setvec(p0_intrp,0L);	/* turn off the reciever interrupt vector */
	if (unitno ==1) setvec(p1_intrp,0L);	/* turn off the reciever interrupt vector */
#endif
	return(E_SUCCESS);
}

/************************************************************************/
/*      p_select                						*/
/************************************************************************/
	 	
LONG	p_select(pb)

	CDSELECT	*pb ;

{
	BYTE	unit;

	unit = pb->unitno;

	if ( (p_pin[unit] = (PINPTR) pb->kbd_pin) != NULLPTR )  ;
					/* we were given no keyboard pin */
	 else if ( (p_pin[unit] = (PINPTR) pb->m_pin) != NULLPTR )  ;
				/* nor mouse pin address so use our routine */
	   else p_pin[unit] = (PINPTR) p_enQ ;

	pdev[unit] = (PHYSBLK *) salloc( (LONG)sizeof(PHYSBLK)) ;
	pdev[unit]->Qrear = pdev[unit]->Qfront = 0 ;

#if POLLING
	if (unit ==0) doasr(p0_getchar,0,0,200);
	if (unit ==1) doasr(p1_getchar,0,0,200);
#else
	OUTP(ports[unit]+1,0x01);	/* enable the interupt mask */
	if (unit ==0) OUTP( 0x21 , INP(0x21) & ~0x08 );
	if (unit ==1) OUTP( 0x21 , INP(0x21) & ~0x10 );
#endif
	return(E_SUCCESS);
}

/************************************************************************/
/*      p_flush                						*/
/************************************************************************/
	 	
LONG	p_flush(pb)

	DPBLK	*pb;
{

#if (POLLING==FALSE)
	BYTE	unit;

	unit = pb->dp_unitno;
	if (unit ==0) OUTP( 0x21, INP(0x21) | 0x08 );	/* disable the */
	if (unit ==1) OUTP( 0x21, INP(0x21) | 0x10 );	/* interupt mask */
#endif
	return( E_SUCCESS );
}

/**** THIS IS NOT NEEDED when there is a true serial/console/printer driver */
/****    so remove it someday, but not today **/
/************************************************************************/
/*      p_read                						*/
/*	read from the serial port.  return when bufsiz chars have been  */
/*	entered, or when a carriage return is encountered.  The carriage*/
/*	return should be placed in the buffer.				*/
/************************************************************************/
LONG p_read( pb )
	DPBLK	*pb ;
{
}

/************************************************************************/
/*	p_write								*/
/*	Adds this character into the unit's Queue, hopeful that the	*/
/*	transmitter interrupts will put the character out the port.	*/
/*									*/
/*	If this is the first character to be EnQueued then enable the	*/
/*	transmitter so that interrupts will happen			*/
/************************************************************************/
LONG	p_write(unit,ch)		
	WORD	unit,ch	;
{					
	PHYSBLK	*u ;
	int	r,f;	/* temp rear and front Queue pointers */

	u = pdev[unit] ;
	r = (u->Qrear +1) % MAXOUTB ;
	f = u->Qfront ;

	if ( f == r )	/* if the advanced rear equals front then Queue */
			/* is full and we should do something appropriate */
	{
/* here we need to do the FLAGEVENT, doASR(until !QFull nextASR), ret(emask */
	QFull |= (1 << unit) ;
	while ((QFull >> unit) &1) /***looks endless***/
		;	/* but what is that appropiate thing is on QFull */
	IsendQc(unit);	/* we could return(E_QFULL) or get/set/wait some */
			/* flag, but the a transmitter at 50baud might be */
			/* the reason which tx_intrp should signal event */
			/* or an XOFF stopped output and only XON at	*/
			/* px_intrp can signal event satisfied		*/
/*****here's what should be done*********/	
/**		if QFull				**/
/**		emask = FLAGEVENT(---,---,---,---)	**/
/**		FLAGSET( ERROR_CODE )			**/
/**		return( emask );			**/
	}

	u->Q[r] = ch;
	u->Qrear = r;	/* char must be stored before length update */

	IsendQc(unit);	/***blotz when tx_intrp is fixed put code here ***/
	return(0L);
}

VOID IsendQc(unit)
BYTE	unit;	/* unit number to send character */
{
	PHYSBLK *u ;

	u = pdev[unit] ;
	if ( u->Qfront==u->Qrear )
		QEmpty |= 1 << unit ;	/* set the empty bit */
	else
	{	QEmpty &= ~(1 << unit) ;  /* clear the empty bit */
		p_out(u->Q[(u->Qfront = ((u->Qfront+1) % MAXOUTB))],unit);
	}
}

VOID p_out(ch,unit)		/**change by moving whole thing to p_write **/
char ch;
BYTE unit ;
{

	while (!(xonflag & (1 << unit))) /* temp fix here for xoff condition */
		;	/** should be changed because no call to p_out is **/
			/*** made without xon true state ***/

	nodisp();	/***if interrupts then this line not needed ***/
	while (!(INP(ports[unit]+5) & 0x20))
		;
	OUTP(ports[unit],ch);

	okdisp();	/***if interrupts then this not needed ***/
}

/************************************************************************/
/*      p_get                						*/
/************************************************************************/
/*** someday we have to extend the table of values retrieved		***/
/*** check for an uninstalled unitno ERROR				***/
LONG	p_get(pb)

	DPBLK	*pb;

{
	BYTE	*buff;	/* ptr to users buffer */
	buff = pb->dp_buffer;
	if ( pb->dp_flags & DPF_UADDR )
		buff = (BYTE *) SADDR(buff);

	if ( (pb->dp_bufsiz >0) && (pb->dp_bufsiz <= sizeof(PORT_TBL)))
		_bmove( (BYTE*) &portab[pb->dp_unitno] ,buff,pb->dp_bufsiz);

	return(E_SUCCESS);
}

/************************************************************************/
/*      p_set                						*/
/************************************************************************/
/*** if a new value is being set into a USART/controller/etc. mode-of-op***/
/*** variable then the hardware needs to be re-initialized		***/
  		
LONG	p_set(pb)

	DPBLK	*pb;

{
	BYTE	*buff;

	buff = pb->dp_buffer;
	if ( pb->dp_flags & DPF_UADDR )
		buff = (BYTE *) SADDR(buff);
	if ( (pb->dp_bufsiz >0) && (pb->dp_bufsiz <= sizeof(PORT_TBL)))
		_bmove( buff, &portab[pb->dp_unitno] ,(WORD)pb->dp_bufsiz);

	/*** re-initialize the USART to the new table values ***/
	p_hinit( pb->dp_unitno );
	return(E_SUCCESS);
}

/************************************************************************/
/*      p_special              						*/
/************************************************************************/
	 	
LONG	p_special(pb)
	DPBLK	*pb;
{
	if ( (pb->dp_option & 0x7F) == 0x13 )
	  {
		pb->dp_buffer = (BYTE *)pb->dp_offset;
		pb->dp_bufsiz = pb->dp_delim;
	  }
	switch( pb->dp_option )
	 {
	 case 0x13 : p_get(pb); break;
	 case 0x93 : p_set(pb); break;
	 default : return(ED_PORT | E_IMPLEMENT);
	 }
	return(E_SUCCESS);
}

/************* sub-routines useful here *********************************/

VOID 	p_hinit(unit)	/* initialize USART to table values */
	BYTE	unit;
{
	WORD	baud;
	BYTE	mode,cntrl;
	PORT_TBL  *ptab;
	 
	ptab = &portab[unit];
	mode = ptab->pt_mode;
	cntrl = ptab->pt_control;
	baud = baudtab[ptab->pt_baud];
	OUTP(ports[unit]+3,0x80);
	OUTP(ports[unit]+1, (baud >> 8) & 0xFF) ;
	OUTP(ports[unit], baud & 0xFF) ;
	OUTP(ports[unit]+3,0) ;
	OUTP(ports[unit]+3, ( (mode & PTM_LENGTH)
			| ( (mode & (PTM_PARITY + PTM_12STOP) & ~4) >> 1 ) ) );
	OUTP(ports[unit]+4, ( ((cntrl & PTC_DTR) >>1)
			| ((cntrl & PTC_RTS) >> 4)
			| 0x08 ) );
}

LONG  p_deQ(unit)
	BYTE	unit;
{
     PHYSBLK	*pd ;

     pd = pdev[unit] ;
     while( (!(INP(ports[unit]+5) & 0x01)) || (pd->Qrear == pd->Qfront) )
		;
     if (pd->Qrear != pd->Qfront)
	return(pd->Q[(pd->Qfront = ((pd->Qfront+1) % MAXOUTB))]);
		/** change to & anding if mod code too long **/
     else
	return(INP(ports[unit]));	
}

VOID	p_enQ(ch,unit)
	LONG	ch;
	LONG	unit;
{
	PHYSBLK	*pd ;

	pd = pdev[unit] ;
	pd->Q[(pd->Qrear = ((pd->Qrear+1) % MAXOUTB))] = (BYTE) ch ;
		/** change to & anding if mod code too long **/
	if (pd->Qrear == pd->Qfront)	/* don't enq if Queue is full */
		pd->Qrear = ((pd->Qrear-1) % MAXOUTB );
}

WORD p0_intrp(e)
LONG	e;
{
	WORD	ch ;

	ch = INP(ports[0]);
	(*p_pin[0])((LONG)ch,0L);
#if POLLING
	aret(e);		/* take emask off the stack */
	p0_getchar();		/* startup another pollevent */
#else
	OUTP(0x20,0x20);	/* non-specific EOI */
#endif
	return(TRUE); 		/* do a dispatch */
}

WORD p1_intrp(e)
LONG	e;
{
	WORD	ch ;

	ch = INP(ports[1]);
	(*p_pin[1])((LONG)ch,1L);
#if POLLING
	aret(e);		/* take emask off the stack */
	p1_getchar();		/* startup another pollevent */
#else
	OUTP(0x20,0x20);	/* non-specific EOI */
#endif
	return(TRUE); 		/* do a dispatch */
}

#if POLLING
VOID p0_getchar()
{
	LONG	emask;

	emask = pollevent( p_stat0 , 0L );
	nextasr( emask, p0_intrp, emask, 0L, 200 ); 
}

VOID p1_getchar()
{
	LONG	emask;

	emask = pollevent( p_stat1 , 0L );
	nextasr( emask, p1_intrp, emask, 0L, 200 ); 
}

WORD p_stat0()
{
	return ( ( INP(BASE0+5) & 1 ) == 1 );
}

WORD p_stat1()
{
	return ( ( INP(BASE1+5) & 1 ) == 1 );
}


#endif	/*if POLLING*/

