/***************************************************************************
*     RNETDRV.C 						Version 1.03
****************************************************************************
* Copyright (c) 1987, Digital Research, Inc. All Rights Reserved. The Software
* Code contained in this listing is proprietary to Digital Research, Inc.,
* Monterey, California and is covered by U.S. and other copyright protection.
* Unauthorized copying, adaptation, distribution, use or display is prohibited
* and may be subject to civil and criminal penalties.  Disclosure to others is
* prohibited.  For the terms and conditions of software code use refer to the
* appropriate Digital Research License Agreement.
****************************************************************************
* Version   Date   Who  Description
* ======= ======== ===  ===================================================
*  1.03   02/29/88 ldt  Renamed modules from hsd*.* to rnet*.*
*  1.02   12/02/87 ldt  Conditionalized the use of 8 port structures.
*  1.01   871120   reb  Added Comment's and NOINT/OKINT code.
*  1.00   871015   reb  Originated
****************************************************************************/
/*
*
*	This is the main module for a high speed serial device driver.
*	Contained within this compiland are the driver header (the interface 
*	to the OS) and the top level logical routines that are called by
*	the OS.
*
*	   The hardware for this implementation is the ARNET Multiport Card
*	with 8 serial ports using the NS8250/16450 UART.  If the ARNET card 
*       is being used, then #define EIGHTPORTS to be TRUE in RNETDRV.H.
*
*
*	This compiland's obj must be the first in the link order, since
*	it contains the Driver Header, and the driver header "must" be
*       the first data item in the first data segment.
*
*/
/*      INCLUDES                                                        */

#include "portab.h"	/* portable coding conventions			*/
#include "io.h"		/* driver level structure definitions and flags	*/
#include "system.h"	/* system level structure definitions and flags	*/
#include "rnetdrv.h"	/* local macro definitions 			*/

/************************************************************************/
/* External Functions							*/
/************************************************************************/
EXTERN	LONG	fix_ds();	/* to set ds for ISR context		*/
				/* in hsdtool.a86			*/
EXTERN	WORD	outp();		/* port I/O routines from DRTL.L86  	*/
EXTERN	WORD	inp();
EXTERN  LONG	setvec();	/* Driver Service to set hardware 	*/
				/* Interrupt Vector.			*/

/************************************************************************/
/* forward referenced functions                                         */
/************************************************************************/

LONG	hsd_init();
LONG	hsd_uninit();
LONG	hsd_subdrvr();
LONG	hsd_select();
LONG	hsd_flush();
LONG	hsd_rd();
LONG	hsd_wrt();
LONG	hsd_get();
LONG	hsd_set();
LONG	hsd_special();
LONG	set_rflag();

WORD	hsd_int_service(void) ;

NEAR	VOID	enable_port(WORD unit, WORD ints) ;
NEAR	VOID	disable_port(WORD unit, WORD ints) ;
NEAR	VOID	port_init(WORD unit) ;
NEAR	VOID	hsd_enq(WORD ch, WORD unit) ;
NEAR	WORD	hsd_deq( BYTE unit ) ;
NEAR	WORD	asr_deq( BYTE unit ) ;

/************************************************************************/
/* Flow Control Routines						*/
/************************************************************************/

NEAR	WORD	stop_port(WORD unit) ;
NEAR	WORD	start_port(WORD unit) ;

/************************************************************************/
/* Local Macros and Defines						*/
/************************************************************************/


#define	MP8_STAT	(0x140+2)		/* MultiPort 8 status port	*/
#define	IRQ_MSK		(bases[local_unit].pic_msk) 
#define HSDINT		(bases[local_unit].vector_number)


/************************************************************************/
/*	Port offsets for I/O to the 8250 UART				*/
/************************************************************************/
#define	DATA		0	/* Serial data port 			*/
#define	IER		1	/* Interrupt Enable Register		*/
#define IIR		2	/* Interrupt Identification Register	*/
#define LCR		3	/* Line Control Rewgister		*/
#define MCR		4	/* Modem Control Register		*/
#define LSR		5	/* Line Status Register			*/
#define MSR		6	/* Modem Status Register		*/

/************************************************************************/
/*	Interrupt Enable Bits for the 8250 UART				*/
/************************************************************************/
#define TX_BIT		0x02	/* transmit interrupt enable bit	*/
#define RX_BIT		0x01	/* receive interrupt enable bit		*/

/* see hsddrv.h for other pertinent defines */
#if EIGHTPORTS
#define MAXUNIT  8	/* number of units that are available.		*/
#define FLAGVAL  0x00	/* no syncing is required			*/
#else
#define MAXUNIT  2	/* number of units that are available.		*/
#define FLAGVAL  0x20	/* no syncing is required			*/
#endif

#define NOINT   _inline(0xfa) ;
#define OKINT   _inline(0xfb) ;

/************************************************************************/
/*  		BEGINNING OF THE DRIVER DATA SEGMENT 			*/
/************************************************************************/


/************************************************************************/
/*      Driver header                                                   */
/*  This header must be the first structure in the data segment         */
/*   so, don't put any data generating code before this section         */
/*  This is the structure that the Operating system Supervisor uses to	*/
/* control the driver.  The initializers below are pointers used by the */
/* supervisor to begin execution of the driver code for selected 	*/
/* functions.								*/
/************************************************************************/

DH  hsd_dh =
{
        DVR_SER,MAXUNIT,FLAGVAL,
        hsd_init,
        hsd_subdrvr,
        hsd_uninit,
        hsd_select,
        hsd_flush,
        hsd_rd,
        hsd_wrt,
        hsd_get,
        hsd_set,
        hsd_special,
        0L,0L,0L,0L,0L
};


/************************************************************************/
/* The following data structures control the ports that are enabled	*/
/************************************************************************/
typedef struct bstrct {
	WORD	port ;		/* io address of the output port base	*/
	WORD	pic_msk ;	/* mask for interrupt enable on th PIC	*/
	WORD	vector_number ;	/* vector number associated with the PIC*/
				/* mask above.				*/
} BASE ;

#if EIGHTPORTS

BASE	bases[MAXUNIT] = {
	{ 0x100, 0x20, 0x6d },
	{ 0x108, 0x20, 0x6d },
	{ 0x110, 0x20, 0x6d },
	{ 0x118, 0x20, 0x6d },
	{ 0x120, 0x20, 0x6d },
	{ 0x128, 0x20, 0x6d },
	{ 0x130, 0x20, 0x6d },
	{ 0x138, 0x20, 0x6d }
} ;

#else

BASE	bases[MAXUNIT] = {
	{ 0x3f8, 0x10, 0x6c },
	{ 0x2f8, 0x08, 0x6b }
} ;

#endif

/************************************************************************/
/*		  Port Get/Set Structure				*/
/************************************************************************/

typedef struct PortInfoTable
{					/* offset: use			*/
	WORD	HSD_type ;		/* 0x00 type of port		*/
	WORD	HSD_state ;		/* 0x02 not used		*/
	BYTE	HSD_baud ;		/* 0x04 baud rate index		*/
	BYTE	HSD_mode ;		/* 0x05 serial mode of operation*/
	BYTE	HSD_control ;		/* 0x06 control information	*/
	BYTE	HSD_1fill ;		/* make it an even num of bytes	*/
} PORT_TBL ;

/************************************************************************/
/*  This is the initialized table used in the get and set flexos calls	*/
/************************************************************************/

/* type ,state,baud   , mode   , cntrl  ,p1 */ 

#if EIGHTPORTS

PORT_TBL  hsd_portab[MAXUNIT] = {
{HSD_TYPE,   0 ,HSD_BAUD, HSD_MODE, HSD_CTRL, 0 },
{HSD_TYPE,   0 ,HSD_BAUD, HSD_MODE, HSD_CTRL, 0 },
{HSD_TYPE,   0 ,HSD_BAUD, HSD_MODE, HSD_CTRL, 0 },
{HSD_TYPE,   0 ,HSD_BAUD, HSD_MODE, HSD_CTRL, 0 },
{HSD_TYPE,   0 ,HSD_BAUD, HSD_MODE, HSD_CTRL, 0 },
{HSD_TYPE,   0 ,HSD_BAUD, HSD_MODE, HSD_CTRL, 0 },
{HSD_TYPE,   0 ,HSD_BAUD, HSD_MODE, HSD_CTRL, 0 },
{HSD_TYPE,   0 ,HSD_BAUD, HSD_MODE, HSD_CTRL, 0 }
} ;

#else

PORT_TBL  hsd_portab[MAXUNIT] = {
{HSD_TYPE,   0 ,HSD_BAUD, HSD_MODE, HSD_CTRL, 0 },
{HSD_TYPE,   0 ,HSD_BAUD, HSD_MODE, HSD_CTRL, 0 }
} ;

#endif

/************************************************************************/
/* This is the translate table for baud rate indexes to NS8250 baud rate*/
/* generator codes.							*/
/************************************************************************/
WORD	hsd_baud_tab[] =
{
	BD50,	BD75,	BD110,	BD134,	BD150,	BD300,	BD600,	BD1200,
	BD1800,	BD2000,	BD2400,	BD3600,	BD4800,	BD7200,	BD9600,	BD19200
};

/************************************************************************/
/*	Please note the use of the ANSI C keyword volatile in the 	*/
/* structure defined below, this is used inorder to force the C compiler*/
/* to refetch the pointer values for each use rather than preserving 	*/
/* them in a register from one use to another.				*/
/* Since these values are changed by ISR code it is necessary that they */
/* be refetched each time they are compared.				*/
/*  This Structure is the basic port control structure and exists for	*/
/* the duration of the port.(i.e. from init to uninit calls.)		*/
/************************************************************************/

typedef	struct	PhysDevQBlock
{
	volatile	BYTE	rq[MAXOUTR];  /* the read queue.	*/
	volatile	BYTE	wq[MAXOUTW] ; /* the write queue.	*/
	volatile	WORD	rqrear;	/* the read get index		*/
	volatile	WORD	rqfront;/* the read put index		*/
	volatile	WORD	wqrear;	/* write get index		*/
	volatile	WORD	wqfront;/* write put index		*/
	WORD	state ;			/* the current stat of IER flgs	*/
	LONG	rpending_event ;	/* pending event flag for read	*/
	LONG	bpdaddr;		/* buffer PD address for mapu	*/
	LONG	fpdaddr;                /* PD address for flagset call	*/
	LONG	rflagno;		/* the read flag number		*/
	LONG	wflagno ;		/* The write flag number	*/
	WORD	port_number ;		/* the number of the port	*/
	DPB	pb;			/* current read parm block	*/
}  PHYSBLK ;

/************************************************************************/
/* This table of pointers is used to manage the dynamic allocation of 	*/
/* control structures on a per unit basis.				*/
/************************************************************************/

#if EIGHTPORTS

PHYSBLK	*ports_hsd[MAXUNIT] =
{
	(PHYSBLK *)0L,
	(PHYSBLK *)0L,
	(PHYSBLK *)0L,
	(PHYSBLK *)0L,
	(PHYSBLK *)0L,
	(PHYSBLK *)0L,
	(PHYSBLK *)0L,
	(PHYSBLK *)0L
} ;

#else

PHYSBLK	*ports_hsd[MAXUNIT] =
{	(PHYSBLK *)0L,
	(PHYSBLK *)0L
} ;

#endif

/************************************************************************/
/* This array of longs holds the previous interrupt vectors		*/
/* returned by the setvec driver service call.				*/
/************************************************************************/

#if EIGHTPORTS

MLOCAL	LONG	next_vector[MAXUNIT] = 
{
	0xffffffffL, 
	0xffffffffL, 
	0xffffffffL, 
	0xffffffffL, 
	0xffffffffL, 
	0xffffffffL, 
	0xffffffffL, 
	0xffffffffL 
} ;

#else

MLOCAL	LONG	next_vector[MAXUNIT] = 
{
	0xffffffffL, 
	0xffffffffL
} ;

#endif

/************************************************************************/
/* This array indicates to the ???_enq routines that a port unit	*/
/* has caused an interrupt but did not receive a byte in its output 	*/
/* transmit holding register.  Therefore, it must be given the next byte*/
/* without the benefit of the xmit int.					*/
/************************************************************************/
BYTE	port_stopped[MAXUNIT] ;

/************************************************************************/
/* This array coordinates the use of set_rflag/ASR and the		*/
/* hsd_int_service/ISR.							*/
/************************************************************************/
BYTE	asr_running[MAXUNIT] ;



/************************************************************************/
/*  		BEGINNING OF THE DRIVER CODE SEGMENT 			*/
/************************************************************************/

/************************************************************************/
/*      hsd_init                                                        */
/* This entry point is called once per unit controlled by the driver    */
/* and serves the purpose of allocating system resources such as	*/
/* system flags and buffers						*/
/************************************************************************/

LONG	hsd_init(unitno)
LONG	unitno; /* the first unit is unit 0, the second is unit 1       */
{
    WORD	local_unit ;
    PHYSBLK	*new_port_block ;

    if ((local_unit = unitno) < MAXUNIT)
    {
	if (next_vector[local_unit] == 0xffffffffL )
	{
	    next_vector[local_unit] = 
              setvec(hsd_int_service,(LONG) bases[local_unit].vector_number ) ;
	}

	if ( ports_hsd[local_unit] == 0L )
	{
		/* init the port and the flags for that port */

		new_port_block = (PHYSBLK *)salloc((LONG)sizeof(PHYSBLK)) ;
		_bfill((BYTE *)new_port_block,sizeof(PHYSBLK),0x00) ;
		if (new_port_block == 0 ) return (ED_SER | E_MEMORY) ;
		else ports_hsd[local_unit] = new_port_block ;

		(*ports_hsd[local_unit]).rflagno = FLAGGET() ;
		(*ports_hsd[local_unit]).rpending_event = 0 ;
		(*ports_hsd[local_unit]).wflagno = FLAGGET() ;
		(*ports_hsd[local_unit]).port_number = bases[local_unit].port ;
		port_stopped[local_unit] = 0 ;
		port_init(local_unit) ;
	}
	else
	{
		/* return an error to the system saying that this port	*/
		/* has been initialized already.			*/

	        return (ED_SER | E_UNIT) ;
	}
    }
    else
    {
        return (ED_SER | E_UNIT) ;
    }	
    return ( E_SUCCESS | DVR_SER ) ;
}

/************************************************************************/
/*      hsd_uninit                                                   	*/
/* This entry point is called only when the driver is being removed from*/
/* the system.  This code must free all of the resources                */
/* allocated during the init and select calls and also must prepare the	*/
/* subdrivers, if any, for disconnection.				*/
/************************************************************************/

LONG	hsd_uninit(unitno)
LONG	unitno;
{
    WORD	local_unit ;

    if ((local_unit = unitno) < MAXUNIT)
    {
	WORD	i ;
	LONG	j = 0L ;
			/* determine if this is the last unit */
	for (i = 0 ; i < MAXUNIT ; i ++) j += (LONG)ports_hsd[local_unit] ;

	if (j == 0L )	/* if this is the last unit then remove the vector */
	{
		setvec(next_vector, HSDINT) ;
		next_vector[local_unit] = 0xffffffffL ;
	}

	if ( ports_hsd[local_unit] != 0L )	/* is port initialized */
	{
		/* init the port and the flags for that port */

		FLAGREL((*ports_hsd[local_unit]).rflagno) ;
		FLAGREL((*ports_hsd[local_unit]).wflagno) ;

		ports_hsd[local_unit] = (PHYSBLK *)SFREE(ports_hsd[local_unit]) ;
	}
	else	/* port is not in the initialized state 		*/
	{
		/* return an error reporting that this port		*/
		/* has already been initialized.			*/
        	return (ED_SER | E_UNIT) ;
	}
    }
    else	/* port unit number is not valid for this driver	*/
    {
        return (ED_SER | E_UNIT) ;
    }	
    return ( E_SUCCESS ) ;
}

/************************************************************************/
/*      hsd_subdrvr                                                     */
/* If this driver uses a subdriver,this entry point establishes		*/
/* the connection between the current driver and its new subdriver.	*/
/************************************************************************/

LONG	hsd_subdrvr()
{
	return (ED_SER | E_IMPLEMENT) ;
}

/************************************************************************/
/*      hsd_select                                                      */
/* The Select entry point specifically prepares a particular		*/
/* hardware unit for I/O.   This routine should contain the code that 	*/
/* starts a hardware state machine(i.e. enable interrupts).		*/
/************************************************************************/

LONG	hsd_select(pb)
DPB	*pb ;           /* Driver Parameter Block                       */
{
	WORD	local_unit ;
	PHYSBLK	*local_phys_blk ;

	local_unit = (*pb).dp_unitno ;
	local_phys_blk = ports_hsd[local_unit] ;
	(*local_phys_blk).rqrear = (*local_phys_blk).rqfront = 0 ;
	enable_port(local_unit, (RX_BIT | TX_BIT) ) ;	
	return (E_SUCCESS) ;	
}

/************************************************************************/
/*      hsd_flush							*/
/* The flush routine either completes or aborts all			*/
/* hardware activities undertaken on a particular unit.  At the time of	*/
/* entry, all buffer pointers that have been passed in from the caller  */
/* are still valid.                                                     */
/* However once this routine exits, buffers may be released/unmapped at	*/
/* any time at the discretion of the OS or calling process.		*/
/************************************************************************/

LONG	   hsd_flush(pb)
DPB     *pb;
{
        LONG	   r;
	BYTE	local_unit ;
	PHYSBLK	*phys_blk ;

	local_unit = (*pb).dp_unitno ;

	phys_blk = ports_hsd[local_unit] ;

	r = 0L ;

	nodisp() ;

	if ( (*phys_blk).rpending_event ) 
	{			/* if there is an event awaiting	*/
		(*phys_blk).rpending_event = 0;
		FLAGSET( (*phys_blk).rflagno,(*phys_blk).fpdaddr,0L );	
							/* signal done	*/
	}
	okdisp() ;

	return( r );
}

/************************************************************************/
/*      hsd_rd - unbuffer up to bufsiz characters 			*/
/* This routine removes characters from a circular buffer maintained by	*/
/* the interrupt service routines, and places them into a user buffer	*/
/* to satisfy the user request.						*/
/************************************************************************/
LONG	hsd_rd( pb )
DPB	*pb ;
{
	BYTE		local_unit  ;	/* unit to be read		*/
	LONG		emask ;	/* event mask for the read		*/
	BYTE		*buff ;	/* ptr to users buffer			*/
	LONG		cnt ;	/* nbr of chars left to transmit	*/
	ERROR		r ;	/* holder for error return codes	*/
	PHYSBLK		*phys_blk ;	/* event data holder		*/
	LONG		nbytes = 0 ;	/* number of chars returned 	*/
        WORD		lbuff ;	/* local buffer for get loop use.	*/

	local_unit = (*pb).dp_unitno ;

	phys_blk = ports_hsd[local_unit];

	emask = FLAGEVENT( (*phys_blk).rflagno,(*pb).dp_swi );

	if ( (*pb).dp_flags & DPF_UADDR )
		buff = (BYTE *)SADDR((*pb).dp_buffer) ;
	else
		buff = (*pb).dp_buffer ;

	for( cnt = (*pb).dp_bufsiz ; cnt ; cnt-- )
	{
		if ((lbuff = hsd_deq(local_unit)) > 0xff ) break ;
                *buff++ = lbuff ;
		nbytes++ ;
	}

	if (nbytes)
	{
	    r = FLAGSET ( (*phys_blk).rflagno, *(hsd_dh.dh_curpd), nbytes );
	    (*phys_blk).rpending_event = 0 ;	/* clear the event	*/
	}
	else 
	{
		(*pb).dp_bufsiz = cnt ;
		(*pb).dp_offset = 0L;
		_bmove((BYTE *)pb,(BYTE *)&((*phys_blk).pb),sizeof(DPB));
		(*phys_blk).bpdaddr = (*pb).dp_pdaddr;
		(*phys_blk).fpdaddr = (LONG)(*(hsd_dh.dh_curpd));
		(*phys_blk).rpending_event = emask ;	/* set the event */
	}

	return(emask);
}

/************************************************************************/
/*      hsd_wrt                                                         */
/*      This routine places characters from the user buffer into the	*/
/* transmit buffer of the specific hardware unit.			*/
/************************************************************************/
LONG	hsd_wrt( pb )
DPB	*pb;
{
	WORD	cnt ;
	BYTE	*buff,local_unit ;
	LONG	evnum ;

	local_unit = (*pb).dp_unitno;


	if ( (*pb).dp_flags & DPF_UADDR )
		buff = (BYTE *)SADDR((*pb).dp_buffer) ;
	else
		buff = (*pb).dp_buffer ;

	for( cnt=(*pb).dp_bufsiz; cnt; buff++,cnt-- )
	{
		hsd_enq(*buff,local_unit) ;
	}

	evnum = FLAGEVENT( (*ports_hsd[local_unit]).wflagno,(*pb).dp_swi );
	FLAGSET( (*ports_hsd[local_unit]).wflagno,*(hsd_dh.dh_curpd),(*pb).dp_bufsiz );
	return( evnum );
}

/************************************************************************/
/*      hsd_get								*/
/*	This routine returns to the calling process a table		*/
/* of information about the unique hardware configuration that exists   */
/* for a specific unit.							*/
/************************************************************************/
	 	
LONG	hsd_get(pb)
DPB	*pb;
{
	BYTE	*buff;	/* ptr to users buffer */

	if ( (*pb).dp_flags & DPF_UADDR )
		buff = (BYTE *) SADDR((*pb).dp_buffer);
	else
		buff = (*pb).dp_buffer;

	if ( ((*pb).dp_bufsiz > 0L) && ((*pb).dp_bufsiz <= sizeof(PORT_TBL)))
		_bmove( (BYTE*) &hsd_portab[(*pb).dp_unitno] ,buff,(*pb).dp_bufsiz);

	return(E_SUCCESS);
}

/************************************************************************/
/*      hsd_set								*/
/* This routine sets the hardware/software configuration of a unit.	*/
/* Some of the fields in the table are not changeable by the calling	*/
/* process and are ignored.						*/
/************************************************************************/
	 	
LONG	hsd_set(pb)
DPB	*pb;
{
	PORT_TBL	*buff;
	WORD	bsize, local_unit ;

	local_unit = (*pb).dp_unitno ;

	if ( (*pb).dp_flags & DPF_UADDR )
		buff = (PORT_TBL *)SADDR((*pb).dp_buffer);
	else
		buff = (PORT_TBL *)(*pb).dp_buffer;

	bsize = (*pb).dp_bufsiz ;

	if ( (bsize > 0) && (bsize <= sizeof(PORT_TBL)))
	{
		switch (bsize)
		{
			/* only the settable fields are moved */
			case 8:
			case 7:
			    hsd_portab[local_unit].HSD_control = (*buff).HSD_control ;
			case 6:	
			    hsd_portab[local_unit].HSD_mode = (*buff).HSD_mode ;
			case 5:
			    hsd_portab[local_unit].HSD_baud = (*buff).HSD_baud ;
			case 4:
			case 3:
			case 2:
			    hsd_portab[local_unit].HSD_control = (*buff).HSD_control ;
			case 1:
			default: break ;
		}
	}

	/*** re-initialize the USART to the new table values ***/

	port_init( (*pb).dp_unitno );
	return(E_SUCCESS);
}

/************************************************************************/
/*      hsd_special                					*/
/* The only currently defined special is the subdriver get/set which    */
/* is simulated by this driver.						*/
/* The normal driver get and set routines are used.			*/
/************************************************************************/
	 	
LONG	hsd_special(pb)
DPB	*pb;
{
    if ( ((*pb).dp_option & 0x7F) == 0x13 )
    {
	(*pb).dp_buffer = (BYTE *)(*pb).dp_offset;
	(*pb).dp_bufsiz = (*pb).dp_parm7 ;
    }
    switch( (*pb).dp_option )
    {
	case 0x13 : hsd_get(pb); break;
	case 0x93 : hsd_set(pb); break;
	default : return(ED_SER | E_IMPLEMENT);
    }
    return(E_SUCCESS);
}

/************************* end of main routines *************************/


/************************************************************************/
/*	set_rflag							*/
/*  This is the flag setting routine for the read path.  If bytes were	*/
/* ready at the time of entry, they were returned and the flag was set.	*/
/* If no bytes were available, an event mask was returned and the DPB	*/
/* was saved in the PHYSBLK structure for this unit.			*/
/* This routine moves the bytes from the queue to the user buffer and   */
/* sets	the flag to indicate that the request is fulfilled.		*/
/************************************************************************/
LONG	set_rflag(phys_blk, unit)
PHYSBLK	*phys_blk ;
LONG	unit ;
{
	BYTE	*buff;
	LONG	cnt;
	LONG	nbytes = 0 ;
	DPB	*parm_blk;
	WORD	lbuff ;

	parm_blk = &((*phys_blk).pb) ;

	if ((*phys_blk).rpending_event)
	{
		(*phys_blk).rpending_event = 0;
	}
	else
	{
		return 0 ;	/* operation cancelled			*/
	}

	mapu((*phys_blk).bpdaddr);	/* get access to buffer memory	*/

	if ( (*parm_blk).dp_flags & DPF_UADDR )
	{
		buff = (BYTE *)saddr((*parm_blk).dp_buffer) ;
	}
	else
	{
		buff = (*parm_blk).dp_buffer ;
	}

	for( cnt=(*parm_blk).dp_bufsiz ; cnt ; cnt-- )
	{
		if ((lbuff = asr_deq((*parm_blk).dp_unitno)) > 0xff) break ;
		*buff++ = lbuff ;
		nbytes++ ;
	}

	unmapu();

	if (nbytes)
	{
		asr_running[unit] = 0 ;
		return(FLAGSET((*phys_blk).rflagno,(*phys_blk).fpdaddr,nbytes));
	}
	else
	{
		(*phys_blk).rpending_event = 1 ;
	}
	asr_running[unit] = 0 ;
}

/************************************************************************/
/* The Physical/Device Specific code begins at this point		*/
/************************************************************************/

/************************************************************************/
/*	hsd_int_service							*/
/* Since the MULTI_PORT 8  only uses a single IRQ line			*/
/* This implementation will use a mixed strategy of Interrupts and 	*/
/* polling.  The basic methodology is as follows: When the interrupt	*/
/* signifies that some channel on the MultiPort Card needs servicing,	*/
/* then service all channels possible during that ISR including those	*/
/* channels that become ready during the processing.			*/
/************************************************************************/
WORD	hsd_int_service()
{
	WORD	i ;
	LONG	old_ds ;
	WORD	retval = 0 ;

	/****************************************************************/
	/* This is the main interrupt service loop, which		*/
	/* goes thru all of the units that are currently enabled	*/
	/* for transmit and/or receive moving data to and from the	*/
	/* queues and user buffers as appropriate.			*/
	/****************************************************************/
	old_ds = fix_ds(0L) ;
        OKINT
	for ( i = 0 ; i < MAXUNIT ; i++ )
	{
		WORD	port, stat ;
		PHYSBLK	*phys_blk = ports_hsd[i] ;
		
		if (!phys_blk) continue ;	/* in case not inited	*/

		port = (*phys_blk).port_number ;
		if (!port) continue ;	/* This port is not in use.	*/

		while (  ( stat = (0x06 & inp(port+IIR))) != 0 )
		{	/* Get TX/RX ints.	*/

		switch (stat>>1)
		{
		    case	0:	/* Modem Status Change Interrupt*/
		    {
			inp(port+MSR) ;	/* read to reset the interrupt	*/
			continue ;
		    }

		    case	1:	/* Tx Register Empty Interrupt	*/
		    {
			/************************************************/
			/* This is the Transmit side of the service loop*/
			/* It removes bytes from the transmit Queue and	*/
			/* places them in the transmit holding register	*/
			/* of the hardware channel.  If the queue is	*/
			/* empty then no character is sent and the state*/
			/* machine for transmit halts until another 	*/
			/* character is placed in the Xmitter Holding   */
                        /* Register (THR).              		*/
			/* If there is data in the Queue.		*/
			/* The write to the TX Holding Register will 	*/
			/* clear the int, if it is not already cleared	*/
			/* by the IIR read above.			*/
			/************************************************/


			if ( (*phys_blk).wqfront != (*phys_blk).wqrear )
			{
				outp(port+DATA,
				(*phys_blk).wq[((*phys_blk).wqrear)] ) ;
				(*phys_blk).wqrear = 
				( ((*phys_blk).wqrear+1) % MAXOUTW );
			}
			else
			{
				port_stopped[i] = 1 ;
			}
			continue ;
		    }
		    case	2:	/* Received Data Avail Interrupt*/
		    {
			/************************************************/
			/* This is the receive side of the service loop.*/
			/* It removes bytes from the port and places	*/
			/* them in the receive Queue. If the Queue is	*/
			/* near being full the appropriate flow control */
			/* action should be taken to stop the other end	*/
			/* of the link, if possible.			*/
			/************************************************/

			WORD	ch ;

			ch = inp(port+DATA) ;
			(*phys_blk).rq[((*phys_blk).rqfront)] = ch ;
			(*phys_blk).rqfront = 
				( ((*phys_blk).rqfront+1) % MAXOUTR) ;

			if ((*phys_blk).rpending_event && !(asr_running[i]) )
			{
				/****************************************/
				/* NOTE:				*/
				/* At this point the driver is required	*/
				/* to go asynchronous because the ISR 	*/
				/* is restricted(by design) from making	*/
				/* certain system calls that can lead 	*/
				/* to non-deterministic behavour.	*/
				/****************************************/

				asr_running[i] = 1 ;

				doasr(set_rflag, phys_blk, (LONG) i, 100) ;
				retval = 1 ;
			}

			if ((*phys_blk).rqrear == (*phys_blk).rqfront)
			{
			    /* Need to do flow control here.		*/
				stop_port(i) ;
			}
			continue ;
		    }

		    case	3:	/* Receiver Line Stat Interrupt	*/
		    			/* Overrun,Framing,Parity,break */
					/* interrupts.			*/
			inp(port+LSR) ;
			continue ;		    
		    default:
		    	break ;
		}
		    break ;
		}
	}

	outp(0x20,0x20);	/* non-specific EOI */
	fix_ds(old_ds) ;
			/************************************************/
			/* The return code from this ISR can cause an 	*/
			/* immediate dispatch.  If the return code is	*/
			/* zero, then an immediate dispatch is not	*/
			/* performed.					*/
			/*  If however the ASR routine is needed to 	*/
			/* comply with critical and/or time-sensitive	*/
			/* protocol operations, then a return code of 	*/
			/* non-zero will cause a dispatch to be done	*/
			/* and all ASR's will be run in priority order.	*/
			/************************************************/
	return (retval) ;
}

/************************************************************************/
/*	enable_port							*/
/*   This routine is called by the Select entry point and is used	*/
/* to enable the chip-level interrupt masks.  It also ensures that	*/
/* that the board level interrupt is also enabled.			*/
/************************************************************************/
VOID	enable_port(local_unit, ints)
	WORD	local_unit, ints ;
{
	WORD	port = (*ports_hsd[local_unit]).port_number ;
	(*ports_hsd[local_unit]).state |= ints ;
	outp(port+IER,(*ports_hsd[local_unit]).state);/* enable the interupt mask */
       	outp(0x21, (inp(0x21) & ~IRQ_MSK)) ;
}

/************************************************************************/
/*	disable_port							*/
/*   This routine is called by the Flush entry point and is used	*/
/* to disable the chip-level interrupt masks.  It also ensures that the	*/
/* board level interrupt is also disabled, if no other channels are	*/
/* active.								*/
/************************************************************************/
VOID	disable_port(local_unit, ints)
	WORD	local_unit, ints ;
{
	WORD	port = (*ports_hsd[local_unit]).port_number ;
	(*ports_hsd[local_unit]).state &=  ~ints ;
	outp(port+IER,(*ports_hsd[local_unit]).state);	/* disable the chip interrupt mask	*/
	outp(0x21, ( inp(0x21) | IRQ_MSK) ) ;
}

/************************************************************************/
/*	Stop_port							*/
/* This routine is called in order to perform the appropriate flow 	*/
/* control action for this unit and to prevent over_run errors.		*/
/************************************************************************/
WORD	stop_port(unit)
WORD	unit ;
{
	return	(unit) ;
}

/************************************************************************/
/*	start_port							*/
/* This routine is called to enable the flow of data from the specified	*/
/* unit. It is assumed that the flow was stopped by the stop_port 	*/
/* routine above.							*/
/************************************************************************/
WORD	start_port(unit)
WORD	unit ;
{
	return (unit) ;
}

/************************************************************************/
/*	port_init							*/
/*  This routine initializes or re-initializes the hardware in response	*/
/* to the SET or INIT call.						*/
/************************************************************************/
VOID	port_init(unit)
WORD	unit ;
{
	WORD	baud, port ;
	BYTE	mode,cntrl ;
	PORT_TBL  *port_table ;

	port_table = &hsd_portab[unit];
	mode = (*port_table).HSD_mode;
	cntrl = (*port_table).HSD_control;
	baud = hsd_baud_tab[(*port_table).HSD_baud];
	port = (*ports_hsd[unit]).port_number ;
	outp(port+LCR, 0x80);
	outp(port+IER, (baud >> 8) & 0xFF) ;
	outp(port+DATA, baud & 0xFF) ;
	outp(port+LCR,0) ;
	outp(port+LCR, ( (mode & PTM_LENGTH)
			| ( (mode & (PTM_PARITY + PTM_12STOP) & ~4) >> 1 ) ) );
	outp(port+MCR, ( ((cntrl & PTC_DTR) >>1)
			| ((cntrl & PTC_RTS) >> 4)
			| 0x08 ) );
}


/************************************************************************/
/*	hsd_enq								*/
/* This routine places as many bytes as will fit into the transmit 	*/
/* queue from the user buffer.						*/
/************************************************************************/
VOID	hsd_enq(ch,unit)
WORD	ch, unit ;	/* the character to be enqueued and the unit.	*/
			/* of the PHYSBLK.wq				*/
{
	PHYSBLK	*phys_blk ;
	WORD	new_qfront, port ;

	phys_blk = ports_hsd[unit] ;
	port = (*phys_blk).port_number ;

        NOINT
	if (port_stopped[unit]) 
	{

		while( !(inp(port+LSR) & 0x20) )
			;	/* spin for xmitter empty		*/
		while( !(inp(port+LSR) & 0x40) )
			;	/* spin for xmitter empty		*/
		outp(port+DATA,ch) ;
                OKINT
	}
	else
	{
                OKINT
		new_qfront = ( ((*phys_blk).wqfront+1) % MAXOUTW ) ;
		/* Now we wait!!! for the xmit int to give us room	*/
		while( new_qfront == ((volatile)(*phys_blk).wqrear) )
			;

		/* Put the byte to be sent into the Queue		*/

		(*phys_blk).wq[(*phys_blk).wqfront] = (BYTE) ch ;
		(*phys_blk).wqfront = new_qfront ;
	}
}

/************************************************************************/
/*	hsd_deq								*/
/* This routine is used by the synchronous portion of the code to pull	*/
/* from the queue maintained by the hsd_int_service routine the bytes	*/
/* needed to satisfy the read request.					*/
/************************************************************************/
WORD	hsd_deq( unit )
BYTE	unit ;
{
	WORD	port ;
	PHYSBLK	*phys_blk ;
	WORD	new_qrear ;
	WORD	ch ;

     	phys_blk = ports_hsd[unit] ;

	port = (*phys_blk).port_number ;

	new_qrear = (((*phys_blk).rqrear+1) % MAXOUTR) ;

	if( (*phys_blk).rqrear == (*phys_blk).rqfront )
	{
		return (0x0100) ;
	}

	while( ((volatile)(*phys_blk).rqrear) == ((volatile)(*phys_blk).rqfront ))
		;

	ch = (*phys_blk).rq[(*phys_blk).rqrear] ;
	(*phys_blk).rqrear = new_qrear ;
	return (ch) ;
}

/************************************************************************/
/*	asr_deq								*/
/* This routine is used by the asynchronous portion of the code to pull	*/
/* from the queue maintained by the hsd_int_service routine the bytes	*/
/* needed to satisfy the read request.					*/
/* NOTE:  The only difference between the synchronous and asynchronus	*/
/* versions of this code is the absence of the while statement.		*/
/* busy waits should not be performed in ISR or ASR contexts.		*/
/************************************************************************/
WORD	asr_deq( unit )
BYTE	unit ;
{
	WORD	port ;
	PHYSBLK	*phys_blk ;
	WORD	new_qrear ;
	WORD	ch ;

     	phys_blk = ports_hsd[unit] ;

	port = (*phys_blk).port_number ;

	new_qrear = (((*phys_blk).rqrear+1) % MAXOUTR) ;

	if( (*phys_blk).rqrear == (*phys_blk).rqfront )
	{
		return (0x0100) ;
	}

	ch = (*phys_blk).rq[(*phys_blk).rqrear] ;
	(*phys_blk).rqrear = new_qrear ;
	return (ch) ;
}

