/*#######################################################################
#									#
#				NOTICE:					#
#									#
#    The source code contained in this listing is a proprietary trade	#
#    secret of DIGITAL RESEARCH, INC., Pacific Grove, California and	#
#    is also subject to copyright protection as an unpublished work	#
#    pursuant to Section 104(a) of Title 17 of the United States Code.	#
#    Unauthorized copying, adaptation, distribution, use or display is	#
#    prohitibed by law and may be subject to civil and criminal		#
#    penalties.								#
#									#
#######################################################################*/

/*#######################################################################
#									#
#  Filename: PICODE01.C							#
#									#
#   Purpose: Global primitives - single functions			#
#									#
#  Maintenance Log:							#
#									#
#  Ver	Date	  Who  SPR/APR	Change/Comments				#
#  ---	--------  ---  -------	--------------------------------------	#
#  1.0	10/02/87  DS	 N/A	Created.				#
#  1.1	11/09/87  WeD	 N/A	Combined with PCOPYSYS.C		#
#  1.2	11/21/87  WeD	 N/A	Enhanced DriveLock code.		#
#  1.3	11/28/87  WeD	 N/A	More DriveLock changes.			#
#  1.4	11/30/87  WeD	 N/A	PCheckDiskArea uses physical read.	#
#######################################################################*/

/*#######################################################################
#									#
#  Outline of General Structures or Algorithms used in the code		#
#									#
#######################################################################*/

/*#######################################################################
#									#
#  The following functions are contained in this file:			#
#									#
#	PCheckDiskArea();						#
#	PConvertClusterToSector();					#
#	PDataStartingSector();						#
#	PDriveLock();							#
#	PGetClusterSize();						#
#	PGetDefaultDrive();						#
#	PGetFatSectors();						#
#	PGetSectorSize();						#
#	PGetRootDirEntries();						#
#	PReadLogicalSector();						#
#	PReadPhysicalSector();						#
#	PUnlockDrive();							#
#	PWriteLogicalSector();						#
#	PWritePhysicalSector();						#
#	ZeroLogicalSectors();						#
#	PRootDirLogicalSector();					#
#	PCopySystem();							#
#	PCopyFat1ToFat2();						#
#									#
#######################################################################*/

/*#######################################################################
#									#
#  Included Files:							#
#									#
#######################################################################*/

#include <stdlib.h>
#include <portab.h>
#include <string.h>

#include "piglobal.h"
#include "pistruct.h"
#include "piprotos.h"

/*#######################################################################
#									#
#  Local Structure Definitions						#
#									#
#######################################################################*/

typedef struct
{
    ULONG Key;
    UBYTE DeviceName[10];
    UWORD DeviceType;
    UWORD AccessMode;
    UWORD InstallStatus;
    UWORD OwnerID;
} DeviceTableEntry;

/*#######################################################################
#									#
#  Static Variables							#
#									#
#######################################################################*/

/*#######################################################################
#									#
#  External Functions and locations:					#
#									#
#    ULONG  PFatEntryOP();	   --  Defined in PICODE02.C File	#
#    ULONG  PDirEntryOP();	   --  Defined in PICODE02.C File	#
#    ULONG  PDataEntryOP();	   --  Defined in PICODE02.C File	#
#    ULONG  PGetTimeDate();	   --  Defined in PICODE03.C File	#
#    LONG   s_open();		   --  High C Runtime Library		#
#    LONG   s_read();		   --  High C Runtime Library		#
#    LONG   s_close();		   --  High C Runtime Library		#
#    LONG   s_define();		   --  High C Runtime Library		#
#    LONG   s_read();		   --  High C Runtime Library		#
#    LONG   s_write();		   --  High C Runtime Library		#
#    LONG   s_open();		   --  High C Runtime Library		#
#    LONG   s_close();		   --  High C Runtime Library		#
#    LONG   s_special();	   --  High C Runtime Library		#
#    LONG   s_lookup();		   --  High C Runtime Library		#
#									#
#    void   *malloc();		   --  High C Runtime Library		#
#    void   free();		   --  High C Runtime Library		#
#									#
#######################################################################*/

/*#######################################################################
#									#
#  External Variables:							#
#									#
#######################################################################*/

extern	POSBootRec		GOSBootRec;
extern	PLockTable		GLockTable;

/*#######################################################################
#									#
# Function Name: PCheckDiskArea						#
#									#
#  Return Value: SUCCESS (0, area is good), Error Code (area is bad),	#
#		 or FAILURE (-1, unable to perform test).		#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: SectorOffset	- (32-bit ULONG) The physical starting	#
#				  sector of a disk.			#
#		 Count		- (32-bit ULONG) The length to checked. #
#		 DiskID		- (32-bit PDiskIDStruct ptr) Used to ID #
#				  the disk.				#
#									#
#   Description: This function read verifies a certain section of a	#
#		 disk. If the area is good, SUCCESS is returned.	#
#		 If bad, the error code is returned. It is used by	#
#		 FDISK to find out if an OS boot record, FAT, and Root	#
#		 Dir can be put on a disk in a certain area.		#
#									#
#######################################################################*/

ULONG PCheckDiskArea(ULONG SectorOffset, ULONG Count, PDiskIDStruct *DiskID)
{
    ULONG      ByteSize;
    BYTE      *BufPtr;
    LONG       ReadCount;
    LONG       RtnCode;


/*  Calculate the buffer size required to do the read verify and allocate
    some memory for the buffer.*/

    ReadCount  = MAX_MALLOC_SIZE / DiskID->MediaDesBlock.SectorSize;

    if( Count > ReadCount )
	ByteSize = MAX_MALLOC_SIZE;
    else
	ByteSize = Count * DiskID->MediaDesBlock.SectorSize;

    BufPtr = (BYTE*)malloc( (unsigned int)ByteSize );

    if( BufPtr == NULL )
	return( FAILURE );

/*  Read verify all sectors or until an error occurs.*/

    while( Count )
    {
	if( Count < ReadCount )
	    ReadCount = Count;

	if( (RtnCode = s_special( 10, 0, DiskID->DriveSVCID, BufPtr, ByteSize,
				  SectorOffset, ReadCount )) != SUCCESS )
	{
	    free( BufPtr );
	    return( RtnCode );
	}
	SectorOffset += ReadCount;
	Count	     -= ReadCount;
    }

/*  Free the allocated memory and return.*/

    free( BufPtr );
    return( SUCCESS );

}/* PCheckDiskArea */

/*#######################################################################
#									#
# Function Name: PConvertClusterToSector				#
#									#
#  Return Value: The logical sector offset of a cluster.		#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: ClusterIndex - 16 bit UWORD.  This is the cluster	#
#		 that's used to figure out a logical sector offset.	#
#									#
#   Description: This function converts a ClusterIndex to a logical	#
#		 sector number.						#
#									#
#######################################################################*/

ULONG PConvertClusterToSector( UWORD ClusterIndex )
{
    register ULONG	InitStartData;
    register UWORD	SectorsPerCluster;

/*  Get the starting logical sector and sectors per cluster.*/

    InitStartData = PDataStartingSector();
    SectorsPerCluster = PGetClusterSize();

    return(((ClusterIndex - 2) * (ULONG)SectorsPerCluster) + InitStartData);

}/* PConvertClusterToSector */

/*#######################################################################
#									#
# Function Name: PDataStartingSector					#
#									#
#  Return Value: The first logical sector address of the data area.	#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: None.							#
#									#
#   Description: Gets the first sector address of the data		#
#		 area and returns it.					#
#									#
#######################################################################*/

ULONG PDataStartingSector( void )
{
    ULONG	SectorCount;
    ULONG	ByteCount;

/*  Get number of sectors occupied by all FATs in partition.*/
    SectorCount = PRootDirLogicalSector();

/*  Calculate number of bytes in the root directory.*/
    ByteCount = (ULONG) GOSBootRec.BPB.NumRootDirEntries *
		 sizeof(PDirEntryStruct);

/*  Convert byte count into sector count.*/
    SectorCount = SectorCount + (ByteCount / PGetSectorSize());

    return(SectorCount);

}/* PDataStartingSector */

/*#######################################################################
#									#
# Function Name: PDriveLock						#
#									#
#  Return Value: 0 for SUCCESS or Error Code.				#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: DiskID      - (32-bit PDiskIDStruct ptr) Used to ID the#
#			       disk in question and the reset of the	#
#			       structure is filled in.			#
#		 LockMode    - 0 == Physical lock of all partition(s)	#
#			       1 == Logical lock of one partition.	#
#			       2 == Logical lock of one partition or	#
#				    unformatted media.			#
#		 ReadWrite   - (16-bit Boolean) If TRUE then lock the	#
#			       drive exclusive, else lock read only.	#
#									#
#   Description: This function serves two purposes:  First, it locks a	#
#		 drive at the physical or logical level depending on the#
#		 user's request (locking it as read or read/write);	#
#		 Second, it fills in the PDiskIDStruct.			#
#									#
#######################################################################*/

ULONG PDriveLock(PDiskIDStruct *DiskID, WORD LockMode, WORD ReadWrite)
{
    LONG	RtnCode;
    UWORD	OpenMode;
    UWORD	LogicalDisk;
    LONG	TempFileID;
    char	DiskName[11], *CharPtr;
    int		DeviceLen;

    PMediaDescriptionBlock	*MediaDesBlock;
    PMediaDescriptionBlock	 TempMediaDesBlock;
    DeviceTableEntry		 DeviceEntry;


/*  If ReadWrite is TRUE then Open the disk exclusive read/write,
    else open the disk shared.*/

    if( ReadWrite )
	OpenMode = OPEN_EXCLUSIVE;
    else
	OpenMode = OPEN_SHARED;

    if( s_define( 2, DiskID->DriveName, DiskName, 10L ) == SUCCESS
      || s_define( 3, DiskID->DriveName, DiskName, 10L ) == SUCCESS )
    {
	if( (CharPtr = strchr( DiskName, ':' )) != NULL )
	{
	    *(CharPtr+1) = '\0';
	}
    }
    else
	strcpy( DiskName, DiskID->DriveName );


/*  Lower-case the DiskName for s_lookup() and _compare().*/

    for( CharPtr = DiskName; *CharPtr; ++CharPtr )
    {
	if( *CharPtr <= 'Z' && *CharPtr >= 'A' )
	    *CharPtr |= 0x20;
    }

    if( (RtnCode = s_open( OpenMode, DiskName )) < SUCCESS )
	return(RtnCode);


/*  Set the DiskID->SVCID to the new ID.*/
    DiskID->DriveSVCID = RtnCode;

    GLockTable.Locks[GLockTable.NumOfLocks++] = RtnCode;

/*  Set up the MDB pointer */
    MediaDesBlock = &DiskID->MediaDesBlock;

/*  Get the drive's MDB and use that information for locking the drive.*/

    if( (RtnCode = s_special(GET_MDB, NEW_MDB, DiskID->DriveSVCID,
	 MediaDesBlock, (ULONG)sizeof(PMediaDescriptionBlock),
	 0L, 0L)) < SUCCESS )
    {
	if( LockMode == PLOGICAL_FORMAT_LOCK )
	{
	    DiskID->DriveType	= UNFORMATTED_TYPE;
	    DiskID->Partitioned = 0;
	    return( SUCCESS );
	}
	else
	{
	    PUnlockDrive();
	    return( FAILURE );
	}
    }

/*  Use the Fat ID byte as an indicator of the current partition's status.*/
    LogicalDisk = MediaDesBlock->FatID;

    if( s_lookup( DEV_INFO, 0, DiskName, &DeviceEntry,
		  (ULONG)sizeof(DeviceTableEntry),
		  (ULONG)sizeof(DeviceTableEntry), 0L ) != 1 )
    {
	PUnlockDrive();
	return( FAILURE );
    }

/*  Setup some more of the DiskID */
    DiskID->DriveType	 = ((DeviceEntry.AccessMode & DISK_NONREMOVABLE) == 0);
    DiskID->Partitioned  = (MediaDesBlock->NumberFats > 0);

/*  Check if caller wants a physical or logical lock.*/

    if( LockMode == 1 && !LogicalDisk
     && (DeviceEntry.AccessMode & DISK_PARTITIONABLE) )
    {
	PUnlockDrive();
	return(FAILURE);
    }

/*  Now check if we are trying to do a physical lock but have logical
    partitions.*/

    if( !LockMode && (DeviceEntry.AccessMode & DISK_PARTITIONABLE) )
    {
/*	Set the key to ZERO */
	DeviceEntry.Key = 0;

/*	If here, we have to find all the other logical partitions on the
	disk and lock them.*/

/*	Run through the known devices, looking for logical paritions on
	our disk.*/

/*	Do a lookup, if it fails we are done, else process the entry */
	while( s_lookup(DEV_INFO, 0, "*:", &DeviceEntry,
			 (ULONG)sizeof(DeviceTableEntry),
			 (ULONG)sizeof(DeviceTableEntry),
			 DeviceEntry.Key) == 1 )
	{
	    DeviceLen = strlen( DeviceEntry.DeviceName );

	    DeviceEntry.DeviceName[ DeviceLen   ] = ':';
	    DeviceEntry.DeviceName[ DeviceLen+1 ] = '\0';

	    if( (DeviceEntry.DeviceType & 0xF0) == DISK_DRIVER
	      && (DeviceEntry.AccessMode & DISK_PARTITIONABLE)
	      && _compare( DeviceEntry.DeviceName, DiskName, DeviceLen+2 ) )
	    {
/*		We need to find out if this is a logical partition on the disk
		we are trying to lock.	First do an open on the disk, so we
		can get it's MDB.*/

		if( (TempFileID = s_open(OPEN_SHARED, DeviceEntry.DeviceName))
		    < SUCCESS )
		{
/*		    If we have an error, than give up.*/
		    PUnlockDrive();
		    return(TempFileID);
		}

/*		Now get the MDB to see if it is a drive we need to lock up.*/

		if( (RtnCode = s_special(GET_MDB, NEW_MDB, TempFileID,
					 &TempMediaDesBlock,
					 (ULONG)sizeof(PMediaDescriptionBlock),
					 0L, 0L)) < SUCCESS )
		{
		    s_close( 0, TempFileID );
		    PUnlockDrive();
		    return(RtnCode);
		}

		s_close(0, TempFileID);

/*		Check if it is a logical disk partition that we want to lock. */

		if( TempMediaDesBlock.PhysicalDriveNumber
		 == MediaDesBlock->PhysicalDriveNumber )
		{

/*		    We have a disk we need to lockup.  So, try to lock it up
		    and update the global lock table.  First, check to see
		    if we have another lock entry */

		    if( GLockTable.NumOfLocks >= PMAX_DISK_LOCKS )
		    {
			PUnlockDrive();
			return(FROM_DISK | 0x4000);
		    }

		    if( (GLockTable.Locks[GLockTable.NumOfLocks++]
			= s_open(OpenMode, DeviceEntry.DeviceName)) < SUCCESS )
		    {
			--GLockTable.NumOfLocks;
			PUnlockDrive();
			return(FAILURE);
		    }
		}
	    }
	}
    }

    return(SUCCESS);

}/* PDriveLock */

/*#######################################################################
#									#
# Function Name: PGetClusterSize					#
#									#
#  Return Value: Number of sectors per cluster.				#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: None.							#
#									#
#   Description: Gets the cluster size in sectors from the GOSBootRec.	#
#									#
#######################################################################*/

UWORD PGetClusterSize( void )
{

    return((UWORD) GOSBootRec.BPB.SectorsPerCluster);

}/* PGetClusterSize */

/*#######################################################################
#									#
# Function Name: PGetDefaultDrive					#
#									#
#  Return Value: 0 for SUCCESS, else FAILURE.				#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: DiskName - 32-bit UBYTE pointer. Used by the function	#
#		 to store the character name of the default disk drive. #
#									#
#   Description: This function accepts a string pointer as it's one	#
#		 argument and fills that string in with the character	#
#		 name of the default disk that the application is	#
#		 running under.  The name returned is 4 characters	#
#		 long (plus null); e.g. fd1:, hd0:, etc.		#
#									#
#######################################################################*/

ULONG PGetDefaultDrive( UBYTE *DiskName )
{

/*  Go get the default drive from the PROCDEF table.*/

    if( s_define( GET_LIT_PROCDEF, "default:", DiskName, 2L )
	 != SUCCESS || DiskName[1] != ':' )
    {
	return( FAILURE );
    }

    DiskName[0] &= ~0x20;		/* Upper case */
    DiskName[2]  = '\0';		/* Null-terminated */

    return( SUCCESS );

}/* PGetDefaultDrive */

/*#######################################################################
#									#
# Function Name: PGetFatSectors						#
#									#
#  Return Value: FatSizeInSectors from OS Boot Record			#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: None.							#
#									#
#   Description: Get the FAT table size in sectors.			#
#									#
#######################################################################*/

UWORD PGetFatSectors( void )
{

    return((UWORD) GOSBootRec.BPB.FatSizeInSectors);

}/* PGetFatSectors */

/*#######################################################################
#									#
# Function Name: PGetSectorSize						#
#									#
#  Return Value: Number of bytes per sector.				#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: None.							#
#									#
#   Description: Gets the sector size from the GOSBootRec structure.	#
#									#
#######################################################################*/

UWORD PGetSectorSize( void )
{

    return((UWORD) GOSBootRec.BPB.BytesPerSector);

}/* PGetSectorSize */

/*#######################################################################
#									#
# Function Name: PGetRootDirEntries					#
#									#
#  Return Value: Number of entries in the root directory.		#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: None.							#
#									#
#   Description: Gets the number of entries in the root directory from	#
#		 the GOSBootRec.					#
#									#
#######################################################################*/

ULONG PGetRootDirEntries( void )
{

    return((ULONG) GOSBootRec.BPB.NumRootDirEntries);

}/* PGetRootDirEntries */

/*#######################################################################
#									#
# Function Name: PReadLogicalSector					#
#									#
#  Return Value: Returns a 0 for SUCCESS or an ErrorCode.		#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: SectorOffset - 32-bit ULONG. Used as an disk sector	#
#				offset.					#
#		 Count	      - 16-bit UWORD. Total number of sectors to#
#				read.					#
#		 Buffer       - 32-bit char ptr.  Sectors are read into.#
#		 DiskID       - 32-bit DiskID ptr. The DiskID field in	#
#				structure is used in the special call to#
#				do the read.				#
#									#
#   Description: This function calls the SPECIAL disk logical devrw read#
#		 call. It uses the first parameter as a logical sector	#
#		 offset from the beginning of the physical disk and then#
#		 reads the sector into memory.				#
#									#
#######################################################################*/

ULONG PReadLogicalSector(ULONG SectorOffset, UWORD Count, UBYTE *Buffer,
	PDiskIDStruct *DiskID)
{
    LONG	ByteOffset;
    ULONG	BufSize;
    LONG	RtnCode;		/* Number of bytes or error code.*/


/*  Setup BufSize and ByteOffset for the s_read.*/

    BufSize = Count * (LONG)DiskID->MediaDesBlock.SectorSize;
    ByteOffset = SectorOffset * (LONG)DiskID->MediaDesBlock.SectorSize;

/*  ...and perform the read, returning any error.*/

    RtnCode = s_read( READ_LOGICAL_FLAGS, DiskID->DriveSVCID, Buffer,
		      BufSize, ByteOffset );

    if( RtnCode != BufSize )
	return( RtnCode );

    return( SUCCESS );

}/* PReadLogicalSector */

/*#######################################################################
#									#
# Function Name: PReadPhysicalSector					#
#									#
#  Return Value: 0 for SUCCESS or Error Code.				#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: SectorOffset - (ULONG) Used as an absolute disk sector #
#				offset.					#
#		 Count	      - (UWORD) Number of sectors to read.	#
#		 Buffer       - (BYTE ptr) Put the sectors in the here. #
#		 DiskID       - (PDiskIDStruct ptr) The disk ID field in#
#				the structure is used for the read in	#
#				the SPECIAL.				#
#									#
#   Description: This function calls the SPECIAL disk physical sector	#
#		 READ call.  It uses SectorOffset as an absolute sector #
#		 offset from the beginning of the physical disk and then#
#		 reads the sectors into memory.				#
#									#
#######################################################################*/

ULONG PReadPhysicalSector(ULONG SectorOffset, UWORD Count, UBYTE *Buffer,
	PDiskIDStruct *DiskID)
{
    LONG	RtnCode;
    ULONG	ReadSize;


/*  Calculate the number of bytes they wish to read.*/
    ReadSize = Count * (LONG)DiskID->MediaDesBlock.SectorSize;

/*  Perform the physical read.*/
    RtnCode = s_special( 10, 0, DiskID->DriveSVCID,
			 Buffer, ReadSize, SectorOffset, (LONG)Count );

    return(RtnCode);

}/* PReadPhysicalSector */

/*#######################################################################
#									#
# Function Name: PUnlockDrive						#
#									#
#  Return Value: 0 for SUCCESS or Error Code.				#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: None.							#
#									#
#   Description: This function Unlocks all drives that were previously	#
#		 locked by the DriveLock function.  The devices are	#
#		 closed and the Lock Table is cleared out.		#
#									#
#######################################################################*/

ULONG PUnlockDrive( void )
{
    LONG	RtnCode = SUCCESS;


/*  Step through the GLockTable and close all necessary devices.*/

    while( GLockTable.NumOfLocks )
    {
	 if( s_close(0, GLockTable.Locks[--GLockTable.NumOfLocks]) != SUCCESS )
	     RtnCode = FAILURE;
    }

    return(RtnCode);

}/* PUnlockDrive */

/*#######################################################################
#									#
# Function Name: PWriteLogicalSector					#
#									#
#  Return Value: Returns a 0 for SUCCESS or an ErrorCode.		#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: SectorOffset - 32-bit ULONG. Used as a disk sector	#
#				offset.					#
#		 Count	      - 16-bit UWORD. Total number of sectors to#
#				read.					#
#		 Buffer       - 32-bit char ptr.  Sectors are written	#
#				from this buffer.			#
#		 DiskID       - 32-bit DiskID ptr. The DiskID field in	#
#				structure is used in the special call to#
#				do the write.				#
#									#
#   Description: This function calls the logical disk write system	#
#		 call. It uses the first parameter as a logical sector	#
#		 offset from the beginning of the physical disk and then#
#		 writes the sectors to the disk.			#
#									#
#######################################################################*/

ULONG PWriteLogicalSector(ULONG SectorOffset, UWORD Count, UBYTE *Buffer,
	PDiskIDStruct *DiskID)
{
    LONG	ByteOffset;
    ULONG	BufSize;
    LONG	RtnCode;		/* Number of bytes or error code.*/


/*  Setup BufSize and ByteOffset for the s_write.*/

    BufSize = Count * (LONG)DiskID->MediaDesBlock.SectorSize;
    ByteOffset = SectorOffset * (LONG)DiskID->MediaDesBlock.SectorSize;

/*  Write the number of bytes that we have calculated as BufSize.*/

    RtnCode = s_write(WRITE_LOGICAL_FLAGS, DiskID->DriveSVCID, Buffer,
		      BufSize, ByteOffset);

    if( RtnCode != BufSize )
	return( RtnCode );

    return( SUCCESS );

}/* PWriteLogicalSector */

/*#######################################################################
#									#
# Function Name: PWritePhysicalSector					#
#									#
#  Return Value: 0 for SUCCESS or Error Code.				#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: SectorOffset - (ULONG) Used as an absolute disk sector #
#				offset.					#
#		 Count	      - (UWORD) Number of sectors to write.	#
#		 Buffer       - (BYTE ptr) Write the data from here.	#
#		 DiskID       - (PDiskIDStruct ptr) The disk ID field in#
#				the structure is used for the write in	#
#				the SPECIAL.				#
#									#
#   Description: This function calls the SPECIAL disk physical sector	#
#		 WRITE call. It uses the parameter as an absolute sector#
#		 offset from the beginning of the physical disk and then#
#		 writes the sectors into memory.			#
#									#
#######################################################################*/

ULONG PWritePhysicalSector(ULONG SectorOffset, UWORD Count, UBYTE *Buffer,
	PDiskIDStruct *DiskID)
{
    LONG	 RtnCode;
    ULONG	 WriteSize;


/*  Calculate the number of bytes they wish to write.*/
    WriteSize = Count * (LONG)DiskID->MediaDesBlock.SectorSize;

/*  Perform the physical write.*/

    RtnCode = s_special( 10, 0x100, DiskID->DriveSVCID,
			 Buffer, WriteSize, SectorOffset, (LONG)Count );

    return(RtnCode);

}/* PWritePhysicalSector */

/*#######################################################################
#									#
# Function Name: ZeroLogicalSectors					#
#									#
#  Return Value: Return a 0 for SUCCESS or Error Code.			#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: FirstSector - ID for starting logical sector.		#
#		 Count	     - Number of sectors to clear.		#
#		 DiskID      - PDiskIDStruct pointer.			#
#									#
#   Description: This function zeros out sectors on the disk.  It uses	#
#		 the FirstSector field as the starting address and the	#
#		 count as the number to zero-out.			#
#									#
#######################################################################*/

ULONG ZeroLogicalSectors(LONG FirstSector, ULONG Count, PDiskIDStruct *DiskID)
{
    LONG	 RtnCode = SUCCESS;
    UWORD	 SecsPerWrite;
    ULONG	 BytesToAllocate;
    UWORD	 BytesAllocated;
    UBYTE	*BufPtr;


/*  Get the sector size and calculate how many bytes the caller needs.*/
    BytesToAllocate = BytesAllocated = Count * PGetSectorSize();

/*  Allocate the required memory (<32K).*/

    if( BytesToAllocate > MAX_MALLOC_SIZE )
	BytesAllocated = MAX_MALLOC_SIZE;

    if( (BufPtr = malloc(BytesAllocated)) == NULL )
	return(FAILURE);

/*  Now zero out the allocated buffer.*/
    _fill_char(BufPtr, BytesAllocated, (UBYTE) 0);

/*  Calculate sectors per write since it may be less than the total.*/
    SecsPerWrite = BytesAllocated / PGetSectorSize();

/*  Clear the specified number of disk sectors.*/

    while( RtnCode == SUCCESS && Count )
    {
	RtnCode = PWriteLogicalSector(FirstSector, SecsPerWrite,
				      BufPtr, DiskID);
/*	Adjust our parameters.*/

	FirstSector += SecsPerWrite;
	Count -= SecsPerWrite;

/*	Check for a remainder. If so, adjust SecsPerWrite.*/
	if( Count < SecsPerWrite )
	    SecsPerWrite = Count;
    }

/*  Free the memory allocated and return.*/

    free(BufPtr);

    return(RtnCode);

}/* ZeroLogicalSectors */

/*#######################################################################
#									#
# Function Name: PRootDirLogicalSector					#
#									#
#  Return Value: Sector offset.						#
#									#
#   Environment: Runs as an application support routine.		#
#									#
#     Arguments: None.							#
#									#
#   Description: This function returns the starting logical sector of	#
#		 the Root Directory.					#
#									#
#######################################################################*/

ULONG PRootDirLogicalSector( void )
{

    return( (ULONG)GOSBootRec.BPB.FatSizeInSectors *
		   GOSBootRec.BPB.NumFatsInPart + 1 );

}/* PRootDirLogicalSector */

/*#######################################################################
#									#
# Function Name:  PCopySystem						#
#									#
#  Return Value:  Size of the file in bytes or FAILURE			#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  FileName - character pointer to the file's name on	#
#			     the current drive.				#
#		  SysFileIndex - an integer flag ( ROOTDIR/SUBDIR ).	#
#		  DiskIDPtr - pointer to PDiskIDStruct structure of the #
#			      destination.				#
#									#
#   Description:  Function copies the named file, either BLOAD.IMG or	#
#		  FLEXOS.SYS to a newly formatted disk.			#
#		  This code assumes that there is enough room on the	#
#		  destination disk to store the file, and that the	#
#		  first directory entry in the root directory is	#
#		  available for use.					#
#									#
#######################################################################*/

ULONG PCopySystem(BYTE *FileName, BYTE *DirName, WORD SysFileIndex,
	PDiskIDStruct *DiskIDPtr)
{
    LONG	     FileNum;
    BYTE	    *BufPtr;
    PDirEntryStruct *DirPtr;
    LONG	     Count;
    LONG	     BufSize;
    LONG	     FileSize;
    LONG	     FileCount;
    UWORD	     CurrentFat, LastFat, FatValue;
    UWORD	     EndOfChain  = ~0;
    ULONG	     ReturnValue = FAILURE;

    /*
	Determine the size of the buffer to allocate for reading and
	writing the system file (size is equal to the size of a cluster
	on the target media) and allocate a buffer.
    */

    BufSize = (LONG)PGetSectorSize() * PGetClusterSize();

    if( ( BufPtr = (char *) malloc( (UWORD) BufSize ) ) == 0L )
	return( FAILURE );

    /*
	Allocate a buffer to hold one directory entry.
    */

    if( (DirPtr =
        (PDirEntryStruct *) malloc( (UWORD) sizeof(PDirEntryStruct) ) ) == 0L )
    {
	free( BufPtr );
	return( FAILURE );
    }

    /*
	Open the appropriate system source file indicated by FileName,
	on the indicated drive/path.
    */

    if( (FileNum = s_open( OPEN_SHARED, FileName )) <= 0 )
    {
	free( BufPtr );
	free( DirPtr );
	return( FAILURE );
    }

    /*
	We will start the FAT scan at the 3rd entry (#2).
    */

    CurrentFat = 1;

    FileSize = FileCount = 0;

    while( (Count = s_read( PREAD_FLAG, FileNum, BufPtr, BufSize, 0L )) > 0L )
    {
	/*
	    Increment the file's size by the read amount.
	*/

	FileSize  += Count;
	FileCount += 1;

	/*
	    Save the previous FAT index for later use.
	*/

	LastFat = CurrentFat;

	do
	{
	    /*
		Search the FAT forward, looking for an empty cluster.
	    */

	    CurrentFat++;
	    PFatEntryOP( CACHE, CurrentFat, PREAD_OP, &FatValue, DiskIDPtr );

	} while( FatValue != 0 );

	if( LastFat == 1 )
	{
	    /*
		Fill in the FirstCluster field in the Directory buffer if
		this is the first FAT entry.
	    */

	    DirPtr->FirstCluster = CurrentFat;
	}
	else
	{
	    /*
		After that, mark the previously marked FAT entry with the
		index of the current FAT.
	    */

	    PFatEntryOP( CACHE, LastFat, PWRITE_OP, &CurrentFat, DiskIDPtr );
	}

	/*
	    Mark the current FAT entry as the end of the chain and write
	    the current data cluster.
	*/

	if( PDataEntryOP( CurrentFat, PWRITE_OP, BufPtr,
			  BufSize, DiskIDPtr ) != SUCCESS )
	    goto TidyUp;

    }/* while */

    if( Count < 0 && (UWORD)Count != 0x4003 )
	goto TidyUp;

    PFatEntryOP( CACHE, CurrentFat, PWRITE_OP, &EndOfChain, DiskIDPtr );

    /*
	Fill in the fields of the Directory entry buffer, and write
	the directory.
    */

    _move( DirName, DirPtr->FileName, LEN_FILENAME );
    DirPtr->Attributes = READ_ONLY_MASK | HIDDEN_FILE_MASK | SYSTEM_FILE_MASK;
    memset( DirPtr->Reserved, 0, 3 );
    DirPtr->RecordSize = 1;
    DirPtr->UserID = 0;
    DirPtr->GroupID = 0;
    DirPtr->FileProtection = PPROTECTION;
    PGetTimeDate( &DirPtr->TimeLastMod, &DirPtr->DateLastMod );
    DirPtr->FileLength = FileSize;

    /*
	DirEntryOP takes as arguments a Root Directory Indicator, ROOTDIR,
	a cluster index (0), the system file indicator ( 0 - bootimage ),
	an operation flag (PWRITE_OP), a pointer to a Directory structure
	(DirPtr), and a disk identifier (DiskIDPtr).
    */

    if( PDirEntryOP( ROOTDIR, 0, (ULONG)SysFileIndex, PWRITE_OP,
		     DirPtr, DiskIDPtr ) == SUCCESS )
	ReturnValue = SUCCESS;

TidyUp:

    /*
	Free up the buffers, close the source file, and return status.
    */

    free( BufPtr );
    free( DirPtr );

    s_close( 0, FileNum );

    if( ReturnValue == SUCCESS )
	return( FileCount * BufSize );
    else
	return( ReturnValue );

}/* PCopySystem */

/*#######################################################################
#									#
# Function Name:  PCopyFat1ToFat2					#
#									#
#  Return Value:  SUCCESS or FAILURE					#
#									#
#   Environment:  Process						#
#									#
#     Arguments:  A pointer to a PDiskIDStruct structure for the disk.	#
#									#
#   Description:  Function creates the 2nd copy of the FAT from the 1st #
#		  copy of the FAT for the disk identified by DiskIDPtr. #
#									#
#######################################################################*/

ULONG PCopyFat1ToFat2( PDiskIDStruct  *DiskIDPtr )
{
    LONG    Offset;		/* Logical sector number		*/
    LONG    Count;		/* Number of cluster chunks to copy	*/
    BYTE   *BufPtr;		/* Pointer to a sector's worth of data	*/
    UWORD   SectorSize;		/* Size of a sector in bytes		*/
    UWORD   CopySize;		/* Number of sectors to copy at a time	*/
    LONG    FATSectors;		/* Number of sectors occupied by a FAT	*/

    /*
	Determine the size of the buffer to allocate for reading and
	writing (size is equal to 8 times the size of a sector) and
	allocate a buffer.
    */

    SectorSize = PGetSectorSize();
    CopySize = 8;

    if( (BufPtr = (char *)malloc( (UWORD)(SectorSize*CopySize) )) == 0L )
	return( FAILURE );

    /*
	Set a displacement variable equal to the distance from the first copy
	of the FAT to the second copy of the FAT.
    */

    FATSectors = PGetFatSectors();
    Count      = FATSectors / CopySize;

    /*
	Copy blocks from logical sector 1 to the end of the fat.
    */

    for( Offset = 1; Count--; Offset += CopySize )
    {
	if( PReadLogicalSector( Offset, CopySize,
				BufPtr, DiskIDPtr ) != SUCCESS
	 || PWriteLogicalSector( Offset + FATSectors, CopySize,
				 BufPtr, DiskIDPtr ) != SUCCESS )
	{
	    free( BufPtr );
	    return( FAILURE );
	}
    }

    /*
	Copy any oddball sectors at the end of the fat.
    */

    if( (CopySize = FATSectors-(Offset-1)) != 0 )
    {
	if( PReadLogicalSector( Offset, CopySize,
				BufPtr, DiskIDPtr ) != SUCCESS
	 || PWriteLogicalSector( Offset + FATSectors, CopySize,
				 BufPtr, DiskIDPtr ) != SUCCESS )
	{
	    free( BufPtr );
	    return( FAILURE );
	}
    }

    free( BufPtr );

    return( SUCCESS );

}/* PCopyFat1ToFat2 */

/*#######################################################################
#									#
#				NOTICE:					#
#									#
#    The source code contained in this listing is a proprietary trade	#
#    secret of DIGITAL RESEARCH, INC., Pacific Grove, California and	#
#    is also subject to copyright protection as an unpublished work	#
#    pursuant to Section 104(a) of Title 17 of the United States Code.	#
#    Unauthorized copying, adaptation, distribution, use or display is	#
#    prohitibed by law and may be subject to civil and criminal		#
#    penalties.								#
#									#
#######################################################################*/
