/***********************************************************************

      File:  pia.h

      (C) Copyright 1992 by GO Corporation, All Rights Reserved.

      Author:      Satya Yenigalla

         Date:      21 August 1991

      This file contains the standard definitions for the 
      PenPoint Information  Architecture (PIA).  PIA is an 
      open standard for PenPoint that allows database applications 
      to access data from multiple back-end DBMS systems for which 
      a database driver exists.

***********************************************************************/

#ifndef SERVICE_INCLUDED
#include <service.h>
#endif

/*==============    MESSAGES   ========================*/

#define   msgDbPrepare			   MakeMsg  (clsDbService, 1  )
#define   msgDbBind     			MakeMsg  (clsDbService, 2  )
#define   msgDbDeclareCursor		MakeMsg  (clsDbService, 3  )
#define   msgDbOpenCursor 			MakeMsg  (clsDbService, 4  )
#define   msgDbCloseCursor			MakeMsg  (clsDbService, 5  ) 
#define   msgDbDeleteCursor 		MakeMsg  (clsDbService, 6  )
#define   msgDbBuildDoQuery		MakeMsg  (clsDbService, 7  )
#define   msgDbColCount			   MakeMsg  (clsDbService, 8  )
#define   msgDbColDescribe			MakeMsg  (clsDbService, 9  )
#define   msgDbFetch 			   MakeMsg  (clsDbService, 10 )
#define   msgDbTables			   MakeMsg  (clsDbService, 11 )
#define   msgDbColumns			   MakeMsg  (clsDbService, 12 )
#define   msgDbIndexes			   MakeMsg  (clsDbService, 13 )
#define   msgDbTransaction			MakeMsg  (clsDbService, 14 )
#define   msgDbError				   MakeMsg  (clsDbService, 15 )




/* PIA datatype  for host variables  */

typedef  enum PIA_TYPE  {
      PIA_STRING,
      PIA_CHAR,
      PIA_SIGNED_16,
      PIA_SIGNED_32,
      PIA_UNSIGNED_16,
      PIA_UNSIGNED_32,
      PIA_DECIMAL,
      PIA_FLOAT,
      PIA_DOUBLE,
      PIA_OS_DATE_TIME,
      PIA_BINARY
}   PIA_TYPE,   *P_PIA_TYPE;


/*   COLUMN  structure  used in the msgDbColDescribe */
/*   COLUMN - aka   ATTRIBUTE, FIELD.   */

typedef  U16       COL_OFST,       *P_COL_OFST;  // Ofst to a col in a tuple
typedef  U16       COL_COUNT,      *P_COL_COUNT; // # of cols in a relation
typedef  U16       COL_LEN,        *P_COL_LEN;   // Length of the column
typedef  U16       COL_INX,        *P_COL_INX;   // A particular column's number


typedef  struct     {
      P_CHAR   	colName;
      PIA_TYPE	   dataTypes;   //  datatypes supported by PIA.
      COL_LEN	   colLength;   //  length of the column
      COL_INX	   colCount;    //  a particular column's number
      COL_OFST	   colOfst;     //  offset of the column in tuple
}   COL_DESC,    * P_COL_DESC;


/*  Error  number  returned  by  messages   */

typedef   S32      PIA_ERROR,   * P_PIA_ERROR ;


/*  Schema Inforamation Tables, for Tables and Columns  */

/*  Table   information */

typedef   P_CHAR        TABLE_NAME;
typedef   U16           TABLE_NUM;
typedef   U32           CARDINALITY;
typedef   P_CHAR        TABLE_OWR;

/*  Column  information  */

typedef  P_CHAR            COLUMN_NAME;
typedef  U16               COLUMN_NUMBER;
typedef  PIA_TYPE          COLUMN_TYPE;
typedef  U16               COLUMN_LENGTH;
typedef  BOOLEAN           NULL_ALLOWED;
typedef  BOOLEAN           COLUMN_UPDATE;



/* database   location  */

typedef  enum PIA_DB_TYPE  {
       LOCAL_DB,
       REMOTE_DB,
}   PIA_DB_TYPE;

/* Capabilities  of the database  drivers and its  components    */

/* Locking  capabilities   */

typedef  struct  LOCK_CAP  {
         BOOLEAN    lockSupported;
}   LOCK_CAP,  * P_LOCK_CAP;



/*  Transaction  Management  capabilities   */

typedef  struct  TRANS_CAP  {
         BOOLEAN      transmgmt;
}   TRANS_CAP,  * P_TRANS_CAP;


/*  Types  of  querying  capabilities  supported  on the database  driver  */

typedef  enum  QUERY_CAP  {
         SELECTABLE		=   flag1,
         UPDATABLE		=   flag2,
         ALTERABLE		=   flag3,
         DELETABLE		=   flag4,
         INSERTABLE		=   flag5,
}   QUERY_CAP,  * P_QUERY_CAP;

/*  Capabilities  structure   */

typedef struct PIA_SERVICE_CAP  {
         QUERY_CAP		      queryCap;  //  Bitwise  OR  of  query capabilities
         TRANS_CAP	      	transcap;
         LOCK_CAP       		lockcap;
         OS_MILLISECONDS		timeOut;
         U8		            	drvVersion;
}   PIA_SERVICE_CAP,  * P_PIA_SERVICE_CAP;



/***********************************************************************

   msgSMOpen


   pArgs :        P_SM_OPEN_CLOSE
   Returns :      STATUS
   Description :  This message opens the service, given its handle. The
   handle can be obtained by msgIMFind. Before doing an Open,
   the client should bind to the service by msgSMBind.


   typedef struct SM_OPEN_CLOSE {
      OBJECT       handle;   // Handle of service to open
      OBJECT       caller;   // Object making this call   
      P_ARGS       pArgs;    // service specific open parameters
      OBJECT       service;  // In: (SMClose)  Out: (SMOpen)     
   }  SM_OPEN_CLOSE,   *P_SM_OPEN_CLOSE;

***********************************************************************/


#define  PIA_DB_OPEN(n)       struct  {                  \
              PIA_ERROR        drvErrCode ;              \
              PIA_ERROR        dbErrCode  ;              \
              P_UNKNOWN        connectParams [(n)];      \
}

typedef     PIA_DB_OPEN(1)   * P_PIA_DB_OPEN ;

/*

   The Service-specific open parameters (pArgs)  is PIA_DB_OPEN.  
   The connectParams element of this structure consists of :
                               0 : Database Userid
                               1 : Database Password
                               2 : Userid of the Server
                               3 : Password of the Server


   These 0 through 3 connectParams are the UserIds and Passwords for 
   connecting to the right database on the right server.  These are 
   followed by the Network parameters :
   (example of network parameters for Oracle via Appletalk)

                               4 : Network 
                               5 : ZoneName
                               6 : Service

   These network parameters will vary according to the type of network 
   and the kind of parameters the network demands to make a connection. 

   The pArgs, service specific connectParams [(n)]  for msgSMOpen 
   consists of a fixed part - connectParams [0]   thru   connectParams [3], 
   and a variable part starting at connectParams [4] which is 
   implementation defined.
                           
*/





/***********************************************************************

   msgSMCharacteristics


   pArgs :          SM_GET_CHARACTERISTICS
   Returns :        STATUS
   Description :  Gets the characteristics of the specified service instance.
   Characteristics are service-specific properties of a particular service.
   The service will return the service-specific characteristics
   via pArgs->pBuf.  pArgs->len specifies the maximum size of
   client's buffer.  If pArgs->len is 0  then the service should return
   actual size of its characteristics in pArgs->len and not pass back
   any data. Clients then allocate this space and make the call again
   with pArgs->len set to this size. 

   typedef     struct   SM_GET_CHARACTERISTICS  {
                        OBJECT      handle;   // Handle of item to get characteristics of.
                        P_UNKNOWN   pBuf;     // Out through Ptr: Characteristics buffer.
                        U16         len;      // In/Out: Buffer size. If 0, actual size ret.
   }   SM_GET_CHARACTERISTICS, * P_SM_GET_CHARACTERISTICS;

Note: Even though these are service characteristics, they mostly relate 
to the DBMS it interacts.

***********************************************************************/


   typedef  struct PIA_DRIVER_CAP    {
                  PIA_SERVICE_CAP          serviceCap;
   }   PIA_DRIVER_CAP,   *P_PIA_DRIVER_CAP;


/*

   pBuf is a pointer to the buffer described by P_PIA_DRIVER_CAP data structure.
   This data structure enumerates various capabilities this service can offer. 
   Some of which are as follows:  what kind of queries can be executed, 
   type of locking scheme supported, transaction management etc.

   This list will be subject to future changes, to accommodate desirable 
   requests from ISV's.

*/




/***********************************************************************

   msgDbDeclareCursor

   pArgs :          P_PIA_DECLARE_CURSOR
   Returns :        STATUS
   Description :  This message declares a cursor and associates the 
   query with the cursor.  Query is not executed at this time.  

***********************************************************************/


/*   Types  of call back  functions.  These  functions will be called by 
     the database service to ask the application some data  (for the 
     bind variables)  or to supply it the fetched data (for the select-
     variables).  The parameteres  :
             -  the name of the host  variable (can be NULL)
             -  the type of the data  (PIA_TYPE)
             -  a pointer  to the buffer
                  -  in which you have to store the data (bind variables)
                  -  where you can find the fetched data (select variables)
             -   the length of the data
             -   a userparameter, not modified by the driver, just passed
        The use of these functions is optional, the alternative is to get/pass the data
        via the buffers.
*/

typedef  void  (EXPORTED  * GetProcType)
                   (char     *     varName,
                    short          type,
                    char     *     varBuff,
                    short    *     len,
                    long           userParam) ;

typedef   void  (EXPORTED   * SetProcType)
                 (char      *   varName,
                  short         type,
                  char      *   varBuff,
                  short         len,
                  long          userParam) ;


typedef  struct   HOST_VARS   {
         P_CHAR		varName;
         PIA_TYPE		dataType;
         S32			rowSize;
         P_UNKNOWN	buffPtr;
}   HOST_VARS,    *P_HOST_VARS;


#define PIA_DECLARE_CURSOR(n)   struct    {  \
            P_CHAR	   	cursorName;       \
            P_CHAR		   query;      		\
            S8	   		   numBinds,	      \
         			   	   numSelects;	      \
            U32			   flags;		      \
            S32			   drvErrCode,	      \
				               dbErrCode;	      \
            P_UNKNOWN		userParam;	      \
            GetProcType		getValue;	      \
            SetProcType		setValue,	      \
				               appendValue;	   \
            HOST_VARS		hostVars [(n)];	\
} 

typedef     PIA_DECLARE_CURSOR(1)    *P_PIA_DECLARE_CURSOR;


/***********************************************************************

   msgDbOpenCursor


   pArgs :          P_PIA_OPEN_CURSOR
   Returns :        STATUS
   Description :  This message  executes a previously  declared query,
   identifying it through its unique name.

***********************************************************************/


typedef   struct  PIA_OPEN_CURSOR   {
            P_CHAR                  cursorName;
            PIA_ERROR               drvErrCode;
            PIA_ERROR               dbErrCode;
            S16                     parseOfst;      //  in case of db parsing error
}   PIA_OPEN_CURSOR,  * P_PIA_OPEN_CURSOR;


/***********************************************************************

   msgDbBuildDoQuery


   pArgs :        P_PIA_BLD_DO_QUERY
   Returns :      STATUS
   Description :  This message taken  on  'extended'  sql-query,  
   parses it and  fills in a P_PIA_DECLARE_CURSOR  structure for you.  
   This structure can then be used in the  Declare Cursor - message. 
   The user still has to supply the flags , userParam, setValue,
   etc - fields.  Because this msg allocates the declareCusor, the 
   user has to deallocate it after use.
   
***********************************************************************/


typedef   struct  PIA_BLD_DO_QUERY  {
            P_CHAR                 doQuery;        // extended SQL - query
            PIA_ERROR              errCode;
            P_PIA_DECLARE_CURSOR   declareCursor;  // Out: pointer to
                                                   // DECLARE_CURSOR structure
}   PIA_BLD_DO_QUERY,  *P_PIA_BLD_DO_QUERY;




/***********************************************************************

   msgDbPrepare


   pArgs :        P_PIA_PREPARE
   Returns :      STATUS
   Description :  This message prepares the query.  Once the query is 
   prepared,  it is ready to accept the host variable values provided thru
   msgDbBind.
   The prepare sends the query to the database for parsing and
   declaration of its host variables.
   A queryNumber, uniquely identifying the query, is returned.
   Error codes  can be returned in the queryInfo field.

***********************************************************************/


#define   PIA_PREPARE(n)   struct  {			\
               U32       queryNumber;	      \
               S16       parseOfst;	         \
               PIA_DECLARE_CURSOR (n)    queryInfo;	\
}

typedef   PIA_PREPARE(1)        *P_PIA_PREPARE;



/***********************************************************************

   msgDbBind


   pArgs :        P_PIA_BIND
   Returns :      STATUS
   Description :  This message binds the host variable values with the query
   that has been prepared thru msgDbPrepare  and executes
   the query.


***********************************************************************/

typedef  struct  PIA_BIND   {
              U32			queryNumber;    // In: number of prepared query
              PIA_ERROR	drvErrCode;
              PIA_ERROR	dbErrCode;
}   PIA_BIND,    * P_PIA_BIND;




/***********************************************************************

   msgDbCloseCursor


   pArgs :        P_PIA_CLOSE_CURSOR
   Returns :      STATUS
   Description :  Closes the cursor. All cursor locks are released during this
   time. When the cursor is reopened, it points to the first tuple of
   the result set.

***********************************************************************/


typedef   struct  PIA_CLOSE_CURSOR {
            P_CHAR                  cursorName;
            PIA_ERROR               drvErrCode;
            PIA_ERROR               dbErrCode;
}   PIA_CLOSE_CURSOR,  *P_PIA_CLOSE_CURSOR;


/***********************************************************************

   msgDbDeleteCursor


   pArgs :           P_PIA_DELETE_CURSOR
   Returns :        STATUS
   Description :  Deletes the cursor.  This is different from closing 
   the cursor. This message actually deletes the cursor from existence. 
   
***********************************************************************/


typedef   struct  PIA_DELETE_CURSOR {
            P_CHAR                  cursorName; 
            PIA_ERROR               drvErrCode;
            PIA_ERROR               dbErrCode;
}   PIA_DELETE_CURSOR,   *P_PIA_DELETE_CURSOR;


/***********************************************************************

   msgDbColCount


   pArgs :           P_PIA_COL_COUNT
   Returns :        STATUS
   Description :  This message is responsible for returning the number of columns
   in the result set.  This number can be used for describing each
   column in the msgDbColDescribe.


***********************************************************************/

typedef   struct  PIA_COL_COUNT  {
            U32			   queryNumber;
            P_COL_COUNT	   colCount;    // Out : returns number of columns
            PIA_ERROR		drvErrCode;
            PIA_ERROR		dbErrCode;
}   PIA_COL_COUNT,  * P_PIA_COL_COUNT;




/***********************************************************************

   msgDbColDescribe


   pArgs :           P_PIA_COL_DESC
   Returns :        STATUS
   Description :  This message returns the data definitions (metadata) 
   information of the result set.  Database driver can then return 
   the mapped data types, corresponding to those returned from the database.

***********************************************************************/


typedef   struct  PIA_COL_DESC   {
            U32		   	queryNumber;
            P_COL_INX		colInx;
            P_COL_DESC		colDesc;
            PIA_ERROR		drvErrCode;
            PIA_ERROR		dbErrCode;
}   PIA_COL_DESC,  * P_PIA_COL_DESC;





/***********************************************************************

   msgDbFetch


   pArgs :        P_PIA_FETCH
   Returns :      STATUS
   Description :  This message fetches the data  from the result set of the data.
   Fetches a row of data as directed by the FETCH_MODE.
   Its not a requirement that all FETCH_MODE be supported by
   the driver, for reasons  that can be attributed to the database system.
   
***********************************************************************/

typedef   enum  FETCH_MODE  {
            FETCH_NEXT,
            FETCH_PREV,
            FETCH_FIRST,
            FETCH_LAST,
            FETCH_RELATIVE
}  FETCH_MODE,   *P_FETCH_MODE;

/* retrieve mode, how the data are stored in the buffers
    - row  or column wise
*/

typedef  enum   RTR_MODE   {
              ROW_WISE,
              COL_WISE
}    RTR_MODE,  *  P_RTR_MODE;

/*  dataValues  array - contains pointers to buffers  with  fetched data  */

#define    PIA_FETCH(n)  struct  {     \
   P_CHAR   			cursorName;	   	\
   U32		   		queryNumber;		\
   FETCH_MODE			fetchMode;		   \
   U32			   	tupleCount;		   \
   U32				   tupleNum;		   \
   RTR_MODE			   rtrMode;		      \
   PIA_ERROR			drvErrCode;		   \
   PIA_ERROR			dbErrCode;		   \
   P_UNKNOWN			dataValues[(n)];	\
}

typedef  PIA_FETCH(1)     *P_PIA_FETCH ;


/***********************************************************************

   msgDbTables


   pArgs :        P_PIA_DICT_TBL
   Returns :      STATUS
   Description :  The database driver will send the appropriate 
   request to the catalog to fetch the matching table names. Once 
   the query is formed, results are fetched like any other query 
   using msgDbFetch.


***********************************************************************/


/*  tableParams  array  -  used for specifying the parameters
    for obtaining the data  definitions  of a system  table 
    (database   dependent)
*/

#define  PIA_DICT_TBL(n)      struct   {                      \
              PIA_ERROR                 drvErrCode;           \
              PIA_ERROR                 dbErrCode;            \
              P_CHAR                    tableParams [(n)];    \
}

typedef    PIA_DICT_TBL(1)       *P_PIA_DICT_TBL;





/***********************************************************************

   msgDbColumns


   pArgs :        P_PIA_DICT_COLS
   Returns :      STATUS
   Description :  The driver makes the appropriate request to the catalog supported
   by the database system.  Once the query is formed,
   results are fetched just like any other query using msgDbFetch.


***********************************************************************/

#define  PIA_DICT_COLS(n)      struct   {	        \
              PIA_ERROR         drvErrCode;       \
              PIA_ERROR         dbErrCode;	     \
              P_CHAR            columnParams[(n)];\
}

typedef    PIA_DICT_COLS(1)       *P_PIA_DICT_COLS;




/***********************************************************************

   msgDbIndexes


   pArgs :        P_PIA_DICT_INDXS
   Returns :      STATUS
   Description :  Requires the driver to send the appropriate request 
   to the catalog to get the list of indexes.  Once the query is formed, 
   the results are fetched just like any other messages - msgDbFetch.

***********************************************************************/



#define  PIA_DICT_INDXS(n)    struct   {                     \
              PIA_ERROR                 drvErrCode;          \
              PIA_ERROR                 dbErrCode;           \
              P_CHAR                    indexParams[(n)];    \
}

typedef    PIA_DICT_INDXS(1)       *P_PIA_DICT_INDXS;





/***********************************************************************

   msgDbTransaction


   pArgs :        P_PIA_TRANSACTION
   Returns :      STATUS
   Description :  This message actually sets the COMMIT or ROLLBACK as
   requested by the application.


***********************************************************************/


typedef  enum  TRANS_TYPE   {
            PIA_ROLLBACK,
            PIA_COMMIT
}  TRANS_TYPE,  *P_TRANS_TYPE;

typedef   struct  PIA_TRANSACTION  {
            TRANS_TYPE         transType;
            PIA_ERROR          drvErrCode;
            PIA_ERROR          dbErrCode;
}   PIA_TRANSACTION,  *P_PIA_TRANSACTION;





/***********************************************************************

   msgDbError


   pArgs :        P_PIA_ERR_MSG
   Returns :      STATUS
   Description :  This message may be called after any message that 
   results in ERROR, WARNING OR SUCCESS_WITH_INFO.  To retrieve
   these error/warning messages, msgDbError can be issued to the
   driver.  The driver maintains in the buffer, the latest error info.
   This info stays in the buffer only until the next error occurs, at
   which time the previous error info is lost.  If more granularity is
   required as the situation demands, can be accommodated in the
   future versions.

***********************************************************************/


typedef   struct  PIA_ERR_MSG  {
            PIA_ERROR          dbError;        //  In : error  returned by db
            PIA_ERROR          errCode;        //  Out : driver  error
            S16                buffLen;        //  length  of  buffer
            P_CHAR             message;        //  ptr  to buffer with message text
}  PIA_ERR_MSG,   * P_PIA_ERR_MSG;

