/*
 A File: addrbook.h

 (C) Copyright 1992 by Go Corporation, All Rights Reserved.

 $Revision:   1.17  $
 $Author:   pma  $
 $Date:   26 Feb 1992 16:25:14  $

 clsAddressBook inherits from clsObject.

 This header file defines the address book protocol.  

 The address book protocol defines what minimal set of information is to
 be kept by an address book app or service, how
 information is to be stored, retrieved, queried by an address book client.
 Please refer to abmgr.h for information on address book manager.  

//REFGEN BEGINIGNORE
 The following diagram illustrates the relationship between different pieces 
 that are involved in storing and retriving address book information.


	--------------------------------------------------			-----------
	|     Address Book Application or Service        |          |  more	  |
	| 	  (includes an Address Book Protocol         |          | Address |
	|     implementation)                            |          | Book App|
	|________________________________________________|  ....... | or Svcs |
	                        |                           /       |_________|
	                        |                         /
	                        |                       /
	                        |                     /
	                        |                   /
	        -------------------------------------
	        |                                   |
	        |       Address Book Manager        |
	        |                                   |
	        |___________________________________|
	      /         |           |                 \
        /           |           |                   \
      AB            AB          AB     ......       AB
	Client        Client      Client              Client


 As the illustration suggests, there can multiple address book applications
 or services registered w/ the system address book manager.  
//REFGEN ENDIGNORE

 All requests to access address book information is channeled
 through the address book manager.
 There can be multiple address book clients at one time.  Whether or not 
 address book clients can access information from more than 1 address book 
 application/service simultaneously is completely up to the implementation
 of the address book manager.  The current implementation of theAddressBookMgr
 provided by GO only allows access to one address book at a time.

 Because theAddressBookMgr uses ObjectSend
 to relay messages to address books, pointers in pArgs in any address 
 book protocol messages should point to some shared memory space.
		
 There are 3 major types of address information defined by the protocol:
	-: individual personal information(e.g.name, phone number, street address)
	-: service information(individual's fax phone number, email address, etc)
	-: distribution list information

 All information is kept/retrieved in attribute-value form.  The basic entity
 in an address book is an "entry"; all information is presented relative to
 an entry.  E.g. to access any information in an address book, a "key" to
 an entry must be presented.  Within an entry, a client can set/get entry
 related information(name, street address, etc.).  Service address information
 is also kept as part of an entry.  Because there can be multiple service
 addresses for each entry(e.g. an individual has 2 fax numbers and 1 email
 address), a service address is accessed through a "service id" or the
 name of the service.(e.g. service name = "fax")

 The Address Book Protocol specifies a minimum set of attributes and
 attribute types to be supported by third party address book applicaitons
 or services.  If a developer thinks that some addition attributes or
 attribute types are common enough that they should be defined in the
 protocol, please contact GO Corporation Developer Support.

****************************************************************************/

#ifndef ADDRBOOK_INCLUDED
#define ADDRBOOK_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef UID_INCLUDED
#include <uid.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef DIALENV_INCLUDED
#include <dialenv.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				Common #defines and typedefs							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 All address book apps should be a sub-class of this app. Being a sub-class
 of clsAddrBookApplication frees an address book application from having
 to register, and unregister itself w/ TheAddressBookMgr.  TheAddressBookMgr
 will notice when an instance of clsAddrBookApplication has been
 created/destroyed, and will automatically register/unregister the instance.
 Aside from providing this auto registeration/unregisteration, 
 clsAddrBookApplication provides no other special behavior to its sub-class.
*/
#define clsAddrBookApplication			MakeWKN(3284, 1, wknGlobal)

/****  Pre-defined Attribute Types  ****/
#define	abNumber		MakeTag(clsAddressBook, 0)	// 32-bit number
#define abString		MakeTag(clsAddressBook, 1)	// null-terminated string
#define abPhoneNumber	MakeTag(clsAddressBook, 2)	// DIALENV_TELEPHONE_NUMBER
#define abOther			MakeTag(clsAddressBook, 3)	// some encoded byte array
													// interpreted by address
													// books simply as a byte
													// stream

/****  Pre-defined attribute ids  ****/
#define AddrBookGroupNameId	MakeTag(clsAddressBook, 0)	// abString
#define	AddrBookGivenNameId	MakeTag(clsAddressBook, 1)	// abString
#define	AddrBookSurNameId	MakeTag(clsAddressBook, 2)	// abString
#define	AddrBookHomePhoneId	MakeTag(clsAddressBook, 3)  // abPhoneNumber
#define	AddrBookBussPhoneId	MakeTag(clsAddressBook, 4)	// abPhoneNumber
#define	AddrBookCountryId	MakeTag(clsAddressBook, 5)	// country in post
														// addr,abString
#define	AddrBookStateId		MakeTag(clsAddressBook, 6)	// state or prefe-
														// cture, abString
#define	AddrBookZipId		MakeTag(clsAddressBook, 7)	// zip, abString
#define	AddrBookCityId		MakeTag(clsAddressBook, 8)	// city , abString
#define	AddrBookDistrictId	MakeTag(clsAddressBook, 9)	// ku in Japanese
														// addr, abString
/* 
 AddrBookStreetId represents street, number, building and other addressing
 information, the character \012(LF in ASCII) can be used to separate
 the different parts.  E.g. a street address can be
	 2650 Durant Avenue
	 Deutsch Hall #406
 In this case, the address should be stored in AddrBookStreetId as
 "2650 Durant Avenue\012Deutsch Hall #406"
*/
#define	AddrBookStreetId	MakeTag(clsAddressBook, 10)	// abString
#define	AddrBookCompanyId	MakeTag(clsAddressBook, 11)	// company name, 
														// abString
#define AddrBookTitleId		MakeTag(clsAddressBook, 18) // title of an 
														// individual entry
														// abString
#define AddrBookPositionId	MakeTag(clsAddressBook, 19) // position of an 
														// individual entry
														// abString
#define AddrBookNickNameId	MakeTag(clsAddressBook, 20) // nickname of an 
														// individual entry
														// abString
#define AddrBookBussPhone2Id	MakeTag(clsAddressBook, 21) // 2nd bussiness 
														// phone #
														// abPhoneNumber
#define AddrBookFaxId		MakeTag(clsAddressBook, 22) // fax # of an
														// individual entry
														// abPhoneNumber
#define	AddrBookSvcNameId	MakeTag(clsAddressBook, 12)	// name of svc, 
														// abString
#define	AddrBookSvcNoteId	MakeTag(clsAddressBook, 13)	// user defined 
														// svc nickname
														// abString
#define	AddrBookSvcShortId	MakeTag(clsAddressBook, 14)	// service short 
														// address
/* 
 The following two special id's are used in specifying a query
*/
#define AddrBookEntryKeyId	MakeTag(clsAddressBook, 15) 
#define AddrBookSvcIdId		MakeTag(clsAddressBook, 16) 

/*
 This is the type for address book transfer protocol. If an address
 book supports move/copy protocol, then it should transfer an entry
 in a XFER_BUF structure, where XFER_BUF.pBuf is a pointer to 
 ADDR_BOOK_ENTRY structure.
*/
#define AddrBookXferType	MakeTag(clsAddressBook, 17)

#define AddrBookAll						(maxU16)
#define AddrBookAllSvcSelectAttrs		(maxU16-1)
#define AddrBookSelectSvcSelectAttrs	(maxU16-2)
#define AddrBookSelectSvcAllAttrs		(maxU16-3)

/*
 If the client wants all attributes(either all entry attributes or
 all service attributes.), the address book should return the attributes
 in some well-known order.  The next batch of #define's specifies
 the order for the common fields
*/
#define	AddrBookSurNameIndex		0
#define	AddrBookGivenNameIndex		1
#define	AddrBookHomePhoneIndex		2
#define	AddrBookBussPhoneIndex		3
#define	AddrBookCountryIndex		4
#define	AddrBookStateIndex			5
#define	AddrBookZipIndex			6
#define	AddrBookCityIndex			7
#define	AddrBookDistrictIndex		8
#define	AddrBookStreetIndex			9
#define	AddrBookCompanyIndex		10
#define AddrBookTitleIndex			11
#define AddrBookPositionIndex		12
#define AddrBookNickNameIndex		13
#define AddrBookBussPhone2Index		14
#define AddrBookFaxIndex			15


#define	AddrBookSvcNameIndex		0
#define	AddrBookSvcNoteIndex		1
#define	AddrBookSvcShortIndex		2

typedef P_UNKNOWN	ADDR_BOOK_SERVICE_ID,	*P_ADDR_BOOK_SERVICE_ID;
typedef TAG			ADDR_BOOK_ATTR_ID,		*P_ADDR_BOOK_ATTR_ID;
typedef TAG			ADDR_BOOK_ATTR_TYPE,	*P_ADDR_BOOK_ATTR_TYPE;
typedef U16			ADDR_BOOK_ATTR_LENGTH,	*P_ADDR_BOOK_ATTR_LENGTH;
typedef P_UNKNOWN	ADDR_BOOK_ATTR_VALUE,	*P_ADDR_BOOK_ATTR_VALUE;
typedef P_UNKNOWN	ADDR_BOOK_KEY,			*P_ADDR_BOOK_KEY;

typedef CHAR	ADDR_BOOK_ATTR_LABEL[nameBufLength];

/*
 ADDR_BOOK_ATTR.length is the length of ADDR_BOOK_ATTR.value.  
 The following table lists what the length field mean, given a certain
 attribute type:

    Attr Type                length
    =======================================================
    abString           length of the string
    abNumber           SizeOf(U32)
    abPhoneNumber      SizeOf(DIALENV_TELEPHONE_NUMBER)
    abOther            length of attribute in bytes

 The following table lists what the value field should be, given a certain
 attribute type:
    Attr Type                  value
    =========================================================
    abString           a ptr to actual storage of the str
    abNumber           the number itself
    abPhoneNumber      P_DIALENV_TELEPHONE_NUMBER
    abOther            a ptr to a byte array that 
                       contains the attribute.
*/

typedef struct ADDR_BOOK_ATTR {
	ADDR_BOOK_ATTR_ID		id;		
	ADDR_BOOK_ATTR_TYPE		type;	
	ADDR_BOOK_ATTR_LENGTH	length;			// length of value, in bytes
	ADDR_BOOK_ATTR_VALUE	value;	
	ADDR_BOOK_ATTR_LABEL	label;			// for display purpose
} ADDR_BOOK_ATTR, *P_ADDR_BOOK_ATTR;

typedef struct ADDR_BOOK_ATTR_DESC {
	ADDR_BOOK_ATTR_ID		id;		
	ADDR_BOOK_ATTR_TYPE		type;	
	ADDR_BOOK_ATTR_LABEL	label;			// for display purpose
} ADDR_BOOK_ATTR_DESC, *P_ADDR_BOOK_ATTR_DESC;

typedef struct ADDR_BOOK_SERVICE {
	ADDR_BOOK_SERVICE_ID	svcId;			// uniquely identify a svc inst
	U16						numAttrs;
	P_ADDR_BOOK_ATTR		attrs;
} ADDR_BOOK_SERVICE, *P_ADDR_BOOK_SERVICE;


Enum16(ADDR_BOOK_ENTRY_TYPE) {
	abIndividual	= 0,
	abGroup			= 1,
};

#define abMaxSvcNameMatch		5

typedef struct ADDR_BOOK_SERVICE_QUAL {
	U16						numAttrIds;
	P_ADDR_BOOK_ATTR_ID		svcAttrIds;
	U16						numSvcNames;
	CHAR					svcNames[abMaxSvcNameMatch][nameBufLength];
} ADDR_BOOK_SERVICE_QUAL, *P_ADDR_BOOK_SERVICE_QUAL;

/*
 .heap field is an in-parameter in msgAddrBookGet and msgAddrBookSearch,
 it is not applicable for other msgs.  A client should specify the
 heap id of the heap that it would like space allocated.  Typically
 a client would use OSTaskSharedHeapId(clientsTaskId).  A client should
 not use osProcessSharedHeapId or osProcessHeapId because they refer
 to different heaps in diffferent processes.  It is very important
 that clients free allocated space.
*/
typedef struct ADDR_BOOK_ENTRY {
	OS_HEAP_ID					heap;			// where should the address
												// book alloc necessary space
												// applicable only for 
												// msgAddrBookGet and
												// msgAddrBookSearch
	ADDR_BOOK_ENTRY_TYPE		type;
	ADDR_BOOK_KEY				key;
	U16							numAttrs;
	P_ADDR_BOOK_ATTR			attrs;
	U16							numServices;	// Read only,abIndividual only
	P_ADDR_BOOK_SERVICE			services;		// abIndividual only
	ADDR_BOOK_SERVICE_QUAL		svcQual;		// service qualifier, for Get
} ADDR_BOOK_ENTRY, *P_ADDR_BOOK_ENTRY;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Status Codes								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
*/

/****  Error Status Values  ****/
#define stsAddrBookBufTooSmall			MakeStatus(clsAddressBook, 1)
#define stsAddrBookEntryExists			MakeStatus(clsAddressBook, 2)
#define stsAddrBookSvcDataExists		MakeStatus(clsAddressBook, 3)
#define stsAddrBookEntryNotFound		MakeStatus(clsAddressBook, 4)
#define stsAddrBookSvcNotFound			MakeStatus(clsAddressBook, 5)
#define stsAddrBookBadKey				MakeStatus(clsAddressBook, 6)
#define stsAddrBookUnknownType			MakeStatus(clsAddressBook, 7)
#define stsAddrBookInvalidAttr			MakeStatus(clsAddressBook, 8)
#define stsAddrBookReadOnlyAttr			MakeStatus(clsAddressBook, 9)
#define stsAddrBookDuplicateAttrId		MakeStatus(clsAddressBook, 10)

/****  Non Error Status Values  ****/
#define stsAddrBookGroupEntry				MakeWarning(clsAddressBook, 7)
#define stsAddrBookNotSupported				MakeWarning(clsAddressBook, 8)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						 Messages					                       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgAddrBookGet			takes P_ADDR_BOOK_ENTRY, returns STATUS
	fills in the specified entry field data, given an address book 
	key for the entry.  
		
 If attribute type is abString and the client-provided
 space is not big enough, stsAddrBookBufTooSmall is returned, and as much
 information as there is room for is filled in(null-terminated).
 Similarly, if attribute type is abOther, stsAddrBookBufTooSmall is 
 returned, and the client-provided buffer is filled in(w/o 
 null-termination).

 Parameters:
	pArgs->key:				In:  specify from which entry to get info
	pArgs->type:			Out: type of the entry
	pArgs->numAttrs:		In:  number of elements in pArgs->attrs array.
								 Each of pArgs->attrs.id specifies the id
								 of the attribute the client wants the address
								 book to return.  If the client sets this
								 field to AddrBookAll, then the address book
								 will return all entry attributes(excluding
								 services), and it will allocate the necessary
								 space.  The client needs to deallocate the
								 space.  If the field is set to 0, then
								 no attributes are returned.
							Out: number of attributes returned
	pArgs->attrs[x].id:		In:  which attributes to get
	pArgs->attrs[x].type:	Out: attribute type
	pArgs->attrs[x].length:	Out: attribute length of each attr specified
								 in entryAttrIds. See previous table on
								 attribute type-attribute length.
	pArgs->attrs[x].value:	In:  if this field is pNull, the address book
								 will allocate space for the value.
							Out: attribute value.  see previous table on
								 attribute value-attribute length.
	pArgs->attrs[x].label:	Out: attribute label, for display.
	pArgs->numServices:		In:  number of elements in pArgs->services array
								 The client should specify AddrBookAll here
								 if it wants all services and all service
								 attributes for each service.  If it wants
								 only selective attributes from all services,
								 then set numServices to 
								 AddrBookAllSvcSelectAttrs.  If it wants all
								 attributes from selective services, then
								 set numServices to AddrBookSelectSvcAllAttrs.
								 Lastly, if the client wants selective attrs 
								 from selective svcs, then set numServices
								 to AddrBookSelectSvcSelectAttrs.In all cases,
								 the address book will allocate the necessary
								 storage for all info, which needs to be 
								 freed by the client.  If the field is set
								 to 0, then no service information is returned
							Out: number of services returned. 
	pArgs->svcQual:			In:  If numServices is AddrBookAllSvcSelectAttrs,
								 or AddrBookSelectSvcSelectAttrs, then
								 numAttrIds is the number of elements in
								 the svcAttrIds array, and svcAttrIds contains
								 the ids of the attributes whose values should
								 be retrieved.  If numServices is
								 AddrBookSelectSvcAllAttrs or
								 AddrBookSelectSvcSelectAttrs,then numSvcNames
								 is the number of elements in the svcNames
								 array, and svcNames contains the names of
								 services whose attribute values should be
								 retrieved.  For any other values of 
								 numServices, this field is irrelevent.
	pArgs->services:		Out: Allocated space if so requested.
	pArgs->services[y].svcId: In:  For each services specifically requested
								 (as opposed to using AddrBookAll or 
								  AddrBookAllSvcsSelectAttrs, and other such
								  constants in pArgs->numServices), there 
								  needs to be a svcId, telling the address 
								  book which service to return
	pArgs->services[y].attrs:In/Out: analogous to pArgs->attrs
										
*/
#define msgAddrBookGet						MakeMsg(clsAddressBook, 1)

/****************************************************************************
 msgAddrBookSet	takes P_ADDR_BOOK_ENTRY, returns STATUS
	Sets the specified entry and service data .

 Parameters:
	pArgs->key:					In:  specify from which entry to get info
	pArgs->numAttrs:			In:  how many attributes in the entry to set
	pArgs->attr[x].id:			In:  which attributes to set
	pArgs->attr[x].type:		NA:  don't need to specify
	pArgs->attr[x].length:		In:  client-specified size of the correspond-
									 ing entryAttrValue field. mandatory for
									 abOther, unnecessary for other types.
	pArgs->attr[x].value:		In: attribute value.  see previous table on
									 attribute value-attribute length.
	pArgs->numServices:			In:  number of services to set. Set it to 0
									 if not setting any service info
	pArgs->svcAttrIds:			NA:  not applicable
	pArgs->services[y].svcId:	In:  service id of the service that set 
									 applies to
	pArgs->services[y].attrs:   In:  analogous to pArgs->attrs.
										
*/
#define msgAddrBookSet						MakeMsg(clsAddressBook, 2)


/****************************************************************************
 msgAddrBookAdd	takes P_ADDR_BOOK_ENTRY, returns STATUS
	Adds the specified entry and service data.

 Parameters:
	pArgs->key:					In:	 If the msg is used to add a service addr
									 then the client specifies the entry key
									 of the entry to which we add the service
									 address.
								Out: if the msg is used to add an entry, then
									 address book fill this field w/ the key
									 of the entry just added
	pArgs->numAttrs:			In:  how many attributes in the entry to have
									 specified initial values.
	pArgs->attr[x].id:			In:  which attributes to add.  To add a
								     brand new individual entry, then at
									 least AddrBookGivenNameId or 
									 AddrBookSurNameId need to be specified.
									 To add a group entry, AddrBookGroupNameId
									 needs to be specified.
	pArgs->attr[x].type:		NA:  don't need to specify
	pArgs->attr[x].length:		In:  mandatory if attribute type is abOther
	pArgs->attr[x].value:		In:  attribute value.  see previous table on
									 attribute value-attribute length.
	pArgs->numServices:			In:  number of services to set. Set it to 0
									 if not adding any service info
	pArgs->svcAttrIds:			NA:  not applicable
	pArgs->services[y].svcId	Out: service id of the service just added
	pArgs->services[y].attrs    In:  analogous to pArgs->attrs.
										
*/
#define msgAddrBookAdd						MakeMsg(clsAddressBook, 3)


/****************************************************************************
 msgAddrBookDelete	takes P_ADDR_BOOK_ENTRY, returns STATUS
	Deletes the specified entry and service data .

 Parameters:
	pArgs->key:					In:  entry id of the entry to be deleted.
									 If deleting a service, then this field
									 still needs to be specified. Only the
									 specified service is deleted.
	pArgs->numServices:			In:  number of services to delete. Set it to 0
									 if deleting the entire entry.
	pArgs->services[x].svcId	In:  Id's of the services to be deleted

	All other fields in ADDR_BOOK_ENTRY structure are not applicable.	
*/
#define msgAddrBookDelete						MakeMsg(clsAddressBook, 4)

/****************************************************************************
 msgAddrBookSearch	takes P_ADDR_BOOK_SEARCH, returns STATUS
    Searches for the entry that matches the search spec.

 pArgs->key is the pArgs->nth entry that matches the search spec,
 sorted by the attribute specified in pArgs->sort, the entry is just
 before/after(depending on the value of pArgs->dir) of pArgs->key
 If key is nil, the enumeration starts with the first element if abEnumNext
 is specified, and the last element if abEnumPrevious is specified.

 Parameters:
	pArgs->key			In: Start point of the search
						Out:Resulting entry id of the match
	pArgs->nth			In: Look for the nth enty meeting the search criteria
	pArgs->sort			In: attribute id of the attribute that the result
							should be sorted by
	pArgs->dir			In: search backwards or forwards.
	pArgs->outType		Out:type of the matched entry
	pArgs->query		In: an elaborate explanation is available below
	pArgs->result		In: How each field is specified is the same as
							that for msgAddrBookGet.  Except for the key
							field, which will be filled in by 
							msgAddrBookSearch
						Out:same as msgAddrBookGet
*/
#define msgAddrBookSearch						MakeMsg(clsAddressBook, 5)

Enum16(ADDR_BOOK_SEARCH_TYPE) {
	abSearchIndividuals = 0,	// Enumerate address book entries
	abSearchGroups		= 1,	// Enumerate groups
	abSearchAll			= 2,	// Enumerate all entries
};

Enum16(ADDR_BOOK_SEARCH_DIR) {
	abEnumNext			= 0, // Search forward
	abEnumPrevious		= 1  // Search backwards
};

Enum16(ADDR_BOOK_ATTR_OPS) {
	abAnd	= 0,
	abOr	= 1
};

Enum16(ADDR_BOOK_VALUE_OPS) {
	abEqual			= 0,
	abNotEqual		= 1,
	abGreater		= 2,
	abLess			= 3,
	abGreaterEqual	= 4,
	abLessEqual		= 5,
	abMatchBeginning = 6,		// string matching
	abMatchEnd		= 7,		// string matching
	abMatchPartial	= 8,		// string matching
	abMaxValue = abMatchPartial
};

/*
 If a client wants to specify a query that says "match an entry whose
 last name is "Smith" and whose zip code is "94024", then the .query
 field in pArgs for msgAddrBookSearch would have 2 elements:
 
    pArgs->query         id              length  value  valueOp  attrOp
    ===================================================================
    attr[0]       AddrBookGivenNameId     N/A    Smith  abEqual   abAnd
    attr[1]       AddrBookZipId           N/A    94024  abEqual   N/A

 Essentially, the attrOp field specifies the operator between attr[x] and 
 attr[x+1]. valueOp specifies the relationship between the attribute id and
 its specified value. e.g. (a == 1) AND (b == 2), the "=="'s are valueOp,
 "AND" is an attrOp.  By definition, pArgs->attrs[pArgs->numAttrs-1].attrOp 
 does not need to be specified.
*/

typedef struct ADDR_BOOK_QUERY_ATTR {
	ADDR_BOOK_ATTR_ID		id;
	ADDR_BOOK_ATTR_LENGTH	length;
	ADDR_BOOK_VALUE_OPS		valueOp;
	ADDR_BOOK_ATTR_VALUE	value;
	ADDR_BOOK_ATTR_OPS		attrOp;	
} ADDR_BOOK_QUERY_ATTR, *P_ADDR_BOOK_QUERY_ATTR;

typedef struct ADDR_BOOK_QUERY {
	U16							numAttrs;		
	P_ADDR_BOOK_QUERY_ATTR		attrs;
} ADDR_BOOK_QUERY, *P_ADDR_BOOK_QUERY;

typedef struct ADDR_BOOK_SEARCH {
	ADDR_BOOK_KEY			key;	// In: Starting Pt. Out: Result
	ADDR_BOOK_SEARCH_TYPE	type;	// In:
	U32						nth;	// In: look for the nth entry meeting 
									//	   the search criteria.  nth = 1
									//	   if looking for the first entry
									//	   meeting the search criteria.
	ADDR_BOOK_ATTR_ID		sort;
	ADDR_BOOK_SEARCH_DIR	dir;
	ADDR_BOOK_ENTRY_TYPE	outType;
	ADDR_BOOK_QUERY			query;	// In: what to look for, set query to
									//	   pNull to enumerate
	ADDR_BOOK_ENTRY			result;	// Out: result entry
} ADDR_BOOK_SEARCH, *P_ADDR_BOOK_SEARCH;

/****************************************************************************
 msgAddrBookGetServiceDesc	takes P_ADDR_BOOK_SERVICES, returns STATUS
	Gets the service address description from the address book.
		
 Parameters:
    pArgs->numServices:  Out: number of installed services
						 an array of ADDR_BOOK_SVC_DESC's is allocated and
						 should be freed by the caller. 

 Return Value
    stsOK
*/
#define msgAddrBookGetServiceDesc			MakeMsg(clsAddressBook, 9)

#define abServiceDescFields							\
	CHAR					name[nameBufLength];	\
	U16						maxPerEntry;			\
	U16						numAttrs;				\
	P_ADDR_BOOK_ATTR_DESC	attrs;					\

typedef struct ADDR_BOOK_SVC_DESC {
	abServiceDescFields
} ADDR_BOOK_SVC_DESC, *P_ADDR_BOOK_SVC_DESC;

typedef struct ADDR_BOOK_SERVICES {
		OS_HEAP_ID				heap;
		U16						numServices;
		P_ADDR_BOOK_SVC_DESC	services;
} ADDR_BOOK_SERVICES, *P_ADDR_BOOK_SERVICES;

/****************************************************************************
 msgAddrBookEnumGroupMembers takes P_ADDR_BOOK_ENUM_GROUP_MEMBER, returns STATUS
    Enumerates through the members in a group.

 Parameters:
	pArgs->groupKey:	In: key of the group
	pArgs->startKey:	In: where to start the group enumeration.  Use pNull
							to start from the beginning.
						Out:last entry key returned in pArgs->pKeys.  Client
							usually uses the out value to be the next in value
							of the next msgAddrBookEnumGroupMembers call.
	pArgs->recurse:		In: whether to recursively enumerate groups
	pArgs->sort:		In: attr id of the field to sort the returned entry
							id by
	pArgs->count:		In:	number of entries to return, which is also the
							number of slots in the pKeys array. Use
							AddrBookAll to get every member.  In this case
							address book will allocate the necessary space,
							and the client should free the space.
						Out:number of entries actually returned
	pArgs->pKeys:		Out:keys of the members of pArgs->groupKey
								
 */
#define msgAddrBookEnumGroupMembers			MakeMsg(clsAddressBook, 6)

typedef struct ADDR_BOOK_ENUM_GROUP_MEMBER {
	ADDR_BOOK_KEY			groupKey;	
	ADDR_BOOK_KEY			startKey;
	BOOLEAN					recurse;
	ADDR_BOOK_ATTR_ID		sort;	
	U32						count;	
	P_ADDR_BOOK_KEY			pKeys;
} ADDR_BOOK_ENUM_GROUP_MEMBER, *P_ADDR_BOOK_ENUM_GROUP_MEMBER;

/****************************************************************************
 msgAddrBookIsAMemberOf takes P_ADDR_BOOK_IS_A_MEMBER_OF, returns STATUS
	Determines if an entry is a member of a group.

 Parameters:
	pArgs->groupKey:	In: key of the group
	pArgs->memberKey:	In: potential member's key
	pArgs->recurse:	In: whether to recursively test for membership

 Return Value
    stsOK:  if pArgs->memberKey is a member of pArgs->groupKey.
	stsNoMatch:  if pArgs->memberKey is not a member of pArgs->groupKey
 */
#define msgAddrBookIsAMemberOf			MakeMsg(clsAddressBook, 7)

typedef struct ADDR_BOOK_IS_A_MEMBER_OF {
	ADDR_BOOK_KEY		groupKey;
	ADDR_BOOK_KEY		memberKey;
	BOOLEAN				recurse;
} ADDR_BOOK_IS_A_MEMBER_OF, *P_ADDR_BOOK_IS_A_MEMBER_OF;

/****************************************************************************
 msgAddrBookGetMetrics	takes P_ADDR_BOOK_METRICS, returns STATUS
	Passes back the metrics for the address book.

*/
#define msgAddrBookGetMetrics				MakeMsg(clsAddressBook, 8)

typedef struct ADDR_BOOK_METRICS {
	U32			numEntries;		// Total number of entries
	U32			numGroups;		// Number of groups in the address book
	U16			numServices;	// Number of known services
	U32			spare1;
	U32			spare2;
} ADDR_BOOK_METRICS, *P_ADDR_BOOK_METRICS;

/****************************************************************************
 msgAddrBookAddAttr	takes P_ADDR_BOOK_ATTR,	returns STATUS
 	Adds a new attribute to active address books.  

 This operation will
 change the address book database schema.  If the attribute is of type
 abNumber, the value is initialized to be 0 for all existing address
 book entries.  If the attribute is of type abPhoneNumber, then the
 value is intialized to be 0.  If the attribute is of type abString
 or abOther, the value is initialized to be 0 length byte array.

 After an attribute is added to an address book, clients can then set
 the attribute value in subsequent msgAddrBookSet's and get the attribute
 value in the subsequent msgAddrBookGet's.  Failure to first make an
 attribute known to an address book and then try to set or get the
 attribute value will cause stsAddrBookInvalidAttr to be returned.

 Parameters:
	pArgs->id:		In: the id(should be a tag) of the new attribute.  
							It has to be different from all other attribute
							ids in the same address book.
	pArgs->type:	In: one of abNumber, abString, abOther, abPhoneNumber
	pArgs->label: 	In: a string, for display purpose.  The address book 
						will copy the string to its own storage.

 Return Value
    stsRequestNotSupported: 	if the address book does not allow
								dynamically changing its database schema.
	stsAddrBookDuplicateAttrId: There is another attribute in the 
								address book w/ the same id.
*/
#define msgAddrBookAddAttr			MakeMsg(clsAddressBook, 12)

/****************************************************************************
 msgAddrBookCount	takes P_ADDR_BOOK_COUNT, returns STATUS
	Finds the number of entries that match the search spec

 Parameters:
	pArgs->key			In: where to stop counting, AddrBookAll to count the
							entire database
	pArgs->dir			In: whether to start counting from the beginning or
							the end of the address book.
	pArgs->query		In:	qualifier.  See msgAddrBookSearch
	pArgs->count		Out:number of entries that satisfy pArgs->query
*/
#define msgAddrBookCount						MakeMsg(clsAddressBook, 13)

typedef struct ADDR_BOOK_COUNT {
	ADDR_BOOK_KEY			key;
	ADDR_BOOK_ATTR_ID		sort;
	ADDR_BOOK_SEARCH_DIR	dir;
	ADDR_BOOK_QUERY			query;
	U16						count;
} ADDR_BOOK_COUNT, *P_ADDR_BOOK_COUNT;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Observer Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgAddrBookEntryChanged	takes P_ADDR_BOOK_ENTRY_CHANGE, returns STATUS
	Sent to observers when an entry has been changed, added or deleted.
	
 If pArgs->type is abServiceChanged, abServiceDeleted, abServiceAdded, then
 the address book fills in pArgs->svcId to be the id of the service address
 affected.  pArgs->entryKey is filled in by the address book except when
 pArgs->type is abServiceInstalled or abServiceDeinstalled.  In that case, 
 the address book is notifying clients that some service has been installed
 or deinstalled,	and the service information returned by the previous 
 msgAddrBookGetServiceDesc is no longer up-to-date.
*/
#define msgAddrBookEntryChanged				MakeMsg(clsAddressBook, 11)

Enum16(ADDR_BOOK_CHANGE_TYPE) {
	abServiceChanged		= 0,
	abServiceDeleted		= 1,
	abServiceAdded			= 2,
	abEntryAdded			= 3,
	abEntryDeleted			= 4,
	abEntryNameChanged		= 5,
	abEntryChanged			= 6,
	abServiceInstalled		= 7,	// svcs have been installed
	abServiceDeinstalled	= 8,	// svcs have been deinstalled
};

typedef struct ADDR_BOOK_ENTRY_CHANGE {
	OBJECT					addrBook;	// Address book UID
	ADDR_BOOK_CHANGE_TYPE	type;		// Type of change
	ADDR_BOOK_KEY			entryKey;	// Internal address book key of the
										// changed entry
	ADDR_BOOK_SERVICE_ID	svcId;		// service id, if applicable
} ADDR_BOOK_ENTRY_CHANGE, *P_ADDR_BOOK_ENTRY_CHANGE;

#endif
