/****************************************************************************
 dvparall.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The public definition of the parallel port device interface.

 $Revision:   1.9  $
   $Author:   thall  $
     $Date:   24 Mar 1992 15:12:04  $
****************************************************************************/
#ifndef DVPARALL_INCLUDED
#define DVPARALL_INCLUDED

#ifndef MIL_INCLUDED
#include <mil.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the parallel device. 
 The definitions contained here are data structures that are necessary
 for the MIL parallel specific functions along with standard
 MIL device data structures with additional fields specific to the
 parallel device.
*/

/**** Parallel Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the parallel device. The function numbers start immediately after
 the last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.

*/

#define milParallelGetDevParameters		milLastCommonFunction+1
#define milParallelSetDevParameters		milLastCommonFunction+2
#define milParallelInitPrinter			milLastCommonFunction+3
#define milParallelWrite				milLastCommonFunction+4

#define fnCountParallel					fnCountBase+4

/**** Parallel Device Function Transfer Table ****/
/*
 This structure is the declaration of the Parallel Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for the MIL Parallel device.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/

typedef struct MIL_FTT_PARALLEL
{
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountParallel];
} MIL_FTT_PARALLEL, *P_MIL_FTT_PARALLEL;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 					Parallel Device Specific Functions 	 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are the parallel device specific
 functions. Access to these functions is provided by the FTT defined
 above. All of the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/


/****************************************************************************
 milParallelGetDevParameters		returns STATUS
 	Provides device specific information about the parallel device.

 This function is a single stage function that provides device
 specific information about the parallel device.
*/

/**** Function Specific Data Structures ****/
/*
 The following data structures are used in the milParallelGetDevParameters
 and milParallelSetDevParameters function calls.
*/

/*
 Device Parameters Data Structure

 The following fields are defined in the MIL_PARALLEL_DEVICE_PARAMETERS
 data structure.

	parallelDeviceFlags:
		This field contains flags that define the capabilities and
		characteristics of the parallel device. There are no values
		defined for this field at this time.

*/
typedef struct MIL_PARALLEL_DEVICE_PARAMETERS
{
	U16 parallelDevFlags;
	U16 reservedU16[2];
	U32 reservedU32[2];
} MIL_PARALLEL_DEVICE_PARAMETERS, *P_MIL_PARALLEL_DEVICE_PARAMETERS;


/*
 parallelStatus

 The following bit definitions are used for the parallelStatus field of the
 MIL_PARALLEL_UNIT_PARAMETERS data structure defined below.

	parallelStatusInterrupt:
		This flag indicates that an interrupt that has occurred was a 
		status interrupt.

	parallelStatusIOError:
		This flag indicates that an I/O error occurs. It is active low.

	parallelStatusSelected:
		This flag indicates the parallel device is selected.

	parallelStatusOutOfPaper:
		This flag indicates the parallel port device is out of paper.

	parallelStatusAck:
		This flag indicates the output character was acknowledged.

	parallelStatusBusy:
		This flag indicates the output device is busy.

*/

#define parallelStatusInterrupt		flag2			// an interrupt occurred
#define parallelStatusIOError		flag3			// an IO error occurred
#define parallelStatusSelected		flag4			// device is selected
#define parallelStatusOutOfPaper	flag5			// device is out of paper
#define parallelStatusAck			flag6			// character acknowledge
#define parallelStatusBusy			flag7			// device busy

/*
 parallelUnitFlags

 The following unit flags are defined for the parallelUnitFlags field
 in the MIL_PARALLEL_UNIT_PARAMETERS.

	parallelAutoLineFeed:
		This flag indicates the mil driver will add a line feed
		after a carriage return.

*/

#define parallelAutoLineFeed		flag0			// cause line feed after

/*

 Unit Parameters Data Structure

 The MIL_PARALLEL_UNIT_PARAMETERS data structure contains the
 following fields.

	parallelUnitFlags:
		This field contains flags that specify the capabilities and
		characteristics of the parallel unit. The bits currently
		defined are given above.

	parallelStatus:
		This field contains flags that indicate the current status
		of the parallel port. The flags are defined above.

*/

typedef struct MIL_PARALLEL_UNIT_PARAMETERS
{
	U16 parallelUnitFlags;
	U16 parallelStatus;			// contents of parallel port status register
	U16 reservedU16[2];
	U32 reservedU32[2];
} MIL_PARALLEL_UNIT_PARAMETERS, *P_MIL_PARALLEL_UNIT_PARAMETERS;


/*

 Time Intervals Data Structure

 The MIL_PARALLEL_TIME_INTERVALS data structure contains the
 following fields.

	initDelay:
		This field indicates the width of the initialization
		pulse in microseconds.

	interruptTimeOut:
		This field specifies the device should be busy after receiving
		a character (in miliseconds).

*/

typedef struct MIL_PARALLEL_TIME_INTERVALS
{
	U32	initDelay;				// width of initialization pulse
								// (in micro-seconds)
	U32	interruptTimeOut;		// maximum time device should be busy after
								// receiving a character (in milli-seconds)
	U16	funcReservedU16[2];
	U32	funcReservedU32[2];
} MIL_PARALLEL_TIME_INTERVALS, *P_MIL_PARALLEL_TIME_INTERVALS;


/**** Device Specific Request Block Fields ****/
/*

 The milParallelGetDevParameters funtion request block contains the
 following fields.

	parallelDevParms:
		This field is a MIL_PARALLEL_DEVICE_PARAMETERS data structure
		used to return information about the parallel device. This
		data structure is defined above.
		
	parallelUnitParms:
		This field is a MIL_PARALLEL_UNIT_PARAMETERS data structure
		used to return information about the parallel unit. This
		data structure is defined above.

	parallelTimeIntervals:
		This field is a MIL_PARALLEL_TIME_INTERVALS data structure
		used to return information about the operation delays required.
		to access the parallel port device. This data structure is
	    defined above.

*/

#define rbPublicFieldsParallelDeviceParameters				\
	MIL_PARALLEL_DEVICE_PARAMETERS	parallelDevParms;		\
	MIL_PARALLEL_UNIT_PARAMETERS	parallelUnitParms;		\
	MIL_PARALLEL_TIME_INTERVALS		parallelTimeIntervals;	\
	U16 funcReservedU16[2];									\
	U32 funcReservedU32[2];


/*
 
 milParallelGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_PARALLEL_DEVICE_PARAMETERS
{
	milRequestBlockCommonFields
	rbPublicFieldsParallelDeviceParameters
} MIL_RB_PUB_PARALLEL_DEVICE_PARAMETERS,
									*P_MIL_RB_PUB_PARALLEL_DEVICE_PARAMETERS;


/****************************************************************************
 milParallelSetDevParameters		returns STATUS
 	Allows setting device specific information of the parallel device.

 This function is a single stage function that provides the capability
 to set the specific information of the parallel device.

 This function uses the same request block as the milParallelGetDevParameters
 function defined above.

*/

/****************************************************************************
 milParallelInitPrinter		returns STATUS
 	Initializes the parallel port printer.

 This function performs an initialization operation on the parallel port
 printer.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milParallelInitPrinter funtion request block contains the
 following fields.

	parallelStatus:
		This field returns that status of the printer after the
		initialization operation. The bits in this field are
		defined above.

*/

#define rbPublicFieldsParallelInitPrinter	\
	U16 parallelStatus;						\
	U16 funcReservedU16[2];					\
	U32 funcReservedU32[2];


/*
 
 milParallelPrinterInit Request Block

*/

typedef struct MIL_RB_PUB_PARALLEL_INIT_PRINTER
{
	milRequestBlockCommonFields
	rbPublicFieldsParallelInitPrinter
} MIL_RB_PUB_PARALLEL_INIT_PRINTER, *P_MIL_RB_PUB_PARALLEL_INIT_PRINTER;

/****************************************************************************
 milParallelWrite		returns STATUS
 	Write data to the parallel printer.

 This function writes the specified data to the parallel printer.

 NOTE: In mil for PC if the boolean dataIsBuffer is false and the bufferSize
 is 1, the character is written without using interrupts.  The countPrinted
 field indicates whether the character was printed or not.  Otherwise,
 the routine does not wait for the device to be ready to accept the
 character.  If more than one character is to be printed, interrupts are
 used to print the characters.

*/

/**** Device Specific Request Block Fields ****/
/*

 The milParallelWrite funtion request block contains the
 following fields.

	data:
		This field is either 1 to 8 characters or a pointer to a
		buffer containing the data to be printed.

	dataIsBuffer:
		This field indicates the data field is a data buffer if
		it is set TRUE.

	countPrinted:		
		This field contains the number of characters actually printed

	parallelStatus:
		This field contains the contents of the parallel port status
		register on the last write
*/

#define rbPublicFieldsParallelWrite			\
	MIL_PTR_OR_BUF		data;				\
	BOOLEAN				dataIsBuffer;		\
	U32					bufferSize;			\
	U32					countPrinted;		\
	U16					parallelStatus;		\
	U16					funcReservedU16[2];	\
	U32					funcReservedU32[2];

/*
 
 milParallelWrite Request Block

*/

typedef struct MIL_RB_PUB_PARALLEL_WRITE
{
	milRequestBlockCommonFields
	rbPublicFieldsParallelWrite
} MIL_RB_PUB_PARALLEL_WRITE, *P_MIL_RB_PUB_PARALLEL_WRITE;


#endif 	// DVPARALL_INCLUDED
