/****************************************************************************
 dvstylus.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The public definition of the stylus device interface.

 $Revision:   1.9  $
   $Author:   thall  $
     $Date:   24 Mar 1992 15:11:28  $
****************************************************************************/
#ifndef DVSTYLUS_INCLUDED
#define DVSTYLUS_INCLUDED

#ifndef PEN_INCLUDED
#include <pen.h>
#endif
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the stylus device. 
 The definitions contained here are data structures that are necessary
 for the MIL stylus specific functions along with standard
 MIL device data structures with additional fields specific to the
 stylus device.
*/

/**** Stylus Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the stylus device. The function numbers start immediately after
 the last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.

*/

#define milStylusGetDevParameters	milLastCommonFunction+1
#define milStylusReadCont			milLastCommonFunction+2
#define milStylusSetSampleRate		milLastCommonFunction+3
#define milStylusSetResolution		milLastCommonFunction+4
#define milStylusSetThreshold		milLastCommonFunction+5

#define fnCountStylus				fnCountBase+5


/**** Stylus Device Function Transfer Table ****/
/*
 This structure is the declaration of the Stylus Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for the MIL Stylus device.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/



typedef struct MIL_FTT_STYLUS {
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountStylus];
} MIL_FTT_STYLUS, *P_MIL_FTT_STYLUS;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 					Stylus Device Specific Functions	   	 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are the stylus device specific
 functions. Access to these functions is provided by the FTT defined
 above. All of the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/


/****************************************************************************
 milStylusGetDevParameters		returns STATUS
 	Provides device specific information about the stylus device.

 This function is a single stage function that provides device
 specific information about the stylus device.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milStylusGetDevParameters function request block contains the
 following fields.

	minResolution:
		This field is the minimum resolution supported by the device
		in lines per meter.

	maxResolution:		
		This field is the maximum resolution supported by the device
		in lines per meter.

	currentResolution:
		This field is the current resolution set on the device.


	maxXPosition:
		This field is the maximum supported X position on the device.

	maxYPosition:
		This field is the maximum supported Y position on the device.

	xPosition:
		This field is the current X position on the device.

	yPosition:
		This field is the current Y position on the device.

	deviceFlags:
		This field defines capabilities, status, and modes of 
		operation for the stylus device. Currently there are no 
		bits defined for this field.

	penTipState:
		This field defines the current pen tip state. This field
		identifies if the pen is down, or in or out of proximity.
		The definitions for this field can be found in pen.h.

	stylusSupportsFlags:
		This field defines the capabilities of the stylus. The bits
		used in this field are defines in pen.h.

	lowSampleRate:
		This field defines the low sampling rate.

	medSampleRate:
		This field defines the medium sampling rate.

	highSampleRate:
		This field defines the high sampling rate.

	currentSampleRate:
		This field defines the current sampling rate.

	reportingThreshold:
		This field represents a threshold time measured between a 
		down and up stroke for generating a tap.

	deviceId:
		
*/

typedef struct MIL_STYLUS_DEVICE_PARAMETERS {
	U32					minResolution;			//	lines per meter
	U32					maxResolution;			//	lines per meter
	U32					currentResolution;		//	lines per meter
	U32					maxXPosition;			//	using pen resolution
	U32					maxYPosition;			//	using pen resolution
	U32					xPosition;				//	using pen resolution
	U32					yPosition;				//	using pen resolution
	U32					deviceFlags;
	U32 				reservedU32[2];
	PEN_TIP_STATE_FLAGS	penTipState;
	PEN_SUPPORTS_FLAGS	stylusSupportsFlags;
	U16					lowSampleRate;
	U16					medSampleRate;
	U16					highSampleRate;
	U16					currentSampleRate;
	U16					reportingThreshold;		//	using pen resolution
	U16					deviceId;
	U16					reservedU16[4];
} MIL_STYLUS_DEVICE_PARAMETERS, *P_MIL_STYLUS_DEVICE_PARAMETERS;

/*
 rbPublicFieldsStylusGetDevParameters

 This define merges the device specific data structure with the standard
 MIL device parameters reserved fields.

*/

#define rbPublicFieldsStylusGetDevParameters \
	MIL_STYLUS_DEVICE_PARAMETERS stylusDevParms; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milStylusGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_STYLUS_GET_DEV_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsStylusGetDevParameters
} MIL_RB_PUB_STYLUS_GET_DEV_PARAMETERS, *P_MIL_RB_PUB_STYLUS_GET_DEV_PARAMETERS;


/****************************************************************************
 milStylusReset		returns STATUS
 	Reset the stylus device.

 This function resets the stylus device.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milStylusReset request block does not contain any
 device or function specific fields.

*/

#define rbPublicFieldsStylusReset \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milStylusReset Request Block

*/

typedef struct MIL_RB_PUB_STYLUS_RESET {
	milRequestBlockCommonFields
	rbPublicFieldsStylusReset
} MIL_RB_PUB_STYLUS_RESET, *P_MIL_RB_PUB_STYLUS_RESET;


/****************************************************************************
 milStylusReadCont		returns STATUS
 	Reports the position of the stylus.

 This function is a continuous function that reports current stylus
 information.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milStylusReadCont request block contains the following fields.
	
	timeStamp:
		This field contains a timestamp that specifies the time the
		stylus event occured.

	xPosition:
		This field specifies the current x position of the stylus.

	yPosition:
		This field specifies the current y position of the stylus.

	pAsyncEventFunc:
		This field is a posinter to the call back function.

	stylusTipState:
		This field specifies the current state of the stylus tip.
		This state can be down, or in or out of proximity. See pen.h
		for the definition of this field.

	zPosition:
		This field specifies the current z position. This field is
		intended to specify the angle of the stylus.

	pressure:
		This field specifies the stylus pressure on the tablet surface.

	penId:
		This field is the ID of the pen device.
	
	xyAngle:
		This field specifies the angle of the pen in the tablet surface
		plane.
	
	zAngle:
		This field specifies the angle of the pen relative to a line
		perpendicular to the tablet surface.

*/

#define rbPublicFieldsStylusReadCont		\
	U32 timestamp;							\
	U32 xPosition;							\
	U32 yPosition;							\
	U32 funcReservedU32[2];					\
	P_MIL_ASYNC_EVENT_FUNC pAsyncEventFunc;	\
	PEN_TIP_STATE_FLAGS stylusTipState; 	\
	U16 zPosition;							\
	U16	pressure;							\
	U16 penId;								\
	U16 xyAngle;							\
	U16	zAngle;								\
	U16 funcReservedU16[2];

/*
 
 milStylusReadCont Request Block

*/

typedef struct MIL_RB_PUB_STYLUS_READ_CONT {
	milRequestBlockCommonFields
	rbPublicFieldsStylusReadCont
} MIL_RB_PUB_STYLUS_READ_CONT, *P_MIL_RB_PUB_STYLUS_READ_CONT;


/****************************************************************************
 milStylusSetSampleRate		returns STATUS
 	Sets the stylus sampling rate.

 This function sets the rate the stylus hardware samples the stylus state
 information.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milStylusSetSampleRate request block contains the following fields.
	
	sampleRate:
		This field specifies the requested sampling rate.

*/

#define rbPublicFieldsStylusSetSampleRate \
	U16 sampleRate; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milStylusSetSampleRate Request Block

*/

typedef struct MIL_RB_PUB_STYLUS_SET_SAMPLE_RATE {
	milRequestBlockCommonFields
	rbPublicFieldsStylusSetSampleRate
} MIL_RB_PUB_STYLUS_SET_SAMPLE_RATE, *P_MIL_RB_PUB_STYLUS_SET_SAMPLE_RATE;

/****************************************************************************
 milStylusSetResolution		returns STATUS
 	Sets the stylus resolution.

 This function sets the resolution of the stylus device.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milStylusSetResolution request block contains the following fields.
	
	resolution:
		This field specifies the requested resolution.

*/


#define rbPublicFieldsStylusSetResolution \
	U16 resolution; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milStylusSetResolution Request Block

*/

typedef struct MIL_RB_PUB_STYLUS_SET_RESOLUTION {
	milRequestBlockCommonFields
	rbPublicFieldsStylusSetResolution
} MIL_RB_PUB_STYLUS_SET_RESOLUTION, *P_MIL_RB_PUB_STYLUS_SET_RESOLUTION;


/****************************************************************************
 milStylusSetThreshold		returns STATUS
 	Sets the stylus threshold.

 This function sets the threshold of the stylus device.
 The set-threshold function is designed to reduce pen-event traffic.
 Pen movement below the set threshold need not be reported to PenPoint.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milStylusSetThreshold request block contains the following fields.
	
	reportingThreshold:
		This field specifies the reporting threshold. Stylus events below
		this threshold will not be reported.

*/

#define rbPublicFieldsStylusSetThreshold \
	U16 reportingThreshold; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milStylusSetThreshold Request Block

*/

typedef struct MIL_RB_PUB_STYLUS_SET_THRESHOLD {
	milRequestBlockCommonFields
	rbPublicFieldsStylusSetThreshold
} MIL_RB_PUB_STYLUS_SET_THRESHOLD, *P_MIL_RB_PUB_STYLUS_SET_THRESHOLD;

#endif
