/****************************************************************************
 File: fontmgr.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.47  $
   $Author:   rfoster  $
     $Date:   22 Feb 1992 15:34:50  $

 This file contains the API definition for clsFontInstallMgr.
 clsFontInstallMgr inherits from clsInstallMgr.
 It performs font installation and maintenance.
 
 There is a single instance of clsFontInstallMgr in the system; the 
 well-known uid theInstalledFonts.

 The font manager maintains the installed and deinstalled fonts on the
 system.  The font manager differs from a generic install manager
 in the area of font identification and the system font.

 A font is a structured file.  The system comes with several pre-defined 
 font files that are loaded at cold boot time.

 Font files typically reside in the \penpoint\font directory on a given
 filesystem volume.  This is not a requirement, however.
 
 Fonts are identified in four ways: 
	-:	a font file handle
	-:	a short font ID
	-:	a string font ID
	-:	the name of a font file

 Font file handles are open file handles on to the font files.  Much of the 
 install manager interface uses these handles.  A short font ID is 
 a pre-defined, 16 bit value that identifies a specific font.  It is a 
 compact, specific reference for a particular font.  The window system API 
 uses short font IDs.  A string font ID is a 4 character string version of a 
 short font ID.  The font file name is the user-visible name for the font.
 Given a handle, you can get the font file name by sending msgIMGetName.
 Given a short font ID, you can get the font file name by sending 
 msgFIMGetNameFromId.

 NOTE: THE MESSAGES IN THIS CLASS ARE SENT TO THE MANAGER, NOT TO 
 THE HANDLES.

 A list of all the font handles in the system is available via superclass
 message msgIMGetList.  A pruned list of the fonts that is appropriate
 for end-user display is available via msgFIMGetInstalledIDList.
 
 The following messages are not understood by clsFontInstallMgr:
	-:  msgIMGetCurrent
	-:  msgIMSetCurrent
	-:  msgIMDup

 The following notification messages are not sent by clsFontInstallMgr:
	-:  msgIMCurrentChanged

 See Also
	instlmgr.h

****************************************************************************/
#ifndef FONTMGR_INCLUDED
#define FONTMGR_INCLUDED

#ifndef INSTLMGR_INCLUDED
#include <instlmgr.h>
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  		Common #defines and typedefs  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Filesystem attribute definitions  ****/
/* 
 Note: Most clients do not deal with attributes directly.
*/

/* Font ID */
#define fimAttrId						FSMakeStrAttr(clsFontInstallMgr, 0)

/* Font ID definitions */
typedef	U16		FIM_SHORT_ID, *P_FIM_SHORT_ID;

typedef struct FIM_LONG_ID {
	U8			pId[5];
} FIM_LONG_ID, *P_FIM_LONG_ID;

/*
 FIM_GET_SET_ID is used by msgFIMGetId and msgFIMSetId.
*/
typedef struct FIM_GET_SET_ID {
	IM_HANDLE			handle;			// Font handle to get IDs on.
	FIM_SHORT_ID		id;				// Out: short version of ID.
	FIM_LONG_ID			longId;			// Out: long ID version.
} FIM_GET_SET_ID, *P_FIM_GET_SET_ID;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  				  	Messages  		  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew						takes P_FIM_NEW, returns STATUS
	category: class message
	Creates a new font install manager.

 There is only one instance of this class, theInstalledFonts, in the system.
 Clients should never send msgNew.
*/

typedef struct FIM_NEW {
	installMgrNewFields
} FIM_NEW, *P_FIM_NEW;


/****************************************************************************
 msgNewDefaults					takes P_FIM_NEW, returns STATUS
	category: class message
    Initializes the FIM_NEW structure to default values.

 Sets
	installMgr.fileMode |= fsReadOnly | fsSystemFile;
*/


/****************************************************************************
 msgFIMGetId					takes P_FIM_GET_SET_ID, returns STATUS
	Gets the short and long font IDs, given a handle.

*/
#define msgFIMGetId			 				MakeMsg(clsFontInstallMgr, 3)


/****************************************************************************
 msgFIMSetId					takes P_FIM_GET_SET_ID, returns STATUS
	Set the font file's ID.

 If the short version of the ID is 0 then the long version of the ID is used.
	
 Note: A font ID is not normally changed.  This message is here to allow a
 tool that edits font IDs to be written.
*/
#define msgFIMSetId			 				MakeMsg(clsFontInstallMgr, 4)


/****************************************************************************
 msgFIMFindId					takes P_FIM_FIND_ID, returns STATUS
	Finds a font handle given a short ID.

 Return Value
	stsNoMatch:		font handle not found.
*/
#define msgFIMFindId	 	 				MakeMsg(clsFontInstallMgr, 5)

typedef struct FIM_FIND_ID {
	FIM_SHORT_ID		id;			// ID, short form
	IM_HANDLE			handle;		// Out: resulting handle
} FIM_FIND_ID, *P_FIM_FIND_ID;


/****************************************************************************
 msgFIMGetNameFromId			takes P_FIM_GET_NAME_FROM_ID, returns STATUS
	Passes back font name given an short ID.

 Return Value
	stsNoMatch:		short ID not found.

 See Also
	msgIMGetName:		Gets the name given a handle.
*/
#define msgFIMGetNameFromId	 				MakeMsg(clsFontInstallMgr, 6)

typedef struct FIM_GET_NAME_FROM_ID {
	FIM_SHORT_ID		id;					
	P_STRING			pName;	// Out: name, max size is nameBufLength
} FIM_GET_NAME_FROM_ID, *P_FIM_GET_NAME_FROM_ID;


/****************************************************************************
 msgFIMGetInstalledIdList	takes P_FIM_GET_INSTALLED_ID_LIST, returns STATUS
	Passes back a list of the short IDs of all installed fonts.
 
 This list is pruned so that it is useable as a user pick list.
 For example, if both Helvetica and Helvetica Bold are in the system, 
 only Helvetica is on this list.

 THE CALLER MUST DESTROY THE LIST OBJECT WHEN IT IS FINISHED USING IT.

 See Also
	msgIMGetList:		Gets a list of all handles.
		
*/
#define msgFIMGetInstalledIdList   			MakeMsg(clsFontInstallMgr, 7)

typedef enum FIM_PRUNE_CONTROL {
	fimNoPruning 		= 0,
	fimPruneDupFamilies	= flag1,
	fimPruneSymbolFonts	= flag2
} FIM_PRUNE_CONTROL, *P_FIM_PRUNE_CONTROL;

typedef struct FIM_GET_INSTALLED_ID_LIST {
	FIM_PRUNE_CONTROL	prune;		// What sort of pruning should be done
	OBJECT				list;		// Out: list
} FIM_GET_INSTALLED_ID_LIST, *P_FIM_GET_INSTALLED_ID_LIST;


#endif	// FONTMGR_INCLUDED


