/****************************************************************************
 input.h
 Copyright 1992, GO Corporation, All Rights Reserved

 $Revision:   1.19  $
   $Author:   kbentley  $
	 $Date:   26 Mar 1992 20:26:02  $

 This file contains the API definition for clsInput and PenPoint's input
 system.

 clsInput inherits from clsObject.

 clsInput provides the object-oriented interface to PenPoint's input
 system.

 The functions described in this file are contained in INPUT.LIB. 
****************************************************************************/

/**** Introduction ****/
/*
 PenPoint's input system collects events generated by devices such as
 thePen and theKeyboard.  It then distributes those events to other objects
 in the system.
 
 The input system is almost always single-threaded.  Usually only one input
 event is being distributed through the system at any given time.  The
 exception is when using msgInputModalStart and msgInputModalEnd.
*/


/**** Road Map ****/
/*
 This file contains general information about PenPoint's input system and
 input events.  Information specific to pen events is in pen.h. Information
 specific to key events is in key.h.

 Most PenPoint application programs do not need to use the PenPoint input
 system directly.  PenPoint has several classes that manage input for
 clients.  Check these classes to see if they meet your needs.  Candidate
 classes include the following (and their subclasses).  (This list is not
 exhaustive.)
	-:	clsGWin (gwin.h)
	-:	clsSPaper (spaper.h)
	-:	clsIP (insert.h)
	-:	clsNotePaper (notepapr.h)
	-:	all toolkit classes.

 Any client handling input directly (rather than using PenPoint classes
 which handle input) needs to understand the following:
    -:  How to set up window input flags so that desired input events are
        received.  See the section "Input Flags."
    -:  How to handle msgInputEvent in general, and how to handle the
        device-specific values for msgInputEvent's pArgs->devCode.  See
        pen.h and key.h.
    -:  How to return appropriate status values in response to
        msgInputEvent.  See "Return Values From msgInputEvent."

 Any client that needs to grab input needs to understand:
    -:  General grabbing information.  See the section "Grabs and
        Grabbers."
    -:  msgInputEvent return values that start a grab and keep a grab going.
        See "Return Values From msgInputEvent."

 Any client that needs to be the input target (and therefore the recipient
 of keyboard events) needs to understand:
	-:	InputSetTarget()

 The other interfaces described in this file are typically used by
 sophisticated clients.
*/


/**** Overview ****/
/*
 This diagram illustrates the flow of events into, through and out of the
 input system:
//{
       +-----------+     +-----------+     +---------------+
       |    Pen    |     | Keyboard  |     | Other Device  |
       +-----+-----+     +-----+-----+     +-------+-------+
             |                 |                   |        
             +-----------------|-------------------+
                               |
                            +-----+  
                            |-----|  Input
                            |-----|  Queue
                            +-----+                    Input
                               |                       Routing
       +-----------------------|-----------------------------+
       |                       |                             |
       |                  +----+----+                        |
       |                  | Filters |                        |
       |                  +----+----+                        |
       |                       |              +----+         |
       |                  +---------+         |----| Grabber |
       |                  | Grabber |- - - - -|----| Stack   |
       |                  +----+----+         +----+         |
       |                       |                             |
       |            +---------------+-------------+          |
       |            |               |             |          |
       |    +-------+-------+  +----+-----+  +----+-----+    |
       |    | Pre-specified |  |    XY    |  |  Target  |    |
       |    |  Destination  |  |          |  |          |    |
       |    +---------------+  +----------+  +----------+    |
       |                                                     |
       +-----------------------------------------------------+
//}

 Each of these major pieces is described below:

    -:  Devices such as the pen and keyboard generate low-level input
        events.  (These "devices" are partially implemented in the MIL and
        partially implemented in PenPoint.)  These low-level input events
        are converted into PenPoint input events and are sent to the Input
        Queue.

    -:  The input system pulls events off the queue one at a time and 
        decides where to send or "route" the event.

 There the "event routing" process starts.

    -:  First the event is run through the list of Filters.  Filters have
        the opportunity to examine each input event.  Filters are ordered
        by their priority.  Filters return a status which indicates how
        processing of the event should continue.
		
    -:  Next the event is sent to the current grab object, or grabber.
        (There might not be a current grabber, in which case this step is
        skipped.)  The grabber returns a status which indicates how
        processing of the event should continue and whether the grab should
        continue.  The input system maintains a stack of grabbers to
        support nested modal behavior.

 The next step in an event's routing depends in part on the event.  (Only
 one of these alternatives is used.)

    -:  If the event has a pre-specified destination, msgInputEvent is
		sent to that destination.  If the event has a pre-specified
		destination, it is found in pArgs->listener for msgInputEvent.
		An event has a pre-specified destination only if the event has
		been programmatically inserted into the input system.

    -:	If the event has a "valid" XY coordinate (which typically means it
		was generated by thePen), the event is routed to window objects. 
        The top-most window (farthest from theRootWindow) which encloses
		the XY coordinate gets the first opportunity to process the event.
		Each window may terminate processing of the event or allow the
		input system to send the event to its (the window's) parent window.

    -:  Otherwise the event is sent to the current input target, if the 
        target is non-null.  (This is how all keyboard events are routed.)
*/


/**** Filters ****/
/*
 Filters are used to implement some types of modal behavior.  Typically
 this modal behavior is relatively long-lived.  For instance, PenPoint's
 Quick Help mode is implemented using filters.  
		
 It is extremely rare for PenPoint application programs to directly use
 or even be aware of filters.

 Object			Priority
 ------			--------
 qhelp win		16
 vkey win		32
 qhelp nb		32
 vkey app		80
 spell (proof)	96
 insertion pad	96 ( if modal )
 menu			112
 note			160
 option			160

*/


/**** Grabs and Grabbers ****/
/*
 Grabbers are used to implement light-weight modal behavior.  These modes
 are typically pen controlled in that they start and end with some pen event,
 such as msgPenDown and msgOutProxUp.  For instance resize handles are
 implemented using grabbers.
 
 Many application programs never use grabbers directly but rather use
 PenPoint classes that use grabbers.

 As illustrated in the "Overview" section, the current grabber gets input
 messages after filters but before "normal" event distribution occurs.  The
 grabber can "swallow" the event and stop any further distribution, or the
 grabber can allow distribution to continue.

 There are two ways to start a grab.
 
    -:  An object that is handling msgInputEvent can return a status value
        that tells the input system that it wants to be the grab object. 
        See the section "Return Values from msgInputEvent."

    -:  Any object can call InputSetGrab() passing in the object to become
        the grabber.

 A grabber terminates a grab by returning from msgInputEvent a status value
 that does not have "Grab" turned on, or by setting the current grabber to
 objNull.

 In order to keep the grab "alive," the grabber must always return a
 status from msgInputEvent that implies "Grab."  If the input system gets a
 status returned that does not have "Grab" implied, it terminates
 the grab.
*/


#ifndef INPUT_INCLUDED
#define INPUT_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef UID_INCLUDED
#include <uid.h>
#endif

#ifndef GEO_INCLUDED
#include <geo.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Miscellaneous ****/

/* 
 * The following flags control event distribution to filters.
 *
 * iflSendMyWindowOnly tells the input system to not bother sending the
 * message to the filter unless the event happened in the filter or in one
 * of the filter's window children or window ancestors.  It is strictly a
 * performance enhancement.
*/
typedef U32	FILTER_FLAGS, *P_FILTER_FLAGS;

#define iflWindow				flag0	// Private.  Internal use only.
#define iflSendMyWindowOnly		flag1


/*
 * This is the number of bytes in the INPUT_EVENT's eventData field.  The
 * data stored in this field depends on the devCode field.  Handlers of
 * msgInputEvent never need to use this value;  all handlers will cast
 * pArgs->eventData to an appropriate type.
*/
#define inptEDataSize	32		// no of bytes INPUT_EVENT's eventData field


//REFGEN BEGINIGNORE
/**** Private ****/

/*
 * The following macros are used internally to construct values.  Clients
 * should not need to use them.
 *
 * The MakeTag macro is on separate line so that the symbolic name gathering
 * mechanism does not try make InputMakeSts a symbol.
*/
#define InputMakeSts(s) \
					MakeTag(clsInput, (s))
#define InputTestSts(s, mask)		((U8)Sts(s) & (U8)mask)
//REFGEN ENDIGNORE


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Return Values From msgInputEvent   				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Overview ****/
/*
 The status returned from msgInputEvent tells the input system how to
 continue processing the event.  This section lists the STATUS values
 that recipients of msgInputEvent may return.  Each of these statuses
 contains several "values."  (Not all possible combinations of these are
 legal or supported.)

    -:  Whether distribution for the event should continue or be
        terminated.

    -:  Grab status.  Whether to start or continue a grab for the
        recipient of msgInputEvent.

    -:  Ancestor interest.  Whether or not the ancestor class was
        interested in the event.

    -:  Filter skip.  For filters only, whether distribution of the event
        should skip certain filters.

 The following table describes the relationship between the legal status
 codes and the values they "contain."  For clarity, the "no" entries are
 left blank and the "Filter skip" information is not shown.
//{
                                                 start     
                                       continue  or        ignored
                                       distri-   continue  by
                                       bution    grab      ancestor
                                       ========  ========  ========
    stsInputContinue                   yes
    stsInputTerminate
    stsInputGrabContinue               yes       yes
    stsInputGrabTerminate                        yes
    stsInputGrab                                 yes
    stsInputIgnored                    yes                 yes
    stsInputGrabIgnored                yes       yes       yes
//}
*/

//REFGEN BEGINIGNORE
/*
 * These values are private.  They are used to construct the legal statuses,
 * but should never be used directly.
 *
 * If flag3 is set, flag4 and flag5 contain a 2 bit value that describe
 * which filter levels to skip.
*/
#define evTerminate		flag0		// distribution should terminate
#define evIgnore		flag1		// event ignored
#define evGrab			flag2		// begin or continue event grabbing
#define evStroke    	flag3		// terminate this strokes event (This 
									// is never set unless evTerminate also 
									// set.)
#define evSkip			flag3		// skip all remaining filters
//						flag4		// used with evSkip
//						flag5		// used with evSkip
#define evOther			flag7		// used for remainder of clsInput tags
//REFGEN ENDIGNORE

/**** Details ****/
/*
 These status values can be returned by any handler of msgInputEvent:

    stsInputContinue:		Distribution of this event should continue.
    stsInputTerminate:		Distribution of this event should terminate.
    stsInputGrabContinue:	Distribution of this event should continue, and
                            the grab should be continued (or started) for
                            the recipient of msgInputEvent.
    stsInputGrabTerminate:	Distribution of this event should terminate,
                            and the grab should be continued (or started)
                            for the recipient of msgInputEvent.
    stsInputGrab:			Same as stsInputGrabTerminate.
    stsInputIgnored:		An ancestor class may return stsInputIgnored to
                            inform a subclass that the ancestor was not
                            interested in the event.  The input system
                            treats stsInputIgnored just like
                            stsInputContinue.
    stsInputGrabIgnored:	An ancestor class may return stsInputGrabIgnored
                            to inform a subclass that the ancestor was not
                            interested in the event, but that the grab
                            should be continued (or started) for the object
                            the received msgInputEvent.  The input system
                            treats stsInputGrabIgnored just like
                            stsInputGrabContinue.

 These statuses should only be returned by Filters:

    stsInputSkip:			Distribution of this event should continue but
                            all remaining filters should be skipped.
    stsInputSkipTo2:		Distribution of this event should continue but
                            all remaining filters in Range 1 (priority
                            less than 64) should be skipped.
    stsInputSkipTo3:		Distribution of this event should continue but
                            all remaining filters in Ranges 1 and 2
                            (priority less than 128) should be skipped.
    stsInputSkipTo4:		Distribution of this event should continue but
                            all remaining filters in Ranges 1, 2 and 3
                            (priority less than 192) should be skipped.
    stsInputTerminateRemoveStroke:  Distribution of this event should
                            terminate, and any other events corresponding
                            to the current stroke should not be sent at
                            all.
    stsInputGrabTerminateRemoveStroke:  Distribution of this event should
                            terminate, the grab should be continued (or
                            started) for the recipient of msgInputEvent,
                            and any other events corresponding to the
                            current stroke should not be sent at all.
*/
#define stsInputContinue				InputMakeSts(0)
#define stsInputSkip					InputMakeSts(evSkip)
#define stsInputSkipTo2					InputMakeSts(evSkip | (1 << 4))
#define stsInputSkipTo3					InputMakeSts(evSkip | (2 << 4))
#define stsInputSkipTo4					InputMakeSts(evSkip | (3 << 4))
#define stsInputTerminate				InputMakeSts(evTerminate)
#define stsInputGrabContinue			InputMakeSts(evGrab)
#define stsInputGrabTerminate			InputMakeSts(evGrab | evTerminate)
#define stsInputGrab					InputMakeSts(evGrab | evTerminate)
#define stsInputTerminateRemoveStroke	InputMakeSts(evTerminate | evStroke)
#define stsInputGrabTerminateRemoveStroke	\
				InputMakeSts(evGrab | evTerminate | evStroke)
#define stsInputIgnored			 		InputMakeSts(evIgnore)
#define stsInputGrabIgnored		 		InputMakeSts(evGrab | evIgnore)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 *                        	Other Statuses								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define stsInputQueueFull			MakeStatus(clsInput, evOther | 2)


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 *						Input Event Construction Values					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * The type of input event is specified in the devCode field of the input
 * event.  This field is of type MESSAGE.  The following INPUT_EVENT_CODE
 * items are used by the pen to create msgPenXXXX to put in the devCode
 * field.  Clients handling msgInputEvent should switch on the message
 * in the devCode field or on the class number of the message rather than
 * looking at these specific event codes.
 * 
 * [Terminology Note:  the eventInProxUp and eventOutProxUp events can be
 * thought of as eventInProx and eventOutProx since the tip is always up when
 * these events are sent.  The trailing "Up" is present for historical
 * reasons only.]
*/
Enum16(INPUT_EVENT_CODE) {
	eventTipUp			= 0,
	eventTipDown		= 1,
	eventMoveUp			= 2,
	eventMoveDown		= 3,
	eventEnterUp		= 4,
	eventEnterDown		= 5,
	eventExitUp			= 6,
	eventExitDown		= 7,
	eventInProxUp		= 8,
	eventInProxDown		= 9,		// Obsolete.  No longer used.
	eventOutProxUp		= 10,
	eventOutProxDown	= 11,		// Obsolete.  No longer used.
	eventStroke			= 12,
	eventTap	 		= 13,
	eventTimeout		= 14,		// Writing timeout.
	eventHoldTimeout	= 18,
	eventHWTimeout		= 17,		// Obsolete.  No longer used.
	eventOther			= 65
};
//REFGEN ENDIGNORE

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 *                          Input Flags									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Overview ****/
/*
 Each window has a set of input flags that are stored in the window's 
 win.flags.input field.  These flags can be manipulated while handling
 msgNew and msgNewDefaults.  They can also be manipulated with several
 other window messages;  see win.h for more information.

 InputSetGrab() and InputFilterAdd use these flags as one of their
 parameters.
*/

typedef U32 INPUT_FLAGS, *P_INPUT_FLAGS;

/**** "Interest" Flags ****/
/*
 PenPoint's input system can generate many messages.  Most clients are only
 interested in a subset of the messages that can be generated.  So clients
 can provide hints to the input system about the input events the client is
 interested in.  This reduces the message traffic and increases
 performance.  For instance, if a client is not interested in pen movement
 events when the pen is up above the writing surface (but within
 proximity), the client can clear the inputMoveUp flag

 Typically, a flag enables or disables several input events.  For instance,
 setting the inputTip flag enables both msgPenDown and msgPenUp (see
 pen.h).

 You should treat these flags as a hint to the input system.  You should
 not assume that a specific input event will not arrive because you have
 not enabled the corresponding bit in the input flags.

 This table contains examples of the messages that are enabled by setting
 various flags.  This table is only representative -- it is not complete!
//{
                       example of 						message defined
    input flag         message(s) enabled               in
    ================   ==============================   ===============
    inputTip           msgPenUp, msgPenDown             pen.h
    inputMoveUp        msgPenMoveUp                     pen.h
    inputMoveDown      msgPenMoveDown                   pen.h
    inputEnter         msgPenEnterUp, msgPenEnterDown   pen.h
    inputExit          msgPenExitUp, msgPenExitDown     pen.h
    inputInProx        msgPenInProxUp                   pen.h
    inputOutProx       msgPenOutProxUp                  pen.h
    inputStroke        msgPenStroke                     pen.h
    inputTap           msgPenTap                        pen.h
    inputHoldTimeout   msgPenHoldTimeout                pen.h
    inputChar          msgKeyChar                       key.h
    inputMultiChar     msgKeyMulti                      key.h
    inputMakeBreak     msgKeyUp, msgKeyDown             key.h
//}
*/
#define inputTip		(U32)(flag0)	// enable TipUp & TipDown events
#define	inputMoveUp		(U32)(flag1)	// enable MoveUp events
#define	inputMoveDown	(U32)(flag7)	// enable MoveDown events
#define inputEnter		(U32)(flag2)	// enable EnterUp & EnterDown
#define inputExit		(U32)(flag3)	// enable ExitUp & ExitDown
#define	inputInProx		(U32)(flag4)	// enable InProxUp events
#define inputOutProx	(U32)(flag5)	// enable OutProxUp events
#define inputStroke		(U32)(flag6)	// enable Stroke events (See pen.h.)
#define inputTap		(U32)(flag10)	// enable tap events
#define inputChar		(U32)(flag13)	// enable character events
#define inputMultiChar	(U32)(flag14)	// enable multi-char events
#define inputMakeBreak	(U32)(flag15)	// enable make/break events
#define inputHoldTimeout (U32)(flag8)	// enable HoldTimeout events  (See
										// "Hold Timeout Events" in pen.h)
#define inputTimeout	(U32)(flag9)	// obsolete.
#define inputHWTimeout	(U32)(flag11)	// obsolete.
#define inputMoveDelta  (U32)(flag18)   // enable compression of multiple
										// delta events into a single delta
										// event.  Good news: fewer messages 
										// and better performance.  Bad news:
										// less information to the client.
#define inputDestOnly   (U32)(flag19)   // send event iff destination is self
#define inputLRContinue (U32)(flag20)   // enable dist. to parent windows
#define inputDisable	(U32)(flag21)   // send no input event messages
#define inputEnableAll	(U32)(flag25)	// enables all events to be sent to 
										// grabbers

/**** Inking Flags ****/
/*
 WARNING: Inking and the acetate are subject to major changes in future
 releases.
*/
#define inputInk		(U32)(flag23)	// enable inking in the acetate layer
#define inputInkThrough (U32)(flag24)	// enable inking in window rather 
										// than acetate layer
#define inputInkDisable	(U32)(flag30)	// disables both inputInk and 
										// inputInkThrough


/**** Miscellaneous Flags ****/
/*

    inputNoBusy:		If cleared, then the input system automatically turns
                        on PenPoint's busy clock if the recipient of a
                        message does not return before a certain timeout.
                        If set, this default busy clock behavior is
                        disabled.

    inputResolution:	If set, msgPenMoveUp and msgPenMoveDown messages are
                        sent each time the pen moves one digitizer unit.
                        (In other words, the input system sends a move
                        event for even the smallest detectable amount of
                        movement.  If cleared, move events are sent only
                        when the pen has moved at least one display pixel's
                        size.

    inputAutoTerm:		Should only be set by a grabber.  Specifies that all
                        events that the grabber is not interested in should
                        be treated as if the grabber returned 
                        stsInputGrabTerminate.

    inputGrabTracker:	Should only be set by a grabber.  Specifies that the
                        grabber does not need the input system to perform
                        its normal hit detection.  This is strictly a
                        performance enhancement.  (The name of this value
                        is an anachronism.  Originally trackers were the 
                        only grabbers that didn't need hit detection.)
*/
#define inputNoBusy		(U32)(flag12)	// disable default busy clock
#define inputResolution (U32)(flag22)	// report pen resolution move events
#define inputAutoTerm	(U32)(flag26)	// automatically terminate all 
										// events' distribution if grabber 
										// doesn't have the event
										// flag enabled.
#define inputGrabTracker (U32)(flag27)	// disables hit detect during grab
#define inputTransparent (U32)(flag31)  // invisible to hit detect operations
										// See win.h.

#define inputSigVerify	(U32)(flag16)	// Sets pen sample rate to high and
										// MIL reporting threshold to 0.
										// This does not guarantee getting
										// every pen move event, so users
										// should check timestamps to see
										// if any data has been lost.

// Shorthand for all flags which correspond to real input events.
#define inputAllRealEventsFlags			(U32)0xEFFF


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 *                       Event Distribution Flags                          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*
	Input distribution flags give some additional information to the
	object being sent the input event.

	evfFilter:			object is getting this event because it is a filter;
	evfGrab:			object is getting this event because it is a grabber;
	evfListener:		object is getting because it was specified in the
						input event listener field;
	evfTarget:			object is getting this event because it is
						the target;
	evfXYLeafToRoot:	object is getting this event as part of the XY
						distribution;

	evfInSelf:			event occurred in this window;
	evfInChild:	  		event occurred in a child of this window;

NOTE:	evfInSelf and evfInChild will become obsolete in future releases.

	evfGrabTracker:		object had input grab tracker flag on.

*/

typedef U32 INPUT_DIST_FLAGS, P_INPUT_DIST_FLAGS;

//REFGEN BEGINIGNORE
/* Obsolete. */
#define evFlag0			((U32)flag0)
#define evFlag1			((U32)flag1)
#define evFlag2			((U32)flag2)
#define evFlag3			((U32)flag3)
#define evFlag4			((U32)flag4)
#define evFlag5			((U32)flag5)
#define evFlag6			((U32)flag6)
#define evFlag7			((U32)flag7)
//REFGEN ENDIGNORE

//	NOTE:	evfInSelf and evfInChild will become obsolete in future releases.
#define evfInParent	 	((U32)flag9)	// Obsolete.
#define evfInChild	  	((U32)flag10)	// event occurred in child window
#define evfInSelf	   	((U32)flag11)	// event occurred in this window
#define evfGrabTracker	((U32)flag12)	// event occurred during grab 

#define evfFilter	   	((U32)flag26)   // event in filter distribution
#define evfGrab		 	((U32)flag27)   // event in grab distribution

#define evfXYLeafToRoot ((U32)flag29)   // event in XY dist
#define evfListener	 	((U32)flag30)   // event in "pre-specified 
										// destination" distribution
#define evfTarget	   	((U32)flag31)   // event in target distribution


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 *                       		Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgInputEvent 					takes P_INPUT_EVENT, returns STATUS
	theInputManager uses this message to deliver input events.

 pArgs->devCode contains the "event" that is being delivered.  These events
 are device-specific.  See pen.h for a list of pen events and key.h for a
 list of key events.

 The pArgs for msgInputEvent is best thought of as a union type.  pArgs can
 always be cast to a P_INPUT_EVENT, but the content of pArgs->eventData
 depends on the value of pArgs->devCode.  For some values of
 pArgs->devCode, the pArgs are actually larger than an INPUT_EVENT
 structure, so use the pArgs->length field to determine the length of
 the input event.  For example, the msgPenStroke and msgKeyMulti events
 both have data which extends past the end of the INPUT_EVENT structure.

 For events that have a valid XY, pArgs->destination is the top-most window
 with input enabled (FlagOff(inputDisable, ...)).  

 The recipient of this message must return one of the status values
 described in the section "Return Values from msgInputEvent."
*/
typedef struct INPUT_EVENT {
	SIZEOF		  		length;					// actual length of pArgs
	INPUT_DIST_FLAGS	flags;					// distribution information
	MESSAGE				devCode;				// input event
	OS_MILLISECONDS		timestamp;				// time event was queued
	XY32				xy;						// location of event
	OBJECT				listener;				// pre-specified destination
												// Normally objNull.
	OBJECT		  		destination;
	OBJECT		  		originator;				// originating device 
	U8					eventData[inptEDataSize];	// event specific data
} INPUT_EVENT, *P_INPUT_EVENT;

#define msgInputEvent				MakeMsg(clsInput, 0)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 *                       	Notification Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

//REFGEN BEGINIGNORE
/****************************************************************************
 msgInputFilterTerminated		takes NULL, returns STATUS
	Obsolete.  Not sent.
*/
#define msgInputFilterTerminated 	MsgNoError(MakeMsg(clsInput, 0x82))
//REFGEN ENDIGNORE


/****************************************************************************
 msgInputGrabPushed 				takes OBJECT, returns STATUS
	Notifies a grabbing object that it is being pushed onto the grabber
	stack and the pArgs is the new grabber.
*/
#define msgInputGrabPushed			MsgNoError(MakeMsg(clsInput, 0x83))


/****************************************************************************
 msgInputGrabPopped 				takes OBJECT, returns STATUS
	Notifies a grabbing object that is being popped from the grabber stack
	and becoming the current grabber.
*/
#define msgInputGrabPopped			MsgNoError(MakeMsg(clsInput, 0x84))


/****************************************************************************
 msgInputTargetDeactivated			takes OBJECT, returns STATUS
	Notifies the input target that some other object is become the input
	target.
*/
#define msgInputTargetDeactivated	MsgNoError(MakeMsg(clsInput, 0x85))


/****************************************************************************
 msgInputTargetActivated			takes OBJECT, returns STATUS
	Notifies an object that it is becoming the input target.
*/
#define msgInputTargetActivated		MsgNoError(MakeMsg(clsInput, 0x86))


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 *                       		Functions								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 InputFilterAdd		returns STATUS
	Adds a filter to the filter list.
*/
STATUS EXPORTED InputFilterAdd(
    OBJECT			newFilter,
    INPUT_FLAGS		inputEventFlags,
	FILTER_FLAGS	filterFlags,
    U8				priority
);


/****************************************************************************
 InputFilterRemove					returns STATUS
	Removes a filter from the filter list.
*/
STATUS EXPORTED0 InputFilterRemove(
	OBJECT listener		// filter to remove
);


/****************************************************************************
 InputEventInsert					returns STATUS
	Adds an event to the input event queue.

 Most clients do not use this message.

 If stamp is true, pEvent->timestamp is filled in with the current time and
 the event is added to the end of the queue.  Otherwise, pEvent->timestamp
 is not modified  and the event is placed at the head of the queue and the

 Return Value
	stsInputQueueFull:		the input system queue is full
*/
STATUS EXPORTED0 InputEventInsert(
	P_INPUT_EVENT	pEvent,
	BOOLEAN			stamp
);	


/****************************************************************************
 InputSetTarget 				returns STATUS
	Sets the input target object and flags.

 Clients use this message to set the input target.  The input target is the
 object that receives msgInputEvent for all events that do not have an XY
 position -- in particular, keyboard events.
 
 PenPoint's UI guidelines state that the selection owner and input target
 should usually be the same object.  PenPoint does not enforce this
 association in any way. See the UI documentation and sel.h for more
 information.

 See Also
	msgInputTargetActivated
	msgInputTargetDeactivated
	sel.h
*/
STATUS EXPORTED InputSetTarget(
	OBJECT	target,		 	// new target object
	U32		flags			// new target flags
);


/****************************************************************************
 InputGetTarget					returns OBJECT
	Returns the current input target.

 See Also
	InputSetTarget
*/
OBJECT EXPORTED InputGetTarget(void);


/****************************************************************************
 InputSetGrab						returns STATUS
	Sets the current grabber and its flags.	

 The previous grabber is pushed onto the grabber stack.

 If the flags parameter is 0, then inputAllRealEventsFlags is used.

 If both parameters are null, the current grabber is removed and the
 grabber on the top of the grabber stack (if the stack isn't empty) becomes
 the current grabber.
*/
STATUS EXPORTED InputSetGrab(
	OBJECT	grabber,	// new grabber
	U32		flags		// new grab input flags
);


/****************************************************************************
 InputGetGrab						returns void
	Passes back the current grabber and its flags.

 See Also
	InputSetGrab
*/
void EXPORTED InputGetGrab(
	P_OBJECT	pGrabber,		// grabber
	P_U32		pFlags			// current grab flags
);


/****************************************************************************
 msgInputModalStart					takes P_INPUT_MODAL_DATA, returns STATUS
	Asks theInputManager to start recursive input.

 This message is used to implement a system-wide mode.  Typical application
 programs should not sent this message.

 You must send this message to the input system using ObjectSendUpdate().
 The sending task is blocked until the recursive task returns.  The
 recursive task can pass data to the first task via pArgs.

 See Also
	msgInputModalEnd
*/
typedef struct INPUT_MODAL_DATA {
	U32		reserved;
	U32		clientData[2];
} INPUT_MODAL_DATA, *P_INPUT_MODAL_DATA;

#define msgInputModalStart			MakeMsg(clsInput, 1)


/****************************************************************************
 msgInputModalEnd					takes P_INPUT_MODAL_DATA, returns STATUS
	Asks theInputManager to terminate recursive input.  

 This message terminates a system-wide mode.  Typical application programs
 should not send this message.

 This message must be paired with msgInputModalStart.

 The sender of this message can pass information to the sender of 
 msgInputModalStart by filling in pArgs.

 This message may be sent with ObjectCall() or ObjectSend().

 See Also
	msgInputModalStart
*/
#define msgInputModalEnd			MakeMsg(clsInput, 2)


/****************************************************************************
 msgInputActivityTimer				takes BOOLEAN, returns STATUS
	Asks theInputManager to enable or disable the activity timer.

 The input system maintains an "activity timer."  Each time the input
 system has no events to process, the input system starts this timer.  If
 no events are received before the timer expires, the input system puts
 PenPoint into Standby mode.  This duration is typically several minutes
 long.

 Long running background tasks should first send msgInputActivityTimer with
 pArgs of false to tell theInputManager to not turn off the machine.  When
 the background operation is complete, the task should send the message
 again, but this time with a pArgs of true.

 theInputManager keeps a nesting count which allows nested pairs of sends of
 this message.
*/
#define msgInputActivityTimer		MakeMsg(clsInput, 5)

#endif
