/****************************************************************************
 File: instlmgr.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.59  $
   $Author:   rfoster  $
     $Date:   27 Feb 1992 23:14:10  $

 This file contains the class definition and methods for clsInstallMgr.
 clsInstallMgr inherits from clsObject.
 Provides the basic facilities for installing items.
 
 NOTE: THE MESSAGES IN THIS CLASS ARE SENT TO THE INSTALL MANAGER, NOT TO 
 THE HANDLES.

 clsInstallMgr provides almost everything needed to manage installable
 items.  An installable item is anything that can be installed and deinstalled
 on a Penpoint machine, such as fonts, applications, services, handwriting
 prototype sets, etc.  You create an instance of clsInstallMgr for each 
 category of installable item.  Penpoint creates well-known install managers
 for the following categories at cold boot time:

	-:  theInstalledHWXProtos:    Handwriting prototype sets
	-:  theInstalledPrefs:        Preference sets
	-:  theInstalledPDicts:       Personal dictionaries

 In addition there are several well-known install managers that are created
 from subclasses of clsInstallMgr:

	-:  theInstalledApps:         Applications (appimgr.h)
	-:  theInstalledServices:     Services (servimgr.h)
	-:  theInstalledFonts:        Fonts (fontmgr.h)

 clsInstallMgr makes use of the filesystem to keep a database of the 
 installed items.  Each item is represented by a file or directory handle.  
 This is a big win for items which *are* files or directories; the 
 InstallMgr's handle is a handle onto the actual item.  There is an extra 
 level of indirection for items which are not files.  The item's ID (whatever 
 that means for a particular type of item) is stored as an attribute of the 
 handle.  An item's name is the name of that item's filesystem node.  This 
 means that items on a given install manager must have unique names.

 An install manager has a base directory in which it keeps its items' 
 filesystem nodes.  The createInitial style bit determines whether the 
 install manager creates an initial set of item handles from whatever is in 
 this directory when the install manager is first created.

 clsInstallMgr provides an API for installing new items and deinstalling 
 existing items.  An item is installed from a location on an external
 filesystem.

 An item can be deinstalled, which removes all traces of the item
 from the system.

 The install manager maintains a bit which specifies if an item has changed.  
 It is the client's responsibility to maintain this bit by sending
 msgIMSetModified when it modifies an item.  The install manager will
 remember the time and date that the item was modified.

 Install managers also maintains a "current" item, and provide 
 an API for getting and setting the current item.  This is used by 
 theInstalledHWXProtos, theInstalledPrefs and theInstalledPDicts to
 specify which handwriting prototype set, preferences, or personal dictionary
 the system is actively using.  A current item is optional; some install
 managers (theInstalledApps, theInstalledServices) do not make use of a 
 current item.

 An item can be marked as being "in use".  This means that the item cannot
 be deinstalled.  The current item is considered to be in use.

 Each install manager can have a verifier object, which it queries whenever
 installation takes place.  The verifier object makes sure that the item
 being installed is valid for this install manager.

 An InstallMgr sends notification to its observers whenever 
 an item is installed, deinstalled, the current item changed, 
 etc.  Subclasses of clsInstallMgr can turn notification generation on and 
 off with msgIMSetNotify.  Notification is on by default.

 A subset of the notification messages are also sent to any observers of
 an item's handle.  This allows clients who are only interested in a
 particular item to monitor just that item.  The messages sent are:
	-:  msgIMNameChanged		
	-:  msgIMInUseChanged
	-:  msgIMModifiedChanged	
	-:  msgIMDeinstalled
	-:  msgIMCurrentChanged (sent to both old and new current handles)

 Clients access installable managers via an ObjectCall interface.  
 clsInstallMgr can accommodate simultaneous access by multiple clients
 if the "shared" style bit is set true (the default).  This causes it to 
 semaphore all of its operations.  This semaphore is available to subclasses 
 via msgIMGetSema, and should be used to protect all subclass messages if 
 multiple clients will be accommodated.  clsInstallMgr also sets objCapCall on
 by default.

 There is a well-known, shared list object (see list.h) that is a list of 
 all the install managers in the system.  This object is called
 theInstallManagers.  You can observe this list and get notification when an
 install manager is	added and removed.  See msgListNotifyAddition and 
 msgListNotifyDeletion.

 clsFontInstallMgr, clsAppInstallMgr, and clsServiceMgr inherit from 
 clsInstallMgr.  See fontmgr.h, appimgr.h and servmgr.h for these classes.
****************************************************************************/
#ifndef INSTLMGR_INCLUDED
#define INSTLMGR_INCLUDED

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif

#ifndef LIST_INCLUDED
#include <list.h>
#endif

#ifndef TKTABLE_INCLUDED
#include <tktable.h>
#endif

#ifndef OPTION_INCLUDED
#include <option.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  		Common #defines and typedefs  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Handle type  ****/
typedef OBJECT			IM_HANDLE, * P_IM_HANDLE;

/****  Warning Codes  ****/

/* Some install manager request has been user cancelled */
#define stsIMRequestCancelled		MakeWarning(clsInstallMgr, 0)

/****  Quick Help Tags  ****/
#define  appQHInstallMgr			MakeTag(clsInstallUISheet, 32)
#define  svcQHInstallMgr         MakeTag(clsInstallUISheet, 33)
#define  hwxQHInstallMgr			MakeTag(clsInstallUISheet, 34)
#define  gestQHInstallMgr        MakeTag(clsInstallUISheet, 35)
#define  dictQHInstallMgr			MakeTag(clsInstallUISheet, 36)
#define  fontsQHInstallMgr       MakeTag(clsInstallUISheet, 37)
#define  userPfleQHInstallMgr    MakeTag(clsInstallUISheet, 38)


/****  Status Codes  ****/

/* The item is current, so cannot be removed. */
#define  stsIMCurrent				MakeStatus(clsInstallMgr, 1)

/* An item to be installed failed verification. */
#define  stsIMInvalidItem			MakeStatus(clsInstallMgr, 2)

/* A new name cannot be created for this item. */
#define stsIMUniqueNameFailed		MakeStatus(clsInstallMgr, 3)

/* The item is in use, so cannot be removed. */
#define  stsIMInUse					MakeStatus(clsInstallMgr, 6)

/* The item to be installed is already installed. */
#define  stsIMAlreadyInstalled		MakeStatus(clsInstallMgr, 8)

/* An invalid handle was passed in.	*/
#define  stsIMBadHandle				MakeStatus(clsInstallMgr, 20)

/****  File System Attribute Definitions  ****/
/* 
 Note: Most clients do not deal with attributes directly. 
*/

/* Node's home on an external volume. Absolute path. */
/* This attribute is used only during installation. */
#define imAttrHome					FSMakeStrAttr(clsInstallMgr, 0)

/* Is this node the current node? Use IM_ATTR_CURRENT values. */
#define imAttrCurrent				FSMakeFix32Attr(clsInstallMgr, 2)
typedef enum IM_ATTR_CURRENT {
	imNotCurrent		  = 0,	 // Same as no attribute
	imCurrent			  = 1
} IM_ATTR_CURRENT;

/* Is this node in use? Use IM_ATTR_INUSE values. */
#define imAttrInUse					FSMakeFix32Attr(clsInstallMgr, 3)
typedef enum IM_ATTR_INUSE {
	imNotInUse			  = 0,	 // Same as no attribute
	imInUse				  = 1
} IM_ATTR_INUSE;

/* Has this node been modified? Use IM_ATTR_MODIFIED values. */
#define imAttrModified				FSMakeFix32Attr(clsInstallMgr, 4)
typedef enum IM_ATTR_MODIFIED {
	imNotModified		  = 0,	 // Same as no attribute
	imModified			  = 1
} IM_ATTR_MODIFIED;

/*
 Ref count. When an item is installed the installer can choose to maintain
 a reference count if the item is already installed.
*/
#define imAttrRefCount			 	FSMakeFix32Attr(clsInstallMgr, 5)

/*
 Is this item on some other item's dependency list? Use IM_ATTR_DEPENDENT 
 values.
*/
#define imAttrDependent				FSMakeFix32Attr(clsInstallMgr, 7)
typedef enum IM_ATTR_DEPENDENT {
	imNotDependent		  = 0,	 // Same as no attribute
	imDependent			  = 1
} IM_ATTR_DEPENDENT;

/*
 Is this item a system inviolate item? Use IM_ATTR_SYSTEM values.
*/
#define imAttrSystem				FSMakeFix32Attr(clsInstallMgr, 8)
typedef enum IM_ATTR_SYSTEM {
	imNotSystem				= 0,	 // Same as no attribute
	imSystemInviolate		= flag0,
	imSystemNotRenameable	= flag1
} IM_ATTR_SYSTEM;

/* Version string */
#define imAttrVersion				FSMakeStrAttr(clsAppInstallMgr, 3)

/****  Debug Flags  ****/
#define installDebugFlag   	'I'


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  				  	Messages  		  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew						takes P_IM_NEW, returns STATUS
	category: class message
	Creates a new install manager.

 The locator field specifies the directory where the managed items live.  If 
 this directory does not exist it will be created.
*/
typedef struct IM_STYLE {
	U16	shared			  : 1,	// Provide concurrency protection.
		createInitial	  : 1,	// Create initial list of handles from 
								//   contents of base directory.
		autoSetCurrent	  : 1,	// Choose any item as the initial current
								//   setting if no one has current attr set.
	 	copyOnInstall	  : 1,  // Copy nodes to manager's dir or create
								//   handles directly on Install locator.
		addToGlobalList	  : 1,	// Add this instlmgr to theInstallManagers.
		createIcon		  : 1,	// Create an icon for this install manager.
		private1	  	  : 1,	// Always set this to false.
		duplicatable	  : 1,	// Items in this installmgr can be duplicated.
		usesVersions	  : 1,	// Items in this installmgr have versions.
		reserved		  : 7;
	U16	sizeCol			  : 1,	// Show size column in Settings NB card.
		hwxTypeCol		  : 1,	// Show hwx engine type column.
		svcTypeCol		  : 1,	// Show service type column.
		modifiedCol		  : 1,	// Show modified column.
		currentCol		  : 1,	// Show current column.
		inUseCol		  : 1,	// Show inUse column.
		reserved1		  : 10;
	U32	helpId;					// Help tag for installmgr's Settings NB card.
	U16	spare1;
	U16 spare2;
} IM_STYLE, *P_IM_STYLE;						 

typedef struct IM_NEW_ONLY {
	IM_STYLE			style; 
	FS_DIR_NEW_MODE		dirMode;		// Default mode for dir handles.
	FS_FILE_NEW_MODE	fileMode;		// Default mode for file handles.
	FS_LOCATOR			locator;		// Base directory.  InstallMgr will
										// create it if it doesn't exist.
	P_STRING			pSingularName; 	// Singular name of installer. Must be
										// <= nameLength in size.
	P_STRING			pName;			// Plural name of installer.  Must be
										// <= nameLength in size.
	P_STRING			pInstallPath;	// Base path for installable items,
										// (i.e. \penpoint\app).
	OBJECT				verifier;		// Verifier object.  Can be null.
	OS_HEAP_ID			heap;			// Installmgr heap.  Must be global.
										// Can be osInvalidHeapId; instlmgr 
										// will use global heap of the task
										// that this object is created in.
	P_TK_TABLE_ENTRY	pSettingsMenu;	// Additional controls for this 
										// installmgr's Settings NB card.
	U32					settingsMenuSize;// Size (in bytes) of pSettingsMenu.
	U32					unused1;
	U32					unused2;
	U32					unused3;
	U32					unused4;
} IM_NEW_ONLY, *P_IM_NEW_ONLY;

#define	installMgrNewFields					\
	objectNewFields							\
	IM_NEW_ONLY   			installMgr;

typedef struct IM_NEW {
	installMgrNewFields
} IM_NEW, *P_IM_NEW;


/****************************************************************************
 msgNewDefaults					takes P_IM_NEW, returns STATUS
	category: class message
    Initializes the IM_NEW structure to default values.	

 Clients do not	normally change the defaults.

 Zeroes out installMgr and sets
//{
	object.cap |= objCapCall;
	installMgr.style.shared = true;
	installMgr.style.createInitial = true;
	installMgr.style.updateOK = true;
	installMgr.style.copyOnInstall = true;
	installMgr.style.addToGlobalList = true;
	installMgr.style.private1 = false;
	installMgr.style.duplicatable = false;
	installMgr.style.createIcon = true;
	installMgr.style.duplicatable = false;
	installMgr.style.usesVersions = false;
	installMgr.style.sizeCol = true;
	installMgr.dirMode = fsUnchangeable;
	installMgr.fileMode = fsSharedMemoryMap;
	installMgr.pInstallPath = pNull;
	installMgr.verifier = objNull;
	installMgr.heap = osInvalidHeapId;
	installMgr.pSettingsMenu = objNull;
	installMgr.settingsMenuSize = 0;
//}
*/


/****************************************************************************
 msgDestroy					takes OBJ_KEY, returns STATUS
	Frees the install manager.
	
 Note: This message does not destroy the install manager's directory, nor
 any files/directories in that directory. 

*/


/****************************************************************************
 msgDump					takes OBJ_KEY, returns STATUS
	Prints out the items in the install manager and their state.

*/


/****************************************************************************
 msgIMGetStyle				takes P_IM_STYLE, returns STATUS
	Passes back the current style settings.

*/
#define msgIMGetStyle		 				MakeMsg(clsInstallMgr, 1)


/****************************************************************************
 msgIMSetStyle				takes P_IM_STYLE, returns STATUS
	Sets the current style.

*/
#define msgIMSetStyle		 				MakeMsg(clsInstallMgr, 2)


/****************************************************************************
 msgIMGetInstallerName		takes P_STRING, returns STATUS
	Passes back the install manager's name.

 pArgs must point to a nameBufLength buffer.
 
 The install manager's name was set at msgNew time in installMgr->pName.

*/
#define msgIMGetInstallerName 				MakeMsg(clsInstallMgr, 3)


/****************************************************************************
 msgIMGetInstallerSingularName		takes P_STRING, returns STATUS
	Passes back the install manager's singular name.

 pArgs must point to a nameBufLength buffer.
 
 The install manager's name was set at msgNew time in installMgr->pName.

*/
#define msgIMGetInstallerSingularName 	 	MakeMsg(clsInstallMgr, 51)


/****************************************************************************
 msgIMGetCurrent			takes P_IM_HANDLE, returns STATUS
	Passes back the current item's handle. 

 Passes back objNull if there is no current handle.

*/
#define msgIMGetCurrent		 				MakeMsg(clsInstallMgr, 4)


/****************************************************************************
 msgIMSetCurrent			takes IM_HANDLE, returns STATUS
	Sets the current item. 

 The argument is the handle to be made current.  It can be objNull to 
 indicate that no handle is the current one.

 If the handle specified in the argument is already current then nothing
 is done (no observer message is generated).

 Causes the install manager to notify observers with msgIMCurrentChanged.

*/
#define msgIMSetCurrent		 				MakeMsg(clsInstallMgr, 5)


/****************************************************************************
 msgIMSetInUse				takes P_IM_SET_INUSE, returns STATUS
	Changes an item's in use setting.

 Setting inUse to true means that the item cannot be deinstalled. 

 Use msgIMGetState to query the value of this field.

 Causes the install manager to notify observers with msgIMInUseChanged.

*/
#define msgIMSetInUse		 		   		MakeMsg(clsInstallMgr, 6)

typedef struct IM_SET_INUSE {
	IM_HANDLE			handle;		// Handle of item to set inUse on.
	BOOLEAN				inUse;		// InUse value.
} IM_SET_INUSE, *P_IM_SET_INUSE;


/****************************************************************************
 msgIMSetModified				takes P_IM_SET_MODIFIED, returns STATUS
	Changes an item's modified setting.

 Use msgIMGetState to query the value of this field.

 Causes the install manager to notify observers with msgIMModifiedChanged.

*/
#define msgIMSetModified	 		   		MakeMsg(clsInstallMgr, 7)

typedef struct IM_SET_MODIFIED {
	IM_HANDLE			handle;		// Handle of item to set modified on.
	BOOLEAN				modified;	// Modified value.
} IM_SET_MODIFIED, *P_IM_SET_MODIFIED;


/****************************************************************************
 msgIMGetName				takes P_IM_GET_SET_NAME, returns STATUS
	Get the name of a item. 

*/
#define msgIMGetName	 			  		MakeMsg(clsInstallMgr, 8)

typedef struct IM_GET_SET_NAME {
	IM_HANDLE			handle;		// Handle of item to get/set name on.
	P_STRING			pName;		// In: (Set) Out: (Get) name.  This 
									// pointer must reference a nameBufLength
									// size buffer.
} IM_GET_SET_NAME, *P_IM_GET_SET_NAME;


/****************************************************************************
 msgIMSetName				takes P_IM_GET_SET_NAME, returns STATUS
	Sets the name of a item. 

 The name must be a legitimate file name and unique amoung all the items on
 this install manager.

 Causes the install manager to notify observers with msgIMNameChanged.

 Return Value
	stsFSNodeExists:		An item with this name already exists.
*/
#define msgIMSetName	 			  		MakeMsg(clsInstallMgr, 9)


/****************************************************************************
 msgIMGetVersion			takes P_IM_GET_VERSION, returns STATUS
	Get the version string for this item.

 Not all install managers have a version string.  pVersion is set to
 pNull if there is no version.
*/
#define msgIMGetVersion	 			  		MakeMsg(clsInstallMgr, 37)

typedef struct IM_GET_VERSION {
	IM_HANDLE			handle;		// Handle of item to get version of.
	P_STRING			pVersion;	// Out: Version string.	Pointer must 
									// reference a nameBufLength
									// size buffer.
} IM_GET_VERSION, *P_IM_GET_VERSION;


/****************************************************************************
 msgIMGetList				takes P_LIST, returns STATUS
	Passes back a list of all the items on this install manager.

 The memory for the list object	is allocated out of the caller's local 
 process heap. 

 CAUTION: Caller must destroy the list object when it is finished using it.
*/
#define msgIMGetList	 					MakeMsg(clsInstallMgr, 14)


/****************************************************************************
 msgIMGetState				takes P_IM_GET_STATE, returns STATUS
	Gets the state of a item.

*/
#define msgIMGetState		 				MakeMsg(clsInstallMgr, 16)

typedef struct IM_GET_STATE {
	IM_HANDLE			handle;		// Handle of item to get state on.
	BOOLEAN				current;	// Out: Is it the current item?
	BOOLEAN				reserved;	// Reserved.
	BOOLEAN				modified;	// Out: Is it modified?
	BOOLEAN				inUse;		// Out: Is it in use?
} IM_GET_STATE, *P_IM_GET_STATE;


/****************************************************************************
 msgIMGetSize				takes P_IM_GET_SIZE, returns STATUS
	Returns the size of an item.

*/
#define msgIMGetSize		 				MakeMsg(clsInstallMgr, 17)

typedef struct IM_GET_SIZE {
	IM_HANDLE			handle;		// Handle of item to get size of.
	U32					size;		// Out: size.
} IM_GET_SIZE, *P_IM_GET_SIZE;


/****************************************************************************
 msgIMInstall				takes P_IM_INSTALL, returns STATUS
	Installs a new item. 

 The install manager derives the item's name from the filesystem
 location specified in pArgs->locator.  pArgs->exist controls what happens if 
 an item of the same name as the item to be installed already exists. 

 pArgs->listAttrLabel and pArgs->listHandle are used to specify an attr
 list to which the install handle is added.  This is used to keep track
 of sub-apps and sub-services.  Set these arguments to 0 if this should not
 be done.

 Causes the install manager to notify observers with msgIMInstalled.
 The install manager also sends msgIMModifiedChanged if the modified
 states changed due to the install.

 Return Value
	stsIMInvalid:			Item to be installed does not pass verification.
	stsIMAlreadyInstalled:	Item already installed and 
							pArgs->exist == imExistGenError.
	stsBadParam:			pArgs->exist is set to an invalid value.
*/
#define msgIMInstall	 					MakeMsg(clsInstallMgr, 18)

typedef enum IM_INSTALL_EXIST {
	imExistUpdate		= 0, // Copy new over existing. 
	imExistReactivate	= 1, // Deactivate existing, then activate new.
	imExistGenError		= 2, // Return stsIMAlreadyInstalled.
	imExistGenUnique	= 3, // Generate a unique name for the new item.
	imExistIncRefCount	= 4	 // Just increment ref count of existing item.
} IM_INSTALL_EXIST, *P_IM_INSTALL_EXIST;

typedef struct IM_INSTALL {
	FS_LOCATOR			locator;	   // Location of item on external 
									   // filesystem.
	IM_INSTALL_EXIST	exist;		   // What to do if item already exists.
	FS_ATTR_LABEL	  	listAttrLabel; // Attr list to add install handle to.
	OBJECT				listHandle;	   // filesystem handle to put attr on.
	IM_HANDLE			handle;		   // Out: Handle of installed item.
} IM_INSTALL, *P_IM_INSTALL;


/****************************************************************************
 msgIMDeinstall					takes P_IM_DEINSTALL, returns STATUS
	Deinstalls an item. 

 All traces of the item are removed, including the item's handle.
	
 Return Value
	stsIMInUse:		Item is in use; cannot be deinstalled.
*/
#define msgIMDeinstall	 					MakeMsg(clsInstallMgr, 19)

typedef struct IM_DEINSTALL {
	IM_HANDLE			handle;		// Item to delete.
} IM_DEINSTALL, *P_IM_DEINSTALL;


/****************************************************************************
 msgIMDup					takes P_IM_DUP, returns STATUS
	Creates a new item that is a duplicate of an existing one. 

 Causes the install manager to notify observers with msgIMInstalled.

 Return Value
	stsIMAlreadyInstalled:	An item with pArgs->name already exists.
*/
#define msgIMDup		 					MakeMsg(clsInstallMgr, 23)

typedef struct IM_DUP {
	IM_HANDLE			handle;		// item to duplicate.
	P_STRING			pName;		// new name.  If pNull then a unique name
									// is generated.
	IM_HANDLE			newHandle;	// Out: Handle to the new item.
} IM_DUP, *P_IM_DUP;


/****************************************************************************
 msgIMFind					takes P_IM_FIND, returns STATUS
	Finds a item's handle, given its name.

 Return Value
	stsNoMatch:			Item not found.
*/
#define msgIMFind		 					MakeMsg(clsInstallMgr, 24)

typedef struct IM_FIND {
	P_STRING			pName;		// Resource name to search for
	IM_HANDLE			handle;		// Out: Resulting handle
} IM_FIND, *P_IM_FIND;


/****************************************************************************
 msgIMGetSema				takes P_OS_FAST_SEMA, returns STATUS
	Gets the concurrency protection semaphore. 

 This message is for subclasses that need to do concurrency	protection to 
 their messages.  Subclasses should get this semaphore and aquire and release
 it at the beginning and end of their messages.  Subclasses should use this
 semaphore instead of creating one of their own in order to avoid
 race conditions.
*/
#define msgIMGetSema		 				MakeMsg(clsInstallMgr, 25)


/****************************************************************************
 msgIMGetDir				takes P_OBJECT, returns STATUS
	Passes back a directory handle on the install manager's directory. 

 This dir handle is owned by the install manager; clients must not 
 destroy it! 
*/
#define msgIMGetDir			 				MakeMsg(clsInstallMgr, 26)


/****************************************************************************
 msgIMGetInstallPath				takes P_STRING, returns STATUS
	Passes back the install base path.

 The install base path is an absolute path to the install manager's 
 directory.

 pArgs must point to an fsPathBufLength sized buffer.
*/
#define msgIMGetInstallPath	 				MakeMsg(clsInstallMgr, 27)


/****************************************************************************
 msgIMGetVerifier				takes P_OBJECT, returns STATUS
	Passes back the current verifier object. 

 This object is sent msgIMVerify whenever an item is attempted to be 
 installed.  The verifier should return stsOK if the item is valid, 
 stsFailed if it isn't.
*/
#define msgIMGetVerifier	 				MakeMsg(clsInstallMgr, 33)


/****************************************************************************
 msgIMSetVerifier				takes OBJECT, returns STATUS
	Sets the current verifier object. 

 This object is sent msgIMVerify whenever an item is attempted to be 
 installed.  The verifier should return stsOK if the item is valid, 
 stsFailed if it isn't.
*/
#define msgIMSetVerifier	 				MakeMsg(clsInstallMgr, 34)


/****************************************************************************
 msgIMVerify				takes OBJECT, returns STATUS
	Verify the validity of an item that is being installed.

 This message is sent to an install manager's verifier object whenever an
 installation is attempted.
 
 pArgs specifies the node being installed.  It is either a file handle
 or a dir handle.  The verifier object should determine if the item to be 
 installed is valid, and return stsOK if so, stsFailed if not.
*/
#define msgIMVerify			 				MakeMsg(clsInstallMgr, 35)


/****************************************************************************
 msgIMExists				takes P_IM_EXISTS, returns STATUS
	Verify the existance of an item that is being installed.

 This message is self sent whenever an installation is attempted.
 
 pArgs specifies the node being installed.  It is either a file handle
 or a dir handle.  The handler should determine if the item to be 
 installed already exists.  Returns stsOK if the item is found; stsFailed
 otherwise.
*/
#define msgIMExists	 				MakeMsg(clsInstallMgr, 61)

typedef struct IM_EXISTS {
	OBJECT		source;		// In: {File|Dir} handle of item to be installed.
	IM_HANDLE	handle;		// Out: Handle of item if found.
} IM_EXISTS, * P_IM_EXISTS;



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	 							UI Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgIMUIInstall				takes P_IM_UI_INSTALL, returns STATUS
	Installs a new item with a user interface. 

 Performs msgIMInstall, but lets the user decide exist behavior.  Pops up
 a progress note which allows the user to cancel the install.  Informs the
 user of successful or unsucessful completion.

 Return Value
	Returns msgIMInstall statuses.
*/
#define msgIMUIInstall	 					MakeMsg(clsInstallMgr, 38)

typedef struct IM_UI_INSTALL {
	FS_LOCATOR			locator;	   // Location of item on external 
									   // filesystem.
	IM_HANDLE			handle;		   	// Out: Handle of installed item.
} IM_UI_INSTALL, *P_IM_UI_INSTALL;


/****************************************************************************
 msgIMUIDeinstall				takes P_IM_UI_DEINSTALL, returns STATUS
	Deinstalls an item with a user interface. 

 Return Value
	Returns msgIMDeinstall statuses.
*/
#define msgIMUIDeinstall	 					MakeMsg(clsInstallMgr, 58)

typedef struct IM_UI_DEINSTALL {
	IM_HANDLE			handle;			// Item to deinstall.
} IM_UI_DEINSTALL, *P_IM_UI_DEINSTALL;


/****************************************************************************
 msgIMUIDup					takes P_IM_UI_DUP, returns STATUS
	Duplicates and item with a UI.

 Return Value
	Returns msgIMDup statuses.
*/
#define msgIMUIDup			 					MakeMsg(clsInstallMgr, 39)

typedef struct IM_UI_DUP {
	IM_HANDLE			handle;		// item to duplicate.
	P_STRING			pName;		// new name.  If pNull then a unique name
									// is generated.
	IM_HANDLE			newHandle;	// Out: Handle to the new item.
} IM_UI_DUP, *P_IM_UI_DUP;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	 						Notification Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgIMNameChanged				takes P_IM_NOTIFY, returns STATUS
	category: observer notification
	The name of a item has changed. 

*/
#define msgIMNameChanged	 		   		MakeMsg(clsInstallMgr, 40)

typedef struct IM_NOTIFY {
	OBJECT				manager;	// manager that sent notification.
	IM_HANDLE			handle;		// handle that changed.
	U8					reserved[40];
} IM_NOTIFY, *P_IM_NOTIFY;


/****************************************************************************
 msgIMCurrentChanged			takes P_IM_CURRENT_NOTIFY, returns STATUS
	category: observer notification
	The current item has changed.

*/
#define msgIMCurrentChanged	 		   		MakeMsg(clsInstallMgr, 42)

typedef struct IM_CURRENT_NOTIFY {
	OBJECT				manager;	// manager that sent notification
	IM_HANDLE			newHandle;	// the new current handle
	IM_HANDLE			oldHandle;	// the previous current handle
	U8					reserved[40];
} IM_CURRENT_NOTIFY, *P_IM_CURRENT_NOTIFY;


/****************************************************************************
 msgIMInUseChanged				takes P_IM_INUSE_NOTIFY, returns STATUS
	category: observer notification
	An item's inUse attribute has changed. 

*/
#define msgIMInUseChanged	 		   		MakeMsg(clsInstallMgr, 43)

typedef struct IM_INUSE_NOTIFY {
	OBJECT				manager;	// manager that sent notification
	IM_HANDLE			handle;		// handle that changed
	BOOLEAN				inUse;		// new inUse state
	U8					reserved[40];
} IM_INUSE_NOTIFY, *P_IM_INUSE_NOTIFY;


/****************************************************************************
 msgIMModifiedChanged			takes P_IM_MODIFIED_NOTIFY, returns STATUS
	category: observer notification
	An item's modified attribute has changed. 

*/
#define msgIMModifiedChanged	 	   		MakeMsg(clsInstallMgr, 44)

typedef struct IM_MODIFIED_NOTIFY {
	OBJECT				manager;	// manager that sent notification
	IM_HANDLE			handle;		// handle that changed
	BOOLEAN				modified; 	// new modified state
	U8					reserved[40];
} IM_MODIFIED_NOTIFY, *P_IM_MODIFIED_NOTIFY;


/****************************************************************************
 msgIMInstalled					takes P_IM_NOTIFY, returns STATUS
	category: observer notification
	A new item was installed.

*/
#define msgIMInstalled		 				MakeMsg(clsInstallMgr, 45)


/****************************************************************************
 msgIMDeinstalled				takes P_IM_DEINSTALL_NOTIFY, returns STATUS
	category: observer notification
	An item has been deinstalled.

 Since the handle is no longer valid when this message is recieved, the
 pArgs includes all information about the item.
*/
#define msgIMDeinstalled	 				MakeMsg(clsInstallMgr, 46)


typedef struct IM_DEINSTALL_NOTIFY {
	OBJECT				manager;			// manager that sent notification.
	IM_HANDLE			handle;				// handle of item that was deinstalled.
	U8					pName[nameBufLength];   // item name.
	U8					pVersion[nameBufLength];// item version.
	U8					reserved[40];
} IM_DEINSTALL_NOTIFY, *P_IM_DEINSTALL_NOTIFY;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Private																   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgIMDeactivate				takes P_IM_DEACTIVATE, returns STATUS
	Deactivate an item. 

 This removes everything but an empty filesytem node with attributes 
 which represents the item.  The item's handle and attributes remain intact.
 
 Returns
	stsRequestNotSupported:		style.copyOnInstall is false.  Install mgrs
								of this style don't support deactivation.
*/
#define msgIMDeactivate	 					MakeMsg(clsInstallMgr, 20)

typedef struct IM_DEACTIVATE {
	IM_HANDLE			handle;		// item to deactivate.
} IM_DEACTIVATE, *P_IM_DEACTIVATE;


/****************************************************************************
 msgIMActivate					takes P_IM_ACTIVATE, returns STATUS
	Activate an item by copying it in from disk. 

 The install manager also sends msgIMModifiedChanged if the modified state 
 changed due to the activate.

 Return Value
	stsIMAlreadyActive:		Item is already active.
	
	stsIMInvalidItem:		There is nothing valid out on disk.
*/
#define msgIMActivate		 				MakeMsg(clsInstallMgr, 21)

typedef struct IM_ACTIVATE {
	IM_HANDLE			handle;		// Item to activate.
} IM_ACTIVATE, *P_IM_ACTIVATE;


/****************************************************************************
 msgAppMgrGetMetrics		takes P_APP_MGR_METRICS, returns STATUS
	Returns generic icon for this installer.

 Install managers understand this message so they can present an icon
 for use by the disk manager.  Install managers look for their icons in
 the system resource file. 

 Only the iconBitmap, smallIconBitmap, and name fields of pArgs are
 filled in.
*/


/****************************************************************************
 msgIMAddCards			takes P_IM_ADD_CARDS, returns STATUS
	Asks the install manager to add option cards for the specified item.

 The handle argument specifies the currently selected item.	It may be
 objNull if there is no selection.

 This message is a superset of msgOptionAddCards.  The optionTag argument
 is exactly the same as that for msgOptionAddCards.
*/
#define msgIMAddCards						MakeMsg(clsInstallMgr, 56)

typedef struct IM_ADD_CARDS {
	IM_HANDLE			handle;		// Item to add cards for. Can be objNull. 
	OPTION_TAG			optionTag;	// msgOptionAddCards argument.
} IM_ADD_CARDS, *P_IM_ADD_CARDS;


/****************************************************************************
 msgIMSetNotify				takes BOOLEAN, returns STATUS
	Turns notification generation on or off.

*/
#define msgIMSetNotify		 				MakeMsg(clsInstallMgr, 28)


/****************************************************************************
 msgIMGetNotify				takes P_BOOLEAN, returns STATUS
	Returns notification generation state.

*/
#define msgIMGetNotify		 				MakeMsg(clsInstallMgr, 29)


/****************************************************************************
 msgIMRemoveHandle			takes OBJECT, returns STATUS
	Removes and frees a handle from our internal list.

*/
#define msgIMRemoveHandle	 				MakeMsg(clsInstallMgr, 30)


/****************************************************************************
 msgIMRenameUninstalledItem		takes P_IM_RENAME_UNINSTALLED, returns STATUS
	Renames an item on disk.

*/
#define msgIMRenameUninstalledItem				MakeMsg(clsInstallMgr, 53)

typedef struct IM_RENAME_UNINSTALLED {
	FS_LOCATOR			locator;	// Location of item to rename.  Must not
									// be an absolute path!
	P_STRING			pOldName;	// Old name.
	P_STRING			pNewName;	// New name.
} IM_RENAME_UNINSTALLED, *P_IM_RENAME_UNINSTALLED;


/****************************************************************************
 msgIMGetSettingsMenu  			takes PP_TK_TABLE_ENTRY, returns STATUS
	Sets a pointer to the tkTable entries for the Settings NB menu.

 pArgs must be the address of a P_TK_TABLE_ENTRY pointer. 

*/
#define msgIMGetSettingsMenu	 				MakeMsg(clsInstallMgr, 54)


/****************************************************************************
 msgIMGetItemIcon				takes P_IM_GET_ITEM_ICON, returns STATUS
	Gets the icons for a given item.

*/
#define msgIMGetItemIcon		 	   		MakeMsg(clsInstallMgr, 57)

typedef struct IM_GET_ITEM_ICON {
	IM_HANDLE		handle;				// Handle of item.
	OBJECT			iconBitmap;			// Out: Icon bitmap.
	TAG				iconTag;			// Out: Icon's tag in resfile.
	BOOLEAN			iconInSystemRes;	// Out: Is this icon in system 
										//		resource file?
	OBJECT			smallIconBitmap;	// Out: Small icon bitmap.
	TAG				smallIconTag;		// Out: Icon's tag in resfile.
	BOOLEAN			smallIconInSystemRes;// Out: Is this icon in system
										//		 resource file?
	U32				reserved;			
} IM_GET_ITEM_ICON, *P_IM_GET_ITEM_ICON;


//REFGEN BEGINIGNORE
/****  UI tags  ****/
#define tagIMInstalling					MakeDialogTag(clsInstallMgr, 0) 
#define tagIMInstallingVersion			MakeDialogTag(clsInstallMgr, 1) 
#define tagIMSaving						MakeDialogTag(clsInstallMgr, 2) 
#define tagIMUpdating					MakeDialogTag(clsInstallMgr, 3) 
#define tagIMCopying					MakeDialogTag(clsInstallMgr, 4) 
#define tagIMDeinstalling				MakeDialogTag(clsInstallMgr, 5) 
#define tagIMDeactivating				MakeDialogTag(clsInstallMgr, 6) 
#define tagIMReactivating				MakeDialogTag(clsInstallMgr, 7) 
#define tagIMDeinstall					MakeDialogTag(clsInstallMgr, 8) 
#define tagIMDeinstallApp				MakeDialogTag(clsInstallMgr, 9) 
#define tagIMDeactivate		 			MakeDialogTag(clsInstallMgr, 10) 
#define tagIMDeactivateApp	 			MakeDialogTag(clsInstallMgr, 11) 
#define tagIMModified					MakeDialogTag(clsInstallMgr, 12) 
#define tagIMReplaceQuery 				MakeDialogTag(clsInstallMgr, 13)
#define tagIMReplaceQueryVersion		MakeDialogTag(clsInstallMgr, 14)

#define	tagIMDynTableProp  				MakeTag(clsInstallMgr, 1)


// Next up: 62

//REFGEN ENDIGNORE
#endif	// INSTLMGR_INCLUDED


