/****************************************************************************
 keycap.h

 Copyright 1992, GO Corporation, All Rights Reserved

 $Revision:   1.2  $
   $Author:   kbentley  $
	 $Date:   16 Mar 1992 14:33:54  $

 Interface for the KeyCap class.

 clsKeyCap inherits from clsWin.
 Provides an array of keycaps for keyboard emulations.

 clsKeyCap inherits from clsWin which provides support for an array
 of keyboard "caps" which can deliver a scan code and make/break events.
 Subclasses are expected to added the glyph which is displayed on the
 cap when the key is painted.  This is accomplished by intercepting
 the self-call msgKeyCapPaintCap.

 WARNING:	These API's are not currently in a suitable state for
			developers.

****************************************************************************/
#ifndef KEYCAP_INCLUDED
#define KEYCAP_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif
#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif
#ifndef UID_INCLUDED
#include <uid.h>
#endif
#ifndef WIN_INCLUDED
#include <win.h>
#endif

#define maxCaps	 150				// max for the KEYCAP_TABLE declaration

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *               Cap Width Descriptors                                     *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*
 A data table based mechanism is used to define the array of key caps.
 Each row is a fixed height (determined by dividing the window by the
 number of rows).  Each cap can have one of five widths, small, medium, 
 large, extra large and huge.  A small cap is the basic unit of measure,
 all other cap sizes are multiples of this size.  This size is determined
 by dividing the window width by the number of switch units.  The cap
 sizes are: small = 1 unit, medium = 1.5 units, large = 2 units, 
 extra large = 2.5 units, and huge = 7 units.
*/
#define kcEND		(0x0000)		// end of row marker
#define kcS			(0x1000)		// small cap
#define kcM			(0x2000)		// medium cap
#define kcL			(0x3000)		// large cap
#define kcX			(0x4000)		// extra large cap
#define kcH			(0x5000)		// huge cap

typedef struct KEYCAP_TABLE {
	U16 rows;						// number of rows
	U16 switches;					// number of column units
	U16 capCodes[maxCaps];			// array of scan codes with
									//  cap width descriptor (high nibble)
									//  Each row must end with kcEnd and
									//  the table must end with two
									//  kcEnd tokens.
} KEYCAP_TABLE, *P_KEYCAP_TABLE;


/****  Class Messages  ****/

/****************************************************************************
 msgNewDefaults						takes P_KEYCAP_NEW, returns STATUS
	Initializes the new arguments.

 The pTable field is initialized to pNull by default.
*/
typedef struct KEYCAP_NEW_ONLY {
	P_KEYCAP_TABLE pTable;		  // pointer to the keycap table
} KEYCAP_NEW_ONLY, *P_KEYCAP_NEW_ONLY;

#define keyCapNewFields			\
	OBJECT_NEW_ONLY object;		\
	WIN_NEW_ONLY	win;		\
	KEYCAP_NEW_ONLY keyCap;

typedef struct KEYCAP_NEW {
	keyCapNewFields
} KEYCAP_NEW, *P_KEYCAP_NEW;

/****************************************************************************
 msgNew							takes P_KEYCAP_NEW, returns STATUS
	Creates a new instance of the keycap object.

 If the pTable pointer is NULL, the standard QWERTY layout is used by
 default.
*/


/****************************************************************************
 msgKeyCapPaintCap				takes P_KEYCAP_INFO, returns STATUS
	Hook to allow painting on top of keycap.

 This is the self-call hook which allows subclasses the ability to paint
 the glyph on the keycap.  This call is made during the response to
 msgWinRepaint and is therefore already bracketed by msgWinBeginRepaint,
 msgWinEndRepaint.
*/
typedef struct KEYCAP_INFO {
	U16 scanCode;				// scan code for the keycap
	RECT32 rect;				// location of the keycap rect, LWC
	BOOLEAN hilite;				// TRUE for highlighted display
	OBJECT dc;					// Drawing context
} KEYCAP_INFO, *P_KEYCAP_INFO;
#define msgKeyCapPaintCap		MakeMsg(clsKeyCap, 1)

/****************************************************************************
 msgKeyCapScan					takes P_KEYCAP_SCAN, returns STATUS
	Locates the keycap under a given x,y.

 This function returns the keycap which is under the Local Window 
 Coordinates (LWC) xy.
*/
typedef struct KEYCAP_SCAN {
	XY32 xy;				// coordinates for search
	U16 scanCode;			// Out: scan code of keycap
	RECT32 rect;			// Out: keycap rect in LWC
} KEYCAP_SCAN, *P_KEYCAP_SCAN;
#define msgKeyCapScan			MakeMsg(clsKeyCap, 2)


/****************************************************************************
 msgKeyCapGetDc					takes P_KEYCAP_GET_DC, returns STATUS
	Returns the DC used by the keycap.
*/
typedef struct KEYCAP_GET_DC {
	OBJECT dc;				// Out: keycap dc object
} KEYCAP_GET_DC, *P_KEYCAP_GET_DC;
#define msgKeyCapGetDc			MakeMsg(clsKeyCap, 3)

/****************************************************************************
 msgKeyCapRedisplay				takes nothing, returns STATUS
	Forces the display to be regenerated.
*/
#define msgKeyCapRedisplay		MakeMsg(clsKeyCap, 5)

/****************************************************************************
 msgKeyCapHilite				takes P_KEYCAP_HILITE, returns STATUS
	Hilites the key with the given scan code.

 The key cap object tracks which keys (by scan code) are highlighted
 at any given time.
*/
typedef struct KEYCAP_HILITE {
	U16 scan;					// In: scan code to hilite
	BOOLEAN on;					// In: true to display as hilite
} KEYCAP_HILITE, *P_KEYCAP_HILITE;
#define msgKeyCapHilite			MakeMsg(clsKeyCap, 6)

/****************************************************************************
 msgKeyCapMake					takes P_KEYCAP_INFO, returns STATUS
	Subclass hook to indicate button press of keycap.

 This message is a self-call when the pen touches down on a keycap.  Note
 that only one make/break event pair is generated for each penDown, penUp
 combination.  Sliding off a key onto another is NOT considered a press
 on the new key.
*/
#define msgKeyCapMake			MakeMsg(clsKeyCap, 0x80)

/****************************************************************************
 msgKeyCapBreak					takes P_KEYCAP_INFO, returns STATUS
	Subclass hook to indicate button release of keycap.

 This message is a self-call when the pen is lifted up from the
 previous make event.
*/
#define msgKeyCapBreak			MakeMsg(clsKeyCap, 0x81)



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *               Private Messages                                          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 msgKeyCapShift					takes void, returns STATUS
	OBSOLETE message, will be removed in a future release. Just calls
	msgKeyCapRedisplay now.
*/
#define msgKeyCapShift			MakeMsg(clsKeyCap, 4)

#endif // KEYCAP_INCLUDED
