/****************************************************************************
 File: label.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.69  $
   $Author:   cmeyer  $
     $Date:   18 Mar 1992 14:46:40  $

 This file contains the API definition for clsLabel.

 clsLabel inherits from clsControl.
 Implements much of the appearance of many toolkit components inside the
 border: font, decoration, scale, orientation, etc.
****************************************************************************/

/**** Debugging Flags ****/
/*
 The clsLabel debugging flag is '%'.  Defined values are:

    flag4  (0x0010):    msgSave/msgRestore info
    flag5  (0x0020):    boxed string/paint dc
*/

#ifndef LABEL_INCLUDED
#define LABEL_INCLUDED

												#ifndef CONTROL_INCLUDED
#include <control.h>
												#endif
												#ifndef SYSGRAF_INCLUDED
#include <sysgraf.h>
												#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT LABEL;

/****  Info style  ****/
#define	lsInfoString			0	// info is pString
#define	lsInfoWindow			1	// info is WIN
#define	lsInfoStringId			2	// info is a resource file string id
//								3	// unused (reserved)

/****  X and Y alignment styles  ****/
#define	lsAlignLeft				0				// left-justified
#define	lsAlignCenter			1				// centered
#define	lsAlignRight			2				// right-justified
#define	lsAlignBottom			lsAlignLeft		// bottom-justified
#define	lsAlignTop				lsAlignRight	// top-justified
#define	lsAlignCustom			3				// send msgLabelAlign to self

/****  Decoration style  ****/
#define	lsDecorationNone			0	// no decoration
#define	lsDecorationBlank			1	// blank space on left
#define	lsDecorationNonExclusiveOn	2	// left check and double bar
#define	lsDecorationExclusiveOff	3	// left blank and single bar
#define	lsDecorationExclusiveOn		4	// left check and single bar
#define	lsDecorationPullRight		5	// pull-right arrow on right
#define	lsDecorationNonExclusiveOff	6	// left blank and double bar
#define	lsDecorationCheck			7	// left checkmark
#define	lsDecorationCircle			8	// left empty circle
#define	lsDecorationBox				9	// left empty box
#define	lsDecorationCheckedBox		10	// left checked box
#define	lsDecorationCheckedCircle	11	// left checked circle
#define	lsDecorationHollowLeft		12	// left hollow left delta
#define	lsDecorationHollowRight		13	// left hollow right delta
#define	lsDecorationSolidLeft		14	// left solid left delta
#define	lsDecorationSolidRight		15	// left solid right delta
#define	lsDecorationPopup			16	// left solid right delta w/space
#define	lsDecorationButtonOff		17	// left off button glyph
#define	lsDecorationButtonOn		18	// left on button glyph
#define	lsDecorationCustomLeft		19	// left custom decoration
#define	lsDecorationCustomRight		20	// right custom decoration
//									21	// unused (reserved)
//									..	// unused (reserved)
//									31	// unused (reserved)

/****  Font Type  ****/
#define	lsFontSystem				0	// use the system font
#define	lsFontCustom				1	// use the specified font
#define	lsFontUser					2	// use the system user font
//									3	// unused (reserved)

/****  Common Scale Values, in layout units  ****/
#define	lsScaleTiny					2	//  2/8 x normal
#define	lsScaleSmall				4	//  4/8 x normal
#define	lsScaleMedium				6	//  6/8 x normal
#define	lsScaleNormal				8	//  8/8 x normal
#define	lsScaleLarge				10	// 10/8 x normal
#define	lsScaleJumbo				12	// 12/8 x normal
#define	lsScaleHuge					14	// 14/8 x normal

/****  Rotation styles  ****/
#define	lsRotateNone		0	//   0 degrees (horizontal, left to right)
#define	lsRotate90			1	//  90 degrees (vertical, bottom to top)
#define	lsRotate180			2	// 180 degrees (horizontal, right to left)
#define	lsRotate270			3	// 270 degrees (vertical, top to bottom)

/****  Underline styles  ****/
#define	lsUnderlineNone		0	// no underline
#define	lsUnderlineSingle	1	// single underline
#define	lsUnderlineDouble	2	// double underline
//							3	// unused (reserved)

/****  Box styles  ****/
#define	lsBoxNone			0	// no boxes
#define	lsBoxSquare			1	// square box around each character
#define	lsBoxTicks			2	// tick mark between characters
#define	lsBoxInvisible		3	// don't draw the box lines


/****  Number of rows/columns  ****/
#define	lsNumAsNeeded		0	// as many rows/columns as needed
#define	lsNumAbsolute		1	// fixed number: rows/cols
//							2	// unused (reserved)
//							3	// unused (reserved)


typedef struct LABEL_STYLE {
	U16	infoType		: 2,	// type of pString field
		xAlignment		: 2,	// x alignment style
		yAlignment		: 2,	// y alignment style
		rotation		: 2,	// text rotation
		underline		: 2,	// underline style
		strikeout		: 1,	// strikeout during msgDcDrawText
		decoration		: 5;	// decoration style

	U16	numCols			: 2,	// style for number of columns
		numRows			: 2,	// style for number of rows
		box				: 2,	// boxing style
		wordWrap		: 1,	// word wrap to next row
		fontType		: 2,	// system or custom font
		scaleUnits		: 6,	// scale units style, e.g. bsUnitsLayout
		stringSelected	: 1;	// whether content string shows sel'd visual

	U16	spare2			: 16;	// unused (reserved)
} LABEL_STYLE, *P_LABEL_STYLE;

/*
 Default LABEL_STYLE:
//{
 	infoType		= lsInfoString
 	xAlignment		= lsLeft
 	yAlignment		= lsBottom
 	decoration		= lsDecorationNone
 	scaleUnits		= bsUnitsLayout
 	rotation		= lsRotateNone
 	underline		= lsUnderlineNone
 	strikeout		= false
 	box				= lsBoxNone
 	numCols			= lsNumAsNeeded
 	numRows			= lsNumAsNeeded
 	wordWrap		= false
 	fontType		= lsFontSystem
 	scaleUnits		= bsUnitsLayout
 	stringSelected	= false
//}
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_LABEL_NEW, returns STATUS
	category: class message
	Creates a label window.

 The fields you commonly set are:
	pArgs->label.style:		appropriate style values
	pArgs->label.pString:	string or child window uid

 In response to msgNew, the label initializes all of its state.
 This is the only time pArgs->fontName would be used.

 Since clsLabel copies the bytes pointed to by pArgs->pString (when
 style.infoType is lsInfoString), the client may free the string after
 msgNew if the string was allocated.

 If style.infoType is lsInfoStringId, clsLabel self-sends
 msgLabelBindStringId to bind the resid to a string.
*/

typedef struct LABEL_NEW_ONLY {
	LABEL_STYLE		style;		// overall style
	P_CHAR			pString;	// string to display or child window
	SYSDC_FONT_SPEC	font;	// spec to open if style.fontType == lsFontCustom
	P_CHAR			fontName;	// font name from which to derive font.id
	U8				scale;		// scale in scaleUnits
	U8				rows;		// number of rows
	U8				cols;		// number of columns (or zero for infinite)
	U16				customGlyph;// custom decoration glyph
	U32				spare1;		// unused (reserved)
	U32				spare2;		// unused (reserved)
} LABEL_NEW_ONLY, *P_LABEL_NEW_ONLY;

#define	labelNewFields	\
	controlNewFields 	\
	LABEL_NEW_ONLY		label;

typedef struct LABEL_NEW {
	labelNewFields
} LABEL_NEW, *P_LABEL_NEW;


/****************************************************************************
 msgNewDefaults				takes P_LABEL_NEW, returns STATUS
	category: class message
	Initializes the LABEL_NEW structure to default values.

 Zeroes out pArgs->label and sets:
//{
	pArgs->win.flags.style |= wsShrinkWrapWidth | wsShrinkWrapHeight;

	pArgs->border.style.leftMargin = bsMarginSmall;
	pArgs->border.style.rightMargin = bsMarginSmall;
	pArgs->border.style.bottomMargin = bsMarginSmall;
	pArgs->border.style.topMargin = bsMarginSmall;

	pArgs->label.style.scaleUnits = bsUnitsLayout;
	pArgs->label.scale = lsScaleNormal;
//}

 Also sets pArgs->label.font to the default system font.
*/


/****************************************************************************
 msgLabelGetStyle		takes P_LABEL_STYLE, returns STATUS
	Passes back the current style values.
*/
#define msgLabelGetStyle 		MakeMsg(clsLabel, 1)


/****************************************************************************
 msgLabelSetStyle		takes P_LABEL_STYLE, returns STATUS
	Sets the style fields.

 If the decoration style changes, the label uses msgWinDirtyRect
 to dirty the appropriate portion of itself.

 If the new style.box is not lsBoxNone, then the label self-sends
 msgLabelProvideBoxSize to obtain the width and height the boxes should be.
 If either of these differ from the old values, then the label 
 self-sends msgWinSetLayoutDirty(true).

 If the style.numCols or style.numRows change, or any of the other style
 values that might require relayout change, label self-sends
 msgWinSetLayoutDirty(true).

 It is the caller's responsibility to re-layout the label if the caller
 has changed any style that affects the layout of the label.
*/
#define msgLabelSetStyle 		MakeMsg(clsLabel, 2)


/****************************************************************************
 msgLabelGetString		takes P_CONTROL_STRING, returns STATUS
	Fills P_ARGS->pString with the current string.

 Will fill the passed buffer up to len bytes worth of the string.  The
 copied string is not null-terminated if the passed len wasn't
 large enough.

 If the passed len is zero, clsLabel sets len to the number of bytes
 the buffer would have to be in order to hold the entire label's string
 (including the terminating null).
*/
#define msgLabelGetString		MakeMsg(clsLabel, 3)


/****************************************************************************
 msgLabelSetString		takes P_CHAR, returns STATUS
	Sets the label string.

 Allocates storage and copies P_ARGS.  Note that clsLabel allocates within
 the context of the current process.
*/
#define msgLabelSetString		MakeMsg(clsLabel, 4)


/****************************************************************************
 msgLabelGetUnicode		takes P_CONTROL_STRING, returns STATUS
	Fills P_ARGS->pString with the current string.

 Like msgLabelGetString, except that the client is requesting the string
 in unicode format (where a character is represented in 16 bits).

 Will fill the passed buffer up to len characters worth of the string.  The
 copied string is not null-terminated if the passed len wasn't
 large enough.

 If the passed len is zero, clsLabel sets len to the number of characters
 the buffer would have to be in order to hold the entire label's string
 (including the terminating null).  Note that the number of bytes would be
 twice this number.
*/
#define msgLabelGetUnicode		MakeMsg(clsLabel, 21)


/****************************************************************************
 msgLabelSetUnicode		takes P_U16 (P_CHAR after its 16-bit), returns STATUS
	Sets the label string.

 Like msgLabelSetString, except that the client is specifying the string
 in unicode format (where a character is represented in 16 bits).

 Allocates storage and copies P_ARGS.  Note that clsLabel allocates within
 the context of the current process.
*/
#define msgLabelSetUnicode		MakeMsg(clsLabel, 22)


/****************************************************************************
 msgLabelGetStringId		takes P_RESID, returns STATUS
	Passes back the string resource id; zero if none.

 clsLabel returns stsFailed if style.infoType is not lsInfoStringId.
*/
#define msgLabelGetStringId		MakeMsg(clsLabel, 25)


/****************************************************************************
 msgLabelSetStringId		takes RESID, returns STATUS
	Sets the string resource id.

 clsLabel immediately binds the specified string id to a string by
 self-sending msgLabelBindStringId.

 The string id is remembered and saved during msgSave.
*/
#define msgLabelSetStringId			MakeMsg(clsLabel, 26)


/****************************************************************************
 msgLabelBindStringId		takes VOID, returns STATUS
	Binds the string resource id to a string.

 clsLabel returns stsFailed if style.infoType is not lsInfoStringId.

 clsLabel binds the current string id to a string by loading
 the string from theProcessResList.
*/
#define msgLabelBindStringId			MakeMsg(clsLabel, 27)


/****************************************************************************
 msgLabelGetWin		takes P_WIN, returns STATUS
	Passes back the child window.

 clsLabel returns stsFailed if style.infoType is not lsInfoWin.
*/
#define msgLabelGetWin			MakeMsg(clsLabel, 5)


/****************************************************************************
 msgLabelSetWin		takes WIN, returns STATUS
	Sets the child window.

 clsLabel returns stsFailed if style.infoType is not lsInfoWin.

 Since changing the window within the label sets the label's wsLayoutDirty
 bit, the caller should re-layout the label.
*/
#define msgLabelSetWin			MakeMsg(clsLabel, 6)


/****************************************************************************
 msgLabelGetFontSpec		takes P_SYSDC_FONT_SPEC, returns STATUS
	Passes back the font spec.

 Note that this font spec is not used unless style.fontType is lsFontCustom.
*/
#define msgLabelGetFontSpec		MakeMsg(clsLabel, 8)


/****************************************************************************
 msgLabelSetFontSpec		takes P_SYSDC_FONT_SPEC, returns STATUS
	Sets the font spec.

 Note that this font spec is not used unless style.fontType is lsFontCustom.

 As with msgLabelSetStyle, it is the caller's responsibility to re-layout
 the label if the caller has changed any style that affects the layout
 of the label (such as certain fields in the font spec).  Note that
 the label does not currently explicitly set its wsLayoutDirty bit
 in response to msgLabelSetFontSpec, but that this may change in the future.
*/
#define msgLabelSetFontSpec		MakeMsg(clsLabel, 9)


/****************************************************************************
 msgLabelGetScale		takes P_U8, returns STATUS
	Passes back the font scale.

 Note that the units of this scale are determined by style.scaleUnits.
*/
#define msgLabelGetScale		MakeMsg(clsLabel, 10)


/****************************************************************************
 msgLabelSetScale		takes U8, returns STATUS
	Sets the font scale.

 Note that the units of this scale are determined by style.scaleUnits.

 As with msgLabelSetStyle, it is the caller's responsibility to re-layout
 the label if the caller has changed any style that affects the layout
 of the label (such as the font scale).  Note that the label does not
 currently explicitly set its wsLayoutDirty bit in response to
 msgLabelSetScale, but that this may change in the future.
*/
#define msgLabelSetScale		MakeMsg(clsLabel, 11)


/****************************************************************************
 msgLabelGetRows		takes P_U8, returns STATUS
	Passes back the number of rows the label will size itself to.

 Note that this is not used unless style.numRows is lsNumAbsolute.
*/
#define msgLabelGetRows			MakeMsg(clsLabel, 12)


/****************************************************************************
 msgLabelSetRows		takes U8, returns STATUS
	Sets the number of rows the label will size itself to.

 Note that this is not used unless style.numRows is lsNumAbsolute.

 As with msgLabelSetStyle, it is the caller's responsibility to re-layout
 the label if the caller has changed any style that affects the layout
 of the label (such as the number of rows).  Note that the label does not
 currently explicitly set its wsLayoutDirty bit in response to
 msgLabelSetRows, but that this may change in the future.
*/
#define msgLabelSetRows			MakeMsg(clsLabel, 13)


/****************************************************************************
 msgLabelGetCols		takes P_U8, returns STATUS
	Passes back the number of columns the label will size itself to.

 Note that this is not used unless style.numCols is lsNumAbsolute.
*/
#define msgLabelGetCols			MakeMsg(clsLabel, 14)


/****************************************************************************
 msgLabelSetCols		takes U8, returns STATUS
	Sets the number of columns the label will size itself to.

 Note that this is not used unless style.numCols is lsNumAbsolute.

 As with msgLabelSetStyle, it is the caller's responsibility to re-layout
 the label if the caller has changed any style that affects the layout
 of the label (such as the number of columnss).  Note that the label does not
 currently explicitly set its wsLayoutDirty bit in response to
 msgLabelSetCols, but that this may change in the future.
*/
#define msgLabelSetCols			MakeMsg(clsLabel, 15)


/****************************************************************************
 msgLabelGetCustomGlyph		takes P_U16, returns STATUS
	Passes back the custom decoration glyph, zero if none.

 Note that this is not used unless style.decoration is
 lsDecorationCustomLeft or lsDecorationCustomRight.
*/
#define msgLabelGetCustomGlyph	MakeMsg(clsLabel, 23)

/****************************************************************************
 msgLabelSetCustomGlyph		takes U16, returns STATUS
	Sets the custom decoration glyph.

 Note that this is not used unless style.decoration is
 lsDecorationCustomLeft or lsDecorationCustomRight.
*/
#define msgLabelSetCustomGlyph	MakeMsg(clsLabel, 24)

/****************************************************************************
 msgLabelGetBoxMetrics		takes P_LABEL_BOX_METRICS, returns STATUS
	Passes back the current box metrics.

 The box metrics describe the arrangement and size of the box cells imaged
 by the label.  These metrics are valid only if style.box is not lsBoxNone.

 All origins and sizes are in device units.
*/
#define msgLabelGetBoxMetrics	MakeMsg(clsLabel, 16)

typedef struct LABEL_BOX_METRICS {
	RECT32	boxRect;		// origin and size of boxed area
	SIZE32	singleBoxSize;	// size of a single box
	U16		rows, cols;		// current # of rows and columns
	U16		baseline;		// positive baseline offset from bottom of box
	U32		spare1;			// unused (reserved)
	U32		spare2;			// unused (reserved)
	U32		spare3;			// unused (reserved)
} LABEL_BOX_METRICS, *P_LABEL_BOX_METRICS;


/****************************************************************************
 msgLabelResolveXY		takes P_LABEL_RESOLVE, returns STATUS
	Resolves a point to a character in the string.

 An index of -1 indicates point is not over any character.
 The xy should be relative to the label and expressed in device units.
*/
#define msgLabelResolveXY		MakeMsg(clsLabel, 17)

typedef struct LABEL_RESOLVE {
	XY32	xy;				// in: point to resolve
	S32		index;			// out: index of char at point
	U32		spare1;			// unused (reserved)
	U32		spare2;			// unused (reserved)
} LABEL_RESOLVE, *P_LABEL_RESOLVE;


/****************************************************************************
 msgLabelAlign		takes P_LABEL_ALIGN, returns STATUS
	category: self-sent
	Self-sent if style.xAlignment or style.yAlignment is lsAlignCustom.

 Allows subclasses to compute alignment.  The subclass should fill in
 pArgs->offset.
*/
#define msgLabelAlign			MakeMsg(clsLabel, 7)

typedef struct LABEL_ALIGN {
	BOOLEAN	alignX;			// in: true if x alignment
	SIZE16	outerSize;		// in: size of label outer rect (in twips)
	SIZE16	innerSize;		// in: size of label inner rect (in twips)
	SIZE16	contentsSize;	// in: size of label contents (in twips)
	COORD16	offset;			// out: computed x or y offset from origin
	U32		spare;			// unused (reserved)
} LABEL_ALIGN, *P_LABEL_ALIGN;


/****************************************************************************
 msgLabelProvideInsPt		takes P_S16, returns STATUS
	category: self-sent
	Self-sent message to obtain where to render insertion point.

 Receiver should return the zero-based index of the character at
 which the insertion point should be drawn.  Non-boxed styles draw
 the insertion point before this character, boxed styles highlight the
 box around this character.
	
 If the returned index is -1, no insertion point is drawn.
 clsLabel responds by default with -1.
*/
#define msgLabelProvideInsPt		MakeMsg(clsLabel, 18)


/****************************************************************************
 msgLabelGetRects			takes P_LABEL_RECT, returns STATUS
	Computes the rectangle for each given character index.

 pArgs points to an array of LABEL_RECTs.  The receiver computes
 the rectangle for the character at the index for each index until
 it encounters one whose value is -1.  The rects are relative to
 the label, and are expressed in device units.

 The indices should be sorted in increasing order.
*/
#define msgLabelGetRects			MakeMsg(clsLabel, 19)

#define lgrInsPtRect	flag0

typedef struct {
	S16		index;
	RECT32	rect;
	U16		flags;
	U16		spare;
} LABEL_RECT, *P_LABEL_RECT;


/****************************************************************************
 msgLabelProvideBoxSize		takes P_SIZE16, returns STATUS
	category: self-sent
	Self-sent message to obtain the char box size.

 Receiver should fill in *pArgs with the size of a character box, in points.
 This message is self-sent when a boxed label is processing the following
 messages: msgInit, msgRestore, msgLabelSetString, and msgLabelSetStyle.
	
 clsLabel responds by filling in *pArgs from the user preferences
 (using prCharBoxWidth and prCharBoxHeight from prefs.h).
*/
#define msgLabelProvideBoxSize		MakeMsg(clsLabel, 20)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from Other Classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgWinLayoutSelf takes P_WIN_METRICS, returns STATUS
    Tell a window to layout its children.

 clsLabel responds by recomputing its layout parameters and by using
 msgWinDelta on its child window (if style.infoType is lsInfoWindow).
*/


/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes an object to file itself in an object file.

 clsLabel responds by filing away all its state, including its string (if
 style.infoType is lsInfoString) or child window (if style.infoType is
 lsInfoWindow).

 Note that the child window must have wsSendFile set to be filed.  If
 wsSendFile is not set, then msgSave does not save it, and a subsequent
 msgRestore sets the label's pString field to objNull.  (wsSendFile
 is the default for clsBorder and its descendents).
*/


/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.

 clsLabel responds by restoring all of its state, including its string (if
 style.infoType is lsInfoString) or child window (if style.infoType is
 lsInfoWindow).

 If the child window was not filed during the msgSave, then after msgRestore
 the label's pString value is objNull.
*/


/****************************************************************************
 msgFree	takes OBJ_KEY, returns STATUS
	Sent as the last of three msgs to destroy an object.

 clsLabel responds by freeing its string if style.infoType is lsInfoString
 and the string pointer is not null.  clsLabel uses OSHeapBlockFree.
*/


/****************************************************************************
 msgWinRepaint takes nothing, returns STATUS
    category: descendant responsibility
    Tells a window to repaint itself.

 clsLabel responds by painting its decoration and string as appropriate.
*/


/****************************************************************************
 msgWinGetBaseline takes P_WIN_METRICS, returns STATUS
    Gets the desired x,y alignment of a window.

 clsLabel responds by setting pArgs->bounds.origin.
 
 If the label is displaying a decoration, the x coordinate is set to the
 x offset of the rightmost decoration position (there's a small gap between
 this position and the start of the string/window).  If the label has no
 decoration, then the x coordinate is set to the offset of the left side
 of the string/window.

 The y coordinate is set to a value derived from the label's innerRect
 origin and the baseline information from the label's font.  This value
 is accurate in those cases where the label's bottom fits snugly
 around the string/window, but is incorrect in cases where this doesn't
 hold (e.g., a non-wsShrinkWrapHeight label that is taller than it needs
 to be).

 See Also
	msgBorderGetInnerRect:	baseline coordinates are derived from this
*/


/****************************************************************************
 msgEmbeddedWinGetMark		takes P_EMBEDDED_WIN_MARK, returns STATUS
	Get an embedded window mark.  

 clsLabel responds by copying into pArgs->label, then ensures that the
 buffer is terminated with a null character.

 If style.infoType is not lsInfoString, or the label's string is null or
 empty, then clsLabel does nothing.
*/

/****************************************************************************
 msgBorderPaintForeground		takes VOID, returns STATUS
	catagory:	subclass window responsibility
	Receiver must paint the foreground, if any.

 clsLabel responds by using msgWinBeginPaint, painting its decoration
 and string as appropriate, and then sending msgWinEndPaint.
*/


/****************************************************************************
 msgControlSetDirty		takes BOOLEAN, returns STATUS
	Clears the dirty bit.

 clsLabel responds by calling ancestor, then checking the
 CONTROL_STYLE.showDirty value.  If this is false, clsLabel just returns.
 Otherwise, if the old CONTROL_STYLE.dirty value is different from the
 new value, then clsLabel uses msgWinDirtyRect to dirty its decoration
 (if it has one).

 See Also
	msgControlSetStyle:		sets the CONTROL_STYLE values
	msgControlSetMetrics:	sets the CONTROL_METRICS values
	msgWinDirtyRect:		dirties a portion of a window
*/


/****************************************************************************
 msgControlSetStyle		takes P_CONTROL_STYLE, returns STATUS
	Sets the style values.

 clsLabel responds by calling ancestor, then checking the
 CONTROL_STYLE.showDirty value.  If this is false, clsLabel just returns.
 Otherwise, if the old CONTROL_STYLE.dirty value is different from the
 new value, then clsLabel uses msgWinDirtyRect to dirty its decoration
 (if it has one).

 See Also
	msgControlSetDirty:		sets the CONTROL_STYLE.dirty bit
	msgControlSetMetrics:	sets the CONTROL_METRICS values
	msgWinDirtyRect:		dirties a portion of a window
*/


/****************************************************************************
 msgControlSetMetrics		takes P_CONTROL_METRICS, returns STATUS
	Sets the metrics.

 clsLabel responds by calling ancestor, then checking the
 CONTROL_STYLE.showDirty value.  If this is false, clsLabel just returns.
 Otherwise, if the old CONTROL_STYLE.dirty value is different from the
 new value, then clsLabel uses msgWinDirtyRect to dirty its decoration
 (if it has one).

 See Also
	msgControlSetStyle:		sets the CONTROL_STYLE values
	msgControlSetDirty:		sets the CONTROL_STYLE.dirty bit
	msgWinDirtyRect:		dirties a portion of a window
*/


#endif	// LABEL_INCLUDED
