/****************************************************************************
 mark.h
 
 (C) Copyright 1992, GO Corporation, All Rights Reserved.
 
 $Revision:   1.14  $
   $Author:   mlentczn  $
     $Date:   26 Feb 1992 13:23:30  $

 clsMark
 
 clsMark inherits from clsObject.

 clsMark provides a path to interact with data in other, possibly
 nested, components.  It is used by Goto Buttons, Search/Replace,
 and Spell.
		  
 See Also goto.h, sr.h
****************************************************************************/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Overview									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 PenPoint allows parts of the system, such as search and replace, spell
 checking and reference buttons, to operate and refer to data in any
 document or nested embedded documents.  This generic reference to data
 is done with clsMark.  In order for clsMark to work, applications must
 support the client side of clsMark's protocol and the client side of
 the various system protocols (which are described elsewhere,
 specifically in sr.h and goto.h).

 What follows describes in more detail how clsMark relates to the rest
 of PenPoint.  You may wish to skip ahead to the Example and/or Quick
 Start sections and refer back to here later.
*/

/*
 In PenPoint, the data that make up a document are held in one or more
 'components'.  Typically these components are descendants of
 clsEmbeddedWin.  Alternatively, your descendant of clsApp might hold
 all the data.  In either case the individual pieces of data (individual
 words in a text component, shapes in a drawing component, etc.) are
 only accessible via the component object.  No other object knows how
 the data is actually stored and the data are not usually accessible as
 objects outside the component.

 There are times, however, when these individual data items need to be
 manipulated from outside of your application.  Goto buttons, for
 example, allow a user to create a link to such a data item, and later
 turn back to it.  Search & Replace, which is driven by a PenPoint
 supplied manager, needs to access successive pieces of text in both
 your application and documents embedded within you.

 An instance of clsMark, (from now on, simply 'a mark',) is a reference
 to a data item in a component.  We call this data item the 'target' as
 it is this data item that a Goto button or Search is really interested
 in, not the component that contains it.  The object that uses the mark
 is called the 'holder' of the mark.  A mark may be persistent or
 temporary.  In the former case, once established a mark will remain
 valid across document shut-downs and re-boots.  In the latter, the mark
 is valid only so long as the component remains active.

 In order to support marks a component must create a mapping between the
 two U32s, or 'token', that a mark holds and its data items.  For
 example, a data base might use this to hold a record number.  Remember
 that the mark might persist beyond a single operation.  Therefore, a
 text editor would NOT use these U32s to be a character position.  This
 is because if a mark is created for a word, and then text is deleted
 before the word, the desired action is for the mark to still refer to
 the word which has now moved in character position.  Remember: once a
 mark has been created for a piece of data, there is no way for the
 component to update the token it has given for it.

 						
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							An Example									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The process of using a mark is best illustrated by the Search & Replace
 mechanism in PenPoint:

 Search/Replace, Spelling, etc.  use the mark mechanism to traverse the
 contents of applications.  All applications that allow themselves to be
 searched, spelled or printed support the component half of this
 protocol.  Implementors of new functionality similar to Search/Replace,
 Spell, or Print must implement the driver half of the protocol.

 When the user selects Find from the Edit menu, the Search Manager
 responds by displaying an option sheet and by creating a mark which
 initially points to the document the user is working in.

 As the user requests find and replace operations, the Search Manager
 calls the mark with msgMarkDeliver with arguments specifying the clsSR
 messages it wants sent to the component.  In turn, the mark sends those
 messages, along with its own messages to the component and, if
 requested, each nested component.  It is these messages that a client
 must implement.  (The clsMark messages are described in this file, the
 clsSR messages are in sr.h.)

 After the component performs the request find and/or replace, the
 status is passed back all the way to the Search Manager which lets the
 user know.  Note that the messages to hilite and select the found text
 are also passed from the Search Manager to the component this way.

*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Quick Start									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** How to be a Client that Supports Marks ****/
/*

 1) You must decide how to refer to the data items in the component via
 tokens.  There are several considerations:  How will you treat marks that
 survive save & restore?  How will the mark be affected by edit
 operations?  What is the ordering of data items (even if the data items
 have no intrinsic ordering, you will still need a way to enumerate over
 them in some serial order)?  Do you inherit markable data from your
 ancestor that you don't take care of.

 2) Support the basic messages:
	-:	msgMarkCreateToken
	-:	msgMarkDeleteToken (if necessary)
	-:	msgMarkGetDataAncestor

 3) For a component that can be traversed, support the following.  These
 are typically very easy to implement, and all markable components should
 support them.
	-:	msgMarkPositionAtEdge
	-:	msgMarkPositionAtToken
	-:	msgMarkCompareTokens

 4) If the component has a graphical view of the data, support the
 following.  This allow Goto Buttons to work (first three messages) and the
 Search & Replace and Spell gestures to work (last message).
	-:	msgMarkShowTarget
	-:	msgMarkSelectTarget (if it can hold the selection)
	-:	msgMarkPositionAtSelection (if it can hold the selection)
	-:	msgMarkPositionAtGesture (if it can target gestures to data)

 5) If the component has any text as data, support the following.  These
 support the text side of both Search & Replace and Spell.  They are also
 used by reference buttons.  See sr.h for a description of these messages.
	-:	msgSRNextChars					 
	-:	msgSRGetChars
	-:	msgSRPositionChars
	-:	msgSRReplaceChars (if replacement is possible)

 6) If your component manages its own embedees, support:
 	-:	msgMarkPositionAtChild			  
	-:	msgMarkNextChild
	-:	msgMarkGetChild


 7) If you component is not a descendant of clsEmbeddedWin or clsApp then
 it must support the following messages:
	-:	msgMarkGetParent
	-:	msgMarkGetUUIDs
*/

/**** How to be a Driver that Uses Marks ****/
/*
 1) Send msgNewDefaults and msgNew to clsMark.  This creates the initial
 mark and sets up the component for the mark.

 2) Send the appropriate msgMarkPosition... message.  This sets the mark at
 the place where you want it.  You are free to define new kinds of
 positioning messages, so long the components you work with support them.
 As a back-up, you should always be prepared to deal with
 stsMsgNotUnderstood as a response from a message sent to a component.  In
 that case, do the default action (try msgMarkPositionAtEdge).

 3) If you need to manipulate the mark, send messages via msgMarkDeliver,
 msgMarkDeliverPos, and/or msgMarkDeliverNext.  These will instruct the
 component to take the appropriate action on the target and the mark.
 Again, be prepared to deal with stsMsgNotUnderstood.  Try to use
 standardized messages, such as msgSRNextChars, when your specific ones
 fail.  Remember: an embedded document may not know the protocol the
 enclosing document does and vice versa.

 4) You can file and unfile the mark as you would with any other object.
 The mark will remain connected to the target.  Note that once a mark
 has been filed it is now permanent; this will likely consume resources
 at the component that has the target.
 
 5) Send msgDestroy to the mark when you are done with it.
*/

#ifndef MARK_INCLUDED
#define MARK_INCLUDED 1

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef UUID_INCLUDED
#include <uuid.h>
#endif

#ifndef GWIN_INCLUDED
#include <gwin.h>
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Statuses									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define stsMarkNoUUIDs				MakeStatus(clsMark, 1)
#define stsMarkRedirectMark			MakeStatus(clsMark, 2)
#define stsMarkNoWin				MakeStatus(clsMark, 3)
#define stsMarkNoComponent			MakeStatus(clsMark, 4)

#define stsMarkComponentsDiffer		MakeWarning(clsMark, 10)
#define	stsMarkTokensEqual			MakeWarning(clsMark, 11)
#define	stsMarkTokenAfter			MakeWarning(clsMark, 12)
#define	stsMarkTokenBefore			MakeWarning(clsMark, 13)

#define stsMarkEnterChild			MakeWarning(clsMark, 20)
#define stsMarkRetry				MakeWarning(clsMark, 21)
#define stsMarkSkipChild			MakeWarning(clsMark, 22)
#define stsMarkNotActive			MakeWarning(clsMark, 23)



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and Types							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT		MARK, * P_MARK;


typedef struct MARK_TOKEN {
	CLASS			classLevel;		// which class level is the data at
	U32				index;			// index to the data item
	U32				index2;			// secondary index if needed
} MARK_TOKEN, * P_MARK_TOKEN;


typedef struct MARK_COMPONENT {
	UUID			appUUID;
	UUID			compUUID;
	UID				compUID;
} MARK_COMPONENT, * P_MARK_COMPONENT;


/* 	MARK_FLAGS */
/*
 These flags are used when creating a mark.  They indicate what kind of
 mark is to be created.
	
    markDocRelative:	makes the mark save its component reference relative
                        to OSThisApp.  This means that if the mark is
                        saved, and both the document that contains the mark
                        and the document it refers to are copied in a
                        single operation, the new set will refer to within
                        itself correctly.  This is what goto buttons do.
						
    markForSelection:	automatically positions the mark at the selection,
                        including finding out who the selection holding
                        component is.  If you set this, then you don't need
                        to set any other fields in the new struct.
	
    markAlwaysDelete:	once a mark has been saved, it remembers that it can
                        never delete the token because it has no idea how
                        many copies of the file it was saved in have been
                        made.  This flag forces marks to always delete the
                        token no matter what.  If you manage the reference
                        of this mark and you can guarantee that what ever
                        happens to the saved mark happens to the component
                        it refers to, then set this flag.
	
    markRelaxActivate:	this keeps a mark from activating the component on
                        entering and exiting.  If the component isn't
                        active on entering, it will be skipped if possible
                        or it will be referred to in its entirety.  If the
                        component is not active on exiting, then it will
                        miss the delete token message.  Note that this can
                        cause resource leaks at the expense of keeping the
                        UI snappy.  This takes precedence over
                        markAlwaysDelete.
*/										
typedef U32 MARK_FLAGS, * P_MARK_FLAGS;

#define markDocRelative		flag1		// if saved, document relative
#define markForSelection	flag2		// make mark for the selection
#define markAlwaysDelete	flag3		// if you manage the destination
#define markRelaxActivate	flag4		// don't always activate
						 

typedef U16 MARK_MSG_FLAGS, * P_MARK_MSG_FLAGS;

/* These flags are only valid with msgMarkDeliverPos & msgMarkDeliverNext */
#define markMsgNormal	0		// standard message send
#define markMsgTry		1		// one in a sequence of possible messages
#define markMsgLastTry	2		// last in a sequence of messages
#define markMsgMode		3		// '&' with flags to extract flag field

/* These flags are only valid with msgMarkDeliverNext */
#define markBackward	flag8			// direction of movement is reversed
#define markEnterNone	0				// enter no children
#define markEnterAll	flag9			// enter all children
#define markEnterOpen	flag10			// enter only open children
#define markEnterMode	(flag9|flag10)	// '&' with flags to extract Enter 
										// field
#define markExitUp		flag11			// at end, move up to parents

// default flag settings:
#define markDefaultMsgFlags			0
#define markDefaultPosMsgFlags		markMsgNormal
#define markDefaultNextMsgFlags		\
	(markMsgNormal | markEnterOpen | markExitUp)


/*
 MARK_MSG_HEADER must be the start of the argument structure for any
 message delivered via msgMarkDeliver, msgMarkDeliverPos, or
 msgMarkDeliverNext.  It allows clsMark to insert the token information
 into the message arguments to indicate which part of the component
 is to be operated on.
*/

typedef struct MARK_MSG_HEADER {
	MARK_TOKEN		token;		// Supplied by mark: the token
	MESSAGE			msg;		// In: the message to send
	SIZEOF			lenArgs;	// In: length of the whole structure
	MARK_MSG_FLAGS	flags;		// In: flags as appropriate			
} MARK_MSG_HEADER, * P_MARK_MSG_HEADER;


typedef struct MARK_MESSAGE {
	MARK_MSG_HEADER	header;
} MARK_MESSAGE, * P_MARK_MESSAGE;


typedef U16 MARK_LOCATION;

#define markLocWhole		0
#define markLocBeginning	1
#define markLocEnd			2

/*
 * The following location codes are only valid for msgMarkPositionAtGesture
 * These may be or'd together and in with the above codes...
*/
#define markLocWholeWord	flag4	// use the whole word under the gesture
#define markLocUseSelection	flag5	// use the selection if the gesture was
									// over it

/*
 Important:  all message handlers for messages sent via msgMarkDeliver,
 msgMarkDeliverPos, or msgMarkDeliverNext, must have the following as
 its first statement.  Replace "clsYourClassHere" with the uid of your
 class.
//{	   
	MarkHandlerForClass(clsYourClassHere);
//}
*/

#define MarkHandlerForClass(cls)	\
	if (WKNValue(((P_MARK_TOKEN)pArgs)->classLevel) != WKNValue(cls)) \
		return ObjectCallAncestor(msg, self, pArgs, ctx);


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
 
/****************************************************************************
 msgNew		takes P_MARK_NEW, returns STATUS
 	category: class message
	Creates a new mark, initialized to the given component (if any).
	
	The fields you might typically set are:
		pArgs->mark.flags:	or in markForSelection to refer to the selection
							object, or in markDocRelative if you ever plan
							on saving the mark object
		pArgs->mark.component.compUID:	the object to refer to (not needed
										if you set markForSelection above)
*/

typedef struct MARK_NEW_ONLY {
	MARK_FLAGS		flags;
	MARK_COMPONENT	component;
	U16				reserved[2];
} MARK_NEW_ONLY, * P_MARK_NEW_ONLY;

#define markNewFields       \
    objectNewFields         \
    MARK_NEW_ONLY 		mark;

typedef struct MARK_NEW {
	markNewFields
} MARK_NEW, * P_MARK_NEW;


/****************************************************************************
 msgNewDefaults	takes P_MARK_NEW, returns STATUS
	category: class message
	Initializes the MARK_NEW structure to default values.

 Zeroes out pNew->mark.  Specifically, this includes:
//{					   	 						
	MakeNilUUID(pArgs->mark.component.appUUID);
	MakeNilUUID(pArgs->mark.component.compUUID);
	pArgs->mark.component.compUID = objNull;
//}
*/


/****************************************************************************
 msgMarkDeliver	takes P_MARK_MESSAGE, returns STATUS
 	Delivers a message to the target that does not move the token.

 The message in pArgs->header.msg is sent to the component after the
 mark fills in the token field.  Note that the pArgs for the sent
 message are the same as the pArgs that are passed in to msgMarkDeliver.
 Various messages that are sent to components have extra fields tacked
 on to this structure.  Therefore, all messages delivered with
 msgMarkDeliver MUST have a pArgs structure that starts with same fields
 as MARK_DELIVER.  Furthermore, the lenArgs field must be set to the
 size of the WHOLE structure.
*/
#define msgMarkDeliver			MakeMsg(clsMark, 1)


/****************************************************************************
 msgMarkDeliverPos				takes P_MARK_MESSAGE, returns STATUS
 	Delivers a message to the target that moves the token but does not
	change the component.

 This is just like msgMarkDeliver, only it is used to deliver a message
 that will potentially reposition the mark elsewhere in the component.  It
 is chiefly used with the msgMarkPosition...  messages.

 The additional flags argument is used to determine how the holder wants to
 interpret the response from the client.  Normally you use markMsgNormal,
 which automatically deals with certain client response codes that the
 holder doesn't need to be aware of.

 For example, if a holder wants to use msgMarkPositionAtEdge the code would
 be:
//{
	MARK_POSITION_EDGE		edgeArgs;

	edgeArgs.msg		= msgMarkPositionAtEdge;
	edgeArgs.lenArgs	= SizeOf(MARK_POSITION_EDGE);
	edgeArgs.flags		= markMsgNormal;
	edgeArgs.location	= markLocBeginning;
	ObjCallRet(msgMarkDeliverPos, aMark, &edgeArgs);
//}	

 However, if the holder wishes to try a different positioning message if
 the first one fails, then the holder must use the flag setting markMsgTry
 on all except the last message which uses markMsgLastTry.  Furthermore,
 these must be in a while loop and repeated if stsMarkRetry is ever
 returned.

 For example, if a holder would like to use the (hypothetical) message
 msgPositionAtVowel, and if that fails use msgPositionAtLetter, and if that
 fails try msgPositionAtCharacter; then it the code would be:
//{
	POS_VOWEL		posVowel;
	POS_LETTER		posLetter;
	POS_CHAR		posChar;
	STATUS			s;

	while (true) {
		posVowel.msg		= msgPositionAtVowel;
		posVowel.lenArgs	= SizeOf(POS_VOWEL);
		posVowel.flags		= markMsgTry;
		posVowel. ...		= ...				// other arguments
		s = ObjectCall(msgMarkDeliverPos, aMark, &posVowel);
		if (s == stsMarkRetry) continue;
		if (s != stsMsgNotUnderstood) break;	// some error occurred
																
		posLetter.msg		= msgPositionAtLetter;	   
		posLetter.lenArgs	= SizeOf(POS_LETTER);
		posLetter.flags		= markMsgTry;
		posLetter. ...		= ...				// other arguments
		s = ObjectCall(msgMarkDeliverPos, aMark, &posLetter);
		if (s == stsMarkRetry) continue;
		if (s != stsMsgNotUnderstood) break;	// some error occurred

		posChar.msg			= msgPositionAtCharacter;
		posChar.lenArgs		= SizeOf(POS_CHAR);
		posChar.flags		= markMsgLastTry;
		posChar. ...		= ...				// other arguments
		s = ObjectCall(msgMarkDeliverPos, aMark, &posChar);
		if (s == stsMarkRetry) continue;
		if (s != stsMsgNotUnderstood) break;	// some error occurred

		//do what you do if none were understood
	}
//}

 While this code is a little complicated, it allows the holder to deal with
 a variety of components that may know different messages.  The while loop
 and stsMarkRetry are necessary for the handling of inherited component
 data and behavior.  (Specifically, while the mark takes care of most of
 the chore of moving from level to level in a components class hierarchy,
 only the holder knows the sequence of messages to try at each level, so
 the stsMarkRetry acts as a sentinel to the holder to retry the full
 sequence again.)
*/
#define msgMarkDeliverPos		MakeMsg(clsMark, 2)


/****************************************************************************
 msgMarkDeliverNext				takes P_MARK_MESSAGE, returns STATUS
 	Delivers a message to the target that moves the token and sometimes
	(but not always) changes the component.

 This is the same as msgMarkDeliverPos, only it is used when the
 repositioning of the token may result moving to a new component.  This may
 happen in messages like msgSRNextChars where the next string to search is
 in an embedded component.							   

 The flags field is used the same way as in msgMarkDeliverPos.  The flags
 field also carries some additional flags:  These indicate which direction
 the movement is in, and what to do about embedded components and what to
 do at the end of a component.

 All components that respond to messages sent via msgMarkDeliverNext are
 responsible for two things:
 
 -:     They must check the markBackward flag to determine the direction
 		of motion.

 -:     If they encounter a child window as the next item, regardless of
 		what the message is looking for, then the token needs to
		be set to refer to that child and stsMarkEnterChild
		needs to be returned.
*/		
#define msgMarkDeliverNext		MakeMsg(clsMark, 3)


/****************************************************************************
 msgMarkSend			takes P_MARK_SEND, returns STATUS
	Sends a message to a component with no further processing.
	
 Sends a message to the component.  Note that this allows you to send
 any arbitrary message.  However, unlike the msgMarkDeliver messages,
 msgMarkSend doesn't copy the token value of the mark into the argument
 structure passed to the component.  Hence, no indication of what the
 target is goes with the message.  This is rarely what you want.

 The rule is: any message designed to be used with marks should use one
 of the msgMarkDeliver forms.  Any message NOT designed to work with
 marks (and thus has no specific target) should use msgMarkSend.
*/
#define msgMarkSend				MakeMsg(clsMark, 9)

typedef struct MARK_SEND {
	MESSAGE			msg;		// the message to send
	P_ARGS			pArgs;		// pointer to the arguments
	SIZEOF			lenArgs;	// length of those arguments
} MARK_SEND, * P_MARK_SEND;


/****************************************************************************
 msgMarkSetComponent	takes P_MARK_COMPONENT, returns STATUS
 	Sets the mark to refer to the given component.
	
 You set the fields of the MARK_COMPONENT one of three ways (zeroing the
 unused fields):
	-:      Set pArgs->compUID to refer to a specific component object
	-:      Set pArgs->appUUID to refer to an application object by UUID
	-:      Set pArgs->appUUID and pArgs->compUUID to refer to a
			component in an application by UUIDs

 This will delete the previous mark, if necessary and send a
 msgMarkCreateToken to the new component.

 To make the mark point at nothing, pass it a pointer to an all-zero
 structure; do NOT pass it a null pointer!

*/
#define msgMarkSetComponent		MakeMsg(clsMark, 4)


/****************************************************************************
 msgMarkGetComponent	takes P_MARK_COMPONENT, returns STATUS
 	Returns the UUID of the app the contains the token and the UUID and
	UID of the component that contain the token.
	
 If the app is not open, then pArgs->compUID will be objNull.  If the
 target is in the app object, then pArgs->compUUID will be zeros.
*/
#define msgMarkGetComponent		MakeMsg(clsMark, 5)


/****************************************************************************
 msgMarkCompareMarks	takes MARK, returns STATUS
 	Determines if two marks refer to the same component, and, if so,
	what order their targets are in.

 Return Value
	stsMarkTokensEqual:	the targets of the marks are the same
	stsMarkTokenAfter: 	the target of the receiver is after the argument
	stsMarkTokenBefore:	the target of the receiver is before the argument
*/
#define msgMarkCompareMarks		MakeMsg(clsMark, 6)


/****************************************************************************
 msgMarkCopyMark		takes P_MARK, returns STATUS
 	Creates a new mark, identical to this mark.

 Because marks can't easily reverse direction across components, it's
 sometimes desirable to save the original position.  Since the duplicate
 mark is independent of the original, it doesn't move when the original
 does.
*/
#define msgMarkCopyMark			MakeMsg(clsMark, 7)



/****************************************************************************
 msgMarkGotoMark		takes P_MARK_GOTO, returns STATUS
 	Causes a mark to be selected and displayed to the user.
	
 By default, the target is selected and scrolled on screen, provided the
 document is on screen.  Optionally, the document can be activated and
 either turned to or floated on screen.
*/
#define msgMarkGotoMark			MakeMsg(clsMark, 8)

typedef struct MARK_GOTO {
	BOOLEAN	noSelect	: 1,	//inhibits the selection of the target
			noDisplay	: 1,	//inhibits the display of the target
			turnTo		: 1,	//if closed, will do a turn to
			bringTo		: 1,	//if closed, will do a bring to
			reserved	: 12;
} MARK_GOTO, * P_MARK_GOTO;


//REFGEN BEGINIGNORE
/****************************************************************************
 msgMarkSetSaveMode	takes U32, returns STATUS
	OBSOLETE MESSAGE, DO NOT USE.
	
*/
#define msgMarkSetSaveMode			MakeMsg(clsMark, 10)

#define markSaveNormal				0
#define markSaveForOtherDoc			1

//REFGEN ENDIGNORE


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					  	Messages Sent to Components						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
 
/*
 Important:  message handlers for the first three messages
 (msgMarkCreateToken, msgMarkDeleteToken, and msgMarkCompareTokens) must
 have the following as its first statement.  Replace "clsYourClassHere"
 with the uid of your class.
//{	   
	MarkHandlerForClass(clsYourClassHere);
//}
*/

/****************************************************************************
 msgMarkCreateToken takes P_MARK_TOKEN, returns STATUS
 	Instructs a component to create a token for its data items, and start
	the token pointing at before all data items.

 You can only forget about the token associated with a mark when a 
 corresponding msgMarkDeleteToken is received, or the target data is 
 deleted.  In the later case you must be careful never to generate that 
 token again as there still might be outstanding tokens for it.
*/
#define msgMarkCreateToken			MakeMsg(clsMark, 40)


/****************************************************************************
 msgMarkDeleteToken takes P_MARK_TOKEN, returns STATUS
 	Tells a component that the given token will no longer be in use.

 See msgMarkCreateToken.
*/
#define msgMarkDeleteToken			MakeMsg(clsMark, 41)


/****************************************************************************
 msgMarkCompareTokens			takes P_MARK_COMPARE_TOKENS, returns STATUS
 	Asks a component to compare the ordering of two tokens.

 Return Value
	stsMarkTokensEqual:		the two tokens point to the same place
	stsMarkTokenAfter:		the first token comes after the second
	stsMarkTokenBefore:		the first token comes before the second
*/
#define msgMarkCompareTokens		MakeMsg(clsMark, 42)

typedef struct MARK_COMPARE_TOKENS {
	MARK_TOKEN		firstToken;
	MARK_TOKEN		secondToken;
} MARK_COMPARE_TOKENS, * P_MARK_COMPARE_TOKENS;


/****************************************************************************
 msgMarkGetDataAncestor				takes P_CLASS, returns STATUS
 	Asks for the next higher superclass that contains traversable data.

 Asks a component what the next ancestor the argument inherits data from. 
 The component's response should is based on what the argument is. Assuming
 the class of the component is clsMyThing:

	objNull:	respond with clsMyThing
	clsMyThing:	respond with the next class clsMyThing gets data from,
				typically clsEmbeddedWin or objNull (if none).
	otherwise:	call the ancestor

 Example
//{		
	if		(*pArgs == objNull)		*pArgs = clsMyThing;
	else if (*pArgs == clsMyThing)	*pArgs = clsEmbeddedWin;
	else							ObjCallAncestorCtxRet(ctx);
	return stsOK;
//}

 If your code doesn't inherit data then you'll do the following:										  
//{													 
	if		(*pArgs == objNull)		*pArgs = clsMyThing;
	else							*pArgs = objNull;
	return stsOK;
//}
*/
#define msgMarkGetDataAncestor			MakeMsg(clsMark, 46)


/****************************************************************************
 msgMarkGetParent				takes P_MARK_COMPONENT, returns STATUS
 	Asks a component to set the argument to its parent (embedding)
	component.
		
 Either the UID or the UUIDs should be filled in, mark will take care of
 the rest.  If the component is descended from clsEmbeddedWin or clsApp, it
 already inherits the correct response and implementation is necessary.
*/										   
#define msgMarkGetParent			MakeMsg(clsMark, 43)


/****************************************************************************
 msgMarkGetUUIDs				takes P_MARK_COMPONENT, returns STATUS
 	Asks a component to set the argument to its own app and component
	UUIDs if it can.
	
 If it can't it should return stsMarkNoUUIDs.  If your component is a
 descendant of clsApp or clsEmbeddedWin then you inherit the correct
 implementation.								 	 
*/
#define msgMarkGetUUIDs				MakeMsg(clsMark, 45)



/****************************************************************************
 msgMarkValidateComponent			takes P_MARK_COMPONENT, returns STATUS
	Asks a component to verify that it is okay to traverse it.

 This message is sent to objects before a mark refers to them.  This gives
 an object a chance to point the mark at a different object as the 
 component.  Typically, a driver might create a mark with the selection
 holder as the component.  However, the selection holder might not be the
 desired component for a mark (the selection could be a data object, but
 the mark component should be the app object).  Mark sends this message to
 the proposed component.  The proposed component then can either not
 implement the message (or return stsOK) or set the argument to another
 component object and return stsMarkRedirectMark.  In the first case the
 proposed component becomes the used component, in the second the returned
 component becomes the new proposed component.
*/
#define msgMarkValidateComponent	MakeMsg(clsMark, 44)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *			  	Messages Sent to Components	via msgMarkDeliver			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Note: As these are defined in clsMark, these messages may be sent to the
 mark directly without using msgMarkDeliver, msgMarkDeliverPos, or
 msgMarkDeliverNext (with mode = markMsgNormal) as appropriate.  As usual,
 the mark and token fields will be filled in by the mark, and then the
 message passed on.

 In addition special processing will be done for some of the  messages
 which would NOT be done if the message was sent via standard delivery
 messages.  This processing is noted for those messages under the heading:
 'If sent directly to mark'
*/

/*
 Important:  all message handlers for these messages must have the
 following as its first statement.  Replace "clsYourClassHere" with the
 uid of your class.
//{	   
	MarkHandlerForClass(clsYourClassHere);
//}
*/

/****************************************************************************
 msgMarkPositionAtEdge	takes P_MARK_POSITION_EDGE, returns STATUS
 	Asks a component to reposition the token to one end or the other of the
	data.
*/
#define msgMarkPositionAtEdge	MakeMsg(clsMark, 80)

typedef struct MARK_POSITION_EDGE {
	MARK_MSG_HEADER	header;
	MARK_LOCATION	location;		// either markLocBeginning or markLocEnd
} MARK_POSITION_EDGE, * P_MARK_POSITION_EDGE;


/****************************************************************************
 msgMarkPositionAtToken		takes P_MARK_POSITION_TOKEN, returns STATUS
 	Asks a component to reposition the token to the same position as
	another token for the same component.

 If sent directly to mark: you only need to fill in the otherMark field,
 the mark will take care of the rest & will check to see that both marks
 point at the same component.  Since you'd have no idea what the other
 Token is, this is the only sensible way to send this message (via
 msgMarkDeliver won't work).
*/
#define msgMarkPositionAtToken	MakeMsg(clsMark, 81)

typedef struct MARK_POSITION_TOKEN {
	MARK_MSG_HEADER	header;
	MARK			otherMark;		// In; the other mark
	MARK_TOKEN		otherToken;		// In: the token to copy
} MARK_POSITION_TOKEN, * P_MARK_POSITION_TOKEN;


/****************************************************************************
 msgMarkPositionAtChild		takes P_MARK_POSITION_CHILD, returns STATUS
 	Asks a component to reposition the token to the given child component
	which is given as a UUID/UID pair.
	
 The UID may be null if it is unknown, but the UUID will always be valid.
*/
#define msgMarkPositionAtChild	MakeMsg(clsMark, 82)

typedef struct MARK_POSITION_CHILD {
	MARK_MSG_HEADER	header;
	MARK_COMPONENT	child;		// In: the child to position to;
} MARK_POSITION_CHILD, * P_MARK_POSITION_CHILD;


/****************************************************************************
 msgMarkPositionAtGesture	takes P_MARK_POSITION_GESTURE, returns STATUS
 	Asks a component to reposition the token at the given gesture.
	
 The location parameter indicates how to position relative to the gesture. 
 Note that there are a variety of location codes that might be or'd
 together.
*/

#define msgMarkPositionAtGesture	MakeMsg(clsMark, 83)

typedef struct MARK_POSITION_GESTURE {
	MARK_MSG_HEADER	header;
	GWIN_GESTURE	gesture;
	MARK_LOCATION	location;
} MARK_POSITION_GESTURE, * P_MARK_POSITION_GESTURE;


/****************************************************************************
 msgMarkPositionAtSelection		takes P_MARK_POSITION_SELECTION, returns STATUS
 	Asks a component to reposition the token to the selection, which
	it presumably owns.
	
 If the component doesn't own the selection, then return stsFailed.
*/
#define msgMarkPositionAtSelection	MakeMsg(clsMark, 85)

typedef struct MARK_POSITION_SELECTION {
	MARK_MSG_HEADER	header;
	MARK_LOCATION	location;
} MARK_POSITION_SELECTION, * P_MARK_POSITION_SELECTION;


/****************************************************************************
 msgMarkNextChild	takes P_MARK_MESSAGE, returns STATUS
 	Requests the component to move the token to the next child.
	
 If a child is found and the token moved to it, return stsMarkEnterChild,
 not stsOK.  If return, the mark is likely (but may not) send
 msgMarkGetChild to find out who the child actually is.
*/
#define msgMarkNextChild		MakeMsg(clsMark, 86)



/****************************************************************************
 msgMarkGetChild				takes P_MARK_GET_CHILD, returns STATUS
 	Requests the component to fill in the component at the current token.

 This is sent because, presumable, the response to some other move
 message was stsMarkEnterChild.  If the token doesn't point at a child,
 return stsFailed.

 pArgs->childIsDoc should set true if the child is an embedded document.
 If the child is just an embedded component that is to be considered
 part of the receiving component, then set this field false.  This field
 is used by clsMark to determine if it should apply the markEnterMode
 bits that control entering embedded documents (they don't control
 entering embedded components, this is always done.)

 If pArgs->childIsDoc is set true, then childIsOpen must be set to
 reflect the "open" status of the embedded doc.

 If your component is managing its own embedees, typically your
 component will only deal with the embedded instances of clsAppWin.
 These are components that are part of your component: you should set
 pArgs->childIsDoc to false (pArgs->childIsOpen doesn't matter in this
 case).  When the appWin is entered, it will handle the proper reporting
 of the embedded document.  (clsAppWin sets pArgs->childIsDoc to true
 and pArgs->childIsOpen appropriately.)
*/
#define msgMarkGetChild			MakeMsg(clsMark, 90)

typedef struct MARK_GET_CHILD {
	MARK_MSG_HEADER	header;
	MARK_COMPONENT	child;			//Out: fill in uid or uuids
	BOOLEAN			childIsDoc;		//Out: is the child is a document?
	BOOLEAN			childIsOpen;	//Out: is the child open?
} MARK_GET_CHILD, * P_MARK_GET_CHILD;





/****************************************************************************
 msgMarkSelectTarget	takes P_MARK_MESSAGE, returns STATUS
 	Requests the component to select the target data item.
	
*/
#define msgMarkSelectTarget		MakeMsg(clsMark, 89)


/****************************************************************************
 msgMarkShowTarget	takes P_MARK_SHOW_TARGET, returns STATUS
 	Request the component to return the window that contains the graphical
	view of the target.

 The rectangle returned is the area within the window that encloses the
 target.
	
 Some components may not have a viewable representations of the target,
 in which case they can return stsMarkNoWin, or simply not implement
 this message.  Other components may have a graphical view only part of
 the time.  In this case, it should ensure that the target has a
 graphical representation, otherwise return stsMarkNoWin if the target
 isn't right now.


 Note that this message requests that the target be scrolled into view.
 That should be done by sending msgEmbeddedWinShowChild to the win showing
 the target (usually the win that is returned in pArgs->win).
*/						
#define msgMarkShowTarget			MakeMsg(clsMark, 88)

typedef struct MARK_SHOW_TARGET {
	MARK_MSG_HEADER	header;
	WIN				win;
	RECT32			rect;
} MARK_SHOW_TARGET, * P_MARK_SHOW_TARGET;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *			  				Messages Sent Internally					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgMarkEnterChild				takes P_MARK_MESSAGE, returns STATUS
 	Sent when a component requests the mark to enter a child (usually via
	returning stsMarkEnterChild to a message send with msgMarkDeliverNext).
	
 This message sends msgMarkGetChild to the component to get the child at
 the token and then enters the child if appropriate.
*/
#define msgMarkEnterChild		MakeMsg(clsMark, 120)


/****************************************************************************
 msgMarkEnterLevel				takes P_MARK_MESSAGE, returns STATUS
 	Sent when a component requests the mark to bump up a level in its
	class chain, or when a position or next message fails and the mark
	tries the next class level.
	
 This message sends msgMarkGetDataAncestor to the component and resets the
 token.							  							
*/
#define msgMarkEnterLevel		MakeMsg(clsMark, 121)


/****************************************************************************
 msgMarkEnterParent				takes P_MARK_MESSAGE, returns STATUS
 	Sent when a component runs out of data altogether and the mark
	needs to move on (and up).
	
 This message may send msgMarkGetParent to the component to find out who
 the parent is.
*/
#define msgMarkEnterParent		MakeMsg(clsMark, 122)


/****************************************************************************
 msgMarkGetToken				takes P_MARK_TOKEN, returns STATUS
 	Sent from one mark to another to get the other's token.
	
 This is not intended to be used by clients of mark.
*/
#define msgMarkGetToken			MakeMsg(clsMark, 123)


#endif


