/****************************************************************************
 File: npdata.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.9  $
   $Author:   bschwart  $
     $Date:   28 Feb 1992 09:32:42  $

 This file contains the API definition for clsNPData.

 clsNPData inherits from clsObject.

 NPData is the data class of PenPoint's ink-management or note-taking
 building block.  (See notepapr.h for more information on the building block.)
 An NPData instance is a data base that manages items that follow the
 clsNPItem protocol.  (See npitem.h).  Its API defines messages for
 inserting, deleting, and enumerating the items it manages.
 
****************************************************************************/
#ifndef NPDATA_INCLUDED
#define NPDATA_INCLUDED

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif
#include <geo.h>

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Types and Constants                                                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define clsNPData			MakeGlobalWKN(2568,1)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Messages		                                                       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Next up: 39;  Recycle: 4 5 6 7 15 20 33 34 */

/****************************************************************************
 msgNewDefaults				takes P_NP_DATA_NEW, returns STATUS
	Initialize pArgs.

 Zeroes out pArgs->npData and sets:
//{
	pArgs->npData.lineSpacing.x = 0;
	pArgs->npData.lineSpacing.y = 360;		// 360 twips = 18 points = 1/4"

	pArgs->npData.baseline.x = 0;
	pArgs->npData.baseline.y = 360;
//}
*/
typedef struct {

	XY32	lineSpacing;
	XY32	baseline;
	BOOLEAN isSubData;	// private to clsNPData
	S32		spare1;
	S32		spare2;

} NP_DATA_NEW_ONLY,  *P_NP_DATA_NEW_ONLY;

#define	npDataNewFields		\
	objectNewFields			\
	NP_DATA_NEW_ONLY	npData;

typedef struct {

	npDataNewFields

} NP_DATA_NEW, *P_NP_DATA_NEW;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages used to manipulate data                       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNPDataInsertItem			takes OBJECT, returns STATUS
	Add item to the data base.
*/
#define msgNPDataInsertItem MakeMsg(clsNPData, 8)

/****************************************************************************
 msgNPDataInsertItemFromView		takes P_NP_DATA_ADDED_NP_ITEM_VIEW, returns STATUS
	Add item to the data base.

 Observers will be notified of which view is responsible for the addition.
*/
#define msgNPDataInsertItemFromView MakeMsg(clsNPData, 38)

typedef struct {

	OBJECT	item;	// item that has been added
	OBJECT	view;	// view that added the item

} NP_DATA_ADDED_NP_ITEM_VIEW, *P_NP_DATA_ADDED_NP_ITEM_VIEW;

/****************************************************************************
 msgNPDataDeleteItem			takes OBJECT, returns STATUS
	Delete an item from the data base.
							
 Returns stsFailed if item is not found.
*/
#define msgNPDataDeleteItem MakeMsg(clsNPData, 9)

/****************************************************************************
 msgNPDataMoveItem			takes P_NP_DATA_XY, returns STATUS
	Move an item within the data base.
*/
#define msgNPDataMoveItem MakeMsg(clsNPData, 10)

typedef struct {

	OBJECT	item;	// item to be moved
	XY32	xy;		// new position for item

} NP_DATA_XY, *P_NP_DATA_XY;

/****************************************************************************
 msgNPDataMoveItems			takes P_MOVE_ITEMS, returns STATUS
	Move all items below pArgs->y by pArgs->yDelta.
*/
#define msgNPDataMoveItems		MakeMsg(clsNPData, 1)

typedef struct {

	COORD32	y;
	COORD32	yDelta;

} MOVE_ITEMS, *P_MOVE_ITEMS;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages used to enumerate over data                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 ENUM_CALLBACK					returns STATUS
	This template describes the the callback function used in item enumeration.

 Your callback function takes a single parameter of type P_NP_DATA_ITEM.
 The clientData field is a copy of that you passed into the enumeration
 message using the ENUM_ITEM or ENUM_RECT_ITEM structures.  During enumeration,
 you can add new items or delete the "current" item begin enumerated.  If you
 delete an item but want to keep using it, use must send it msgNPItemHold
 before deleting it and msgNPItemRelease when you are done using it.
																		 	
 Some of the enumeration messages refer to bPaintOrder or "Reverse" order.
 Paint order refers to the top-to-bottom, left-to-right ordering of items.
 Non-paint or reverse order is simply the opposite ordering.  Items are
 sorted first by line and then by their left edge.  An item is considered
 to be on the line closest to its baseline.  The lines are "line spacing"
 apart starting from the top of the page.  If no lines are displayed
 to the user, it is possible that non-intuitive item ordering will result.
 										   	  
 Return an error status from the callback to terminate the enumeration.
*/
typedef struct {

	OBJECT		data;			// in - the data being enumerated over
	OBJECT		item;			// in - the item being enumerated
	P_UNKNOWN	clientData;		// in - the client supplied data (or pointer)

} NP_DATA_ITEM, *P_NP_DATA_ITEM;
typedef STATUS FunctionPtr(P_ENUM_CALLBACK)(P_NP_DATA_ITEM pItem);

/****************************************************************************
 msgNPDataEnumOverlappedItems		takes P_ENUM_RECT_ITEMS, returns STATUS
	Enumerates each item that overlaps the given rectangle.
*/
#define msgNPDataEnumOverlappedItems MakeMsg(clsNPData, 2)

typedef struct {

    P_ENUM_CALLBACK	function;		// in -- callback function described above
	RECT32			hitRect;		// in -- enum items overlapping hitRect
	BOOLEAN			bPaintOrder;	// in -- enum in paint order?
	P_UNKNOWN		clientData;		// in

} ENUM_RECT_ITEMS, *P_ENUM_RECT_ITEMS;

/****************************************************************************
 msgNPDataEnumBaselineItems		takes P_ENUM_RECT_ITEMS, returns STATUS
	Enumerates each item whose baseline overlaps the given rectangle.
*/
#define msgNPDataEnumBaselineItems MakeMsg(clsNPData, 19)

/****************************************************************************
 msgNPDataEnumSelectedItems		takes P_ENUM_ITEMS, returns STATUS
	Enumerates each item that is selected (in paint order).
*/
#define msgNPDataEnumSelectedItems MakeMsg(clsNPData, 13)

typedef struct {

    P_ENUM_CALLBACK	function;		// in -- callback function described above
	P_UNKNOWN		clientData;		// in

} ENUM_ITEMS, *P_ENUM_ITEMS;


/****************************************************************************
 msgNPDataEnumSelectedItemsReverse		takes P_ENUM_ITEMS, returns STATUS
	Enumerates each item that is selected (in reverse paint order).
*/
#define msgNPDataEnumSelectedItemsReverse MakeMsg(clsNPData, 26)

/****************************************************************************
 msgNPDataEnumAllItems		takes P_ENUM_ITEMS, returns STATUS
	Enumerates each item (in paint order).
*/
#define msgNPDataEnumAllItems MakeMsg(clsNPData, 14)

/****************************************************************************
 msgNPDataEnumAllItemsReverse		takes P_ENUM_ITEMS, returns STATUS
	Enumerates each item (in reverse paint order).
*/
#define msgNPDataEnumAllItemsReverse MakeMsg(clsNPData, 27)

/****************************************************************************
 msgNPDataSendEnumSelectedItems		takes P_SEND_ENUM_ITEMS, returns STATUS
	Enumerates each selected item (in paint order).

 This message is the same as msgNPDataEnumSelectedItems, except that it
 it intended to be used in conjunction with ObjectSend rather than
 ObjectCall.  It is used to enumerate the items in a data object that is not
 in the caller's process.  Rather than a pointer to the client data being
 passed around, the client data is put into an array that is passed around.
*/
#define msgNPDataSendEnumSelectedItems MakeMsg(clsNPData, 22)
typedef struct {

    P_ENUM_CALLBACK	function;		// in -- callback function described above
	U8				clientData[32];	// in/out

} SEND_ENUM_ITEMS, *P_SEND_ENUM_ITEMS;

/****************************************************************************
 msgNPDataGetCurrentItem	takes P_OBJECT, returns STATUS
	Passes back the current item in the receiver.
*/
#define msgNPDataGetCurrentItem		MakeMsg(clsNPData, 30)

/****************************************************************************
 msgNPDataGetNextItem	takes P_OBJECT, returns STATUS
	Increments the current item to the next item and sets *pArgs to it.

 Set *pArgs to the current item before sending this message.  If you
 set it to NULL, the first item will be returned.  The next time you call
 this message after you reach the last item, stsEndOfData will be returned
 and *pArgs will be set to objNull.
*/
#define msgNPDataGetNextItem		MakeMsg(clsNPData, 31)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages used to access internal state                 *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNPDataItemCount			takes P_U32, returns STATUS
	Passes back the count of items in receiver.
*/
#define msgNPDataItemCount	MakeMsg(clsNPData, 17)

/****************************************************************************
 msgNPDataSelectedCount	takes P_U32, returns STATUS
	Passes back the count of selected items in receiver.
*/
#define msgNPDataSelectedCount	MakeMsg(clsNPData, 18)

/****************************************************************************
 msgNPDataSetBaseline	takes P_XY32, returns STATUS
	Sets the receiver's baseline (used for alignment).
*/
#define msgNPDataSetBaseline		MakeMsg(clsNPData, 24)

/****************************************************************************
 msgNPDataGetBaseline	takes P_XY32, returns STATUS
	Gets the receiver's baseline (used for alignment).
*/
#define msgNPDataGetBaseline		MakeMsg(clsNPData, 25)

/****************************************************************************
 msgNPDataSetLineSpacing	takes P_XY32, returns STATUS
	Sets receiver's line spacing (used as the font size).
*/
#define msgNPDataSetLineSpacing		MakeMsg(clsNPData, 35)

/****************************************************************************
 msgNPDataGetLineSpacing	takes P_XY32, returns STATUS
	Gets receiver's line spacing (used as the font size).
*/
#define msgNPDataGetLineSpacing		MakeMsg(clsNPData, 36)

/****************************************************************************
 msgNPDataGetBounds	takes P_RECT32, returns STATUS
	Passes back the bounding rectangle for all items in receiver.
*/
#define msgNPDataGetBounds		MakeMsg(clsNPData, 23)

/****************************************************************************
 msgNPDataGetSelBounds	takes P_RECT32, returns STATUS
	Passes back the bounding rectangle for all selected items in receiver.
*/
#define msgNPDataGetSelBounds		MakeMsg(clsNPData, 32)

/****************************************************************************
 msgNPDataGetFontSpec	takes P_SYSDC_FONT_SPEC, returns STATUS
	Passes back the receiver's font specification.
*/
#define msgNPDataGetFontSpec		MakeMsg(clsNPData, 28)

/****************************************************************************
 msgNPDataSetFontSpec	takes P_SYSDC_FONT_SPEC, returns STATUS
	Sets the receiver's font specification.
*/
#define msgNPDataSetFontSpec		MakeMsg(clsNPData, 29)

/****************************************************************************
 msgNPDataGetCachedDCs	takes P_NP_DATA_DC, returns STATUS
	Passes back DC's with normal and bold fonts at the given line spacing.

 Used by items that want to measure text without the overhead of creating
 a DC.  These DC's cannot be used for drawing!!
*/
#define msgNPDataGetCachedDCs		MakeMsg(clsNPData, 37)

typedef struct {

	OBJECT		dcNormal;	// normal font dc
	OBJECT		dcBold;		// bold font dc

} NP_DATA_DCS, *P_NP_DATA_DCS;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages sent to observers                             *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNPDataAddedItem				takes P_NP_DATA_ADDED_ITEM, returns STATUS
	category: observer notification
	Observers notified when item has been has been added or moved.
*/
#define msgNPDataAddedItem MakeMsg(clsNPData, 11)

typedef struct {

	OBJECT	data;	// the data that the item has been added to
	OBJECT	item;	// item that has been added
	OBJECT	view;	// view that added the item

} NP_DATA_ADDED_ITEM, *P_NP_DATA_ADDED_ITEM;

/****************************************************************************
 msgNPDataItemChanged			takes P_NP_DATA_ITEM_CHANGED, returns STATUS
	category: observer notification
	Observers notified when item has been changed.

 Currently called when item is selected or deselected.
*/
#define msgNPDataItemChanged MakeMsg(clsNPData, 12)
typedef struct {

	OBJECT	data;	// the data
	OBJECT	item;	// item that has been changed
	OBJECT	view;	// view that changed the item
	RECT32	bounds;	// maximum bounds affected by the change

} NP_DATA_ITEM_CHANGED, *P_NP_DATA_ITEM_CHANGED;

/****************************************************************************
 msgNPDataHeightChanged			takes P_NP_DATA_ITEM_CHANGED, returns STATUS
	category: observer notification
	Observers notified when receiver's height has been changed.

 Currently called by msgNPDataMoveItems.  The bounds.origin.y field of pArgs
 contains the delta in the height of the data object.
*/
#define msgNPDataHeightChanged MakeMsg(clsNPData, 21)

/****************************************************************************
 msgNPDataItemEnumDone		takes NULL, returns STATUS
	category: observer notification
	Observers notified when an enumeration that deleted or moved items is complete.

 When this message is received by an observer client, all deletions have been
 completed and all moved items have been temporarily removed from the data
 object.  Thus the client has the option of repainting all remaining items
 at this time and then painting moved items as they are reinserted.

 This message is handled by clsNotePaper and should not be handled by
 subclasses of clsNotePaper.
*/
#define msgNPDataItemEnumDone	MakeMsg(clsNPData, 16)

#endif
