/****************************************************************************
 scribble.h

 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.5  $
   $Author:   kbentley  $
     $Date:   11 Mar 1992 19:28:22  $

 This file contains the API definition for clsScribble.

 clsScribble inherits from clsObject.

 Instances of clsScribble (or scribbles for short) store pen strokes.  
 Scribbles also interact with translators.
****************************************************************************/

/**** Introduction ****/
/*
 An scribble is a collection of pen strokes.  Clients can add strokes to (and
 remove strokes from) a scribble.  Clients can use msgScrRender to render
 a scribble in a given drawing context. 

 A client typically adds strokes to a scribble during the client's response
 to msgPenStroke type input events.

 clsScribble is a relatively low-level piece of PenPoint.  Many clients can
 and should use clsGWin (gwin.h) or clsSPaper (spaper.h) rather than
 clsScribble.
*/


/**** Coordinates and the Base ****/
/*
 A scribble's coordinates are in Pen Units. (See msgDcUnitsPen in
 sysgraf.h.)

 Each scribble has a "base."  By default, a scribble's base is the origin
 of the first stroke added to the scribble (via msgScrAddStroke).
 Whenever a stroke is added to a scribble, the scribble's base is
 subtracted from the origin of the stroke before the stroke is made part
 of the scribble.  In other words, all strokes are stored relative to
 the scribble's base.  This allows repositioning the entire scribble
 by adjusting the base.  For instance, the client might do this in
 response to a window resize operation to keep the scribble in the
 "same" position relative to the upper left corner of a window.

 This base is not transparent when extracting a stroke from a scribble.
 When using msgScrStrokePtr to get a pointer to a stroke in a scribble,
 it is necessary to add the scribble base back to the stroke origin in
 order to get the original stroke origin.

*/


/**** Adding Strokes to Scribbles ****/
/*
 This example shows how strokes are added to a scribble by a window that is
 handling msgInputEvent when pArgs->devCode is msgPenStroke.  Note that 
 pArgs->base is set to the origin of the scribble. 
//{ 
	// msgPenStroke's stroke data arrives in root window-relative device
	// units. Convert the stroke data to be self-relative.  Steps:
	// (1) Compute the origin of self relative to the root window,
	// (2) Convert that origin to pen units.
	wm.parent = theRootWindow;
	wm.child = NULL;
	wm.bounds.origin.x = 0;
	wm.bounds.origin.y = 0;
	wm.bounds.size.h = 0;
	wm.bounds.size.w = 0;
	ObjectCall(msgWinTransformBounds, self, &wm);
	ConvertOriginToPenCoordinates(&wm.bounds.origin);

	// Now add the scribble to the stroke.  Note that the scrAdd.base is
	// the base (i.e., origin) of the STROKE, not the scribble.
	scrAdd.base.x = pStroke->bounds.origin.x - wm.bounds.origin.x;
	scrAdd.base.y = pStroke->bounds.origin.y - wm.bounds.origin.y;
	ObjectCall(msgScrAddStroke, scribble, &scrAdd);
//}

 This code gives a rough idea of how a scribble adds a stroke in response
 to msgScrAddStroke.  This is provided so that you can see how the base is
 used.  Basically, the base of the scribble is subtracted from pArgs->base
 and used as the origin of the stroke.  
//{
	// Make a local copy of the stroke.  Then convert the stroke's origin 
	// to be relative to the scribble's base.
	pNewStroke = <Copy of pArgs->pStroke>;
	pNewStroke->bounds.origin.x = pArgs->base.x - scribble.base.x;
	pNewStroke->bounds.origin.y = pArgs->base.y - scribble.base.y;
	<add stroke to internal data structures>
//}
*/


/**** Repositioning Scribbles ****/
/*
 To reposition a scribble, (1) compute the delta by which you want to move
 the scribble, (2) get the scribble's current base using msgScrGetBase, (3)
 add the delta to the current base, and (4) set the base using
 msgScrSetBase.  Be sure to use msgScrSetBase since it will readjust the
 bounds of the scribble.
*/


/**** Debugging Flags ****/
/*
 clsScribble uses the debugging flag set 'h'.  Defined flags are:
	0100:	General scribble debugging information
	0800:	Scribble save and restore debugging information
*/


/**** Limitations ****/
/*
 Strokes in a scribble must be within ((2^15)-1) units of each other.

 Memory for deleted strokes is only recovered upon msgScrClear.  No other
 messages recover deleted stroke memory.  Deleted strokes are saved and
 restored.
*/


#ifndef SCRIBBLE_INCLUDED
#define SCRIBBLE_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif

#ifndef GEO_INCLUDED
#include <geo.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef PEN_INCLUDED
#include <pen.h>
#endif

#ifndef DEBUG_INCLUDED
#include <debug.h>
#endif

// Next Up:	18	Recycled: 2, 7, 8, 


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT SCRIBBLE;

/*
 * stsScrOutOfRange is returned from msgScrAddStroke if the coordinates for
 * the base of the scribble are out of the allowable range for strokes.
*/
#define stsScrOutOfRange		MakeStatus(clsScribble,1)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Messages Defined by Other Classes				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew			takes P_SCR_NEW, returns STATUS
	category: class message
	Creates and initializes a new scribble. 
*/


/****************************************************************************
 msgNewDefaults					takes P_SCR_NEW, returns STATUS
	Sets the default values for the new arguments.

 Zeros out pNew->scribble.
*/

typedef struct SCR_NEW_ONLY {
    XY32 base;                  // initial base value, default (0,0)
	U32  reserved;				// reserved for future use
} SCR_NEW_ONLY;

#define scribbleNewFields       \
	OBJECT_NEW_ONLY	object;     \
	SCR_NEW_ONLY    scribble;
    
typedef struct SCR_NEW {
    scribbleNewFields
} SCR_NEW, *P_SCR_NEW;


/****************************************************************************
 msgFree	takes P_OBJ_KEY, returns STATUS
	Defined in clsmgr.h

 The scribble frees all of its strokes.
*/


/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Defined in clsmgr.h.

 Saves all strokes to pArgs->file.
*/


/****************************************************************************
 msgRestore		takes P_OBJ_RESTORE, returns STATUS
	Defined in clsmgr.h.

 Restores all strokes from pArgs->file.
*/


/****************************************************************************
 msgPicSegPaintObject		 		takes P_PIC_SEG_OBJECT, returns STATUS
	Defined in picseg.h.

 In response to this message, a scribble paints itself.  Any object which
 responds to msgPicSegPaintObject can be placed into a PicSeg (and instance
 of clsPicSeg).
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgScrSetBase					takes P_XY32, returns STATUS
	Sets the scribble's base.

 Recomputes the bounds of the scribble to reflect the new base.

 See the section "Coordinates and the Base" for more information.
*/
#define msgScrSetBase			MakeMsg(clsScribble,11)


/****************************************************************************
 msgScrGetBase					takes P_XY32, returns STATUS
	Passes back the base for the scribble.

 See the section "Coordinates and the Base" for more information.
*/
#define msgScrGetBase			MakeMsg(clsScribble, 10)


/****************************************************************************
 msgScrGetBounds				takes P_RECT32, returns STATUS
	Passes back the bounds of the scribble.

 Passes back the bounding box that contains all the strokes in the
 scribble.  The bounding box is in pen units.
*/
#define msgScrGetBounds			MakeMsg(clsScribble,12)


/****************************************************************************
 msgScrCount					takes P_U16, returns STATUS
	Passes back the number of strokes in the scribble.
*/
#define msgScrCount				MakeMsg(clsScribble,1)


/****************************************************************************
 msgScrAddStroke				takes P_SCR_ADD_STROKE, returns STATUS
	Adds a stroke to the scribble.

 In response to this message, the scribble makes a copy of the stroke data
 and adds the stroke to itself.  Observers are notified with
 msgScrAddedStroke.  Note the SCR_ADD_STROKE base is the base (i.e.,
 origin) of the STROKE.

 If this is the first stroke to be added to the scribble, the scribble's
 base is set to pArgs->base.  Otherwise the base of the stroke is shifted
 by the scribble base as follows:
//{
	stroke.bounds.origin = pArgs->base - scribble.base;
//}
 
 Return Value
	stsScrOutOfRange: 	The computed base for the stroke was out of
						the allowable range.
*/
#define msgScrAddStroke			MakeMsg(clsScribble,3)

typedef struct SCR_ADD_STROKE {
	XY32			base;			// In: origin of the stroke.
	P_PEN_STROKE	pStroke;		// In: pointer to stroke data
	U16				index;		  	// Out: index of the newly added stroke
} SCR_ADD_STROKE, * P_SCR_ADD_STROKE;


/****************************************************************************
 msgScrDeleteStroke				takes U16, returns STATUS
	Deletes the stroke from the scribble.

 In response to this message, the scribble marks as deleted the stroke with
 the index value of pArgs.  Observers receive msgScrRemovedStroke.

 Note: this does not actually free any memory, the scribble is just marked
 as deleted.
 
 See Also
	msgScrRemovedStroke
*/
#define msgScrDeleteStroke		MakeMsg(clsScribble,4)


/****************************************************************************
 msgScrDeleteStrokeArea		takes P_SCR_DELETE_STROKE_AREA, returns STATUS
	Deletes all of the strokes in the scribble which intersect pArgs->rect.

 The scribble uses msgScrHit and msgScrDeleteStroke to do the deletions.
 Observers receive one msgScrRemovedStroke for each intersecting stroke.
 
 If pArgs->window is non-null, the scribble dirties the appropriate
 rectangle in the window.

 See Also
	msgScrHit
	msgScrDeleteStroke
	msgScrRemovedStroke
*/
#define msgScrDeleteStrokeArea	MakeMsg(clsScribble, 2)

typedef struct SCR_DELETE_STROKE_AREA {
	RECT32		rect;		// Rectangle of area to delete strokes from.
							// In pen units.
	OBJECT		window;		// Window to dirty
	U32			spare;		// Reserved.
} SCR_DELETE_STROKE_AREA, *P_SCR_DELETE_STROKE_AREA;


/****************************************************************************
 msgScrCat						takes SCRIBBLE, returns STATUS
	Adds (concatenates) the strokes from another scribble to self.

 The receiving scribble makes copies of all of the strokes in the pArgs
 scribble and adds them to itself by self sending msgScrAddStroke.

 See Also
	msgScrAddStroke
*/
#define msgScrCat				MakeMsg(clsScribble,6)


/****************************************************************************
 msgScrComplete					takes void, returns STATUS
	Sent to a scribble to indicate completion.

 Clients send this message to a scribble to tell the scribble that it is
 "complete."  The client is responsible for determining when a scribble is 
 complete.  (For instance, the client might decide that a scribble is
 complete when the client receives an out-of-proximity pen event, or when a
 certain amount of time has elapsed since the last input event.)

 A scribble does nothing in response to this message except to send
 msgScrCompleted to all observers.  

 A translator is a typical observer of a scribble.  See xlate.h for
 information about how a translator responds to msgScrCompleted.

 See Also
	msgScrCompleted
*/
#define msgScrComplete	 		MakeMsg(clsScribble, 5)


/****************************************************************************
 msgScrStrokePtr				takes P_SCR_STROKE_PTR, returns STATUS
	Passes back the pointer to the stroke identified by pArgs->index.

 Be Careful!  pArgs->pStroke is a pointer to internal data, not a copy.

 Strokes retrieved from scribbles are relative to the scribble's base.  The
 stroke's origin is NOT the same as was passed to msgScrAddStroke -- you
 need to add the scribble's base back.  Note that this may still not be
 the same as the original stroke origin if the scribble base has been
 adjusted.
*/
#define msgScrStrokePtr			MakeMsg(clsScribble,9)

typedef struct SCR_STROKE_PTR {
	U16				index;		// In: index to the stroke
	P_PEN_STROKE	pStroke;	// Out: pointer to the index'th stroke, or
								// pNull if no such stroke exists.
} SCR_STROKE_PTR, *P_SCR_STROKE_PTR;


/****************************************************************************
 msgScrClear					takes void, returns STATUS
	Deletes all the strokes in the scribble.

 In response to this message, the scribble sets its stroke count to zero.
 scribble's stroke count to 0.  It also frees all allocated memory. 
 
 Important: Observers are not notified!
*/
#define msgScrClear   			MakeMsg(clsScribble,15)


/****************************************************************************
 msgScrRender					takes P_SCR_RENDER, returns STATUS
	Renders a scribble in a window through a DC.

 Draws the strokes in the scribble using pArgs->dc.  The start and stop
 indices describe the inclusive range of strokes to be rendered. Only
 strokes intersecting pArgs->rect are rendered.  pArgs->rect must be in
 window device coordinates.  pArgs->dc must be set up to draw in pen
 coordinates (using msgDcUnitsPen as described in sysgraf.h).
*/
#define msgScrRender			MakeMsg(clsScribble,13)

typedef struct SCR_RENDER {
	U16		start;			// stroke start index (0 for first)
	U16		stop;			// stroke stop index (maxU16 for last)
	XY32	base;			// unused
	OBJECT	dc;				// dc for rendering the stroke
	RECT32	rect;			// dirty rect
} SCR_RENDER, *P_SCR_RENDER;


/****************************************************************************
 msgScrHit						takes P_SCR_HIT, returns STATUS
	Passes back the next stroke which intersects pArgs->rect.

 This message is typically sent multiple times to find all strokes that
 intersect pArgs->rect.  The hit-test is quite simple -- a stroke
 "intersects" if the stroke's bounding box intersects pArgs->rect.

 Clients should set pArgs->index to 0 before first sending this message and
 then not disturb that field between sends.

 If a hit is found, pArgs->hit is true and pArgs->index is the stroke
 index.  Otherwise pArgs->hit is false.

 Example:
//{
	P_SCR_HIT hit;
	hit.rect = <rect to check>
	hit.index = 0;
	hit.hit = TRUE;
	while (hit.hit) {
		ObjectCall(msgScrHit, scribble, &hit);
		if (hit.hit) {
			// hit.index now equals the first stroke that intersected
		}
	}
//}
*/
#define msgScrHit   			MakeMsg(clsScribble,14)

typedef struct SCR_HIT {
	RECT32	rect;			// In: rect to test against, in pen coordinates.
	U16		index;			// In: For the first send, should be 0.  Do
							// not disturb between sends.  Out: if 
							// pArgs->hi is true, the index of the next
							// stroke that intersects pArgs->rect.
	BOOLEAN hit;			// Out: true if another stroke intersect 
							// pArgs->rect; false when no more strokes 
							// intersect.
} SCR_HIT, *P_SCR_HIT;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Notifications Sent to Observers						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgScrCompleted				takes NULL, returns STATUS
	Notifies observers that scribble input has been completed.

 This notification is sent as part of a scribble's response to
 msgScrComplete.

 Typical use:  Translators that are observing the scribble may perform
 their translation in response to this message.  (See xlate.h for more
 information.)
*/
#define msgScrCompleted			MakeMsg(clsScribble, 0x80)


/****************************************************************************
 msgScrAddedStroke				takes P_SCR_ADDED_STROKE, returns STATUS
	Notifies observers that a stroke has been added to the scribble.

 This notification sent as part of a scribble's response to
 msgScrAddStroke. 
*/
#define msgScrAddedStroke		MakeMsg(clsScribble, 0x81)

typedef struct SCR_ADDED_STROKE {
	U16				index;		// index of the added stroke
	P_PEN_STROKE	pStroke;	// pointer to the stroke data structure
} SCR_ADDED_STROKE, *P_SCR_ADDED_STROKE;


/****************************************************************************
 msgScrRemovedStroke			takes P_SCR_REMOVED_STROKE, returns STATUS
	Notifies observers that a stroke has been removed from the scribble.

 This notification is sent as part of a scribble's response to
 msgScrDeleteStroke.
*/
#define msgScrRemovedStroke		MakeMsg(clsScribble, 0x82)

typedef struct SCR_REMOVED_STROKE {
	U16		index;			// index of the removed stroke
	U16		id;	  		 	// stroke identifier
	RECT32	bounds;		  	// bounds of the removed stroke (in pen units)
} SCR_REMOVED_STROKE, *P_SCR_REMOVED_STROKE;

#endif
