/****************************************************************************
 File: sio.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.10  $
   $Author:   rgratacap  $
     $Date:   28 Feb 1992 13:12:18  $

 This file contains the API for clsMILAsyncSIODevice.
 clsMILAsyncSIODevice inherits from clsStream.
 Provides the serial port interface, see also stream.h for the stream 
 messages.
****************************************************************************/
#ifndef SIO_INCLUDED
#define SIO_INCLUDED

#include <go.h>
#include <clsmgr.h>
#include <milserv.h>

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				Common #defines and typedefs							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define	stsSioPortInUse		MakeStatus(clsMILAsyncSIODevice, 1)

#define	milDefaultBaudRate		9600
#define	milDefaultXonChar		0x11
#define	milDefaultXoffChar		0x13
#define	milDefaultModemControl	milDataTerminalReady | milRequestToSend
#define	milDefaultStopBits		milOneStopBit
#define	milDefaultParityType	milNoParity
#define	milDefaultWordLength	milEightBitWord
#define	milDefaultXonTimeout	(U32)30000
#define	milDefaultLineToStop	milRequestToSend

typedef OBJECT		SIO;
typedef SIO *		P_SIO;

Enum16(SIO_EVENT_MASK) {
	sioEventCTS             = flag0,	// CTS line has changed state 
	sioEventDSR             = flag1,	// DSR line has changed state 
	sioEventDCD             = flag2,	// DCD line has changed state 
	sioEventRI              = flag3,	// RI line has changed state 
	sioEventRxChar          = flag4,	// Rx buffer has become not empty.
                                        // Note: The receive buffer must be
                                        // empty for a received character
                                        // to generate this event!  
	sioEventRxBreak         = flag5,	// Break condition has been received
	sioEventTxBufferEmpty   = flag6,	// Tx buffer has become empty
	sioEventRxError         = flag7,	// parity, framing, or overrun error
	sioAllEvents			= flag0 | flag1 | flag2 | flag3 | flag4
							  | flag5 | flag6 | flag7
};


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Asynchronous SIO Class Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSioBaudSet takes U32, returns STATUS
    Sets the serial port baud rate.

 Maximum possible setting 115200.
 Actual baud rate = (115200/((U32)(115200/baudRate)))
 Default setting 9600 baud
*/
#define msgSioBaudSet		MakeMsg(clsMILAsyncSIODevice,4)



/****************************************************************************
 msgSioLineControlSet takes P_SIO_LINE_CONTROL_SET, returns STATUS
    Sets serial port data bits per character, stop bits, and parity.

 Default setting 8 bits, 1 stop bit, no parity.
*/
#define msgSioLineControlSet	MakeMsg(clsMILAsyncSIODevice,5)

Enum16(SIO_DATA_BITS) {
	sioSixBits = 6,
	sioSevenBits = 7,
	sioEightBits = 8
};

Enum16(SIO_STOP_BITS) {
	sioOneStopBit = 0,
	sioOneAndAHalfStopBits = 1,
	sioTwoStopBits = 2
};

Enum16(SIO_PARITY) {
	sioNoParity = 0,
	sioOddParity = 1,
	sioEvenParity = 2
};

typedef struct {
	SIO_DATA_BITS   dataBits;       
	SIO_STOP_BITS   stopBits;
	SIO_PARITY      parity;
} SIO_LINE_CONTROL_SET, *P_SIO_LINE_CONTROL_SET;


/****************************************************************************
 msgSioControlOutSet takes P_SIO_CONTROL_OUT_SET, returns STATUS
    Controls serial port output lines dtr and rts.

 Default setting dtr active, rts active.
*/
#define msgSioControlOutSet		MakeMsg(clsMILAsyncSIODevice,6)

typedef struct {
	BOOLEAN         dtr;		// true activates, false deactivates 
	BOOLEAN         rts;		// true activates, false deactivates
	BOOLEAN         out1;		// true activates, false deactivates
	BOOLEAN         out2;		// true activates, false deactivates
} SIO_CONTROL_OUT_SET, *P_SIO_CONTROL_OUT_SET;


/****************************************************************************
 msgSioControlInStatus takes P_SIO_CONTROL_IN_STATUS, returns STATUS
     Reads the current state of the serial port input control lines.

*/
#define msgSioControlInStatus	MakeMsg(clsMILAsyncSIODevice,7)

typedef struct {
	BOOLEAN         cts;		// out - true = active (Clear To Send)
	BOOLEAN         dsr;		// out - true = active (Data Set Ready)
	BOOLEAN         dcd;		// out - true = active (Data Carrier Detect)
    BOOLEAN         ri;			// out - true = active (Ring Indicator)
} SIO_CONTROL_IN_STATUS, *P_SIO_CONTROL_IN_STATUS;

#define	rlsd	dcd

/****************************************************************************
 msgSioFlowControlCharSet takes P_SIO_FLOW_CONTROL_CHAR_SET, returns STATUS
    Defines serial port XON/XOFF flow control characters.  
    
 Valid only if xon-xoff flow control is enabled.  

 Default xon character 0x11 (control-q), default xoff character 0x13
 (control-s).
*/
#define msgSioFlowControlCharSet	MakeMsg(clsMILAsyncSIODevice,8)

typedef struct {
	U8              xonChar;        // xon character (default control-Q)
	U8              xoffChar;       // xoff character (default control-S)
} SIO_FLOW_CONTROL_CHAR_SET, *P_SIO_FLOW_CONTROL_CHAR_SET;


/****************************************************************************
 msgSioBreakSend takes P_SIO_BREAK_SEND, returns STATUS
    Sends a break for the specified duration.

 Constant 0's transmitted on the serial line for the specified duration.  
 Typical durations are around 200-400 milliseconds).
*/
#define msgSioBreakSend		MakeMsg(clsMILAsyncSIODevice,11)

typedef struct {
    OS_MILLISECONDS	milliseconds;	// break duration
} SIO_BREAK_SEND, *P_SIO_BREAK_SEND;


/****************************************************************************
 msgSioBreakStatus takes P_SIO_BREAK_STATUS, returns STATUS
    Sends back the number of breaks received so far.  
    
 Also clears the internal break counter.
*/
#define msgSioBreakStatus	MakeMsg(clsMILAsyncSIODevice,13)

typedef struct {
	U32				breaksReceived;	// out
} SIO_BREAK_STATUS, *P_SIO_BREAK_STATUS;
    

/****************************************************************************
 msgSioReceiveErrorsStatus takes P_SIO_RECEIVE_ERRORS_STATUS, returns STATUS
    Sends back the number of receive errors and the number of dropped bytes
	(due to buffer overflows).
    
 Also clears the internal counters.
*/
#define msgSioReceiveErrorsStatus	MakeMsg(clsMILAsyncSIODevice,36)

typedef struct {
	U32				droppedBytes;	// out
	U32				receiveErrors;	// out
} SIO_RECEIVE_ERRORS_STATUS, *P_SIO_RECEIVE_ERRORS_STATUS;
    

/****************************************************************************
 msgSioInputBufferStatus takes P_SIO_INPUT_BUFFER_STATUS, returns STATUS
    Provides input buffer status.
 
 Sends back the number of characters in the input buffer and the amount of 
 empty room in the input buffer.
*/
#define msgSioInputBufferStatus	MakeMsg(clsMILAsyncSIODevice,16)

typedef struct {
    U32				bufferChars;	// out, number of chars in buffer
    S32				bufferRoom;		// out, amount of empty room in buffer
    BOOLEAN			receiverFrozen;	// out, is receive frozen?
} SIO_INPUT_BUFFER_STATUS, * P_SIO_INPUT_BUFFER_STATUS;


/****************************************************************************
 msgSioOutputBufferStatus takes P_SIO_OUTPUT_BUFFER_STATUS, returns STATUS
    Provides output buffer status.

 Sends back the number of characters in the output buffer and the amount of 
 empty room in the output buffer.
*/
#define msgSioOutputBufferStatus	MakeMsg(clsMILAsyncSIODevice,17)

typedef struct {
    U32				bufferChars;	// out, number of chars in buffer
    S32				bufferRoom;		// out, amount of empty room in buffer
    BOOLEAN			transmitterFrozen;  // out, is transmit frozen?
} SIO_OUTPUT_BUFFER_STATUS, * P_SIO_OUTPUT_BUFFER_STATUS;


/****************************************************************************
 msgSioInputBufferFlush takes pNull, returns STATUS
    Flushes the contents of the input buffer.

*/
#define msgSioInputBufferFlush	MakeMsg(clsMILAsyncSIODevice,18)


/****************************************************************************
 msgSioOutputBufferFlush takes pNull, returns STATUS
    Flushes the contents of the output buffer.

*/
#define msgSioOutputBufferFlush MakeMsg(clsMILAsyncSIODevice,19)


/****************************************************************************
 msgSioFlowControlSet takes P_SIO_FLOW_CONTROL_SET, returns STATUS
    Selects flow control type.
    
 Flow control types: no flow control, XON/XOFF flow control, or 
 hardware flow control.  Default: XON/XOFF flow control.
*/
#define msgSioFlowControlSet		MakeMsg(clsMILAsyncSIODevice,20)

Enum16(SIO_FLOW_TYPE) {
	sioNoFlowControl            = 0x11,
	sioXonXoffFlowControl       = 0x22,
	sioHardwareFlowControl      = 0x44,

    // To independently set receive and transmit flow control OR together
    // one from each of the following two sets. 
    // i.e.,     .flowControl = sioRxXonXoff | sioTxHardware; 
    // YOU MUST SET BOTH THE TX AND RX FLOW CONTROL!

    // Transmit flow control
    sioTxNone                   = 0x01,
    sioTxXonXoff                = 0x02,
    sioTxHardware               = 0x04,

    // Receive flow control
    sioRxNone                   = 0x10,
    sioRxXonXoff                = 0x20,
    sioRxHardware               = 0x40
};

typedef struct {
    SIO_FLOW_TYPE           flowControl;
} SIO_FLOW_CONTROL_SET, *P_SIO_FLOW_CONTROL_SET;

/****************************************************************************
 msgSioEventStatus takes P_SIO_EVENT_STATUS, returns STATUS
    Sends back current state of event word, and then clears the event word.

 Bit set indicates an event has occurred.  Events do not have to be enabled
 for eventMask to be set.
*/
#define msgSioEventStatus 		MakeMsg(clsMILAsyncSIODevice,21)

typedef struct {
    SIO_EVENT_MASK	eventMask;		// out
} SIO_EVENT_STATUS, *P_SIO_EVENT_STATUS;


/****************************************************************************
 msgSioEventSet takes P_SIO_EVENT_SET, returns STATUS
    Enables event notification.

 Bits set in the eventMask enable msgSioEventHappened to be sent to 
 uid.  Default: eventMask = 0, all event notification disabled.
*/
#define msgSioEventSet			MakeMsg(clsMILAsyncSIODevice,22)

typedef struct {
    SIO_EVENT_MASK	eventMask;		// in, events to respond to
    OBJECT			client;			// object to inform when event happens
} SIO_EVENT_SET, *P_SIO_EVENT_SET;


/****************************************************************************
 msgSioEventGet takes P_SIO_EVENT_SET, returns STATUS
    Gets the current sio event setting.

*/
#define msgSioEventGet			MakeMsg(clsMILAsyncSIODevice,29)


/****************************************************************************
 msgSioEventHappened takes P_SIO_EVENT_HAPPENED, returns STATUS
    Notifies client of event occurance.

 Message sent to object to notify it of event occurrance.
 Possibly, more than one bit will be set in the event mask (bits may be
 set from disabled events, although disabled events never cause this 
 message to be generated.  Clears event mask.
*/
#define msgSioEventHappened		MakeMsg(clsMILAsyncSIODevice,23)

typedef struct {
    SIO_EVENT_MASK	eventMask;		// out, bits set indicate event happened.
    OBJECT			self;           // object which generated message.
} SIO_EVENT_HAPPENED, *P_SIO_EVENT_HAPPENED;


/****************************************************************************
 msgSioInit takes P_SIO_INIT, returns STATUS
	Initializes the serial device to its default state.

*/
#define msgSioInit              MakeMsg(clsMILAsyncSIODevice,26)

typedef struct {
    U32				inputSize;		// size of the input buffer
    U32				outputSize;		// size of the output buffer
} SIO_INIT, *P_SIO_INIT;



/****************************************************************************
 msgSioGetMetrics takes P_SIO_METRICS, returns STATUS
    Sends back the sio metrics.

*/
#define msgSioGetMetrics        MakeMsg(clsMILAsyncSIODevice,24)

typedef struct {
    U32                         baud;           // out/in
    SIO_LINE_CONTROL_SET        line;           // out/in
    SIO_CONTROL_OUT_SET         controlOut;     // out/in
    SIO_FLOW_CONTROL_CHAR_SET   flowChar;       // out/in
    SIO_FLOW_CONTROL_SET        flowType;       // out/in
    //
    // Changing the bufferSize fields causes reinitialization of serial
    // chip!
    SIO_INIT                    bufferSize;     // out/in
    U8                          spare[12];
} SIO_METRICS, *P_SIO_METRICS;


/****************************************************************************
 msgSioSetMetrics takes P_SIO_METRICS, returns STATUS
    Sets the sio metrics.

*/
#define msgSioSetMetrics        MakeMsg(clsMILAsyncSIODevice,25)


/****************************************************************************
 msgSioSetReplaceCharProc takes P_SIO_REPLACE_CHAR, returns STATUS
    Replaces the built in receive character interrupt routine.
    
 This message calls the user defined function when a character is received.
 The procedure has the option to filter the character or to return and have
 the character processed normally.  The user defined fuction returns a
 BOOLEAN indicating whether the function filtered the character or not.

*/
#define msgSioSetReplaceCharProc	MakeMsg(clsMILAsyncSIODevice,72)

typedef BOOLEAN (EXPORTED *P_SIO_CHAR_HANDLER)(U16		ch,
											   BOOLEAN	DCDState,
											   U32		handle);

typedef struct SIO_REPLACE_CHAR
{
	P_SIO_CHAR_HANDLER	pRxHandler;	// address of character handler
	U32					handle;		// user data (meaningless to
									// clsMILAsyncSIO)
} SIO_REPLACE_CHAR, *P_SIO_REPLACE_CHAR;


/****************************************************************************
 msgNew takes P_SIO_NEW, returns STATUS
    Creates a new clsMILAsyncSIODevice object.

*/

typedef struct SIO_NEW
{
	milServiceNewFields
} SIO_NEW, *P_SIO_NEW;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Asynchronous SIO Option Card Tags					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define sioTagOptionCard    MakeTag(clsMILAsyncSIODevice, 19) // Card tag

#define sioTagName		    MakeTag(clsMILAsyncSIODevice, 20)
#define sioTagBaud			MakeTag(clsMILAsyncSIODevice, 21)
#define sioTagFlowControl	MakeTag(clsMILAsyncSIODevice, 22)
#define sioTagParity		MakeTag(clsMILAsyncSIODevice, 23)
#define sioTagDataBits		MakeTag(clsMILAsyncSIODevice, 24)
#define sioTagStopBits		MakeTag(clsMILAsyncSIODevice, 25)

#define sioTagBaud300		MakeTag(clsMILAsyncSIODevice, 40)
#define sioTagBaud600		MakeTag(clsMILAsyncSIODevice, 41)
#define sioTagBaud1200		MakeTag(clsMILAsyncSIODevice, 42)
#define sioTagBaud2400		MakeTag(clsMILAsyncSIODevice, 43)
#define sioTagBaud4800		MakeTag(clsMILAsyncSIODevice, 44)
#define sioTagBaud9600		MakeTag(clsMILAsyncSIODevice, 45)
#define sioTagBaud19200		MakeTag(clsMILAsyncSIODevice, 46)
#define sioTagBaud38400		MakeTag(clsMILAsyncSIODevice, 47)
#define sioTagBaud57600		MakeTag(clsMILAsyncSIODevice, 48)
#define sioTagBaud115200	MakeTag(clsMILAsyncSIODevice, 49)

#define sioTagFlowNone      MakeTag(clsMILAsyncSIODevice, 55)
#define sioTagFlowXonXoff   MakeTag(clsMILAsyncSIODevice, 56)
#define sioTagFlowHardware  MakeTag(clsMILAsyncSIODevice, 57)

#define sioTagParityNone    MakeTag(clsMILAsyncSIODevice, 60)
#define sioTagParityOdd     MakeTag(clsMILAsyncSIODevice, 61)
#define sioTagParityEven    MakeTag(clsMILAsyncSIODevice, 62)

#define sioTagBits7         MakeTag(clsMILAsyncSIODevice, 65)
#define sioTagBits8         MakeTag(clsMILAsyncSIODevice, 66)

#define sioTagStopBitsOne   MakeTag(clsMILAsyncSIODevice, 70)
#define sioTagStopBitsTwo   MakeTag(clsMILAsyncSIODevice, 71)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Function prototypes									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

STATUS	EXPORTED ClsSioInit(void);
void	EXPORTED SioSemaClear(P_UNKNOWN pHandle);

#endif	// SIO_INCLUDED
