/****************************************************************************
 File: stream.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.15  $
   $Author:   rgratacap  $
	 $Date:   01 Apr 1992 19:19:06  $

 This file contains the API definition for clsStream.

 clsStream inherits from clsObject.

 clsStream is an abstract class -- it does not completely implement its own
 protocol.  Subclasses of clsStream must complete the implementation. 
 clsFileHandle is an important subclass of clsStream (see fs.h).

 The functions described in this file are contained in PENPOINT.LIB. 
****************************************************************************/
#ifndef STREAM_INCLUDED
#define STREAM_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef UID_INCLUDED
#include <uid.h>
#endif

#ifndef OSTYPES_INCLUDED
#include <ostypes.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   				Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define streamNewFields	\
	objectNewFields


typedef struct STREAM_NEW {
	streamNewFields
} STREAM_NEW, * P_STREAM_NEW;


/*
 * Several types in this file contain "streamElements."
 *
 * The streamElements fields are:
 *    -:  numBytes:	In: size of buffer
 *    -:  pBuf:		In: buffer
 *    -:  count:	Out: number of bytes transferred
*/
#define streamElements		\
	U32			numBytes;	\
	P_UNKNOWN	pBuf;		\
	U32			count;		

/* Status codes */
#define stsTimeOutWithData			MakeWarning(clsStream, 1)

/*
 * stsStreamDisconnected status is returned by all stream calls when the
 * service executing the stream function is no longer in a connected 
 * state (A disconnectable service is clsMILAsyncSIO).
 *
 * Clients must not send other stream messages to the disconnected service.
 *
 * Penpoint can notify clients or clients may find services' connected
 * states (see service.h and servmgr.h).
*/
#define stsStreamDisconnected		MakeStatus(clsStream, 1)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgStreamRead	takes P_STREAM_READ_WRITE, returns STATUS
	category: descendant responsibility
	Reads data from stream.

 msgStreamRead reads data from the stream into pBuf.  pBuf must point to a
 buffer which can hold at least numBytes bytes.  The number of bytes read
 is passed back in count.

 If you try to read 0 bytes when at the end of the data stream stsOK is
 returned.

 Return Value
	< stsOK:				No data read.
 	>= stsOK:				Count of bytes is non-zero.
	stsEndOfData:			Count is zero and at the end of data.
*/
#define msgStreamRead				MakeMsg(clsStream,1)


typedef struct {
	streamElements
} STREAM_READ_WRITE, * P_STREAM_READ_WRITE;


/****************************************************************************
 msgStreamWrite			takes P_STREAM_READ_WRITE, returns STATUS
	category: descendant responsibility
	Writes data to stream.

 Writes numBytes from pBuf into the stream.  Returns stsOK if all bytes
 are written.
*/
#define msgStreamWrite				MakeMsg(clsStream,2)


/****************************************************************************
 msgStreamReadTimeOut	takes P_STREAM_READ_WRITE_TIMEOUT, returns STATUS
	category: descendant responsibility
	Reads data from stream with timeout.

 msgStreamReadTimeOut reads data from the stream into pBuf.  pBuf must 
 point to a buffer which can hold least numBytes bytes.  The number of
 bytes read is passed back in count.
 
 When count is greater than zero the status returned is always greater than
 or equal to stsOK.

 Return Value
	stsTimeOutWithData:		Count is greater than zero but less than 
							numBytes because of a timeout.
	stsTimeOut:				Count is zero and the timeout has expired.
	stsEndOfData:			Count is zero and at the end of data.
*/
#define msgStreamReadTimeOut		MakeMsg(clsStream,3)


typedef struct {
	streamElements
	OS_MILLISECONDS			timeOut;		// In: milliseconds until timeout
} STREAM_READ_WRITE_TIMEOUT, * P_STREAM_READ_WRITE_TIMEOUT;	 


/****************************************************************************
 msgStreamWriteTimeOut	takes P_STREAM_READ_WRITE_TIMEOUT, returns STATUS
	category: descendant responsibility
	Writes to the stream with timeout.

 Writes numBytes from pBuf into the stream.

 Return Value
 	stsOK:					All bytes were written.
	stsTimeOut:				Timeout has expired before all data written.
*/
#define msgStreamWriteTimeOut		MakeMsg(clsStream,4)


/****************************************************************************
 msgStreamFlush		takes pNull, returns STATUS
	category: descendant responsibility
	The stream flushes any buffered data.

 clsStream's default response is to return stsMessageIgnored.  Most 
 subclasses override clsStream's response.

 Return Value
 	stsOK:					Buffers were successfully emptied.
	stsFailed:				Buffers do not empty after some timeout period.
*/
#define msgStreamFlush				MakeMsg(clsStream,5)


/****************************************************************************
 msgStreamSeek	  takes P_STREAM_SEEK, returns STATUS
	Sets the stream's Current Byte Position.

 clsStream's default response is to return stsMessageIgnored.  Most 
 subclasses override clsStream's response.
*/
#define msgStreamSeek				MakeMsg(clsStream,6)


Enum16(STREAM_SEEK_MODE) {

	// Relative to beginning of file, end of file, or Current Byte Position
	streamSeekBeginning		= 0,
	streamSeekEnd			= 1,
	streamSeekCurrent		= 2,

	// Default setting
	streamSeekDefaultMode	= streamSeekBeginning

};


typedef struct STREAM_SEEK {
	STREAM_SEEK_MODE	mode;			//
	S32					offset;			// relative change from seek origin
	U32					curPos;			// Out: byte position after seek
	U32					oldPos;			// Out: byte position before seek
	BOOLEAN				eof;			// Out: Is new pos at end of file?
} STREAM_SEEK, * P_STREAM_SEEK;


/****************************************************************************
 msgStreamBlockSize takes P_STREAM_BLOCK_SIZE, returns STATUS
	category: descendant responsibility
	Passes back the most efficient write block size for this stream.

 clsStream's default response is to return a blockSize of 512.  Most
 subclasses override clsStream's response.
*/
#define msgStreamBlockSize	   MakeMsg(clsStream,7)

typedef struct {
	U32			blockSize;		// out: preferred write block size
} STREAM_BLOCK_SIZE, * P_STREAM_BLOCK_SIZE;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Functions								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * The P_UNKNOWN declarations for the following are assumed to be FILE*.
 * Maintaining a clean separation between ANSI and PenPoint header files
 * prevents the use of the true type.
*/

/****************************************************************************
 StdioStreamBind	returns pointer to FILE
    Returns a stdio file pointer bound to a stream object.
*/
P_UNKNOWN EXPORTED StdioStreamBind(
	OBJECT 	obj);


/****************************************************************************
 StdioStreamUnbind	returns int
	Frees the stdio file handle bound to a stream object.
*/
int EXPORTED StdioStreamUnbind(
	P_UNKNOWN	pFile);


/****************************************************************************
 StdioStreamToObject	returns OBJECT
    Returns the stream object bound to a stdio file pointer.
*/
OBJECT EXPORTED StdioStreamToObject(
	P_UNKNOWN	pFile);

//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Private Functions							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 ClsStreamInit	returns STATUS
    Initializes the stream class.
*/
STATUS GLOBAL ClsStreamInit(void);

//REFGEN ENDIGNORE

#endif
