/****************************************************************************
 File: timer.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.5  $
   $Author:   twiegman  $
	 $Date:   06 Mar 1992 12:05:38  $

 This file contains the API definition for clsTimer.  
 
 Notes:
 
 "theTimer" is a well known object that provides timer and alarm support.
 clsTimer inherits from clsObject.
****************************************************************************/
#ifndef TIMER_INCLUDED
#define TIMER_INCLUDED

/* Include file dependencies for this include file */
#ifndef GO_INCLUDED
#include <go.h>
#endif
#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif
#ifndef OS_INCLUDED
#include <os.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Class Timer Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgTimerRegister	takes P_TIMER_REGISTER_INFO, returns STATUS
	Registers a request for notification.

 Sent by the client to the timer object for notification after a 
 specified time period has elapsed.	At that time,
 msgTimerNotify will be sent (via ObjectPost) to the client.	 
 See that message for details.

 When the machine is turned off, the time period will stop counting
 down until the machine is turned back on.

 To stop the timeout message, use msgTimerStop.

 The use of ObjectPost to send the msgTimerNotify message means that
 it will be synchronous with input events.

 Return Value
	stsBadObject:			The client field cannot be a local object.
*/
#define msgTimerRegister			MakeMsg(clsTimer, 1)

typedef struct TIMER_REGISTER_INFO {
  OBJECT			client;				// client object to notify
  OS_MILLISECONDS	time;				// waiting period before msg is sent
  P_UNKNOWN			clientData;			// Uninterpreted client data
  OS_HANDLE			transactionHandle;	// Out: transaction handle
} TIMER_REGISTER_INFO, * P_TIMER_REGISTER_INFO;


/****************************************************************************
 msgTimerRegisterAsync	takes P_TIMER_REGISTER_INFO, returns STATUS
	Registers a request for notification.

 Sent by the client to the timer object for notification after a 
 specified time period has elapsed.	At that time,
 msgTimerNotify will be sent (via ObjectPostAsync) to the client.  
 See that message for details.

 When the machine is turned off, the time period will stop counting
 down until the machine is turned back on.

 To stop the timeout message, use msgTimerStop.

 The use of ObjectPostAsync to send the msgTimerNotify message means 
 that it will NOT be synchronous with input events.
*/
#define msgTimerRegisterAsync		MakeMsg(clsTimer, 9)


/****************************************************************************
 msgTimerRegisterDirect	takes P_TIMER_REGISTER_INFO, returns STATUS
	Registers a request for notification.

 Sent by the client to the timer object for notification after a 
 specified time period has elapsed.	At that time,
 msgTimerNotify will be sent (via ObjectPostDirect) to the client.  
 See that message for details.

 When the machine is turned off, the time period will stop counting
 down until the machine is turned back on.

 To stop the timeout message, use msgTimerStop.

 The use of ObjectPostDirect to send the msgTimerNotify message means 
 that it will NOT be synchronous with input events.
*/
#define msgTimerRegisterDirect		MakeMsg(clsTimer, 12)


/****************************************************************************
 msgTimerRegisterInterval	takes P_TIMER_INTERVAL_INFO, returns STATUS
	Registers a request for interval notification.
	
 Sent by the client to the timer for a notification message on a 
 specified time interval.  After each time interval, msgTimerNotify 
 will be posted (via ObjectPost) to the client.	

 When the machine is turned off, the time period will stop counting
 down until the machine is turned back on.

 To stop the interval messages, use msgTimerStop.

 The use of ObjectPost to send the msgTimerNotify message means 
 that it will be synchronous with input events.

 Return Value
	stsBadObject:			The client field cannot be a local object.
*/
#define msgTimerRegisterInterval			MakeMsg(clsTimer, 2)

typedef struct TIMER_INTERVAL_INFO {
  OBJECT			client;				// client object to notify
  OS_MILLISECONDS	interval;			// waiting interval before msg is sent
  P_UNKNOWN			clientData;			// Uninterpreted client data
  OS_HANDLE			transactionHandle;	// Out: transaction handle
} TIMER_INTERVAL_INFO, * P_TIMER_INTERVAL_INFO;


/****************************************************************************
 msgTimerStop	takes OS_HANDLE, returns STATUS
	Stops a timer transaction.
*/
#define msgTimerStop						MakeMsg(clsTimer, 11)


/****************************************************************************
 msgTimerTransactionValid	takes OS_HANDLE, returns STATUS
	Determines if a timer transaction is valid.
*/
#define msgTimerTransactionValid			MakeMsg(clsTimer, 10)


/****************************************************************************
 msgTimerNotify takes P_TIMER_NOTIFY, returns nothing
	category: advisory message
	Notifies the client that the timer request has elapsed.	 

 Sent by the timer object to the client.
*/
#define msgTimerNotify						MakeMsg(clsTimer, 3)

typedef struct TIMER_NOTIFY {
  P_UNKNOWN clientData;				// client data returned
  OS_HANDLE transactionHandle;		// transaction handle
} TIMER_NOTIFY, * P_TIMER_NOTIFY;


/****************************************************************************
 msgTimerAlarmRegister	takes P_TIMER_ALARM_INFO, returns STATUS
	Registers a request for alarm notification.

 Alarms differ from timer requests in that a time and date specifies 
 when an alarm is to occur. The timer will ObjectPost msgTimerAlarmNotify 
 to the client when the alarm goes off.	See that message for details.

 Alarms will alarm within a minute of the alarm time.  

 When the machine is turned off, the alarm is still active.	An
 alarm will turn the machine on.

 To stop the alarm, use the message msgTimerAlarmStop.

 Return Value
	stsBadObject:			The client field cannot be a local object.
*/
#define msgTimerAlarmRegister				MakeMsg(clsTimer, 5)

Enum16(TIMER_ALARM_MODE) {
  timerAbsoluteDate,			// alarm only on specified date and time
  timerEveryWeek,				// alarm when dayOfWeek, hours, minutes match
  timerEveryDay					// alarm when hours and minutes match
};

typedef struct TIMER_ALARM_INFO {
  OBJECT			client;				// client object to notify
  OS_DATE_TIME		alarmTime;			// alarm time
  P_UNKNOWN			clientData;			// Uninterpreted client data
  OS_HANDLE			transactionHandle;	// Out: transaction handle
  TIMER_ALARM_MODE	alarmMode;
} TIMER_ALARM_INFO, * P_TIMER_ALARM_INFO;


/****************************************************************************
 msgTimerAlarmStop	takes OS_HANDLE, returns STATUS
	Stops a pending alarm request. 
*/
#define msgTimerAlarmStop					MakeMsg(clsTimer, 6)


/****************************************************************************
 msgTimerAlarmNotify	takes P_ALARM_NOTIFY, returns nothing
	category: advisory message
	Notifies the client that the alarm request has elapsed.	 

 Sent by the timer object to the client.
*/
#define msgTimerAlarmNotify					MakeMsg(clsTimer, 7)

typedef struct ALARM_NOTIFY {
  P_UNKNOWN clientData;				// client data returned
  OS_HANDLE transactionHandle;		// transaction handle
  BOOLEAN	alarmCausedPoweron;		// power up occurred due to alarm
} ALARM_NOTIFY, * P_ALARM_NOTIFY;

//REFGEN BEGINIGNORE

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Private Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgTimerCheck	takes nothing, returns nothing
	category: advisory message
	Notifies the timer that a timer interrupt has occurred.

 This message is sent by the timer "interrupt" to the timer object to tell
 it that a timer request has elapsed.  This is used internally by PenPoint.
*/
#define msgTimerCheck						MakeMsg(clsTimer, 4)


/****************************************************************************
 msgTimerPowerUpCheck	takes nothing, returns nothing
	category: advisory message
	Notifies the timer object that power up has just occurred.

 This message is sent by the power up code to the timer object to 
 tell it to check if the power up occurred because of an alarm.

 NOTE: This message is used internally by PenPoint and is now obsolete.
*/
#define msgTimerPowerUpCheck				MakeMsg(clsTimer, 8)

//REFGEN ENDIGNORE

#endif	// TIMER_INCLUDED
