/****************************************************************************
 File: track.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.24  $
   $Author:   cmeyer  $
     $Date:   13 Mar 1992 19:04:38  $

 This file contains the API definition for clsTrack.

 clsTrack inherits from clsObject.
 Provides transient drawing feedback for various pen dragging situations,
 such as resizing and dragging frames.
****************************************************************************/

/**** Debugging Flags ****/
/*
 The clsTrack debugging flag is 'K'.  Defined values are:

    flag15 (0x8000):    general debug info
*/

#ifndef TRACK_INCLUDED
#define TRACK_INCLUDED

													#ifndef WIN_INCLUDED
#include <win.h>
													#endif

/* * * * * * * * *   Common #defines and typedefs   * * * * * * * * * * * */

/****  Track Styles  ****/
#define tsTrackMove			0
#define tsTrackResize		1

/****  Anchor Styles  ****/
#define tsAnchorUL			0	// upper-left
#define tsAnchorUR			1	// upper-right
#define tsAnchorLR			2	// lower-right
#define tsAnchorLL			3	// lower-left

/****  Draw Styles  ****/
#define tsDrawRect			0	// simple rectangle
#define tsDrawTabBarRect	1	// rectangle with vertical tab bar on right
#define tsDrawCmdBarRect	2	// rectangle with command bar at bottom
#define tsDrawTabCmdBarRect	3	// rectangle with both tab and command bars
#define tsDrawBitmap		4	// not implemented
#define tsDrawViaMessages	5	// forward msgTrackShow/Hide to client
#define tsDrawDoubleRect	6	// double rect as in clsBorder double thickness

/****  Thickness Styles  ****/
#define	tsThicknessSingle	0	// single-thick lines
#define	tsThicknessDouble	1	// double-thick lines

/****  Line Pattern Styles  ****/
#define	tsPatForeground		0	// foreground ink
#define	tsPatDashed			1	// sysDcPatLD50
//							2	// unused (reserved)
//							3	// unused (reserved)


typedef struct TRACK_STYLE {
	U16		track			: 2,	// track style (move or resize)
			anchor			: 2,	// corner to anchor (tsTrackResize only)
			draw			: 4,	// visual to draw
			update			: 1,	// send msgTrackUpdate to client
			autoDestroy		: 1,	// destroy self when done
			thickness		: 2,	// thickness of drawn lines
			pattern			: 2,	// line pattern of drawn lines
			startThickness	: 2;	// thickness of initial drawn lines

	U16		useThreshold	: 1,	// start tracking after msgPenMoveDown
			spare			: 15;	// reserved
} TRACK_STYLE, *P_TRACK_STYLE;

/****************************************************************************
 msgNew						takes P_TRACK_NEW, returns STATUS
	category: class message
	Creates a tracker.

 Note that if you change the default value for pArgs->track.constrainRect
 you should also insure pArgs->track.keepRect is correct for your new
 constrainRect.

 Here is some sample code for creating an instance of clsTrack to resize
 a window.  This is taken from clsGrabBox.  pInst->client is the window
 to be resized.

//{
	// distance to stay away from edge of parent after resize, in device units
	#define	trBottomParentMargin	0
	#define	trRightParentMargin	0

	// min. distance from bottom of child to top of parent, in device units
	#define	trTopParentMargin		12

	// min. distance from right of child to left of parent, in device units
	#define	trLeftParentMargin	12

	// absolute minimum resize width and height, in device units
	#define	trMinResizeWidth		20
	#define	trMinResizeHeight		20

 	TRACK_NEW		tn;

	// start a resize tracker
	ObjCallRet(msgNewDefaults, clsTrack, &tn, s);

	ObjCallRet(msgWinGetMetrics, pInst->client, &wm, s);

	tn.track.style.track = tsTrackResize;
	tn.track.win = wm.parent;
	tn.track.client = self;
	tn.track.clientData = pInst->client;		// window being resized
	tn.track.tag = tagBorderResize;
	tn.track.initRect = wm.bounds;

	// don't allow the grabbox to go off the edge of client's parent
	ObjCallRet(msgWinGetMetrics, wm.parent, &rm, s);

	tn.track.maxWH.w = rm.bounds.size.w - trRightParentMargin -
	  wm.bounds.origin.x;
	tn.track.maxWH.h = RectTop(&wm.bounds) - trBottomParentMargin;

	tn.track.minWH.w = RectRight(&wm.bounds) - (rm.bounds.size.w - trLeftParentMargin);
	tn.track.minWH.h = RectTop(&wm.bounds) - (rm.bounds.size.h - trTopParentMargin);

	tn.track.minWH.w = Max(tn.track.minWH.w, trMinResizeWidth);
	tn.track.minWH.h = Max(tn.track.minWH.h, trMinResizeHeight);

	switch (pInst->style.loc) {
		case gbLocTopEdge:
		case gbLocULCorner:
			tn.track.style.anchor = tsAnchorLR;
			break;

		case gbLocRightEdge:
		case gbLocURCorner:
			tn.track.style.anchor = tsAnchorLL;
			break;

		case gbLocLeftEdge:
		case gbLocLLCorner:
			tn.track.style.anchor = tsAnchorUR;
			break;

		case gbLocBottomEdge:
		default:
			tn.track.style.anchor = tsAnchorUL;
			break;
	}

	switch (pInst->style.loc) {
		default:
			// unconstrained
			break;

		case gbLocLeftEdge:
		case gbLocRightEdge:
			// constrained to horizontal
			tn.track.minWH.h = wm.bounds.size.h;
			tn.track.maxWH.h = wm.bounds.size.h;
			break;

		case gbLocBottomEdge: 
		case gbLocTopEdge:
			// constrained to vertical
			tn.track.minWH.w = wm.bounds.size.w;
			tn.track.maxWH.w = wm.bounds.size.w;
			break;
	}

	ObjCallRet(msgTrackProvideMetrics, pInst->client, &tn.track, s);
	ObjCallRet(msgNew, clsTrack, &tn, s);

	// start tracking at the initial down point
	wm.bounds.origin = *pXY;
	ObjCallRet(msgWinTransformBounds, theRootWindow, &wm, s);
	ObjCallRet(msgTrackStart, tn.object.uid, &wm.bounds.origin, s);
//}

 Here is some sample code for creating an instance of clsTrack to drag
 a window.  This is taken from clsBorder.  deltaWin is the window
 to be dragged.

//{
	// keep rect size for drag, in device units
	#define	trDefaultMoveKeep		12

 	TRACK_NEW		tn;

	ObjCallRet(msgNewDefaults, clsTrack, &tn, s);

	// constraint to parent's bounds
	ObjSendUpdateRet(msgWinGetMetrics, deltaWin, &clientMetrics, SizeOf(clientMetrics), s);
	if (!clientMetrics.parent)
		return stsOK;

	ObjSendUpdateRet(msgWinGetMetrics, clientMetrics.parent, &wm, SizeOf(wm), s);

	tn.track.style.startThickness = tsThicknessDouble;
	tn.track.win = clientMetrics.parent;
	tn.track.client = self;
	tn.track.clientData = deltaWin;
	tn.track.initRect = clientMetrics.bounds;
	tn.track.constrainRect.size = wm.bounds.size;
	tn.track.tag = tagBorderDrag;

	// start tracking at the initial point
	wm.parent = clientMetrics.parent;
	wm.bounds.origin = *pXY;
	ObjCallRet(msgWinTransformBounds, self, &wm, s);

	tn.track.keepRect.size.w = tn.track.keepRect.size.h = trDefaultMoveKeep;
	tn.track.keepRect.origin = wm.bounds.origin;
	tn.track.keepRect.origin.x -= trDefaultMoveKeep / 2;
	tn.track.keepRect.origin.y -= trDefaultMoveKeep / 2;

	ObjSendUpdateRet(msgTrackProvideMetrics, deltaWin, &tn.track, SizeOf(tn.track), s);
	ObjCallRet(msgNew, clsTrack, &tn, s);

	ObjCallRet(msgTrackStart, tn.object.uid, &wm.bounds.origin, s);
//}

*/
typedef struct TRACK_NEW_ONLY {
	TRACK_STYLE		style;
	WIN				win;			// objNull means use theRootWindow
	OBJECT			client;			// client to send msgTrackDone to
	P_UNKNOWN		image;			// optional image instead of box (not implemented)
	P_UNKNOWN		clientData;		// data for client to set
	OBJECT			tracker;		// ignored in msgInit
	RECT32			initRect;		// in device units, relative to win
	RECT32			rect;			// in device units, relative to win
	S32				tabBarW;		// tsDrawTabBarRect | tsDrawTabCmdBarRect
	S32				cmdBarH;		// tsDrawCmdBarRect | tsDrawTabCmdBarRect
	XY32			origXY;			// in device units, relative to win
	XY32			curXY;			// in device units, relative to win
	TAG				tag;			// optional distinguishing tag

	// if tsTrackMove
	RECT32			keepRect;		// in device units, relative to win
	RECT32			constrainRect;	// in device units, relative to win

	// if tsTrackResize
	SIZE32			minWH;			// in device units
	SIZE32			maxWH;			// in device units

	U32				spare;			// unused (reserved)
	U32				spare1;			// unused (reserved)
} TRACK_METRICS, *P_TRACK_METRICS,
  TRACK_NEW_ONLY, *P_TRACK_NEW_ONLY;

#define trackNewFields	\
	objectNewFields		\
	TRACK_NEW_ONLY		track;

typedef struct TRACK_NEW {
	trackNewFields
} TRACK_NEW, *P_TRACK_NEW;


/****************************************************************************
 msgNewDefaults					takes P_TRACK_NEW, returns STATUS
	category: class message
	Initializes the TRACK_NEW structure to default values.

	Sets all of pArgs->track to 0, then ...

//{
	pArgs->object.cap |= objCapCall;
	pArgs->track.style.autoDestroy		= true;
	pArgs->track.constrainRect.size.w	= maxS32 / 2;
	pArgs->track.constrainRect.size.h	= maxS32 / 2;
	pArgs->track.keepRect.origin.x		= maxS32 / 4;
	pArgs->track.keepRect.origin.y		= maxS32 / 4;
	pArgs->track.keepRect.size.w		= 1;
	pArgs->track.keepRect.size.h		= 1;
	pArgs->track.maxWH.w				= maxS32 / 2;
	pArgs->track.maxWH.h				= maxS32 / 2;
//}
*/

/*
 Default style:
//{
 	track			= tsTrackMove
 	anchor			= tsAnchorUL		(ignored when tsTrackMove)
 	draw			= tsDrawRect
 	update			= false
 	autoDestroy		= true
 	thickness		= tsThicknessSingle
 	pattern			= tsPatForeground
 	startThickness	= tsThicknessSingle
 	useThreshold	= false
//}
 */

/****************************************************************************
 msgTrackGetStyle				takes P_TRACK_STYLE, returns STATUS
	Passes back current style values.

*/
#define msgTrackGetStyle		MakeMsg(clsTrack, 1)

/****************************************************************************
 msgTrackSetStyle				takes P_TRACK_STYLE, returns STATUS
	Sets style values.

 If the new style values result in a different visual, and msgTrackStart
 has been sent, you should first send msgTrackHide with pArgs of the old
 TRACK_METRICS, then msgTrackSetStyle, then msgTrackShow with the new
 TRACK_METRICS.
*/
#define msgTrackSetStyle		MakeMsg(clsTrack, 2)

/****************************************************************************
 msgTrackGetMetrics				takes P_TRACK_METRICS, returns STATUS
	Passes back the current metrics.

*/
#define msgTrackGetMetrics		MakeMsg(clsTrack, 3)

/****************************************************************************
 msgTrackSetMetrics				takes P_TRACK_METRICS, returns STATUS
	Sets the metrics.

 See msgTrackSetStyle for notes on changing metrics after msgTrackStart
 has been sent.

 See Also
 	msgTrackSetStyle
*/
#define msgTrackSetMetrics		MakeMsg(clsTrack, 4)

/****************************************************************************
 msgTrackStart					takes P_XY32, returns STATUS
	Starts the tracker.

 The pArgs indicates the initial position of the pen (in device units,
 in the space of the metrics.win).  If pArgs is pNull, then
 metrics.origXY is used as the initial pen position.

 clsTrack will do the following:
	-:  self-send msgTrackConstrain to constrain the initial point.
	-:  grab all input events using InputSetGrab().
	-:  self-send msgTrackShow(&metrics) to paint the tracker.
*/
#define msgTrackStart			MakeMsg(clsTrack, 5)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *									Client Messages							   			*
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgTrackDone					takes P_TRACK_METRICS, returns STATUS
	category: client notification
	Sent by clsTrack to metrics.client when the track is done.

*/
#define msgTrackDone			MakeMsg(clsTrack, 6)

/****************************************************************************
 msgTrackUpdate					takes P_TRACK_METRICS, returns STATUS
	category: client notification
	Sent by clsTrack to metrics.client when the pen moves if style.update is
	true.

*/
#define msgTrackUpdate			MakeMsg(clsTrack, 7)

/****************************************************************************
 msgTrackProvideMetrics			takes P_TRACK_METRICS, returns STATUS
	category: third-party notification
	Sent to a tracker client before tracker is created.

 Before it sends msgNew to clsTrack, code creating a tracker may choose to 
 send out this message to another object, allowing it to modify the tracker
 metrics.  See frame.h for a sample response to msgTrackProvideMetrics.
*/
#define msgTrackProvideMetrics	MsgNoError(MakeMsg(clsTrack, 9))


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *									Self-sent Messages						   			*
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgTrackConstrain				takes P_XY32, returns STATUS
	category: self-sent
	Constrains a point.

 If style.track is tsTrackMove, a new value for metrics.keepRect is computed
 based on the offset from metrics.origXY to pArgs.  pArgs is altered to
 insure the new keepRect lies within metrics.constrainRect.

 If style.track is tsTrackResize, a new value for metrics.rect is computed
 based on the offset from metrics.origXY to pArgs.  pArgs is altered to
 insure the new rect.size lies within metrics.maxWH and metrics.minWH.
*/
#define msgTrackConstrain		MakeMsg(clsTrack, 8)

/****************************************************************************
 msgTrackShow					takes P_TRACK_METRICS, returns STATUS
	category: self-sent
	Displays the tracker visuals at pArgs->rect.

 clsTrack will self-send this message when the tracker needs to be displayed.
*/
#define msgTrackShow			MakeMsg(clsTrack, 10)

/****************************************************************************
 msgTrackHide					takes P_TRACK_METRICS, returns STATUS
	category: self-sent
	Removes the tracker visuals at pArgs->rect.

 clsTrack will self-send this message when the tracker needs to be erased.
*/
#define msgTrackHide			MakeMsg(clsTrack, 11)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from other classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgInputEvent 					takes P_INPUT_EVENT, returns STATUS
	Notification of an input event. 

 clsTrack will respond to input events by updating and/or terminating
 the tracker.

 If pArgs->devCode is not one of msgPenMoveDown, msgPenUp, or
 msgPenOutProxDown stsInputGrabTerminate is returned.

 The new point is constrained by self-sending msgTrackConstrain.
 The new value for metrics.rect and metrics.curXY is computed based on
 the constrained pArgs->xy.

 If pArgs->devCode is msgPenUp or msgPenOutProxDown, clsTrack does the
 following:
	-:  send msgTrackDone(&metrics) to metrics.client
	-:  self-send msgTrackHide to remove the old tracker visuals
	-:  if style.autoDestroy is true, self-send msgDestroy(pNull)

 If pArgs->devCode is msgPenMoveDown, and the constrained version of
 pArgs->xy is different from metrics.curXY, clsTrack does the following:
	-:  if style.update is true, send msgTrackUpdate(&metrics) to metrics.client
	-:  self-send msgTrackHide to remove the old tracker visuals
	-:  self-send msgTrackShow to paint the new tracker visuals
*/

#endif		// TRACK_INCLUDED
