/****************************************************************************
 File: view.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.22  $
   $Author:   pstahl  $
     $Date:   03 Mar 1992 11:46:08  $

 This file contains the API definition for clsView.

 clsView inherits from clsCustomLayout.

 clsView is an abstract class that defines an association between a data
 object and a view onto that data.

 Since clsView is an abstract class it should never be directly instantiated.
****************************************************************************/
#ifndef VIEW_INCLUDED
#define VIEW_INCLUDED

#ifndef CLAYOUT_INCLUDED
#include <clayout.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT VIEW, *P_VIEW;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew     takes P_VIEW_NEW, returns STATUS
	category: class message
	Creates a new view.

 If pArgs->view.dataObject is non-null, the new view object becomes an 
 observer of the data object.

 clsView does not act on the value of pArgs->view.createDataObject.
 It is up to descendants of clsView to act on this field, typically by
 creating a new data object if the field is true.  This behavior may
 not be appropriate of all descendants, however.

 Descendants: You should never handle msgNew directly. Instead, handle
 msgInit by initializing your instance data. The ancestor must be called
 before your msgInit handler.
*/
typedef struct VIEW_NEW_ONLY {
	OBJECT		dataObject;			// Data object to view.
	BOOLEAN		createDataObject;	// Auto-create data object?
} VIEW_NEW_ONLY, *P_VIEW_NEW_ONLY;

#define	viewNewFields			\
	customLayoutNewFields		\
	VIEW_NEW_ONLY				view;

typedef struct VIEW_NEW {
	viewNewFields
} VIEW_NEW, *P_VIEW_NEW;

/****************************************************************************
 msgNewDefaults     takes P_VIEW_NEW, returns STATUS
	category: class message
	Initializes the VIEW_NEW structure to default values.

 In response to this message, clsView does the following:
//{
	pArgs->embeddedWin.style.embeddor	= true;
	pArgs->embeddedWin.style.embeddee	= true;
	pArgs->view.dataObject				= objNull;
	pArgs->view.createDataObject		= false;
//}

 Descendants: You should handle msgNewDefaults by initializing your _NEW 
 structure to default values. The ancestor must be called before your
 handler.
*/

/****************************************************************************
 msgFree	takes OBJ_KEY, returns STATUS
	Defined in clsmgr.h.

 In addition to standard msgFree behavior, the view removes itself as an
 observer of its data object.  It does NOT send msgFree to the data object.

 Descendants: You should handle msgFree by destroying all objects and
 resources you have created. It may be appropriate for you to destroy the
 data object if your view is the only observer of it.  The ancestor must
 be called after your handler.
*/

/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Defined in clsmgr.h.
 
 In response to this message, the view sends msgResPutObject to pArgs->file
 with the data object as the value of pArgs.  In effect, this means that
 saving the view also saves the data object.  (If the data object is null,
 this writes the "null object" id into the resource file.)

 Descendants: You should handle msgSave by saving your instance data.
 The ancestor must be called before your handler.
*/

/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Defined in clsmgr.h.

 In response to this message, the view sends msgResGetObject to
 pArgs->file.  In effect, this means that restoring the view also restores
 the data object.  (If the data object was null when the view was saved, the
 data object is null after msgRestore is handled.)

 If the restored data object is non-null, the view becomes an observer of
 the data object.

 Descendants: You should handle msgSave by restoring your instance data.
 The ancestor must be called before your handler.
*/

/****************************************************************************
 msgFreePending	takes OBJECT, returns STATUS
	Defined in clsmgr.h.

 If the object being freed is the view's data object, the view sets its
 data object to objNull.

 Descendants: If you maintain instance data on the data object, you may
 need to handle this message by updating your instance data to reflect
 the impending destruction of the data object.  The ancestor should be
 called before your handler.  It is recommended, however, that your view
 not keep any information on the data object, thus maintaining a strict
 view/data separation.  In such cases, you will not need to handle
 msgFreePending.
*/

/****************************************************************************
 msgViewSetDataObject	takes OBJECT, returns STATUS
	Specifies a view's data object.

 If the current data object is non-null, the view removes itself as an
 observer of the current data object.  It then sets the current data object
 to pArgs and, if the new data object is non-null, becomes an observer of it.

 Descendants: If you maintain instance data on the data object, you may
 need to handle this message by updating your instance data to reflect
 the changed data object.  The ancestor may be called before or after
 your handler.  It is recommended, however, that your view not keep
 any information on the data object, thus maintaining a strict
 view/data separation.  In such cases, you will not need to handle
 msgViewSetDataObject.
*/
#define msgViewSetDataObject 				MakeMsg(clsView, 1)

/****************************************************************************
 msgViewGetDataObject	takes P_OBJECT, returns STATUS
	Passes back a view's current data object

 Descendants: You do not normally handle this message.    
*/
#define msgViewGetDataObject 				MakeMsg(clsView, 2)

#endif	// VIEW_INCLUDED
