/****************************************************************************
 File: bytebuf.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.13  $
   $Author:   kguzik  $
     $Date:   24 Feb 1992 19:34:38  $

 This file contains the API definition for clsByteBuf.

 clsByteBuf inherits from clsObject.

 clsByteBuf provides a facility to store uninterpreted byte strings. Each
 object of clsByteBuf stores a single buffer. This class provides
 convenient object filing of the buffer data. Storage for each object's
 buffer is allocated out of the creator's shared process heap using
 OSHeapBlockAlloc.

 Clients who want to store null terminated strings should use clsString
 (see strobj.h).
 
****************************************************************************/
#ifndef BYTEBUF_INCLUDED
#define BYTEBUF_INCLUDED

#include <go.h>
#include <clsmgr.h>

typedef OBJECT BYTEBUF, *P_BYTEBUF;

typedef struct BYTEBUF_DATA {
	U16		bufLen;		// In/Out: Length (in bytes) of the pBuf buffer.
	P_U8	pBuf;		// In/Out: Object buffer.
} BYTEBUF_DATA, *P_BYTEBUF_DATA;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Class Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_BYTEBUF_NEW, returns STATUS
	category: class message
	Creates a new buffer object. 

 This message allocates shared heap storage for the specified buffer.

 allowObservers indicates whether the object will send the clsByteBuf
 observer messages (See msgByteBufChanged). Only clsByteBuf messages are
 affected by this option. Adding and removing observers is not affected by
 this option.
*/

typedef struct BYTEBUF_NEW_ONLY {
	BOOLEAN			allowObservers;	// In:	Send clsByteBuf observer messages
									//		to the object's observers?
	BYTEBUF_DATA	data;			// In/Out: Buffer data.
} BYTEBUF_NEW_ONLY, *P_BYTEBUF_NEW_ONLY;

#define byteBufNewFields		\
	objectNewFields				\
	BYTEBUF_NEW_ONLY	bytebuf;

typedef struct BYTEBUF_NEW {
	byteBufNewFields
} BYTEBUF_NEW, *P_BYTEBUF_NEW;

/****************************************************************************
 msgNewDefaults		takes P_BYTEBUF_NEW, returns STATUS
	category: class message
	Initializes the BYTEBUF_NEW structure to default values.

 Sets
 //{
	pNew->bytebuf.allowObservers	= true;
	pNew->bytebuf.data.bufLen		= 0;
	pNew->bytebuf.data.pBuf			= pNull;
 //}

 allowObservers indicates whether the object will send the clsByteBuf
 observer messages. (See msgByteBufChanged)
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Object Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgByteBufGetBuf	takes P_BYTEBUF_DATA, returns STATUS
	Passes back the object's buffer.

 The pointer passed back references the object's global storage. Clients
 must not modify or free this storage.
*/
#define msgByteBufGetBuf	 				MakeMsg(clsByteBuf, 1)

/****************************************************************************
 msgByteBufSetBuf	takes P_BYTEBUF_DATA, returns STATUS
	Copies the specified buffer data into the object's buffer.

 Previously retrieved bytebuf pointers will be invalid after this operation.
 Clients must call msgByteBufGetBuf to retrieve a pointer to the valid
 object buffer.
*/
#define msgByteBufSetBuf	 				MakeMsg(clsByteBuf, 2)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Observer Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgByteBufChanged	takes OBJECT, returns nothing
	category: observer notification
	Sent to observers when the object data changes. 

 The message argument is the UID of the clsByteBuf object that changed.

 This message is not sent if the creator did not specify allowObservers
 during msgNew.
*/
#define msgByteBufChanged 					MakeMsg(clsByteBuf, 3)

#endif
