/****************************************************************************
 File: clayout.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.48  $
   $Author:   kcatlin  $
     $Date:   17 Mar 1992 14:38:26  $

 This file contains the API definition for clsCustomLayout.

 clsCustomLayout inherits from clsBorder.
 Provides container windows which position and size their child windows
 according to complex constraints you specify for each child.

 See Also
	clsTableLayout

****************************************************************************/

/**** Debugging Flags ****/
/*
 The clsCustomLayout debugging flag is 'W'.  Defined values are:

    flag1  (0x0002):    msgWinLayoutSelf info

 You can also set the '%' flag to:

    flag8  (0x0100):    layout timing
*/

#ifndef CLAYOUT_INCLUDED
#define CLAYOUT_INCLUDED

													#ifndef BORDER_INCLUDED
#include <border.h>
													#endif
													#ifndef _STRING_H_INCLUDED
#include <string.h>
													#endif

/* * * * * * * * *   Common #defines and typedefs   * * * * * * * * * * * */

typedef OBJECT CSTM_LAYOUT;

typedef struct CSTM_LAYOUT_STYLE {
	U16	limitToRootWin	: 1;		// limit bounds to stay within theRootWindow
	U16	spare			: 15;		// unused (reserved)
} CSTM_LAYOUT_STYLE, *P_CSTM_LAYOUT_STYLE;

/*
 Default CSTM_LAYOUT_STYLE:
//{
	limitToRootWin		= false
//}
 */

typedef struct CSTM_LAYOUT_METRICS {
	CSTM_LAYOUT_STYLE	style;	// overall style
	U32					spare1;	// unused (reserved)
	U32					spare2;	// unused (reserved)
} CSTM_LAYOUT_METRICS, *P_CSTM_LAYOUT_METRICS;

/*
 Constraints for Custom layout.  For each of these,
 relWin of pNull and relWinTag of zero maps to parent.
*/
Enum16(CSTM_LAYOUT_CONSTRAINT) {
	// for x, y, width, height
	clAsIs			= 0,	// x, y: leave unchanged; w, h: use desired size
	clAbsolute		= 1,	// use absolute value specified in spec

	// for x, y, width, height: all relative to relWin
	clBefore		= 2,	// clBefore clMinEdge is one pixel less than 
							// the border rect; clBefore clMaxEdge is
							// on the border inner rect 

	clSameAs		= 3,	// same as relWin

	clAfter			= 4,	// clAfter clMaxEdge is one pixel after max edge
							// clAfter clMinEdge is on the border inner rect 

	clPctOf			= 5		// value * relWin / 100
};

/* possible edge specifications */
#define	clMinEdge		0	// x: left edge, y: bottom edge
#define	clCenterEdge	1	// x, y: mid point
#define	clMaxEdge		2	// x: right edge, y: top edge
#define	clBaselineEdge	3	// x: horiz. baseline, y: vertical baseline

/* macro for defining an x or y constraint to align two edges */
#define	ClAlign(childEdge, constraint, relWinEdge)	\
	( ((childEdge) << 6) | ((relWinEdge) << 4) | (constraint) )

/* macro for defining a w or h constraint to extend to an edge */
#define	ClExtend(constraint, relWinEdge)	\
	ClAlign(clMaxEdge, constraint, relWinEdge)

/*
 * can be or'ed into any constraint (except clAsIs or clAbsolute) to
 * refer to opposite dimension.
 */
#define	clOpposite		flag8

/*
 * can be or'ed into any constraint to compute new value as
 * Max(as-is value, constraint-computed value)
 * useful for children which need to be at least their desired
 * size, but can be bigger (e.g. extend to parent's edge)
 */
#define	clAtLeastAsIs	flag9

/*
 * can be or'ed into any constraint to compute new value as
 * specified constraint or clAsIs if the custom layout window
 * has wsShrinkWrapWidth/Height on.  This allows a child to be
 * shrink wrapped around if the custom layout window is computing
 * its own size; or, for example, have the child's width extend to
 * the edge of the parent if the parent is forced to a bigger size.
 */
#define	clAsIsIfShrinkWrap	flag10

/*
 * can be or'ed into width or height constraint to
 * exclude a child's width or height from the shrink-wrap computation.
 * This is useful for children which align to parent's max edge and overlap
 * other children.
 */
#define	clShrinkWrapExclude	flag11

/* macros to extract the parts of a constraint */
#define	ClChildEdge(c)		(((c) >> 6) & 0x3)
#define	ClRelWinEdge(c)		(((c) >> 4) & 0x3)
#define	ClConstraint(c)		((c) & 0x7)

typedef struct CSTM_LAYOUT_DIMENSION {
	CSTM_LAYOUT_CONSTRAINT	constraint;
	S32						value;				// offset or absolute value
	WIN						relWin;				// relative window
	U32						relWinTag;			// tag of relative window
	U16						valueUnits	: 6,	// units for value (e.g. bsUnitsLayout)
							spare1		: 10;	// unused (reserved)
	U32						spare;				// unused (reserved)
} CSTM_LAYOUT_DIMENSION, *P_CSTM_LAYOUT_DIMENSION;

typedef struct CSTM_LAYOUT_SPEC {
	CSTM_LAYOUT_DIMENSION	x, y, w, h;
} CSTM_LAYOUT_SPEC, *P_CSTM_LAYOUT_SPEC;

typedef struct CSTM_LAYOUT_CHILD_SPEC {
	WIN					child;
	CSTM_LAYOUT_SPEC	metrics;
	BOOLEAN				parentShrinkWrapWidth;
	BOOLEAN				parentShrinkWrapHeight;
	U32					spare;		// unused (reserved)
} CSTM_LAYOUT_CHILD_SPEC, *P_CSTM_LAYOUT_CHILD_SPEC;


/****************************************************************************
 msgNew		takes P_CSTM_LAYOUT_NEW, returns STATUS
	category: class message
	Creates a custom layout window.

*/
typedef CSTM_LAYOUT_METRICS CSTM_LAYOUT_NEW_ONLY, *P_CSTM_LAYOUT_NEW_ONLY;

#define	customLayoutNewFields		\
	borderNewFields					\
	CSTM_LAYOUT_NEW_ONLY	customLayout;

typedef struct CSTM_LAYOUT_NEW {
	customLayoutNewFields
} CSTM_LAYOUT_NEW, *P_CSTM_LAYOUT_NEW;

/****************************************************************************
 msgNewDefaults				takes P_CSTM_LAYOUT_NEW, returns STATUS
	category: class message
	Initializes the CSTM_LAYOUT_NEW structure to default values.

 Zeroes out pNew->customLayout.
*/

/****************************************************************************
 msgCstmLayoutGetMetrics		takes P_CSTM_LAYOUT_METRICS, returns STATUS
	Passes back the current metrics.

*/
#define msgCstmLayoutGetMetrics		MakeMsg(clsCustomLayout, 1)

/****************************************************************************
 msgCstmLayoutSetMetrics		takes P_CSTM_LAYOUT_METRICS, returns STATUS
	Sets the current metrics.

 If style.limitToRootWin is changed, msgWinSetLayoutDirty(true) will be
 self-sent.
*/
#define msgCstmLayoutSetMetrics		MakeMsg(clsCustomLayout, 2)


/****************************************************************************
 msgCstmLayoutGetStyle		takes P_CSTM_LAYOUT_STYLE, returns STATUS
	Passes back current style values.

*/
#define msgCstmLayoutGetStyle	MakeMsg(clsCustomLayout, 5)

/****************************************************************************
 msgCstmLayoutSetStyle		takes P_CSTM_LAYOUT_STYLE, returns STATUS
	Sets style values.

 If style.limitToRootWin is changed, msgWinSetLayoutDirty(true) will be
 self-sent.
*/
#define msgCstmLayoutSetStyle	MakeMsg(clsCustomLayout, 6)


/****************************************************************************
 CstmLayoutSpecInit		returns VOID
	Zeros the P_CSTM_LAYOUT_SPEC.

 This is equivalent to the following:
	x, y, w, h: constraint = clAsIs

 You should use CustmLayoutSpecInit to initialize the layout spec that you
 pass in to msgCstmLayoutSetChildSpec.  For example:

	CSTM_LAYOUT_CHILD_SPEC	cs;
 	
	CstmLayoutSpecInit(&cs.metrics);
	cs.child = child;
	cs.metrics.x.constraint	= ClAlign(clMinEdge, clSameAs, clMinEdge);
	cs.metrics.y.constraint	= ClAlign(clMinEdge, clSameAs, clMinEdge);
	ObjCallRet(msgCstmLayoutSetChildSpec, clayout, &cs, s);
*/
#define	CstmLayoutSpecInit(lm)	memset((lm), 0, sizeof(CSTM_LAYOUT_SPEC))


/****************************************************************************
 msgCstmLayoutSetChildSpec		takes P_CSTM_LAYOUT_CHILD_SPEC, returns STATUS
	Sets layout spec for given child.

 Storage will be allocated for the spec.  The child spcecification will
 be used in response to msgCstmLayoutGetChildSpec, which is self-sent during
 msgWinLayoutSelf.

 clsCustomLayout will self-send msgWinSetLayoutDirty(true).

 See Also
 	CstmLayoutSpecInit
	msgCstmLayoutGetChildSpec
	msgCstmLayoutRemoveChildSpec
	msgWinLayoutSelf
*/
#define msgCstmLayoutSetChildSpec 	MakeMsg(clsCustomLayout, 3)

/****************************************************************************
 msgCstmLayoutRemoveChildSpec		takes WIN, returns STATUS
	Removes the spec for the specified child (pArgs).

 If a child is extracted or destroyed, and msgCstmLayoutSetChildSpec was
 used to set the child layout constraints, you must use this message to 
 remove the child layout constraints.

 See Also
 	msgCstmLayoutSetChildSpec
*/
#define msgCstmLayoutRemoveChildSpec 	MakeMsg(clsCustomLayout, 7)

/****************************************************************************
 msgCstmLayoutGetChildSpec		takes P_CSTM_LAYOUT_CHILD_SPEC, returns STATUS
	category: self-sent, subclass responsibility
	Passes back the current spec for the specified child.

 Self-sent during msgWinLayout to retrieve the current spec from subclasses.
 clsCustomLayout responds by returning the stored spec, or an initialized
 spec (CstmLayoutSpecInit()) if none is found.

 Subclasses can catch this message, look at pArgs->child and return the
 layout constraints for known children.

 If pArgs->relWin is not objNull, this uid will be used as the relative
 window.  Otherwise, if pArgs->relWinTag will be used to find the relative
 window (i.e. relWinTag should be the window tag of the relative window).
 The relative window must be objNull (in which case the parent is used) or
 a sibling of pArgs->child.
*/
#define msgCstmLayoutGetChildSpec 	MakeMsg(clsCustomLayout, 4)

/* status values */
#define	stsCstmLayoutBadRelWin		MakeStatus(clsCustomLayout, 1)
#define	stsCstmLayoutBadRelWinTag	MakeStatus(clsCustomLayout, 4)
#define	stsCstmLayoutLoop		  	MakeStatus(clsCustomLayout, 2)
#define	stsCstmLayoutBadConstraint 	MakeStatus(clsCustomLayout, 3)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from other classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes an object to file itself in an object file.

 clsCustomLayout will save the constraints for each child that has wsSendFile
 on in its WIN_METRICS.flags.style.  If a child's constraint specifies a
 relWin that does not file, the relWin will be filed as objNull.
*/

/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.
	
 clsCustomLayout will restore the constraints for each child that was filed.

 clsCustomLayout will self-send msgWinSetLayoutDirty(true) if the system font
 or system font scale changed since the table was filed.  pArgs->pEnv is
 cast to a P_WIN_RESTORE_ENV and must be a valid window environment pointer.
*/

/****************************************************************************
 msgWinLayoutSelf takes P_WIN_METRICS, returns STATUS
    Tell a window to layout its children (sent during layout).

 clsCustomLayout responds by laying out its children.  For each child,
 the following is done:
	- msgCstmLayoutGetChildSpec is self-sent with the following
	  CSTM_LAYOUT_CHILD_SPEC parameters:
	  	child					= child to be layed out;
		metrics					= result of CstmLayoutSpecInit();
		parentShrinkWrapWidth	= true if self can shrink wrap width;
		parentShrinkWrapWidth	= true if self can shrink wrap height;

	  Self can shrink wrap width/height if pArgs->options has wsLayoutResize
	  on and self's WinShrinkWrapWidth/Height(WIN_METRICS.flags.style) is
	  true.
		
	  The passed-back pArgs will be used as the child's layout spec.

	- msgBorderGetOuterOffsets is sent to the child with a default pArgs
	  (RECT32) of (1, 1, 1, 1).  The outer offsets are used to define
	  "after min edge" or "before max edge" constraints.

	- The x, y, w, h of the child is computed based on its constraints.
	  If the either w or h constraints are clAsIs, msgWinGetDesiredSize is
	  sent to the child to determine its desired size.
	  
 If pArgs->options has wsLayoutResize on and self has shrink wrap width/height
 on, the bounding box around the layed out children will be computed and
 passed back in pArgs->bounds.size.  If style.limitToRootWin is true, and
 self has no parent or self's parent is theRootWindow, the computed size
 will be limited to insure that self will fit on theRootWindow and self's
 origin may be altered (via msgWinDelta) to insure the window is fully on
 screen.

 Return Value
	stsCstmLayoutBadRelWin:  The relWin specified for a child spec was
		not the uid of a sibling window.

	stsCstmLayoutBadRelWinTag:  The relWinTag specified for a child spec was
		not the tag of a sibling window.

	stsCstmLayoutLoop:  The specified set of child constraints results in
		a circular layout loop.  For example, child A's width clSameAs child
		B's width and child B's width clSameAs child A's width.

	stsCstmLayoutBadConstraint:  A constraint specified for a child is not
		a valid value.
*/

#endif	// CLAYOUT_INCLUDED
