/****************************************************************************
 dvalap.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The public definition of the Apple Link Access Protocol device interface.

 $Revision:   1.10  $
   $Author:   unknown  $
     $Date:   25 Mar 1992 16:39:36  $
****************************************************************************/
#ifndef DVALAP_INCLUDED
#define DVALAP_INCLUDED

#ifndef MIL_INCLUDED
#include <mil.h>
#endif
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the ALAP device. 
 The definitions contained here are data structures that are necessary
 for the MIL ALAP specific functions along with standard
 MIL device data structures with additional fields specific to the
 ALAP device.
*/

/**** ALAP Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the ALAP device. The function numbers start immediately after
 the last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.

*/

#define milAlapGetDevParameters		milLastCommonFunction+1
#define milAlapSetDevParameters		milLastCommonFunction+2
#define milAlapWaitCarrierSense		milLastCommonFunction+3
#define milAlapSetBaudRate			milLastCommonFunction+4
#define milAlapStartTxRx			milLastCommonFunction+5
#define milAlapInterrupt			milLastCommonFunction+6

#define fnCountAlap					fnCountBase+6

/**** ALAP Device Function Transfer Table ****/
/*
 This structure is the declaration of the ALAP Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for the MIL ALAP device.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/


typedef struct MIL_FTT_ALAP
{
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountAlap];
} MIL_FTT_ALAP, *P_MIL_FTT_ALAP;



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 					ALAP Device Specific Functions	   	 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are the ALAP device specific
 functions. Access to these functions is provided by the FTT defined
 above. All of the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/


/****************************************************************************
 milAlapGetDevParameters		returns STATUS
 	Provides device specific information about the ALAP device.

 This function is a single stage function that provides device
 specific information about the ALAP device.

*/

/**** Device Specific Request Block Fields ****/
/*
 The milAlapGetDevParameters request block contains the following
 fields.

	address:
		This is the ALAP address for this device.

*/

typedef struct MIL_ALAP_DEVICE_PARAMETERS
{
	U16	address;
} MIL_ALAP_DEVICE_PARAMETERS, *P_MIL_ALAP_DEVICE_PARAMETERS;

#define rbPublicFieldsAlapDeviceParameters	\
	MIL_ALAP_DEVICE_PARAMETERS		alapDeviceParameters;

/*
 
 milAlapGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_ALAP_DEVICE_PARAMETERS
{
	milRequestBlockCommonFields
	rbPublicFieldsAlapDeviceParameters
} MIL_RB_PUB_ALAP_DEVICE_PARAMETERS, *P_MIL_RB_PUB_ALAP_DEVICE_PARAMETERS;


/****************************************************************************
 milAlapSetDevParameters		returns STATUS
 	Sets device specific information for the ALAP device.

 This function is a single stage function that sets device
 specific information for the ALAP device.

*/

/**** Device Specific Request Block Fields ****/
/*
 The milAlapSetDevParameters request block is the same as the 
 milAlapGetDevParameters request block defined above.

*/

/****************************************************************************
 milAlapWaitCarrierSense		returns STATUS
 	Returns current carrier detect status.

 This function is a single stage function that returns 
 the current carrier detect status of the Alap device.

*/
/**** Function Specific Definitions ****/
/*
 Wait Times

 The following definition contains the defined wait times.

	milAlapblipWait:
		This value indicates the wait is the time for 1 blip (2 baud
		intervals).

	milAlapflagWait:
		This value indicates the wait is for 8 clocks.

	milAlapspeedSwitchWait:
		This value indicates the wait is the time it takes to switch
		speeds

	milAlapflashFlagWait:
		This value indicates the wait is the time for a flash (8 clocks)
		at high speed.

	milAlaptwelveOnesWait:
		This indicates the wait is the time to send 12 ones.

	milAlapWait100us:
		This value indicates the wait is 100 us.

*/

Enum16(MIL_ALAP_WAIT_TIMES)
{
	milAlapblipWait,
	milAlapflagWait,
	milAlapspeedSwitchWait,
	milAlapflashFlagWait,
	milAlaptwelveOnesWait,
	milAlapWait100us
};

/*
 Wait Flags

 The following flags are defined for the milAlapWaitCarrierSense
 function.

	milAlapCarrierSense:
		This value indicates the wait should wait the specified
		time or until a carrier is sensed.

	milAlapWaitNoCarrier:
		This value indicates the wait should be for the full time
		period.

	milAlapWaitCarrierOff:
		This value indicates the wait should be the time indicated
		or until the carrier drops which ever is sooner

	milAlapWaitCarrierOn:
		This value indicates the wait should be the time indicated
		or until the carrier is on which ever is sooner
		
*/

#define milAlapCarrierSense			0
#define milAlapWaitNoCarrier		1
#define milAlapWaitCarrierOff		2
#define milAlapWaitCarrierOn		3


/**** Device Specific Request Block Fields ****/
/*
 The milAlapWaitCarrierSense request block contains the following
 fields.

	flags:
		This field identifies the type of wait.

	time:
		This field specifies the time interval to wait.
	
	multiplier:
		This field allows a wait of multiple intervals.

	carrierStatus:
		This field is a return value that indicates the carrier 
		status.

*/


typedef struct MIL_RB_ALAP_WAIT_CARRIER_SENSE
{
	U16						flags;
	MIL_ALAP_WAIT_TIMES	time;
	U16						multiplier;
	U16						carrierStatus;
} MIL_ALAP_WAIT_CARRIER_SENSE, *P_MIL_ALAP_WAIT_CARRIER_SENSE;

#define rbPublicFieldsAlapWaitCarrierSense	\
	MIL_ALAP_WAIT_CARRIER_SENSE	alapWaitCarrierSense;	

/*
 
 milAlapWaitCarrierSense Request Block

*/

typedef struct MIL_RB_PUB_ALAP_WAIT_CARRIER_SENSE
{
	milRequestBlockCommonFields
	rbPublicFieldsAlapWaitCarrierSense
} MIL_RB_PUB_ALAP_WAIT_CARRIER_SENSE, *P_MIL_RB_PUB_ALAP_WAIT_CARRIER_SENSE;

/****************************************************************************
 milAlapSetBaudRate		returns STATUS
 	Sets the ALAP device baud rate.

 This function is a single stage function that is used to set the 
 baud rate for the Alap device. For FlashTalk the two supported
 rates are 230,400 and 768,000 bits per second.

*/

/**** Device Specific Request Block Fields ****/
/*
 The milAlapSetBaudRate request block contains the following
 fields.

	baudRate:
		This field indicates the baud rate to be set. It is 
		in bits per second.

*/
typedef struct MIL_ALAP_SET_BAUD_RATE
{
	U32 baudRate;
} MIL_ALAP_SET_BAUD_RATE, *P_MIL_ALAP_SET_BAUD_RATE;

#define rbPublicFieldsAlapSetBaudRate \
	MIL_ALAP_SET_BAUD_RATE		alapSetBaudRate;

/*
 
 milAlapSetBaudRate Request Block

*/

typedef struct MIL_RB_PUB_ALAP_SET_BAUD_RATE
{
	milRequestBlockCommonFields
	rbPublicFieldsAlapSetBaudRate
} MIL_RB_PUB_ALAP_SET_BAUD_RATE, *P_MIL_RB_PUB_ALAP_SET_BAUD_RATE;



/****************************************************************************
 milAlapDiagnostic		returns STATUS
 	Performs a diagnostic test of the ALAP device.

 This single stage function requests that a diagnostic test be
 performed on the Alap device.

*/
/**** Device Specific Request Block Fields ****/
/*
 The milAlapDiagnostic request block does not contain any function
 specific fields.

*/

#define rbPublicFieldsAlapDiagnostic

/*
 
 milAlapDiagnostic  Request Block

*/

typedef struct MIL_RB_PUB_ALAP_DIAGNOSTIC
{
	milRequestBlockCommonFields
	rbPublicFieldsAlapDiagnostic
} MIL_RB_PUB_ALAP_DIAGNOSTIC, *P_MIL_RB_PUB_ALAP_DIAGNOSTIC;


/****************************************************************************
 milAlapDevInit		returns STATUS
 	Performs initialization on the ALAP device.

 This single stage function performs a device initialization on the
 Alap device.

*/
/**** Device Specific Request Block Fields ****/
/*
 The milAlapDevInit request block contains the following fields.

	port:

*/

typedef struct MIL_ALAP_DEVICE_INIT
{
	U16	port;
} MIL_ALAP_DEVICE_INIT, *P_MIL_ALAP_DEVICE_INIT;

#define rbPublicFieldsAlapDeviceInit	\
	MIL_ALAP_DEVICE_INIT		alapDeviceInit;

/*
 
 milAlapDevInit Request Block

*/

typedef struct MIL_RB_PUB_ALAP_DEVICE_INIT
{
	milRequestBlockCommonFields
	rbPublicFieldsAlapDeviceInit
} MIL_RB_PUB_ALAP_DEVICE_INIT, *P_MIL_RB_PUB_ALAP_DEVICE_INIT;



/****************************************************************************
 milAlapStartTxRx		returns STATUS
 	Initiates tansmit and receive operations

 This function initiates a the transmit and receiveing
 of LocalTalk frames.

*/

/**** Device Specific Request Block Fields ****/
/*
 The milAlapStartTxRx request block does not contain any function
 specific fields.

*/

/*
 
 milAlapStartTxRx  Request Block

*/

typedef struct MIL_RB_PUB_ALAP_START_TX_RX
{
	milRequestBlockCommonFields
} MIL_RB_PUB_ALAP_START_TX_RX, *P_MIL_RB_PUB_ALAP_START_TX_RX;



/****************************************************************************
 milAlapTransmitReceive		returns STATUS
 	Transmits or receives LocalTalk frames.

 This multi-stage function transmits and receives LocalTalk frames.

*/

/**** Function Specific Definitions ****/
/*
 Transmit and Receive Flags

 The following flags are defined for the transmit and receive flags.

	milAlapDoNothingFlag:
		This flag indicates there is nothing to do extra on a receive.

	milAlapReceiveFlag:
		This flag indicates a receive operation.

	milAlapSlowReceiveFlag:
		This flag indicates a receive is done in slow mode.

	milAlapFastReceiveFlag:
		This flag indicates a receive in fast mode.

	milAlapTransmitFlag:
		This flag indicates a transmit.

	milAlapSlowTransmitFlag:
		This flag indicates a transmit in slow mode.

	milAlapFastTransmitFlag:
		This flag indicates a transmit in fast mode.

	milAlapNoBlipTransmitFlag:
		This flag indicates to not send a blip.

	milAlapBlipTransmitFlag:
		This flag indicates to send a blip on a transmit.

*/ 

 
#define milAlapDoNothingFlag			0x00
#define milAlapReceiveFlag				0x01
#define milAlapSlowReceiveFlag			0x00
#define milAlapFastReceiveFlag			0x02
#define milAlapTransmitFlag				0x10
#define milAlapSlowTransmitFlag			0x00
#define milAlapFastTransmitFlag			0x20
#define milAlapNoBlipTransmitFlag		0x00
#define milAlapBlipTransmitFlag			0x40

/*
 Frame Status Flags

 The following flags are defined for the frame status

	milAlapTransmitFrameOK:
		This flag indicates the frame was transmitted without errors.

	milAlapValidFrameReceived:
		This flag indicates the received frame was valid.

	milAlapReceivedAbort:
		This flag indicates the receive operation was aborted.

	milAlapReceivedBadCrc:
		This flag indicates the received frame did not pass the CRC check.

	milAlapReceiverOverRun:
		This flag indicates an overrun error was detected on a receive.

	milAlapReceivedShortFrame:
		This flag indicates a short frame was received.

	milAlapReceivedInvalidFrame:
		This flag indicates the received frame was invalid.


*/

#define milAlapTransmitFrameOK			0x01
#define milAlapValidFrameReceived		0x02
#define milAlapReceivedAbort			0x04
#define milAlapReceivedBadCrc			0x08
#define milAlapReceiverOverRun			0x10
#define milAlapReceivedShortFrame		0x20
#define milAlapReceivedInvalidFrame		0x40

/*
 Connect Status Flags

 The following flags are defined for the connect status.

	milTxNotConnected:
		This field indicates there is no connection.

	milTxNoInt:
		This field indicates there was no transmit interrupt.

	milTxOK:
		This field indicates the transmit was successful.

*/

#define milTxNotConnected				0x01
#define milTxNoInt						0x02
#define milTxOK							0x03


/**** Device Specific Request Block Fields ****/
/*

 The mil milAlapTransmitReceive funtion request block contains the
 following fields.

	flags:
		This field specifies active modes of operation for the ALAP
		device. The bits are defined above.

	connected:
		This field contains bits that indicate the current connection
		status. The bits are defined above.

	frameStatus:
		This field conatins bits that indicatr the status of the current
		frame.

	pTxBuffer:
  		This field is a pointer to the transmit buffer.

	txBufferLen
		This field is a count of the number of bytes in the transmit
		buffer.

	txBufferPhysAddress:
		This field is the physical addres of the transmit buffer.

	pRxBuffer:
		This field is a pointer to the receive buffer.

	rxBufferLen:	
		This field is a count of the number of bytes in the receive
		buffer.

	rxBufferPhysAddress:
		This field is the physical address of the receive buffer.

	rxLength:
		This the current receive length.

	pAlapEventFunc:
		This field contains a pointer to a call back function that
		will be called if an event occurs.

*/

typedef struct MIL_ALAP_TRANSMIT_RECEIVE
{
	U16							flags;
	U16							connected;
	U16							frameStatus;
	P_U8						pTxBuffer;
	U16							txBufferLen;
	U32							txBufferPhysAddress;
	P_U8						pRxBuffer;
	U16							rxBufferLen;
	U32							rxBufferPhysAddress;
	U16							rxLength;
	P_MIL_ASYNC_EVENT_FUNC		pAlapEventFunc;
} MIL_ALAP_TRANSMIT_RECEIVE, *P_MIL_ALAP_TRANSMIT_RECEIVE;

#define rbPublicFieldsAlapTransmitReceive	\
	MIL_ALAP_TRANSMIT_RECEIVE	alapTransmitReceive;

/*
 
 milAlapTransmitReceive Request Block

*/

typedef struct MIL_RB_PUB_ALAP_TRANSMIT_RECEIVE
{
	milRequestBlockCommonFields
	rbPublicFieldsAlapTransmitReceive
} MIL_RB_PUB_ALAP_TRANSMIT_RECEIVE, *P_MIL_RB_PUB_ALAP_TRANSMIT_RECEIVE;

#endif
