/****************************************************************************
 File: fs.h

 (C) Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.31  $
   $Author:   twiegman  $
	 $Date:   16 Mar 1992 12:57:44  $

 This file contains the API for clsDirHandle and clsFileHandle.
 The functions described in this file are contained in PENPOINT.LIB.

 clsFileSystem inherits from clsObject.
 Provides file system support.
 theFileSystem is the only instance of clsFileSystem.

 clsDirHandle inherits from clsObject.
 Provides file system directory support.
 theBootVolume is a well known instance of clsDirHandle.
 theSelectedVolume is a well known instance of clsDirHandle.
 theWorkingDir is a well known instance of clsDirHandle.

 clsFileHandle inherits from clsStream.
 Provides file system file access support.
****************************************************************************/
#ifndef FS_INCLUDED
#define FS_INCLUDED


/**** Debugging Flags ****/
/*
 FileSystem Debugging Flag is '$', values are:

	0001:	Debug info when fs cache layer calls volume layer
	0200:	Breaks into debugger before asking to insert disk
	20000:	Display list of known volumes when prompting for unmounted disk
*/

/* Include file dependencies for this include file */

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef UID_INCLUDED
#include <uid.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef UUID_INCLUDED
#include <uuid.h>
#endif

#ifndef STREAM_INCLUDED
#include <stream.h>
#endif

/*
 Common abbreviations, terms:

	FS:		File System
	Node:	A file or a directory
	Dir:	A directory

 Rules concerning the destination of file system messages:

 All messages defined in this file are directed to their destination via
 ObjectCall, the file system does not accept messages that are sent.  All
 messages (with the exception of msgFSGetInstalledVolumes) of clsFileSystem
 can be "sent" to either a file or a dir object.  Messages of clsDirHandle
 can only be "sent" to directory objects.  Messages of clsFileHandle can only
 be "sent" to file objects.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Defines */

#define fsMaxPathLength			254			// Max path length
											// (Excluding null terminator)
#define fsPathBufLength	(fsMaxPathLength+1) // Buffer size for max path
#define fsSeparator				'\\'		// Pathname separator
#define	fsEscapeChar			'|'			// Escape char (invalid in paths)
#define fsUniqueSeparator		' '			// Char for unique name postfix
#define	fsMaxHandles			255			// Max handles on a single node
#define fsMaxUnique				255			// Max tries to make name unique
#define fsMaxReadWrite			0x40000000	// Max size for single read/write
#define fsMaxNestingLevel		20			// Max nesting for recursive ops


/****  FS Attribute Intrinsics	****/

/*
 These are used to build file/directory attribute labels or
 to get component pieces from an attribute label.

 A client can define their own attribute using one of the FSMakeXXXAttr
 intrinsics, specifying a class and a tag.	The attribute type will allow
 for storage of a 32 bit value (Fix32), a 64 bit value (Fix64), a null
 terminated string of any length up to 32K (Str), or a variable length
 value up to 32K (Var).	 The messages msgFSGetAttr, msgFSSetAttr,
 msgFSReadDir, msgFSReadDirFull and msgFSTraverse use file system
 attributes to represent the attribute label.
*/

#define fsFixAttr				0
#define fsFix64Attr				1
#define fsVarAttr				2
#define fsStrAttr				3

#define fsMaxAttrLength			255

//REFGEN BEGINIGNORE
/*
 Note: This must be broken into two lines because the gensyms
 utility treats defines followed by MakeTag... on the same line
 as a special case (creating problems during gen symbols)
*/
//REFGEN ENDIGNORE
#define FSMakeAttr(cls,t,f)		\
			MakeTagWithFlags(cls,t,f)
#define FSMakeFix32Attr(cls,t)	FSMakeAttr(cls,t,fsFixAttr)
#define FSMakeFix64Attr(cls,t)	FSMakeAttr(cls,t,fsFix64Attr)
#define FSMakeVarAttr(cls,t)	FSMakeAttr(cls,t,fsVarAttr)
#define FSMakeStrAttr(cls,t)	FSMakeAttr(cls,t,fsStrAttr)

#define FSAttr(attr)			TagNum(attr)
#define FSAttrCls(attr)			ClsNum(attr)

#define FSAttrIsFix32(attr)		(TagFlags(attr) == fsFixAttr)
#define FSAttrIsFix64(attr)		(TagFlags(attr) == fsFix64Attr)
#define FSAttrIsVar(attr)		(TagFlags(attr) == fsVarAttr)
#define FSAttrIsStr(attr)		(TagFlags(attr) == fsStrAttr)


/****  File System Attributes  ****/

/*
 These are the predefined attributes managed by the file system.
*/

#define fsNullAttrLabel			FSMakeFix32Attr(objNull,0)

#define fsAttrName				FSMakeStrAttr(clsFileSystem,0)

#define fsAttrFlags				FSMakeFix32Attr(clsFileSystem,0)
#define fsAttrDateCreated		FSMakeFix32Attr(clsFileSystem,2)
#define fsAttrDateModified		FSMakeFix32Attr(clsFileSystem,3)
#define fsAttrFileSize			FSMakeFix32Attr(clsFileSystem,4)

#define fsAttrDirIndex			FSMakeFix64Attr(clsDirHandle, 0)
#define fsAttrOldDirIndex		FSMakeFix64Attr(clsDirHandle, 1)

#define fsAttrFileType			FSMakeFix32Attr(clsFileHandle, 0)

/*
 See msgFSGetAttr for an explanation when to use these constants.
*/

#define fsAllocAttrLabelsBuffer	((P_FS_ATTR_LABEL)maxU32)
#define fsAllocAttrValuesBuffer	((P_UNKNOWN)maxU32)
#define fsAllocAttrSizesBuffer	((P_FS_ATTR_SIZE)maxU32)


/****  Status Codes	 ****/

/*
 Common return values:

 There are a few status return values that are common to either all messages
 or to a group of messages (i.e. messages that try to change the volume).

	stsFSHandleInvalid:		The dir/file object refers to a node that has
							been previously deleted.
	stsFSVolDisconnected:	The volume is not connected.
	stsFSVolFull:			The message cannot complete,
							due to insufficient space on the volume.
	stsFSVolReadOnly:		The message cannot complete,
							because the volume is write protected.
*/

/* Error Status Codes */

#define	 stsFSVolDisconnected	MakeStatus(clsFileSystem,0)
#define	 stsFSVolReadOnly		MakeStatus(clsFileSystem,1)
#define	 stsFSVolFull			MakeStatus(clsFileSystem,2)
#define	 stsFSNodeNotFound		MakeStatus(clsFileSystem,3)
#define	 stsFSNodeReadOnly		MakeStatus(clsFileSystem,4)
#define	 stsFSAccessDenied		MakeStatus(clsFileSystem,5)

#define	 stsFSCircularMoveCopy	MakeStatus(clsFileSystem,6)

#define	 stsFSVolBusy			MakeStatus(clsFileSystem,7)
#define	 stsFSNodeBusy			MakeStatus(clsFileSystem,8)
#define	 stsFSBadPath			MakeStatus(clsFileSystem,9)
#define	 stsFSUniqueFailed		MakeStatus(clsFileSystem,10)
#define	 stsFSDirFull			MakeStatus(clsFileSystem,11)

#define	 stsFSNodeExists		MakeStatus(clsFileSystem,12)
#define	 stsFSNotDir			MakeStatus(clsFileSystem,13)
#define	 stsFSNotFile			MakeStatus(clsFileSystem,14)

#define	 stsFSReadOnlyAttr		MakeStatus(clsFileSystem,15)
#define	 stsFSBufTooSmall		MakeStatus(clsFileSystem,16)
#define	 stsFSNestingTooDeep	MakeStatus(clsFileSystem,17)

#define	 stsFSNoParent			MakeStatus(clsFileSystem,18)
#define	 stsFSUnchangeable		MakeStatus(clsFileSystem,19)
#define	 stsFSNotAncestor		MakeStatus(clsFileSystem,20)
#define	 stsFSDirPositionLost	MakeStatus(clsFileSystem,21)

#define	 stsFSHandleInvalid		MakeStatus(clsFileSystem,22)
#define	 stsFSDifferent			MakeStatus(clsFileSystem,23)
#define	 stsFSTooManyHandles	MakeStatus(clsFileSystem,24)

#define	 stsFSDirIndexExists	MakeStatus(clsFileSystem,25)
#define	 stsFSDirIndexNotFound	MakeStatus(clsFileSystem,26)

#define	 stsFSVolCorrupt		MakeStatus(clsFileSystem,27)


/* Informational Status Codes */

#define	 stsFSAttrBufTooSmall	MakeWarning(clsFileSystem,1)


/****  Types  ****/

/*
 Locators are structures used to describe the location of a file or dir node.
 There are two types of locators: explicit and implicit.  An explicit
 locator is defined with FS_LOCATOR which specifies both the starting node
 (uid) and the path relative to the starting node (pPath).	An implicit
 locator is made up of a starting node (the object that receives a message)
 and the path relative to the starting node (pPath).  msgFSMove is a good
 example of a message that contains both types of locators.	 The receiver of
 msgFSMove and move.pSourcePath defines the implicit location of the source
 of the move.  move.destLocator defines the explicit location of the dest
 of the move.

 The uid field of a locator must be filled in and must be non-null.	 If no
 other choice can be decided upon, theWorkingDir may be a good one.
 The uid field does not always have to be a dir handle object.	The uid can
 be a file handle object if the pPath field points to a path that begins with
 .. (parent), \ (root) or \\ (fully specified path including volume name).

 The path field of locators (explicit and implicit) are relative to the node
 defined by the uid (or object receiving the message) unless the path begins
 with a \ (root relative) or \\ (fully specified path).
*/

typedef struct	FS_LOCATOR {
	OBJECT				uid;
	P_STRING			pPath;		// Relative to node defined by uid
} FS_LOCATOR, * P_FS_LOCATOR;

/*
 The file system interface never uses flat locators, but if it is more
 convenient to hold the entirety of the locator in a linear structure
 using flat locators.
*/

typedef struct	FS_FLAT_LOCATOR {
	OBJECT			uid;
	U8				path[fsPathBufLength];
} FS_FLAT_LOCATOR, * P_FS_FLAT_LOCATOR;

Enum16(FS_NODE_FLAGS) {
	fsNodeReadOnly			= flag0,	// Node is read-only.
	fsNodeHidden			= flag1,	// System hidden file.
	fsNodeDir				= flag4,	// Directory or file?
	fsNodeGoFormat			= flag8,	// Node has non-native attrs
	fsNodePenPointHidden	= flag9		// Should this node be hidden from
										// the user in Penpoint browsers?
};

#define validFSNodeFlags \
	(fsNodeReadOnly | fsNodeHidden | fsNodeDir | \
	 fsNodeGoFormat | fsNodePenPointHidden)

#define readOnlyFSFlags (fsNodeDir | fsNodeGoFormat)

/*
 FS_NODE_FLAGS_ATTR is used to set or get the flags attribute stored with
 a file/dir node.  When setting the flags, only those flags with a one in
 the mask word will be affected.  When getting flags, all flags are returned
 and mask is set to all ones (as a convenience for set after get).
*/

typedef struct FS_NODE_FLAGS_ATTR {
	FS_NODE_FLAGS		flags;
	U16					mask;
} FS_NODE_FLAGS_ATTR, * P_FS_NODE_FLAGS_ATTR;

typedef U32		FS_DATE_TIME, * P_FS_DATE_TIME;
typedef U32		FS_FILE_SIZE, * P_FS_FILE_SIZE;

typedef U16		FS_ATTR_SIZE,	* P_FS_ATTR_SIZE;
typedef U32		FS_ATTR_LABEL,	* P_FS_ATTR_LABEL;

Enum16(FS_VOL_TYPE) {
	fsAnyVolType		= 0,			// Match any vol type for msgNew
	fsVolTypeMemory		= 0,
	fsVolTypeDisk		= 1,
	fsVolTypeRemote		= 2
};

Enum16(FS_VOL_FLAGS) {
	fsVolReadOnly		= flag0,
	fsVolConnected		= flag1,
	fsVolRemovableMedia	= flag2,
	fsVolEjectableMedia	= flag3,
	fsVolDirsIndexable	= flag4,
	fsVolFormattable	= flag5,
	fsVolDuplicatable	= flag6
};

/*
 This information is returned by msgFSGetVolMetrics.
*/

typedef struct FS_VOL_HEADER {
	FS_VOL_TYPE			type;
	FS_VOL_FLAGS		flags;
	OBJECT				rootDir;
	OBJECT				volObj;
	U32					serialNum;
	U32					created;
	U16					optimalSize;
	U32					totalBytes;
	U32					freeBytes;
	U32					commSpeed;
	U8					pName [nameBufLength];
	U8					alignSpare;		// Word align following values
	CLASS				browserClass;	// Class of browser to use for volume
										// If null, use system default
	U32					nativeFS;
	RES_ID				iconResId;
	U32					spare1;
	U32					spare2;
	U32					spare3;
	U32					spare4;
} FS_VOL_HEADER, * P_FS_VOL_HEADER;

typedef FS_VOL_HEADER	FS_VOL_METRICS, * P_FS_VOL_METRICS;

Enum16(FS_EXIST) {
 // Lower byte: what to do if the node exists
	fsExistOpen			= 0,
	fsExistGenError		= 1,
	fsExistGenUnique	= 2,
	fsExistTruncate		= 3,
 // Upper byte: what to do if the node doesn't exist
	fsNoExistCreate		  = MakeU16(0, 0),
	fsNoExistGenError	  = MakeU16(0, 1),
	fsNoExistCreateUnique = MakeU16(0, 2),
 // Default setting
	fsExistDefault		= fsExistOpen | fsNoExistCreate
};

Enum16(FS_MOVE_COPY_EXIST) {
 // What to do if the destination node exists
	fsMoveCopyExistOverwrite	= 0,
	fsMoveCopyExistGenError		= 1,
	fsMoveCopyExistGenUnique	= 2,
	fsMoveCopyExistDelete		= 3,
 // Default setting
	fsMoveCopyExistDefault		= fsMoveCopyExistGenError
};

Enum16(FS_DIR_NEW_MODE) {
 // Delete directory at handle free time?
	fsTempDir			= flag0,
 // Is handle changeable?
	fsUnchangeable		= flag1,
 // Find node via its dir index?
	fsUseDirIndex		= flag2,
 // Disable prompts (insert disk, write protected, etc)
 //	fsDisablePrompts	= flag4, (Defined in FS_FILE_NEW_MODE below)
 // System owned dir handle - ring 0 only
	fsSystemDir			= flag7,
 // Default setting
	fsDirNewDefaultMode	= 0 // permanent, changeable directory
};

Enum16(FS_FILE_NEW_MODE) {
 // Lower byte: flags
 // Delete file at handle free time?
	fsTempFile			= flag0,
 // Read/write intentions for this handle
	fsReadOnly			= flag2,
 // Memory mapped files accessibility
	fsSharedMemoryMap	= flag3,
 // Disable prompts (insert disk, write protected, etc)
	fsDisablePrompts	= flag4,
 // System owned file handle - ring 0 only
	fsSystemFile		= flag7,
 // Upper byte: exclusivity requirements for other handles
	fsNoExclusivity		= MakeU16(0, 0),
	fsDenyWriters		= MakeU16(0, 1),
	fsExclusiveOnly		= MakeU16(0, 2),
 // Default setting
	fsFileNewDefaultMode= 0 // perm, read/write (noExclusivity)
};

Enum16(FS_GET_PATH_MODE) {
 // Get path relative to root, dir passed in, just name or vol and path
	fsGetPathRoot		= 0,
	fsGetPathRelative	= 1,
	fsGetPathName		= 2,
	fsGetPathAbsolute	= 3,
 // Default setting
	fsGetPathDefaultMode= fsGetPathRoot
};

Enum16(FS_MOVE_COPY_MODE) {
 // Use destination as container.
	fsMoveCopyIntoDest			= flag0,
 // Check but don't move or copy.
	fsMoveCopyVerifyOnly		= flag1,
 // Does source have live dir indexes.
	fsMoveCopySourceArchived	= flag2,
 // Does dest have live dir indexes.
	fsMoveCopyArchiveDest		= flag3,
 // Default setting
	fsMoveCopyDefaultMode	= 0
};

Enum16(FS_TRAVERSE_MODE) {
 // Call back on files?
	fsCallBackOnFiles	= flag0,
 // Call back before stepping into directory?
	fsCallBackPreDir	= flag1,
 // Call back after stepping into directory?
	fsCallBackPostDir	= flag2,
 // Default setting
	fsTraverseDefaultMode= fsCallBackOnFiles | fsCallBackPreDir
};

Enum16(FS_SEEK_MODE) {
 // Relative to beginning of file, end of file, or Current Byte Position
	fsSeekBeginning		= 0,
	fsSeekEnd			= 1,
	fsSeekCurrent		= 2,
 // Default setting
	fsSeekDefaultMode	= fsSeekBeginning
};

typedef OBJECT			DIR_HANDLE, * P_DIR_HANDLE;
typedef OBJECT			FILE_HANDLE, * P_FILE_HANDLE;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Class FileSystem Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgFSGetInstalledVolumes	takes P_LIST, returns STATUS
	Returns list of all installed volumes.

 This message can only be directed to the well known class theFileSystem.
 Each object in the list is a directory handle object that references the
 root node of the volume.  The list is passed back and is not used as an
 input parameter.  The caller must free the returned list when finished using
 it, but do not free any of the objects in the list.

 See Also
	msgFSEjectMedia:		to eject media from a floppy drive.
	msgFSGetVolMetrics:		to get more info about the volume
	msgFSSame:				to compare root dir to a well-known dir handle
*/
#define msgFSGetInstalledVolumes		MakeMsg(clsFileSystem, 21)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Class File System Messages understood by dirHandles and fileHandles	   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew						takes P_FS_NEW, returns STATUS
	category: class message
	Creates a directory or file handle object on a new or existing dir/file.

 The fields you commonly set are:
	pNew->fs.locator:	Location of the node
	pNew->fs.mode:		Options for opening file/dir handle
	pNew->fs.exist:		Action to take if the file/dir exists or doesn't exist

 Accessing a directory using a dirIndex:
 Three pieces of information must be provided to open a directory by dirIndex.
 The fsUseDirIndex flag must be set in new.fs.mode, a valid dirIndex must
 be supplied in new.fs.dirIndex and the volume that the directory resides on
 must be identified.  This can be done by specifying some location on the
 volume by filling in new.fs.locator.  Either the uid can point to the root
 or any other handle on the volume or the path can be an absolute path that
 identifies the volume.  See msgFSSetAttr on how to store a dir index with
 a directory so it can later be accessed by its dir index.

 Use FS_DIR_NEW_MODE for mode if new is for dir handle.
 Use FS_FILE_NEW_MODE for mode if new is for file handle.

 Return Value
	stsBadParam:			locator.uid is not a valid object.
	stsFSAccessDenied:		Access cannot be granted because node is locked
							for exclusive access, read only access or write
							only access.
	stsFSBadPath:			locator.pPath is malformed or
							a specified dir node is in fact a file.
	stsFSDirFull:			There is no space in the dir for a new node.
	stsFSDirIndexNotFound:	There is not a dirIndex for the dir node.
	stsFSNodeBusy:			Node cannot be deleted/truncated because it is
							being access by another client.
	stsFSNodeExists:		The requested node already exists.
	stsFSNodeNotFound:		The root node does not exist.
	stsFSNodeReadOnly:		Node cannot be deleted/truncated or read/write
							access has been denied because the read only flag
							is set on the node.
	stsFSNotDir:			A requested dir node already exists as a file.
	stsFSNotFile			A requested file node already exists as a dir.
	stsFSTooManyHandles:	There are already fsMaxHandles on this node.
	stsFSUniqueFailed:		fsMaxUnique variants of the name already exist.

 See Also
	FSNameValid
*/
typedef struct FS_NEW_ONLY {
	FS_LOCATOR			locator;		// location of the target directory
	FS_VOL_TYPE			volType;		// hint for uninstalled fullpath vols
	UUID				dirIndex;		// used with fsUseDirIndex mode only
	U16					mode;			// options for opening file/dir handle
	FS_EXIST			exist;			// action to take if exists or doesn't
	P_UNKNOWN			pVolSpecific;	// volume specific information
										// Note: this is an in only parm
	U32					spare1;			// for future use
	U32					spare2;			// for future use
	BOOLEAN				alreadyExisted;	// Out: indicates if already exists
} FS_NEW_ONLY, * P_FS_NEW_ONLY;

#define fsNewFields		\
	objectNewFields		\
	FS_NEW_ONLY			fs;

typedef struct FS_NEW {
	fsNewFields
} FS_NEW, * P_FS_NEW;


/****************************************************************************
 msgNewDefaults				takes P_FS_NEW, returns STATUS
	category: class message
	Initializes the FS_NEW structure to default values.

 Zeroes out pNew->fs and sets:
//{
	pNew->fs.locator.uid = theWorkingDir;
	pNew->object.cap |= objCapCall;
//}
*/


/****************************************************************************
 msgDestroy					takes OBJ_KEY, returns STATUS
	Destroys a directory or file handle.

 This destroys the handle, NOT the actual node.	 An exception to this is
 if the fsTempFile/fsTempDir flag was set in pNew->fs.mode when the handle
 was created.

 Return Value
	stsFSNodeBusy:			Temporary node cannot be deleted because it is
							being access by another client.
	stsFSNodeReadOnly:		Temporary node cannot be deleted because the read
							only flag is set on the node.
*/


/****************************************************************************
 msgFSNull					takes void, returns STATUS
	Does nothing.
	
	This message is used to time entering and exiting the file system.
*/
#define msgFSNull						MakeMsg(clsFileSystem, 20)


/****************************************************************************
 msgFSGetVolMetrics			takes P_FS_GET_VOL_METRICS, returns STATUS
	Returns metrics of the volume.

 Return Value
	stsFSVolDisconnected:	This will never be returned, even if the volume
							is disconnected.  Instead test fsVolConnected
							in volMetrics.flags.

 You must set updateInfo to TRUE if you want the volMetrics.freeBytes field or
 the fsVolConnected flag of the volMetrics.flags field to be updated before
 returning the vol metrics.  Setting updateInfo to FALSE will make this
 request faster, but these fields may not be correct.
*/
#define msgFSGetVolMetrics				MakeMsg(clsFileSystem, 22)

typedef struct FS_GET_VOL_METRICS {
	BOOLEAN				updateInfo;		// have volume recompute values?
	FS_VOL_METRICS		volMetrics;		// Out: the volume's metrics
} FS_GET_VOL_METRICS, * P_FS_GET_VOL_METRICS;


/****************************************************************************
 msgFSSetVolName			takes P_STRING, returns STATUS
	Changes the name of a volume.

 Return Value
	stsBadParam:			New vol name is invalid (checked by FSNameValid).
	stsFSHandleInvalid:		The dir/file object refers to a node that has
							been previously deleted.
	stsFSVolDisconnected:	The volume is not connected.
	stsFSVolReadOnly:		The new volume name cannot be set,
							because the volume is write protected.

 See Also
	FSNameValid:			Mechanism to precheck validity of new volume name.
*/
#define msgFSSetVolName					MakeMsg(clsFileSystem, 36)


/****************************************************************************
 msgFSNodeExists			takes P_FS_NODE_EXISTS, returns STATUS
	Tests the existence of a file or directory node.

 The return parm isDir is useful in deciding whether the msgNew, to create
 a handle to the node, should be sent to clsDirHandle or clsFileHandle.
 The parm pPath is relative to the object that receives this message.

 Return Value
	stsOK:					The node exists.
	stsFSNodeNotFound:		The node does not exist.
*/
#define msgFSNodeExists					MakeMsg(clsFileSystem, 37)

typedef struct FS_NODE_EXISTS {
	P_STRING			pPath;			// path to node that may exist
	BOOLEAN				isDir;			// Out: dir or file
} FS_NODE_EXISTS, * P_FS_NODE_EXISTS;


/****************************************************************************
 msgFSGetHandleMode			takes P_U16, returns STATUS
	Returns the "new" mode for the object's fs handle.

 Directory handles interpret the P_U16 as a P_FS_FILE_NEW_MODE.
 File handles interpret the P_U16 as a P_FS_DIR_NEW_MODE.
*/
#define msgFSGetHandleMode				MakeMsg(clsFileSystem, 23)


/****************************************************************************
 msgFSSetHandleMode			takes P_FS_SET_HANDLE_MODE, returns STATUS
	Changes the "new" mode for the object's fs handle.

 Directory handles interpret mode as a FS_FILE_NEW_MODE.
 File handles interpret mode as a FS_DIR_NEW_MODE.
*/
#define msgFSSetHandleMode				MakeMsg(clsFileSystem, 24)

typedef struct FS_SET_HANDLE_MODE {
	U16					mode;			// value of mode flags to change
	U16					mask;			// which mode flags are to change
} FS_SET_HANDLE_MODE, * P_FS_SET_HANDLE_MODE;


/****************************************************************************
 msgFSSame					takes OBJECT, returns STATUS
	Tests if another directory or file handle references the same node.
*/
#define msgFSSame						MakeMsg(clsFileSystem, 25)


/****************************************************************************
 msgFSGetPath				takes P_FS_GET_PATH, returns STATUS
	Gets the path to (or name of) a directory or file handle node.

 If mode is fsGetPathRoot or fsGetPathAbsolute the root dir handle is
 passed back in dir.  If mode is fsGetPathRelative the path passed back
 begins at the dir represent by dir and terminates at the node represented
 by the recipient of this client.

 Return Value
	stsFSBufTooSmall:		User supplied pPathBuf is not large enough.
	stsFSNotAncestor:		dir is not ancestor of recipient of msgFSGetPath.
*/
#define msgFSGetPath					MakeMsg(clsFileSystem, 26)

typedef struct FS_GET_PATH {
	FS_GET_PATH_MODE	mode;			// options for get path operation
	DIR_HANDLE			dir;			// In-Out: rel dir or root dir
	U16					bufLength;		// length of pPathBuf
	P_STRING			pPathBuf;		// Out: user buffer for path
} FS_GET_PATH, * P_FS_GET_PATH;


/****************************************************************************
 msgFSGetAttr				takes P_FS_GET_SET_ATTR, returns STATUS
	Gets an attribute or attributes of a file or directory node.

 Specify which attributes you wish returned via an array of attribute
 labels pointed to by pAttrLabels.	The number of attribute labels is
 specified by numAttrs.	 The values are passed back via an array of values.
 If the nth value represents a string or variable attribute a pointer
 must be filled in for the destination of the string/variable.	If the nth
 value represents a Fix64 provide space for two consecutive U32s.
 The sizes are passed back via an array of sizes.
 
 If either the values are of no interest or the sizes are of no interest,
 set pAttrValues to pNull and/or set pAttrSizes to pNull.

 If you want all attributes of a node, but do not know what they may be
 set numAttrs to maxU16, pAttrLabels to fsAllocAttrLabelsBuffer, and
 pAttrValues to fsAllocAttrValuesBuffer (or pNull if unwanted) and
 pAttrSizes to fsAllocAttrSizesBuffer (or pNull if unwanted).  Any buffers
 returned as a result of fsAllocXXXBuffer must be freed with OSHeapBlockFree.

 The parm pPath is relative to the object that receives this message.
*/
#define msgFSGetAttr					MakeMsg(clsFileSystem, 27)

typedef struct FS_GET_SET_ATTR {
	P_STRING			pPath;			// path to node to get/set attrs
	U16					numAttrs;		// number of attrs of interest
	P_FS_ATTR_LABEL		pAttrLabels;	// In-Out: attr labels
	P_UNKNOWN			pAttrValues;	// In-Out: attr values
	P_FS_ATTR_SIZE		pAttrSizes;		// In-Out: attr sizes
} FS_GET_SET_ATTR, * P_FS_GET_SET_ATTR;

/****************************************************************************
 msgFSSetAttr				takes P_FS_GET_SET_ATTR, returns STATUS
	Sets the attribute or attributes of a file or directory node.

 Specify which attributes you wish to set via an array of attribute
 labels pointed to by pAttrLabels.	The number of attribute labels is
 specified by numAttrs.	 The values are specified via an array of values.
 If the nth value represents a string or variable attribute supply
 the pointer to the string/variable.  If the nth value represents a Fix64
 attribute two consecutive U32 values are expected.	 If there are no
 variable length attributes, pAttrSizes can be set to pNull, because
 the size of Fix32, Fix64 and string attributes can be inferred.

 pAttrLabels, pAttrValues & pAttrSizes are inputs only for this message.
 The parm pPath is relative to the object that receives this message.

 The attr fsAttrDirIndex (dir indexes) can be set on directories to establish
 an alternate access to a directory without having to specify the path to the
 directory.  See msgNew above on how to access directories with a dir index.
 Only directories that reside under the PenPoint tree (any directories below
 the PenPoint directory on a given volume) can have dir index attributes.
 If another directory already has the same dir index as the one given then a
 stsFSDirIndexExists error is returned.

 NOTE: Most attributes (with the exception of dir index and old dir index
 attributes) can be stored with either files or directories.  The root of
 a volume is the exception.  No attributes may be stored with the root.

 Return Value
	stsFSBadPath:			New name for name attr is invalid.
	stsFSNotDir:			Dir index attr cannot be set on a file.
	stsFSReadOnlyAttr:		File size cannot be set via set attr,
							use msgFSSetSize.
*/
#define msgFSSetAttr					MakeMsg(clsFileSystem, 28)


/****************************************************************************
 msgFSMove					takes P_FS_MOVE_COPY, returns STATUS
	Moves a node (and any children) to a new destination.

 The destination file/dir name of a move is derived as follows.

 For "fsMoveCopyToDest" (the default): If non null path is provided then
 dest name is the leaf name of the path and the path up to the leaf name
 determines the destination directory.	If the path is null then the name
 of the destination object is used as the dest name and the parent of the
 destination object is used as the destination directory.

 For fsMoveCopyIntoDest:  The entire destination uid and path are used for
 the destination directory.	 And the destination name is taken from the
 source name.
 
 The parm pSourcePath is relative to the object that receives this message.

 NOTE: pNewDestName is not an in parameter.  It is an output parameter that
 gives the (new, if fsMoveCopyGenUnique was specified for exist) name of the
 copied node.  Set pNewDestName to a buffer if you want to know the name,
 set pNewDestName to pNull if you do not.

 Return Value
	stsFSBadPath:			Path or parts of path are too large.
	stsFSCircularMoveCopy:	Occurs when copying dir to an ancestor (parent).

 See Also
	msgFSMoveNotify, msgFSCopy
*/
#define msgFSMove						MakeMsg(clsFileSystem, 29)

typedef struct FS_MOVE_COPY {
	P_STRING			pSourcePath;	// path of source of move or copy
	FS_LOCATOR			destLocator;	// locator to destination node
	FS_MOVE_COPY_MODE	mode;			// options that affect move or copy
	FS_MOVE_COPY_EXIST	exist;			// action to take if exists or doesn't
	P_STRING			pNewDestName;	// Out: See comment above
	BOOLEAN				alreadyExisted;	// Out: indicates if already exists
	U32					spare;
} FS_MOVE_COPY, * P_FS_MOVE_COPY;


/****************************************************************************
 msgFSCopy					takes P_FS_MOVE_COPY, returns STATUS
	Copies a node (and any children) to a new destination.

 The destination file/dir of a copy is derived as follows.

 For "fsMoveCopyTo" (the default): If non null path is provided then dest
 name is the leaf name of the path and the path up to the leaf name
 determines the destination directory.	If the path is null then the name
 of the destination object is used as the dest name and the parent of the
 destination object is used as the destination directory.

 For fsMoveCopyInto:  The entire destination uid and path are used for the
 destination directory.	 And the destination name is taken from the source
 name.

 The parm pSourcePath is relative to the object that receives this message.

 NOTE: pNewDestName is not an in parameter.  It is an output parameter that
 gives the (new, if fsMoveCopyGenUnique was specified for exist) name of the
 copied node.  Set pNewDestName to a buffer if you want to know the name,
 set pNewDestName to pNull if you do not.

 Return Value
	stsFSBadPath:			Path or parts of path are too large.
	stsFSCircularMoveCopy:	Occurs when copying dir to an ancestor (parent).

 See Also
	msgFSCopyNotify, msgFSMove
*/
#define msgFSCopy						MakeMsg(clsFileSystem, 30)


/****************************************************************************
 msgFSMoveNotify                takes P_FS_MOVE_COPY_NOTIFY, returns STATUS
    Same as msgFSMove with notification routine extensions.

 The parm pSourcePath is relative to the object that receives this message.
*/
#define msgFSMoveNotify         MakeMsg(clsFileSystem, 70)

// the time that the current event occurred
Enum16 ( FS_NOTIFY_TIME ) {
    fsBeginOperation = 1,                   // beginning of whole operation
    fsBeforeOperation = 2,                  // before the sub operation 
    fsDuringOperation = 3,                  // during the sub operation
    fsAfterOperation = 4,                   // after the sub operation
    fsEndOperation = 5                      // end of the whole operation
};

// the operation of the current event
Enum16 ( FS_NOTIFY_OP ) {
    fsReadOperation = 1,                    // read operation 
    fsWriteOperation = 2,                   // write operation
    fsCreateOperation = 3,                  // create operation
    fsVerifyOperation = 4,                  // verify operation
    fsDeleteOperation = 5                   // delete operation
};

// information required by the notification routine
typedef struct FS_NOTIFY_RTN_INFO {
    OBJECT              source;             // a handle to the current file
    BOOLEAN             moveOperation;      // if move operation
    BOOLEAN             isADirectory;       // if source is a directory
    P_FS_GET_SET_ATTR   pFSGetSetAttr;      // attributes for current file
    FS_NOTIFY_TIME      fsNotifyTime;       // time context of notification
    FS_NOTIFY_OP        fsNotifyOp;         // op context of notification
    U32                 bufferSize;         // max size of operation buffer
    U32                 operationSize;      // actual size of operation
    U32                 fileSize;           // actual size of file
    U32                 spare1;             // spare: unused
    U32                 spare2;             // spare: unused
} FS_NOTIFY_RTN_INFO, *P_FS_NOTIFY_RTN_INFO;

// the definition of the notification routine
typedef STATUS FunctionPtr ( P_FS_NOTIFY_RTN ) ( P_FS_NOTIFY_RTN_INFO pFSNotifyRtnInfo,
                                                 P_UNKNOWN pClientData );
// the information required for FSMove/CopyNotify
typedef struct FS_MOVE_COPY_NOTIFY {
	P_STRING			pSourcePath;	    // path of source of move or copy
	FS_LOCATOR			destLocator;	    // locator to destination node
	FS_MOVE_COPY_MODE	mode;			    // options that affect move or copy
	FS_MOVE_COPY_EXIST	exist;			    // action to take if exists or doesn't
	P_STRING			pNewDestName;	    // Out: See comment w/msgFSMove
	BOOLEAN				alreadyExisted;	    // Out: indicates if already exists
	P_UNKNOWN			pNotifyRtn;         // notification routine
    P_UNKNOWN           pClientData;        // client data to notification routine
    P_UNKNOWN           pQuickSortRtn;      // quicksort routine
    U32                 spare1;             // spare: unused
    U32                 spare2;             // spare: unused
} FS_MOVE_COPY_NOTIFY, * P_FS_MOVE_COPY_NOTIFY;


/****************************************************************************
 msgFSCopyNotify                takes P_FS_MOVE_COPY_NOTIFY, returns STATUS
    Same as msgFSCopy with notification routine extensions.

 The parm pSourcePath is relative to the object that receives this message.
*/
#define msgFSCopyNotify         MakeMsg(clsFileSystem, 71)


/****************************************************************************
 msgFSDelete				takes P_STRING, returns STATUS
	Deletes a node (and all of its children).

 The object of msgFSDelete is typically a dir handle, but it can also be a
 file handle, but the argument passed must be set to pNull.
 After a node is deleted, its handle is marked corrupt (since it is no longer
 valid).  A dir handle object can be reused via msgFSSetTarget or destroyed
 via msgDestroy.  A file handle must be destroyed after the node is deleted.
 The argument (a path) is relative to the object that receives this message.

 Return Value
	stsFSVolDisconnected:	The volume is not connected.
	stsFSVolReadOnly:		A node cannot be deleted,
							because the volume is write protected.
	stsFSNodeReadOnly:		Node cannot be deleted because the read only
							flag is set on the node.
	stsFSNodeBusy:			Node cannot be deleted because it is
							being access by another client.

 See Also
	msgFSForceDelete
*/
#define msgFSDelete						MakeMsg(clsFileSystem, 31)


/****************************************************************************
 msgFSFlush					takes void, returns STATUS
	Flushes any buffers and attributes associated with the file or directory.

 This can be used to guarantee that cached buffers are flushed to the disk
 and can also be used to flush memory mapped files to disk.
*/
#define	msgFSFlush						MakeMsg(clsFileHandle, 20)


/****************************************************************************
 msgFSMakeNative			takes P_FS_MAKE_NATIVE, returns STATUS
	Removes anything not supported by the native file system.

 The parm pPath is relative to the object that receives this message.
*/
#define msgFSMakeNative					MakeMsg(clsFileSystem, 32)

typedef struct FS_MAKE_NATIVE {
	P_STRING			pPath;			// path to node to make native
	P_STRING			pNewName;		// Out: native name if changed
} FS_MAKE_NATIVE, * P_FS_MAKE_NATIVE;


/****************************************************************************
 msgFSEjectMedia			takes void, returns STATUS
	Ejects media from an ejectable, removable volume.

 Return Value
	stsOK:						The volume media has been ejected.
	stsFSVolDisconnected:		The volume media is already ejected.
	stsRequestNotSupported:		The volume does not have ejectable media
*/
#define msgFSEjectMedia					MakeMsg(clsFileSystem, 34)


/****************************************************************************
 msgFSForceDelete			takes P_STRING, returns STATUS
	Forcibly deletes a node (and all of its childen).

 WARNING.  Normal restrictions do not apply.  The node will still be deleted
 even if it is being accessed via another handle or if it is marked read only.
 However, if the volume is not connected or is write protected, the forced
 delete will still fail.

 After a node is deleted, its handle is marked corrupt (since it is no longer
 valid).  A dir handle object can be reused via msgFSSetTarget or destroyed
 via msgDestroy.  A file handle must be destroyed after the node is deleted.
 The argument (a path) is relative to the object that receives this message.

 See Also
	msgFSDelete
*/
#define msgFSForceDelete				MakeMsg(clsFileSystem, 35)


/****************************************************************************
 msgFSVolSpecific			takes P_FS_VOL_SPECIFIC, returns STATUS
	Sends a volume specific message via a dir or file handle.

 Return Value
	Volume specific errors.
*/
#define msgFSVolSpecific				MakeMsg(clsFileSystem, 40)

typedef struct FS_VOL_SPECIFIC {
	P_STRING			pPath;			// path of node to receive msg
	MESSAGE				msg;			// message to pass on to volume
	P_UNKNOWN			pArgs;			// In-Out: message specific args
} FS_VOL_SPECIFIC, * P_FS_VOL_SPECIFIC;


/****************************************************************************
 msgFSChanged				takes P_FS_CHANGE_INFO, returns STATUS
	category: observer notification
	Notifies observers of directory changes.

 This notifies observers of directories (not files) when a file or dir
 within the directory changes.  The change reasons described below are
 changes to the directory or file node, not the handle referencing the node.
*/
#define msgFSChanged					MakeMsg(clsFileSystem, 50)

typedef struct FS_CHANGE_INFO {
	MESSAGE				reason;			// fs message that caused the change
	OBJECT				observed;		// observed dir whose content changed
	U32					spare1;
	U32					spare2;
} FS_CHANGE_INFO, * P_FS_CHANGE_INFO;

/*
 These messages are the reason observers of a dir handle would be
 notified of a change and the circumstances that the change happens:

	msgInit:			A file or dir has been created.
	msgFree:			A temp file or temp directory has been deleted.
	msgFSDelete:		A file or directory has been deleted.
	msgFSForceDelete:	A file or directory has been deleted.
	msgFSMove:			A file or directory has been "fast" moved.
*/

/****************************************************************************
 msgFSVolChanged			takes P_FS_VOL_CHANGE_INFO, returns STATUS
	category: observer notification
	Notifies observer of volume changes.

	Observe the well known object, theFileSystem, if you want to receive this.
*/
#define msgFSVolChanged					MakeMsg(clsFileSystem, 51)

Enum16(FS_VOL_CHANGE_FLAGS) {
	fsVolChangeWhilePrompting	= flag0		// FS prompting caused change
};

typedef struct FS_VOL_CHANGE_INFO {
	MESSAGE				reason;			// fs message that caused the change
	OBJECT				rootDir;		// root dh of volume that changed
	FS_VOL_CHANGE_FLAGS	flags;			// more info related to reason
	U16					spare1;
	U32					spare2;
} FS_VOL_CHANGE_INFO, * P_FS_VOL_CHANGE_INFO;

/*
 These messages are the reason observers of theFileSystem would be
 notified of a volume addition, removal or change of state.
 Note: msgFSSetVolName (defined above) is also a volume change reason.
*/

#define msgFSInstallVol					MakeMsg(clsFileSystem, 1)
#define msgFSRemoveVol					MakeMsg(clsFileSystem, 2)
#define msgFSConnectVol					MakeMsg(clsFileSystem, 3)
#define msgFSDisconnectVol				MakeMsg(clsFileSystem, 4)

//REFGEN BEGINIGNORE
/****************************************************************************
 msgFSForceVolRemoval		takes void, returns STATUS
	Forces the removal of a volume from the file system.

 All handles on this volume will be marked corrupt and the volume will
 be freed.	If the volume has ejectable media, it will be ejected.

 Return Value
	stsRequestNotSupported:		Some volumes do not allow themselves to be
								removed.  For example the selected volume
								cannot be removed.
*/
#define msgFSForceVolRemoval			MakeMsg(clsFileSystem, 33)
//REFGEN ENDIGNORE

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Class DirHandle Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgFSSetTarget				takes P_FS_LOCATOR, returns STATUS
	Changes the target directory to directory specified by locator.

 Setting a dir handle object to a new target also resets the read dir pointer.

 Return Value
	stsFSUnchangeable:		The recipient of this message has been "opened"
							with the fsUnchangeable flag set in pNew->mode.
*/
#define msgFSSetTarget					MakeMsg(clsDirHandle, 20)


/****************************************************************************
 msgFSReadDir				takes P_FS_READ_DIR, returns STATUS
	Reads the next entry (its attributes) from a directory.

 Specify which attributes you wish returned via an array of attribute
 labels pointed to by pAttrLabels.	The number of attribute labels is
 specified by numAttrs.	 See msgFSGetAttr for a description on setting
 pAttrValues and pAttrSizes.
*/
#define msgFSReadDir					MakeMsg(clsDirHandle, 21)

typedef struct FS_READ_DIR {
	struct FS_READ_DIR	* pNext;		// Out: only used w/msgFSReadDirFull
	U16					numAttrs;		// In-Out: attrs of interest
	P_FS_ATTR_LABEL		pAttrLabels;	// In-Out: ptr to attr labels
	P_UNKNOWN			pAttrValues;	// In-Out: ptr to attr values
	P_FS_ATTR_SIZE		pAttrSizes;		// In-Out: ptr to attr sizes
} FS_READ_DIR, * P_FS_READ_DIR;


/****************************************************************************
 msgFSReadDirReset			takes void, returns STATUS
	Resets the ReadDir position to the beginning.

 This will direct msgFSReadDir to begin reading from the first entry
 in the directory.	This has no effect on msgFSReadDirFull.	 The default
 after creating a handle to a directory is to point to the first entry.
*/
#define msgFSReadDirReset				MakeMsg(clsDirHandle, 22)


/****************************************************************************
 msgFSReadDirFull			takes P_FS_READ_DIR_FULL, returns STATUS
	Reads all the entries in a directory into a local buffer.

 Specify which attributes you wish returned via an array of attribute
 labels pointed to by pAttrLabels.	The number of attribute labels is
 specified by numAttrs.

 The returned data is a linked list of FS_READ_DIR entries, linked by the
 pNext field.  The last link is specified by a pLink == pNull.

 The client must free the returned buffer pDirBuf, using OSHeapBlockFree.
 The buffer should not be freed if it has a value of pNull, which will
 be the case if there are any errors or if numEntries is zero.
*/
#define msgFSReadDirFull				MakeMsg(clsDirHandle, 23)

typedef struct FS_READ_DIR_FULL {
	U16					numAttrs;		// num of labels in label array
	P_FS_ATTR_LABEL		pAttrLabels;	// attrs of interest to be read
	U32					numEntries;		// Out: number of dir entries
	U32					bufLength;		// Out: length of pDirBuf
	P_FS_READ_DIR		pDirBuf;		// Out: points to first entry
} FS_READ_DIR_FULL, * P_FS_READ_DIR_FULL;

												
/****************************************************************************
 msgFSTraverse				takes P_FS_TRAVERSE, returns STATUS
	Traverse through the nodes of a tree starting with the target of this msg.

 This message traverses the file system tree beginning with the directory
 which is the recipient of this message and traverses the node tree depth
 first.	 The client will be called back via pCallBackRtn at each node
 depending on mode (see FS_TRAVERSE_MODE above).  Optionally, the nodes at
 each directory level can be sorted before being returned by specifying a
 quick sort routine via pQuickSortRtn (See quicksort in sort.h).

 Specify which attributes you wish returned via an array of attribute
 labels pointed to by pAttrLabels.	The number of attribute labels is
 specified by numAttrs.	 At a minimum, pAttrLabels must contain fsAttrName
 and fsAttrFlags.

 Return Value
	stsBadParam:			Did not specify fsAttrName/fsAttrFlags in labels.
	stsFSUnchangeable:		The recipient of this message has been "opened"
							with the fsUnchangeable flag set in pNew->mode.
							This is a common error if trying to traverse from
							the root dir (which is unchangeable) provided by
							msgFSGetInstalledVolumes/msgFSGetVolMetrics.
							Create a handle to the root and use that to
							traverse instead.
	stsFSNestingTooDeep:	Dir tree is deeper than fsMaxNestingLevel levels.
*/
#define msgFSTraverse						MakeMsg(clsDirHandle, 24)

/* Prototype for the call back routine used by msgFSTraverseTree */
typedef STATUS FunctionPtr(P_FS_TRAVERSE_CALL_BACK) (
	OBJECT				dir,				// dir handle to current node
	U16					level,				// level in the hierarchy
	P_FS_READ_DIR		pNextEntry,			// info about next entry
	P_UNKNOWN			pClientData			// the client's data
);

typedef struct FS_TRAVERSE {
	FS_TRAVERSE_MODE		mode;			// call back order and criteria
	U16						numAttrs;		// num of labels in label array
	P_FS_ATTR_LABEL			pAttrLabels;	// attr label array
	P_FS_TRAVERSE_CALL_BACK	pCallBackRtn;	// called for each dir & file
	P_UNKNOWN				pClientData;	// passed to call back routine
	P_UNKNOWN				pQuickSortRtn;	// optional quick sort routine
} FS_TRAVERSE, * P_FS_TRAVERSE;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Class FileHandle Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgStreamRead				takes P_STREAM_READ_WRITE, returns STATUS
	category: descendant responsibility
	Reads data from the file.

 The maximum number of bytes read with a single request is determined by
 fsMaxReadWrite.

 Return Value
	stsBadParam:			Requesting more than fsMaxReadWrite bytes.

 See Also
	msgStreamRead in stream.h
*/

/****************************************************************************
 msgStreamWrite				takes P_STREAM_READ_WRITE, returns STATUS
	category: descendant responsibility
	Writes data to the file.

 The maximum number of bytes writable with a single request is determined by
 fsMaxReadWrite.  Note that writes to a memory mapped file that cause the
 file to grow will result in a stsFSNodeBusy error.  Free the memory map file
 pointer before growing the file.

 Return Value
	stsBadParam:			Requesting more than fsMaxReadWrite bytes.
	stsFSNodeReadOnly:		This is a read only file.
	stsFSVolFull:			The file could not be written - no space on volume.
	stsFSNodeBusy:			The file is memory mapped and this write request
							would cause the file to be grown beyond the
							memory mapped size.

 See Also
	msgStreamWrite in stream.h
*/

/****************************************************************************
 msgStreamFlush				takes void, returns STATUS
	category: descendant responsibility
	Flushes any buffers associated with the file.

 See Also
	msgStreamFlush in stream.h
*/

												
/****************************************************************************
 msgStreamSeek				takes P_STREAM_SEEK, returns STATUS
	category: descendant responsibility
	Seeks to new position within the file.

 Return Value
	stsBadParam:			Seek mode is out of range.

 See Also
	msgStreamSeek in stream.h
*/

/****************************************************************************
 msgFSSeek					takes P_FS_SEEK, returns STATUS
	Sets the value of the current byte position.

 Return Value
	stsBadParam:			Seek mode is out of range.
*/
#define msgFSSeek						MakeMsg(clsFileHandle, 21)

typedef struct FS_SEEK {
	FS_SEEK_MODE		mode;			// seek from bof, cur pos, eof 
	S32					offset;			// relative change from seek origin
	U32					curPos;			// Out: cur byte pos after seek
	U32					oldPos;			// Out: cur byte pos before seek
	BOOLEAN				eof;			// Out: Is new pos at end of file?
} FS_SEEK, * P_FS_SEEK;


/****************************************************************************
 msgFSGetSize				takes P_FS_FILE_SIZE, returns STATUS
	Gets the size of the file.
*/
#define msgFSGetSize					MakeMsg(clsFileHandle, 22)


/****************************************************************************
 msgFSSetSize				takes P_FS_SET_SIZE, returns STATUS
	Sets the size of the file.

 Note that a set size to a memory mapped file that causes the file to grow
 will result in a stsFSNodeBusy error.  Free the memory map file pointer
 before growing the file.

 Return Value
	stsFSNodeReadOnly:		This is a read only file.
	stsFSVolFull:			The file could not be grown - no space on volume.
	stsFSNodeBusy:			The file is memory mapped and this set size
							request would cause the file to be grown beyond
							the memory mapped size.
*/
#define msgFSSetSize					MakeMsg(clsFileHandle, 23)

typedef struct FS_SET_SIZE {
	FS_FILE_SIZE		newSize;		// new file size
	FS_FILE_SIZE		oldSize;		// Out: prior file size
} FS_SET_SIZE, * P_FS_SET_SIZE;


/****************************************************************************
 msgFSMemoryMap				takes PP_MEM, returns STATUS
	Associates the file with a directly accessible memory pointer.

 To get a memory mapped file pointer from shared memory, the file handle
 must be created with pNew->fs.mode |= fsSharedMemoryMap.
*/
#define msgFSMemoryMap					MakeMsg(clsFileHandle, 24)


/****************************************************************************
 msgFSMemoryMapFree			takes void, returns STATUS
	Frees the memory map pointer currently associated with the file.

 NOTE: Memory map pointers are freed for you at msgFree of a file handle.
*/
#define msgFSMemoryMapFree				MakeMsg(clsFileHandle, 25)


/****************************************************************************
 msgFSMemoryMapSetSize		takes SIZEOF, returns STATUS
	Sets the size of the file's memory map.

 Determines the limit of a memory map for the file.  The size can't be less
 than the file size, nor less than a limit set by another client but can be
 larger.  The memory map size must be set before memory mapping the file.

 Return Value
	stsFSNodeBusy:			The file is currently memory mapped.
*/
#define msgFSMemoryMapSetSize			MakeMsg(clsFileHandle, 26)


/****************************************************************************
 msgFSMemoryMapGetSize		takes P_SIZEOF, returns STATUS
	Gets the size of the file's memory map.
*/
#define msgFSMemoryMapGetSize			MakeMsg(clsFileHandle, 27)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Public Functions							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 FSNameValid	returns STATUS
	Checks a file/dir name for validity.

 Return Value
	stsOK:			The node name is valid.
	stsFailed:		The node name was invalid.

 Name is bad if it has no characters, is greater than 32 characters,
 has leading or trailing spaces, contains the pathname delimeter char,
 contains the file system escape character, or is the name of self (.)
 or parent (..).
*/
STATUS EXPORTED FSNameValid(
	P_STRING			pName			// name of file/dir to validate
);

#endif // FS_INCLUDED
