/****************************************************************************
 File: hash.h
	  
 (C) Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.6.2.1  $
   $Author:   jvierra  $
     $Date:   21 Apr 1992 19:51:50  $

 This package implements an "Open Addressing, Linear Probe" hash table.   

 The functions described in this file are contained in SYSUTIL.LIB. 
****************************************************************************/

/**** Introduction ****/
/*
 This package implements hash tables.  Hash tables offer relatively fast
 key-based random access to data at the expense of some memory.  The
 performance improvement over linear searching is substantial.

 The defaults supplied by this package are probably fine for most data. 
 However, hash table performance depends on both a good hash function and
 proper size parameters.  If your data's keys are unevenly distributed then
 consider writing your own hash function.  Try to get the hash table's
 initial size close to the number of expected entries divided by the fill
 percentage.  You can vary the fill percentage to meet your tradeoffs
 between space and time.
*/

/**** Creating a Hash Table ****/
/*
 To create a hash table:
 	-:	Allocate space for the hash table (either on the stack or in a heap
		block)
 	-:	Call HashInitDefaults()
	-:	Optionally customize the HASH_INFO structure
	-:	Call HashInit()
*/

/**** Examples ****/
/*
 Here's some sample code based on a 32 bit key.  (The package has various
 built-in Hash and Compare functions;  see section "Hash and Compare
 Functions.")

//{
    // Client data structure.  (The structure must contain a key field, 
	// though it need not be named key and it need not be the first field.)
    typedef struct {
        U32     data;
        U32     key;
    } YOUR_DATA, *P_YOUR_DATA;

	{
    	P_HASH_INFO     pHashInfo;
    	P_YOUR_DATA     pMD;
		U32				key;

		// Create table.  
		OSHeapBlockAlloc(osProcessHeapId, sizeof(*pHashInfo), &pHashInfo);
    	HashInitDefaults(pHashInfo);
		// Optionally customize between calls to HashInitDefaults() and 
		// HashInit().  For instance, if you have 16 bit keys, you
		// might do the following:
		//   pHashInfo->pHashFunction = HashFunction16;
		//   pHashInfo->pHashCompare = HashCompare16;
    	HashInit(pHashInfo, offsetof(YOUR_DATA, key));

    	// Add entry to hash table
		OSHeapBlockAlloc(osProcessHeapId, SizeOf(YOUR_DATA), &pMD);
    	pMD->key = 25;
    	pMD->data = someData;
    	HashAddEntry(pHashInfo, pMD);

    	// Find entry in hash table. Returns stsNoMatch if not found.
    	key = 25;
    	HashFindData(pHashInfo, &key, &pMD);
    	Debugf("Data for key %d is %d", key, pMD->data);

    	// Delete entry in hash table without freeing client data.
    	// Returns stsNoMatch if not found.
    	key = 25;
    	HashDeleteEntry(pHashInfo, &key, &pMD, false);
    	OSHeapBlockFree(pMD);

    	// Delete entry in hash table and free the client data.
    	// Returns stsNoMatch if not found.
    	key = 25;
		HashDeleteEntry(pHashInfo, &key, &pMD, true);

    	// Free hash table, and call OSHeapBlockFree() on all client data.
    	HashFree(pHashInfo, true);

		OSHeapBlockFree(pHashInfo);
	}
//}
*/

/**** Enumerating Hash Table Entries ****/
/*
 All of the entries in a hash table can be enumerated by examining the
 entries field of the HASH_INFO structure.  Empty entries are null.  Note
 that there are numEntries slots, numFilled of which are non-null.
//{
	P_HASH_INFO		pHashInfo;
	P_HASH_ENTRY	pEntries;

	pEntries = pHashInfo->entries;
	for (i = 0; i < pHashInfo->numEntries; i++) {
		if (pEntries[i].pData) {
			// Do something with entry
		}
	}
//}
*/


/**** Hash and Compare Functions ****/
/*
 The package includes good Hash and Compare functions for the following
 types of keys:
	-:	16 bit numbers
	-:	32 bit numbers
	-:	64 bit numbers
	-:	null-terminated strings

 Clients with other key types need to provide their own Hash and Compare
 functions.  Sophisticated clients may want to provide their own Hash and
 Compare functions even if they have keys with one of the above types.

 Replacement Hash and Compare functions should look like the following:
//{
	typedef struct {
		U8	major;
		U16	minor;
	} MY_KEY, * P_MY_KEY;

	typedef struct {
		MY_KEY		key;
		P_UNKNOWN	pData;
	} MY_DATA, * P_MY_DATA;

	U32 EXPORTED
	MyKeyHashFunction(
		P_HASH_KEY	pKey)
	{
		P_MY_KEY	pMyKey = (P_MY_KEY)pKey;
		U32			hash;
		hash = pMyKey->major * 9551;	// 9551 is prime
		hash += pMyKey->minor * 113;	// 113 is prime
		return hash;
	}

	BOOLEAN EXPORTED MyKeyHashCompare(
    	P_HASH_KEY      pKey1,
    	P_HASH_KEY      pKey2)
	{
		P_MY_KEY	pMyKey1 = (P_MY_KEY)pKey1;
		P_MY_KEY	pMyKey2 = (P_MY_KEY)pKey2;
		return ((pMyKey1->major == pMyKey2->major) AND
				(pMyKey1->minor == pMyKey2->minor));
	}	
//}
*/


/**** Space / Time Tradeoff ****/
/*
 The following table show the space / time tradeoff for a variety of 
 percentFull values, normalized to 80%.  This table is, of course, a gross
 simplification.  Among other things, it assumes well distributed keys.
//{
    full per-   relative   relative
    centage     speed      memory use
    ---------  --------   ----------
    10           2.8        8.0
    20           2.7        4.0
    30           2.5        2.7
    40           2.3        2.0
    50           2.0        1.6
    60           1.7        1.3
    70           1.4        1.2
    80           1.0        1.0
    90            .6         .9
    95            .3         .8
//}
*/

#ifndef HASH_INCLUDED
#define HASH_INCLUDED

#include <string.h>

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef OSTYPES_INCLUDED
#include <ostypes.h>
#endif

#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif

#include <stddef.h>

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Default values */
#define minHashTableInitialSize	15		// minimum initial size
#define minHashTableExpandSize	16		// minimum expand increment
#define hashTableMaxFillPct	    80		// expand when the table gets this
										// percentage full.

/* Key and Data Pointer Types */
typedef	void *	P_HASH_KEY;
typedef	void *	P_HASH_DATA;


/* Type for Hash function */
typedef U32 FunctionPtr(HASH_FUNCTION) (
	P_HASH_KEY	pKey
);


/*
 * Type for Compare function.  Function should return true if pKey1 and pKey2
 * point to keys with identical values.
*/
typedef BOOLEAN FunctionPtr(HASH_COMPARE)(
    P_HASH_KEY      pKey1,
    P_HASH_KEY      pKey2
);


/* A hash table entry. */
typedef struct HASH_ENTRY {
	P_HASH_DATA		pData;	// Points to user data
} HASH_ENTRY, * P_HASH_ENTRY, ** PP_HASH_ENTRY;
			  

/*
 * The hash table itself.  Space for the table is allocated by the client.
 * Space for the entries is allocated by hash table functions and is freed
 * via a call to HashFree().
 *
 * The debugging version of the hash table gathers statistics.
*/
typedef struct HASH_INFO {

	U32				numEntries;			// number of entries allocated.  
										// Should be prime!
	U32				numFilled;			// number of entries in use.  Not
										// too small or table will expand too 
										// often.  Should be even.
    U32             expandNumber;       // number of entries to expand by
    U32             percentFull;        // max percentage full at expand time.
										// Performance falls off rapidly if 
										// table allowed to get much fuller
										// than 80%.
    U16             keyOffset;          // offset of key in P_HASH_DATA
	OS_HEAP_ID		heap;				// heap to expand into
	P_HASH_ENTRY	entries;            // points to hash table array.
                                        // Array can be indexed sequentially
                                        // to find all the entries in the
                                        // table.  Empty slots are null.
    HASH_FUNCTION   pHashFunction;      // Hash function
    HASH_COMPARE    pHashCompare;       // Compare function

    // Statistics maintained for DEBUG version
    U32             numProbes;          // Counts number of hash probes
    U32             numProbeMisses;     // Counts number of probe retries
    U32             numAdds;            // Counts number of adds  
    U32             numDeletes;         // Counts number of deletes

} HASH_INFO, * P_HASH_INFO;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Functions								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 HashFindData		returns STATUS
	Given a key, passes back a P_HASH_DATA.

 Return Value
	stsNoMatch:		the key is not in the table.  *ppData is undefined.
	stsOK:			the key is in the table.

 See Also
	HashFindTableEntry
*/
STATUS EXPORTED HashFindData (
	P_HASH_INFO	    pInfo,
	P_HASH_KEY		pKey,
	P_HASH_DATA	  * ppData);


/****************************************************************************
 HashFindTableEntry		returns STATUS
	Given a key, passes back a pointer to client data.

 Return Value
	stsNoMatch:		the key is not in the table.  *ppEntry is undefined.
	stsOK:			the key is in the table.

 See Also
	HashFindData
*/
STATUS EXPORTED HashFindTableEntry (
	P_HASH_INFO	    pInfo,
	P_HASH_KEY   	pKey,
	PP_HASH_ENTRY	ppEntry);


/****************************************************************************
 HashAddEntry		returns STATUS
    Adds an entry to a hash table.

 The hash table expands if adding this entry causes the table to exceed the
 expand threshold.

 Return Value
	stsFailed:		the key is already in the table
*/
STATUS EXPORTED HashAddEntry (
	P_HASH_INFO	    pInfo,
	P_HASH_DATA	    pData);


/****************************************************************************
 HashDeleteEntry		returns STATUS
 	Deletes entry from hash table.

 If freeClientData is true then the client data is deallocated using ppData
 is undefined.  Otherwise *ppData contains the pointer to client data.
 
 Freeing entries does not cause the table to shrink.

 Return Value
	stsNoMatch:		the key is not in the table.
*/
STATUS EXPORTED HashDeleteEntry (
	P_HASH_INFO	    pInfo,
	P_HASH_KEY  	pKey,
    P_HASH_DATA   * ppData,
    BOOLEAN         freeClientData);						  


/****************************************************************************
 HashInitDefaults		returns STATUS
	Initializes hash table parameters.

 Warning: HashInitDefaults() MUST be called before HashInit.  See the
 section "Examples."

 Default values:
//{
    memset(pInfo, 0, sizeof(HASH_INFO));
	pInfo->numEntries       = 31;
    pInfo->expandNumber     = 24;
	pInfo->heap             = osProcessHeapId;
    pInfo->pHashFunction    = HashFunction32;   // Default 32 bit key
    pInfo->pHashCompare     = HashCompare32;    // Default 32 bit key
    pInfo->percentFull      = 80;
//}
*/
STATUS EXPORTED HashInitDefaults(
	P_HASH_INFO	pInfo);


/****************************************************************************
 HashInit	returns STATUS
	Causes the hash table to allocate internal tables.

 The client must call this function after calling HashInitDefaults() and
 performing any optional customization.

 Example:
//{
    HashInitDefaults(pInfo);
    HashInit(pInfo, offsetof(YOUR_DATA, key));
//}
*/
STATUS EXPORTED HashInit(
	P_HASH_INFO		pInfo, 
	U32 			keyOffset);		// offset of key in client data.


/****************************************************************************
 HashFree	returns STATUS
    Frees internal hash table memory.  Optionally deallocates any remaining
    user data blocks.

 If freeAllEntries is true, then the hash table calls OSHeapBlockFree() on
 each remaining piece of client data.  
 
 If the client is going to call HashFree() with freeAllEntries false,  the
 client must free all client data beforehand.

 Note that this function does NOT free the HASH_INFO structure.  If the
 client allocated it before calling HashInit() then the client should free
 the table after calling HashFree().
*/
STATUS EXPORTED HashFree(
	P_HASH_INFO	    pInfo,
    BOOLEAN         freeAllEntries);


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				Built-in Hash and Compare Functions						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * The functions in this section are useful default hash and compare
 * functions for common key types.  The 64 bit, 32 bit, and 16 bit functions
 * work equally well for signed or unsigned values.
*/

/**** 64 bit keys ****/
U32 EXPORTED HashFunction64(P_HASH_KEY pKey);
BOOLEAN EXPORTED HashCompare64(P_HASH_KEY pKey1, P_HASH_KEY pKey2);

/**** 32 bit keys ****/
U32 EXPORTED HashFunction32(P_HASH_KEY pKey);
BOOLEAN EXPORTED HashCompare32(P_HASH_KEY pKey1, P_HASH_KEY pKey2);

/**** 16 bit keys ****/
U32 EXPORTED HashFunction16(P_HASH_KEY pKey);
BOOLEAN EXPORTED HashCompare16(P_HASH_KEY pKey1, P_HASH_KEY pKey2);

/**** String keys ****/
U32 EXPORTED HashFunctionString(P_HASH_KEY pKey);
BOOLEAN EXPORTED HashCompareString(P_HASH_KEY pKey1, P_HASH_KEY pKey2);

#endif
