/****************************************************************************
 File: icon.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.28  $
   $Author:   cmeyer  $
     $Date:   16 Mar 1992 17:34:40  $

 This file contains the API definition for clsIcon.

 clsIcon inherits from clsMenuButton.
 Icons support drawing a picture as well as a label string.
 Several picture types are supported, including bitmap.
****************************************************************************/
#ifndef ICON_INCLUDED
#define ICON_INCLUDED

													#ifndef MBUTTON_INCLUDED
#include <mbutton.h>
													#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT ICON;

/****  Picture Styles  ****/
#define	isPictureBitmap			0	// picture is a bitmap
#define	isPictureNone			1	// no picture
#define	isPicturePixelmap		2	// picture is a pixelmap
//								3	// unused (reserved)

/****  Aspect Ratio Styles  ****/
#define	isAspectWidthFromHeight		0	// compute width from height & sample size
#define	isAspectHeightFromWidth		1	// compute height from width & sample size
#define	isAspectAsIs				2	// use the width and height as-is
//									3	// unused (reserved)

/****  Common Layout Units Picture Sizes  ****/
#define	iconSizeNormal			21	// standard size, both width and height
#define	iconSizeSmall			10	// standard small size


typedef struct ICON_STYLE {
	U16	transparent		: 2,	// make the background transparent
		picture			: 2,	// type of picture
		freeBitmap		: 1,	// true => msgDestroy to bitmap after provide
		open			: 1,	// modify picture to look open
		sizeUnits		: 6,	// units for pictureSize, e.g. bsUnitsPoints
		sampleBias		: 1,	// true => alter pictureSize for quality
		aspect			: 2,	// aspect ration rule (e.g. isAspectWidthFromHeight)
		spare1	  		: 1;	// unused (reserved)

	U16	spare2			: 16;	// unused (reserved)
} ICON_STYLE, *P_ICON_STYLE;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_ICON_NEW, returns STATUS
	category: class message
	Creates an icon window.

 If pArgs->icon.style.transparent is true, wsTransparent is turned on in
 pArgs->win.flags.style and bsInkExclusive will be or-ed into 
 pArgs->border.style.backgroundInk.  
*/

typedef struct ICON_NEW_ONLY {
	ICON_STYLE			style;			// overall style
	SIZE16				pictureSize;	// picture size, in device units
	U32					spare;			// unused (reserved)
} ICON_NEW_ONLY, *P_ICON_NEW_ONLY;

#define	iconNewFields		\
	menuButtonNewFields		\
	ICON_NEW_ONLY			icon;

typedef struct ICON_NEW {
	iconNewFields
} ICON_NEW, *P_ICON_NEW;

/****************************************************************************
 msgNewDefaults				takes P_ICON_NEW, returns STATUS
	category: class message
	Initializes the ICON_NEW structure to default values.

 Zeroes out pArgs->icon and sets
//{
	pArgs->gWin.style.gestureEnable = true;

	pArgs->border.style.backgroundInk = bsInkWhite | bsInkExclusive;
	pArgs->border.style.borderInk = bsInkWhite | bsInkExclusive;

	pArgs->border.style.previewAlter = bsAlterBorders;
	pArgs->border.style.selectedAlter = bsAlterBorders;
	pArgs->border.style.edge = bsEdgeBottom;
	pArgs->border.style.shadow = bsShadowNone;

	pArgs->control.style.showDirty = true;

	pArgs->label.style.xAlignment = lsAlignCenter;

	pArgs->icon.style.freeBitmap	= true;
	pArgs->icon.style.sampleBias	= true;
	pArgs->icon.pictureSize.w = pArgs->icon.pictureSize.h = iconSizeNormal;
//}
*/

/*
 Default ICON_STYLE:
//{
 	transparent		= false
 	picture			= isPictureBitmap
 	freeBitmap		= true
 	open			= false
 	sizeUnits		= bsUnitsLayout
 	sampleBias		= true
 	aspect			= isAspectWidthFromHeight
//}
*/

/****************************************************************************
 msgIconGetStyle		takes P_ICON_STYLE, returns STATUS
	Passes back the current style values.
*/
#define msgIconGetStyle	MakeMsg(clsIcon, 1)

/****************************************************************************
 msgIconSetStyle		takes P_ICON_STYLE, returns STATUS
	Sets the style values.

 If style.open changes, the rect of the picture is dirtied by self-sending
 msgWinDirtyRect.

 Note that changing style.transparent is not implemented.
*/
#define msgIconSetStyle	MakeMsg(clsIcon, 2)

/****************************************************************************
 msgIconGetPictureSize		takes P_SIZE16, returns STATUS
	Passes back the picture size in style.sizeUnits.
*/
#define msgIconGetPictureSize	MakeMsg(clsIcon, 3)

/****************************************************************************
 msgIconSetPictureSize		takes P_SIZE16, returns STATUS
	Sets the picture size.

 The new picture size should be in style.sizeUnits (e.g. bsUnitsLayout).
 clsIcon will free the cached picture as in msgIconFreeCache.
*/
#define msgIconSetPictureSize	MakeMsg(clsIcon, 4)

/****************************************************************************
 msgIconGetActualPictureSize		takes P_SIZE16, returns STATUS
	Computes and passes back the actual picture size in device units.

 This is equivalent using msgIconGetPictureSize and converting to device
 units if style.sampleBias is false or style.picture is not isPictureBitmap.

 Otherwise, clsIcon will compute and pass back the actual picture size used
 based on the sample size of the bitmap, the specified picture size and
 style.sizeUnits, style.aspect, and the device resolution of the icon's
 window device.
*/
#define msgIconGetActualPictureSize	MakeMsg(clsIcon, 10)

/****************************************************************************
 msgIconFreeCache		takes pNull, returns STATUS
	Frees the cached picture, if any.

 If style.picture isPictureBitmap, is clsIcon will send msgIconProvideBitmap
 on the next msgWinRepaint.

 Note that clsIcon does not self-send msgWinDirtyRect here.  You should
 send msgWinDirty rect after msgIconFreeCache if you want the icon to repaint
 before it is otherwise damaged.
*/
#define msgIconFreeCache			MakeMsg(clsIcon, 8)

/****************************************************************************
 msgIconGetRects		takes P_RECT32, returns STATUS
	Passes back the bounds for the picture in pArgs[0] and the label in
	pArgs[1].

 Note that pArgs is an array of two RECT32 structs.
 Bounds are in device units, relative to the origin of the icon.
*/
#define msgIconGetRects			MakeMsg(clsIcon, 6)

/****************************************************************************
 msgIconProvideBitmap		takes P_ICON_PROVIDE_BITMAP, returns STATUS
	category: self-sent and client notification
	Sent to control client when icon needs the picture bitmap.

 clsIcon will self-send this message when it needs the picture bitmap.
 Subclasses can catch this message and provide the appropriate bitmap.

 If clsIcon receives this message, the message will be forwarded on to the
 icon's control client.

 After the subclass or client provides the bitmap, clsIcon will copy the
 bitmap to a cached data structure for use when painting.  If style.freeBitmap
 is true, clsIcon will send msgDestroy to the bitmap after creating the cache.
*/
#define msgIconProvideBitmap	MakeMsg(clsIcon, 7)

typedef struct ICON_PROVIDE_BITMAP {
	WIN				icon;			// in: icon asking for the bitmap
	TAG				tag;			// in: window tag of icon
	OBJECT			device;			// in: device on which bitmap will be rendered
	SIZE16			pictureSize;	// in: size of picture, device units
	OBJECT			bitmap;			// out: bitmap to render
	U32				spare1;			// unused (reserved)
	U32				spare2;			// unused (reserved)
} ICON_PROVIDE_BITMAP, *P_ICON_PROVIDE_BITMAP;


/****************************************************************************
 msgIconCopyPixels		takes P_ICON_COPY_PIXELS, returns STATUS
	Causes the icon to copy pixels from pArgs->srcWin to a pixelmap.

 If style.picture is not isPicturePixelmap or pArgs->srcWin is objNull,
 clsIcon will return stsBadParam.

 The area copied has size of pictureSize and origin pArgs->srcXY
 in pArgs->srcWin space. The pixelmap will be used during msgWinRepaint.
*/
#define msgIconCopyPixels		MakeMsg(clsIcon, 9)

typedef struct ICON_COPY_PIXELS {
	WIN				srcWin;			// in: source window
	XY32			srcXY;			// in: origin of area to copy, srcWin space
	U32				spare1;			// unused (reserved)
} ICON_COPY_PIXELS, *P_ICON_COPY_PIXELS;

/****************************************************************************
 msgIconSampleBias		takes P_ICON_SAMPLE_BIAS, returns STATUS
	category: class message
	Computes the sample-biased size for a given picture size.

 clsIcon will alter pArgs->size.w/h to be a multiple of
 pArgs->sampleSize.w/h.  If the new pArgs->size.w/h is within
 pArgs->tolerance units from pArgs->sampleSize.w/h, the size is
 rounded up or down to the sample size.

 pArgs->sampleSize should be in device units.  pArgs->size should be in
 the units described by pArgs->sizeUnits (e.g. bsUnitsLayout).
 pArgs->tolerance should be in layout units.  pArgs->win is any window
 on the related device.

 If pArgs->aspect is isAspectWidthFromHeight, the width will be
 computed from the final height as
  	
	size.w = size.h * (sampleSize.w / sampleSize.h);

 If pArgs->aspect is isAspectHeightFromWidth, the height will be
 computed from the final width as
  	
	size.h = size.w * (sampleSize.h / sampleSize.w);

 This message can be sent to clsIcon or any instance of clsIcon.

 Here is the current "size bias" code.  In this fragment sampleSize is
 the sample's width or height, size is the proposed picture with or height,
 tolerance is the "snap-to" tolerance.  All values are in device units.
 The computed value is the sample-biased picture width or height.

//{
	if (size > sampleSize) {
		S32		mult;
		S32		lowerValue, lowerDelta;
		S32		upperValue, upperDelta;

		mult = size / sampleSize;
		lowerValue = mult * sampleSize;
		lowerDelta = size - lowerValue;
		upperValue = (mult + 1) * sampleSize;
		upperDelta = upperValue - size;
		if (lowerDelta < upperDelta) {
			value = lowerValue;
			delta = lowerDelta;
		} else {
			value = upperValue;
			delta = upperDelta;
		}
	} else {
		delta = sampleSize - size;
		value = sampleSize;
	}
		
	if (delta <= tolerance)
		size = value;

	return size;
//}

*/
#define msgIconSampleBias			MakeMsg(clsIcon, 11)

// default tolerance used buy clsIcon
// amount a picture size can be adjusted up or down for sample bias, 
// in layout units
#define	iconSampleTolerance		4

typedef struct ICON_SAMPLE_BIAS {
	WIN		win;				// in: device window
	U32		tolerance;			// in: snap-to tolerance, in layout units
	SIZE32	sampleSize;			// in: sample size, in device units
	SIZE32	size;				// in/out: picture size, in device units
	U16		sizeUnits	: 6,	// in: units for size
			aspect		: 2,	// in: aspect ratio style
			spare1		: 8;	// unused (reserved)
	U32		spare2;				// unused (reserved)
} ICON_SAMPLE_BIAS, *P_ICON_SAMPLE_BIAS;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from other classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgWinSetTag takes P_WIN_METRICS, returns STATUS
    Sets the window tag.

 If pArgs->tag is the same as the current window tag, nothing is done
 and stsOK is returned.

 If style.picture is isPictureBitmap, clsIcon will self-send
 msgIconFreeCache followed by msgWinDirtyRect to force a redraw of the
 icon picture.
*/

#endif	// ICON_INCLUDED

