/****************************************************************************
 File: listbox.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.35  $
   $Author:   cmeyer  $
     $Date:   18 Mar 1992 16:57:52  $

 This file contains the API for clsListBox.

 clsListBox inherits from clsScrollWin.
 Implements a scrolling list of windows (of arbitrary length).

 The windows that the listBox manages may be of any class, and they may
 be of different classes within a listBox.  The windows may have different
 heights as well.  The listBox will constrain their widths as per various
 style settings.

 A listBox is useful when the number of windows that could be displayed
 is unknown, variable, or large (say 30 or more).  The listBox will,
 by default, destroy those windows that have scrolled out of view, thus
 keeping the number of windows in existence to a reasonable quantity.

 By using a listBox, you trade performance for generality.  If the number
 of windows is likely to be small and not particularly variable, you may
 choose to put a tableLayout window in as the clientWin of a scrollWin
 instead.  The visual effect would be the same as for a listBox, but
 each of the tableLayout's child windows would, by default, be around
 for the lifetime of the parent (and as more windows are put on the screen,
 the overall performance of the UI degrades).

 As with most UI Toolkit classes, you may use clsListBox as-is, or create
 your own subclass for special purposes.  Since a common use of a listBox
 is to present a simple list of strings to the user, you may use
 clsStringListBox instead (see strlbox.h).  That class presents a somewhat
 simpler API for this common usage.  A subclass of clsStringListBox is
 clsFontListBox, which gets its strings from the list of currently installed
 fonts on the system.  clsFontListBox proves useful in situations such
 as option sheets (see fontlbox.h).
****************************************************************************/

/**** Debugging Flags ****/
/*
 The clsListBox debugging flag is 'K'.  Defined values are:

    flag12 (0x1000):    general
*/

#ifndef LISTBOX_INCLUDED
#define LISTBOX_INCLUDED

												#ifndef SWIN_INCLUDED
#include <swin.h>
												#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Common #defines and typedefs					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT LIST_BOX;

/****  ListBox Filing Styles  ****/
#define lbFileMin			0		// file minimum data necessary
#define lbFileEntryInfo		1		// lbFileMin + entry info except windows
#define lbFileAll			2		// lbFileEntryInfo + windows

typedef struct {
	U16		filing	:  2,
			spare	: 14;
} LIST_BOX_STYLE, *P_LIST_BOX_STYLE;

/*
 Default style:
//{
    filing = lbFileAll
//}
*/

typedef struct {
	LIST_BOX_STYLE	style;
	OBJECT			client;			// client to send list box messages to.
	U16				nEntries;		// total number of entries in list box.
	U16				nEntriesToView;	// show this many entries at a time.
	U32				spare;
} LIST_BOX_METRICS, *P_LIST_BOX_METRICS;

Enum16(LIST_BOX_DATA_FREE_MODE) {
	lbFreeDataNotVisible	= flag0, 
	lbFreeDataWhenDestroyed	= flag1,
	lbFreeDataByMessage		= flag2,
	lbFreeDataDefault		= lbFreeDataNotVisible | lbFreeDataWhenDestroyed
};

Enum16(LIST_BOX_ENTRY_STATE) {
	lbSelected		= flag0,
	lbOpen			= flag1,
	lbBusy			= flag2,
	lbStateDefault	= 0			// Not selected, not open
};

typedef struct LIST_BOX_ENTRY {
	WIN			listBox;	// in/out:	requestor
	U16			position;	// in:		entry position
	WIN			win;		// in/out: 	entry window to display
	U16			freeEntry;	// in/out: 	LIST_BOX_DATA_FREE_MODE
	U16			state;		// in/out:	LIST_BOX_ENTRY_STATE
	P_UNKNOWN	data;		// in/out:	client data
	P_UNKNOWN   arg;        // message specific argument
	U32         spare;      // reserved
} LIST_BOX_ENTRY, *P_LIST_BOX_ENTRY;

typedef struct LIST_BOX_POSITION_XY {
	XY32		place;		// in
	U16			position;	// in/out
	U32			spare;		// unused (reserved)
} LIST_BOX_POSITION_XY, *P_LIST_BOX_POSITION_XY;

typedef struct LIST_BOX_ENTRY_ENUM {
	U16					max;	// in	= size of pEntry[] array.
	U16					count;	// in	= # of entries to return in array.
                                // 		  If count > max then memory may be 
								//		  allocated.
                                // out	= # of valid entries in array.
	U16					next;	// in	= 0 to start at beginning 
                                //        OR previous out value to pick up
                                //        where we left off.
	P_LIST_BOX_ENTRY	pEntry;	// in	= Ptr to array of entries.
                                // out	= If memory was allocated client 
								//		  should free the memory.
	U16                 flags;  // in   = state flags to filter on.
	U32					spare;	// unused (reserved)
} LIST_BOX_ENTRY_ENUM, *P_LIST_BOX_ENTRY_ENUM;

#define stsListBoxEmpty						MakeStatus(clsListBox, 1)

/****************************************************************************
 msgNew						takes P_LIST_BOX_NEW, returns STATUS
	category: class message
	Creates a list box (initially empty).

 clsListBox sets the following values before calling its ancestor:
//{
	pArgs->scrollWin.style.getDelta = false;
	pArgs->scrollWin.style.vertClient = swClientWin;
	pArgs->scrollWin.style.horizClient = swClientScrollWin;
	pArgs->scrollWin.style.getSize = true;
	pArgs->scrollWin.style.forward = swForwardGesture;
//}
*/

typedef LIST_BOX_METRICS LIST_BOX_NEW_ONLY, *P_LIST_BOX_NEW_ONLY;

#define listBoxNewFields	\
	scrollWinNewFields		\
	LIST_BOX_NEW_ONLY		listBox;

typedef struct {
	listBoxNewFields
} LIST_BOX_NEW, *P_LIST_BOX_NEW;


/****************************************************************************
 msgNewDefaults					takes P_LIST_BOX_NEW, returns STATUS
	category: class message
	Initializes the LIST_BOX_NEW structure to default values.

 clsListBox sets the following values:
//{
	pArgs->win.flags.style |= wsShrinkWrapHeight;
	pArgs->win.flags.style &= ~wsShrinkWrapWidth;
	pArgs->border.style.edge = bsEdgeAll;
	pArgs->scrollWin.style.expandChildWidth = true;
	pArgs->listBox.style.filing = lbFileAll;
	pArgs->listBox.client = objNull;
	pArgs->listBox.nEntries = 0;
	pArgs->listBox.nEntriesToView = 6;
	pArgs->listBox.spare = 0;
//}
*/


/****************************************************************************
 msgListBoxGetMetrics			takes P_LIST_BOX_METRICS, returns STATUS
	Passes back the metrics for a listBox.
*/
#define msgListBoxGetMetrics				MakeMsg(clsListBox, 1)


/****************************************************************************
 msgListBoxSetMetrics			takes P_LIST_BOX_METRICS, returns STATUS
	Sets the metrics for a listBox.

 You should send msgWinLayout to the listBox if the value of nEntriesToView
 has changed.

 The listBox might ask for new entries after the SetMetrics call returns
 if the value of nEntries has changed.
*/
#define msgListBoxSetMetrics				MakeMsg(clsListBox, 2)


/****************************************************************************
 msgListBoxAppendEntry			takes P_LIST_BOX_ENTRY, returns STATUS
	Appends an entry to the list box after the specified position.

 If win is objNull, the client will receive msgListBoxProvideEntry
 when the entry needs to be displayed.

 This is computationally expensive when the listBox has a parent.  In other
 words, all work necessary to fix up the listBox is performed immediately.
 If you want to batch several calls, consider extracting the listBox first.

 See Also
 	msgListBoxInsertEntry:		similar, but inserts before
*/
#define msgListBoxAppendEntry				MakeMsg(clsListBox, 3)


/****************************************************************************
 msgListBoxInsertEntry			takes P_LIST_BOX_ENTRY, returns STATUS
	Insert an entry to the list box before the specified position.

 If win is objNull, the client will receive msgListBoxProvideEntry
 when the entry needs to be displayed.

 This is computationally expensive when the listBox has a parent.  In other
 words, all work necessary to fix up the listBox is performed immediately.
 If you want to batch several calls, consider extracting the listBox first.

 See Also
 	msgListBoxAppendEntry:		similar, but appends after
*/
#define msgListBoxInsertEntry				MakeMsg(clsListBox, 4)


/****************************************************************************
 msgListBoxRemoveEntry			takes U16, returns STATUS
	Removes an entry from the list box.

 If the item was added with freeEntry != 0, then the item is freed 
 automatically by the list box.

 This is computationally expensive when the listBox has a parent.  In other
 words, all work necessary to fix up the listBox is performed immediately.
 If you want to batch several calls, consider extracting the listBox first.

 Return Value
 	stsBadParam:	the specified position is >= number of entries
*/
#define msgListBoxRemoveEntry 				MakeMsg(clsListBox, 5)


/****************************************************************************
 msgListBoxGetEntry				takes P_LIST_BOX_ENTRY, returns STATUS
	Gets an entry in a listBox by position.

 Will pass back the last one if the passed position is maxU16.

 Return Value
	stsListBoxEmpty:	the list box has no entries
	stsNoMatch:			the list box has no entry at that position
*/
#define msgListBoxGetEntry	 				MakeMsg(clsListBox, 6)


/****************************************************************************
 msgListBoxSetEntry				takes P_LIST_BOX_ENTRY, returns STATUS
	Sets an entry's information.

 Typically this message is used to set an entry's data or flag values.

 This message prohibits the caller from changing whether the entry has a
 window (by specifying an objNull pArgs->win when the entry has a window
 or vice versa).  Clients should use Append/Insert/Remove for this purpose.
 msgListBoxSetEntry does support replacing a window with a different one.

 Replacing an entry window is computationally expensive when the listBox has
 a parent.

 Return Value
	stsListBoxEmpty:	the list box has no entries
	stsBadParam:		attempt to add or remove an entry
*/
#define msgListBoxSetEntry	 				MakeMsg(clsListBox, 7)


/****************************************************************************
 msgListBoxFindEntry			takes P_LIST_BOX_ENTRY, returns STATUS
	Finds the position of the given entry window/data.

 If pArgs->win is non-null, clsListBox searches for an entry whose window
 matches pArgs->win.  If pArgs->win is null, then clsListBox searches
 for an entry whose data fields matches pArgs->data.

 Return Value
	stsListBoxEmpty:	the list box has no entries
	stsNoMatch:			the list box had no matching entry
*/
#define msgListBoxFindEntry	 				MakeMsg(clsListBox, 8)


/****************************************************************************
 msgListBoxEnum					takes P_LIST_BOX_ENTRY_ENUM, returns STATUS
	Enumerates the entries of a listBox according to the given flags.
*/
#define msgListBoxEnum                      MakeMsg(clsListBox, 9)


/****************************************************************************
 msgListBoxEntryIsVisible		takes P_LIST_BOX_ENTRY, returns STATUS
    Passes back the visibility of an entry in a listBox.

 Sets the 'arg' field to zero if not visible, non-zero if visible.
 If the position is maxU16, then uses pArgs->win instead.
*/
#define msgListBoxEntryIsVisible            MakeMsg(clsListBox, 10)


/****************************************************************************
 msgListBoxXYToPosition			takes P_LIST_BOX_POSITION_XY, returns STATUS
	Gets the position for a given listBox window coordinate. 

 This message resolves positions only to currently visible entry windows.
 It does not attempt to interpolate arbitrary coordinates to positions.

 pArgs->place should be in the listBox's clientWin space.

 Return Value
	stsNoMatch:		the place did not intersect any currently visible
					entry windows
*/
#define msgListBoxXYToPosition	 			MakeMsg(clsListBox, 11)


/****************************************************************************
 msgListBoxMakeEntryVisible		takes P_LIST_BOX_ENTRY, returns STATUS
	Makes the specified entry visible.

 If the specified position is maxU16, msgListBoxFindEntry is first used
 to find the given window.  If the position is not visible, it will be
 scrolled so that its top is at the center of the view.  Otherwise, the
 minimum amount is scrolled.  No msgWinUpdate is required.
*/
#define msgListBoxMakeEntryVisible			MakeMsg(clsListBox, 12)



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Self-Sent/Client Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/*
 All of the messages in this section are first sent to the listBox
 itself.  This is so that subclasses of clsListBox may intercept these
 messages and process them as desired.  If these messages reach the
 clsListBox message handler, they will be forwarded on to the listBox
 client.
*/

/****************************************************************************
 msgListBoxProvideEntry			takes P_LIST_BOX_ENTRY, returns STATUS
	category: self-sent/client responsibility
	Self-sent when a listBox needs a window for display.

 The client should pass back a window UID in the win field.  The client
 should also set the freeEntry, state, and data fields as desired.
 Note that the state and data fields have no meaning to clsListBox; they're
 uninterpreted fields for the client to use for any purpose.

 A listBox will send this message even when the position it's asking for
 is >= the number of entries specified for the listBox.  In this case,
 the client is free to return a non-zero status value, indicating to
 the listBox that no entry should be created for that position.  Providing
 another entry window in this case allows the user to create new entries
 by merely scrolling past the end of the list.

 If the message reaches the standard listBox message procedure, the 
 listBox will forward the message to the client.  This scheme allows
 subclasses of clsListBox to handle the message in a different way.
*/
#define msgListBoxProvideEntry				MakeMsg(clsListBox, 13)


/****************************************************************************
 msgListBoxDestroyEntry			takes P_LIST_BOX_ENTRY, returns STATUS
	category: self-sent/client responsibility
	Sent to the client for an entry that has lbFreeDataByMessage enabled.

 The client should destroy the entry win and free any storage pointed to
 by the entry's 'data' field.
*/
#define msgListBoxDestroyEntry				MakeMsg(clsListBox, 14)


/****************************************************************************
 msgListBoxEntryGesture    		takes P_LIST_BOX_ENTRY, returns STATUS
	category: self-sent/client responsibility
	Notifies the subclass / client that a gesture occurred over an entry.

 The 'arg' field contains a P_GWIN_GESTURE pointer.

 If the position is maxU16, this means that the listbox currently has
 no entry windows.  Any other value indicates the position of the entry
 window to which the gesture is directed.  The listbox will use
 msgGWinTransformGesture to translate the coordinates in the GWIN_GESTURE
 to be relative to the entry window.

 The listbox returns (from its msgGWinGesture/msgGWinForwardedGesture handler)
 the status resulting from self-sending msgListBoxEntryGesture.  This means
 that the client should return stsOK, stsRequestDenied, or stsRequestForward
 as appropriate.  See gwin.h.
*/
#define msgListBoxEntryGesture				MakeMsg(clsListBox, 15)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from Other Classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgWinStartPage takes pNull, returns STATUS
    category: advisory message
    Advises window that it is on a printer, and printing is about to start.

 clsListBox responds by ensuring that its clientWin is appropriately
 populated with entry windows.
*/


/****************************************************************************
 msgWinGetBaseline takes P_WIN_METRICS, returns STATUS
    Gets the desired x,y alignment of a window.

 clsListBox will set pArgs->bounds.origin.x to 0.  If there is an entry
 window visible, pArgs->bounds.origin.y is set to:
	<top of scrollWin's inner window> - <row height>
	+ <y baseline of first visible entry window>
 If no entry window is visible, the y is set to 0.
*/


#endif	// LISTBOX_INCLUDED

