/***********************************************************************
 mil.h
 Copyright 1992, GO Corporation, All Rights Reserved

 The definition of the public Machine Interface Layer (a.k.a. MIL).



 "A foolish consistency is the hobgoblin of little minds..." - Emerson



 $Revision:   1.46  $
   $Author:   CPAYNE  $
     $Date:   19 Mar 1992 11:03:34  $

************************************************************************/
#ifndef MIL_INCLUDED
#define MIL_INCLUDED

#ifndef GO_INCLUDED
// use <> form for private version of go.h in PenPoint boot program
#include <go.h>
#endif

#ifndef UID_INCLUDED
#include "uid.h"
#endif

// PenPoint machine types - assigned by GO Corp. to provide unique Ids
// for each machine type of each manufacture.
#define machineTypeUnknown	MakeTag(theMILMachineType, 0)
#define machineTypePC		MakeTag(theMILMachineType, 1)
#define machineTypeGoOne	MakeTag(theMILMachineType, 2)
#define machineTypeNCR3125	MakeTag(theMILMachineType, 3)
#define machineTypeIBM2521	MakeTag(theMILMachineType, 4)
#define machineTypeSamsung1	MakeTag(theMILMachineType, 5)
#define machineTypeGrid1	MakeTag(theMILMachineType, 6)

#define milVersionStringResId MakeWknResId (theMIL, 0)

// In general this is used to flag unknown logical device Ids
#define unknownMILDevice maxU16

// Device type Ids - the assignment and administration of these Ids is
// controlled by GO Corp. Some spare Ids for testing of new device types
// are provided at the end of this list.

// a dummy device defining base functionality
#define milDeviceBase			clsMILBaseDevice

// At least one device of each of these types is required to run PenPoint:
#define milDeviceInit			clsMILInitDevice
#define milDevicePower			clsMILPowerDevice
#define milDeviceTimer			clsMILTimerDevice
#define milDeviceRealTimeClock	clsMILRealTimeClockDevice
#define milDeviceInterrupt		clsMILInterruptDevice
#define milDeviceScreen		clsMILScreenDevice
#define milDeviceStylus		clsMILStylusDevice
#define milDeviceNMI			clsMILNMIDevice

// PenPoint supports (but does not require) these additional device types:
#define milDeviceSound			clsMILSoundDevice
#define milDeviceKeyboard 		clsMILKeyboardDevice
#define milDeviceAsyncSIO		clsMILAsyncSIODevice
#define milDeviceParallelPort	clsMILParallelPortDevice
#define milDeviceAppleLAP		clsMILAppleLAPDevice
#define milDeviceNVMem			clsMILNVMemDevice
#define milDeviceSCSI			clsMILSCSIDevice
#define milDeviceFlash			clsMILFlashDevice
#define milDeviceCompression	clsMILCompressionDevice
#define milDeviceDebug			clsMILDebugDevice
#define milDeviceHSPacket		clsMILHSPacketDevice

// The generic block-structured device and its functional supersets
#define milDeviceBlock			clsMILBlockDevice
#define milDeviceFDisk			clsMILFDiskDevice
#define milDeviceDiskette		clsMILDisketteDevice
#define milDeviceFlashDisk		clsMILFlashDiskDevice
#define milDeviceMemoryCard	clsMILMemoryCardDevice

// These device Ids may be used for temporary testing of new device types.
// Code using these device types SHOULD NEVER BE RELEASED.
#define milDeviceTest1			clsMILTest1Device
#define milDeviceTest2			clsMILTest2Device
#define milDeviceTest3			clsMILTest3Device

// Normal and warning statuses returned by the MIL
#define stsMILStageOnInterrupt			MakeWarning(theMIL, 1)
#define stsMILStageOnTime				MakeWarning(theMIL, 2)
#define stsMILStageOnVirtual8086		MakeWarning(theMIL, 3)
#define stsMILNotMyInterrupt			MakeWarning(theMIL, 4)
#define stsMILCorrectedWriteError		MakeWarning(theMIL, 5)
#define stsMILCorrectedReadError		MakeWarning(theMIL, 6)
#define stsMILSkipThisDevice			MakeWarning(theMIL, 7)
#define stsMILStageOnAcquireDeviceSema	MakeWarning(theMIL, 8)
#define stsMILStageOnReleaseDeviceSema	MakeWarning(theMIL, 9)
#define stsMILStageOnPowerUp			MakeWarning(theMIL, 10)
#define stsMILStageOnIntOrPwrUp			MakeWarning(theMIL, 11)
#define stsMILStageOnTimeOrPwrUp		MakeWarning(theMIL, 12)

#define	stsMILRequestCancelled			MakeWarning(theMIL, 13)

// Error statuses returned by the MIL
#define stsMILInternalError				MakeStatus(theMIL, 0)
#define stsMILNotSupported				MakeStatus(theMIL, 1)
#define stsMILInvalidCommonData			MakeStatus(theMIL, 2)
#define stsMILInvalidInterrupt			MakeStatus(theMIL, 3)
#define stsMILInvalidLogicalId			MakeStatus(theMIL, 4)
#define stsMILInvalidUnitId				MakeStatus(theMIL, 5)
#define stsMILInvalidFunctionCode		MakeStatus(theMIL, 6)
#define stsMILInvalidReqBlk				MakeStatus(theMIL, 7)
#define stsMILInvalidReqBlkLength		MakeStatus(theMIL, 8)
#define stsMILInvalidReqBlkPtr			MakeStatus(theMIL, 9)
#define stsMILInvalidParameter			MakeStatus(theMIL, 10)
#define stsMILInvalidStage				MakeStatus(theMIL, 11)
#define stsMILUnexpectedStage			MakeStatus(theMIL, 12)
#define stsMILUnexpectedInterrupt		MakeStatus(theMIL, 13)
#define stsMILUnexpectedTimeout			MakeStatus(theMIL, 14)
#define stsMILUnexpectedV8086Req		MakeStatus(theMIL, 15)
#define stsMILDeviceBusy				MakeStatus(theMIL, 16)
#define stsMILHardwareError				MakeStatus(theMIL, 17)
#define stsMILReadError					MakeStatus(theMIL, 18)
#define stsMILWriteError				MakeStatus(theMIL, 19)
#define stsMILWriteProtectError			MakeStatus(theMIL, 20)
#define stsMILMediaChangedError			MakeStatus(theMIL, 21)
#define stsMILDeviceNotReadyError       MakeStatus(theMIL, 22)
#define stsMILBlockSeekError            MakeStatus(theMIL, 23)
#define stsMILBadAddressMarkError       MakeStatus(theMIL, 24)
#define stsMILBlockNotFound				MakeStatus(theMIL, 25)
#define stsMILDMAOverrun				MakeStatus(theMIL, 26)
#define stsMILDMAOver64KBoundary		MakeStatus(theMIL, 27)
#define stsMILBadCRCError				MakeStatus(theMIL, 28)
#define stsMILFormatUnknown				MakeStatus(theMIL, 29)
#define	stsMILRequestTimeOut			MakeStatus(theMIL, 30)
#define	stsMILNoRequestBlock			MakeStatus(theMIL, 31)
#define	stsMILNoDeviceAttached			MakeStatus(theMIL, 32)

// This macro calculates the size of a structure whose last element is a
// variable-size array. *** NOTE WELL *** - the ASSUMPTION is made that
// the base structure was declared with ONE element:
//
//		typedef struct A_STRUCTURE {
//			U16 field1;
//			U32 field2;
//			ELEMENT_TYPE array[1];  // ONE must be used here
//		};
//
//	MilStructSize(A_STRUCTURE, ELEMENT_TYPE, 20) sizes one with 20 elements
//
#define MilStructSize(baseStruct, arrayElement, elements) \
	(SizeOf(baseStruct) + SizeOf(arrayElement) * (elements-1))

/***********************************************************************
 A circular buffer of bytes. "pStart" and "pEnd" define the upper
 and lower (inclusive) limits to the buffer. "pIn" points to the next 
 free location to store a byte into. "pOut" points to the next byte
 to remove UNLESS it is equal to "pIn" in which case the buffer is empty.
 If the next value for "pIn" (with wrap) would be equal to "pOut" then 
 the buffer is full. Note that this means that you can actually only put
 "bufSize-1" bytes into a buffer of length "bufSize". "bufSize" is 
 simply "pEnd - pStart + 1" and saves recalculating this over and over.
 The initial state for an empty buffer is:

     pIn = pOut = pStart;

***********************************************************************/

typedef struct MIL_CIRCULAR_BYTE_BUFFER {
	P_U8	pStart;
	P_U8	pEnd;
	U32		bufSize;
	P_U8	pIn;
	P_U8	pOut;
} MIL_CIRCULAR_BYTE_BUFFER, *P_MIL_CIRCULAR_BYTE_BUFFER;

#define MilCBufPop(cBuf) \
	((cBuf)->pIn-(cBuf)->pOut>=0?\
	(cBuf)->pIn-(cBuf)->pOut:\
	(cBuf)->bufSize+(cBuf)->pIn-(cBuf)->pOut)

#define MilCBufIncPtr(cBuf, ptr) \
	if ((cBuf)->ptr++ == (cBuf)->pEnd) \
		(cBuf)->ptr = (cBuf)->pStart

#define MilCBufEmpty(cBuf) ((cBuf)->pIn == (cBuf)->pOut)

/*************************************************************************
 This macro defines the portion of a device block common to all.

 The "pUnitAttributeList" field points to an array of "numUnits" entries
 containing descriptions of each unit. Each descriptor holds a fixed
 (i.e. non-localizable) name string, the tag to a string in a resource 
 file which can be localized, and a "conflict group". 
 
 This last field is used to allow higher levels of software to discover
 which units of which devices share some underlying piece of hardware or
 resource. For example the debug device may be using the same serial port
 as the async device. Any units in the MIL which can not be used 
 simultaneously due to some shared resource should have the same tag in 
 this field. Otherwise it should be zero. The tags should be defined from
 the predefined UIDs in UID.H: theMILConflictGroup<n>.
**************************************************************************/

#define milDeviceBlockCommonFields \
	SIZEOF deviceBlockLength; \
	U16 revision; \
	U16 secondaryDeviceId; \
	U16 logicalId; \
	UID deviceId; \
	UID subDeviceId; \
	U16 numUnits; \
	SIZEOF minDevReqBlkLength; \
	U16 memoryClass; \
	U32 interruptLevels; \
	MIL_DEVICE_POWER_STATE devicePowerState; \
	U32 deviceFlags; \
	P_MIL_UNIT_ATTRIBUTES pUnitAttributeList; \
	U32	defaultPowerTimeout; \
	U32 powerTimeoutCounter; \
	U32 powerManagerSpareU32; \
	U32 exclusiveAccessSema; \
	U16 reservedU16[2]; \
	U32 reservedU32[2];

// All devices support the devPowerOn and devPowerStandby states
Enum16(MIL_DEVICE_POWER_STATE) {
	milDevPowerOn,		// device is fully functional
	milDevPowerStandby,	// device can detect attachment, disk insertion
	milDevPowerOff,		// device can only power-up
	milDevPowerSuspend,	// save HW state for restore on devPowerOn
	milDevPowerResume	// restore HW state after devPowerSuspend 
};

#define milMaxFixedUnitNameLen 31
#define milFixedUnitNameBufferSize (milMaxFixedUnitNameLen+1)

typedef CHAR MIL_FIXED_UNIT_NAME[milFixedUnitNameBufferSize];
typedef MIL_FIXED_UNIT_NAME *P_MIL_FIXED_UNIT_NAME;

typedef struct MIL_UNIT_ATTRIBUTES {
	MIL_FIXED_UNIT_NAME	fixedName;
	TAG					unitResourceTag;
	UID					conflictGroup;
	// These two fields belong to the service manager. It can store whatever
	// it wants in these fields, no one else will read or write them.
	OBJECT				serviceInstance;
	P_UNKNOWN			pServiceReserved;
	U16 				reservedU16[2];
	U32 				reservedU32[2];
} MIL_UNIT_ATTRIBUTES, *P_MIL_UNIT_ATTRIBUTES;
	
// flags used in deviceFlags
#define milDevFlagUsesV8086Int		flag2
#define milDevFlagDetachable		flag3
#define milDevFlagDetached			flag4
#define milDevFlagNeedsHousekeeping	flag5
#define milDevFlagSupportsOff		flag6
#define milDevFlagSupportsStandby	flag7

typedef struct MIL_DEVICE_BLOCK {
	milDeviceBlockCommonFields
} MIL_DEVICE_BLOCK, *P_MIL_DEVICE_BLOCK;

// place holder for forward defined struct
struct MIL_FTT_BASE;

typedef struct MIL_DEVICE_POINTERS {
	P_MIL_DEVICE_BLOCK	 	pDeviceBlock;
	struct MIL_FTT_BASE	*pFuncTransTable;
} MIL_DEVICE_POINTERS, *P_MIL_DEVICE_POINTERS;

// A union used to pass a few U8's or a pointer to a large buffer of U8's.
// Because C does not allow us to cleanly mix unions and structs in the same
// declaration the variable indicating how to interpret the union is declared
// seperately.
//
// Example:
//	struct s {
//		MIL_PTR_OR_BUF data;
//		BOOLEAN dataIsBuffer;
//		U32 bufferSize;
//	} aStruct;
//
//	void Print(U32 length, P_U8 pBuffer);
//
//	Print(
//		aStruct.bufferSize,
//		(aStruct.dataIsBuffer ? aStruct.data.buffer: aStruct.data.pBuffer) 
//	);

typedef union MIL_PTR_OR_BUF {
	P_U8 pBuffer;
	U8 buffer[4];
} MIL_PTR_OR_BUF, *P_MIL_PTR_OR_BUF;

// ZZZ the obsolete fields will be removed in the next build
typedef struct MIL_V8086_INFO {
	U32 initCode;			// Real-mode address CS:IP
	U32 stackPointer;		// Initial SS:SP for V8086 code
	U32 obsoleteA;			// buffer: A DMA-able buffer. NULL if not present
	U32 obsoleteB;			// bufLength: The length of the above
	U16 interruptNum;		// The interrupt that the v8086 task uses to idle.
	U32 intVectTable[16];	// real mode interrupt vectors
} MIL_V8086_INFO, *P_MIL_V8086_INFO;

typedef U32 MIL_PHYSICAL_ADDR, *P_MIL_PHYSICAL_ADDR;

typedef P_UNKNOWN FunctionPtr(P_PHYSICAL_TO_LOGICAL_FUNC) (
	MIL_PHYSICAL_ADDR	physAddr,	// address of memory mapped area
	U32 		length 		// length of memory to map
);

typedef MIL_PHYSICAL_ADDR FunctionPtr(P_LOGICAL_TO_PHYSICAL_FUNC) (
	P_UNKNOWN	pMem,
	U32 		length		// length of memory which contiguous
);

Enum16(MIL_MEMORY_TYPE) {
	milMemTypeNone,
	milMemTypeOther,
	milMemTypeRAM,
	milMemTypeNonVolatileRAM,
	milMemTypeDeviceBufferRAM,
	milMemTypeROM,
	milMemTypeExtensionROM,
	milMemTypeFlash
};

typedef struct MIL_MEMORY_REGION {
	U32 lowLimit;
	U32 highLimit;
	U32 granularity;	// for flash this defines the unit of "eraseability"
	MIL_MEMORY_TYPE memoryType;
	U16 memoryClass;
} MIL_MEMORY_REGION, *P_MIL_MEMORY_REGION;

// Well known memory classes
#define anyMemClass				0
#define dmaMemClass				1

// the common data can be (somewhat) validated by checking for 
// this signature "CMIL" (assumes Intel-style byte-ordering)
#define validCommonDataSignature 0x4C494D43

// The common data that is passed with all calls to the MIL.
typedef struct MIL_COMMON_DATA {
	U32 commonDataSignature;
	SIZEOF minReqBlkLength;
	U16 numLogicalIds;
	U16 maxLogicalIds;

	P_MIL_V8086_INFO pV8086Info;	// NULL if v8086 not used

	TAG machineType;				// unique for each manufacture and 
									// machine type
	U32 machineSerialNumber;		// home for optional serial number
									// NOT REQUIRED!

	U16 bootDevice;
	U16 bootUnit;

	P_PHYSICAL_TO_LOGICAL_FUNC pPhysicalToLogicalFunc;
	P_LOGICAL_TO_PHYSICAL_FUNC pLogicalToPhysicalFunc;

	U16 numMemoryRegions;
	P_MIL_MEMORY_REGION pMemoryRegions;

	U32 privateDataSize;
	P_UNKNOWN pPrivateData;

	U16 reservedU16[2];
	U32 reservedU32[2];

	MIL_DEVICE_POINTERS devicePtrs[1];

} MIL_COMMON_DATA, *P_MIL_COMMON_DATA;

// *** Note *** - there is code that assumes that milReqComplete equals 0
typedef U16 MIL_REQ_STAGE, * P_MIL_REQ_STAGE;
enum MIL_REQ_STAGE {
	milReqComplete = 0,
	milReqStageOnInterrupt = 1,
	milReqStageOnTime = 2,
	milReqStageOnVirtual8086 = 3,
	milReqStageOnAcquireDeviceSema = 4,
	milReqStageOnReleaseDeviceSema = 5,
	milReqStageOnPowerUp = 6,
	milReqStageOnIntOrPwrUp = 7,
	milReqStageOnTimeOrPwrUp = 8
};

#define maxReqStage milReqStageOnTimeOrPwrUp

// place holder for forward defined struct
struct MIL_REQUEST_BLOCK;

typedef struct MIL_REQUEST_BLOCK *P_MIL_REQUEST_BLOCK;

// A pointer to a async-event function for continuous, multi-staged functions
typedef void FunctionPtr(P_MIL_ASYNC_EVENT_FUNC) (
	P_MIL_COMMON_DATA pCommonData,
	P_MIL_REQUEST_BLOCK pRB
);	

// These structures are used to communicate with a virtual 8086 task. A 
// typical operation would entail one or more of the following steps:
//
//		- copying some data from protected to real memory
//		- invoking a real-mode interrupt or calling a FAR subroutine
//		- copying some data from real to protected memory

typedef union MIL_V8086_PTR {
	U32 all;
	U16 offset, segment;
} MIL_V8086_PTR, *P_MIL_V8086_PTR;

typedef struct MIL_V8086_REGS {
	U16 cflag;
	U16 ax;
	U16 bx;
	U16 cx;
	U16 dx;
	U16 si;
	U16 di;
	U16 ds;
	U16 es;
} MIL_V8086_REGS, *P_MIL_V8086_REGS;

typedef struct MIL_INTER_MODE_MOVE_PTRS {
	P_UNKNOWN pProtectedData;
	MIL_V8086_PTR pV8086Data;  // seg:off
	U32 dataLength;
} MIL_INTER_MODE_MOVE_PTRS, *P_MIL_INTER_MODE_MOVE_PTRS;

// This structure will exist in any MIL request block that is returned
// with a status of stsMILStageOnVirtual8086. For such a request block
// field pMILV8086Request points to this structure.

typedef struct MIL_V8086_REQUEST {

	// Both pointers and the length in this structure are nil if 
	// there is nothing to move in
	MIL_INTER_MODE_MOVE_PTRS toV8086;

	// Both pointers and the length in this structure are nil if 
	// there is nothing to move out
	MIL_INTER_MODE_MOVE_PTRS fromV8086;

	MIL_V8086_REGS regs;		// in:out

	// Note that for a give request only one of the 
	// following two fields is used. The other should be zero
	MIL_V8086_PTR pSubroutine;	// FAR pointer (seg:off) 
	U8 interruptNum;

	STATUS status;

} MIL_V8086_REQUEST, *P_MIL_V8086_REQUEST;

typedef struct MIL_RB_LINK_ELEMENT {
	P_MIL_REQUEST_BLOCK		pReqBlock;
	struct MIL_RB_LINK_ELEMENT	*pPrevRBLink;
	struct MIL_RB_LINK_ELEMENT	*pNextRBLink;
} MIL_RB_LINK_ELEMENT, *P_MIL_RB_LINK_ELEMENT;

// flags used in REQUEST_BLOCK (reqBlkFlags)
#define	milDebugging		flag0	// Validate request blocks at each
									// DrvMILRequest() call.
#define	milNoTaskScheduling	flag1	// At ISR time, no task scheduling 
									// is to take place.  
									// Do NOT set if a semaphore is reset
									// within ISR (or call back from ISR)
									// for quickest task scheduling of
									// waiting task.

// The portion of a request block common to all.
// This code assumes that OS_SEMA_ID and U32 are the same size
#define milRequestBlockCommonFields \
	SIZEOF blockLength; \
	U32 reqBlkSignature; \
	U16 functionCode; \
	U16 logicalId; \
	U16 unit; \
	U32 timeout; \
	P_MIL_RB_LINK_ELEMENT pRBLinkElements; \
	U16 reqBlkFlags; \
	U16 replacedU16; \
	U32 stagingSema; \
	P_MIL_V8086_REQUEST pMILV8086Request; \
	P_UNKNOWN pPrivateReqData; \
	U32 extendedErrorCode; \
	MIL_REQ_STAGE reqStage; \
	STATUS status; \
	U16 callerDataU16A; \
	U16 callerDataU16B; \
	U32 callerDataU32A; \
	U32 callerDataU32B; \
	U32	timerHandle; \
	U16 reservedU16[2]; \
	U32 reservedU32;

// any request block can be (somewhat) validated by checking for 
// this signature "REQB" (assumes Intel-style byte-ordering)
#define validRBSignature 0x42514552

typedef struct MIL_REQUEST_BLOCK {

	milRequestBlockCommonFields

} MIL_REQUEST_BLOCK;

// A pointer to a MIL Common Entry Point (Common Start, Next Stage or Cancel).
// ** NOTE ** This must match MILCommonXXXXX() declarations.
typedef STATUS FunctionPtr(P_MIL_COMMON_ENTRY) (
	P_MIL_COMMON_DATA pCommonData,
	P_MIL_REQUEST_BLOCK pRB
);	

typedef void FunctionPtr(P_MIL_DEBUG_STR_OUT) (P_STRING pString);

typedef void FunctionPtr(P_MIL_DEBUG_BEEP) (void);

typedef void FunctionPtr(P_MIL_DEBUG_BUF_OUT) (P_U8 pBuf, U32 len);

typedef void FunctionPtr(P_MIL_DEBUG_NUM_OUT) (U32 num, U8 base, U8 width);

// a pointer to a Device Entry Point (Device function, Next Stage or Cancel)
typedef STATUS FunctionPtr(P_MIL_DEVICE_ENTRY) (
	P_MIL_COMMON_DATA,
	P_UNKNOWN,				// P_MIL_RB_ALL_<dev>_<func>
	P_UNKNOWN				// P_MIL_DEV_BLK_<dev>
);	

// The structure of a function descriptor used in a function transfer table.
// The function flags allow a caller to anticipate the behavior of a function.
// This simplifies the implemenation of common MIL request driver code in 
// PenPoint.

// The function is DEFINED by the MIL specification to be a 
// continuous function
#define milContFunc				flag0

// The function is implemented as, or defined by the MIL specification to be,
// a single-stage function
#define milSingleStageFunc			flag1

// The function is implemented with (any or all) stages of this/these type(s):
//    Stage-On-Interrupt
//    Stage-On-Time
//    Stage-On-Virtual-8086
//    Stage-On-Acquire-Device-Sema
//    Stage-On-Release-Device-Sema
//    Stage-On-Power-Up
//    Stage-On-Int-Or-Power-Up
//    Stage-On-Time-Or-Power-Up

#define milStageOnInterruptFunc	flag2
#define milStageOnTimeFunc			flag3
#define milStageOnVirtual8086Func	flag4
#define	milStageOnDeviceSemaFunc	flag5
#define	milStageOnPowerUpFunc		flag6

#define	milMultiStageFunc (\
	milStageOnInterruptFunc | \
	milStageOnTimeFunc | \
	milStageOnVirtual8086Func | \
	milStageOnDeviceSemaFunc | \
	milStageOnPowerUpFunc)

typedef struct MIL_FUNCTION_DESC {
	SIZEOF minFuncReqBlkLength;
	U16 functionFlags;
	P_MIL_DEVICE_ENTRY deviceFunction;
} MIL_FUNCTION_DESC, *P_MIL_FUNCTION_DESC;


/***********************************************************************

 All MIL devices have entry-points for the following nine functions:

	milBaseInitDevice		- implemented by all
	milBaseResetDevice		- implemented by all
	milBaseGetParameters	- implemented by all
	milBaseClearInterrupt	- implemented by all interrupting devices
	milBaseDiagnostic		- implemented by all
	milBaseMachineSpecific	- optional
	milBaseSetDevPowerState	- implementation indicated by device flags
	milBaseHousekeepingCont	- implementation indicated by device flags
	milBaseAttachmentCont	- implementation indicated by device flags

 Unimplemented functions will return stsMILNotSupported.

***********************************************************************/

#define milBaseInitDevice             0
#define milBaseResetDevice            1
#define milBaseGetParameters          2
#define milBaseClearInterrupt         3
#define milBaseDiagnostic             4
#define milBaseMachineSpecific        5
#define milBaseSetDevPowerState       6
#define milBaseHousekeepingCont       7
#define milBaseAttachmentCont         8

#define milLastCommonFunction	milBaseAttachmentCont

#define fnCountBase 9

#define	funcTransTableCommonFields \
	P_MIL_DEVICE_ENTRY pDeviceNextStage; \
	P_MIL_DEVICE_ENTRY pDeviceCancelRequest; \
	U16 functionCount; \
	U16 reservedU16[2]; \
	U32 reservedU32[2];

typedef struct MIL_FTT_BASE {
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountBase];
} MIL_FTT_BASE, *P_MIL_FTT_BASE;

typedef void FunctionPtr(P_EXIT_TO_REAL_FUNC) (void);

// Struct passed from milInit to PenPointInit w/MIL entrypoints
typedef struct MIL_ENTRY_INFO {
	P_MIL_COMMON_DATA		pInitCommonData;
	P_MIL_COMMON_ENTRY		pMILCommonStart;
	P_MIL_COMMON_ENTRY		pMILCommonNextStage;
	P_MIL_COMMON_ENTRY		pMILCommonCancelRequest;
	P_MIL_DEBUG_STR_OUT		pMILDbgStrOut;
	P_MIL_DEBUG_BEEP		pMILDbgBeep;
	P_MIL_DEBUG_BUF_OUT		pMILDbgBufOut;
	P_MIL_DEBUG_NUM_OUT		pMILDbgNumOut;
	// More to be defined...
} MIL_ENTRY_INFO, *P_MIL_ENTRY_INFO;

// a pointer to a device data block initialization routine
typedef STATUS FunctionPtr(P_MIL_DEVICE_BLOCK_INIT) (
	U16 relativeLogicalId,
	P_MIL_COMMON_DATA pCommonData,
	P_UNKNOWN pDeviceBlock	// P_MIL_DEVICE_BLOCK
);	

typedef struct MIL_DEVICE_INFO {
	U16 numLogicalDevices;		// Out: number of logical devices of this
								//      device type id
	P_MIL_DEVICE_BLOCK_INIT		// Out: Pointer to routine to initialize the
		initDevBlkRoutine;   	//      Device data block
	U16 fTTLength;	  			// Out: SizeOf(FTT)
	P_UNKNOWN pFuncTransTable;	// Out: pointer to FTT 
	MIL_DEVICE_BLOCK
		commonDeviceBlock;		// Out: values for public device block fields
} MIL_DEVICE_INFO, *P_MIL_DEVICE_INFO;

#define milValidExtensionSignature 0x5458454d  // "MEXT", Intel-style

typedef struct MIL_EXTENSION {
	U32 extensionLength;
	U32 extensionSignature;
	U16 checksum;
	MIL_DEVICE_INFO deviceInfo;
	U8 extensionImage[1];
} MIL_EXTENSION, *P_MIL_EXTENSION;

/***********************************************************************

 Function milBaseInitDevice - used to initialize all devices.

***********************************************************************/

#define rbPublicFieldsBaseInitDevice \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

typedef struct MIL_RB_PUB_BASE_INIT_DEVICE {
	milRequestBlockCommonFields
	rbPublicFieldsBaseInitDevice
} MIL_RB_PUB_BASE_INIT_DEVICE, *P_MIL_RB_PUB_BASE_INIT_DEVICE;


/***********************************************************************

 Function milBaseResetDevice - used to reset any devices.

***********************************************************************/

#define rbPublicFieldsBaseResetDevice \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

typedef struct MIL_RB_PUB_BASE_RESET_DEVICE {
	milRequestBlockCommonFields
	rbPublicFieldsBaseResetDevice
} MIL_RB_PUB_BASE_RESET_DEVICE, *P_MIL_RB_PUB_BASE_RESET_DEVICE;

/***********************************************************************

 Function milBaseGetParameters - this single-stage function returns the 
 attributes the specified unit and information about the specified 
 function and the public portion of a device block.

***********************************************************************/

#define rbPublicFieldsBaseGetParameters \
	U16 queryFunctionCode; \
	MIL_UNIT_ATTRIBUTES unitAttributes; \
	SIZEOF minFuncReqBlkLength; \
	U16 functionFlags; \
	MIL_DEVICE_BLOCK devBlk; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

typedef struct MIL_RB_PUB_BASE_GET_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsBaseGetParameters
} MIL_RB_PUB_BASE_GET_PARAMETERS, *P_MIL_RB_PUB_BASE_GET_PARAMETERS;

/***********************************************************************

 Function milBaseClearInterrupt - used by PenPoint to clear an unexpected
 interrupt from any device.

***********************************************************************/

#define rbPublicFieldsBaseClearInterrupt \
	U16	interruptLevel; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

typedef struct MIL_RB_PUB_BASE_CLEAR_INTERRUPT {
	milRequestBlockCommonFields
	rbPublicFieldsBaseClearInterrupt
} MIL_RB_PUB_BASE_CLEAR_INTERRUPT, *P_MIL_RB_PUB_BASE_CLEAR_INTERRUPT;

/***********************************************************************

 Function milBaseDiagnostic - performs destructive or non-detructive
 diagnostics.

***********************************************************************/

#define rbPublicFieldsBaseDiagnostic \
	BOOLEAN destructive; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

typedef struct MIL_RB_PUB_BASE_DIAGNOSTIC {
	milRequestBlockCommonFields
	rbPublicFieldsBaseDiagnostic
} MIL_RB_PUB_BASE_DIAGNOSTIC, *P_MIL_RB_PUB_BASE_DIAGNOSTIC;

/***********************************************************************

 Function milBaseMachineSpecific - used by an OEM or machine specific
 program or service to access device internals.

***********************************************************************/

#define rbPublicFieldsBaseMachineSpecific \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

typedef struct MIL_RB_PUB_BASE_MACHINE_SPECIFIC {
	milRequestBlockCommonFields
	rbPublicFieldsBaseMachineSpecific
} MIL_RB_PUB_BASE_MACHINE_SPECIFIC, *P_MIL_RB_PUB_BASE_MACHINE_SPECIFIC;

/***********************************************************************

 Function milBaseSetDevPowerState - set a device to the specified power
 state: devPowerOn, devPowerStandby, devPowerOff or devPowerSuspend

***********************************************************************/

#define rbPublicFieldsBaseSetDevPowerState \
	MIL_DEVICE_POWER_STATE devPowerState; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

typedef struct MIL_RB_PUB_BASE_SET_DEV_POWER_STATE {
	milRequestBlockCommonFields
	rbPublicFieldsBaseSetDevPowerState
} MIL_RB_PUB_BASE_SET_DEV_POWER_STATE, *P_MIL_RB_PUB_BASE_SET_DEV_POWER_STATE;

/***********************************************************************

 Function milBaseHousekeepingCont - allows MIL devices to perform internal
 housekeeping staged by an interrupt or interval. Example: can be used to
 time-out floppy motors or erase ahead on flash-disks.

***********************************************************************/

#define rbPublicFieldsBaseHousekeepingCont \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

typedef struct MIL_RB_PUB_BASE_HOUSEKEEPING_CONT {
	milRequestBlockCommonFields
	rbPublicFieldsBaseHousekeepingCont
} MIL_RB_PUB_BASE_HOUSEKEEPING_CONT, *P_MIL_RB_PUB_BASE_HOUSEKEEPING_CONT;

/***********************************************************************

 Function milBaseAttachmentCont - this function monitors and reports
 (via a call-back function) changes in a devices attachment state

***********************************************************************/

Enum16(MIL_DEV_ATTACHMENT_EVENT) {
	milDevAttachedEvent,
	milDevDetachedEvent,
	milDevMediaInsertedEvent,
	milDevMediaRemovedEvent,
	// for memory cards containing back-up batteries
	milDevBatteryLowEvent,	
	milDevBatteryFailedEvent
};

#define rbPublicFieldsBaseAttachmentCont \
	P_MIL_ASYNC_EVENT_FUNC pAsyncEventFunc; \
	MIL_DEV_ATTACHMENT_EVENT event; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

typedef struct MIL_RB_PUB_BASE_ATTACHMENT_CONT {
	milRequestBlockCommonFields
	rbPublicFieldsBaseAttachmentCont
} MIL_RB_PUB_BASE_ATTACHMENT_CONT, *P_MIL_RB_PUB_BASE_ATTACHMENT_CONT;

#endif


