/****************************************************************************
 pen.h

 Copyright 1992, GO Corporation, All Rights Reserved

 $Revision:   1.11  $
   $Author:   kbentley  $
	 $Date:   26 Mar 1992 17:32:12  $

 This file contains the API definition for the pen driver.

 clsPen inherits from clsObject.

 The functions described in this file are contained in INPUT.LIB. 

 This file contains information about pen-generated input events.  See
 input.h for general information on PenPoint's input system and input
 events.  You should probably read at least the "Road Map" section of
 input.h before trying to understand this file in detail.
****************************************************************************/

/**** Pen Events ****/
/*
 When the pen generates input events, the events are delivered via
 msgInputEvent.  The following values are the value of pArgs->devCode for
 msgInputEvent.

    msgPenUp:			sent when the pen tip is lifted from the screen.
    msgPenDown:			sent when the pen tip touches the screen.
    msgPenMoveUp:		sent as the pen moves while above the screen and in
                        proximity.
    msgPenMoveDown:		sent as the pen moves while touching the screen.
    msgPenEnterUp:		sent when the pen enters a window while above the
                        screen and in proximity.
    msgPenEnterDown:	sent when the pen enters into a window while touching
                        the screen.
    msgPenExitUp:		sent when the pen exits a window while above the
						screen and in proximity.
    msgPenExitDown:		sent when the pen exits a window while touching the
                        screen.
    msgPenInProxUp:		sent when the pen comes into proximity.  This
                        message is also sent when certain timeouts occur; 
                        see the section "Proximity Timeout Events" for more
                        information.
    msgPenOutProxUp:	sent when the pen leaves proximity.  This message is
                        also sent when certain timeouts occur;  see the
                        section "Proximity Timeout Events" for more 
						information.
    msgPenStroke:		sent with the collected stroke data.  See the "Stroke
                        Events" section.
    msgPenTap:			sent when taps are recognized by the driver. See the
                        "Tap Events" section.  The taps field of PEN_DATA
                        contains the number of taps.
    msgPenHoldTimeout:	sent after pen down and hold timeout. See the
                        "Hold Timeout Events" section.  The taps field of
                        PEN_DATA contains the number of taps that occurred
                        before the Hold.

 [Terminology Note:  the msgPenInProxUp and msgPenOutProxUp events can be
 thought of as msgPenInProx and msgPenOutProx since the pen tip is always
 up when these events are sent.  The trailing "Up" is present for
 historical reasons only.]
*/


/**** Input Flags ****/
/*
 Pen events can be screened out using input flags.  See input.h for more
 information.  The relevant flags for pen are:
//{
    input flag         enables             see section 
    ================   =================   ================
    inputTip           msgPenUp
                       msgPenDown
    inputMoveUp        msgPenMoveUp
    inputMoveDown      msgPenMoveDown
    inputEnter         msgPenEnterUp
                       msgPenEnterDown
    inputExit          msgPenExitUp
                       msgPenExitDown
    inputInProx        msgPenInProxUp
    inputOutProx       msgPenOutProxUp
    inputStroke        msgPenStroke
    inputTap           msgPenTap
    inputHoldTimeout   msgPenHoldTimeout   "Hold Timeout Events"
//}
*/

/**** Enter Exit Window Events ****/
/*
 msgPenEnterUp, msgPenEnterDown, msgPenExitUp and msgPenExitDown are
 generated when the pen transits a window boundary.  The window that
 the pen was previously in will receive msgPenExitUp or msgPenExitDown
 (if its input flags request them).  The window that the pen is now in
 will receive msgPenEnterUp or msgPenEnterDown (if its input flags request
 them).  Note that if the pen leaves proximity, the window will receive
 a msgPenOutProxUp and not msgPenExitUp.  Similarly, if the pen enters
 proximity, the window will receive msgPenInProxUp and not msgPenEnterUp.

 The timestamp, strokeSeqNum and penXY field of the PEN_DATA structure
 will be valid.  All other fields will be 0.

*/


/**** Hold Timeout Events ****/
/*
 msgPenHoldTimeout events are generated when the user puts the pen on the
 display and leaves it there for the "Hold" timeout duration.  This message
 is also generated if the user taps 1 or more times before holding the pen
 down.
 
 For example, msgPenHoldTimeout is the event that triggers PenPoint's move
 and copy, and is also used by some applications to trigger wipe-through
 area selection.
 
 msgPenHoldTimeout events are sent if the window's input flags have the
 inputHoldTimeout flag set.

 The strokeSeqNum field of the PEN_DATA structure will be the sequence
 number of the most recent pen down.  The penXY field of the PEN_DATA
 structure will be the pen device coordinates of the first pen down.

 will be valid.  All other fields will be 0.


*/


/**** Proximity Timeout Events ****/
/*
 The input system also has a proximity-related timeout.  These are used
 if the user lifts the pen off the display but leaves the pen in proximity.
 
 This timer is typically used to trigger translation because some users
 don't lift their pen tips out of proximity but still want translation to
 occur.

 If this timer expires with the pen still in proximity, the
 input system sends msgPenOutProxUp, followed by msgPenInProxUp.  In other
 words, the input system generates events to make it look like the user
 temporarily lifted the pen out of proximity.

 [Note: the obsolete messages msgPenTimeout and msgPenHWTimeout are not
 sent.]
*/


/**** Stroke Events ****/
/*
 Each time the pen goes down, moves, and comes up, the input system
 generates msgInputEvent with a pArgs->devCode of msgPenStroke.  The pArgs
 also includes a compressed representation of the stroke.

 One way to think about a stroke event is as a "summary" or "reprise" of
 msgPenDown, zero or more msgPenMoveDowns, and a msgPenUp.  
					
 Stroke events are delivered to the window in which the stroke started (if
 that window has the input flag inputStroke flag set), even if the stroke
 crosses that window's boundaries.

 Stroke events are generated in addition to the other, lower level,
 messages that occur as the stroke event is being accumulated.  Typical
 clients either handle msgPenStroke or the lower-level messages
 (msgPenDown, msgPenMoveDown, msgPenUp), but NOT both.

 The input system assigns a sequence number to each stroke.  Each pen event
 contains the stroke number that the event is a part of.  This number is
 found in the "strokeSeqNum" field of PEN_DATA.

 See the "Sample Code" section for an example of how to extract stroke
 information from the pArgs of a stroke event.
*/


/**** Tap Events ****/
/*
 A msgPenTap is generated if there is a msgPenDown followed by a msgPenUp
 and (1) any pen motion between the Down and Up is below some threshold and
 (2) the Down and Up happen within a certain time interval and (3) the Down
 and Up occur over the same window and (4) no other input event (excepting
 an optional Out of Proximity) event happens within a certain time
 threshold of the Up.

 msgPenTap is sent if the input flag inputTap is set.

 If the pen is "tapped" repeatedly, a single msgPenTap is sent and the taps
 field of PEN_DATA contains the number of pen taps.

 The strokeSeqNum field of the PEN_DATA structure will be the sequence
 number of the most recent pen down.  The penXY field of the PEN_DATA
 structure will be the pen device coordinates of the first pen down.


*/


/**** Typical Sequences of Events ****/
/*
 This sections illustrates some typical sequences of pen events.  It does
 not include tap, timeout and stroke events.  It also does not show
 forwarding up the window parent chain.

 This table contains the flow of events if the pen comes down, moves
 around, and comes back up, all within a single window.
//{
    quantity    event
    =========   ===============
    1           msgPenInProxUp
    0 or more   msgPenMoveUp
    1           msgPenDown
    0 or more   msgPenMoveDown
    1           msgPenUp
    0 or more   msgPenMoveUp
    1           msgPenOutProxUp
//}

 This sequence is complicated if the pen crosses a window boundary while
 moving.  When this happens, the input system generates Enter and Exit
 events to notify the windows being entered and exited.  In the following
 example, assume that the window hierarchy isn't changing and that the pen
 crosses a window boundary between windows A and B while the pen is down.
//{
    quantity   events seen by A   events seen by B
    =========   ================   ================
    1           msgPenInProxUp
    0 or more   msgPenMoveUp
    1           msgPenDown
    0 or more   msgPenMoveDown
    1           msgPenExitDown
    1                              msgPenEnterDown
    1                              msgPenUp
    0 or more                      msgPenMoveUp
    1                              msgPenOutProxUp
//}

 If the pen goes down in window A and in response window A "pops up" a
 window B right where the pen is, and the user lifts the pen, the following
 sequence occurs:
//{
    quantity    events seen by A   events seen by B
    =========   ================   ================
    1           msgPenInProxUp
    0 or more   msgPenMoveUp
    1           msgPenDown
    1           msgPenExitDown
    1                              msgPenEnterDown
    1                              msgPenUp
    0 or more                      msgPenMoveUp
    1                              msgPenOutProxUp
//}
*/


/**** Sample Code ****/
/*
 You can verify that your msgInputEvent handler is handling a pen message
 by checking as follows:
//{
	if (ClsNum(pArgs->devCode) == ClsNum(clsPen)) {
//}

 Once you've decided that you're looking at a pen event, you can cast
 pArgs->eventData as follows:
//{
 	P_PEN_DATA	pPenData;
	pPenData = (P_PEN_DATA)(pArgs->eventData);
//}

 If pArgs->devCode is msgPenStroke, you can get a pointer to the stroke
 data as follows:
//{
	P_PEN_STROKE	pStroke;
	pStroke = (P_PEN_STROKE)((P_PEN_DATA)(pArgs->eventData))->data;
//}
*/

#ifndef PEN_INCLUDED
#define PEN_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef GEO_INCLUDED
#include <geo.h>
#endif

#ifndef INPUT_INCLUDED
#include <input.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Pen Event Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define msgPenUp			MakeMsg(clsPen, eventTipUp)
#define msgPenDown			MakeMsg(clsPen, eventTipDown)
#define msgPenMoveUp		MakeMsg(clsPen, eventMoveUp)
#define msgPenMoveDown		MakeMsg(clsPen, eventMoveDown)
#define msgPenEnterUp		MakeMsg(clsPen, eventEnterUp)
#define msgPenEnterDown		MakeMsg(clsPen, eventEnterDown)
#define msgPenExitUp		MakeMsg(clsPen, eventExitUp)
#define msgPenExitDown		MakeMsg(clsPen, eventExitDown)
#define msgPenInProxUp		MakeMsg(clsPen, eventInProxUp)
#define msgPenOutProxUp		MakeMsg(clsPen, eventOutProxUp)
#define msgPenStroke		MakeMsg(clsPen, eventStroke)
#define msgPenTap			MakeMsg(clsPen, eventTap)
#define msgPenHoldTimeout	MakeMsg(clsPen, eventHoldTimeout)


//REFGEN BEGINIGNORE
/*
 * The following are obsolete.  They are never sent.
*/
#define msgPenInProxDown	MakeMsg(clsPen, eventInProxDown)	// Obsolete
#define msgPenOutProxDown	MakeMsg(clsPen, eventOutProxDown)	// Obsolete
#define msgPenTimeout		MakeMsg(clsPen, eventTimeout)		// Obsolete
#define msgPenHWTimeout		MakeMsg(clsPen, eventHWTimeout)		// Obsolete
//REFGEN ENDIGNORE

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * All pen events report coordinates in standard pen resolution, which units
 * of 0.1 mm.
*/
#define penStdResolution	10000	//	lines per meter


/* Possible states of the pen tip. */
typedef U16 PEN_TIP_STATE_FLAGS, *P_PEN_TIP_STATE_FLAGS;
#define penOutOfProximity	0
#define penInProximity		flag0
#define penTipDown			flag1


/* Possible states of the pen tip. */
Enum16(PEN_TIP_STATE_TYPE) {
	penTipOutOfProxState = 0,
	penTipInProxState = 1,
	penTipDownState = 2
};


typedef U16 PEN_SUPPORTS_FLAGS, *P_PEN_SUPPORTS_FLAGS;
#define penSupportsProximity		flag0
#define penSupportsPressure			flag1		// For future use.
#define penSupportsZPosition		flag2		// For future use.
#define penSupportsZAngle			flag3		// For future use.
#define penSupportsXYAngle			flag4		// For future use.
#define penSupportsPenId			flag5		// For future use.
#define penSeparateFromScreen		flag6		// digitizer is not 
												// integrated with display.
#define penDataIsStroke				flag7		// For future use.
#define penSupportsInk				flag12		// For future use.
#define penSupportsStrokes			flag13		// For future use.

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					msgInputEvent Argument Types						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * PEN_DATA is the "true" type of msgInputEvent's pArgs->eventData for all
 * pen event messages.
 *    timeStamp:      time the event occurred, as defined by the pen driver.
 *                    This may differ from pArgs->timeStamp, which is time the
 *                    event was enqueued by the input system.
 *    strokeSeqNum:   Number of the stroke that this event is in.  See the 
 *                    section "Stroke Events" for more information.
 *    taps:           if pArgs->devCode is msgPenHoldTimeout this field
 *                    contains the number of taps that occurred before the
 *                    hold.  If pArgs->devCode is msgPenTap, this field
 *                    contains the tap count.
 *    data:           Variable length data.  Contents depends on the
 *                    msgInputEvent's pArgs->devCode.  For instance, if
 *                    pArgs->devCode is msgPenStroke, then this field is the
 *                    beginning of the event's stroke information.
*/
typedef struct PEN_DATA {
	U32					timestamp;
	P_UNKNOWN			reservedPointer;
	U32					strokeSeqNum;
	XY16				penXY;				// in 0.1 mm pen units
	PEN_SUPPORTS_FLAGS	penSupportsFlags;
	S16					pressure;			// For future use.
	S16					zPosition;			// For future use.
	U16					penId;				// For future use.
	S16					xyAngle;			// For future use.
	S16					zAngle;				// For future use.
	U16					reserved[1];
	U8					tipState;			// one of PEN_TIP_STATE_FLAGS
	U8					taps;
	U8					data[1];			// start of variable length data
} PEN_DATA, *P_PEN_DATA;


/*
 PEN_STROKE holds the variable length data for msgInputEvent with a devCode
 of msgPenStroke.  See the section "Stroke Events" for more information. It
 holds the stroke data.  It consists of header information followed by the
 compressed XY data.

 The stroke data can be converted into other useful forms using the
 functions described in the section "Stroke Manipulation Functions."
*/
typedef struct PEN_STROKE {
	RECT16				bounds;		// bounds in pen coordinates
	U16					count;		// number of points
	U16					id;			// stroke id when added to scribble
	struct
		{
		U16 len:15,					// # bytes in the data field 
			selected:1;				// used by scribble object
		}				info;
	U8					data[1];	// byte array of compressed points 
} PEN_STROKE, *P_PEN_STROKE;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Other Types									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef struct CURRENT_STD_PEN_DATA {
	S16					xPosition;			// in 0.1 mm pen units
	S16					yPosition;			// in 0.1 mm pen units
	PEN_TIP_STATE_TYPE	penTipState;
	U16					zPosition;			// For future use.
	U16					pressure;			// For future use.
	U16					penId;				// For future use.
	U16					xyAngle;			// For future use.
	U16					zAngle;				// For future use.
	XY32				positionAcetate;
} CURRENT_STD_PEN_DATA, *P_CURRENT_STD_PEN_DATA;


typedef struct PEN_METRICS {
	U32					minResolution;			// lines per meter
	U32					maxResolution;			// lines per meter
	U32					currentResolution;		// lines per meter
	U32					maxXPosition;			// using pen resolution
	U32					maxYPosition;			// using pen resolution
	U32					xPosition;				// using pen resolution
	U32					yPosition;				// using pen resolution
	U32					deviceFlags;
	U32 				reservedU32[2];
	PEN_TIP_STATE_FLAGS	penTipState;
	PEN_SUPPORTS_FLAGS	penSupportsFlags;
	U16					lowSampleRate;
	U16					medSampleRate;
	U16					highSampleRate;
	U16					currentSampleRate;
	U16					reportingThreshold;		// using pen resolution
	U16					deviceId;
	U16					reservedU16[4];
} PEN_METRICS, *P_PEN_METRICS;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 *                       		Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgPenMetrics					takes P_PEN_METRICS, returns STATUS
	Sent to thePen.  thePen passes back the current pen device metrics.
*/
#define msgPenMetrics			MakeMsg(clsPen, 0xFE)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
 *                      Stroke Manipulation Functions					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 PenExpander					returns S16
	Decompresses a point from the compressed stroke structure.  
	
 pX and pY must point to the previous point value.  (They must be set to
 the bounding box origin for the first point).  The new point is passed
 back using the same pointers.
  
 The return value is the number of bytes to advance pData to get to the
 next point.
*/
S16 PASCAL PenExpander(P_U8 pData, P_S16 pX, P_S16 pY);


/****************************************************************************
 PenStrokeRetrace			returns S16
	Iterates the points in a compressed stroke structure.
*/
typedef void (PASCAL * P_DRAW_PROC)(P_UNKNOWN, S16, S16);

S16 PASCAL PenStrokeRetrace(
	P_PEN_STROKE	pStroke,	// ptr to the stroke structure 
	P_DRAW_PROC		pProc,		// ptr to a function to process the points 
	S16				xOff,		// x base offset 
	S16				yOff,		// y base offset 
	P_UNKNOWN 		appData		// application specific data 
);


/****************************************************************************
 PenStrokeUnpack16 				returns STATUS
	Expands a compressed stroke to an array of XY16.
*/
STATUS PASCAL PenStrokeUnpack16(
	P_PEN_STROKE	pStroke,	// compressed stroke
	P_XY32			pBase,		// stroke window offset
	P_XY16			pBuffer,	// point buffer
	BOOLEAN			toLWC		// true to transform points to LWC
);


/****************************************************************************
 PenStrokeUnpack32				returns STATUS
	Expands a compressed stroke to an array of XY32.
*/
STATUS PASCAL PenStrokeUnpack32(
	P_PEN_STROKE	pStroke,		// compressed stroke
	P_XY32			pBase,			// stroke window offset
	P_XY32			pBuffer,		// point buffer
	BOOLEAN			toLWC			// true to transform points to LWC
);


/****************************************************************************
 XY16ToPenStroke	returns STATUS
	Converts an array of XY16 values to into a PEN_STROKE.
	
 The function allocates memory for the PEN_STROKE from heapId.  If
 pPointData is null or numPoints is 0 then only the PEN_STROKE data
 structure is allocated.
*/
STATUS EXPORTED XY16ToPenStroke(
	XY16			pPointData[],	//	In: XY point data
	U16				numPoints,		//	In: number of points in pPointData
	OS_HEAP_ID		heapId,			//	In: heap to allocate stroke from
	P_PEN_STROKE	*ppNewPenStroke	//	Out: pointer to new pen stroke
	);


/****************************************************************************
 PenCurrentStandardData		returns nothing
	Fills in the most recent pen data in standard units.
*/
void PASCAL PenCurrentStandardData(P_CURRENT_STD_PEN_DATA	pPenStdData);


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                        		Private									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgPenSetOrientation			takes nothing, returns STATUS
	Private.  Sets the orientation of the pen to match theScreen.
*/
#define msgPenSetOrientation	MakeMsg(clsPen, 0xFD)


/****************************************************************************
 msgPenSetProxTimeout			takes OS_MILLISECONDS, returns STATUS
	Private.  Sets the out-of-proximity de-bounce timeout for the pen.
*/
#define msgPenSetProxTimeout	MakeMsg(clsPen, 0xFC)


/****************************************************************************
 msgPenWarmStart				takes nothing, returns STATUS
	Private.  Resets the Pen after power-up.
*/
#define msgPenWarmStart	 		MakeMsg(clsPen, 0xFB)


/****************************************************************************
 msgPenSetGestureTimeout		takes OS_MILLISECONDS, returns STATUS
	Private.  Sets the writing timeout for the pen.  "Gesture" is an old
	name.
*/
#define msgPenSetGestureTimeout	MakeMsg(clsPen, 0xFA)


/****************************************************************************
 msgPenSetHWTimeout				takes OS_MILLISECONDS, returns STATUS
	Obsolete.
*/
#define msgPenSetHWTimeout		MakeMsg(clsPen, 0xF9)


/****************************************************************************
 msgPenSetHoldTimeout			takes OS_MILLISECONDS, returns STATUS
	Private.  Sets the Tap-Hold gesture timeout for the pen.
*/
#define msgPenSetHoldTimeout	MakeMsg(clsPen, 0xF8)


/****************************************************************************
 msgPenInk						takes BOOLEAN, returns STATUS
	Private.  Sets the static inking mode of the pen driver.  

 Should only be used by Quick Help to prevent all ink until the mode ends.
*/
#define msgPenInk				MakeMsg(clsPen, 0xF7)


/****************************************************************************
 PenObjectCreate	returns STATUS
	Private.  Creates thePen.
*/
STATUS PenObjectCreate(
	void
	);
//REFGEN ENDIGNORE

#endif
