/****************************************************************************
 File: resfile.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.21  $
   $Author:   jbennett  $
	 $Date:   06 Mar 1992 16:12:28  $

 This file contains the API definition for clsResFile.

 clsResFile inherits from clsFileHandle.
 Provides resource and object filing support.

 theSystemResFile is a well known instance of clsResFile.

 clsResList inherits from clsList.
 ResLists are lists of resource files that act like a single resource file
 for reading and searching (but not writing).

 theProcessResList is a process well known instance of clsResList.

 A resource file maintains a collection of 'resources' each identified by
 a 'resource ID'.  A resource is filed data or a filed object.  The types
 of data supported are: byte array, string, and array of strings.  It is
 also possible to create an 'agent' that reads and writes other kinds of
 data.
 
 A resource ID is a 32 bit TAG used as a unique (per file) key to identify
 and select a desired resource.

****************************************************************************/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Overview								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Resource files are used in three general ways: filing & unfiling objects,
 reading theProcessResList for configuration and customization information,
 and application specific data storage.

	-:	The most common case of filing & unfiling objects is a page turn, 
		which needs to save the state of a running process on the disk, and 
		restore the  state of another process from the disk.  This is done by 
		(un)filing the application framework, which (if everything is set up 
		correctly) (un)files directly or indirectly all the objects that make 
		up the state of the process.
		
		Filling of an object is initiated with msgResWriteObject which ends 
		up sending msgSave to the object.  The save procedure uses  
		msgStreamWrite (everything except objects) and msgResPutObject 
		(objects) to write out its instance data.  Unfiling of the object is 
		initiated with msgResReadObject which sends msgRestore to the (newly 
		created) object. the restore procedure uses msgStreamRead and 
		msgResGetObject to read its instance data back in.

	-:	theProcessResList is used for several reasons: to allow text to
		be stored separately from the code, to store pre-built UI objects,
		to allow applications to override system provided items, to provide
		a central set of system wide preferences, etc.  To do this it
		normally (inside an application) contains four resource files:
		DOC.RES (specific to the document), APP.RES (specific to the
		application), current system preferences file, and PENPOINT.RES
		(system wide resource file).  They are searched in the order
		listed above.  There are some utility functions to access
		theProcessResList, see RESUTIL.H for more information on them.

	-:	There are many other ways to use resource files, but they are
		application specific.  If you think you have a use for resource
		files, it is worth checking out, but do be careful, resource files
		are designed and optimized for the first two uses, and do not work
		well for everything that it at first seems like they should.
*/

/****  How a Resource ID is put together  ****/

/*
 The fields in a Resource ID:
	TagNum (which resource object)  = 8 bits
	Flags (see below)               = 2 bits
	Admin (as usual)                = 20 or 19 bits
	Scope (as usual)                = 1 or 2 bits

 They are laid out this way:
//{
	        ---------------------------------
	Name:   0|tagNum |F|     Admin+Scope
	        +-------+-------+-------+-------+
	Size:   1| 8     |2|    20+1 or 19+2
	        ---------------------------------
//}

 The flags are interpreted as follows:
	0	Well-Known Resource ID
	1	Dynamic Resource ID
	2	Well-Known List Resource ID
	3	RESERVED

 The Well-Knowns used here are the same ones used in other tags.  This
 gives us three possible scopes: global, process and local.  Because
 resource files are not tied to a process context, there is no difference
 between the global and process Well-Knowns.  System and service classes
 should only use there own well known.  Applications can not only use the
 well knowns for there own classes, they can also use all local well known
 values.

 Well-Known Resource Ids (flag == 0) can be used to store any kind of
 resource.

 The Dynamic Resource IDs (flag == 1) are used by the resource file in
 msgResPutObject to file nested objects.  It is also possible for other
 code to allocate them using msgResNextDynResId.  They may be used to
 file any kind of resource.  We get 29 bits worth of Dynamic Resource IDs
 by combining the tagNum, admin and scope fields.

 Well-Known List Resource IDs (flag == 2) must be used with list resources
 to allow the Indexed Resource IDs (see below) to work.  The only list
 resource defined by GO is the string array, but it is possible to define
 others.  The tagNum field is split into two fields for List Resource IDs.
 
 The fields in a List Resource ID:
	Group of lists                  = 6 bits
	List in group		            = 2 bits
	Flags (always set to 0x2)       = 2 bits        
	Admin (as usual)                = 20 or 19 bits
	Scope (as usual)                = 1 or 2 bits

 They are laid out this way:
//{
	        ---------------------------------
	Name:   0| Grp |L|2|     Admin+Scope
	        +-------+-------+-------+-------+
	Size:   1|  6  |2|2|     20+1 or 19+2
	        ---------------------------------
//}

 The Groups are allocated as follows:

	00 - 1F		AVAILABLE TO DEVELOPERS
	20          TK Table Lists
	21          Standard Message Lists
	22          Quick Help Lists
	23 - 3F     RESERVED FOR GO
*/

/****  What an Indexed Resource ID is  ****/

/*	
 Indexed Resource IDs are used to access list resources.  They are NOT
 Resource IDs.  Each must be converted into the List Resource ID of the
 desired list plus an index into the list to fetch the desired data.

 The fields in a Indexed Resource ID:
	TagNum (index into list)	= 8 bits
	Flags (which list)			= 2 bits
	Admin (as usual)            = 20 or 19 bits
	Scope (as usual)            = 1 or 2 bits

 They are laid out this way:
//{
	        ---------------------------------
	Name:   0|tagNum |F|     Admin+Scope
	        +-------+-------+-------+-------+
	Size:   1| 8     |2|     20+1 or 19+2
	        ---------------------------------
//}

 You will note that this provides eight bits not provided by a List Resource
 ID (the index) and is missing eight bits needed by it (the flags and group).

 The eight bits of index allow each list to contain up to 256 items.
 Actually they can have more, but only the first 256 can be accessed this way.
 Since there are four lists for each Well-Known, it is possible to access
 up to 1,024 items per group per Well-Known.

 We provide the missing bits as follows.  Since we always map to a List
 Resource ID we know that the flags will be set to 0x2.  Which group to use
 is determined by which API it is used with.  Thus, the passing the same
 Indexed Resource ID to both Quick help and a TK table will result in
 different data items being used.

*/

/****  Warnings to those going off the beaten path.  ****/

/*
 -:  The description above gives the standard way of allocating resource IDs.
 While there is special support for using them this way, and some other parts
 of the system in fact require this usage, the resource file itself does not
 care.  The only time it puts a special interpretation on a resource ID is
 for well-known-object resource IDs.  They have the top (sign) bit set to
 one.  These are automatically created by the resource file, and to avoid
 trouble, should never be created by anything else.

 -:  Dynamic resource IDs are based off of a 29-bit count, and gaps are
 not reused.  Because of this it is possible to run out.  While this
 will not happen in 'normal' use, it is possible for uses that seem
 reasonable.   So if you use them for anything other than normal object
 filing, or are repeatedly filing objects, make sure you do not run
 into this.

 -:  When an object resource is deleted from a resource file, the other
 objects it filed are NOT deleted, and there is no easy way of finding
 them to delete them.  Because of this, repeatedly filing objects will
 result in the file growing without bound unless you work very hard
 to prevent it.

 -:  Opening multiple handles on the same resource file has some limitations.
 It is possible to have as many read-only handles as desired, as long as
 there are no writable handles.  If there is a writable handle, no other
 handles may be opened.  This is do to a limitation of the current
 implementation.  It maintains index information into the file on a per
 handle basis.  If writing was allowed with multiple handles open, these
 tables would become invalid resulting in fatal errors of many kinds.

 -:  While it is possible to use a resource file as a kind of mini-database,
 it was not designed or optimized for such a use.  So, don't be surprised if
 you find it is not up to the task you would like to use it for.

*/

/****  ResFile Debugging Flags (Shared with Penpoint kernel & fs)  ****/

/*
 ResFile flag is 'G', values are:

	1-80	= Used by PenPoint Kernel (see os.h)
	100-800	= Used by File System (see fs.h)

	1000	= Turns on debugging info for reading and writing resources.
	2000	= Turns on timing stats
	4000	= Turns on debugging info for intercepted Stream & FS messages.
	8000	= TBD
*/

#ifndef RESFILE_INCLUDED
#define RESFILE_INCLUDED

#ifndef UUID_INCLUDED
#include <uuid.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif

#ifndef LIST_INCLUDED
#include <list.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif
/*
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 These are used to define resource IDs, both well known (client-defined)
 and dynamic (See uuid.h for comparison).  Note that the count used for
 the dynamic resource ID's is managed internal to the resource file,
 and no attempt should be made to create them elsewhere.
*/

#define resFlagsWkn     0x0
#define resFlagsDyn     0x1
#define resFlagsLists   0x2
#define resFlagsSpare   0x3

#define resFlagWknObj	((RES_ID)0x80000000)

#define MakeWknResId(wkn,i)			\
	MakeTagWithFlags(wkn, i, resFlagsWkn)
#define MakeDynResId(count)			\
	MakeTagWithFlags(((U32)(count))>>8, ((count)&0xFF), resFlagsDyn)
#define MakeListResId(wkn,grp,lst)	\
	MakeTagWithFlags(wkn, ((((U32)(grp))<<2)+((lst)&0x03)), resFlagsLists)
#define MakeWknObjResId(obj)		((RES_ID)(obj) | resFlagWknObj)

/* Extract the pieces from resource IDs. */

#define ResWknObjResId(resId)		((OBJECT)((resId) & ~resFlagWknObj))
#define ResDynIdCount(resId)		(WKNValue(resId)<<8 | Tag(resId))
#define ResListGroup(resId)			(Tag(resId) >> 2)
#define ResListList(resId)			(Tag(resId) & 0x3)

/* Tests on resource ID's */

#define WknObjResId(resId)	((resId) & resFlagWknObj)
#define WknResId(resId)		\
					(!WknObjResId(resId) && TagFlags(resId) != resFlagsDyn)
#define WknItemResId(resId)	\
					(!WknObjResId(resId) && TagFlags(resId) == resFlagsWkn)
#define WknListResId(resId)	\
					(!WknObjResId(resId) && TagFlags(resId) == resFlagsLists)
#define DynResId(resId)		\
					(!WknObjResId(resId) && TagFlags(resId) == resFlagsDyn)

/* Constants */

#define resNilResId		Nil(RES_ID)

/* OBOLETE Resource IDs do NOT use. */

#define residRfSystemVersion			MakeWknResId(clsResFile, 1)
#define residRfApplicationVersion		MakeWknResId(clsResFile, 2)

/* How to make a Indexed resource ID. */

#define MakeIndexedResId(wkn,list,index)\
   MakeTagWithFlags(wkn,index,list)

/*
 The group identifiers used to convert from Indexed resource IDs to normal
 resource IDs. Values from 0x00 to 0x1F are available for use by
 applications. Values from 0x20 to 0x3F are reserved to the system.
*/

#define resGrpTK        0x20
#define resGrpStdMsg    0x21
#define resGrpQhelp     0x22

/****  Predefined Resource Agents  ****/

/*
 These are used by both the resource compiler to define data resources
 and by msgResWriteData to dynamically write a resource.
*/

// Don't use these definitions, use the derived values below
#define resDefaultObjAgent			3	// Use resObjectResAgent
#define resDefaultDataAgent		4	// Use resDataResAgent
#define resStringAgent			5	// Use resStringResAgent
#define resStringArrayAgent		6	// Use resStringArrayResAgent

#define MakePrivateResAgent(x) \
	  ((UID)MakeTag(clsResFile, x))

// These are the pre-defined resource types
#define resDefaultResAgent		objNull
#define resObjectResAgent		MakePrivateResAgent(resDefaultObjAgent)
#define resDataResAgent			MakePrivateResAgent(resDefaultDataAgent)
#define resStringResAgent		MakePrivateResAgent(resStringAgent)
#define resStringArrayResAgent	MakePrivateResAgent(resStringArrayAgent)

/****  Status Codes	 ****/

#define stsResResourceNotFound			MakeStatus(clsResFile, 1)
#define stsResNotDataResource			MakeStatus(clsResFile, 2)
#define stsResNotObjectResource			MakeStatus(clsResFile, 3)
#define stsResBufferTooSmall			MakeStatus(clsResFile, 4)
#define stsResNotFullyRead				MakeStatus(clsResFile, 5)
#define stsResGetNotFromRestore			MakeStatus(clsResFile, 6)
#define stsResPutNotFromSave			MakeStatus(clsResFile, 7)
// removed unused 				MakeStatus(clsResFile, 8)
#define stsResWriteObjDynamicClass		MakeStatus(clsResFile, 9)
// removed unused 				MakeStatus(clsResFile, 10)
#define stsResCompactInReadOrWrite		MakeStatus(clsResFile, 11)
#define stsResIncorrectFileType			MakeStatus(clsResFile, 12)
#define stsResFileCorrupt				MakeStatus(clsResFile, 13)
#define stsResResourceTooBig			MakeStatus(clsResFile, 14)
#define stsResOutOfDynResIds			MakeStatus(clsResFile, 15)

/****  Types  ****/

// Object types.
typedef OBJECT RES_FILE, *P_RES_FILE;
typedef OBJECT RES_LIST, *P_RES_LIST;

/*
 NOTE: That RES_ID is already defined in clsmgr.h because it is
 referenced by msgSave & msgRestore:
//{
	typedef TAG		RES_ID, *P_RES_ID;			// Resource ID
//}
*/

// Modes used in msgNew to control the creation of the resource file.
Enum16(RES_NEW_MODE) {
	// Will the file handle be shared?  Also guarantees concurrence
	resSharedResFile		= flag0,
	// Remove "deleted" fields on close
	resCompactOnClose		= flag1,
	// Compact file when ratio of deleted to non-deleted reaches compactRatio.
	resCompactAuto			= flag2,
	// Check to see that system version is new enough for resources.
	resVerifyVersions		= flag3,
	// Allow unsafe opens, internal use only.
	resUnsafeOpen			= flag4,
	// Default - No Concurrence, compact on close, verify versions.
	resNewDefault			= resCompactOnClose | resVerifyVersions
};

// Duplicate object checking flag for reading objects.
Enum16(RES_READ_OBJ_MODE) {
	resReadObjectOnce	 = 0,	// Should object resource be read once?
	resReadObjectMany	 = 1	// Should object resource be read many times?
};

// Duplicate object checking flag for writing objects.
Enum16(RES_WRITE_OBJ_MODE) {
	resWriteObjectOnce	  = 0,	// Should object resource be written once?
	resWriteObjectMany	  = 1	// Should object resource be written many?
};

// Mode used to control msgResEnumResources.
Enum16(RES_ENUM_MODE) {
	resEnumAll			 = 0,	// Enumerate all resource entries?
	resEnumByResIdClass	 = 1,	// Enumerate by wkn resource ID admin field?
	resEnumByObjectClass = 2,	// Enumerate by object resource's class?
	resEnumByObjectUID	 = 3,	// Enumerate by object resource's uid?
	resEnumByAgent		 = 4,	// Enumerate by resource's agent?
	resEnumNext			 = flag14,	// Or in to enumerate the next item.
	resEnumDefault		 = resEnumAll	// Default - all resources.
};

// Internal flag used to enumerate across resource lists.
#define	resEnumNextFile		0x8000

// Indexed resource IDs.
typedef TAG		IX_RES_ID, *P_IX_RES_ID;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Class ResFile Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew						takes P_RES_FILE_NEW, returns STATUS
	category: class message
	Creates a resource file object.

 Return Value
	stsIncompatibleVersion:		Filed data is incompatible with system.
	stsResIncorrectFileType:	File is not a resource file.
	stsResFileCorrupt:			Size or contents of the file are not valid.
	stsFSAccessDenied:			Incompatible with existing handles(*)

 (*) Note that there can be only one open handle to a writable resource
 file.  The file mode is automatically set to enforce this.

 A resource file compacts itself at close time if the resCompactOnClose
 flag was set in pNew->resFile.mode.
 
 If the resCompactAuto flag is set in pArgs->res.mode then it compacts 
 itself when a resource is written or deleted, if the number of records is 
 greater than compactMinimum and the number of deleted records is greater 
 than compactRatio percent of the records in the file.

 For example, a value of 10 for compactMinimum and 50 for compactRatio 
 implies that compaction should happen whenever there are more than 10 
 resources in the resource file and 50% of them have been marked as deleted.
*/
typedef struct RES_FILE_NEW_ONLY {
	RES_NEW_MODE		mode;
	U16					compactMinimum;
	U16					compactRatio;
	U32					spare1;
	U32					spare2;
} RES_FILE_NEW_ONLY, *P_RES_FILE_NEW_ONLY;

#define	resFileNewFields	\
	fsNewFields				\
	RES_FILE_NEW_ONLY	resFile;

typedef struct RES_FILE_NEW {
	resFileNewFields
} RES_FILE_NEW, *P_RES_FILE_NEW;


/****************************************************************************
 msgNewDefaults				takes P_RES_FILE_NEW, returns STATUS
	category: class message
	Initializes the RES_FILE_NEW structure to default values.

	Zeroes out pArgs->resFile and sets...
	resFile.mode = resNewDefault;
	resFile.compactRatio = 33;
	resFile.compactMinimum = 50;
*/


/****************************************************************************
 msgResFindResource			takes P_RES_FIND, returns STATUS
	Finds a resource in a resource file or a resource list.

 *** This message is obsolete, you should use msgResGetInfo instead.

 This message may be used to determine if a resource exists and to get
 information about that resource.  You must use it before writing or deleting
 a resource if you do not know which resource file (out of a resource list)
 contains the resource (Resource lists only act upon non-destructive
 messages).

 Return Value
	stsBadParam:				resId is a nil resource ID.
	stsResResourceNotFound:		No resource with the given resId exists.
*/
#define msgResFindResource			MakeMsg(clsResFile, 1)

typedef struct RES_FIND {
	RES_ID				resId;			// In: Resource to find
	RES_FILE			file;			// Out: File location of resource
	UID					agent;			// Out: Agent of the resource
	U32					offset;			// Out: Offset in file (Careful!)
	U16					minSysVersion;	// Out: Min sys vers for the resource
	U16					reserved;
} RES_FIND, *P_RES_FIND;

/****************************************************************************
 msgResGetInfo			takes P_RES_INFO, returns STATUS
	Gets information on a resource in a resource file or a resource list.

 This message may be used to determine if a resource exists and to get
 information about that resource.  You must use it before writing or deleting
 a resource if you do not know which resource file (out of a resource list)
 contains the resource (Resource lists only act upon non-destructive
 messages).  This is an improved version of msgResFindResource.  It gives
 a more useful set of values for agent (as in exactly what is in the file),
 and it returns the size of the resource in the file.

 Return Value
	stsBadParam:				resId is a nil resource ID.
	stsResResourceNotFound:		No resource with the given resId exists.
*/
#define msgResGetInfo			MakeMsg(clsResFile, 17)

typedef struct RES_INFO {
	RES_ID				resId;			// In: Resource to find
	RES_FILE			file;			// Out: File location of resource
	UID					agent;			// Out: Agent of the resource
	UID					objClass;		// Out: Class of object (if is object)
	U32					offset;			// Out: Offset in file (Careful!)
	U32					size;			// Out: Size in file (Careful!)
	U16					minSysVersion;	// Out: Min sys vers for the resource
	U16					reserved1;
	U32					reserved;
} RES_INFO, *P_RES_INFO;

/****************************************************************************
 msgResReadData				takes P_RES_READ_DATA, returns STATUS
	Reads resource data from a resource file or resource list.

 This message requires a destination for the read data.	 There are two
 choices.  You can specify a pointer and a length for the data passed back
 (heap = null, pData = ptr, length = xx) or you can specify a valid heap
 from which the resource file will allocate memory for the data
 (heap = heap ID, pData = doesn't matter, length = doesn't matter).
 Typically if the size of the data is already known and it is small and
 short lived, then the data is "allocated" on the stack.  Otherwise, the
 data is allocated on behalf of a heap.

 Some resources require additional data to identify the actual data to be
 passed back.  For example, a string arrays resource requires additional
 information (the index into the array) to find the string to pass back.
 You specify an index in pAgentData (pAgentData = (P_UNKNOWN)index).

 Return Value
	stsBadParam:				resId is a nil resource ID or reading a string
								from a string array resource and the index
								specified in pAgentData is out of range.
	stsResResourceNotFound:		No resource with the given resId exists.
	stsResNotDataResource:		The found resource was an object resource.
	stsResBufferTooSmall:		Supplied buffer isn't big enough to hold data.

 See Also
	msgResWriteData:			To write data to resource file.
	msgResReadObject:			To read an object from a resource file.
*/
#define msgResReadData				MakeMsg(clsResFile, 2)

typedef struct RES_READ_DATA {
	RES_ID				resId;			//
	OS_HEAP_ID			heap;			// Nil if pData is user supplied buf
	P_UNKNOWN			pData;			// I/O: In: user buffer, Out: res data
	U32					length;			// I/O: In: user buf len, Out: res len
	P_UNKNOWN			pAgentData;		// Agent-specific data
	U32					spare1;
} RES_READ_DATA, *P_RES_READ_DATA;

/****************************************************************************
 msgResWriteData			takes P_RES_WRITE_DATA, returns STATUS
	Writes resource data to a file.

 This message writes data to the resource file.	 If the resource already
 exists it is marked as deleted and the new data is written to the end of
 the file.

 Return Value
	stsBadParam:				resId is a nil resource ID.
	stsResResourceTooBig:		Tried to write resource bigger than
								resource file can handle (16Meg).

 See Also
	msgResReadData:				To read data from resource file.
	msgResUpdateData:			To re-write data in a resource file.
	msgResWriteObject:			To write an object to a resource file.
*/
#define msgResWriteData				MakeMsg(clsResFile, 3)

typedef struct RES_WRITE_DATA {
	RES_ID				resId;			//
	P_UNKNOWN			pData;			// Data to be written
	U32					length;			// Optional if agent can compute size
	UID					agent;			// Not used by msgResUpdateData
	P_UNKNOWN			pAgentData;		// Agent-specific data
	U32					spare1;
} RES_WRITE_DATA, *P_RES_WRITE_DATA;

/****************************************************************************
 msgResUpdateData			takes P_RES_WRITE_DATA, returns STATUS
	Updates existing data resource data.

 Use this message if you know that a resource already exists and is only being
 updated.  The only advantage of this message over msgWriteData is that you
 don't have to specify the agent.

 Return Value
	stsBadParam:				resId is a nil resource ID.
	stsResResourceNotFound:		No resource with the given resId exists.
	stsResNotDataResource:		The found resource was an object resource.
	stsResResourceTooBig:		Tried to write resource bigger than
								resource file can handle (16Meg).

 See Also
	msgResReadData:				To read data from resource file.
	msgResWriteData:			To write data to a resource file.
*/
#define msgResUpdateData				MakeMsg(clsResFile, 4)

/****************************************************************************
 msgResReadObject			takes P_RES_READ_OBJECT, returns STATUS
	Reads a resource object from a resource file or resource list.

 An object must be initialized before it can be read.  You must send
 msgNewDefault to clsObject.

 There are two modes that can be applied to reading an object resource,
 resReadObjectOnce and resReadObjectMany.

 Setting mode to resReadObjectOnce, passed back the object that is associated
 with the resource stored in the resource file (per open).  This guarantees
 that all filed references to a given object refer to the same object.  This 
 is the mode to use if you are unfiling data in a msgRestore procedure. There 
 are other uses of it, but they can be very tricky, so make sure you read all 
 of the documentation and understand it thoroughly before you try to use this
 any place other than a msgSave procedure.

 Setting mode to resReadObjectMany, passes back a new copy of the object 
 without regard as to whether the object has already been read in before or 
 not.  This guarantees that each reader gets his own unique instance of the 
 object.  This is the mode to use if you are reading an object resource 
 "template" (the normal case).

 Return Value
	stsBadParam:					resId is a nil resource ID.
	stsResResourceNotFound:			No resource with the given resId exists.
	stsResNotObjectResource:		The found resource was a data resource.
	stsResNotFullyRead:				The msgRestore routine did not read the
									same amount of data as the msgSave wrote.

 See Also
	msgResWriteObject:			To write an object to a resource file.
	msgResReadData:				To read data from a resource file.

 Pseudo code for reading an object resource:
//{
 	#define sampleResId		MakeWknResId(clsXXX, 17)

	readObj.resId = sampleResId;
	readObj.mode  = resReadObjectMany;
	ObjCallRet(msgNewDefaults, clsObject, &readObj.objectNew, status);
	status = ObjCallWarn(msgResReadObject, file, &readObj);
	object = readObj.new.uid;
//}
*/
#define msgResReadObject				MakeMsg(clsResFile, 5)

typedef struct RES_READ_OBJECT {
	RES_READ_OBJ_MODE	mode;			// Duplicate checking mode
	RES_ID				resId;			//
	OBJECT_NEW			objectNew;		// Object passed back in new.uid
	RES_SAVE_RESTORE_FLAGS	sysFlags;	// Only for msgResReadObjectWithFlags
	U16					appFlags;		// Only for msgResReadObjectWithFlags
	U32					spare1;
} RES_READ_OBJECT, *P_RES_READ_OBJECT;

/****************************************************************************
 msgResWriteObject			takes P_RES_WRITE_OBJECT, returns STATUS
	Writes a resource object to a file.

 There are two modes that can be applied to writing an object resource,
 resWriteObjectOnce and resWriteObjectMany.

 Setting mode to resWriteObjectOnce, will only write the object to the
 resource file once (per open).  This guarantees that all filed references
 to a given object refer to the same object.  This is the mode is used by 
 msgResPutObject, and should be used by you if you bypass it and use 
 msgResWriteObject directly in a msgSave procedure.  There are other uses
 of it, but they can be very tricky, so make sure you read all of the
 documentation and understand it thoroughly before you try to use this
 any place other than a msgSave procedure.
 
 Setting mode to resWriteObjectMany, will write a new copy of the object to 
 the resource file whether the object has already been written before or not. 
 This is the mode to use if you are writing an object resource "template"
 (the normal case).

 Return Value
	stsBadParam:					resId is a nil resource ID.
	stsResWriteObjDynamicClass:		Class of object cannot be dynamic.
	stsResResourceTooBig:			Tried to write resource bigger than
									resource file can handle (16Meg).

 See Also
	msgResReadObject:				To read an object from resource file.
	msgResWriteData:				To write data to a resource file.
*/
#define msgResWriteObject			MakeMsg(clsResFile, 6)

typedef struct RES_WRITE_OBJECT {
	RES_WRITE_OBJ_MODE	mode;			// Duplicate checking mode
	RES_ID				resId;			// 
	OBJECT				object;			// Object to write
	RES_SAVE_RESTORE_FLAGS	sysFlags;	// Only for msgResWriteObjectWithFlags
	U16					appFlags;		// Only for msgResWriteObjectWithFlags
	U32					spare1;
} RES_WRITE_OBJECT, *P_RES_WRITE_OBJECT;

/****************************************************************************
 msgResGetObject			takes P_OBJECT, returns STATUS
	Reads the filed object resource from the current file position.

 This should only be called by routines responding to msgRestore.
 This message is provided as a convenience.	 It eliminates the need for
 everyone to duplicate the same code and guarantees that the parallel
 operation (msgResPutObject) will work.

 Return Value
	stsResGetNotFromRestore:	This was sent in a context other than in
								response to a msgRestore.

 This message is equivalent to this pseudo code:

//{
	STREAM_READ_WRITE	fsRead;
	RES_READ_OBJECT		resRead;
	STATUS				status;

	// Read the object's resource ID from the file.
	fsRead.numBytes	= SizeOf(resRead.resId);
	fsRead.pBuf		= &resRead.resId;
	ObjCallRet(msgStreamRead, pArgs->file, &fsRead, status);

	// Set up the read resource object request.
	resRead.mode = resReadObjectOnce;
	ObjCallRet(msgNewDefaults, clsObject, &resRead.new, status);

	// Read the object if one was filed.
	if (resRead.resId != resNilResId) {
		ObjCallRet(msgResReadObject, pArgs->file, &resRead, status);
	}
//}
*/
#define msgResGetObject				MakeMsg(clsResFile, 8)


/****************************************************************************
 msgResPutObject			takes OBJECT, returns STATUS
	Writes the object as a filed object resource to the current file position.

 This should only be called by routines responding to msgSave.
 This message is provided as a convenience.	 It eliminates the need for
 everyone to duplicate the same code and guarantees that the parallel
 operation (msgResGetObject) is done in the correct order.

 Return Value
	stsResPutNotFromSave:		This was sent in a context other than in
								response to a msgSave.

 This message is equivalent to this pseudo code:
//{
	STREAM_READ_WRITE	fsWrite;
	RES_WRITE_OBJECT	resWrite;
	STATUS				status;

	if (object != Nil(OBJECT)) {

		// Assign an appropriate resource ID to the object.
		if (!ObjectIsDynamic(object)) {
			resWrite.resId	= MakeWknObjResId(object);
		} else {
			ObjCallRet(
				msgResNextDynResId, pArgs->file, &resWrite.resId, status
			);
		}

		// Write the object.
		resWrite.mode	= resWriteObjectOnce;
		resWrite.object	= object;
		ObjCallRet(msgResWriteObject, pArgs->file, &resWrite, status);

	} else {

		// No object.
		resWrite.resId	= resNilResId;

	}

	// Write the object's resId.
	fsWrite.numBytes = SizeOf(resWrite.resId);
	fsWrite.pBuf	 = &resWrite.resId;
	ObjCallRet(msgStreamWrite, pArgs->file, &fsWrite, status);
//}
*/
#define msgResPutObject				MakeMsg(clsResFile, 9)

/****************************************************************************
 msgResReadObjectWithFlags		takes P_RES_READ_OBJECT, returns STATUS
	Reads a resource object, passing the supplied flags.

 This is identical to msgResReadObject except that it copies the flag
 values supplied into all msgRestore calls done by this or any
 object reads that are done recursively from this.

 The values for the sysFlags field are defined by GO and should be examined
 by any object that needs special behavior for any of the defined cases
 (currently only on copy).

 The values for the appFlags field are defined by an application writer.
 Great care must be used with setting or testing these flags.  If the
 flags from one application are used with a class of a second application,
 disaster can result.  E.g. set this field to 0 unless you are very sure
 you know what you are doing.

 Return Value
	stsBadParam:					resId is a nil resource ID.
	stsResResourceNotFound:			No resource with the given resId exists.
	stsResNotObjectResource:		The found resource was a data resource.
	stsResNotFullyRead:				The msgRestore routine did not read the
									same amount of data as the msgSave wrote.

 See Also
	msgResReadObject:				Normal message to read an object
	msgResWriteObjectWithFlags:		The matching write call.
*/
#define msgResReadObjectWithFlags		MakeMsg(clsResFile, 15)

/****************************************************************************
 msgResWriteObjectWithFlags		takes P_RES_WRITE_OBJECT, returns STATUS
	Writes a resource object, passing the supplied flags.

 This is identical to msgResWriteObject except that it copies the flag
 values supplied into all msgSave calls done by this or any object writes
 that are done recursively from this.

 The values for the sysFlags field are defined by GO and should be examined
 by any object that needs special behavior for any of the defined cases
 (currently only on copy).

 The values for the appFlags field are defined by an application writer.
 Great care must be used with setting or testing these flags.  If the
 flags from one application are used with a class of a second application,
 disaster can result.  E.g. set this field to 0 unless you are very sure
 you know what you are doing.

 Return Value
	stsBadParam:					resId is a nil resource ID.
	stsResWriteObjDynamicClass:		Class of object cannot be dynamic.
	stsResResourceTooBig:			Tried to write resource bigger than
									resource file can handle (16Meg).

 See Also
	msgResWriteObject:				Normal message to write an object.
	msgResReadObjectWithFlags:		The matching Read call.
*/
#define msgResWriteObjectWithFlags		MakeMsg(clsResFile, 16)

/****************************************************************************
 msgResDeleteResource		takes RES_ID, returns STATUS
	Deletes the resource identified by RES_ID.

 This marks the resource deleted in the resource file index.  The space taken
 by the resource is reclaimed whenever the resource file is compacted.
 Auto compaction may happen after a resource is deleted.

 Return Value
	stsBadParam:				resId is a nil resource ID.
	stsResResourceNotFound:		No resource with the given resId exists.
*/
#define msgResDeleteResource			MakeMsg(clsResFile, 10)

/****************************************************************************
 msgResCompact				takes void, returns STATUS
	Compacts the resource file.

 This message removes all deleted entries from the file and frees any
 unused space that results.  This can be called automatically in a couple
 of ways.  See msgNew for an explanation of them.
 
 Return Value
	stsResCompactInReadOrWrite:		Can not compact during read or write.
									This only happens if msgCompact is sent
									during msgSave or msgRestore.
*/
#define msgResCompact				MakeMsg(clsResFile, 11)


/****************************************************************************
 msgResFlush				takes void, returns STATUS
	Flushes the resource file index.

 The resource file keeps track of all objects that have filed themselves in
 the resource file.	 It needs this information to implement the
 resReadObjectOnce / resWriteObjectOnce behavior.  If you wish to override
 the resReadObjectOnce / resWriteObjectOnce behavior, then flush the resource
 file.
 
 Clients rarely use this message.  Instead, use the resReadObjectMany /
 resWriteObjectMany modes with msgResReadObject / msgResWriteObject.

 This also sends a msgFSFlush to the file.	If all you want to do is flush
 the file then use msgFSFlush instead of msgResFlush.

 See Also
	msgResReadObject:		To get info on read once / read many.
	msgResWriteObject:		To get info on write once / write many.
*/
#define msgResFlush					MakeMsg(clsResFile, 12)


/****************************************************************************
 msgResEnumResources		takes RES_ENUM, returns STATUS
	Enumerates resources in a resource file or resource list.

 This message will enumerate all resources of a given category (based on mode
 and match) in either a single resource file or a resource list.  The
 max and count fields behave as all other enum messages.  This passes back the
 resource IDs and files that contain the resources in the pResId and pResFile
 arrays.  Mode must always have resEnumNext clear the first time this is
 called and set subsequent times.  Other mode flags selectively filter what
 is being enumerated.

 Return Value
	stsBadParam:			resEnumNext was specified first time.

 Here is some pseudo-code for enumerating:
//{
	#define resMaxEnums	12
	STATUS				status;
	RES_ENUM			rEnum;
	RES_ID				enumResIds[resMaxEnums];
	RES_FILE			enumResFiles[resMaxEnums];

	// Enumerate only objects belonging to clsString of the resources.
	rEnum.max		= resMaxEnums;
	rEnum.count		= resMaxEnums;
	rEnum.mode		= resEnumByObjectClass;
	rEnum.match		= clsString;
	rEnum.pResId	= enumResIds;
	rEnum.pResFile	= enumResFiles;
	for (status = stsOK; status == stsOK; ) {
		status = ObjectCall(msgResEnumResources, resFile, &rEnum);
		for (index = 0; index < rEnum.count; index++) {
			// Process the data, etc, etc
		}
		rEnum.mode |= resEnumNext;
	}
//}
*/
#define msgResEnumResources			MakeMsg(clsResFile, 13)

typedef struct RES_ENUM {
	U16				max;		// size of pResId[] and pResFile[] arrays
								
	U16				count;		// # to pass back in arrays
								// if count > max then memory may be allocated
								// Out: # of valid entries in arrays

	RES_ENUM_MODE	mode;		// Enumerate based on what and first/next.

	UID				match;		// key to match on (i.e. class; agent; etc)
								
	P_RES_ID		pResId;		// Out: ptr to array of resource IDs
	P_RES_FILE		pResFile;	// Out: ptr to array of resource file handles
								// Note: if memory was alloc'd for previous 2
								// fields, client should heap free the memory
//REFGEN BEGINIGNORE
	//
	// Internally managed fields!


	// These fields are used to get to the next item and are not and
	// should not be filled in by the client.
	//
	RES_ID			lastResId;
	RES_FILE		lastResFile;
	BOOLEAN			allocated;
	U16				maxCount;
	U32				spare1;
//REFGEN ENDIGNORE
} RES_ENUM, *P_RES_ENUM;

/****************************************************************************
 msgResNextDynResId			takes P_RES_ID, returns STATUS
	Allocates the next available dynamic resource ID.

 This message may be used to allocate the next dynamic resource ID available,
 so that the caller can write dynamic items without using msgResPutObject.
 WARNING: dynamic IDs are based on a 29 bit count, and the values are not
 recycled.  If you run out of available counts, this will fail.

 Return Value
	stsResOutOfDynResIds:			ran out of dynamic resIDs
*/
#define msgResNextDynResId			MakeMsg(clsResFile, 14)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						   ResFile Agent Message						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgResAgent				takes P_RES_AGENT, returns STATUS
	Message sent by resource file to resource agent when forwarding messages.

 Messages forwarded are msgResReadData, msgResReadObject, msgResWriteData,
 msgResUpdateData, msgResWriteObject and msgResUpdateObject.

 For reads, current file pointer will be positioned at resource entry and
 length of the entry will be passed in length field.  For writes, current
 file pointer will be positioned where write should begin.
*/
#define msgResAgent					MakeMsg(clsResFile, 20)

typedef struct RES_AGENT {
	RES_FILE			file;			// File containing the resource
	U32					length;			// Length of resource entry
	MESSAGE				msg;			// message passed on to agent
	P_UNKNOWN			pArgs;			// In-Out: message specific args
	U16					sysVersion;		// Min sys version if write
	U16					spare;
	U32					spare1;
	U32					spare2;
} RES_AGENT, *P_RES_AGENT;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Class ResList Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew						takes P_RES_LIST_NEW, returns STATUS
	category: class message
	Creates a resource file (search) list object.

 clsResList adds no additional msgNew parameters to clsList.  There are no
 messages specific to clsResList.	 It adds additional behavior.
*/
typedef struct RES_LIST_NEW_ONLY {
	U16				resvd1;
	U16				resvd2;
} RES_LIST_NEW_ONLY, *P_RES_LIST_NEW_ONLY;

#define	resListNewFields	\
	listNewFields			\
	RES_LIST_NEW_ONLY		resList;

typedef struct RES_LIST_NEW {
	resListNewFields
} RES_LIST_NEW, *P_RES_LIST_NEW;

/****************************************************************************
 msgResXxx					takes P_RES_XXX, returns STATUS
 	Non-destructive resource file messages.

 Resource lists accept only non-destructive resource file messages
 (msgResReadData, msgResReadObject, msgResReadObjectWithFlags,
 msgResGetObject, msgResFindResource and msgResEnumResources) and forwards
 the message to each resource file in the list.	Resource files that are
 null are skipped and are not considered an error. The resource list stops
 forwarding the message when either all resource files in the list have
 been exhausted or when one of them responds with a status greater than or
 equal to stsOK.

 Sending msgResEnumResource to a resource file list is special, because it
 forwards the message to all resource files in the list until the list is
 exhausted.	 Thus the enumerated data is representative of the entire resource
 list.

 Return Value
	stsRequestNotSupported:		Msg was not read, find or enum.
	stsListEmpty:				No valid resource files in the list.

 See Also
	stsXXX:						Return values from the resource file messages
								that are sent to the resource list.
*/

#endif	// RESFILE_INCLUDED
