/****************************************************************************
 File: tlayout.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.51  $
   $Author:   cmeyer  $
     $Date:   13 Mar 1992 19:04:30  $

 This file contains the API definition for clsTableLayout.

 clsTableLayout inherits from clsBorder.
 Table layout windows position (and optionally size) their child windows in
 a grid whose parameters you specify.
****************************************************************************/

/**** Debugging Flags ****/
/*
 The clsTableLayout debugging flag is '%'.  Defined values are:

    flag4  (0x0010):    msgWinLayoutSelf info
    flag7  (0x0080):    layout timing
*/

#ifndef TLAYOUT_INCLUDED
#define TLAYOUT_INCLUDED

													#ifndef BORDER_INCLUDED
#include <border.h>
													#endif

/* * * * * * * * *   Common #defines and typedefs   * * * * * * * * * * * */

typedef OBJECT TBL_LAYOUT;

/****  X and Y Alignment Styles  ****/
#define	tlAlignLeft				0				// left-justified
#define	tlAlignCenter			1				// centered
#define	tlAlignRight			2				// right-justified
#define	tlAlignBottom			tlAlignLeft		// bottom-justified
#define	tlAlignTop				tlAlignRight	// top-justified
#define	tlAlignBaseline			3				// vertical/horizontal baseline aligned

/****  Placement Styles  ****/
#define	tlPlaceRowMajor			0	// across each row first
#define	tlPlaceColMajor			1	// down each column first
#define	tlPlaceStack			2	// stack on top of each other
#define	tlPlaceOrientation		3	// landscape: RowMajor, portrait: ColMajor

/****  Extra Space Styles  ****/
#define	tlExtraNone				0	// leave extra space alone
#define	tlExtraFirst			1	// add extra space to 1st row/col
#define	tlExtraAfterFirst		2	// put extra space after 1st row/col
#define	tlExtraLast 			3	// add extra space to last row/col
#define	tlExtraBeforeLast 		4	// put extra space before last row/col
#define	tlExtraAll				5	// add extra space evenly to each row/col
#define	tlExtraBetweenAll		6	// divide extra space after each row/col
//								7	// unused (reserved)
//								..	// unused (reserved)
//								15	// unused (reserved)

typedef struct TBL_LAYOUT_STYLE {
	U16	tblXAlignment	: 2,		// table x alignment within window
		tblYAlignment 	: 2,		// table y alignment within window
		childXAlignment	: 2,		// child x alignment within grid cell
		childYAlignment	: 2,		// child y alignment within grid cell
		placement		: 2,		// order for placing children in the table
		growChildWidth	: 1,		// true to size child to col width
		growChildHeight	: 1,		// true to size child to row height
		senseOrientation: 1,		// adjust according to current orientation
		reverseX		: 1,		// layout from right to left
		reverseY		: 1,		// layout from bottom to top
		wrap			: 1;		// wrap around row/column

	U16	widthExtra		: 4,		// what to do with extra width
		heightExtra		: 4,		// what to do with extra height
		spare1			: 8;		// unused (reserved)
} TBL_LAYOUT_STYLE, *P_TBL_LAYOUT_STYLE;

/*
 Default TBL_LAYOUT_STYLE:
//{
 	tblXAlignment		= tlLeft
 	tblYAlignment		= tlTop
 	childXAlignment		= tlLeft
 	childYAlignment		= tlBottom
 	growChildWidth		= true
 	growChildHeight		= true
 	placement			= tlPlaceRowMajor
 	reverseX			= false
 	reverseY			= false
 	widthExtra			= tlExtraNone
 	heightExtra			= tlExtraNone
//}
 */

/* constraints for Table Layout */
Enum16(TBL_LAYOUT_CONSTRAINT) {
	// for numRows, numCols, colWidth, rowHeight
	tlAbsolute		= 0,		// fixed

	// for colWidth, rowHeight; can also or-in tlBaselineBox
	tlChildrenMax	= 1,		// max of all children
	tlGroupMax		= 2, 		// max of all children on same row/column

	// for numRows, numCols, colWidth, rowHeight
	tlMaxFit		= 3, 		// as many rows/cols as fit given current
								// rowHeight, colWidth, gaps, and parent size
								// or as wide a col/high a row as possible
								// given current numRows, numCols

	// for numRows, numCols
	tlInfinite		= 4			// unbounded number of rows/cols
};

/*
 * The following can be OR'ed into tlChildrenMax or tlGroupMax to 
 * use max. ascender and descender of each child
 *		Note: not implemented for tlChildrenMax
 */
#define	tlBaselineBox		flag7

/*
 * The following can be OR'ed into any colWidth/rowHeight constraint
 * to use the provided baseline rather than the max. baseline
 * 
 * Note: not implemented yet.
 */
#define	tlAbsoluteBaseline	flag6

/*
 * The following can be OR'ed into any colWidth/rowHeight constraint
 * to use tlMaxFit if the width/height is constrained during layout
 * (i.e. wsLayoutResize is off or wsShrinkWrapWidth/Height is off).
 * 
 */
#define	tlMaxFitIfConstrained	flag8

/* macros to extract the parts of a constraint */
#define	TlConstraint(c)		((c) & 0xF)

typedef struct TBL_LAYOUT_COUNT {
	TBL_LAYOUT_CONSTRAINT	constraint;	// see above
	S16						value;		// absolute value
	U32						spare;		// unused (reserved)
} TBL_LAYOUT_COUNT, *P_TBL_LAYOUT_COUNT;

typedef struct TBL_LAYOUT_SIZE {
	TBL_LAYOUT_CONSTRAINT	constraint;			// see above
	S16						value;				// absolute value
	S16						gap;				// space between rows/columns
	S16						baseline;			// absolute baseline (not implemented)
	U16						valueUnits	: 6,	// units for value/gap/baseline
												// (e.g. bsUnitsLayout)
							spare1		: 10;	// unused (reserved)
	U32						spare;				// unused (reserved)
} TBL_LAYOUT_SIZE, *P_TBL_LAYOUT_SIZE;

typedef struct TBL_LAYOUT_METRICS {
	TBL_LAYOUT_COUNT	numRows, numCols;
	TBL_LAYOUT_SIZE		rowHeight, colWidth;
	TBL_LAYOUT_STYLE	style;
	U32					spare;					// unused (reserved)
} TBL_LAYOUT_METRICS, *P_TBL_LAYOUT_METRICS;

/****  Status Values  ****/
/* These are possible return values from msgWinLayoutSelf */
#define	stsTblLayoutLoop		  	MakeStatus(clsTableLayout, 1)
#define	stsTblLayoutBadConstraint 	MakeStatus(clsTableLayout, 2)

/****************************************************************************
 msgNew		takes P_TBL_LAYOUT_NEW, returns STATUS
	category: class message
	Creates a table layout window.

 You first create a table layout window, then insert the children,
 then send msgWinLayout to layout the children.

 Note: if you are using tlAlignBaseline for the childX/YAlignment, you
 must use a colWidth/rowHeight constraint of tlGroupMax | tlBaselineBox.
 Baseline alignment is not implemented with other colWidth or rowHeight
 constraints.

 See Also
 	msgWinLayoutSelf
*/
typedef TBL_LAYOUT_METRICS TBL_LAYOUT_NEW_ONLY, *P_TBL_LAYOUT_NEW_ONLY;

#define	tableLayoutNewFields		\
	borderNewFields					\
	TBL_LAYOUT_NEW_ONLY		tableLayout;

typedef struct {
	tableLayoutNewFields
} TBL_LAYOUT_NEW, *P_TBL_LAYOUT_NEW;

/****************************************************************************
 msgNewDefaults				takes P_TBL_LAYOUT_NEW, returns STATUS
	category: class message
	Initializes the TBL_LAYOUT_NEW structure to default values.

 Zeroes out pArgs->tableLayout and sets
//{
	pArgs->win.flags.style |= 
				  wsShrinkWrapWidth | wsShrinkWrapHeight | wsFileInline;
	pArgs->tableLayout.style.tblXAlignment = tlAlignLeft;
	pArgs->tableLayout.style.tblYAlignment = tlAlignTop;
	pArgs->tableLayout.style.childXAlignment = tlAlignLeft;
	pArgs->tableLayout.style.childYAlignment = tlAlignBottom;
	pArgs->tableLayout.style.growChildWidth = true;
	pArgs->tableLayout.style.growChildHeight = true;

	// Default is horizontal layout.
	pArgs->tableLayout.numRows.constraint = tlAbsolute;
	pArgs->tableLayout.numRows.value = 1;

	pArgs->tableLayout.numCols.constraint = tlInfinite;
	pArgs->tableLayout.numCols.value = 0;

	pArgs->tableLayout.rowHeight.constraint = tlChildrenMax;
	pArgs->tableLayout.rowHeight.value = 0;
	pArgs->tableLayout.rowHeight.gap = 0;

	pArgs->tableLayout.colWidth.constraint = tlGroupMax;
	pArgs->tableLayout.colWidth.value = 0;
	pArgs->tableLayout.colWidth.gap = 0;
//}
*/

/****************************************************************************
 msgTblLayoutGetMetrics		takes P_TBL_LAYOUT_METRICS, returns STATUS
	Passes back current metrics.

*/
#define msgTblLayoutGetMetrics	MakeMsg(clsTableLayout, 1)

/****************************************************************************
 msgTblLayoutSetMetrics		takes P_TBL_LAYOUT_METRICS, returns STATUS
	Sets current metrics.

 clsTableLayout self-sends msgWinLayoutDirty(true).
*/
#define msgTblLayoutSetMetrics	MakeMsg(clsTableLayout, 2)


/****************************************************************************
 msgTblLayoutGetStyle		takes P_TBL_LAYOUT_STYLE, returns STATUS
	Passes back current style values.

*/
#define msgTblLayoutGetStyle	MakeMsg(clsTableLayout, 3)

/****************************************************************************
 msgTblLayoutSetStyle		takes P_TBL_LAYOUT_STYLE, returns STATUS
	Sets style values.

 clsTableLayout self-sends msgWinLayoutDirty(true).
*/
#define msgTblLayoutSetStyle	MakeMsg(clsTableLayout, 4)

/****************************************************************************
 msgTblLayoutXYToIndex		takes P_TBL_LAYOUT_INDEX, returns STATUS
	Determines a child zero-based index from an xy position.

 The index returned is such that if a child were inserted there and
 the table layed out, that child would be at the given xy.
*/
#define msgTblLayoutXYToIndex	MakeMsg(clsTableLayout, 5)

typedef struct TBL_LAYOUT_INDEX {
	XY32	xy;			//  In: table-relative coords
	U16		index;		// Out: zero-based position at which to insert a child
	U32		spare;		// unused (reserved)
} TBL_LAYOUT_INDEX, *P_TBL_LAYOUT_INDEX;


/****************************************************************************
 msgTblLayoutAdjustSections		takes BOOLEAN, returns STATUS
	Adjusts the border edges and margins of children to correctly reflect
	a sectioned table.

 If you have a table layout window in one column and many rows, and the
 children have top or bottom border edges on to demarcate groups, you should
 send msgTblLayoutAdjustSections to the table layout window after you add
 or remove children.  clsTableLayout will turn off borders that are not
 needed.

 If the table needs to be relayed out, msgWinLayout will be self-sent
 if pArgs is true; otherwise msgWinSetLayoutDirty(true) will be self-sent.

 Note that the current implementation assumes the table is one column,
 infinite rows.
*/
#define msgTblLayoutAdjustSections		MakeMsg(clsTableLayout, 6)


/****************************************************************************
 msgTblLayoutComputeGrid		takes P_TBL_LAYOUT_GRID, returns STATUS
	Computes the table grid parameters given the current constraints.

 This message is self-sent by clsTableLayout in response to msgWinLayoutSelf.
 clsTableLayout responds by computing all of the grid information based on
 the current TBL_LAYOUT_METRICS and current children.

 You can send this message at any time to determine the grid parameters.

 When you send msgTblLayoutComputeGrid, you must set pArgs->pData to pNull.

 You should send msgTblLayoutFreeGrid(pArgs) when finished to free any storage
 allocated by msgTblLayoutComputeGrid.

 If you subclass clsTableLayout, you can respond to this message and compute
 custom grid parameters (e.g. different per-column absolute column widths).

 Note that pArgs->xy is not computed here.  The location of the first grid
 cell can be computed by sending msgTblLayoutComputeGridXY.

 See Also
 	msgTblLayoutFreeGrid
	msgTblLayoutComputeGridXY
*/
#define msgTblLayoutComputeGrid		MakeMsg(clsTableLayout, 7)

/* typical number of children in a table layout window */
#define	tblLayoutAvgChildren	10

typedef struct TBL_LAYOUT_GRID_VALUE {
	S32		value;			// value in device units
	S32		maxBaseline;	// max. baseline for the column/row
	S32		gap;			// gap after row/col, in device units
	U32		spare;			// unused (reserved)
} TBL_LAYOUT_GRID_VALUE, *P_TBL_LAYOUT_GRID_VALUE;

typedef struct TBL_LAYOUT_GRID {
	U16						numCols;	// # of columns
	U16						numRows;	// # of rows
	S32						colWidth;	// column width if pColWidths is pNull
	S32						rowHeight;	// row height if pRowHeights is pNull
	P_TBL_LAYOUT_GRID_VALUE	pColWidths;	// per-column widths, if not pNull
	P_TBL_LAYOUT_GRID_VALUE	pRowHeights;// per-row heights, if not pNull
	TBL_LAYOUT_METRICS		metrics;	// actual metrics
	SIZE32					gap;		// col/row gap, in device units
	U8						placement;	// actual placement
	XY32					xy;			// 1st grid cell in parent space 
	// default storage for column widths, row heights
	TBL_LAYOUT_GRID_VALUE	colWidthBuf[tblLayoutAvgChildren];
	TBL_LAYOUT_GRID_VALUE	rowHeightBuf[tblLayoutAvgChildren];
	P_UNKNOWN				pData;		// reserved for clsTableLayout
	U32						spare1;		// unused (reserved)
	U32						spare2;		// unused (reserved)
} TBL_LAYOUT_GRID, *P_TBL_LAYOUT_GRID;


/****************************************************************************
 msgTblLayoutComputeGridXY		takes P_TBL_LAYOUT_GRID, returns STATUS
	Computes the table grid start xy given the other grid parameters.

 This message is self-sent by clsTableLayout in response to msgWinLayoutSelf.
 clsTableLayout responds by computing the lower-left of the first grid cell
 given the specified grid information.

 You should first send msgTblLayoutComputeGrid(pArgs) to compute the grid
 parameters, then send msgTblLayoutComputeGridXY to determine the location
 of the first cell.

 If style.reverseX is true, the first grid cell is actually at
 pArgs->xy.x - pArgs->colWidth.

 If style.reverseY is true, the first grid cell is actually at
 pArgs->xy.y - pArgs->rowHeight.

 If you subclass clsTableLayout, you can respond to this message and compute
 a custom grid starting location (e.g. something not based on 
 style.tblXAlignment or style.tblYAlignment).

 See Also
 	msgTblLayoutComputeGrid
*/
#define msgTblLayoutComputeGridXY		MakeMsg(clsTableLayout, 8)

/****************************************************************************
 msgTblLayoutFreeGrid		takes P_TBL_LAYOUT_GRID, returns STATUS
	Frees any storage allocated by msgTblLayoutComputeGrid.

 This message is self-sent by clsTableLayout after self-sending
 msgTblLayoutComputeGrid.

 You should send msgTblLayoutFreeGrid when finished with the grid 
 information computed using msgTblLayoutComputeGrid to free any storage
 allocated by msgTblLayoutComputeGrid.

 See Also
 	msgTblLayoutComputeGrid
*/
#define msgTblLayoutFreeGrid		MakeMsg(clsTableLayout, 9)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from other classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.
	
 clsTableLayout will self-send msgWinSetLayoutDirty(true) if the system font
 or system font scale changed since the table was filed.  pArgs->pEnv is
 cast to a P_WIN_RESTORE_ENV and must be a valid window environment pointer.
*/

/****************************************************************************
 msgWinLayoutSelf takes P_WIN_METRICS, returns STATUS
    Tell a window to layout its children (sent during layout).

 clsTableLayout responds by laying out its children.  The grid cells of the
 table are computed based on the TBL_LAYOUT_METRICS specified.  Each
 child is placed in the corresponding grid cell.

 clsTableLayout will self-send msgTblLayoutComputeGrid to compute the grid
 in which the children will be placed.  msgTblLayoutComputeGridXY will be
 self-sent to determine the origin of the grid in self's window.

 The number of columns and rows are computed based on the numCols and numRows
 constraints.  The width and height of each column and row are computed based
 on the colWidth and rowHeight constraints.

 The children are placed acording to style.placement.  For example,
 if style.placement is tlPlaceRowMajor, the children are placed across
 the first row, then the next row, etc..  If style.placement is
 tlPlaceOrientation, then the placement will be based on the current
 orientation of self's window device:

 		Orientation				Placement
		orientPortraitNormal	tlPlaceColMajor
		orientPortraitReverse	tlPlaceColMajor
		orientLandscapeNormal	tlPlaceRowMajor
		orientLandscapeReverse	tlPlaceRowMajor

 If style.senseOrientation is true and the orientation is Landscape, the
 layout metrics are "swapped" as follows:

 	if style.placement is tlPlaceRowMajor, tlPlaceColMajor is used
 	else if style.placement is tlPlaceColMajor, tlPlaceRowMajor is used

	metrics.numRows and metrics.numCols are swapped
	metrics.rowHeight and metrics.colWidth are swapped

 So if you want a layout that is sensitive to the orientation, set the
 constraints to make sense for Portrait orientation and turn on
 style.senseOrientation.  If the orientation is Landscape when the window
 is layed out, the metrics will be altered for you.

 Within each grid cell, each child is aligned acording to style.childXAlignment
 and style.yAlignment.  For example, if style.childXAlignment and 
 style.childYAlignment are both tlAlignCenter, the children are centered in
 each grid cell.

 If style.growChildWidth/Height is true, the width/height of each child is
 set to the width/height of the child's grid cell.

 The entire table is aligned within self acording to style.tblXAlignment and
 style.tblYAlignment. For example, if style.tblXAlignment and 
 style.tblYAlignment are both tlAlignCenter, the table is centered in
 self's window.

 The rows and columns of the table are normally filled out top to bottom,
 left to right.  If style.reverseY is true, the rows are filled out bottom
 to top.  If style.reverseX is true, the columns are filled out right to
 left.

 If pArgs->options has wsLayoutResize on and self has shrink wrap width/height
 on, the width and height of the resulting table will be passed back in
 pArgs->bounds.size.

 Return Value
	stsTblLayoutLoop:  The specified set of constraints results in
		a circular layout loop.  For example, tlMaxFit for numCols and
		tlMaxFit for colWidth.

	stsTblLayoutBadConstraint:  A constraint specified is not a valid value.
*/

/****************************************************************************
 msgWinGetBaseline takes P_WIN_METRICS, returns STATUS
    Gets the desired x,y alignment of a window.

 If the table is one column, clsTableLayout will return the x-baseline of
 the first child in the table (i.e. send msgWinGetBaseline to the first
 child).  Otherwise the x-baseline will be zero.

 If the table is one row, clsTableLayout will return the y-baseline of
 the first child in the table (i.e. send msgWinGetBaseline to the first
 child). Otherwise the y-baseline will be zero.
*/

/****************************************************************************
 msgControlEnable		takes P_CONTROL_ENABLE, returns STATUS
	The control re-evaluates whether it is enabled.

 clsTableLayout recursively enumerates its children (i.e. wsEnumRecursive 
 option to msgWinEnum) and forwards this message to each child
 that inherits from clsControl.  This allows each control in the table
 to respond to alter its enabled state.

 This is used by, for example, clsMenuButton when menuButton.style.enableMenu
 is set to true.

 See Also
 	clsMenuButton
*/

#endif	// TLAYOUT_INCLUDED

